/*
** Copyright 2001-2006 Double Precision, Inc.
** See COPYING for distribution information.
**
*/

#include "config.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <popt.h>
#include <libintl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
#define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
#define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif

#include "superfloppy.h"
#include "sysconfdir.h"
#include "mkfs.h"
#include "mkdosfs.h"
#include "mke2fs.h"
#include "bindir.h"

static const char rcsid[]="$Id: superfloppy.c,v 1.5 2006/01/29 16:15:50 mrsam Exp $";

#define RCFILE SYSCONFDIR "/floppy"

const char rcfile[]=RCFILE;
const char mke2fs[]=MKE2FS;
const char mkdosfs[]=MKDOSFS;

static const struct driver_info idefloppy_driver={
	"idefloppy",
	&idefloppy_probe,
	&idefloppy_capacity,
	&idefloppy_format,
	&idefloppy_eject
} ;

static const struct driver_info floppy_driver={
	"floppy",
	&floppy_probe,
	&floppy_capacity,
	&floppy_format,
	NULL
} ;

static const struct driver_info * const all_drivers[] = {
	&idefloppy_driver,
	&floppy_driver,
	NULL
} ;

struct probe_info {
	const char *device;
	const struct driver_info *driver;
};

static const struct probe_info default_probe[]={
	{
		"/dev/fd0",
		&floppy_driver,
	},
	{
		"/dev/fd1",
		&floppy_driver,
	},
	{
		"/dev/fd2",
		&floppy_driver,
	},
	{
		"/dev/fd3",
		&floppy_driver,
	},
	{
		"/dev/fd4",
		&floppy_driver,
	},
	{
		"/dev/fd5",
		&floppy_driver,
	},
	{
		"/dev/fd6",
		&floppy_driver,
	},
	{
		"/dev/fd7",
		&floppy_driver,
	},
	{
		"/dev/hda",
		&idefloppy_driver,
	},
	{
		"/dev/hdb",
		&idefloppy_driver,
	},
	{
		"/dev/hdc",
		&idefloppy_driver,
	},
	{
		"/dev/hdd",
		&idefloppy_driver,
	},
	{
		"/dev/hde",
		&idefloppy_driver,
	},
	{
		"/dev/hdf",
		&idefloppy_driver,
	},
	{
		"/dev/hdg",
		&idefloppy_driver,
	},
	{
		"/dev/hdh",
		&idefloppy_driver,
	},
	{ NULL, NULL }};

static int doprobe=0, docreaterc=0, docapacity=0, flags=0, doformat=0,
	doshowversion=0, doshowrc=0;
static const char *formatsize=0;
static int fs_flag=0;

static int (* const fs_ptrs[])(const char *, int) = {
	NULL, &mkfs_fat, &mkfs_ext2 };
     
static const char * const fs_bins[]={NULL, mkdosfs, mke2fs};

static int probe()
{
	int i;
	int rc;

	for (i=0; default_probe[i].device; i++)
	{
		rc=(*default_probe[i].driver->probe)(default_probe[i].device,
						     NULL);
		if (rc)
			return (rc);
	}
	return (0);
}

static int createrc()
{
	int found[sizeof(default_probe)/sizeof(default_probe[0])];

	int i;
	int rc;
	int cnt=0;
	int n;

	for (i=0; default_probe[i].device; i++)
	{
		rc=(*default_probe[i].driver->probe)(default_probe[i].device,
						     found+i);
		if (rc)
			return (rc);
		if (found[i])
			++cnt;
	}

	for (n='A', i=0; default_probe[i].device; i++)
	{
		if (!found[i])
			continue;

		printf("%s\t%s%c\t%s\n", default_probe[i].driver->name,
		       cnt <= 2 ? "":"F", n++,
		       default_probe[i].device);
	}
	return (0);
}

static const char *createrc_help()
{
	const char *fmt=_("probe and create %s configuration file (on standard output)");
	char *p=malloc(strlen(fmt)+sizeof(rcfile));

	if (!p)
	{
		perror("malloc");
		exit(1);
	}
	sprintf(p, fmt, rcfile);
	return (p);
}

static const char *showrc_help()
{
	const char *fmt=_("display floppies configured in %s");
	char *p=malloc(strlen(fmt)+sizeof(rcfile));

	if (!p)
	{
		perror("malloc");
		exit(1);
	}
	sprintf(p, fmt, rcfile);
	return (p);
}

static const struct driver_info *find_driver1(const char *, char *, char **);

static const struct driver_info *find_driver(const char *device, char **dev)
{
	const struct driver_info *drvinfo;
	FILE *f;

	if ((f=fopen(rcfile, "r")) != NULL)
	{
		char buf[BUFSIZ];

		while (fgets(buf, sizeof(buf), f) != NULL)
		{
			char *p=strchr(buf, '\n');
			char *driver_name;
			int l;

			if (p)
				*p=0;
			else
			{
				int c;

				while ((c=getc(f)) != EOF && c != '\n')
					;
			}

			p=strchr(buf, '#');
			if (p)
				*p=0;

			driver_name=strtok(buf, "\t");

			if (!driver_name)
				continue;

			p=strtok(NULL, "\t");

			if (!p)
				continue;

			l=strlen(p);

			if (strncasecmp(p, device, l) == 0 &&
			    strcmp(device+l, ":") == 0)
			{
				p=strtok(NULL, "\t");
				if (!p)
					continue;
				drvinfo=find_driver1(driver_name, p, dev);
				fclose(f);
				return (drvinfo);
			}
			p=strtok(NULL, "\t");
			if (!p)
				continue;

			if (strcmp(p, device) == 0)
			{
				drvinfo=find_driver1(driver_name, p, dev);
				fclose(f);
				return (drvinfo);
			}
		}
	}
	return (NULL);
}

static int showrc()
{
	FILE *f;

	if ((f=fopen(rcfile, "r")) != NULL)
	{
		char buf[BUFSIZ];

		while (fgets(buf, sizeof(buf), f) != NULL)
		{
			char *p=strchr(buf, '\n');
			char *driver_name;

			if (p)
				*p=0;
			else
			{
				int c;

				while ((c=getc(f)) != EOF && c != '\n')
					;
			}

			p=strchr(buf, '#');
			if (p)
				*p=0;

			driver_name=strtok(buf, "\t");

			if (!driver_name)
				continue;

			p=strtok(NULL, "\t");

			if (!p)
				continue;

			printf("%s:\n", p);
		}
		fclose(f);
	}
	return (0);
}


static const struct driver_info *find_driver1(const char *name,
					      char *devfile,
					      char **devfilebuf)
{
	int i;

	for (i=0; all_drivers[i]; i++)
		if (strcmp(all_drivers[i]->name, name) == 0)
		{
			*devfilebuf=strdup(devfile);

			if (!*devfilebuf)
			{
				perror("malloc");
				exit(1);
			}
			return (all_drivers[i]);
		}

	fprintf( stderr, _("%s: driver %s not known.\n"), rcfile, name);
	return (NULL);
}

static int main2(int, const char **);

int main(int argc, const char **argv)
{
	setlocale(LC_ALL, "");
	/*	bindtextdomain(PACKAGE, LOCALEDIR); */
	textdomain(PACKAGE);
	return (main2(argc, argv));
}

static void gtkfloppy(const char **argv)
{
	const char *display_env=getenv("DISPLAY");

	/* If called from X, just run floppygtk */

	if (display_env && *display_env &&
	    access(BINDIR "/floppygtk", X_OK) == 0)
	{
		pid_t p, p2;
		int waitstat;

		signal(SIGCHLD, SIG_DFL);
		p=fork();

		if (p < 0)
		{
			perror("fork");
			exit(1);
		}

		if (p > 0)
		{
			p=fork();
			if (p < 0)
			{
				perror("fork");
				exit(1);
			}

			if (p == 0)
			{
				setpgrp();
				execv( BINDIR "/floppygtk", (char **)argv);
				perror( BINDIR "/floppygtk");
				exit(1);
			}
			exit(0);
		}

		while ((p2=wait(&waitstat)) != p)
		{
			if (p2 < 0 && errno == ECHILD)
				break;
		}
		exit(0);
	}
}

static int main2(int argc, const char **argv)
{
	poptContext optContext;
	int rc;
	struct stat stat_buf;

	const char *showversion_help=
		_("Display program version");
	const char *opt_probe_help=
		_("probe and report available floppy devices");
	const char *opt_capacity_help=
		_("show device format sizes");
	const char *opt_brief_help=
		_("no prompts, and brief output");
	const char *opt_verify_help=
		_("verify format");
	const char *opt_ideverify_help=
		_("manually verify ide-floppy format");
	const char *opt_format_help=
		_("format device");
	const char *opt_eject_help=
		_("eject floppy (ide-floppies only)");
	const char *opt_size_help=
		_("format size");
	const char *opt_fat_help=
		_("create a FAT (DOS) filesystem");
	const char *opt_ext2_help=
		_("create an EXT2 (Linux) filesystem");
	const char *opt_createrc_help=createrc_help();
	const char *opt_showrc_help=showrc_help();

	const char *c;
	char *device;

	const struct poptOption opt_list[] = {
		{
			"version",
			0,
			POPT_ARG_NONE,
			&doshowversion,
			0,
			showversion_help,
			NULL
		},
		{
			"probe",
			'p',
			POPT_ARG_NONE,
			&doprobe,
			0,
			opt_probe_help,
			NULL
		},
		{
			"createrc",
			'r',
			POPT_ARG_NONE,
			&docreaterc,
			0,
			opt_createrc_help,
			NULL
		},
		{
			"showrc",
			0,
			POPT_ARG_NONE,
			&doshowrc,
			0,
			opt_showrc_help,
			NULL
		},
		{
			"capacity",
			'c',
			POPT_ARG_NONE,
			&docapacity,
			0,
			opt_capacity_help,
			NULL
		},
		{
			"format",
			'f',
			POPT_ARG_NONE,
			&doformat,
			0,
			opt_format_help,
			NULL
		},
		{
			"size",
			's',
			POPT_ARG_STRING,
			&formatsize,
			0,
			opt_size_help,
			NULL
		},
		{
			"noprompt",
			'n',
			POPT_ARG_VAL | POPT_ARGFLAG_OR,
			&flags,
			FLAGS_BRIEF,
			opt_brief_help,
			NULL
		},
		{
			"verify",
			'v',
			POPT_ARG_VAL | POPT_ARGFLAG_OR,
			&flags,
			FLAGS_FMTVERIFY,
			opt_verify_help,
			NULL
		},
		{
			"eject",
			'j',
			POPT_ARG_VAL | POPT_ARGFLAG_OR,
			&flags,
			FLAGS_EJECT,
			opt_eject_help,
			NULL
		},
		{
			NULL,
			'V',
			POPT_ARG_VAL | POPT_ARGFLAG_OR,
			&flags,
			FLAGS_FMTIDEVERIFY,
			opt_ideverify_help,
			NULL
		},
		{
			"fat",
			0,
			POPT_ARG_VAL,
			&fs_flag,
			1,
			opt_fat_help,
			NULL
		},
		{
			"ext2",
			0,
			POPT_ARG_VAL,
			&fs_flag,
			2,
			opt_ext2_help,
			NULL
		},
		POPT_AUTOHELP
		{ NULL, 0 },
	} ;
	const struct driver_info *driver;


	optContext=poptGetContext("superfloppy",
					      argc,
					      argv,
					      opt_list,
					      0);

	if (!optContext)
	{
		perror("poptGetContext");
		exit (1);
	}

	rc=poptGetNextOpt(optContext);

	if (rc < -1)
	{
		fprintf(stderr, _("%s: %s\n"),
			poptBadOption(optContext, POPT_BADOPTION_NOALIAS),
			poptStrerror(rc));
		exit (1);
	}

	if (doshowversion)
	{
		printf("%s\n", VERSION);
		poptFreeContext(optContext);
		exit (0);
	}

	if (!(flags & FLAGS_BRIEF))
		fprintf(stderr,
			"%s %s Copyright 2001-2006, Double Precision, Inc.\n\n",
			PACKAGE, VERSION);

	device=NULL;
	driver=NULL;

	if ((c=poptGetArg(optContext)) != NULL)
	{
		driver=find_driver(c, &device);
		if (!driver)
		{
			poptFreeContext(optContext);
			fprintf(stderr, _("%s: device %s not found in %s\n"),
				argv[0], c, rcfile);
			exit(1);
		}
	}

	if (doprobe)
	{
		poptFreeContext(optContext);
		return (probe());
	}

	if (docreaterc)
	{
		poptFreeContext(optContext);
		return (createrc());
	}

	if (doshowrc)
	{
		poptFreeContext(optContext);
		return (showrc());
	}

	if (!device)
	{
		gtkfloppy(argv);
		poptPrintHelp(optContext, stdout, 0);
		poptFreeContext(optContext);
		exit(1);
	}

	if (doformat && !formatsize)
	{
		docapacity=1;
		flags |= FLAGS_FORMAT;
	}

	if (fs_bins[fs_flag] && stat(fs_bins[fs_flag], &stat_buf))
	{
		perror(fs_bins[fs_flag]);
		exit(1);
	}

	if (docapacity)
	{
		int rc;

		if (!(flags & (FLAGS_BRIEF|FLAGS_FORMAT)))
			printf( _("Formattable capacities for %s:\n"), device);

		rc=(*driver->capacity)(device, fs_ptrs[fs_flag], flags);

		if (rc)
		{
			fprintf(stderr,
				_("No formattable capacities for %s\n"),
				device);
		}

		poptFreeContext(optContext);
		exit(rc);
		return (0);
	}

	if (doformat)
	{
		int rc;
		if (!formatsize)
		{
			fprintf(stderr, _("%s: format size required.\n"),
				argv[0]);
			exit(1);
		}

		rc=(*driver->format)(device, formatsize, fs_ptrs[fs_flag],
				     flags);
		poptFreeContext(optContext);
		exit(rc);
		return (0);
	}

	if ((flags & FLAGS_EJECT) && driver->eject)
	{
		rc=(*driver->eject)(device, flags);
		poptFreeContext(optContext);
		exit(rc);
		return (0);
	}

	gtkfloppy(argv);
	poptFreeContext(optContext);
	return (0);
}
