/*******************************************************************************
 * Copyright (c) 2007 Red Hat, Inc.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Red Hat Incorporated - initial API and implementation
 *******************************************************************************/
package com.redhat.eclipse.cdt.autotools.ui.editors.parser;

import java.io.IOException;
import java.io.LineNumberReader;
import java.io.StringReader;

import org.eclipse.core.resources.IMarker;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.rules.ICharacterScanner;
import org.eclipse.jface.text.rules.IWordDetector;
import org.eclipse.ui.IEditorInput;

import com.redhat.eclipse.cdt.autotools.AutotoolsPlugin;
import com.redhat.eclipse.cdt.autotools.internal.text.hover.AutoconfPrototype;
import com.redhat.eclipse.cdt.autotools.internal.text.hover.AutoconfTextHover;
import com.redhat.eclipse.cdt.autotools.internal.ui.preferences.AutotoolsEditorPreferenceConstants;
import com.redhat.eclipse.cdt.autotools.ui.editors.AutoconfEditor;
import com.redhat.eclipse.cdt.autotools.ui.editors.AutoconfEditorMessages;
import com.redhat.eclipse.cdt.autotools.ui.editors.AutoconfErrorHandler;
import com.redhat.eclipse.cdt.autotools.ui.editors.AutoconfM4WordDetector;
import com.redhat.eclipse.cdt.autotools.ui.editors.AutoconfMacroWordDetector;
import com.redhat.eclipse.cdt.autotools.ui.editors.ParseException;

public class AutoconfParser {
	protected IWordDetector acMacroDetector;
	protected IWordDetector m4MacroDetector;
	protected AutoconfEditor fEditor;
	
	private String startQuote = "`";
	private String endQuote = "\'";
	
	public final String MISSING_SPECIFIER = "MissingSpecifier"; //$NON-NLS-1$
	public final String INVALID_SPECIFIER = "InvalidSpecifier"; //$NON-NLS-1$
	public final String UNTERMINATED_CONSTRUCT = "UnterminatedConstruct"; //$NON-NLS-1$
	public final String MISSING_CONDITION = "MissingCondition"; //$NON-NLS-1$
	public final String INVALID_ELSE = "InvalidElse"; //$NON-NLS-1$
	public final String INVALID_FI = "InvalidFi"; //$NON-NLS-1$
	public final String INVALID_DONE = "InvalidDone"; //$NON-NLS-1$
	public final String INVALID_ESAC = "InvalidEsac"; //$NON-NLS-1$
	public final String IMPROPER_CASE_CONDITION = "ImproperCaseCondition"; //$NON-NLS-1$
	public final String UNTERMINATED_CASE_CONDITION = "UnterminatedCaseCondition"; //$NON-NLS-1$
	public final String UNTERMINATED_INLINE_DOCUMENT = "UnterminatedInlineDocument"; //$NON-NLS-1$
	public final String INCOMPLETE_INLINE_MARKER="IncompleteInlineMarker"; //$NON-NLS-1$
	public final String MISSING_INLINE_MARKER="MissingInlineMarker"; //$NON-NLS-1$
	public final String UNMATCHED_RIGHT_PARENTHESIS = "UnmatchedRightParenthesis"; //$NON-NLS-1$
	public final String UNMATCHED_LEFT_PARENTHESIS = "UnmatchedLeftParenthesis"; //$NON-NLS-1$
	public final String UNMATCHED_RIGHT_QUOTE = "UnmatchedRightQuote"; //$NON-NLS-1$
	public final String UNMATCHED_LEFT_QUOTE = "UnmatchedLeftQuote"; //$NON-NLS-1$
	public final String MACRO_ARGS_TOO_FEW = "MacroArgsTooFew"; //$NON-NLS-1$
	public final String MACRO_ARGS_TOO_MANY = "MacroArgsTooMany"; //$NON-NLS-1$
	
	public AutoconfParser(AutoconfEditor editor) {
		acMacroDetector = new AutoconfMacroWordDetector();
		m4MacroDetector = new AutoconfM4WordDetector();
		fEditor = editor;
	}
	
	protected class AutoconfLineReader {
		private LineNumberReader reader;
		private String lastLine;
		private int mark;
		public AutoconfLineReader(LineNumberReader reader) {
			this.reader = reader;
		}
		
		public String getLastLine() {
			return lastLine;
		}
		
		public String getLineFromMark() {
			return lastLine.substring(mark);
		}
		
		public int getLineNumber() {
			return reader.getLineNumber();
		}
		
		public String readLine() {
			String line = null;
			try {
				line = reader.readLine();
			} catch (IOException e) {
				// do nothing
			}
			lastLine = line;
			mark = 0;
			return line;
		}
		
		public void setMark(int value) {
			mark = value;
		}
		
		public int getMark() {
			return mark;
		}
		
		public int getOffset(String s) {
			return lastLine.indexOf(s, mark);
		}
	}
	
	public AutoconfElement parse(IDocument document, IEditorInput input) {

		LineNumberReader reader = new LineNumberReader(new StringReader(
				document.get()));
		AutoconfElement root = new AutoconfRootElement();
//		IStorageEditorInput f = (IStorageEditorInput)AutoconfEditor.getDefault().getEditorInput();
//		AutoconfErrorHandler handler = new AutoconfErrorHandler(f, document);
		AutoconfErrorHandler handler = new AutoconfErrorHandler(document, input);
		handler.removeAllExistingMarkers();
		parseLines("", new AutoconfLineReader(reader), root, handler);
		return root;
	}

	protected boolean isQuoted(String s, int pos) {
		boolean quoted = false;
		String tmp = s;
		// Check if position is inside a string
		int quotePos = tmp.lastIndexOf('\"', pos);
		while (quotePos >= 0) {
			quoted = !quoted;
			tmp = tmp.substring(0, quotePos);
			quotePos = tmp.lastIndexOf('\"');
		}
		if (!quoted) {
			// Check for shell command quoting.
			tmp = s;
			quotePos = s.lastIndexOf('`', pos);
			while (quotePos >= 0) {
				quoted = !quoted;
				tmp = tmp.substring(0, quotePos);
				quotePos = tmp.lastIndexOf('`');
			}

			//TODO: what to do with m4 quotes?
//			int startPos = s.lastIndexOf(startQuote, pos);
//			if (startPos >= 0) {
//				quoted = true;
//				// Is not a quote if quote is completed before position.
//				if (s.lastIndexOf(endQuote, pos) > startPos)
//					quoted = false;
//			}
//			if (quoted) {
//				// Ignore m4 quotes if they are within a string themselves.
//				tmp = s;
//				quotePos = tmp.lastIndexOf('\"', startPos);
//				while (quotePos >= 0) {
//					quoted = !quoted;
//					tmp = tmp.substring(0, quotePos);
//					quotePos = tmp.lastIndexOf('\"');
//				}
//			}
		}
		return quoted;
	}
	
	protected int findKeyword(String s, String val) {
		int pos = -1;
		int startPos = 0;
		boolean finished = false;
		while (!finished) {
			int tmpPos = s.indexOf(val, startPos);
			if (tmpPos < 0)
				finished = true;
			else if (tmpPos == 0) {
				finished = true;
				pos = tmpPos;
			}
			else {
				char preChar = s.charAt(tmpPos - 1);
				// Ignore $val as this is not a keyword
				if (preChar == '$' ||
						!Character.isWhitespace(preChar) &&
						preChar != ';') {
					startPos = tmpPos + val.length();
					continue;
				}
				// Don't allow anything after keyword except whitespace
				// or end of line.
				if (s.length() > tmpPos + val.length()) {
					char postChar = s.charAt(tmpPos + val.length());
					if (!Character.isWhitespace(postChar)) {
						startPos = tmpPos + 1;
						continue;
					}
				}
				// Verify keyword isn't hidden within a string or quoted
				// in which case it is not a keyword.
				if (!isQuoted(s, tmpPos)) {
					pos = tmpPos;
					finished = true;
				} else {
					startPos = tmpPos + val.length();
				}
			}
		}
		return pos;
	}
	
	protected int findDelimeter(String s, char ch) {
		int pos = -1;
		int startPos = 0;
		boolean finished = false;
		while (!finished) {
			int tmpPos = s.indexOf(ch, startPos);
			if (tmpPos < 0)
				finished = true;
			else if (tmpPos == 0) {
				finished = true;
				pos = tmpPos;
			}
			else {
				char preChar = s.charAt(tmpPos - 1);
				// Ignore $ch
				if (preChar == '$' || preChar == '\\') {
					startPos = tmpPos + 1;
					continue;
				}
				// Verify char isn't hidden within a string or quoted in which
				// case it is not a delimeter.
				if (!isQuoted(s, tmpPos)) {
					pos = tmpPos;
					finished = true;
				} else {
					startPos = tmpPos + 1;
				}
			}
		}
		return pos;
	}

	public int findComment(String line) {
		// Check for dnl being part of an identifier (we don't want to
		// treat such as a comment (e.g. endnl=x or dnlb=y).
		int dnlPos = findKeyword(line, "dnl"); //$NON-NLS-1$
		int poundPos = findDelimeter(line, '#');
		if (dnlPos >= 0 && poundPos >= 0)
			return (dnlPos < poundPos ? dnlPos : poundPos);
		else if (dnlPos >= 0)
			return dnlPos;
		else if (poundPos >= 0)
			return poundPos;
		return -1;
	}

	protected String parseLines(String line, AutoconfLineReader reader, AutoconfElement parent,
			AutoconfErrorHandler handler) {
		try {
			while (line != null) {
				if (line.length() == 0)
					line = reader.readLine();
				// The following is expected to generate a NullPointerException when we
				// reach EOF
				line = line.trim();
				// 0. Process changequote() command to get any quote changes.
				if (line.startsWith("changequote")) { //$NON-NLS-1$
					line = parseChangequote(line, reader, parent, handler);
					if (line.length() == 0)
						continue;
				}
				

				// 2. Look for macro references.
				if (line.startsWith("AC_") || line.startsWith("AM_")) { //$NON-NLS-1$ //$NON-NLS-2$
					line = parseMacro(line, reader, parent, handler, acMacroDetector);
					if (line.length() == 0)
						continue;
				}

				if (line.startsWith("m4")) { //$NON-NLS-1$
					line = parseMacro(line, reader, parent, handler, m4MacroDetector);
					if (line.length() == 0)
						continue;
				}
				
				// 2. Strip out comments
				int commentStart = findComment(line);
				if (commentStart >= 0) {
					line = line.substring(0, commentStart).trim();
					if (line.length() == 0)
						continue;
				}
				
				// 3. Look for if, elif, or else statements
				if (line.startsWith("if")) { //$NON-NLS-1$
					line = parseIf(line, reader, parent, handler);
					if (line.length() == 0)
						continue;
				}

				if (line.startsWith("elif")) { //$NON-NLS-1$
					String tmp = line;
					line = parseElif(line, reader, parent, handler);
					if (tmp != line)
						return line;
				}
				
				if (line.startsWith("else")) { //$NON-NLS-1$
					String tmp = line;
					line = parseElse(line, reader, parent, handler);
					if (tmp != line)
						return line;
				}
				
				// 4. Look for for/while loops
				if (line.startsWith("for")) { //$NON-NLS-1$
					line = parseFor(line, reader, parent, handler);
					if (line.length() == 0)
						continue;
				}

				if (line.startsWith("while")) { //$NON-NLS-1$
					line = parseWhile(line, reader, parent, handler);
					if (line.length() == 0)
						continue;
				}

				// 5. Look for case statements
				if (line.startsWith("case")) { //$NON-NLS-1$
					line = parseCase(line, reader, parent, handler);
					if (line.length() == 0)
						continue;
				}
				
				if (line.startsWith("esac")) { //$NON-NLS-1$
					String tmp = line;
					line = parseEsac(line, reader, parent, handler);
					if (tmp != line)
						return line;
				}

				if (parent instanceof AutoconfCaseElement) {
					line = parseCaseCondition(line, reader, parent, handler);
					if (line.length() == 0 || line.startsWith("esac"))
						continue;
				}
				
				// 6. Look for construct endings.
				if (line.startsWith("fi")) { //$NON-NLS-1$
					// We use a trick to see if the fi is really a fi.
					// If the same line is returned, then we didn't find
					// a fi.
					String tmp = line;
					line = parsefi(line, reader, parent, handler);
					if (tmp != line)
						return line;	
				}
				
				if (line.startsWith("done")) { //$NON-NLS-1$
					String tmp = line;
					line = parseDone(line, reader, parent, handler);
					if (tmp != line)
						return line;
				}

				// Look for command separator.
				int semi = findDelimeter(line, ';');
				if (semi >= 0) {
					// If we are processing a case condition, ";;" means
					// end of case.
					if (parent instanceof AutoconfCaseConditionElement &&
							line.length() > semi + 1 &&
							line.charAt(semi + 1) == ';')
						return line.substring(semi + 2).trim();
					// Otherwise, we have a multistatement line.  Process
					// next statement.
					int realSemi = findDelimeter(reader.getLineFromMark(), ';') + reader.getMark();
					AutoconfElement element = parent.getLastChild();
					if (element != null && 
							element.getEndLineNumber() == reader.getLineNumber() - 1 &&
							element.getEndOffset() == 0) {
						element.setEndOffset(realSemi);
					}
					reader.setMark(realSemi + 1);
					line = line.substring(semi + 1).trim();
					reader.setMark(reader.getOffset(line));
					continue;
				}
				
				int lt = findDelimeter(line, '<');
				if (lt >= 0) {
					// We might have a HERE document (<<[-]WORD) situation in which case we need
					// to ignore all lines up to the line that only contains WORD
					if (line.length() > lt + 1 && line.charAt(lt + 1) == '<') {
						line = parseHERE(line, reader, parent, handler);
					}
				}
				else
					line = reader.readLine();
			}
		} catch (NullPointerException e) {
			if (line != null)
				e.printStackTrace();
		} catch (Exception e) {
			e.printStackTrace();
		}
		
		// Verify that we reached EOF without a dangling construct. 
		if (!(parent instanceof AutoconfRootElement)) {
			try {
				handler.handleError(new ParseException(
						AutoconfEditorMessages.getFormattedString(UNTERMINATED_CONSTRUCT, parent.getName()),
						parent.getLineNumber(), 0, handler.getDocument().getLineLength(parent.getLineNumber()),
						IMarker.SEVERITY_ERROR));
			} catch (BadLocationException e2) {
				// ignore at this point
			}
		}
		return null;
	}
	
	protected String readerRemainder(StringReader x) {
		StringBuffer buffer = new StringBuffer();
		try {
			int ch = x.read();
			while (ch != ICharacterScanner.EOF) {
				buffer.append((char)ch);
				ch = x.read();
			}
		} catch (IOException e) {
			// Do nothing.
		}
		return buffer.toString();
	}
	
	protected String matchParentheses(StringReader x, AutoconfLineReader reader, 
			AutoconfMacroElement macro, AutoconfErrorHandler handler) {
		boolean finished = false;
		int depth = 1;
		int quoteDepth = 0;
		int commaCount = 0;
		StringBuffer var = new StringBuffer();
		StringBuffer line = new StringBuffer();
		try { 
			while (!finished) {
				int c = x.read();
				if (c == ICharacterScanner.EOF) {
					String newLine = reader.readLine();
					if (newLine != null) {
						x = new StringReader(newLine.trim());
						line.delete(0, line.length());
					} else {
						finished = true;
					}
					continue;
				}

				line.append((char)c);
				
				if (commaCount == 0)
					var.append((char)c);
				
				if (c == '[') {
					++quoteDepth;
				}
				else if (c == ']') {
					--quoteDepth;
					if (quoteDepth < 0)
						finished = true;
				}
				if (quoteDepth == 0) {
					if (c == ')') {
						--depth;
						if (depth <= 0)
							finished = true;
					}
					else if (c == '(') {
						++depth;
					}
					else if (c == ',') {
						++commaCount;
					}
				}
			}
			macro.setParmCount(commaCount + 1);
			macro.setQuoteDepth(quoteDepth);
			macro.setDepth(depth);
			if (var.length() > 0)
				var.deleteCharAt(var.length() - 1);
			if (var.length() > 15)
				macro.setVar(var.substring(0, 15) + "..."); //$NON-NLS-1$
			else
				macro.setVar(var.toString());
		} catch (IOException e) {
			// Do nothing.
		}
		return readerRemainder(x);
	}
	
	public String parseMacro(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler,
			IWordDetector macroDetector) {
		StringReader x = new StringReader(line);
		StringBuffer buffer = new StringBuffer();
		try {
			int ch = x.read();
			while (ch != ICharacterScanner.EOF && macroDetector.isWordPart((char)ch)) {
				buffer.append((char)ch);
				ch = x.read();
			}
			AutoconfMacroElement macro = new AutoconfMacroElement(buffer.toString(), reader.getLineNumber() - 1, reader.getMark());
			parent.addChild(macro);
			if ((char)ch == '(') {
				line = matchParentheses(x, reader, macro, handler);
			} else {
				line = readerRemainder(x);
			}
			int lineNumber = macro.getLineNumber();
			// Check basics: matching parentheses and matching quotes
			if (macro.getQuoteDepth() < 0) {
				handler.handleError(new ParseException(
						AutoconfEditorMessages.getString(UNMATCHED_RIGHT_QUOTE),
						lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
						IMarker.SEVERITY_ERROR));
			} else if (macro.getQuoteDepth() > 0) {
				handler.handleError(new ParseException(
						AutoconfEditorMessages.getString(UNMATCHED_LEFT_QUOTE),
						lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
						IMarker.SEVERITY_ERROR));
			} else if (macro.getDepth() < 0) {
				handler.handleError(new ParseException(
						AutoconfEditorMessages.getString(UNMATCHED_RIGHT_PARENTHESIS),
						lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
						IMarker.SEVERITY_ERROR));
			} else if (macro.getDepth() > 0) {
				handler.handleError(new ParseException(
						AutoconfEditorMessages.getString(UNMATCHED_LEFT_PARENTHESIS),
						lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
						IMarker.SEVERITY_ERROR));
			}
			AutoconfPrototype p = AutoconfTextHover.getPrototype(macro.getName(), fEditor);
			if (p != null) {
				boolean tooFew = false;
				boolean tooMany = false;
				boolean justRight = false;
				int parms = macro.getParmCount();
				int numPrototypes = p.getNumPrototypes();
				int minParms = 0;
				int maxParms = 0;
				for (int i = 0; i < numPrototypes; ++i) {
					if (parms < p.getMinParms(i)) {
						tooFew = true;
						minParms = p.getMinParms(i);
					} else if (parms > p.getMaxParms(i)) {
						tooMany = true;
						maxParms = p.getMaxParms(i);
					} else {
						justRight = true;
						break;
					}
				}
				if (!justRight) {
					if (tooFew) {
						String formatString = AutoconfEditorMessages.getFormattedString(MACRO_ARGS_TOO_FEW, 
								AutotoolsPlugin.getDefault().getPreferenceStore().getString(AutotoolsEditorPreferenceConstants.AUTOCONF_VERSION),
								p.getName(), Integer.toString(minParms));
						handler.handleError(new ParseException(
								formatString,
								lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
								IMarker.SEVERITY_WARNING));
					} else if (tooMany) {
						String formatString = AutoconfEditorMessages.getFormattedString(MACRO_ARGS_TOO_MANY,
								AutotoolsPlugin.getDefault().getPreferenceStore().getString(AutotoolsEditorPreferenceConstants.AUTOCONF_VERSION),
								p.getName(), Integer.toString(maxParms));
						handler.handleError(new ParseException(
								formatString,
								lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
								IMarker.SEVERITY_WARNING));
					}
				}
			}
			macro.setEndLineNumber(reader.getLineNumber() - 1);
		} catch (IOException e) {
			// Can't happen with String.
		} catch (BadLocationException e) {
			// Don't care if we fail trying to issue an error marker
		}
		return line;	
	}

	protected String checkForAdditionalSpecifier(String line, AutoconfLineReader reader, AutoconfElement element, 
			AutoconfErrorHandler handler, String construct, String keyword) {
		int pos = findDelimeter(line, ';');
		int keywordPos = findKeyword(line, keyword);
		int lineNumber = element.getLineNumber();
		int constructLen = construct.length();
		if (pos < 0 || keywordPos < 0) {
			element.setVar(line.substring(constructLen).trim());
			try {
				char lastChar = line.charAt(line.length() - 1);
				boolean endsInAnd = line.charAt(line.length() - 2) == '&' &&
				line.charAt(line.length() - 1) == '&';
				boolean endsInOr = line.charAt(line.length() - 2) == '|' &&
				line.charAt(line.length() - 1) == '|';
				if (pos < 0) {
					while (keywordPos < 0 && 
							(lastChar == '\\' || endsInAnd || endsInOr)) {
						line = reader.readLine();
						if (line != null) {
							line = line.trim();
							pos = findDelimeter(line, ';');
							keywordPos = findKeyword(line, keyword);
						}
					}
				}
				if (keywordPos < 0) {
					// Check if next line starts with keyword
					pos = -1;
					line = reader.readLine();
					if (line != null) {
						line = line.trim();
						keywordPos = findKeyword(line, keyword);
					}
				}
				if (keywordPos < 0) {
					if (line != null && line.length() == constructLen) {
						handler.handleError(new ParseException(AutoconfEditorMessages.getFormattedString(MISSING_CONDITION, construct), 
								lineNumber,
								0, handler.getDocument().getLineLength(lineNumber), IMarker.SEVERITY_ERROR));

					} else {
						handler.handleError(new ParseException(AutoconfEditorMessages.getFormattedString(MISSING_SPECIFIER, keyword), 
								lineNumber,
								0, handler.getDocument().getLineLength(lineNumber), IMarker.SEVERITY_ERROR));
					}
				} else {
					line = line.substring(keywordPos + keyword.length());
					// Check that either keyword starts a new line or there
					// is a semicolon between conditional and keyword
					if (pos > keywordPos || pos < 0 && keywordPos != 0) {
						handler.handleError(new ParseException(AutoconfEditorMessages.getFormattedString(INVALID_SPECIFIER, keyword), 
								lineNumber,
								0, handler.getDocument().getLineLength(lineNumber), IMarker.SEVERITY_ERROR));
					}
				}
			} catch (BadLocationException e) {
				// Do nothing.  We got here trying to issue an error marker.
			}
		} else {
			// Skip over construct to get condition
			element.setVar(line.substring(constructLen, pos).trim());
			// Skip over keyword
			line = line.substring(keywordPos + keyword.length());
		}
		return line;
	}
	
	public String parseIf(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		int lineNumber = reader.getLineNumber() - 1;
		AutoconfElement element = new AutoconfIfElement(lineNumber, reader.getMark());
		// Process line(s) looking for "then"
		line = checkForAdditionalSpecifier(line, reader, element, handler, "if", "then"); //$NON-NLS-1$ //$NON-NLS-2$
		if (line == null)
			return null;
		parent.addChild(element);
		String returnLine = parseLines(line, reader, element, handler);
		element.setEndLineNumber(reader.getLineNumber() - 1);
		return returnLine;
	}

	public String parseElif(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		int lineNumber = reader.getLineNumber() - 1;
		AutoconfElement element = new AutoconfElifElement(lineNumber, reader.getMark());
		// Process line(s) looking for "then"
		line = checkForAdditionalSpecifier(line, reader, element, handler, "elif", "then"); //$NON-NLS-1$ //$NON-NLS-2$
		if (line == null)
			return null;
		parent.addChild(element);
		String returnLine = parseLines(line, reader, element, handler);
		element.setEndLineNumber(reader.getLineNumber() - 1);
		return returnLine;
	}
	
	public String parseElse(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		// Make sure we don't have elif as part of an identifier.
		if (line.length() > 4 && !Character.isWhitespace(line.charAt(4)))
			return line;
		int lineNumber = reader.getLineNumber() - 1;
		AutoconfElement element = new AutoconfElseElement(lineNumber, reader.getMark());
		line = line.substring(4).trim();
		element.setVar(line);
		if (!(parent instanceof AutoconfIfElement) &&
				!(parent instanceof AutoconfElifElement)) {
			try {
				handler.handleError(new ParseException(
						AutoconfEditorMessages.getString(INVALID_ELSE),
						lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
						IMarker.SEVERITY_ERROR));
			} catch (BadLocationException e) {
				// Do nothing if we blew up trying to mark an error.
			}
			parent.addChild(element);
		} else 
			parent.addSibling(element);
		String returnLine = parseLines(line, reader, element, handler);
		element.setEndLineNumber(reader.getLineNumber() - 1);
		return returnLine;
	}	

	public String parsefi(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		// Make sure we don't have fi as part of an identifier.
		if (line.length() > 2 && !Character.isWhitespace(line.charAt(2)) &&
				line.charAt(2) != ';')
			return new String(line); // We want to return a new String != line
		
		int lineNumber = reader.getLineNumber() - 1;
		if (!(parent instanceof AutoconfIfElement) &&
				!(parent instanceof AutoconfElifElement) &&
				!(parent instanceof AutoconfElseElement)) {
			try {
				AutoconfElement element = parent;
				boolean finished = false;
				while (!finished && element.getParent() != null) {
					element = element.getParent();
					if (element instanceof AutoconfIfElement ||
							element instanceof AutoconfElifElement ||
							element instanceof AutoconfElseElement) {
						finished = true;
					}
				}
				// Unwound to top.  "fi" is simply in the middle of nowhere.
				if (element instanceof AutoconfRootElement) {
					handler.handleError(new ParseException(
							AutoconfEditorMessages.getString(INVALID_FI),
							lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
							IMarker.SEVERITY_ERROR));
				} else {
					// We are completing an if/elif/else construct somewhere.  Flag all
					// parents above this "fi" as being incomplete.  Unwind the recursive
					// stack by continuously returning "fi" string back to be reprocessed.
					handler.handleError(new ParseException(
							AutoconfEditorMessages.getFormattedString(UNTERMINATED_CONSTRUCT, parent.getName()),
							parent.getLineNumber(), 0, handler.getDocument().getLineLength(parent.getLineNumber()),
							IMarker.SEVERITY_ERROR));
				    return new String(line); // We want to return a new String != line
				}
			} catch (BadLocationException e) {
				// Do nothing if we blew up trying to mark an error.
			}
		}
		return line.substring(2).trim();
	}	

	public String parseCase(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		// Skip over initial "case".
		int lineNumber = reader.getLineNumber() - 1;
		AutoconfElement element = new AutoconfCaseElement(lineNumber, reader.getMark());
		int pos = findKeyword(line, "in"); //$NON-NLS-1$
		if (pos < 0) { //$NON-NLS-1$
			element.setVar(line.substring(4).trim());
			try {
				// Check if line is continued and specifier is in continued section of line.
				char lastChar = line.charAt(line.length() - 1);
				while (pos < 0 && lastChar == '\\') {
					line = reader.readLine();
					if (line != null) {
						line = line.trim();
						pos = findKeyword(line, "in"); //$NON-NLS-1$
					}
				}
				if (pos < 0) {
					handler.handleError(new ParseException(AutoconfEditorMessages.getFormattedString(MISSING_SPECIFIER, "in"), //$NON-NLS-1$ 
							lineNumber,
							0, handler.getDocument().getLineLength(lineNumber), IMarker.SEVERITY_ERROR));

				}
			} catch (BadLocationException e) {
				// Don't care if we blow up trying to issue error marker.
			}
		} else {
			// Skip over in
			element.setVar(line.substring(4, pos).trim());
			line = line.substring(pos + 2);
		}
		parent.addChild(element);
		String returnLine = parseLines(line, reader, element, handler);
		element.setEndLineNumber(reader.getLineNumber() - 1);
		return returnLine;	
	}

	public String parseEsac(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		// Make sure we don't have elif as part of an identifier.
		if (line.length() > 4 && !Character.isWhitespace(line.charAt(4)))
			return line;

		int lineNumber = reader.getLineNumber() - 1;
		if (parent instanceof AutoconfCaseConditionElement) {
			try {
				handler.handleError(new ParseException(
						AutoconfEditorMessages.getString(UNTERMINATED_CASE_CONDITION),
						parent.getLineNumber(), 0, 
						handler.getDocument().getLineLength(parent.getLineNumber()),
						IMarker.SEVERITY_ERROR));
			} catch (BadLocationException e) {
				// Do nothing if we blew up trying to mark an error.
			}
			// Let parent "case" see "esac" and think it is terminated.
			// We can't just return the line "as-is" as there is a check
			// for this in case "esac" is the start of an identifier.
			return new String(line);
		}
		if (!(parent instanceof AutoconfCaseElement)) {
			try {
				AutoconfElement element = parent;
				boolean finished = false;
				while (!finished && element.getParent() != null) {
					element = element.getParent();
					if (element instanceof AutoconfCaseElement)
						finished = true;
				}
				// Unwound to top.  "esac" is simply in the middle of nowhere.
				if (element instanceof AutoconfRootElement) {
					handler.handleError(new ParseException(
							AutoconfEditorMessages.getString(INVALID_ESAC),
							lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
							IMarker.SEVERITY_ERROR));
				} else {
					// We are completing a case construct somewhere.  Flag all
					// parents above this "esac" as being incomplete.  Unwind the recursive
					// stack by continuously returning "esac" string back to be reprocessed.
					handler.handleError(new ParseException(
							AutoconfEditorMessages.getFormattedString(UNTERMINATED_CONSTRUCT, parent.getName()),
							parent.getLineNumber(), 0, handler.getDocument().getLineLength(parent.getLineNumber()),
							IMarker.SEVERITY_ERROR));
				    return new String(line); // We want to return a new String != line
				}
			} catch (BadLocationException e) {
				// Do nothing if we blew up trying to mark an error.
			}
		}
		return line.substring(4).trim();
	}
	
	public String parseCaseCondition(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		// We expect to find conditions ended with ')'
		boolean finished = false;
		line = line.trim();
		if (line.length() == 0)
			return line;
		String tmp = line;
		int endPos = -1;
		AutoconfElement element = null;
		int lineNumber = reader.getLineNumber() - 1;
		
		try {
			while (!finished) {
				endPos = findDelimeter(tmp, ')');
				int startPos = findDelimeter(tmp, '(');
				if (startPos > 0 && startPos < endPos) {
					tmp = tmp.substring(endPos+1);
				} else if (endPos < 0) {
					char lastChar = tmp.charAt(tmp.length() - 1);
					if (lastChar == '\\') {
						line = reader.readLine();
						if (line != null) {
							line = line.trim();
							tmp = line;
						} else
							finished = true;
					}
					else
						finished = true;
				} else
					finished = true;
			}
		} catch (Exception e) {
			// Do nothing.  We don't have a proper condition.
		}
		
		if (endPos > 0) {
			element = new AutoconfCaseConditionElement(tmp.substring(0, endPos), lineNumber);
			line = line.substring(endPos + 1);
		} else {
			// Generate warning that case condition is improper.  This will allow for some unforeseen macros
			// being used.
			try {
				handler.handleError(new ParseException(
						AutoconfEditorMessages.getString(IMPROPER_CASE_CONDITION),
						lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
						IMarker.SEVERITY_WARNING));
			} catch (BadLocationException e) {
				// We don't care if we blow up trying to issue an error marker.
			}
			int caseEnd = findDelimeter(tmp, ';');
			if (caseEnd > 0 && tmp.length() > caseEnd + 1 && 
					tmp.charAt(caseEnd+1) == ';') {
				return tmp.substring(caseEnd + 2);
			}
			return "";
		}	
		parent.addChild(element);
		return parseLines(line, reader, element, handler);	
	}

	public String parseFor(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		int lineNumber = reader.getLineNumber() - 1;
		AutoconfElement element = new AutoconfForElement(lineNumber, reader.getMark());
		// Process line(s) looking for "do"
		line = checkForAdditionalSpecifier(line, reader, element, handler, "for", "do"); //$NON-NLS-1$ //$NON-NLS-2$
		if (line == null)
			return null;
		parent.addChild(element);
		String returnLine = parseLines(line, reader, element, handler);
		element.setEndLineNumber(reader.getLineNumber() - 1);
		return returnLine;
	}
	
	public String parseWhile(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		int lineNumber = reader.getLineNumber() - 1;
		AutoconfElement element = new AutoconfWhileElement(lineNumber, reader.getMark());
		// Process line(s) looking for "do"
		line = checkForAdditionalSpecifier(line, reader, element, handler, "while", "do"); //$NON-NLS-1$ //$NON-NLS-2$
		if (line == null)
			return null;
		parent.addChild(element);
		String returnLine = parseLines(line, reader, element, handler);
		element.setEndLineNumber(reader.getLineNumber() - 1);
		return returnLine;
	}
	
	public String parseDone(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		// Make sure we don't have done as part of an identifier.
		if (line.length() > 4 && !Character.isWhitespace(line.charAt(4)) &&
				line.charAt(4) != ';')
			return line;
	
		int lineNumber = reader.getLineNumber() - 1;
		if (!(parent instanceof AutoconfForElement) &&
				!(parent instanceof AutoconfWhileElement)) {
			try {
				AutoconfElement element = parent;
				boolean finished = false;
				while (!finished && element.getParent() != null) {
					element = element.getParent();
					if (element instanceof AutoconfForElement ||
							element instanceof AutoconfWhileElement)
						finished = true;
				}
				// Unwound to top.  "done" is simply in the middle of nowhere.
				if (element instanceof AutoconfRootElement) {
					handler.handleError(new ParseException(
							AutoconfEditorMessages.getString(INVALID_DONE),
							lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
							IMarker.SEVERITY_ERROR));
				} else {
					// We are completing a for/while construct somewhere.  Flag all
					// non-loop parents above this "done" as being incomplete.  Unwind the recursive
					// stack by continuously returning "done" string back to be reprocessed.
					handler.handleError(new ParseException(
							AutoconfEditorMessages.getFormattedString(UNTERMINATED_CONSTRUCT, parent.getName()),
							parent.getLineNumber(), 0, handler.getDocument().getLineLength(parent.getLineNumber()),
							IMarker.SEVERITY_ERROR));
				    return new String(line); // We want to return a new String != line
				}
			} catch (BadLocationException e) {
				// Do nothing if we blew up trying to mark an error.
			}
		}
		return line.substring(4).trim();
	}

	public String parseChangequote(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		// Make sure we don't have changequote as part of an identifier.
		if (line.length() > 11 && Character.isJavaIdentifierPart(line.charAt(11)))
			return line;
		
		line = line.substring(11);
		
		if (line.charAt(0) != '(') {
			// Default quotation
			startQuote = "`";
			endQuote = "\'";
		} else {
			//TODO: parse changequote
			int endpos = line.indexOf(')');
			line = line.substring(endpos + 1);
		}
		return line.trim();
	}

	public String parseHERE(String line, AutoconfLineReader reader,
			AutoconfElement parent, AutoconfErrorHandler handler) {
		int index = line.indexOf("<<");
		int lineNumber = reader.getLineNumber() - 1;
		if (!isQuoted(line, index)) {
			line = line.substring(index+2).trim();
			String tmp = line;

			if (tmp.length() > 0) {
				// Skip over '-' as this is part of the HERE specifier
				if (tmp.charAt(0) == '-')
					tmp = tmp.substring(1);
				else if (tmp.charAt(0) == '\'') {
					index = tmp.indexOf('\'', 1);
					if (index > 0)
						tmp = tmp.substring(1, index);
					else {
						tmp = tmp.substring(1);
						try {
							handler.handleError(new ParseException(
									AutoconfEditorMessages.getString(INCOMPLETE_INLINE_MARKER),
									lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
									IMarker.SEVERITY_ERROR));
						} catch (BadLocationException e) {
							// Don't care if we blow up trying to issue marker
						}
						tmp = tmp.substring(1);
					}
				}
				String[] tokens = tmp.split("\\s+");
				String eofMarker = tokens[0].replaceAll("\\\\", "");
				try {
					line = reader.readLine();
					while (line != null && !line.equals(eofMarker)){
						line = reader.readLine();
					}
					if (line == null) {
						handler.handleError(new ParseException(
								AutoconfEditorMessages.getString(UNTERMINATED_INLINE_DOCUMENT),
								lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
								IMarker.SEVERITY_ERROR));
					} else
						line = "";
				} catch (BadLocationException e) {
					// Don't care if we blow up trying to issue marker.
				}
			} else {
				try {
					handler.handleError(new ParseException(
							AutoconfEditorMessages.getString(MISSING_INLINE_MARKER),
							lineNumber, 0, handler.getDocument().getLineLength(lineNumber),
							IMarker.SEVERITY_ERROR));
				} catch (BadLocationException e) {
					// Don't care if we blow up trying to issue marker
				}
			}
		}
		return line;
	}
}

