/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

//Own
#include "install.h"

//Qt
#include <qdir.h>

//KDE
#include <kdesu/su.h>
#include <kmenu.h>
#include <kmessagebox.h>
#include <kmountpoint.h>
#include <kprogressdialog.h>

//KGRUBEditor
#include "core/path.h"
#include "core/root.h"

void HardDiskThread::run()
{
	KDESu::SuProcess install( QByteArray( "root" ), QByteArray().append( installCommand ).append( " " ).append( m_location ) );
	emit exitCode( install.exec( Core::Root::Password.toLocal8Bit() ) );
}

void RescueFloppyThread::run()
{
	KDESu::SuProcess rootProcess( QByteArray( "root" ) );

	emit currentProcess( i18nc( "@info", "Formatting Floppy..." ) );
	emit currentStep( 20 );
	rootProcess.setCommand( QByteArray( "mke2fs /dev/fd0" ) );
	if ( rootProcess.exec( Core::Root::Password.toLocal8Bit() ) != 0 )
	{
		emit exitCode( -1 );
		return;
	}

	emit currentProcess( i18nc( "@info", "Creating Mountpoint..." ) );
	emit currentStep( 40 );
	if ( !QFile::exists( mountpoint ) && !QDir().mkpath( mountpoint ) )
	{
		emit exitCode( -2 );
		return;
	}

	emit currentProcess( i18nc( "@info", "Mounting Floppy..." ) );
	emit currentStep( 60 );
	rootProcess.setCommand( QByteArray( "mount -t ext2 /dev/fd0 " ).append( mountpoint ) );
	if ( !KMountPoint::currentMountPoints().findByDevice( "/dev/fd0" ) && rootProcess.exec( Core::Root::Password.toLocal8Bit() ) != 0 )
	{
		emit exitCode( -3 );
		return;
	}

	emit currentProcess( i18nc( "@info", "Installing GRUB..." ) );
	emit currentStep( 80 );
	rootProcess.setCommand( QByteArray().append( installCommand ).append( " --root-directory=" ).append( mountpoint ).append( " /dev/fd0" ) );
	if ( rootProcess.exec( Core::Root::Password.toLocal8Bit() ) != 0 )
	{
		emit exitCode( -4 );
		return;
	}

	if ( !m_menuFile.isEmpty() )
	{
		emit currentProcess( i18nc( "@info", "Copying Menu File..." ) );
		emit currentStep( 100 );
		rootProcess.setCommand( QByteArray( "cp " ).append( m_menuFile ).append( " " ).append( mountpoint ).append( "/boot/grub/menu.lst" ) );
		if ( rootProcess.exec( Core::Root::Password.toLocal8Bit() ) != 0 )
		{
			emit exitCode( -5 );
			return;
		}
	}

	emit exitCode( 0 );
}

InstallAssistant::InstallAssistant( QWidget *parent ) : KAssistantDialog( parent )
{
//SETUP UI
	setCaption( i18nc( "@window:title", "Install Assistant" ) );
	setWindowIcon( KIcon( "drive-harddisk" ) );
	setAttribute( Qt::WA_DeleteOnClose );
	showButton( KDialog::Help, false );
	setInitialSize( QSize( 600, 400 ) );

	QWidget *intro = new QWidget( this );
	ui_intro.setupUi( intro );
	addPage( intro, i18nc( "@title", "Introduction" ) );
	QWidget *choice = new QWidget( this );
	ui_choice.setupUi( choice );
	choicePage = addPage( choice, i18nc( "@title", "Select where to install GRUB" ) );
	QWidget *hdd = new QWidget( this );
	ui_hdd.setupUi( hdd );
	hardPage = addPage( hdd, i18nc( "@title", "Hard Disk Installation" ) );
	QWidget *floppy = new QWidget( this );
	ui_floppy.setupUi( floppy );
	floppyPage = addPage( floppy, i18nc( "@title", "Floppy Disk Installation" ) );
//MISC UI SETUP
	ui_choice.label_iconHard->setPixmap( KIcon( "drive-harddisk" ).pixmap( 48, 48 ) );
	ui_choice.label_iconFloppy->setPixmap( KIcon( "media-floppy" ).pixmap( 48, 48 ) );

	ui_hdd.klineedit->setValidator( new QRegExpValidator( QRegExp( "(/dev/[sh]d[a-z]\\d?|hd\\d(,\\d)?|'\\(hd\\d(,\\d)?\\)')" ), this ) );

	menu_deviceSuggestions = new KMenu( i18n( "Hard Disks" ), this );
	menu_partitionSuggestions = new KMenu( i18n( "Partitions" ), this );
	foreach( const GRUB::Misc::Device device, Core::Devices::DeviceList )
	{
		if ( !alreadyExists( device.device() ) )
			menu_deviceSuggestions->addAction( device.device() )->setData( device.grubDevice() );

		menu_partitionSuggestions->addAction( device.partition() + QString( " (" ) + device.mountPoint() + QString( ")" ) )->setData( device.grubPartition() );
	}
	menu_suggestions = new KMenu( this );
	menu_suggestions->addMenu( menu_deviceSuggestions );
	menu_suggestions->addMenu( menu_partitionSuggestions );
	ui_hdd.kpushbutton_suggestions->setMenu( menu_suggestions );
	ui_hdd.kpushbutton_suggestions->setIcon( KIcon( "tools-wizard" ) );
//SETUP CONNECTIONS
	setupConnections();
}

void InstallAssistant::setupConnections()
{
	connect( menu_suggestions, SIGNAL( triggered( QAction * ) ), SLOT( suggestionTriggered( QAction * ) ) );
	connect( ui_floppy.checkBox, SIGNAL( toggled( bool ) ), ui_floppy.kurlrequester, SLOT( setEnabled( bool ) ) );
}
bool InstallAssistant::alreadyExists( const QString &device )
{
	foreach( QAction *action, menu_deviceSuggestions->actions() )
		if ( action->text() == device )
			return true;
	return false;
}
void InstallAssistant::clearMountPoint()
{
	if ( QFile::exists( mountpoint ) )
	{
		if ( KMountPoint::currentMountPoints().findByDevice( "/dev/fd0" ) )
		{
			KDESu::SuProcess rootProcess( QByteArray( "root" ), QByteArray( "umount " ).append( mountpoint ) );
			rootProcess.exec( Core::Root::Password.toLocal8Bit() );
		}
		QDir().rmpath( mountpoint );
	}
}

void InstallAssistant::suggestionTriggered( QAction *action )
{
	ui_hdd.klineedit->setText( action->data().toString().remove( "(" ).remove( ")" ) );
}

void InstallAssistant::hddResult( int exitCode )
{
	switch( exitCode )
	{
		case -2:
			KMessageBox::error( this, i18nc( "@info", "<command>%1</command> could not be started and the operation was aborted.", installCommand ) );
		break;
		case -1:
			KMessageBox::error( this, i18nc( "@info", "<command>%1</command> crashed and the operation was aborted.", installCommand ) );
		break;
		case 0:
			KMessageBox::information( this, i18nc( "@info", "The process was successfully completed." ) );
		break;
		default:
			KMessageBox::sorry( this, i18nc( "@info", "<command>%1</command> returned unknown exit code %2. An error could have occured.", installCommand, exitCode ) );
		break;
	}

	exitCode == 0 ? accept() : reject();
}

void InstallAssistant::floppyResult( int exitCode )
{
	switch( exitCode )
	{
		case -1:
			KMessageBox::error( this, i18nc( "@info", "The floppy formatting process failed." ) );
		break;
		case -2:
			KMessageBox::error( this, i18nc( "@info", "Unable to create temporary mountpoint." ) );
		break;
		case -3:
			KMessageBox::error( this, i18nc( "@info", "The floppy mounting process failed." ) );
			clearMountPoint();
		break;
		case -4:
			KMessageBox::error( this, i18nc( "@info", "The installation process failed." ) );
			clearMountPoint();
		break;
		case -5:
			KMessageBox::error( this, i18nc( "@info", "Copying the configuration file failed." ) );
			clearMountPoint();
		break;
		case 0:
			KMessageBox::information( this, i18nc( "@info", "The process was successfully completed." ) );
			clearMountPoint();
		break;
	}

	exitCode == 0 ? accept() : reject();
}
void InstallAssistant::setProgressLabelText( const QString &text )
{
	if ( progressDialog )
		progressDialog->setLabelText( text );
}
void InstallAssistant::setProgressValue( int value )
{
	if ( progressDialog )
		progressDialog->progressBar()->setValue( value );
}

void InstallAssistant::slotButtonClicked( int button )
{
	//Next
	if ( button == KDialog::User2 )
	{
		if ( currentPage() == choicePage )
		{
			setAppropriate( hardPage, ui_choice.radioButton_hard->isChecked() );
			setAppropriate( floppyPage, ui_choice.radioButton_floppy->isChecked() );
		}
		KDialog::slotButtonClicked( button );
	}
	//Finish
	else if ( button == KDialog::User1 )
	{
		//hard disk
		if ( currentPage() == hardPage )
		{
			if ( !ui_hdd.klineedit->hasAcceptableInput() )
			{
				KMessageBox::sorry( this, i18nc( "@info", "<para>Please enter a valid installation location.</para><para>If you are unsure what to enter, use the <interface>Suggestions</interface> list.</para>" ) );
				return;
			}
			else
			{
				if ( !Core::Root::requestPassword( this ) )
					return;

				HardDiskThread install( ui_hdd.klineedit->text(), this );
				connect( &install, SIGNAL( exitCode( int ) ), SLOT( hddResult( int ) ) );
				install.start();
				install.wait();
			}
		}
		else if ( currentPage() == floppyPage )
		{
			RescueFloppyThread install;
			if ( ui_floppy.checkBox->isChecked() && !ui_floppy.kurlrequester->url().isEmpty() )
				install.setMenuFile( ui_floppy.kurlrequester->url().path() );

			progressDialog = new KProgressDialog( this, i18nc( "@window:title", "Creating Rescue Floppy..." ), i18nc( "@label", "Starting..." ) );
			progressDialog->setAutoClose( false );
			progressDialog->setAllowCancel( false );
			progressDialog->progressBar()->setMaximum( 100 );

			connect( &install, SIGNAL( currentProcess( const QString & ) ), SLOT( setProgressLabelText( const QString & ) ) );
			connect( &install, SIGNAL( currentStep( int ) ), SLOT( setProgressValue( int ) ) );
			connect( &install, SIGNAL( finished() ), progressDialog, SLOT( close() ) );
			connect( &install, SIGNAL( exitCode( int ) ), SLOT( floppyResult( int ) ) );
			install.start();
			progressDialog->exec();

			delete progressDialog;
			progressDialog = 0;
		}
	}
	else
		KDialog::slotButtonClicked( button );
}

#include "install.moc"
