/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

#ifndef FILE_INPUT_OUTPUT_H
#define FILE_INPUT_OUTPUT_H

//KDE
class KUrl;

//KGRUBEditor
#include "datatypes.h"

namespace Core
{
	namespace FileIO
	{
		/**
		* Reads input from the menulst file (which by default is /boot/grub/menu.lst)
		* The contents are saved to the settings and entries containers.
		*
		* @param menulst The GRUB's configuration file. Needed to extract GRUB Entry and Settings information. Usually this is /boot/grub/menu.lst.
		* @param settings A class which contains all GRUB Settings which were detected. Used to extract information from menulst.
		* @param entries A vector containing all GRUB entries. Used to extract information from menulst.
		* @param parent The parent window. Used to center the error dialog (if any) above them.
		*
		* @return whether the operation was successfully succeeded.
		*/
		bool readMenulst( const KUrl &menulst, GRUB::ConfigFile::Settings *const settings, QVector<GRUB::ConfigFile::Entry> *const entries, QWidget *parent );
		/**
		* Reads input from the devicemap file (which by default is /boot/grub/device.map)
		* The contents are saved to the settings and entries containers.
		*
		* @param devicemap The GRUB's map file. In conjuction with mtab, it retrieves  information about how GRUB names each device. Usually this is /boot/grub/device.map.
		* @param devices A class which contains all GRUB Device Naming information found in devicemap.
		* @param parent The parent window. Used to center the error dialog (if any) above them.
		*
		* @return whether the operation was successfully succeeded.
		*/
		bool readDevicemap( const KUrl &devicemap, QVector<GRUB::Misc::Device> *const devices, QWidget *parent );
		/**
		* Reads the UUIDs of all devices found using /dev/disk/by-uuid/
		* (Unix-only way).
		*
		* @param devices A class which contains all GRUB Device Naming information found in devicemap.
		*/
		void readUuids( QVector<GRUB::Misc::Device> *const devices );
		/**
		* Makes sure that the file we want to read from is actually readable.
		*
		* In case it is a remote file and is not readable, we can't do anything
		* and an empty KUrl is returned.
		*
		* If however it is a local file (and not readable) it tries to create
		* a temporary copy using root permissions. If the user refuses to enter
		* a password or the process fails, an empty KUrl is returned.
		*
		* @param file The file we want to read from.
		* @param parent Needed for some NetAccess functions (eg. KIO::NetAccess::exists)
		*
		* @return a valid KUrl if the file is readable; an empty one if we can't read
		*         from it.
		*/
		KUrl readFile( const KUrl &file, QWidget *parent );
		/**
		* Reads input from the 2 files specified and extracts information. File errors are handled through message boxes and debug messages.
		* Supports Network access using KIO::NetAccess.
		*
		* For parameters see:
		* @see readMenulst
		* @see readDevicemap
		* @see readUuids
		* @see readFile
		*/
		void readFileInput( const KUrl &menulst, const KUrl &devicemap, GRUB::ConfigFile::Settings *const settings, QVector<GRUB::ConfigFile::Entry> *const entries, QVector<GRUB::Misc::Device> *const devices, QWidget *parent );

		/**
		* Writes information found in the settings and items parameters to the file menulst.
		* 
		* @param menulst The GRUB's configuration file. Needed to write GRUB Entry and Settings information. Usually this is /boot/grub/menu.lst.
		* @param settings A class which contains all GRUB Settings which were detected. Used to write information to menulst.
		* @param entries A vector containing all GRUB entries. Used to write information to menulst.
		* @param parent The QWidget parent window.
		*/
		bool writeMenulst( const KUrl &menulst, const GRUB::ConfigFile::Settings &settings, const QVector<GRUB::ConfigFile::Entry> &entries, QWidget *parent );
		/**
		* Makes sure that the file we want to write to is actually writable.
		*
		* In case it is a remote file, we try to upload a temporary copy.
		*
		* If it is a local file (and not writable) it creates a temporary
		* copy, which is afterwards copied to the original using root permissions. 
		* If the user refuses to enter a password or the process fails, false is returned.
		*
		* Uses writeMenulst() to write the contents to a file (either the temporary or
		* the original if it is writable).
		*
		* For the parameters check writeMenulst().
		*
		* @return whether the data were successfully written to the file.
		*/
		bool writeFileOutput( const KUrl &menulst, const GRUB::ConfigFile::Settings &settings, const QVector<GRUB::ConfigFile::Entry> &entries, QWidget *parent );

		/**
		* Opens file and returns the contents.
		*
		* @param file The file to view.
		* @param parent See readFile()
		*
		* @return the contents of file.
		*
		* @see readFile()
		*/
		QString view( const KUrl &file, QWidget *parent );
	}
}

#endif
