/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

//Own
#include "fileio.h"

//Qt
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qtextstream.h>
#include <qvector.h>

//KDE
#include <kdebug.h>
#include <kdesu/su.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kmountpoint.h>
#include <kprocess.h>
#include <ksavefile.h>
#include <ktemporaryfile.h>
#include <kurl.h>

//KGRUBEditor
#include "root.h"

namespace Core
{
	namespace FileIO
	{
		bool readMenulst( const KUrl &menulst, GRUB::ConfigFile::Settings *const settings, QVector<GRUB::ConfigFile::Entry> *const entries, QWidget *parent )
		{
			settings->clear();
			entries->clear();

			kDebug() << "Reading input from" << menulst;

			QFile fileMenulst( menulst.path() );
			if ( !fileMenulst.open( QIODevice::ReadOnly ) )
			{
				kWarning() << fileMenulst.errorString();
				KMessageBox::error( parent, fileMenulst.errorString() );
				return false;
			}
			kDebug() << "Successfully opened file for reading";

			QRegExp separator( "(\\s+|=|\\s+=\\s+)" );
			GRUB::Misc::Automagic tmp_automagic;
			QString line;
			QTextStream stream( &fileMenulst );
			while ( !stream.atEnd() )
			{
				line = stream.readLine().trimmed();
				//Comments
				if ( line.startsWith( "### BEGIN AUTOMAGIC KERNELS LIST", Qt::CaseInsensitive ) )
				{
					tmp_automagic.setFirstEntry( entries->size() );
					tmp_automagic.appendComment( line );
					continue;
				}
				if ( line.startsWith( "### END DEBIAN AUTOMAGIC KERNELS LIST", Qt::CaseInsensitive ) )
				{
					if ( tmp_automagic.firstEntry() != entries->size() )
					{
						tmp_automagic.appendComment( line );
						tmp_automagic.setLastEntry( entries->size() - 1 );
						settings->setAutomagic( tmp_automagic );
					}
					continue;
				}
				if ( line.startsWith( "#" ) )
				{
					if ( !tmp_automagic.isEmpty() && ( !tmp_automagic.comments().last().startsWith( "## ## End Default Options ##", Qt::CaseInsensitive ) || !tmp_automagic.comments().last().startsWith( "### END DEBIAN AUTOMAGIC KERNELS LIST", Qt::CaseInsensitive ) ) )
						tmp_automagic.appendComment( line );
					continue;
				}
				if ( line.isEmpty() )
				{
					if ( !tmp_automagic.isEmpty() && ( !tmp_automagic.comments().last().startsWith( "## ## End Default Options ##", Qt::CaseInsensitive ) || !tmp_automagic.comments().last().startsWith( "### END DEBIAN AUTOMAGIC KERNELS LIST", Qt::CaseInsensitive ) ) )
						tmp_automagic.appendComment( line );
					continue;
				}
				//*General* settings
				if ( line.startsWith( "splashimage", Qt::CaseInsensitive ) )
				{
					settings->setSplashImage( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "gfxmenu", Qt::CaseInsensitive ) )
				{
					settings->setGfxMenu( line.section( separator, 1 ) );
					continue;
				}
				if (line.startsWith( "default", Qt::CaseInsensitive ) )
				{
					( line.section( separator, 1 ).toLower() == "saved" ? settings->setDefault( -2 ) : settings->setDefault( line.section( separator, 1, 1 ).toInt() ) );
					continue;
				}
				if (line.startsWith( "fallback", Qt::CaseInsensitive ) )
				{
					settings->setFallback( line.section( separator, 1, 1 ).toInt() );
					continue;
				}
				if ( line.startsWith( "timeout", Qt::CaseInsensitive ) )
				{
					settings->setTimeout( line.section( separator, 1 ).toInt() );
					continue;
				}
				if ( line.startsWith( "hiddenmenu", Qt::CaseInsensitive ) )
				{
					settings->setHiddenMenu( true );
					continue;
				}
				//*Shared* settings (can be both *General* and *Entry*)
				if ( line.startsWith( "map", Qt::CaseInsensitive ) )
				{
					if ( entries->isEmpty()  )
						settings->addMap( GRUB::ComplexCommand::Map( line.section( separator, 1 ) ) );
					else
						entries->last().addMap( GRUB::ComplexCommand::Map( line.section( separator, 1 ) ) );
					continue;
				}
				if ( line.startsWith( "color", Qt::CaseInsensitive ) )
				{
					if ( entries->isEmpty()  )
						settings->setColor( line.section( separator, 1 ) );
					else
						entries->last().setColor( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "password", Qt::CaseInsensitive ) )
				{
					if ( entries->isEmpty()  )
						settings->setPassword( line.section( separator, 1 ) );
					else
						entries->last().setPassword( line.section( separator, 1 ) );
					continue;
				}
				//*Entry* settings
				if ( line.startsWith( "title", Qt::CaseInsensitive ) )
				{
					GRUB::ConfigFile::Entry tmp_entry;
					entries->append( tmp_entry );

					entries->last().setTitle( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "lock", Qt::CaseInsensitive ) && !entries->last().title().isEmpty() )
				{
					entries->last().setLock( true );
					continue;
				}
				if ( line.startsWith( "root", Qt::CaseInsensitive ) && !entries->last().title().isEmpty() )
				{
					entries->last().setRoot( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "kernel", Qt::CaseInsensitive ) && !entries->last().title().isEmpty() )
				{
					entries->last().setKernel( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "initrd", Qt::CaseInsensitive ) && !entries->last().title().isEmpty() )
				{
					entries->last().setInitrd( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "chainloader", Qt::CaseInsensitive ) && !entries->last().title().isEmpty() )
				{
					entries->last().setChainLoader( line.section( separator, 1 ) );
					continue;
				}
				if ( line.startsWith( "savedefault", Qt::CaseInsensitive ) && !entries->last().title().isEmpty() )
				{
					entries->last().setSaveDefault( true );
					continue;
				}
				if ( line.startsWith( "makeactive", Qt::CaseInsensitive ) && !entries->last().title().isEmpty() )
				{
					entries->last().setMakeActive( true );
					continue;
				}
			}
			kDebug() << "Data extraction successful. Closing file";
			fileMenulst.close();

			return true;
		}
		bool readDevicemap( const KUrl &devicemap, QVector<GRUB::Misc::Device> *const devices, QWidget *parent )
		{
			devices->clear();

			//Unix specific code
			foreach( const KMountPoint::Ptr mp, KMountPoint::currentMountPoints() )
				if ( mp->mountedFrom().startsWith( "/dev" ) )
					devices->append( GRUB::Misc::Device( mp->mountedFrom().remove( QRegExp( "\\d" ) ), mp->mountedFrom(), mp->mountPoint() ) );

			kDebug() << "Reading input from" << devicemap;

			QFile fileDevicemap( devicemap.path() );
			if ( !fileDevicemap.open( QIODevice::ReadOnly ) )
			{
				kWarning() << fileDevicemap.errorString();
				KMessageBox::error( parent, fileDevicemap.errorString() );
				return false;
			}
			kDebug() << "Successfully opened file for reading";

			QRegExp space( "\\s+" );
			QString line;
			QTextStream stream( &fileDevicemap );
			while ( !stream.atEnd() )
			{
				line = stream.readLine().trimmed();
				for ( int i = 0 ; i < devices->size() ; i++ )
				{
					if ( devices->at( i ).device() == line.section( space, 1, 1 ) )
					{
						(*devices)[ i ].setGrubDevice( line.section( space, 0, 0 ) );
						(*devices)[ i ].setGrubPartition( QString( (*devices)[ i ].grubDevice() ).replace( QString( ")" ), QString( "," ) + QString().setNum( (*devices)[ i ].partition().right( 1 ).toInt() - 1 ) + QString( ")" ) ) );
					}
				}
			}
			kDebug() << "Data extraction successful. Closing file";
			fileDevicemap.close();

			return true;
		}
		void readUuids( QVector<GRUB::Misc::Device> *const devices )
		{
			QRegExp space( "\\s+" );

			KProcess uuid;
			uuid.setProgram( "ls", QStringList() << "-l" << "/dev/disk/by-uuid/" );
			uuid.setOutputChannelMode( KProcess::MergedChannels );
			uuid.start();
			uuid.waitForFinished();

			QString output = uuid.readAllStandardOutput();
			for ( int j = 1 ; j < output.count( "\n" ) ; j++ )
				for ( int i = 0 ; i < devices->size() ; i++ )
					if ( devices->at( i ).partition() == output.section( "\n", j, j ).section( space, 9, 9 ).remove( "../.." ).prepend( "/dev" ) )
						(*devices)[ i ].setUuid( output.section( "\n", j, j ).section( space, 7, 7 ) );
		}
		KUrl readFile( const KUrl &file, QWidget *parent )
		{
			KUrl ret( file );
			KTemporaryFile *tmpFile = 0;
			if ( KIO::NetAccess::exists( file, true, parent ) )
			{
				if ( file.isLocalFile() && !QFileInfo( file.path() ).isReadable() )
				{
					if ( Core::Root::requestPassword( parent ) )
					{
						tmpFile = new KTemporaryFile;
						if ( tmpFile->open() )
						{
							KDESu::SuProcess cat( QByteArray( "root" ), QByteArray( "cat " ).append( file.path() ).append( " > " ).append( tmpFile->fileName() ) );
							cat.exec( Core::Root::Password.toLocal8Bit() );

							ret = tmpFile->fileName();
						}
						else
						{
							kWarning() << tmpFile->errorString();
							KMessageBox::error( parent, tmpFile->errorString() );
							ret = KUrl();
						}
					}
					else
					{
						KMessageBox::sorry( parent, i18nc( "@info", "Operation was aborted. Unable to read from <filename>%1</filename>.", file.path() ) );  
						ret = KUrl();
					}
				}
				else if ( !file.isLocalFile() )
				{
					tmpFile = new KTemporaryFile;
					QString tmpFileName = tmpFile->fileName();
					if ( KIO::NetAccess::download( file, tmpFileName, parent ) )
					{
						ret = tmpFile->fileName();
					}
					else
					{
						kWarning() << KIO::NetAccess::lastErrorString();
						KMessageBox::error( parent, KIO::NetAccess::lastErrorString() );
						ret = KUrl();
					}
				}
			}
			else
			{
				kWarning() << file << "does not exist!";
				KMessageBox::error( parent, i18nc( "@info", "<filename>%1</filename> does not exist!", file.path() ) );
				ret = KUrl();
			}

			if ( tmpFile )
				tmpFile->deleteLater();

			return ret;
		}
		void readFileInput( const KUrl &menulst, const KUrl &devicemap, GRUB::ConfigFile::Settings *const settings, QVector<GRUB::ConfigFile::Entry> *const entries, QVector<GRUB::Misc::Device> *const devices, QWidget *parent )
		{
			KUrl tmpUrl = Core::FileIO::readFile( menulst, parent );
			if ( !tmpUrl.isEmpty() )
			{
				Core::FileIO::readMenulst( tmpUrl, settings, entries, parent );
			}
			tmpUrl = Core::FileIO::readFile( devicemap, parent );
			if ( !devicemap.isEmpty() )
			{
				Core::FileIO::readDevicemap( tmpUrl, devices, parent );
				Core::FileIO::readUuids( devices );
			}
		}

		bool writeMenulst( const KUrl &menulst, const GRUB::ConfigFile::Settings &settings, const QVector<GRUB::ConfigFile::Entry> &entries, QWidget *parent )
		{
			kDebug() << "Writing output to" << menulst;

			KSaveFile fileMenulst( menulst.path() );
			if ( !fileMenulst.open() )
			{
				kWarning() << fileMenulst.errorString();
				KMessageBox::error( parent, fileMenulst.errorString() );
				return false;
			}
			kDebug() << "Successfully opened file for writing";

		//Settings
			QTextStream stream( &fileMenulst );
			if ( !settings.splashImage().isEmpty() )
					stream << "splashimage " << settings.splashImage() << endl;
			if ( !settings.gfxMenu().isEmpty() )
					stream << "gfxmenu " << settings.gfxMenu() << endl;
			if ( settings.hiddenMenu() )
					stream << "hiddenmenu" << endl;
			if ( settings._default() != -1 )
					stream << "default " << ( settings._default() != -2 ? QString().setNum( settings._default() ) : "saved" ) << endl;
			if ( settings.fallback() != -1 )
					stream << "fallback " << settings.fallback() << endl;
			if ( settings.timeout() != -1 )
					stream << "timeout " << settings.timeout() << endl;
			if ( !settings.maps().isEmpty() )
				for ( int i = 0; i < settings.maps().size(); i++ )
					stream << "map " << settings.maps().at( i ) << endl;
			if ( !settings.color().isEmpty() )
					stream << "color " << settings.color() << endl;
			if ( !settings.password().isEmpty() )
					stream << "password " << settings.password() << endl;
			stream << endl;
		//Entries
			for ( int i = 0; i < entries.size(); i++ )
			{
				if ( !settings.automagic().isEmpty() && settings.automagic().firstEntry() == i )
				{
					QStringList comments = settings.automagic().comments();
					comments.removeLast();
					while( comments.last().isEmpty() )
						comments.removeLast();
		
					foreach( const QString &comment, comments )
						stream << comment << endl;
					stream << endl;
				}
		
					stream << "title " << entries.at( i ).title() << endl;
				if ( entries.at( i ).lock() )
					stream << "lock" << endl;
				if ( !entries.at( i ).password().isEmpty() )
					stream << "password " << entries.at( i ).password() << endl;
				if ( !entries.at( i ).root().isEmpty() )
					stream << "root " << entries.at( i ).root() << endl;
				if ( !entries.at( i ).kernel().isEmpty() )
					stream << "kernel " << entries.at( i ).kernel() << endl;
				if ( !entries.at( i ).initrd().isEmpty() )
					stream << "initrd " << entries.at( i ).initrd() << endl;
				if ( !entries.at( i ).maps().isEmpty() )
					for ( int j = 0; j < entries.at( i ).maps().size(); j++ )
						stream << "map " << entries.at( i ).maps().at( j ) << endl;
				if ( !entries.at( i ).color().isEmpty() )
					stream << "color " << entries.at( i ).color() << endl;
				if (!entries.at( i ).chainLoader().isEmpty())
					stream << "chainloader " << entries.at( i ).chainLoader() << endl;
				if ( entries.at( i ).saveDefault() )
					stream << "savedefault" << endl;
				if ( entries.at( i ).makeActive() )
					stream << "makeactive" << endl;
				stream << endl;
		
				if ( !settings.automagic().isEmpty() && settings.automagic().lastEntry() == i )
				{
					stream << settings.automagic().comments().last() << endl;
					stream << endl;
				}
			}

			if ( !fileMenulst.finalize() )
			{
				kWarning() << fileMenulst.errorString();
				KMessageBox::error( parent, fileMenulst.errorString() );
				return false;
			}
			kDebug() << "Successfully wrote and finalized file";

			return true;
		}
		bool writeFileOutput( const KUrl &menulst, const GRUB::ConfigFile::Settings &settings, const QVector<GRUB::ConfigFile::Entry> &entries, QWidget *parent )
		{
			KTemporaryFile *tmpMenulst = 0;
			bool needsRootPassword = menulst.isLocalFile() && !QFileInfo( menulst.path() ).isWritable();

			if ( needsRootPassword || !menulst.isLocalFile() )
			{
				tmpMenulst = new KTemporaryFile;
				if ( !tmpMenulst->open() )
				{
					kWarning() << tmpMenulst->errorString();
					KMessageBox::error( parent, tmpMenulst->errorString() );
					return false;
				}
			}

			writeMenulst( KUrl( tmpMenulst ? tmpMenulst->fileName() : menulst.path() ), settings, entries, parent );

			if ( needsRootPassword )
			{
				if ( Core::Root::requestPassword( parent ) )
				{
					KDESu::SuProcess cp( QByteArray( "root" ), QByteArray( "cp " ).append( tmpMenulst->fileName() ).append( " " ).append( menulst.path() ) );
					cp.exec( Core::Root::Password.toLocal8Bit() );
				}
				else
				{
					KMessageBox::sorry( parent, i18nc( "@info", "Operation was aborted. Unable to write to %1.", menulst.path() ) );                                                                                                             
					return false;
				}
			}
			else if ( !menulst.isLocalFile() )
			{
				if ( !KIO::NetAccess::upload( tmpMenulst->fileName(), menulst, parent ) )
				{
					kWarning() << KIO::NetAccess::lastErrorString();
					KMessageBox::error( parent, KIO::NetAccess::lastErrorString() );
					return false;
				}
				kDebug() << "Successfully uploaded temporary file to" << menulst;
			}

			delete tmpMenulst;
			return true;
		}

		QString view( const KUrl &file, QWidget *parent )
		{
			KUrl tmpUrl = Core::FileIO::readFile( file, parent );
			if ( tmpUrl.isEmpty() )
				return QString();

			kDebug() << "Opening" << tmpUrl << "for viewing";

			QFile _file( tmpUrl.path() );
			if ( !_file.open( QIODevice::ReadOnly ) )
			{
				kWarning() << _file.errorString();
				KMessageBox::error( parent, _file.errorString() );
				return QString();
			}
			kDebug() << "Successfully opened file for viewing";

			return QTextStream( &_file ).readAll();
		}
	}
}
