/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

#ifndef DATA_TYPES_H
#define DATA_TYPES_H

//Qt
#include <qvector.h>
#include <qtextstream.h>
#include <qmap.h>
#include <qstringlist.h>

namespace GRUB
{
	namespace ComplexCommand
	{
		/**
		* @brief A GRUB 'map' command wrapper.
		*
		* Fully supports the GRUB 'map' command syntax:
		*
		* map to_drive from_drive
		*
		* Virtually swaps the drive from_drive to the drive to_drive.
		* So, this example:
		*
		* @code
		* map (hd0) (hd1)
		* map (hd1) (hd0)
		* @endcode
		*
		* Virtually exchanges the order between (hd1) <--> (hd0)
		*
		* All attributes are necessary and thus, considered fundamental.
		*
		* See GRUB's documentation for more info:
		* @see http://www.gnu.org/software/grub/manual/html_node/map.html#map
		*/
		class Map
		{
			friend QTextStream &operator<<( QTextStream &output, const Map &map )
			{
				output << map.m_toDrive << " " << map.m_fromDrive;
				return output;
			}
		public:
		//constructors
			/**
			* Constructs an empty map using clear().
			*
			* @see clear()
			*/
			Map();
			/**
			* Constructs a map initialized with the arguments passed.
			*
			* @param toDrive The To-Drive of this map.
			* @param fromDrive The From-Drive of this map.
			*/
			Map( const QString toDrive, const QString fromDrive );
			/**
			* Constructs a map initialized with drive-strings found in map,
			* after analyzing it.
			*
			* @param map QString passed to resolve().
			*
			* For the analysis process see:
			* @see resolve()
			*/
			Map( const QString map );
		//special functions
			/** 
			* Resets all of the map's attributes to their default values.
			*/
			void clear();
			/**
			* Analyses the QString parameter and extracts information 
			* with which the attributes are filled.
			*
			* @param map QString to be analysed.
			*/
			void resolve( const QString map );
			/**
			* Formats the input into a single line in order to show
			* it to the user through the GUI (more conceivable).
			*
			* @return the result of the above process.
			*/
			const QString result() const;
			/**
			* @return true if no fundamental attribute is set;
			*         otherwise returns false.
			*/
			inline bool isEmpty() const { return ( m_toDrive.isEmpty() && m_fromDrive.isEmpty() ); };
		//get functions
			inline const QString toDrive() const { return m_toDrive; };
			inline const QString fromDrive() const { return m_fromDrive; };
		//set functions
			inline void setToDrive( const QString toDrive ) { m_toDrive = toDrive; };
			inline void setFromDrive( const QString fromDrive ) { m_fromDrive = fromDrive; };
		//reset functions
			inline void clearToDrive() { m_toDrive.clear(); };
			inline void clearFromDrive() { m_fromDrive.clear(); };
		private:
			QString m_toDrive;
			QString m_fromDrive;
		};
	
		/**
		* @brief A wrapper for the GRUB command 'color'.
		*
		* Fully supports the GRUB 'color' command syntax:
		*
		* color normal [highlight]
		*
		* Where:
		* @li normal = [blink-]foreground/background
		* @li highlight = [blink-]foreground/background
		*
		* Background can be:
		* @li black
		* @li blue
		* @li green
		* @li cyan
		* @li red
		* @li magenta
		* @li brown
		* @li light-gray
		*
		* Foreground can be:
		* @li black
		* @li blue
		* @li green
		* @li cyan
		* @li red
		* @li magenta
		* @li brown
		* @li light-gray
		*
		* @li dark-gray
		* @li light-blue
		* @li light-green
		* @li light-cyan
		* @li light-red
		* @li light-magenta
		* @li yellow
		* @li white 
		*
		* You can prefix blink- to foreground if you want a blinking foreground color.
		*
		* Usage:
		* @code
		* # Set default colors.
		* color light-gray/blue black/light-gray
		*
		* # Change the colors.
		* title OS-BS like
		* color magenta/blue black/magenta
		* @endcode
		*
		* Normal is a fundamental attribute (both foreground and background).
		* If these attributes are set to QString() the class is considered
		* to be incomplete and isEmpty() returns true.
		*
		* See GRUB's documentation for more info:
		* @see http://www.gnu.org/software/grub/manual/html_node/color.html#color
		*/
		class Color
		{
			friend QTextStream &operator<<( QTextStream &output, const Color &color )
			{
				output 
				<< ( color.m_blinkNormal ? "blink-" : QString() ) << color.m_normalForeground << ( color.normalIsEnabled() ? "/" : QString() ) << color.m_normalBackground 
				<< " " 
				<< ( color.m_blinkHighlighted ? "blink-" : QString() ) << color.m_highlightedForeground << ( color.highlightedIsEnabled() ? "/" : QString() ) << color.m_highlightedBackground;
				return output;
			}
		private:
			/**
			* A function used to initialize the (static) m_colorMap object. 
			* It should be run only once, when giving the object an initial value.
			*
			* @return a QMap where all GRUB-recognised colors are mapped
			*         to an integer value, starting from 0. The QString
			*         attribute of the map contains the color in plain
			*         form (what GRUB uses/sees).
			*/
			static const QMap<int, QString> mappedColors();
		public:
		//constructors
			/**
			* Constructs an empty color using clear().
			*
			* @see clear()
			*/
			Color();
			/**
			* Constructs a color initialized with the color-attributes
			* found in color, after analyzing it.
			*
			* @param color QString passed to resolve().
			*
			* For the analysis process see:
			* @see resolve()
			*/
			Color( const QString color );
		//special functions
			/** 
			* Resets all of the object's attributes to their default values.
			*/
			void clear();
			/**
			* Analyses the QString parameter and extracts information.
			* If some info are missing (for example highlight or the blink- prefix)
			* sets the corresponding attribute to QString() (or to false if it is a bool).
			*
			* @param color QString to be analysed.
			*/
			void resolve( const QString color );
			/**
			* Formats the input into a single line in order to show
			* it to the user through the GUI (more conceivable).
			*
			* @return the result of the above process.
			*/
			const QString result() const;
			/**
			* Converts the color provided into the corresponding
			* color keyword as defined by the World Wide Web Consortium.
			*
			* Needed in order to automatically resolve the color using
			* QColor::QColor( const QString &name ) which creates a color
			* according to the string passed. This in turn is needed by
			* the GRUBColorPreview widget.
			*
			* It is only used for colors where GRUB separates them with a
			* dash (eg. light-gray) and for colors that do not have a
			* corresponding keyword in the 'list of the recognised color
			* keyword names' (these are magenta, light-magenta and light-red).
			*
			* For the list, see:
			* @see http://www.w3.org/TR/SVG/types.html#ColorKeywords
			*
			* @param color The color to be converted.
			*
			* @return The result of the conversion.
			*/
			const QString previewOptimisedColor( const QString color ) const;
			/**
			* @return true if no fundamental attribute is set;
			*         otherwise returns false.
			*/
			inline bool isEmpty() const { return ( m_normalBackground.isEmpty() && m_normalForeground.isEmpty() ); };
			/**
			* @return true if normal color is enabled;
			*         otherwise returns false.
			*/
			inline bool normalIsEnabled() const { return !( m_normalBackground.isEmpty() && m_normalForeground.isEmpty() ); };
			/**
			* @return true if highlighted color is enabled;
			*         otherwise returns false.
			*/
			inline bool highlightedIsEnabled() const { return !( m_highlightedBackground.isEmpty() && m_highlightedForeground.isEmpty() ); };
		//get functions
			inline bool blinkNormal() const { return m_blinkNormal; };
			inline const QString normalBackground() const { return m_normalBackground; };
			inline const QString normalForeground() const { return m_normalForeground; };
			inline bool blinkHighlighted() const { return m_blinkHighlighted; };
			inline const QString highlightedBackground() const { return m_highlightedBackground; };
			inline const QString highlightedForeground() const { return m_highlightedForeground; };

			inline static const QMap<int, QString> colorMap() { return m_colorMap; };
		//set functions
			inline void setBlinkNormal( const bool blinkNormal ) { m_blinkNormal = blinkNormal; };
			inline void setNormalBackground( const QString normalBackground ) { m_normalBackground = normalBackground; };
			inline void setNormalForeground( const QString normalForeground ) { m_normalForeground = normalForeground; };
			inline void setBlinkHighlighted( const bool blinkHighlighted ) { m_blinkHighlighted = blinkHighlighted; };
			inline void setHighlightedBackground( const QString highlightedBackground ) { m_highlightedBackground = highlightedBackground; };
			inline void setHighlightedForeground( const QString highlightedForeground ) { m_highlightedForeground = highlightedForeground; };
		//reset functions
			inline void clearBlinkNormal() { m_blinkNormal = false; };
			inline void clearNormalBackground() { m_normalBackground.clear(); };
			inline void clearNormalForeground() { m_normalForeground.clear(); };
			inline void clearBlinkHighlighted() { m_blinkHighlighted = false; };
			inline void clearHighlightedBackground() { m_highlightedBackground.clear(); };
			inline void clearHighlightedForeground() { m_highlightedForeground.clear(); };
		private:
			bool m_blinkNormal;
			QString m_normalBackground;
			QString m_normalForeground;
			bool m_blinkHighlighted;
			QString m_highlightedBackground;
			QString m_highlightedForeground;

			static QMap<int, QString> m_colorMap;
		};
	
		/**
		* @brief A GRUB 'password' command wrapper.
		*
		* Fully supports the GRUB 'password' command syntax:
		*
		* password [`--md5'] passwd [new-config-file]
		*
		* Where:
		* @li --md5 = passwd is encrypted with md5crypt.
		* @li passwd = the password string
		* @li new-config-file = the new config file that will be loaded after the user enters the passwd.
		*
		* Fundamental attribute is passwd. If these attribute is set to QString()
		* the class is considered to be incomplete and isEmpty() returns true.
		*
		* See GRUB's documentation for more info:
		* @see http://www.gnu.org/software/grub/manual/html_node/password.html#password
		*/
		class Password
		{
			friend QTextStream &operator<<( QTextStream &output, const Password &password )
			{
				output << ( password.m_md5crypted ? "--md5 " : QString() ) << password.m_password << ( password.m_configFile.isEmpty() ? QString() : QString( " " ) + password.m_configFile );
				return output;
			}
		public:
		//constructors
			/**
			* Constructs an empty password using clear().
			*
			* @see clear()
			*/
			Password();
			/**
			* Constructs a password initialized with the arguments passed.
			*
			* @param md5crypted Whether this password is encrypted using MD5.
			* @param password The actual password.
			* @param configFile The config file that will be loaded.
			*/
			Password( const bool md5crypted, const QString password, const QString configFile );
			/**
			* Constructs a password initialized with the password-attributes
			* found in password, after analyzing it.
			*
			* @param password QString passed to resolve().
			*
			* For the analysis process see:
			* @see resolve()
			*/
			Password( const QString password );
		//help functions
			/** 
			* Resets all of the object's attributes to their default values.
			*/
			void clear();
			/**
			* Analyses the QString parameter and extracts information.
			* If some info are missing (for example new-config-file or the --md5 switch)
			* sets the corresponding attribute to QString() (or to false if it is a bool).
			*
			* @param password QString to be analysed.
			*/
			void resolve( const QString password );
			/**
			* Formats the input into a single line in order to show
			* it to the user through the GUI (more conceivable).
			*
			* @return the result of the above process.
			*/
			const QString result() const;
			/**
			* @return true if no fundamental attribute is set;
			*         otherwise returns false.
			*/
			inline bool isEmpty() const { return m_password.isEmpty(); };
		//get functions
			inline bool md5crypted() const { return m_md5crypted; };
			inline const QString password() const { return m_password; };
			inline const QString configFile() const { return m_configFile; };
		//set functions
			inline void setMd5crypted( const bool md5crypted ) { m_md5crypted = md5crypted; };
			inline void setPassword( const QString password ) { m_password = password; };
			inline void setConfigFile( const QString configFile ) { m_configFile = configFile; };
		//reset functions
			inline void clearMd5crypted() { m_md5crypted = false; };
			inline void clearPassword() { m_password.clear(); };
			inline void clearConfigFile() { m_configFile.clear(); };
		private:
			bool m_md5crypted;
			QString m_password;
			QString m_configFile;
		};
	
		/**
		* @brief A GRUB 'kernel' command wrapper.
		*
		* Doesn't fully support the GRUB 'kernel' command syntax, which is:
		*
		* kernel [`--type=type'] [`--no-mem-option'] file ...
		*
		* since it doesn't take into consideration the optional options.
		*
		* Fundamental attribute is file. If these attribute is set to QString()
		* the class is considered to be incomplete and isEmpty() returns true.
		*
		* See GRUB's documentation for more info:
		* @see http://www.gnu.org/software/grub/manual/html_node/kernel.html#kernel
		*/
		class Kernel
		{
			friend QTextStream &operator<<( QTextStream &output, const Kernel &kernel )
			{
				output << kernel.m_kernel << ( kernel.m_arguments.isEmpty() ? QString() : QString( " " ) + kernel.m_arguments );
				return output;
			}
		public:
		//constructors
			/**
			* Constructs an empty password using clear().
			*
			* @see clear()
			*/
			Kernel();
			/**
			* Constructs a kernel initialized with the arguments passed.
			*
			* @param kernel The kernel file.
			* @param arguments The arguments to pass to the kernel.
			*/
			Kernel( const QString kernel, const QString arguments );
			/**
			* Constructs a kernel initialized with the kernel-strings
			* found in kernel, after analyzing it.
			*
			* @param kernel QString passed to resolve().
			*
			* For the analysis process see:
			* @see resolve()
			*/
			Kernel( const QString kernel );
		//help functions
			/** 
			* Resets all of the object's attributes to their default values.
			*/
			void clear();
			/**
			* Analyses the QString parameter and extracts information.
			* If kernel arguments are missing sets the corresponding 
			* attribute to QString().
			*
			* @param kernel QString to be analysed.
			*/
			void resolve( const QString kernel );
			/**
			* Formats the input into a single line in order to show
			* it to the user through the GUI (more conceivable).
			*
			* @return the result of the above process.
			*/
			const QString result() const;
			/**
			* @return true if no fundamental attribute is set;
			*         otherwise returns false.
			*/
			inline bool isEmpty() const { return m_kernel.isEmpty(); };
		//get functions
			inline const QString kernel() const { return m_kernel; };
			inline const QString arguments() const { return m_arguments; };
		//set functions
			inline void setKernel( const QString kernel ) { m_kernel = kernel; };
			inline void setArguments( const QString arguments ) { m_arguments = arguments; };
		//reset functions
			inline void clearKernel() { m_kernel.clear(); };
			inline void clearArguments() { m_arguments.clear(); };
		private:
			QString m_kernel;
			QString m_arguments;
		};
	}

	namespace Misc
	{
		/**
		* @brief A class which contains all Device-related info collected.
		*
		* Consists of:
		* @li mountpoint
		* @li partition name
		* @li partition name (in GRUB device naming scheme)
		* @li device name
		* @li device name (in GRUB device naming scheme)
		* @li uuid
		*/
		class Device
		{
		public:
		//constructors
			/**
			* Constructs an empty device using clear().
			*
			* @see clear()
			*/
			Device();
			/**
			* Constructs a device initialized with the arguments passed.
			*
			* @param device The device's name.
			* @param partition The partition's name.
			* @param mountpoint The mountpoint of the partition.
			*/
			Device( const QString device, const QString partition, const QString mountpoint );
		//help functions
			/** 
			* Resets all of the object's attributes to their default values.
			*/
			void clear();
		//get functions
			inline const QString device() const { return m_device; };
			inline const QString partition() const { return m_partition; };
			inline const QString grubDevice() const { return m_grubDevice; };
			inline const QString grubPartition() const { return m_grubPartition; };
			inline const QString mountPoint() const { return m_mountPoint; };
			inline const QString uuid() const { return m_uuid; };
		//set functions
			inline void setDevice( const QString device ) { m_device = device; };
			inline void setPartition( const QString partition ) { m_partition = partition; };
			inline void setGrubDevice( const QString grubDevice ) { m_grubDevice = grubDevice; };
			inline void setGrubPartition( const QString grubPartition ) { m_grubPartition = grubPartition; };
			inline void setMountPoint( const QString mountPoint ) { m_mountPoint = mountPoint; };
			inline void setUuid( const QString uuid ) { m_uuid = uuid; };
		//reset functions
			inline void clearDevice() { m_device.clear(); };
			inline void clearPartition() { m_partition.clear(); };
			inline void clearGrubDevice() { m_grubDevice.clear(); };
			inline void clearGrubPartition() { m_grubPartition.clear(); };
			inline void clearMountPoint() { m_mountPoint.clear(); };
			inline void clearUuid() { m_uuid.clear(); };
		private:
			QString m_device;
			QString m_partition;
			QString m_grubDevice;
			QString m_grubPartition;
			QString m_mountPoint;
			QString m_uuid;
		};

		/**
		* @brief A wrapper for the AUTOMAGIC KERNELS LIST.
		*
		* It is used to store everything concerning the AUTOMAGIC
		* KERNELS LIST which is to be found in Debian[-derived] distros
		* eg. Ubuntu.
		*
		* Such distro-specific code should not be added but an exception
		* was done due to many users asking for it.
		*
		* Can handle comments and provides 2 properties for fast access to
		* the first and last entry which are protected from AUTOMAGIC.
		* All of the attributes are fundamental -@see isEmpty()-.
		*/
		class Automagic
		{
		public:
		//constructor
			/**
			* Constructs an empty device using clear().
			*
			* @see clear()
			*/
			Automagic();
		//help functions
			/** 
			* Resets all of the object's attributes to their default values.
			*/
			void clear();
			/**
			* @return true if no fundamental attribute is set;
			*         otherwise returns false.
			*/
			inline bool isEmpty() const { return m_comments.isEmpty() && m_firstEntry == -1 && m_lastEntry == -1; };
		//get functions
			inline const QStringList comments() const { return m_comments; };
			inline int firstEntry() const { return m_firstEntry; };
			inline int lastEntry() const { return m_lastEntry; };
		//set functions
			inline void setComments( const QStringList comments ) { m_comments = comments; };
			inline void setFirstEntry( const int firstEntry ) { m_firstEntry = firstEntry; };
			inline void setLastEntry( const int lastEntry ) { m_lastEntry = lastEntry; };
		//reset functions
			inline void clearComments() { m_comments.clear(); };
			inline void clearFirstEntry() { m_firstEntry = -1; };
			inline void clearLastEntry() { m_lastEntry = -1; };
		//special functions
			/**
			* Appends a new comment to the list of comments.
			*
			* @param comment The comment to append.
			*/
			inline void appendComment( const QString &comment ) { m_comments.append( comment ); };
		private:
			QStringList m_comments;
			int m_firstEntry;
			int m_lastEntry;
		};
	}

	namespace ConfigFile
	{
		/**
		* @brief A GRUB Entry wrapper.
		*
		* Supports the following GRUB Commands:
		*
		* @li title
		* @li lock
		* @li password
		* @li root
		* @li kernel
		* @li initrd
		* @li color
		* @li maps
		* @li chainloader
		* @li savedefault
		* @li makeactive
		*
		* It also has the following helper attributes which are
		* not per entry attributes but general settings. They are
		* provided for conveinience.
		*
		* @li default
		* @li fallback
		* @li automagic
		*/
		class Entry
		{
		public:
		//constructors
			/**
			* Constructs an empty entry using clear().
			*
			* @see clear()
			*/
			Entry();
		//help functions
			/** 
			* Resets all of the object's attributes to their default values.
			*/
			void clear();
		//get functions
			inline const QString title() const { return m_title; };
			inline bool lock() const { return m_lock; };
			inline const GRUB::ComplexCommand::Password password() const { return m_password; };
			inline const QString root() const { return m_root; };
			inline const GRUB::ComplexCommand::Kernel kernel() const { return m_kernel; };
			inline const QString initrd() const { return m_initrd; };
			inline const GRUB::ComplexCommand::Color color() const { return m_color; };
			inline const QVector<GRUB::ComplexCommand::Map> maps() const { return m_maps; };
			inline const QString chainLoader() const { return m_chainLoader; };
			inline bool saveDefault() const { return m_saveDefault; };
			inline bool makeActive() const { return m_makeActive; };
		//set functions
			inline void setTitle( const QString title ) { m_title = title; };
			inline void setLock( const bool lock ) { m_lock = lock; };
			inline void setPassword( const GRUB::ComplexCommand::Password &password ) { m_password = password; };
			inline void setPassword( const QString password ) { m_password.resolve( password ); };
			inline void setRoot( const QString root ) { m_root = root; };
			inline void setKernel( const GRUB::ComplexCommand::Kernel &kernel ) { m_kernel = kernel; };
			inline void setKernel( const QString kernel ) { m_kernel.resolve( kernel ); };
			inline void setInitrd( const QString initrd ) { m_initrd = initrd; };
			inline void setColor( const GRUB::ComplexCommand::Color &color ) { m_color = color; };
			inline void setColor( const QString color ) { m_color.resolve( color ); };
			inline void setMaps( const QVector<GRUB::ComplexCommand::Map> &maps ) { m_maps = maps; };
			inline void setChainLoader( const QString chainLoader ) { m_chainLoader = chainLoader; };
			inline void setSaveDefault( const bool saveDefault ) { m_saveDefault = saveDefault; };
			inline void setMakeActive( const bool makeActive ) { m_makeActive = makeActive; };
		//reset functions
			inline void clearTitle() { m_title.clear(); };
			inline void clearLock() { m_lock = false; };
			inline void clearPassword() { m_password.clear(); };
			inline void clearRoot() { m_root.clear(); };
			inline void clearKernel() { m_kernel.clear(); };
			inline void clearInitrd() { m_initrd.clear(); };
			inline void clearColor() { m_color.clear(); };
			inline void clearMaps() { m_maps.clear(); };
			inline void clearChainLoader() { m_chainLoader.clear(); };
			inline void clearSaveDefault() { m_saveDefault = false; };
			inline void clearMakeActive() { m_makeActive = false; };
		//special functions
			/**
			* Appends a new map to the end of the list of maps.
			*
			* @param map The map to be appended.
			*/
			inline void addMap( const GRUB::ComplexCommand::Map &map ) { m_maps.append( map ); };
			/**
			* Replaces the map positioned at index with map.
			*
			* @param index The position of the map to be replaced in the vector
			*              that holds all maps.
			* @param map The map that will replace the above mentioned map.
			*/
			inline void replaceMap( const int index, const GRUB::ComplexCommand::Map &map ) { m_maps.replace( index, map ); };
			/**
			* Copies the map positioned at index to the end of the list.
			*
			* @param index The position of the map that will be copied.
			*/
			inline void copyMap( int index ) { m_maps.append( m_maps.at( index ) ); };
			/**
			* Removes the map positioned at index.
			*
			* @param index The position of the map that will be removed.
			*/
			inline void removeMap( int index ) { m_maps.remove( index ); };
		private:
			QString m_title;
			bool m_lock;
			GRUB::ComplexCommand::Password m_password;
			QString m_root;
			GRUB::ComplexCommand::Kernel m_kernel;
			QString m_initrd;
			GRUB::ComplexCommand::Color m_color;
			QVector<GRUB::ComplexCommand::Map> m_maps;
			QString m_chainLoader;
			bool m_saveDefault;
			bool m_makeActive;
		};

		/**
		* @brief A class which holds all GRUB Settings detected.
		*
		* Supports the following GRUB Settings:
		*
		* @li splash image
		* @li gfxmenu
		* @li timeout
		* @li default
		* @li fallback
		* @li hidden menu
		* @li maps
		* @li color
		* @li password
		*
		* @li Debian-agnostic: AUTOMAGIC KERNELS LIST
		*/
		class Settings
		{
		public:
		//constructor
			/**
			* Constructs an empty settings-container using clear().
			*
			* @see clear()
			*/
			Settings();
		//help functions
			/** 
			* Resets all of the object's attributes to their default values.
			*/
			void clear();
		//get functions
			inline const QString splashImage() const { return m_splashImage; };
			inline const QString gfxMenu() const { return m_gfxMenu; };
			inline int timeout() const { return m_timeout; };
			inline int _default() const { return m_default; };
			inline int fallback() const {return m_fallback; };
			inline bool hiddenMenu() const { return m_hiddenMenu; };
			inline const QVector<GRUB::ComplexCommand::Map> maps() const { return m_maps; };
			inline const GRUB::ComplexCommand::Color color() const { return m_color; };
			inline const GRUB::ComplexCommand::Password password() const { return m_password; };

			inline const GRUB::Misc::Automagic automagic() const { return m_automagic; };
		//set functions
			inline void setSplashImage( const QString splashImage ) { m_splashImage = splashImage; };
			inline void setGfxMenu( const QString gfxMenu ) { m_gfxMenu = gfxMenu; };
			inline void setTimeout( const int timeout ) { m_timeout = timeout; };
			inline void setDefault( const int _default ) { m_default = _default; };
			inline void setFallback( const int fallback ) { m_fallback = fallback; };
			inline void setHiddenMenu( const bool hiddenMenu ) { m_hiddenMenu = hiddenMenu; };
			inline void setMaps( const QVector<GRUB::ComplexCommand::Map> &maps ) { m_maps = maps; };
			inline void setColor( const GRUB::ComplexCommand::Color &color ) { m_color = color; };
			inline void setColor( const QString color ) { m_color.resolve( color ); };
			inline void setPassword( const GRUB::ComplexCommand::Password &password ) { m_password = password; };
			inline void setPassword( const QString password ) { m_password.resolve( password ); };

			inline void setAutomagic( const GRUB::Misc::Automagic automagic ) { m_automagic = automagic; };
		//reset functions
			inline void clearSplashImage() { m_splashImage.clear(); };
			inline void clearGfxMenu() { m_gfxMenu.clear(); };
			inline void clearTimeout() { m_timeout = -1; };
			inline void clearDefault() { m_default = -1; };
			inline void clearFallback() { m_fallback = -1; };
			inline void clearHiddenMenu() { m_hiddenMenu = false; };
			inline void clearMaps() { m_maps.clear(); };
			inline void clearColor() { m_color.clear(); };
			inline void clearPassword() { m_password.clear(); };

			inline void clearAutomagic() { m_automagic.clear(); };
		//special functions
			/**
			* Appends a new map to the end of the list of maps.
			*
			* @param map The map to be appended.
			*/
			inline void addMap( const GRUB::ComplexCommand::Map &map ) { m_maps.append( map ); };
			/**
			* Replaces the map positioned at index with map.
			*
			* @param index The position of the map to be replaced in the vector
			*              that holds all maps.
			* @param map The map that will replace the above mentioned map.
			*/
			inline void replaceMap( const int index, const GRUB::ComplexCommand::Map &map ) { m_maps.replace( index, map ); };
			/**
			* Copies the map positioned at index to the end of the list.
			*
			* @param index The position of the map that will be copied.
			*/
			inline void copyMap( int index ) { m_maps.append( m_maps.at( index ) ); };
			/**
			* Removes the map positioned at index.
			*
			* @param index The position of the map that will be removed.
			*/
			inline void removeMap( int index ) { m_maps.remove( index ); };
		private:
			QString m_splashImage;
			QString m_gfxMenu;
			int m_timeout;
			int m_default;
			int m_fallback;
			bool m_hiddenMenu;
			QVector<GRUB::ComplexCommand::Map> m_maps;
			GRUB::ComplexCommand::Color m_color;
			GRUB::ComplexCommand::Password m_password;

			GRUB::Misc::Automagic m_automagic;
		};
	}
}

#endif
