/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

//Own
#include "backup.h"

//Qt
#include <qfileinfo.h>

//KDE
#include <kdebug.h>
#include <kdesu/su.h>
#include <kio/netaccess.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kurl.h>

//KGRUBEditor
#include "root.h"

namespace Core
{
	namespace Backup
	{
		bool backup( const KUrl &source, const KUrl &target, QWidget *parent, bool verbose )
		{
			if ( KIO::NetAccess::exists( source, true, parent ) )
			{
				if ( ( source.isLocalFile() && target.isLocalFile() ) && ( !QFileInfo( source.path() ).isReadable() || !QFileInfo( source.path() ).isWritable() ) )
				{
					if ( Core::Root::requestPassword( parent ) )
					{
						KDESu::SuProcess cp( QByteArray( "root" ), QByteArray( "cp " ).append( source.path() ).append( " " ).append( target.path() ) );
						if ( cp.exec( Core::Root::Password.toLocal8Bit() ) == 0 )
						{
							kDebug() << source << "was successfully copied to" << target;
							if ( verbose )
								KMessageBox::information( parent, i18nc( "@info", "%1 was successfully copied to %2.", source.path(), target.path() ) );
							return true;
						}
						else
						{
							kWarning() << source << "could not be copied to" << target;
							if ( verbose )
								KMessageBox::error( parent, i18nc( "@info", "%1 could not be copied to %2.", source.path(), target.path() ) );
						}
					}
					else
					{
						kWarning() << "The backup operation was aborted by the user.";
						if ( verbose )
							KMessageBox::sorry( parent, i18nc( "@info", "The backup operation was aborted by the user." ) );  
					}
				}
				else
				{
					if ( KIO::NetAccess::upload( source.path(), target, parent ) )
					{
						kDebug() << source << "was successfully copied to" << target;
						if ( verbose )
							KMessageBox::information( parent, i18nc( "@info", "%1 was successfully copied to %2.", source.path(), target.path() ) );
						return true;
					}
					else
					{
						kWarning() << KIO::NetAccess::lastErrorString();
						if ( verbose )
							KMessageBox::error( parent, KIO::NetAccess::lastErrorString() );
					}
				}
			}
			else
			{
				kWarning() << "The backup operation failed." << source << "does not exist!";
				if ( verbose )
					KMessageBox::error( parent, i18nc( "@info", "<para>The backup operation failed.</para><para>%1 does not exist!</para>", source.path() ) );
			}
			return false;
		}
		bool deleteBackup( const KUrl &backup, QWidget *parent, bool verbose )
		{
			if ( backup.isLocalFile() && !QFileInfo( backup.path() ).isWritable() )
			{
				if ( Core::Root::requestPassword( parent ) )
				{
					KDESu::SuProcess rm( QByteArray( "root" ), QByteArray( "rm " ).append( backup.path() ) );
					if ( rm.exec( Core::Root::Password.toLocal8Bit() ) == 0 )
					{
						kDebug() << backup << "was successfully deleted";
						if ( verbose )
							KMessageBox::information( parent, i18nc( "@info", "%1 was successfully deleted.", backup.path() ) );
						return true;
					}
					else
					{
						kWarning() << backup << "could not be deleted";
						if ( verbose )
							KMessageBox::error( parent, i18nc( "@info", "%1 could not be deleted.", backup.path() ) );
					}
				}
				else
				{
					kWarning() << "The backup operation was aborted by the user.";
					if ( verbose )
						KMessageBox::sorry( parent, i18nc( "@info", "The backup operation was aborted by the user." ) );  
				}
			}
			else
			{		
				if ( KIO::NetAccess::del( backup, parent ) )
				{
					kDebug() << backup << "was successfully deleted";
					if ( verbose )
						KMessageBox::information( parent, i18nc( "@info", "%1 was successfully deleted.", backup.path() ) );
					return true;
				}
				else
				{
					kWarning() << KIO::NetAccess::lastErrorString();
					if ( verbose )
						KMessageBox::error( parent, KIO::NetAccess::lastErrorString() );
				}
			}
			return false;
		}
	}
}
