/* Analog VU meter plugin for xmms
 *
 * Copyright (C) 2002 Pekka Harjamki <analogvu@mcfish.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>

#include <stdio.h> 
#include <stdlib.h> 
#include <string.h>

#include <sys/types.h> 
#include <dirent.h>
#include <sys/stat.h> 
#include <unistd.h>

#include "vumeter.h"

extern struct vumeter_skin_info vumeter_skin;
extern struct vumeter_cfg_info vumeter_cfg;

extern GdkPixbuf 	*background,
			*titlebar_on,*titlebar_off,
			*skin_pic,
			*vumeterwin,
			*overlayimg;

extern GdkGC *linestyle;

extern GtkWidget *clist_skinlist;

/*****************************************************************
  	        Load list of skins in directory
*****************************************************************/

int vumeter_test_skin(char *name)
{
 char dir[1024];
 struct stat statbuf;

 /* Let's try to load skin defined in name variable, or fall back to default images */
 if(name=="" || name==NULL || strcmp(name,".")==0 || strcmp(name,"..")==0 || strlen(name)>150) return(0);
  
 sprintf(dir,"%s/.xmms/VU_Meter_skins/%s",g_get_home_dir(),name);
 if(lstat(dir,&statbuf) || !S_ISDIR(statbuf.st_mode))
 {
  sprintf(dir,"%s/%s",global_skin_dir,name);
  if(lstat(dir,&statbuf) || !S_ISDIR(statbuf.st_mode))
  {
   printf("[VUmeter plugin error]: No skins found matching name %s!\n",name);
   return(0);
  }
 }

 return(1);
}

void scan_skin_dir(char *dir_path)
{
 DIR *dir;
 struct dirent *dir_ent;
 struct stat statbuf;
 char file[1024],directory[1024];
 char *skin_data[2];
 
 if(strcmp(dir_path,"home")==0)
 {
  sprintf(directory,"%s/.xmms/VU_Meter_skins",g_get_home_dir());
 } else if(strcmp(dir_path,"global")==0) {
  sprintf(directory,global_skin_dir);
 } else {
  strcpy(directory,dir_path);
 }

 if( (dir=opendir(directory)) )
 {
  while( (dir_ent=readdir(dir)) )
  if( strcmp(dir_ent->d_name,"..")!=0 && strcmp(dir_ent->d_name,".")!=0 )
  {
   sprintf(file,"%s/%s",directory,dir_ent->d_name);
   if(!lstat(file,&statbuf) && S_ISDIR(statbuf.st_mode))
   {
	   skin_data[0]=dir_ent->d_name;
	   skin_data[1]=file;
	   gtk_clist_append(GTK_CLIST(clist_skinlist),skin_data);
   }
  }
  closedir(dir);
 }
}

void splitstr(char *dst1, char *dst2, char *src, char delim)
{
        int i=0, j=0, srclen=0;
        srclen=strlen(src);
        for (i=0; i<srclen && src[i]!=delim; i++) dst1[i]=src[i];
        dst1[i++]='\0';
        for (j=0; i<srclen; i++,j++) dst2[j]=src[i];
        dst2[j++]='\0';
}

/*******************************************************
     Convert line color value from hex value to gint32
 *******************************************************/

gint32 convert_linecolor(char *value)
{
 unsigned char tmp[8],i;
 char length=strlen(value),*p;
 gint32 returnvalue;

 // Unknown value, return black color
 if(length!=8 && length!=6) return(0);

 // Put all characters to uppercase
 g_strup(value);

 // see if there's 0x at the beginning
 if(length==8) for(i=2; i<length; i++) tmp[i-2]=value[i];
 if(length==6) strcpy(tmp,value);

 returnvalue = strtol(tmp, &p, 16);
 return(returnvalue);
}

/*******************************************************
			Load skin
 *******************************************************/

int vumeter_load_skin(char *name)
{
 char dir[1024],file[1024],line[1024],
      var_name[1024],var_value[1024],overlay_img[1024],
      back_img[1024],titleon_img[1024],titleoff_img[1024];
 FILE *fp;

 /* Clean buffers */
 strcpy(overlay_img,"");
 strcpy(titleon_img,"");
 strcpy(titleoff_img,"");
 strcpy(back_img,"");

 /* Set defaults */

 vumeter_skin.exit_x1=265;		vumeter_skin.exit_x2=271;
 vumeter_skin.exit_y1=4;		vumeter_skin.exit_y2=10;
 vumeter_skin.config_x1=6;		vumeter_skin.config_x2=14;	
 vumeter_skin.config_y1=4;		vumeter_skin.config_y2=10;	
 
 vumeter_skin.digitalvu_enabled=0;
 vumeter_skin.analogvu_enabled=1;
 
 vumeter_skin.left_rad=60.0;		vumeter_skin.left_x=71;		vumeter_skin.left_y=111;
 vumeter_skin.right_rad=60.0;		vumeter_skin.right_x=203;	vumeter_skin.right_y=111;
 vumeter_skin.width=275;		vumeter_skin.height=116;	vumeter_skin.titlebar_ymax=11;	
 vumeter_skin.line_width=1;		vumeter_skin.line_color=0;	vumeter_skin.overlay_x=0;
 vumeter_skin.needle_min_angle=0;	vumeter_skin.left_mirror=0;	vumeter_skin.overlay_y=0;
 vumeter_skin.needle_max_angle=180; 	vumeter_skin.right_mirror=0;	vumeter_skin.overlay_enabled=1;


 /* Let's try to load skin defined in name variable, or fall back to default images */
 if(vumeter_test_skin(name)==0)
 {
  if(vumeter_test_skin("default")==0)
  {
   printf("[VUmeter plugin error]: Error loading requested skin and could not fall back to default skin!\n");
   return(0); 
  } else strcpy(var_name,"default");
 } else strcpy(var_name,name); 

 sprintf(dir,"%s/.xmms/VU_Meter_skins/%s",g_get_home_dir(),var_name);
 
 /* Start reading skin.cfg file, and try to look for atleast image names */
 sprintf(file,"%s/skin.cfg",dir);
 if( (fp=fopen(file,"r")) !=NULL)
 {
  strcpy(vumeter_skin.name,var_name);
  while(fgets(line,1023,fp))
  {
   g_strstrip(line);
   if( line[0]!='\n' && line[0]!='#' && line!="" && strlen(line)>3 )
   {   
    splitstr(var_name,var_value,line,'=');
    g_strstrip(var_name);
    g_strstrip(var_value);
    g_strup(var_name);



    if     (strcmp(var_name, "EXIT_X1")==0)		vumeter_skin.exit_x1=atoi(var_value);
    else if(strcmp(var_name, "EXIT_X2")==0)		vumeter_skin.exit_x2=atoi(var_value);
    else if(strcmp(var_name, "EXIT_Y1")==0)		vumeter_skin.exit_y1=atoi(var_value);
    else if(strcmp(var_name, "EXIT_Y2")==0)		vumeter_skin.exit_y2=atoi(var_value);

    else if(strcmp(var_name, "ENABLE_LEFT_MIRROR")==0)	vumeter_skin.left_mirror=atoi(var_value);
    else if(strcmp(var_name, "ENABLE_RIGHT_MIRROR")==0) vumeter_skin.right_mirror=atoi(var_value);
    else if(strcmp(var_name, "ENABLE_OVERLAY_IMG")==0)  vumeter_skin.overlay_enabled=atoi(var_value);
    else if(strcmp(var_name, "ENABLE_DIGITAL_VU")==0)	vumeter_skin.digitalvu_enabled=atoi(var_value);
    else if(strcmp(var_name, "ENABLE_ANALOG_VU")==0)	vumeter_skin.analogvu_enabled=atoi(var_value);

    else if(strcmp(var_name, "CONFIG_X1")==0)		vumeter_skin.config_x1=atoi(var_value);
    else if(strcmp(var_name, "CONFIG_X2")==0)		vumeter_skin.config_x2=atoi(var_value);
    else if(strcmp(var_name, "CONFIG_Y1")==0)		vumeter_skin.config_y1=atoi(var_value);
    else if(strcmp(var_name, "CONFIG_Y2")==0)		vumeter_skin.config_y2=atoi(var_value);

    else if(strcmp(var_name, "LEFT_RADIUS")==0)		vumeter_skin.left_rad=atof(var_value);
    else if(strcmp(var_name, "LEFT_XSTART")==0)		vumeter_skin.left_x=atoi(var_value);
    else if(strcmp(var_name, "LEFT_YSTART")==0)		vumeter_skin.left_y=atoi(var_value);

    else if(strcmp(var_name, "RIGHT_RADIUS")==0)	vumeter_skin.right_rad=atof(var_value);
    else if(strcmp(var_name, "RIGHT_XSTART")==0)	vumeter_skin.right_x=atoi(var_value);
    else if(strcmp(var_name, "RIGHT_YSTART")==0)	vumeter_skin.right_y=atoi(var_value);

    else if(strcmp(var_name, "OVERLAY_X")==0)		vumeter_skin.overlay_x=atoi(var_value);
    else if(strcmp(var_name, "OVERLAY_Y")==0)		vumeter_skin.overlay_y=atoi(var_value);

    else if(strcmp(var_name, "TITLEBAR_HEIGHT")==0)	vumeter_skin.titlebar_ymax=atoi(var_value);

    else if(strcmp(var_name, "NEEDLE_WIDTH")==0)	vumeter_skin.line_width=atoi(var_value);
    else if(strcmp(var_name, "NEEDLE_COLOR")==0)	vumeter_skin.line_color=convert_linecolor(var_value);

    else if(strcmp(var_name, "NEEDLE_MIN_ANGLE")==0)	vumeter_skin.needle_min_angle=atoi(var_value);
    else if(strcmp(var_name, "NEEDLE_MAX_ANGLE")==0)	vumeter_skin.needle_max_angle=atoi(var_value);

    else if(strcmp(var_name, "BACKGROUND_IMG")==0)	strcpy(back_img,var_value);
    else if(strcmp(var_name, "OVERLAY_IMG")==0)		strcpy(overlay_img,var_value);
    else if(strcmp(var_name, "TITLEBAR_ON_IMG")==0)	strcpy(titleon_img,var_value);
    else if(strcmp(var_name, "TITLEBAR_OFF_IMG")==0)	strcpy(titleoff_img,var_value);

//    else					printf("%s = %s\n",var_name,var_value);

   }
  }
  fclose(fp);
 } else {
  printf("[VUmeter plugin error]: Directory %s found, but no skin.cfg so loading aborted!\n",dir);
  return(0);
 }

 /* Release old image */
 if(skin_pic!=NULL) 	{ gdk_pixbuf_unref(skin_pic);		skin_pic=NULL; }
 if(titlebar_on!=NULL) 	{ gdk_pixbuf_unref(titlebar_on);	titlebar_on=NULL; }
 if(titlebar_off!=NULL) { gdk_pixbuf_unref(titlebar_off);	titlebar_off=NULL; }
 if(overlayimg!=NULL) 	{ gdk_pixbuf_unref(overlayimg); 	overlayimg=NULL; }

 /* Load png files */

 if(back_img!="" && strlen(back_img)>4 && back_img[0]!=0)
 {
  sprintf(file,"%s/%s",dir,back_img);
  skin_pic=gdk_pixbuf_new_from_file(file);
  if(!skin_pic) { printf("[VUmeter plugin error]: Error loading Skin (background) (%s)!\n",file); }
 }

 if(titleon_img!="" && strlen(titleon_img)>4 && titleon_img[0]!=0)
 {
  sprintf(file,"%s/%s",dir,titleon_img);
  titlebar_on=gdk_pixbuf_new_from_file(file);
  if(!titlebar_on) { printf("[VUmeter plugin error]: Error loading Skin (titlebar_on) (%s)!\n",file); }
 }

 if(titleoff_img!="" && strlen(titleoff_img)>4 && titleoff_img[0]!=0)
 {
  sprintf(file,"%s/%s",dir,titleoff_img);
  titlebar_off=gdk_pixbuf_new_from_file(file);
  if(!titlebar_off) { printf("[VUmeter plugin error]: Error loading Skin (titlebar_off) (%s)!\n",file); }
 }

 if(overlay_img!="" && strlen(overlay_img)>4 && overlay_img[0]!=0 && vumeter_skin.overlay_enabled==1)
 {
  sprintf(file,"%s/%s",dir,overlay_img);
  overlayimg=gdk_pixbuf_new_from_file(file);

  if(!overlayimg) { printf("[VUmeter plugin error]: Error loading Skin (overlay image) (%s)!\n",file); vumeter_skin.overlay_enabled=0; } 
             else { vumeter_skin.overlay_enabled=1; }

 } else { vumeter_skin.overlay_enabled=0; }

 if(background==NULL)
 {
  background=gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8,vumeter_skin.width , vumeter_skin.height);
  if(!background) { printf("[VUmeter plugin error]: Allocation of background buffer failed\n"); }
 }

 /* Set Draw style */
 gdk_rgb_gc_set_foreground (linestyle,vumeter_skin.line_color);
 gdk_gc_set_line_attributes(linestyle, vumeter_skin.line_width , GDK_LINE_SOLID, GDK_CAP_BUTT, GDK_JOIN_MITER);

 /* Create basic skin */
 gdk_pixbuf_copy_area(	titlebar_on,0,0,vumeter_skin.width,gdk_pixbuf_get_height(titlebar_on),	background,0,0);
 gdk_pixbuf_copy_area(	skin_pic,0,0,vumeter_skin.width,gdk_pixbuf_get_height(skin_pic),	background,0,gdk_pixbuf_get_height(titlebar_on));
 return(1);
}
