/***************************************************************************
    smb4kshareslistviewitem  -  The shares list view item class of Smb4K.
                             -------------------
    begin                : Sa Jun 30 2007
    copyright            : (C) 2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

// Qt includes
#include <qpixmap.h>
#include <qcolor.h>

// KDE includes
#include <kiconeffect.h>
#include <kdebug.h>

// application specific includes
#include "smb4kshareslistviewitem.h"
#include "smb4kshareslistview.h"


Smb4KSharesListViewItem::Smb4KSharesListViewItem( Smb4KShare *share, bool mountpoint,
                                                  Smb4KSharesListView *parent )
: KListViewItem( parent ), m_share( *share ), m_mountpoint( mountpoint ),
  m_initial_setup( true )
{
  setDropEnabled( true );
  setDragEnabled( true );

  m_loader = new KIconLoader();

  setupItem( m_share, m_mountpoint );
}


Smb4KSharesListViewItem::~Smb4KSharesListViewItem()
{
  // Do not touch the Smb4KShare object!
  delete m_loader;
}


void Smb4KSharesListViewItem::setupItem( const Smb4KShare &share, bool mountpoint )
{
  // Only do something here if either the item hasn't been set up
  // yet, or share and mountpoint changed, respectively.
  if ( m_initial_setup || !m_share.equals( share ) || m_mountpoint != mountpoint )
  {
    if ( m_initial_setup || m_share.isBroken() != share.isBroken() )
    {
      QPixmap pix;

      int icon_state = m_share.isForeign() ? KIcon::DisabledState : KIcon::DefaultState;

      if ( m_share.isBroken() )
      {
        QImage over = m_loader->loadIcon( "button_cancel", KIcon::Small,
                                          0, icon_state, 0L, false ).convertToImage();
        QImage src = m_loader->loadIcon( "hdd_mount", KIcon::Small,
                                         0, icon_state, 0L, false ).convertToImage();

        QImage over_desk = m_loader->loadIcon( "button_cancel", KIcon::Desktop,
                                               0, icon_state, 0L, false ).convertToImage();
        QImage src_desk = m_loader->loadIcon( "hdd_mount", KIcon::Desktop,
                                              0, icon_state, 0L, false ).convertToImage();

        KIconEffect e;
        e.semiTransparent( over );
        e.overlay( src, over );
        e.semiTransparent( over_desk );
        e.overlay( src_desk, over_desk );

        pix = QPixmap( src );
        m_desktop_pixmap = QPixmap( src_desk );
      }
      else
      {
        pix = m_loader->loadIcon( "hdd_mount", KIcon::Small,
                                  0, icon_state, 0L, false );

        m_desktop_pixmap = m_loader->loadIcon( "hdd_mount", KIcon::Desktop,
                                               0, icon_state, 0L, false );
      }

      setPixmap( Item, pix );
    }
    else
    {
      // Do nothing
    }

    if ( !m_initial_setup )
    {
      if ( m_mountpoint != mountpoint )
      {
        setText( Item, (m_mountpoint ? m_share.path() : m_share.name()) );
      }

      // The file system, owner and login won't change

      QString total, free, used, total_dim, free_dim, used_dim;

      if ( shareObject()->totalDiskSpace() - shareObject()->freeDiskSpace() > 1024 )
      {
        double tmp_used = (shareObject()->totalDiskSpace() - shareObject()->freeDiskSpace()) / 1024;
        used_dim = "MB";

        if ( tmp_used >= 1024 )
        {
          tmp_used = tmp_used / 1024;
          used_dim = "GB";
        }

        used = QString( "%1" ).arg( tmp_used, 0, 'f', 1 );
      }
      else
      {
        used_dim = "kB";
        double tmp_used = shareObject()->totalDiskSpace() - shareObject()->freeDiskSpace();
        used = QString( "%1" ).arg( tmp_used, 0, 'f', 1 );
      }

      if ( shareObject()->freeDiskSpace() >= 1024 )
      {
        double tmp_free = shareObject()->freeDiskSpace() / 1024;
        free_dim = "MB";

        if ( tmp_free >= 1024 )
        {
          tmp_free = tmp_free / 1024;
          free_dim = "GB";
        }

        free = QString( "%1" ).arg( tmp_free, 0, 'f', 1 );
      }
      else
      {
        free_dim = "kB";
        free = QString( "%1" ).arg( shareObject()->freeDiskSpace(), 0, 'f', 1 );
      }

      if ( shareObject()->totalDiskSpace() >= 1024 )
      {
        double tmp_total = shareObject()->totalDiskSpace() / 1024;
        total_dim = "MB";

        if ( tmp_total >= 1024 )
        {
          tmp_total = tmp_total / 1024;
          total_dim = "GB";
        }

        total = QString( "%1" ).arg( tmp_total, 0, 'f', 1 );
      }
      else
      {
        total_dim = "kB";
        total = QString( "%1" ).arg( shareObject()->totalDiskSpace(), 0, 'f', 1 );
      }

      setText( Free, QString( "%1 %2" ).arg( free, free_dim ) );
      setText( Used, QString( "%1 %2" ).arg( used, used_dim ) );
      setText( Total, QString( "%1 %2" ).arg( total, total_dim ) );
    }
    else
    {
      setText( Item, (m_mountpoint ? m_share.path() : m_share.name()) );

      setText( Owner, QString::compare( m_share.filesystem(), "smbfs" ) == 0 ?
               QString( "%1 - %2" ).arg( m_share.user() ).arg( m_share.group() ) :
               QString::null );
#ifndef __FreeBSD__
      setText( Login, QString::compare( m_share.filesystem(), "cifs" ) == 0 ?
               m_share.cifsLogin() :
               QString::null );
#endif
      setText( FileSystem, m_share.filesystem().upper() );

      QString total, free, used, total_dim, free_dim, used_dim;

      if ( shareObject()->totalDiskSpace() - shareObject()->freeDiskSpace() > 1024 )
      {
        double tmp_used = (shareObject()->totalDiskSpace() - shareObject()->freeDiskSpace()) / 1024;
        used_dim = "MB";

        if ( tmp_used >= 1024 )
        {
          tmp_used = tmp_used / 1024;
          used_dim = "GB";
        }

        used = QString( "%1" ).arg( tmp_used, 0, 'f', 1 );
      }
      else
      {
        used_dim = "kB";
        double tmp_used = shareObject()->totalDiskSpace() - shareObject()->freeDiskSpace();
        used = QString( "%1" ).arg( tmp_used, 0, 'f', 1 );
      }

      if ( shareObject()->freeDiskSpace() >= 1024 )
      {
        double tmp_free = shareObject()->freeDiskSpace() / 1024;
        free_dim = "MB";

        if ( tmp_free >= 1024 )
        {
          tmp_free = tmp_free / 1024;
          free_dim = "GB";
        }

        free = QString( "%1" ).arg( tmp_free, 0, 'f', 1 );
      }
      else
      {
        free_dim = "kB";
        free = QString( "%1" ).arg( shareObject()->freeDiskSpace(), 0, 'f', 1 );
      }

      if ( shareObject()->totalDiskSpace() >= 1024 )
      {
        double tmp_total = shareObject()->totalDiskSpace() / 1024;
        total_dim = "MB";

        if ( tmp_total >= 1024 )
        {
          tmp_total = tmp_total / 1024;
          total_dim = "GB";
        }

        total = QString( "%1" ).arg( tmp_total, 0, 'f', 1 );
      }
      else
      {
        total_dim = "kB";
        total = QString( "%1" ).arg( shareObject()->totalDiskSpace(), 0, 'f', 1 );
      }

      setText( Free, QString( "%1 %2" ).arg( free, free_dim ) );
      setText( Used, QString( "%1 %2" ).arg( used, used_dim ) );
      setText( Total, QString( "%1 %2" ).arg( total, total_dim ) );
    }

    m_initial_setup = false;
    m_share = share;
    m_mountpoint = mountpoint;
  }
  else
  {
    // Do nothing
  }
}


bool Smb4KSharesListViewItem::sameShareObject( Smb4KShare *share )
{
  return m_share.equals( *share );
}


void Smb4KSharesListViewItem::replaceShareObject( Smb4KShare *share )
{
  setupItem( *share, m_mountpoint );
}


void Smb4KSharesListViewItem::paintCell( QPainter *p, const QColorGroup &cg, int col, int width, int align )
{
  // Set the color of the item text:
  QColorGroup colorgrp( cg );

  if ( m_share.isForeign() )
  {
    colorgrp.setColor( QColorGroup::Text, Qt::gray );
  }

  if ( col != Usage )
  {
    KListViewItem::paintCell( p, colorgrp, col, width, align );

    return;
  }

  // Draw the usage:
  // This code was inspired by KAudioCreator.
  p->setPen( colorgrp.base() );
  p->drawRect( 0, 0, width, height() );

  if ( isSelected() )
  {
    p->fillRect( 1, 1, width-2, height()-2, colorgrp.highlight() );
  }
  else
  {
    p->fillRect( 1, 1, width-2, height()-2, colorgrp.base() );
  }

  if ( !m_share.isBroken() )
  {
    int percent = (int)(((double)(width-2)) * (m_share.percentage()/100));

    p->fillRect( 1, 1, percent, height()-2, !m_share.isForeign() ? Qt::red : Qt::red.light( 175 ) );
    p->fillRect( percent+1, 1, width-percent-2, height()-2, !m_share.isForeign() ? Qt::green : Qt::green.light( 175 ) );

    p->setPen( !m_share.isForeign() ? colorgrp.foreground() : Qt::gray );
    p->drawRect( 1, 1, width-2, height()-2 );

    // Show the text:
    p->setPen( colorgrp.text() );

    //   if ( isSelected() )
    //   {
    //     p->setPen( colorgrp.highlightedText() );
    //   }

    p->drawText( 0, 0, width-1, height()-1, Qt::AlignCenter, QString( "%1 \%" ).arg( m_share.percentage(), 0, 'f', 1 ) );
  }
  else
  {
    p->fillRect( 1, 1, width-2, height()-2, colorgrp.base() );

    p->setPen( !m_share.isForeign() ? colorgrp.foreground() : Qt::gray );
    p->drawRect( 1, 1, width-2, height()-2 );
  }
}


bool Smb4KSharesListViewItem::acceptDrop( const QMimeSource *source ) const
{
  if ( source->provides( "text/plain" ) )
  {
    return true;
  }

  return false;
}
