/***************************************************************************
    smb4kfileio  -  Does file IO operations for Smb4K
                             -------------------
    begin                : Do Jan 1 2004
    copyright            : (C) 2004-2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KFILEIO_H
#define SMB4KFILEIO_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstring.h>
#include <qcstring.h>

// KDE include
#include <kprocess.h>


/**
 * This class belongs to the core classes of Smb4K. It handles IO
 * operations that are performed on system configuration files.
 *
 * @author Alexander Reinholdt <dustpuppy@mail.berlios.de>
 */

class Smb4KFileIO : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor
     *
     * @param parent      The parent object of this class
     *
     * @param name        The name of this class
     */
    Smb4KFileIO( QObject *parent = 0, const char *name = 0 );

    /**
     * The destructor
     */
    ~Smb4KFileIO();

    /**
     * Enumeration that determines for which super user program we perform
     * the current action.
     */
    enum Operation { Insert, Remove, NoOperation };

    /**
     * This function writes changes to the sudoers configuration file.
     *
     * @param operation   Defines whether entries should be inserted or if
     *                    they should be removed. With Smb4KFileIO::NoOperation
     *                    the function exits immediately and returns TRUE.
     *
     * @returns TRUE if the write process was successfully initiate and FALSE
     * otherwise.
     *
     * @note You need to connect to the finished() and failed() signals to find
     * out if the the write process finished successfully.
     *
     */
    bool writeSudoers( Smb4KFileIO::Operation operation );

    /**
     * This function writes changes to the super.tab configuration file.
     *
     * @param operation   Defines whether entries should be inserted or if
     *                    they should be removed. With Smb4KFileIO::NoOperation
     *                    the function exits immediately and returns TRUE.
     *
     * @returns TRUE if the write process was successfully initiate and FALSE
     * otherwise.
     *
     * @note You need to connect to the finished() and failed() signals to find
     * out if the the write process finished successfully.
     *
     */
    bool writeSuperTab( Smb4KFileIO::Operation operation );

  signals:
    /**
     * This signal is emitted when somthing went wrong with the writing to
     * the system configuration files.
     */
    void failed();

    /**
     * This signal is emitted when the writing to the system configuration
     * files has finished. It is emitted in case the writing was successful
     * as well as in case it wasn't.
     */
    void finished();

  protected slots:
    /**
     * This slot is invokes when the application is closed. It is connected
     * to KApplication::shutDown().
     */
    void slotShutdown();

    /**
     * This slot receives shell program output from Stderr.
     *
     * @param proc            The KProcess object
     *
     * @param buf             The buffer that holds the error output
     *
     * @param len             The length of the buffer
     */
    void slotReceivedStderr( KProcess *proc, char *buf, int len );

    /**
     * This slot receives shell program output from Stdout.
     *
     * @param proc            The KProcess object
     *
     * @param buf             The buffer that holds the output
     *
     * @param len             The length of the buffer
     */
    void slotReceivedStdout( KProcess *proc, char *buf, int len );

    /**
     * This slot is called, when the process exited.
     *
     * @param proc            The KProcess object
     */
    void slotProcessExited( KProcess *proc );

  private:
    /**
     * This function creates a lock file in /tmp if it does not
     * exist already. If the user is not allowed to write to the
     * desired file a the moment, the user will be shown an error
     * dialog and the function will return FALSE.
     *
     * Checks are performed to make sure it is save to write to an
     * existing lock file using the system call lstat().
     *
     * @param filename        The name of the file that is to be modified.
     *
     * @returns TRUE if the creation was successful and FALSE if
     * something went wrong.
     */
    bool createLockFile( const QString &filename );

    /**
     * This function removes the lock file or at least the
     * entry within it.
     *
     * Checks are performed to make sure it is save to write to an
     * existing lock file using the system call lstat().
     *
     * @param shutdown        Should be set to FALSE if you do not want to have
     *                        any error message shown. Otherwise you should set it
     *                        to TRUE.
     *
     * @returns TRUE if the removal was successful and FALSE if
     * something went wrong.
     */
    bool removeLockFile( const bool error_message = true );

    /**
     * This function finds a file.
     *
     * @param filename        The name of the file
     *
     * @returns the canonical path of the file or an empty string if it could not be
     *          found.
     */
    const QCString findFile( const QString &filename );

    /**
     * Enumeration that is used to tell the process what has to be done.
     */
    enum State { ReadSudoers, ReadSuperTab, WriteSudoers, WriteSuperTab, Idle };

    /**
     * This integer holds one of the entries of the enumeration above.
     */
    State m_state;

    /**
     * This buffer holds the output that was received at Stdout.
     */
    QString m_buffer;

    /**
     * Process the sudoers file.
     */
    void processSudoers();

    /**
     * Process the sudoers file.
     */
    void processSuperTab();

    /**
     * This is the absolute path of the lock file.
     */
    QCString m_lock_file;

    /**
     * Which operation should be performed
     */
    Operation m_operation;

    /**
     * The KProcess object
     */
    KProcess *m_proc;

    /**
     * Tell if an error occurred.
     */
    bool m_error_occurred;
};


#endif
