/*----------------------------------------------------------------------*
 * File:        main.C
 *----------------------------------------------------------------------*
 *
 * All portions of code are copyright by their respective author/s.
 * Copyright (c) 1992      John Bovey, University of Kent at Canterbury <jdb@ukc.ac.uk>
 *                              - original version
 * Copyright (c) 1994      Robert Nation <nation@rocket.sanders.lockheed.com>
 *                              - extensive modifications
 * Copyright (c) 1995      Garrett D'Amore <garrett@netcom.com>
 * Copyright (c) 1997      mj olesen <olesen@me.QueensU.CA>
 *                              - extensive modifications
 * Copyright (c) 1997,1998 Oezguer Kesim <kesim@math.fu-berlin.de>
 * Copyright (c) 1998-2001 Geoff Wing <gcw@pobox.com>
 *                              - extensive modifications
 * Copyright (c) 2003-2006 Marc Lehmann <pcg@goof.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *---------------------------------------------------------------------*/

#include "../config.h"          /* NECESSARY */
#include "rxvt.h"               /* NECESSARY */
#include "keyboard.h"
#include "rxvtperl.h"

#include <limits>

#include <csignal>
#include <cstring>

#ifdef TTY_GID_SUPPORT
# include <grp.h>
#endif

#ifdef HAVE_TERMIOS_H
# include <termios.h>
#endif

vector<rxvt_term *> rxvt_term::termlist;

// used to tell global functions which terminal instance is "active"
rxvt_t rxvt_current_term;

static char curlocale[128], savelocale[128];

bool
rxvt_set_locale (const char *locale) NOTHROW
{
  int size = strlen (locale) + 1;

  if (size > sizeof (curlocale))
    rxvt_fatal ("locale string too long, aborting.\n");

  if (!locale || !memcmp (locale, curlocale, size))
    return false;

  memcpy (curlocale, locale, size);
  setlocale (LC_CTYPE, curlocale);
  return true;
}

void
rxvt_push_locale (const char *locale) NOTHROW
{
  strcpy (savelocale, curlocale);
  rxvt_set_locale (locale);
}

void
rxvt_pop_locale () NOTHROW
{
  rxvt_set_locale (savelocale);
}

#if ENABLE_COMBINING
class rxvt_composite_vec rxvt_composite;

text_t rxvt_composite_vec::compose (unicode_t c1, unicode_t c2)
{
  compose_char *cc;
  
  // break compose chains, as stupid readline really likes to duplicate
  // composing characters for some reason near the end of a line.
  cc = (*this)[c1];
  while (cc)
    {
      if (cc->c2 == c2) return c1;
      cc = (*this)[cc->c1];
    }

  // check to see wether this combination already exists otherwise
  for (cc = v.end (); cc-- > v.begin (); )
    {
      if (cc->c1 == c1 && cc->c2 == c2)
        return COMPOSE_LO + (cc - v.begin ());
    }

  // allocate a new combination
  if (v.size () == COMPOSE_HI - COMPOSE_LO + 1)
    {
      static int seen;

      if (!seen++)
        fprintf (stderr, "too many unrepresentable composite characters, try --enable-unicode3\n");

      return REPLACEMENT_CHAR;
    }

  v.push_back (compose_char (c1, c2));

  return v.size () - 1 + COMPOSE_LO;
}

int rxvt_composite_vec::expand (unicode_t c, wchar_t *r)
{
  compose_char *cc = (*this)[c];

  if (!cc)
    {
      if (r) *r = c;
      return 1;
    }

  int len = expand (cc->c1, r);

  if (r) r += len;

  if (cc->c2 != NOCHAR)
    {
      len++;
      if (r) *r++ = cc->c2;
    }

  return len;

}
#endif

rxvt_term::rxvt_term ()
    :
#if TRANSPARENT || ENABLE_PERL
    rootwin_ev (this, &rxvt_term::rootwin_cb),
#endif
#ifdef HAVE_SCROLLBARS
    scrollbar_ev (this, &rxvt_term::x_cb),
#endif
#ifdef CURSOR_BLINK
    cursor_blink_ev (this, &rxvt_term::cursor_blink_cb),
#endif
#ifdef TEXT_BLINK
    text_blink_ev (this, &rxvt_term::text_blink_cb),
#endif
#ifndef NO_SCROLLBAR_BUTTON_CONTINUAL_SCROLLING
    cont_scroll_ev (this, &rxvt_term::cont_scroll_cb),
#endif
#ifdef SELECTION_SCROLLING
    sel_scroll_ev (this, &rxvt_term::sel_scroll_cb),
#endif
#if defined(MOUSE_WHEEL) && defined(MOUSE_SLIP_WHEELING)
    slip_wheel_ev (this, &rxvt_term::slip_wheel_cb),
#endif
#ifdef POINTER_BLANK
    pointer_ev (this, &rxvt_term::pointer_cb),
#endif
#ifdef USE_XIM
    im_ev (this, &rxvt_term::im_cb),
#endif
#ifndef NO_BELL     
    bell_ev (this, &rxvt_term::bell_cb),
#endif
    termwin_ev (this, &rxvt_term::x_cb),
    vt_ev (this, &rxvt_term::x_cb),
    child_ev (this, &rxvt_term::child_cb),
    check_ev (this, &rxvt_term::check_cb),
    flush_ev (this, &rxvt_term::flush_cb),
    destroy_ev (this, &rxvt_term::destroy_cb),
    pty_ev (this, &rxvt_term::pty_cb),
    incr_ev (this, &rxvt_term::incr_cb)
{
  cmdbuf_ptr = cmdbuf_endp = cmdbuf_base;

  termlist.push_back (this);

#ifdef KEYSYM_RESOURCE
  keyboard = new keyboard_manager;
#endif
}

// clean up the most important stuff, do *not* call x or free mem etc.
// for use before an emergency exit
void rxvt_term::emergency_cleanup ()
{
  if (cmd_pid)
    kill (-cmd_pid, SIGHUP);

  pty_ev.stop ();
  delete pty; pty = 0;
}

rxvt_term::~rxvt_term ()
{
  termlist.erase (find (termlist.begin (), termlist.end(), this));

  emergency_cleanup ();

#if ENABLE_STYLES
  for (int i = RS_styleCount; --i; )
    if (fontset[i] != fontset[0])
      delete fontset[i];
#endif
  delete fontset[0];

  if (display)
    {
      selection_clear ();

#ifdef USE_XIM
      im_destroy ();
#endif
#ifdef XTERM_SCROLLBAR
      if (xscrollbarGC) XFreeGC (dpy, xscrollbarGC);
      if (ShadowGC)     XFreeGC (dpy, ShadowGC);
#endif
#ifdef PLAIN_SCROLLBAR
      if (pscrollbarGC) XFreeGC (dpy, pscrollbarGC);
#endif
#ifdef NEXT_SCROLLBAR
      if (blackGC)      XFreeGC (dpy, blackGC);
      if (whiteGC)      XFreeGC (dpy, whiteGC);
      if (grayGC)       XFreeGC (dpy, grayGC);
      if (darkGC)       XFreeGC (dpy, darkGC);
      if (stippleGC)    XFreeGC (dpy, stippleGC);
      if (dimple)       XFreePixmap (dpy, dimple);
      if (upArrow)      XFreePixmap (dpy, upArrow);
      if (downArrow)    XFreePixmap (dpy, downArrow);
      if (upArrowHi)    XFreePixmap (dpy, upArrowHi);
      if (downArrowHi)  XFreePixmap (dpy, downArrowHi);
#endif
#ifdef RXVT_SCROLLBAR
      if (topShadowGC)  XFreeGC (dpy, topShadowGC);
      if (botShadowGC)  XFreeGC (dpy, botShadowGC);
      if (scrollbarGC)  XFreeGC (dpy, scrollbarGC);
#endif
      if (gc)   XFreeGC (dpy, gc);

      delete drawable;
      // destroy all windows
      if (parent[0])
        XDestroyWindow (dpy, parent[0]);

      for (int i = 0; i < TOTAL_COLORS; i++)
        if (ISSET_PIXCOLOR (i))
          {
            pix_colors_focused   [i].free (this);
#if OFF_FOCUS_FADING
            pix_colors_unfocused [i].free (this);
#endif
          }

      clear ();
    }

  delete [] pix_colors_focused;
#if OFF_FOCUS_FADING
  delete [] pix_colors_unfocused;
#endif

  displays.put (display);

  scr_release ();

  /* clear all resources */
  for (int i = 0; i < allocated.size (); i++)
    free (allocated [i]);

  free (selection.text);
  // TODO: manage env vars in child only(!)
  free (env_display);
  free (env_term);
  free (locale);
  free (v_buffer);
  free (incr_buf);

  delete envv;
  delete argv;

#ifdef KEYSYM_RESOURCE
  delete keyboard;
#endif
}

// child has exited, usually destroys
void
rxvt_term::child_cb (child_watcher &w, int status)
{
  HOOK_INVOKE ((this, HOOK_CHILD_EXIT, DT_INT, status, DT_END));

  cmd_pid = 0;

  if (!OPTION (Opt_hold))
    destroy ();
}

void
rxvt_term::destroy ()
{
  if (destroy_ev.active)
    return;

  HOOK_INVOKE ((this, HOOK_DESTROY, DT_END));

#if ENABLE_OVERLAY
  scr_overlay_off ();
#endif

  if (display)
    {
#if USE_XIM
      im_ev.stop (display);
#endif
#if HAVE_SCROLLBARS
      scrollbar_ev.stop (display);
#endif
#if TRANSPARENT || ENABLE_PERL
      rootwin_ev.stop (display);
#endif
      incr_ev.stop ();
      termwin_ev.stop (display);
      vt_ev.stop (display);
    }

  check_ev.stop ();
  pty_ev.stop ();
#ifdef CURSOR_BLINK
  cursor_blink_ev.stop ();
#endif
#ifdef TEXT_BLINK
  text_blink_ev.stop ();
#endif
#ifndef NO_SCROLLBAR_BUTTON_CONTINUAL_SCROLLING
  cont_scroll_ev.stop ();
#endif
#ifdef SELECTION_SCROLLING
  sel_scroll_ev.stop ();
#endif
#ifdef POINTER_BLANK
  pointer_ev.stop ();
#endif

  destroy_ev.start (0);
}

void
rxvt_term::destroy_cb (time_watcher &w)
{
  make_current ();

  delete this;
}

/*----------------------------------------------------------------------*/
/*
 * Exit gracefully, clearing the utmp entry and restoring tty attributes
 * TODO: if debugging, this should free up any known resources if we can
 */
static XErrorHandler old_xerror_handler;

static void
rxvt_emergency_cleanup ()
{
  for (rxvt_term **t = rxvt_term::termlist.begin (); t < rxvt_term::termlist.end (); t++)
    (*t)->emergency_cleanup ();
}

#if !ENABLE_MINIMAL
static void
print_x_error (Display *dpy, XErrorEvent *event)
{
    char buffer[BUFSIZ];
    char mesg[BUFSIZ];
    char number[32];
    char *mtype = "XlibMessage";
    XGetErrorText(dpy, event->error_code, buffer, BUFSIZ);
    XGetErrorDatabaseText(dpy, mtype, "XError", "X Error", mesg, BUFSIZ);
    rxvt_warn ("An X Error occured, trying to continue after report.\n");
    rxvt_warn ("%s:  %s\n", mesg, buffer);
    XGetErrorDatabaseText(dpy, mtype, "MajorCode", "Request Major code %d", mesg, BUFSIZ);
    rxvt_warn (strncat (mesg, "\n", BUFSIZ), event->request_code);
    sprintf(number, "%d", event->request_code);
    XGetErrorDatabaseText(dpy, "XRequest", number, "", buffer, BUFSIZ);
    rxvt_warn ("(which is %s)\n", buffer);
    if (event->request_code >= 128) {
	XGetErrorDatabaseText(dpy, mtype, "MinorCode", "Request Minor code %d",
			      mesg, BUFSIZ);
        rxvt_warn (strncat (mesg, "\n", BUFSIZ), event->minor_code);
    }
    if ((event->error_code == BadWindow) ||
	       (event->error_code == BadPixmap) ||
	       (event->error_code == BadCursor) ||
	       (event->error_code == BadFont) ||
	       (event->error_code == BadDrawable) ||
	       (event->error_code == BadColor) ||
	       (event->error_code == BadGC) ||
	       (event->error_code == BadIDChoice) ||
	       (event->error_code == BadValue) ||
	       (event->error_code == BadAtom)) {
	if (event->error_code == BadValue)
	    XGetErrorDatabaseText(dpy, mtype, "Value", "Value 0x%x",
				  mesg, BUFSIZ);
	else if (event->error_code == BadAtom)
	    XGetErrorDatabaseText(dpy, mtype, "AtomID", "AtomID 0x%x",
				  mesg, BUFSIZ);
	else
	    XGetErrorDatabaseText(dpy, mtype, "ResourceID", "ResourceID 0x%x",
				  mesg, BUFSIZ);
	rxvt_warn (strncat (mesg, "\n", BUFSIZ), event->resourceid);
    }
    XGetErrorDatabaseText(dpy, mtype, "ErrorSerial", "Error Serial #%d", 
			  mesg, BUFSIZ);
    rxvt_warn (strncat (mesg, "\n", BUFSIZ), event->serial);
}
#endif

int
rxvt_xerror_handler (Display *display, XErrorEvent *event)
{
  if (GET_R->allowedxerror == -1)
    GET_R->allowedxerror = event->error_code;
  else
    {
      // GET_R is most likely not the terminal which caused the error,
      // so just output the error and continue
#if ENABLE_MINIMAL
      old_xerror_handler (display, event);
#else
      print_x_error (display, event);
#endif
    }

  return 0;
}

int
rxvt_xioerror_handler (Display *display)
{
  rxvt_warn ("X connection to '%s' broken, unable to recover, exiting.\n",
             DisplayString (display));
  rxvt_emergency_cleanup ();
  _exit (EXIT_FAILURE);
}

/*----------------------------------------------------------------------*/
bool
rxvt_term::init (int argc, const char *const *argv, stringvec *envv)
{
  this->envv = envv;

  SET_R (this);
  set_locale ("");
  set_environ (envv); // few things in X do not call setlocale :(

  if (!init_vars ())
    return false;

  init_secondary ();

  const char **cmd_argv = init_resources (argc, argv);

#ifdef KEYSYM_RESOURCE
  keyboard->register_done ();
#endif

#ifdef HAVE_SCROLLBARS
  if (OPTION (Opt_scrollBar))
    scrollBar.setIdle ();    /* set existence for size calculations */
#endif

  pty = ptytty::create ();

  create_windows (argc, argv);

  init_xlocale ();

  scr_reset (); // initialize screen

#if 0
  XSynchronize (dpy, True);
#endif

#ifdef HAVE_SCROLLBARS
  if (OPTION (Opt_scrollBar))
    resize_scrollbar ();      /* create and map scrollbar */
#endif
#ifdef TRANSPARENT
  if (OPTION (Opt_transparent))
    {
      XSelectInput (dpy, display->root, PropertyChangeMask);
      check_our_parents ();
      rootwin_ev.start (display, display->root);
    }
#endif
#if ENABLE_PERL
  rootwin_ev.start (display, display->root);
#endif

  set_colorfgbg ();

  init_command (cmd_argv);

  free (cmd_argv);

  if (pty->pty >= 0)
    pty_ev.start (pty->pty, EVENT_READ);

  check_ev.start ();

  HOOK_INVOKE ((this, HOOK_START, DT_END));

#if ENABLE_XEMBED
  if (rs[Rs_embed])
    {
      long info[2] = { 0, XEMBED_MAPPED };

      XChangeProperty (dpy, parent[0], xa[XA_XEMBED_INFO], xa[XA_XEMBED_INFO],
                       32, PropModeReplace, (unsigned char *)&info, 2);
    }
#endif

  XMapWindow (dpy, vt);
  XMapWindow (dpy, parent[0]);

  return true;
}

static struct sig_handlers
{
  sig_watcher sw_term, sw_int;
  
  /*
   * Catch a fatal signal and tidy up before quitting
   */
  void
  sig_term (sig_watcher &w)
  {
    rxvt_emergency_cleanup ();
    signal (w.signum, SIG_DFL);
    kill (getpid (), w.signum);
  }

  sig_handlers ()
  : sw_term (this, &sig_handlers::sig_term),
    sw_int  (this, &sig_handlers::sig_term)
  {
  }
} sig_handlers;

char **rxvt_environ; // startup environment

void
rxvt_init ()
{
  ptytty::init ();

  rxvt_environ = environ;

  signal (SIGHUP,  SIG_IGN);
  signal (SIGPIPE, SIG_IGN);

  sig_handlers.sw_term.start (SIGTERM);
  sig_handlers.sw_int.start  (SIGINT);

  /* need to trap SIGURG for SVR4 (Unixware) rlogin */
  /* signal (SIGURG, SIG_DFL); */

  old_xerror_handler = XSetErrorHandler ((XErrorHandler) rxvt_xerror_handler);
  // TODO: handle this with exceptions and tolerate the memory loss
  XSetIOErrorHandler (rxvt_xioerror_handler);

  XrmInitialize ();
}

/* ------------------------------------------------------------------------- *
 *                         MEMORY ALLOCATION WRAPPERS                        *
 * ------------------------------------------------------------------------- */
void *
rxvt_malloc (size_t size)
{
  void *p = malloc (size);

  if (!p)
    rxvt_fatal ("memory allocation failure. aborting.\n");

  return p;
}

void *
rxvt_calloc (size_t number, size_t size)
{
  void *p = calloc (number, size);

  if (!p)
    rxvt_fatal ("memory allocation failure. aborting.\n");

  return p;
}

void *
rxvt_realloc (void *ptr, size_t size)
{
  void *p = realloc (ptr, size);

  if (!p)
    rxvt_fatal ("memory allocation failure. aborting.\n");

  return p;
}

/*----------------------------------------------------------------------*/
/*
 * window size/position calculcations for XSizeHint and other storage.
 * if width/height are non-zero then override calculated width/height
 */
void
rxvt_term::window_calc (unsigned int newwidth, unsigned int newheight)
{
  short recalc_x, recalc_y;
  int x, y, sb_w, flags;
  unsigned int w, h;
  unsigned int max_width, max_height;

  szHint.flags = PMinSize | PResizeInc | PBaseSize | PWinGravity;
  szHint.win_gravity = NorthWestGravity;
  /* szHint.min_aspect.x = szHint.min_aspect.y = 1; */

  recalc_x = recalc_y = 0;
  flags = 0;

  if (!parsed_geometry)
    {
      parsed_geometry = 1;

      if (rs[Rs_geometry])
        flags = XParseGeometry (rs[Rs_geometry], &x, &y, &w, &h);

      if (flags & WidthValue)
        {
          ncol = clamp (w, 0, std::numeric_limits<int16_t>::max ());
          szHint.flags |= USSize;
        }

      if (flags & HeightValue)
        {
          nrow = clamp (h, 0, std::numeric_limits<int16_t>::max ());
          szHint.flags |= USSize;
        }

      if (flags & XValue)
        {
          szHint.x = x;
          szHint.flags |= USPosition;

          if (flags & XNegative)
            {
              recalc_x = 1;
              szHint.win_gravity = NorthEastGravity;
            }
        }

      if (flags & YValue)
        {
          szHint.y = y;
          szHint.flags |= USPosition;

          if (flags & YNegative)
            {
              recalc_y = 1;

              if (szHint.win_gravity == NorthEastGravity)
                szHint.win_gravity = SouthEastGravity;
              else
                szHint.win_gravity = SouthWestGravity;
            }
        }
    }

  /* TODO: BOUNDS */
  width = ncol * fwidth;
  height = nrow * fheight;
  max_width = MAX_COLS * fwidth;
  max_height = MAX_ROWS * fheight;

  szHint.base_width = szHint.base_height = 2 * int_bwidth;

  sb_w = 0;
  window_vt_x = window_vt_y = int_bwidth;

  if (scrollBar.state)
    {
      sb_w = scrollbar_TotalWidth ();
      szHint.base_width += sb_w;

      if (!OPTION (Opt_scrollBar_right))
        window_vt_x += sb_w;
    }

  szHint.width_inc  = fwidth;
  szHint.height_inc = fheight;
  szHint.min_width  = szHint.base_width + szHint.width_inc;
  szHint.min_height = szHint.base_height + szHint.height_inc;

  if (newwidth && newwidth - szHint.base_width < max_width)
    {
      szHint.width = newwidth;
      width = newwidth - szHint.base_width;
    }
  else
    {
      min_it (width, max_width);
      szHint.width = szHint.base_width + width;
    }

  if (newheight && newheight - szHint.base_height < max_height)
    {
      szHint.height = newheight;
      height = newheight - szHint.base_height;
    }
  else
    {
      min_it (height, max_height);
      szHint.height = szHint.base_height + height;
    }

  if (scrollBar.state && OPTION (Opt_scrollBar_right))
    window_sb_x = szHint.width - sb_w;

  if (recalc_x)
    szHint.x += DisplayWidth  (dpy, display->screen) - szHint.width  - 2 * ext_bwidth;

  if (recalc_y)
    szHint.y += DisplayHeight (dpy, display->screen) - szHint.height - 2 * ext_bwidth;

  ncol = width / fwidth;
  nrow = height / fheight;
}

/*----------------------------------------------------------------------*/
/*
 * Tell the teletype handler what size the window is.
 * Called after a window size change.
 */
void
rxvt_term::tt_winch ()
{
  if (pty->pty < 0)
    return;

  struct winsize ws;

  ws.ws_col = ncol;
  ws.ws_row = nrow;
  ws.ws_xpixel = width;
  ws.ws_ypixel = height;
  (void)ioctl (pty->pty, TIOCSWINSZ, &ws);

#if 0
  // TIOCSWINSZ⎈ is supposed to do this automatically and correctly
  if (cmd_pid)               /* force through to the command */
    kill (-cmd_pid, SIGWINCH);
#endif
}

/*----------------------------------------------------------------------*/
/* set_fonts () - load and set the various fonts
 *
 * init = 1   - initialize
 *
 * fontname == FONT_UP  - switch to bigger font
 * fontname == FONT_DN  - switch to smaller font
 */
bool
rxvt_term::set_fonts ()
{
  rxvt_fontset *fs = new rxvt_fontset (this);
  rxvt_fontprop prop;

  if (!fs
      || !fs->populate (rs[Rs_font] ? rs[Rs_font] : "fixed")
      || !fs->realize_font (1))
    {
      delete fs;
      return false;
    }

#if ENABLE_STYLES
  for (int i = RS_styleCount; --i; )
    if (fontset[i] != fontset[0])
      delete fontset[i];
#endif

  delete fontset[0];
  fontset[0] = fs;

  prop = (*fs)[1]->properties ();
  prop.height += lineSpace;

  fs->set_prop (prop, false);

  fwidth  = prop.width;
  fheight = prop.height;
  fbase   = prop.ascent;

  for (int style = 1; style < 4; style++)
    {
#if ENABLE_STYLES
      const char *res = rs[Rs_font + style];

      if (res && !*res)
        fontset[style] = fontset[0];
      else
        {
          fontset[style] = fs = new rxvt_fontset (this);
          rxvt_fontprop prop2 = prop;

          if (res)
            {
              fs->populate (res);
              fs->set_prop (prop2, false);
            }
          else
            {
              fs->populate (fontset[0]->fontdesc);

              if (SET_STYLE (0, style) & RS_Bold)   prop2.weight = rxvt_fontprop::bold;
              if (SET_STYLE (0, style) & RS_Italic) prop2.slant  = rxvt_fontprop::italic;

              fs->set_prop (prop2, true);
            }

        }
#else
      fontset[style] = fontset[0];
#endif
    }

  if (parent[0])
    {
      resize_all_windows (0, 0, 0);
      scr_remap_chars ();
      scr_touch (true);
    }   

  return true;
}

void rxvt_term::set_string_property (Atom prop, const char *str, int len)
{
  XChangeProperty (dpy, parent[0],
                   prop, XA_STRING, 8, PropModeReplace,
                   (const unsigned char *)str, len >= 0 ? len : strlen (str));
}

void rxvt_term::set_utf8_property (Atom prop, const char *str, int len)
{
  wchar_t *ws = rxvt_mbstowcs (str, len);
  char *s = rxvt_wcstoutf8 (ws);

  XChangeProperty (dpy, parent[0],
                   prop, xa[XA_UTF8_STRING], 8, PropModeReplace,
                   (const unsigned char *)s, strlen (s));

  free (s);
  free (ws);
}

/*----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*/
/* xterm sequences - title, iconName, color (exptl) */
void
rxvt_term::set_title (const char *str)
{
  set_string_property (XA_WM_NAME, str);
#if ENABLE_EWMH
  set_utf8_property (xa[XA_NET_WM_NAME], str);
#endif
}

void
rxvt_term::set_icon_name (const char *str)
{
  set_string_property (XA_WM_ICON_NAME, str);
#if ENABLE_EWMH
  set_utf8_property (xa[XA_NET_WM_ICON_NAME], str);
#endif
}

#ifdef XTERM_COLOR_CHANGE
void
rxvt_term::set_window_color (int idx, const char *color)
{
  rxvt_color xcol;
  int i;
  
  if (color == NULL || *color == '\0')
    return;

  color = strdup (color);
  allocated.push_back ((void *)color);
  rs[Rs_color + idx] = color;

  /* handle color aliases */
  if (isdigit (*color))
    {
      i = atoi (color);

      if (i >= 8 && i <= 15)
        {        /* bright colors */
          i -= 8;
          pix_colors_focused[idx] = pix_colors_focused[minBrightCOLOR + i];
          goto done;
        }

      if (i >= 0 && i <= 7)
        { /* normal colors */
          pix_colors_focused[idx] = pix_colors_focused[minCOLOR + i];
          goto done;
        }
    }

  set_color (xcol, color);

  /*
   * FIXME: should free colors here, but no idea how to do it so instead,
   * so just keep gobbling up the colormap
   */

  pix_colors_focused[idx] = xcol;

  /* XSetWindowAttributes attr; */
  /* Cursor cursor; */
done:
  /*TODO: handle Color_BD, scrollbar background, etc. */

  update_fade_color (idx);
  recolour_cursor ();
  scr_recolour ();
}

#else
# define set_window_color (idx,color)   ((void)0)
#endif                          /* XTERM_COLOR_CHANGE */

void
rxvt_term::recolour_cursor ()
{
  XColor fg, bg;

  (ISSET_PIXCOLOR (Color_pointer_fg)
     ? pix_colors_focused[Color_pointer_fg]
     : pix_colors_focused[Color_fg]).get (fg);

  (ISSET_PIXCOLOR (Color_pointer_bg)
     ? pix_colors_focused[Color_pointer_bg]
     : pix_colors_focused[Color_bg]).get (bg);

  XRecolorCursor (dpy, TermWin_cursor, &fg, &bg);
}

/*----------------------------------------------------------------------*/
/*
 * find if fg/bg matches any of the normal (low-intensity) colors
 */
void
rxvt_term::set_colorfgbg ()
{
  unsigned int i;
  const char *xpmb = "\0";
  char fstr[sizeof ("default") + 1], bstr[sizeof ("default") + 1];

  strcpy (fstr, "default");
  strcpy (bstr, "default");
  for (i = Color_Black; i <= Color_White; i++)
    if (pix_colors[Color_fg] == pix_colors[i])
      {
        sprintf (fstr, "%d", (i - Color_Black));
        break;
      }

  for (i = Color_Black; i <= Color_White; i++)
    if (pix_colors[Color_bg] == pix_colors[i])
      {
        sprintf (bstr, "%d", (i - Color_Black));
#ifdef XPM_BACKGROUND
        xpmb = "default;";
#endif
        break;
      }

  sprintf (env_colorfgbg, "COLORFGBG=%s;%s%s", fstr, xpmb, bstr);
}

/*----------------------------------------------------------------------*/

bool
rxvt_term::set_color (rxvt_color &color, const char *name)
{
  if (color.set (this, name))
    return true;

  rxvt_warn ("can't get colour '%s', continuing without.\n", name);
  return false;
}

void
rxvt_term::alias_color (int dst, int src)
{
  pix_colors[dst].set (this, rs[Rs_color + dst] = rs[Rs_color + src]);
}

/* -------------------------------------------------------------------- *
 * -                         WINDOW RESIZING                          - *
 * -------------------------------------------------------------------- */
void
rxvt_term::resize_all_windows (unsigned int newwidth, unsigned int newheight, int ignoreparent)
{
  int fix_screen;
  int old_width  = szHint.width;
  int old_height = szHint.height;

  window_calc (newwidth, newheight);

  if (!HOOK_INVOKE ((this, HOOK_RESIZE_ALL_WINDOWS, DT_INT, newwidth, DT_INT, newheight, DT_END)))
    XSetWMNormalHints (dpy, parent[0], &szHint);

  if (!ignoreparent)
    {
#ifdef SMART_RESIZE
      /*
       * resize by Marius Gedminas <marius.gedminas@uosis.mif.vu.lt>
       * reposition window on resize depending on placement on screen
       */
      int x, y, x1, y1;
      int dx, dy;
      unsigned int unused_w1, unused_h1, unused_b1, unused_d1;
      Window unused_cr;

      XTranslateCoordinates (dpy, parent[0], display->root,
                             0, 0, &x, &y, &unused_cr);
      XGetGeometry (dpy, parent[0], &unused_cr, &x1, &y1,
                    &unused_w1, &unused_h1, &unused_b1, &unused_d1);
      /*
       * if display->root isn't the parent window, a WM will probably have offset
       * our position for handles and decorations.  Counter it
       */
      if (x1 != x || y1 != y)
        {
          x -= x1;
          y -= y1;
        }

      x1 = (DisplayWidth  (dpy, display->screen) - old_width ) / 2;
      y1 = (DisplayHeight (dpy, display->screen) - old_height) / 2;
      dx = old_width  - szHint.width;
      dy = old_height - szHint.height;

      /* Check position of the center of the window */
      if (x < x1)             /* left half */
        dx = 0;
      else if (x == x1)       /* exact center */
        dx /= 2;
      if (y < y1)             /* top half */
        dy = 0;
      else if (y == y1)       /* exact center */
        dy /= 2;

      XMoveResizeWindow (dpy, parent[0], x + dx, y + dy,
                         szHint.width, szHint.height);
#else
      XResizeWindow (dpy, parent[0], szHint.width, szHint.height);
#endif
    }

  fix_screen = ncol != prev_ncol || nrow != prev_nrow;

  if (fix_screen || newwidth != old_width || newheight != old_height)
    {
      if (scrollBar.state)
        {
          XMoveResizeWindow (dpy, scrollBar.win,
                             window_sb_x, 0,
                             scrollbar_TotalWidth (), szHint.height);
          resize_scrollbar ();
        }

      XMoveResizeWindow (dpy, vt,
                         window_vt_x, window_vt_y,
                         width, height);

      scr_clear ();
#ifdef XPM_BACKGROUND
      resize_pixmap ();
#endif
    }

  if (fix_screen || old_height == 0)
    scr_reset ();

  // TODO, with nvidia-8178, resizes kill the alpha channel, report if not fixed in newer version
  //scr_touch (false);

#ifdef XPM_BACKGROUND
  if (pixmap)
    scr_touch (false);
#endif

#ifdef USE_XIM
  IMSetPosition ();
#endif
}

/*
 * Set the width/height of the vt window in characters.  Units are pixels.
 * good for toggling 80/132 columns
 */
void
rxvt_term::set_widthheight (unsigned int newwidth, unsigned int newheight)
{
  XWindowAttributes wattr;

  if (newwidth == 0 || newheight == 0)
    {
      XGetWindowAttributes (dpy, display->root, &wattr);

      if (newwidth == 0)
        newwidth = wattr.width - szHint.base_width;
      if (newheight == 0)
        newheight = wattr.height - szHint.base_height;
    }

  if (newwidth != width || newheight != height)
    {
      newwidth += szHint.base_width;
      newheight += szHint.base_height;
      resize_all_windows (newwidth, newheight, 0);
    }
}

/* -------------------------------------------------------------------- *
 * -                      X INPUT METHOD ROUTINES                     - *
 * -------------------------------------------------------------------- */
#ifdef USE_XIM

void
rxvt_term::im_set_color (unsigned long &fg, unsigned long &bg)
{
  fg = pix_colors [Color_fg];
  bg = pix_colors [Color_bg];
}

void
rxvt_term::im_set_size (XRectangle &size)
{
  // the int_bwidth terms make no sense to me
  size.x      = int_bwidth;
  size.y      = int_bwidth;
  size.width  = Width2Pixel (ncol) + int_bwidth;
  size.height = Height2Pixel (nrow) + int_bwidth;
}

void
rxvt_term::im_set_preedit_area (XRectangle &preedit_rect,
                                XRectangle &status_rect,
                                const XRectangle &needed_rect)
{
  preedit_rect.x      = needed_rect.width;
  preedit_rect.y      = 0;
  preedit_rect.width  = Width2Pixel (ncol) - needed_rect.width + 1;
  preedit_rect.height = fheight;

  status_rect.x       = 0;
  status_rect.y       = 0;
  status_rect.width   = needed_rect.width ? needed_rect.width : Width2Pixel (ncol) + 1;
  status_rect.height  = fheight;
}

/* Checking whether input method is running. */
bool
rxvt_term::IMisRunning ()
{
  char *p;
  Atom atom;
  Window win;
  char server[IMBUFSIZ];

  /* get current locale modifier */
  if ((p = XSetLocaleModifiers (NULL)) != NULL)
    {
      strcpy (server, "@server=");
      strncat (server, & (p[4]), IMBUFSIZ - 9); /* skip "@im=" */

      if ((p = strchr (server + 1, '@')) != NULL)      /* first one only */
        *p = '\0';

      atom = XInternAtom (dpy, server, False);
      win = XGetSelectionOwner (dpy, atom);

      if (win != None)
        return True;
    }

  return False;
}

void
rxvt_term::IMSendSpot ()
{
  XPoint nspot;
  XVaNestedList preedit_attr;

  if (!Input_Context
      || !focus
      || !(input_style & (XIMPreeditPosition | XIMPreeditCallbacks)))
    return;

  im_set_position (nspot);

  if (nspot.x == spot.x && nspot.y == spot.y)
    return;

  spot = nspot;

  preedit_attr = XVaCreateNestedList (0, XNSpotLocation, &spot, NULL);
  XSetICValues (Input_Context, XNPreeditAttributes, preedit_attr, NULL);
  XFree (preedit_attr);
}

void
rxvt_term::im_destroy ()
{
  if (input_method)
    {
      if (Input_Context && input_method->xim)
        XDestroyIC (Input_Context);

      display->put_xim (input_method);
      input_method = 0;
    }

  Input_Context = 0;
}

#ifdef ENABLE_XIM_ONTHESPOT

static void
xim_preedit_start (XIC ic, XPointer client_data, XPointer call_data)
{
  ((rxvt_term *)client_data)->make_current ();
  HOOK_INVOKE (((rxvt_term *)client_data, HOOK_XIM_PREEDIT_START, DT_END));
}

static void
xim_preedit_done (XIC ic, XPointer client_data, XPointer call_data)
{
  ((rxvt_term *)client_data)->make_current ();
  HOOK_INVOKE (((rxvt_term *)client_data, HOOK_XIM_PREEDIT_DONE, DT_END));
}

static void
xim_preedit_draw (XIC ic, XPointer client_data, XIMPreeditDrawCallbackStruct *call_data)
{
  rxvt_term *term = (rxvt_term *)client_data;
  XIMText *text = call_data->text;

  term->make_current ();

  if (text)
    {
      void *str;

      if (!text->encoding_is_wchar && text->string.multi_byte)
        {
          // of course, X makes it ugly again
          if (term->rs[Rs_imLocale])
            SET_LOCALE (term->rs[Rs_imLocale]);

          str = rxvt_temp_buf ((text->length + 1) * sizeof (wchar_t));
          mbstowcs ((wchar_t *)str, text->string.multi_byte, text->length + 1);

          if (term->rs[Rs_imLocale])
            SET_LOCALE (term->locale);
        }
      else
        str = (void *)text->string.wide_char;
      
      HOOK_INVOKE ((term, HOOK_XIM_PREEDIT_DRAW,
                    DT_INT, call_data->caret,
                    DT_INT, call_data->chg_first,
                    DT_INT, call_data->chg_length,
                    DT_LCS_LEN, (void *)text->feedback, text->feedback ? (int)text->length : 0,
                    DT_WCS_LEN, str, str ? (int)text->length : 0,
                    DT_END));
    }
  else
    HOOK_INVOKE ((term, HOOK_XIM_PREEDIT_DRAW,
                  DT_INT, call_data->caret,
                  DT_INT, call_data->chg_first,
                  DT_INT, call_data->chg_length,
                  DT_END));
}

#if 0
static void
xim_preedit_caret (XIC ic, XPointer client_data, XIMPreeditCaretCallbackStruct *call_data)
{
  ((rxvt_term *)client_data)->make_current ();
  HOOK_INVOKE (((rxvt_term *)client_data, HOOK_XIM_PREEDIT_CARET,
                DT_INT, call_data->position,
                DT_INT, call_data->direction,
                DT_INT, call_data->style,
                DT_END));
}
#endif

#endif

/*
 * Try to open a XIM with the current modifiers, then see if we can
 * open a suitable preedit type
 */
bool
rxvt_term::IM_get_IC (const char *modifiers)
{
  int i, j, found;
  XIM xim;
  XPoint spot;
  XRectangle rect, status_rect, needed_rect;
  unsigned long fg, bg;
  const char *p;
  char **s;
  XIMStyles *xim_styles;
#ifdef ENABLE_XIM_ONTHESPOT
  XIMCallback xcb[4];
#endif

  set_environ (envv);

  if (! ((p = XSetLocaleModifiers (modifiers)) && *p))
    return false;

  input_method = display->get_xim (locale, modifiers);
  if (input_method == NULL)
    return false;

  xim = input_method->xim;
  spot.x = spot.y = -1;

  xim_styles = NULL;
  if (XGetIMValues (xim, XNQueryInputStyle, &xim_styles, NULL)
      || !xim_styles || !xim_styles->count_styles)
    {
      im_destroy ();
      return false;
    }

  const char *pet[] = { rs[Rs_preeditType], "OverTheSpot,OffTheSpot,Root,None" };

  for (int pi = 0; pi < 2; pi++)
    {
      p = pet[pi];

      if (!p)
        continue;

      s = rxvt_splitcommastring (p);

      for (i = found = 0; !found && s[i]; i++)
        {
          if (!strcmp (s[i], "OverTheSpot"))
            input_style = XIMPreeditPosition | XIMStatusNothing;
          else if (!strcmp (s[i], "OffTheSpot"))
            input_style = XIMPreeditArea | XIMStatusArea;
          else if (!strcmp (s[i], "Root"))
            input_style = XIMPreeditNothing | XIMStatusNothing;
          else if (!strcmp (s[i], "None"))
            input_style = XIMPreeditNone | XIMStatusNone;
#ifdef ENABLE_XIM_ONTHESPOT
          else if (SHOULD_INVOKE (HOOK_XIM_PREEDIT_START) && !strcmp (s[i], "OnTheSpot"))
            input_style = XIMPreeditCallbacks | XIMStatusNothing;
#endif
          else
            input_style = XIMPreeditNothing | XIMStatusNothing;

          for (j = 0; j < xim_styles->count_styles; j++)
            if (input_style == xim_styles->supported_styles[j])
              {
                rxvt_freecommastring (s);

                found = 1;
                goto foundpet;
              }

        }

      rxvt_freecommastring (s);
    }

foundpet:

  XFree (xim_styles);

  if (!found)
    {
      im_destroy ();
      return false;
    }

  XFontSet fs = 0;
  XVaNestedList preedit_attr = 0, status_attr = 0;

  if (input_style & (XIMPreeditPosition | XIMPreeditArea))
    {
      // fake us a font-set, please
      char **missing_charset_list;
      int missing_charset_count;
      char *def_string;
      char pat[512];

      sprintf (pat,
               "-*-*-*-R-*-*-%d-*-*-*-*-*-*,"
               "-*-*-*-R-*-*-%d-*-*-*-*-*-*,"
               "-*-*-*-R-*-*-%d-*-*-*-*-*-*,"
               "-*-*-*-R-*-*-%d-*-*-*-*-*-*,"
               "-*-*-*-R-*-*-%d-*-*-*-*-*-*,"
               "*",
               fheight,
               fheight + 1, fheight - 1,
               fheight - 2, fheight + 2);

      fs = XCreateFontSet (dpy, rs[Rs_imFont] ? rs[Rs_imFont] : pat,
                           &missing_charset_list, &missing_charset_count, &def_string);

      if (missing_charset_list)
        XFreeStringList (missing_charset_list);

      if (!fs)
        {
          input_style &= ~(XIMPreeditPosition | XIMPreeditArea);
          rxvt_warn ("unable to create fontset for input method, try \"-pt Root\". Continuing.\n");
        }
    }

  if (input_style & XIMPreeditPosition)
    {
      im_set_size (rect);
      im_set_position (spot);
      im_set_color (fg, bg);

      preedit_attr = XVaCreateNestedList (0,
                                          XNForeground, fg,
                                          XNBackground, bg,
                                          XNArea, &rect,
                                          XNSpotLocation, &spot,
                                          XNFontSet, fs,
                                          NULL);
    }
  else if (input_style & XIMPreeditArea)
    {
      im_set_color (fg, bg);

      /*
       * The necessary width of preedit area is unknown
       * until create input context.
       */
      needed_rect.width = 0;
      im_set_preedit_area (rect, status_rect, needed_rect);

      preedit_attr = XVaCreateNestedList (0,
                                          XNForeground, fg,
                                          XNBackground, bg,
                                          XNArea, &rect,
                                          XNFontSet, fs,
                                          NULL);
      status_attr = XVaCreateNestedList (0,
                                         XNForeground, fg,
                                         XNBackground, bg,
                                         XNArea, &status_rect,
                                         XNFontSet, fs,
                                         NULL);
    }
#if ENABLE_XIM_ONTHESPOT
  else if (input_style & XIMPreeditCallbacks)
    {
      im_set_position (spot);

      xcb[0].client_data = (XPointer)this; xcb[0].callback = (XIMProc)xim_preedit_start;
      xcb[1].client_data = (XPointer)this; xcb[1].callback = (XIMProc)xim_preedit_done;
      xcb[2].client_data = (XPointer)this; xcb[2].callback = (XIMProc)xim_preedit_draw;
# if 0
      xcb[3].client_data = (XPointer)this; xcb[3].callback = (XIMProc)xim_preedit_caret;
# endif

      preedit_attr = XVaCreateNestedList (0,
                                          XNSpotLocation, &spot,
                                          XNPreeditStartCallback, &xcb[0],
                                          XNPreeditDoneCallback , &xcb[1],
                                          XNPreeditDrawCallback , &xcb[2],
# if 0
                                          XNPreeditCaretCallback, &xcb[3],
# endif
                                          NULL);
    }
#endif

  Input_Context = XCreateIC (xim,
                             XNInputStyle, input_style,
                             XNClientWindow, vt,
                             XNFocusWindow, parent[0],
                             preedit_attr ? XNPreeditAttributes : NULL,
                             preedit_attr,
                             status_attr ? XNStatusAttributes : NULL,
                             status_attr, NULL);

  if (preedit_attr) XFree (preedit_attr);
  if (status_attr)  XFree (status_attr);
  if (fs)           XFreeFontSet (dpy, fs);

  if (Input_Context == NULL)
    {
      rxvt_warn ("failed to create input context, continuing without XIM.\n");
      im_destroy ();
      return false;
    }

#if 0
  // unfortunately, only the focus window is used by XIM, hard to fix
  if (!XGetICValues (Input_Context, XNFilterEvents, &vt_emask_xim, NULL))
    vt_select_input ();
#endif

  IMSetPosition ();

  return true;
}

void
rxvt_term::im_cb ()
{
  int i;
  const char *p;
  char **s;
  char buf[IMBUFSIZ];

  make_current ();

  im_destroy ();

  if (Input_Context)
    return;

#if defined(HAVE_XSETLOCALE) || defined(HAVE_SETLOCALE)
  if (rs[Rs_imLocale])
    SET_LOCALE (rs[Rs_imLocale]);
#endif

  p = rs[Rs_inputMethod];
  if (p && *p)
    {
      bool found = false;

      s = rxvt_splitcommastring (p);

      for (i = 0; s[i]; i++)
        {
          if (*s[i])
            {
              strcpy (buf, "@im=");
              strncat (buf, s[i], IMBUFSIZ - 5);
              if (IM_get_IC (buf))
                {
                  found = true;
                  break;
                }
            }
        }

      rxvt_freecommastring (s);

      if (found)
        goto done;
    }

  /* try with XMODIFIERS env. var. */
  if (IM_get_IC (""))
    goto done;

  /* try with no modifiers base IF the user didn't specify an IM */
  if (IM_get_IC ("@im=none"))
    goto done;

done:
#if defined(HAVE_XSETLOCALE) || defined(HAVE_SETLOCALE)
  if (rs[Rs_imLocale])
    SET_LOCALE (locale);
#endif
}

void
rxvt_term::IMSetPosition ()
{
  XRectangle preedit_rect, status_rect, *needed_rect;
  XVaNestedList preedit_attr, status_attr;

  if (!Input_Context
      || !focus
      || !(input_style & (XIMPreeditArea | XIMPreeditPosition))
      || !IMisRunning ())
    return;

  if (input_style & XIMPreeditPosition)
    {
      im_set_size (preedit_rect);
      preedit_attr = XVaCreateNestedList (0, XNArea, &preedit_rect, NULL);
    
      XSetICValues (Input_Context,
                    XNPreeditAttributes, preedit_attr, NULL);
    }
  else
    {
      /* Getting the necessary width of preedit area */
      status_attr = XVaCreateNestedList (0, XNAreaNeeded, &needed_rect, NULL);
      XGetICValues (Input_Context, XNStatusAttributes, status_attr, NULL);
      XFree (status_attr);

      im_set_preedit_area (preedit_rect, status_rect, *needed_rect);
      XFree (needed_rect);

      preedit_attr = XVaCreateNestedList (0, XNArea, &preedit_rect, NULL);
      status_attr  = XVaCreateNestedList (0, XNArea, &status_rect,  NULL);

      XSetICValues (Input_Context,
                    XNPreeditAttributes, preedit_attr,
                    XNStatusAttributes,  status_attr,
                    NULL);

      XFree (status_attr);
    }

   XFree (preedit_attr);
}
#endif                          /* USE_XIM */

/*----------------------- end-of-file (C source) -----------------------*/
