/****************************************************************************
** Subtitles - class
**
**   Created : Tue Dec 27 12:09:08 2006
**        by : Varol Okan using XEmacs editor
** Copyright : (c) Varol Okan
**   License : GPL v 2.0
**
** This class encapsulates the attributes for the subtitle generation.
**
****************************************************************************/

#include <qprogressbar.h>
#include <qapplication.h>
#include <qdatetime.h>
#include <qfileinfo.h>
#include <qpainter.h>
#include <qdialog.h>
#include <qlayout.h>
#include <qimage.h>
#include <qdom.h>
#include <qdir.h>

#include "global.h"
#include "utils.h"
#include "xml_dvd.h"
#include "sourcefileentry.h"
#include "subtitles.h"
#include "qplayer/mediacreator.h"
	 
SubtitleEntry::SubtitleEntry ( )
{
  m_iSubtitleNumber = -1;
}

SubtitleEntry &SubtitleEntry::operator= ( SubtitleEntry &theOther )
{
  m_iSubtitleNumber = theOther.m_iSubtitleNumber;
  m_qsIso639        = theOther.m_qsIso639;
  return *this;
}

Subtitles::entry::entry ()
{
  iIndex = 0;
  iTimeStart = -1;
  iTimeStop  = -1;
}

Subtitles::entry &Subtitles::entry::operator = ( Subtitles::entry &theOther )
{
  iIndex      = theOther.iIndex;
  qsTimeStart = theOther.qsTimeStart;
  qsTimeStop  = theOther.qsTimeStop;
  iTimeStart  = theOther.iTimeStart;
  iTimeStop   = theOther.iTimeStop;
  qsText      = theOther.qsText;
  return *this;
}

Subtitles::Subtitles ()
{
  m_subColors[0]    = Rgba ( TRANSPARENT_COLOR       );
  m_subColors[1]    = Rgba ( START_HIGHLIGHTED_COLOR );
  m_subColors[2]    = Rgba ( START_SELECTED_COLOR    );
  //m_subColors[3]    = Rgba ( START_FRAME_COLOR       );

  m_iSubtitleNumber = 0;
  m_bFit            = false;
  m_iOutline        = 2; // Default to outline with a shadow of 2 pixels...
  m_bTextSubtitles  = false; // default to image based subtitles (render subtitles in this class)
  m_iTextFontSize   = 24;
  m_qsTextFont      = "Vera";
  m_alignment       = QPoint ( 1, 1 ); // HAlign=Center / VAlign=bottom
  m_subtitleState   = STATE_MANUAL;
  m_rect            = QRect ( 0, 0, 720, 480 ); // standard NTSC res.
}

Subtitles::~Subtitles ()
{
  for (uint t=0;t<m_listOfSubtitles.count(); t++)
    delete m_listOfSubtitles[t];
  m_listOfSubtitles.clear ();
}

Subtitles &Subtitles::operator = ( Subtitles &theOther )
{
  uint t;
  m_bTextSubtitles  = theOther.m_bTextSubtitles;
  m_qsTextFont      = theOther.m_qsTextFont;
  m_iTextFontSize   = theOther.m_iTextFontSize;
  m_bFit            = theOther.m_bFit;
  m_iOutline        = theOther.m_iOutline;
  m_font            = theOther.m_font;
  m_rect            = theOther.m_rect;
  m_alignment       = theOther.m_alignment;
  m_iSubtitleNumber = theOther.m_iSubtitleNumber;
  m_qsIso639        = theOther.m_qsIso639;
  m_qsXmlFile       = theOther.m_qsXmlFile;
  m_subtitleState   = theOther.m_subtitleState;

  for ( t=0;t<MAX_SUBTITLE_COLORS;t++)
    m_subColors[t] = theOther.m_subColors[t];
  
  if ( m_listOfSubtitles.count () ) {
    for ( t=0;t<theOther.m_listOfSubtitles.count();t++) 
      delete m_listOfSubtitles[t];
  }
  m_listOfSubtitles.clear ();

  for ( t=0;t<theOther.m_listOfSubtitles.count();t++) {
    entry *pNewEntry = new entry;
    *pNewEntry = *theOther.m_listOfSubtitles[t];
    m_listOfSubtitles.append ( pNewEntry );
  }

  return *this;
}

bool Subtitles::readProjectFile ( QDomNode &xmlNode )
{
  int t;
  QString     qsAttribute, infoName, infoText;
  QDomNode    infoNode;
  QDomElement infoElement, theElement = xmlNode.toElement();
  infoName  = theElement.tagName();

  qsAttribute = theElement.attributeNode ( SUBTITLES_NUMBER ).value();
  if ( ! qsAttribute.isEmpty ( ) )
    m_iSubtitleNumber = qsAttribute.toInt ();
  qsAttribute = theElement.attributeNode ( SUBTITLES_LANG ).value();
  if ( ! qsAttribute.isEmpty ( ) )
    m_qsIso639 = qsAttribute;
  qsAttribute = theElement.attributeNode ( SUBTITLES_TEXT_BASED ).value ( );
  if ( ! qsAttribute.isEmpty ( ) )
    m_bTextSubtitles = ( qsAttribute == "true" );
  qsAttribute = theElement.attributeNode ( SUBTITLES_STATE ).value ( );
  if ( ! qsAttribute.isEmpty ( ) )
    m_subtitleState = (enState)qsAttribute.toInt ( );
  qsAttribute = theElement.attributeNode ( SUBTITLES_TEXT_FONT ).value ( );
  if ( ! qsAttribute.isEmpty ( ) )
    m_qsTextFont = qsAttribute;
  qsAttribute = theElement.attributeNode ( SUBTITLES_TEXT_SIZE ).value ( );
  if ( ! qsAttribute.isEmpty ( ) )
    m_iTextFontSize = qsAttribute.toInt ( );
  qsAttribute = theElement.attributeNode ( SUBTITLES_FIT ).value();
  if ( ! qsAttribute.isEmpty ( ) )
    m_bFit = ( qsAttribute == "true" );
  qsAttribute = theElement.attributeNode ( SUBTITLES_OUTLINE ).value();
  m_iOutline = -1;
  if ( ! qsAttribute.isEmpty ( ) )
    m_iOutline = qsAttribute.toInt ( );
  qsAttribute = theElement.attributeNode ( SUBTITLES_SUBTITLE_FILE ).value();
  if ( ! qsAttribute.isEmpty ( ) )
    m_qsXmlFile = qsAttribute;
  qsAttribute = theElement.attributeNode ( SUBTITLES_FONT ).value();
  if ( ! qsAttribute.isEmpty ( ) ) {
    QFont theFont;
    if (theFont.fromString ( qsAttribute ))
      m_font = theFont;
  }
  qsAttribute = theElement.attributeNode ( SUBTITLES_RECT ).value();
  if ( ! qsAttribute.isEmpty ( ) ) {
    // QString "x,y,width,height" 
    QStringList list = QStringList::split ( ",", qsAttribute );
    QRect theRect;
    if (list.count() == 4)
      theRect = QRect(list[0].toInt(), list[1].toInt(),  list[2].toInt(),  list[3].toInt() );
    m_rect = theRect;
  }
  qsAttribute = theElement.attributeNode ( SUBTITLES_ALIGNMENT ).value();
  if ( ! qsAttribute.isEmpty ( ) ) {
    // QString "x,y,width,height" 
    QStringList list = QStringList::split ( ",", qsAttribute );
    QPoint theAlignment;
    if (list.count() == 2)
      theAlignment = QPoint( list[0].toInt(), list[1].toInt() );
    m_alignment = theAlignment;
  }
  qsAttribute = theElement.attributeNode ( SUBTITLES_COLORS ) .value ();
  if ( ! qsAttribute.isEmpty ( ) ) {
    QStringList list = QStringList::split ( ",", qsAttribute );
    if ( list.count() == MAX_SUBTITLE_COLORS ) {
      for ( t=0;t<MAX_SUBTITLE_COLORS; t++ )  
	m_subColors[t].fromString ( list[t] ); 
    }
  }
  qsAttribute = theElement.attributeNode ( SUBTITLES_TRANSPARENCY ) .value ();
  if ( ! qsAttribute.isEmpty ( ) ) {
    // old format when color / transparency was separate.
    QStringList list = QStringList::split ( ",", qsAttribute );
    if ( list.count ( ) == MAX_SUBTITLE_COLORS ) {
      for ( t=0;t<MAX_SUBTITLE_COLORS; t++ )  
	m_subColors[t].setAlpha ( list[t].toInt ( ) );
    }
  }
  
  infoNode = theElement.firstChild ();  
  while (!infoNode.isNull())	{
    infoElement = infoNode.toElement();
    // Okay, here we read the stored data from the xml file.
    infoName = infoElement.tagName();
    infoText = infoElement.text ();
    if (SUBTITLES_SUBTITLE_FILE == infoName)	{
      m_qsTempFile = infoText;
    }
    else if (SUBTITLES_SUBTITLE_ENTRY == infoName)	{
      Subtitles::entry *pNewEntry = new entry;
      qsAttribute = infoElement.attributeNode ( SUBTITLES_SUBTITLE_INDEX ).value();
      if (!qsAttribute.isEmpty())
	pNewEntry->iIndex = qsAttribute.toInt ();
      qsAttribute = infoElement.attributeNode ( SUBTITLES_SUBTITLE_START ).value();
      if (!qsAttribute.isEmpty())
	pNewEntry->qsTimeStart = qsAttribute;
      qsAttribute = infoElement.attributeNode ( SUBTITLES_SUBTITLE_STOP ).value();
      if (!qsAttribute.isEmpty())
	pNewEntry->qsTimeStop = qsAttribute;
      pNewEntry->iTimeStart = getTimeFromString ( pNewEntry->qsTimeStart );
      pNewEntry->iTimeStop  = getTimeFromString ( pNewEntry->qsTimeStop  );

      pNewEntry->qsText = infoText;

      m_listOfSubtitles.append ( pNewEntry );
    }
    infoNode = infoNode.nextSibling ();
  }

  return true;
}

bool Subtitles::writeProjectFile ( QDomElement &inElement )
{
  uint t;
  QDomDocument xmlDoc = inElement.ownerDocument();
  QDomElement  infoElement, tmpFileElement, theElement;
  QDomText     text;  
  entry       *pEntry;
  QString      qsColors, qsTransparency;

  if ( m_listOfSubtitles.count ( ) < 1 )
    return true;

  theElement = xmlDoc.createElement ( SOURCE_OBJECT_SUBTITLES );

  // Here we set the attributes of the <dvdauthor> tag
  //  theElement.setAttribute( SUBTITLES_TRANSPARENCY,  QString ("%1").arg ( (int)iTransparency   ) );
  theElement.setAttribute( SUBTITLES_NUMBER,        QString ("%1").arg ( (int)m_iSubtitleNumber ) );
  theElement.setAttribute( SUBTITLES_LANG,          m_qsIso639  );
  theElement.setAttribute( SUBTITLES_FONT,          m_font.toString ( ) );
  theElement.setAttribute( SUBTITLES_STATE,         QString ("%1").arg ( (int)m_subtitleState ) );
  if ( m_bTextSubtitles ) {
    theElement.setAttribute( SUBTITLES_TEXT_BASED,  QString ("%1").arg ( m_bTextSubtitles ? "true" : "false" ) );
    theElement.setAttribute( SUBTITLES_TEXT_FONT,   m_qsTextFont );
    theElement.setAttribute( SUBTITLES_TEXT_SIZE,   QString ("%1").arg ( m_iTextFontSize ) );
  }
  theElement.setAttribute( SUBTITLES_FIT,           QString ("%1").arg ( m_bFit ? "true" : "false" ) );
  if ( m_iOutline > 0 )
    theElement.setAttribute( SUBTITLES_OUTLINE,     QString ("%1").arg ( m_iOutline ) );
  theElement.setAttribute( SUBTITLES_SUBTITLE_FILE, m_qsXmlFile );
  for ( t=0;t<MAX_SUBTITLE_COLORS; t++ ) 
    qsColors += QString ( "%1," ).arg ( m_subColors[t].toString ( ) );
  theElement.setAttribute( SUBTITLES_COLORS, qsColors );
  theElement.setAttribute( SUBTITLES_RECT, QString ("%1,%2,%3,%4").arg(m_rect.x ( ) ).arg(m_rect.y ( ) ).arg(m_rect.width ( ) ).arg(m_rect.height ( ) ) );
  theElement.setAttribute( SUBTITLES_ALIGNMENT, QString ("%1,%2").arg(m_alignment.x ( ) ).arg(m_alignment.y ( ) ) );

  // store the temp file name too ...
  if ( ! m_qsTempFile.isEmpty () ) {
    tmpFileElement = xmlDoc.createElement( SUBTITLES_SUBTITLE_FILE );
    text = xmlDoc.createTextNode( m_qsTempFile );
    tmpFileElement.appendChild( text );
    theElement.appendChild ( tmpFileElement );
  }

  if ( m_listOfSubtitles.count ( ) > 0)	{
    for (t=0;t<m_listOfSubtitles.count();t++)	{
      pEntry = m_listOfSubtitles[t];
      infoElement = xmlDoc.createElement( SUBTITLES_SUBTITLE_ENTRY );
      infoElement.setAttribute( SUBTITLES_SUBTITLE_INDEX, QString ("%1").arg( pEntry->iIndex ) );
      infoElement.setAttribute( SUBTITLES_SUBTITLE_START, pEntry->qsTimeStart );
      infoElement.setAttribute( SUBTITLES_SUBTITLE_STOP,  pEntry->qsTimeStop  );

      text = xmlDoc.createTextNode( pEntry->qsText );
      infoElement.appendChild( text );
      
      theElement.appendChild ( infoElement );
    }
  }

  inElement.appendChild ( theElement );
  return true;
}

long Subtitles::getTimeFromString ( QString &qsTime )
{
  long iMSeconds = -1;
  QTime theTime;
  iMSeconds = (long)theTime.msecsTo ( QTime::fromString ( qsTime ) );

  return iMSeconds;
}

QString Subtitles::getStringFromTime ( long iMSTime )
{
  QString qsReturn;
  QTime theTime;
  theTime  = theTime.addMSecs ( iMSTime );
  qsReturn = theTime.toString ( "hh:mm:ss.zzz" );
  return qsReturn;
}

QString Subtitles::getXMLFileName ()
{
  QFileInfo fileInfo ( m_qsTempFile );
  QString qsReturn;

  qsReturn = m_qsXmlFile;
  if ( m_qsXmlFile.isEmpty ( ) )
    qsReturn = QString ("%1/%2_%3.xml").arg( fileInfo.filePath () ).arg ( fileInfo.baseName ( TRUE ).arg ( m_iSubtitleNumber ) );
  return qsReturn;
}

QRect Subtitles::getSubtitlesRect ( QString &qsResolution )
{
  QRect theRect;
  Utils theUtils;
  int iVideoWidth, iVideoHeight;
  int iFontSize  = m_font.pointSize ();
  if ( iFontSize == -1 ) {
       iFontSize = m_font.pixelSize ();
       if ( iFontSize == -1 )
	    iFontSize = 24;
  }
  iFontSize *= 3;

  iVideoWidth  = theUtils.getWHFromResolution ( qsResolution, true  );
  iVideoHeight = theUtils.getWHFromResolution ( qsResolution, false );
  if ( iVideoWidth  < 350 ) // out of spec ?
       iVideoWidth  = 720;
  if ( iVideoHeight < 240 ) // out of spec ?
       iVideoHeight = 480;
  
  theRect = m_rect;
  // Try to figure out if the user was too lazy to input values ...
  if ( ( m_rect.x() == 0 ) && ( m_rect.y() == 0 ) ) {
    if ( ( ( m_rect.width  () == iVideoWidth  ) && 
	   ( m_rect.height () == iVideoHeight )  ) ||   // rect == video's rect
	 ( ( m_rect.width  () == 0            ) && 
	   ( m_rect.height () == 0            )  )  ) { // or rect is empty
      theRect = QRect ( 0, iVideoHeight - 45 - iFontSize, iVideoWidth, iFontSize );
    }
  }
  else { // Otherwise default to the complete width and adjust the height accordingly
    if ( m_rect.height ( ) > 2 )
      iFontSize = m_rect.height ( );
    theRect = QRect ( 0, iVideoHeight - 45 - iFontSize, iVideoWidth, iFontSize );
  }

  return theRect;
}

bool Subtitles::alreadyRendered ( QString qsBasePath )
{
  Subtitles::entry *pEntry;
  QString   qsSubIdx, qsStart, qsThePath, qsFileName;
  QFileInfo fileInfo;
  int       t, iTotalCount;

  qsSubIdx.sprintf ( "/sub%02d_", m_iSubtitleNumber );
  qsThePath = qsBasePath + qsSubIdx;
  
  iTotalCount = (int)m_listOfSubtitles.count ();  
  for (t=0;t<iTotalCount; t++) {
    pEntry  = m_listOfSubtitles[t];
    qsStart = pEntry->qsTimeStart;
    qsStart.remove ( ":"      );
    qsStart.replace( ".", "_" );
    qsFileName = QString ("%1%2.png").arg ( qsThePath ).arg( qsStart );
    fileInfo.setFile ( qsFileName );
    if ( ( ! fileInfo.exists ( ) ) || ( fileInfo.size ( ) < 10 ) )
      return false;
  }
  return true; // AFAIK this subtitle has been rendered already
}

Subtitles *Subtitles::render ( QString qsBasePath, QString qsResolution, QString qsFileInfoName, bool bForce )
{
  // If we are using Spumux's version to render subtitles, then we won't need to render them here ... right ?
  if ( m_bTextSubtitles )
    return NULL;

  // The first check is to see if the subtitles have been rendered already. If so we return this
  // and ask the user if they should be re-created or not.
  if ( ! bForce && alreadyRendered ( qsBasePath ) )
    return this;

  // Here we handle this in just one function as we can call 
  // qApp->processEvents ( 100 ); // process for 100 ms
  int       t, j, iTotalCount, iWidth, iHeight;
  QString   qsThePath, qsFileName, qsStart, qsSubIdx, qsLang;
  QRgb      theColor, colors[ MAX_SUBTITLE_COLORS ];
  Utils     theUtils;
  QImage    theImage;
  QRect     theRect;
  QPainter  thePainter;
  QDialog   progressDialog   ( NULL );
  QProgressBar *pProgressBar = NULL;
  QFileInfo fileInfo ( qsFileInfoName );
  iTotalCount = (int)m_listOfSubtitles.count ( );

  // The following widgets will create th progress dialog on the fly.
  if ( iTotalCount > 20 ) { // no progress for only a few subtitles.
    QGridLayout  *pProgressLayout = new QGridLayout  ( &progressDialog ); //, 1, 1 ); //, 11, 6, "progressWidgetLayout" );
    pProgressBar = new QProgressBar ( &progressDialog );
    pProgressLayout->addWidget( pProgressBar, 0, 0 );
    progressDialog.resize( QSize(537, 50).expandedTo ( progressDialog.minimumSizeHint ( ) ) );
    progressDialog.show  ( );
  }
  Subtitles::entry *pEntry;
  qsLang = theUtils.iso639  ( m_qsIso639, false );
  progressDialog.setCaption ( QObject::tr ( "Creating %1 Subtitles for %2" ).arg ( qsLang ).arg ( fileInfo.fileName ( ) ) );
  
  qsSubIdx.sprintf ( "/sub%02d_", m_iSubtitleNumber );
  qsThePath = qsBasePath + qsSubIdx;
  
  // get the rect of the video ...
  iWidth  = theUtils.getWHFromResolution ( qsResolution, true  );
  iHeight = theUtils.getWHFromResolution ( qsResolution, false );
  if ( iWidth  < 350 )
       iWidth  = 720;
  if ( iHeight < 240 )
       iHeight = 480;

  // get the rect for the subtitles. Try to be clever about it.
  theRect = getSubtitlesRect ( qsResolution );
  m_rect  = theRect;
  
  // Now we do some magic with colors ...
  // we will set the Background Color to be completely GREEN
  // and we set the foreground color to be completely RED
  // Then we do the drawing
  // After the drawing has been done we wll reduce the colors in the palette to MAX_SUBTITLE_COLORS (4)
  // Lastly we will replace the RED VALUES with the actual requested foreground color.
  // We will use the GREEN color as the transparency value for the color.
  // Note: color0 = background / color1 actual foreground / 
  //       color[2 .. MAX_SUBTITLE_COLORS] = fading foreground to background with delta transparency
  colors[0] = 0x0000FF00; // background
  colors[1] = 0x00FF0000; // foreground
  
  int   iMaxColors = MAX_SUBTITLE_COLORS - 2;
  float fDelta   = (float)0xFF / (iMaxColors+1);
  int   iBGRed, iBGGreen, iBGBlue, iRed, iGreen, iBlue, iTransparency;
  for ( t=0;t<iMaxColors;t++ ) {
    iGreen  = (int)((t+1) * fDelta);
    iRed    = 255 - iGreen;
    colors[t+2] = ( iRed << 16 ) + ( iGreen << 8 );
  }
  
  QRect boundingRect;
  QPen  thePen ( colors[1], 2, Qt::DashDotLine );
  int   iFlags = Qt::DontClip|Qt::WordBreak|Qt::AlignHCenter|Qt::AlignVCenter;
  
  iTotalCount = (int)m_listOfSubtitles.count ();
  // Processing some Qt events / messages
  qApp->processEvents ( 100 ); // do some event processing ...

  int    iDiameter = m_iOutline*2 + 1;
  QImage theBrush;
  if ( m_iOutline > 0 ) {
    // First we need to create a brush to follow ...
    QPixmap  createCircle ( iDiameter, iDiameter ); //, 1 ); // monochrome pixmap
    QPainter circlePainter;
    QPen     circlePen  ( colors[1], 1, Qt::SolidLine );
    circlePainter.begin ( &createCircle ); {
      createCircle.fill ( colors[0] );
      circlePainter.setPen   ( circlePen );
      circlePainter.setBrush ( QBrush ( colors[1] ) );
      circlePainter.drawEllipse ( 0, 0, iDiameter, iDiameter );
    } circlePainter.end ( );
    theBrush = createCircle.convertToImage ( );
    // Then we reduce the colors to the max avail color space.
    theUtils.reduceColors ( theBrush, 2, (QRgb *)&colors );
  }
  
  for (t=0;t<iTotalCount; t++) {
    pEntry  = m_listOfSubtitles[t];
    qsStart = pEntry->qsTimeStart;
    qsStart.remove ( ":"      );
    qsStart.replace( ".", "_" );
    qsFileName = QString ("%1%2.png").arg ( qsThePath ).arg( qsStart );
    
    // First we should fill the drawing board with the propper background  color.
    QPixmap theDrawingBoard ( iWidth, iHeight );
    thePainter.begin ( &theDrawingBoard  ); {
      theDrawingBoard.fill( colors[0] );
      thePainter.setPen   ( thePen );
      thePainter.setBrush ( QBrush ( colors[0] ) );
      thePainter.setFont  ( m_font );
      
      // Here we draw onto the drawing board
      thePainter.drawText ( theRect, iFlags, pEntry->qsText, -1, &boundingRect );
      
      int x, y, w, h, d;
      d = boundingRect.height ( ) - theRect.height ( );
      x = theRect.x ( ); // should be 0
      y = theRect.y ( );
      w = theRect.width  ( ); // should be iWidth
      h = theRect.height ( ); // should be iHeight
      // We should always adjust the height
      if ( m_bFit ) {
	w = boundingRect.width  ( );
	h = boundingRect.height ( );
	y = theRect.y ( ) - (int)( (float)d / 2.0 );

	switch ( m_alignment.x ( ) ) {
	case 2:
	case 4:
	  x = 0;
	  break;
	case 3:
	  x = iWidth - w;
	  break;
	default:
	  x = (int)( ( iWidth - w ) / 2.0 );
	}
      }
      pEntry->rect = QRect ( x, y, w, h );

      // create new dimensioned pixmap
      QPixmap thePixmap = QPixmap ( w, h );
#if ( QT_VERSION > 0x0301FF )
      copyBlt ( &thePixmap, 0, 0, &theDrawingBoard, (int)( ( iWidth - w ) / 2.0 ), y, w, h );
#else
      bitBlt  ( &thePixmap, 0, 0, &theDrawingBoard, (int)( ( iWidth - w ) / 2.0 ), y, w, h, Qt::CopyROP);
#endif
      theImage = thePixmap.convertToImage ();
    }  thePainter.end ();
    // Enable transparency ...
    theImage.setAlphaBuffer ( true );
    
    // Then we reduce the colors to the max avail color space.
    theUtils.reduceColors ( theImage, MAX_SUBTITLE_COLORS, (QRgb *)&colors );
    
    // next we generate the backround color ...
    theColor = m_subColors[0].rgb ( ) & 0x00FFFFFF;
    iBGRed   = qRed   ( theColor );
    iBGGreen = qGreen ( theColor );
    iBGBlue  = qBlue  ( theColor );
    theImage.setColor ( 0, m_subColors[0].rgb ( ) );
    
    // and then the rest of the colors
    theColor = m_subColors[1].rgb ( ) & 0x00FFFFFF;
    iRed     = qRed   ( theColor );
    iGreen   = qGreen ( theColor );
    iBlue    = qBlue  ( theColor );
    // Here we'll blend the max 4 colors for the subtitles
    for ( j=1; j<MAX_SUBTITLE_COLORS; j++ ) {
      int iDelta, r, g, b;
      iTransparency = 255 - (( colors[j] & 0xFF00 ) >> 8); // shift green to get the base transparency value
      // which must be added to the multiplied by the user defined transparency
      fDelta = (float)(255 - m_subColors[j].alpha ( ) ) / 0xFF;
      iTransparency = (int)( fDelta * (255-iTransparency ) ) + iTransparency;
      
      iDelta = ( (colors[j] & 0xFF0000 ) >> 16 );
      fDelta = (float)(0xFF - iDelta) / 0xFF;
      r = iRed   + (int)( ( iBGRed   - iRed   ) * fDelta );
      g = iGreen + (int)( ( iBGGreen - iGreen ) * fDelta );
      b = iBlue  + (int)( ( iBGBlue  - iBlue  ) * fDelta );

      theColor   =qRgba ( r, g, b, iTransparency );
      theImage.setColor ( j, theColor );
    }

    if ( m_iOutline > 0 ) {
      // Next is to inclrease the number of colors to 4
      theImage.setNumColors ( 4 );
      //theImage.setColor ( 3, 0xFF101010 ); // FF == Opaque // 00=Transparent
      theImage.setColor ( 0, 0x00000000 );
      theImage.setColor ( 3, m_subColors[0].rgb ( ) ); //real deal ...
      // Okay, we have outline enabled which means we should replace all background color with either 
      // transparency or with the pixelcolor of the backdrop or with the pixel color of the generated subtitle.
      int w, h, x, y, xb, yb, brushX, brushY, idx;
      w = theImage.width  ( );
      h = theImage.height ( );

      int pIdx;
      for ( x=0; x<w; x++ ) {
	for ( y=0; y<h; y++ ) { // Image x, and y ...
	  // here we take m_iOutline number of pixels before and after an set them to the fourth color.
	  pIdx = theImage.pixelIndex ( x, y );
	  if ( ( pIdx == 1 ) || ( pIdx == 2 ) ) { // not background
	    // So we should underlay the brush ...
	    for ( yb=0; yb<iDiameter; yb++ ) { // copy brush over ...
	      brushY = y - m_iOutline + yb;
	      if ( ( brushY < 0 ) || ( brushY >= h ) )  // out of bounds
		continue;
	      for ( xb=0; xb<iDiameter; xb++ ) {
		brushX = x - m_iOutline + xb; 
		if ( ( brushX < 0 ) || ( brushX >= w ) ) // out of bounds
		  continue;
		idx = theImage.pixelIndex ( brushX, brushY );
 		if ( ( idx == 0 ) && ( theBrush.pixelIndex ( xb, yb ) == 1 ) )
		  theImage.setPixel   ( brushX, brushY, 3 );
	      }
	    }
	  }
	}
      }
    }
    theImage.save ( qsFileName, "PNG", 100 );  // 100%== No compression. png image
    
    if ( ( iTotalCount > 20 ) && ( pProgressBar ) ) {
      pProgressBar->setProgress ( (int)((float)t / iTotalCount * 100.0) );
      if ( t%10 == 0 ) 
	qApp->processEvents ( 100 ); // do some event processing ...
    }
  }
  // that's it ...
  return NULL;
}

// static function to auto extract MetaInfo from a DV video stream in a backround thread
void Subtitles::generateSubtitles ( QObject *pOrigObject, SourceFileInfo *pInfo, QString qsIso639 )
{
  if ( ( ! pInfo ) || ( ! pInfo->bMetaSubtitles ) )
    return;

  QFileInfo  fileInfo   ( pInfo->qsFileName );
  Subtitles *pSubtitles = NULL;
  QString    qsLang, qsXmlFile;
  Utils      theUtils;
  int        t, iFirstNewSubtitleNumber = 0;
  bool       bOkay;
  float      fFPS;

  // First we see if we already have a subtitle with MetaInfo created.
  for ( t=0; t<MAX_SUBTITLES; t++ ) {
    pSubtitles = pInfo->arraySubtitles [ t ];
    if ( pSubtitles ) { 
      if ( ( pSubtitles->m_subtitleState == Subtitles::STATE_META_INFO  ) ||
	   ( pSubtitles->m_subtitleState == Subtitles::STATE_EXTRACTING )  )
	return;
    }
  }
  // lastly we see if we can extract the MetaInfo ( DV only at this time )
  if ( fileInfo.extension ( ).lower ( ) != "dv" )
    return;

  // Okay we do actually want to create subtitles here ...    
  for ( t=0; t<MAX_SUBTITLES; t++ ) {
    iFirstNewSubtitleNumber = t;
    if ( ! pInfo->arraySubtitles [ t ] ) 
      break;
  }

  // Next we should import the subtitles ...
  fFPS =  pInfo->qsFPS.toFloat ( &bOkay );
  if ( ( ! bOkay ) || ( fFPS < 19.0f ) ) {
    fFPS = 29.97f;
    if ( pInfo->qsVideoFormat.upper ( ) == "PAL" )
      fFPS = 25.0f;
  }

  //	sanityCheckSubtitleTrack ( iFirstNewSubtitleNumber );
  if ( ( qsIso639.isEmpty ( ) ) || ( qsIso639.length ( ) != 2 ) ) 
    qsIso639 = "en";
  qsLang = theUtils.iso639 ( qsIso639, false );

  fileInfo.setFile ( pInfo->qsFileName );
  qsXmlFile.sprintf ( "/subtitle_%d.xml", iFirstNewSubtitleNumber );
  qsXmlFile = theUtils.getTempFile ( fileInfo.baseName ( TRUE ) ) + qsXmlFile;

  pSubtitles = new Subtitles;
  // Set subtitle parameters ...
  pSubtitles->m_subtitleState       = Subtitles::STATE_EXTRACTING;
  pSubtitles->m_iSubtitleNumber     = iFirstNewSubtitleNumber;
  pSubtitles->m_qsIso639            = qsIso639;
  pSubtitles->m_qsXmlFile           = qsXmlFile;
  pSubtitles->m_subColors[0]        = Rgba (   0,   0,  0, 0 ); // black background
  pSubtitles->m_subColors[1]        = Rgba ( 250, 250, 10, 0 ); // Yellow font color
  pSubtitles->m_iOutline            = 2;
  pSubtitles->m_font.fromString ( "Sans Serif,24,-1,5,50,0,0,0,0,0" );

  // link it with the SourceFileInfo ...
  if ( pInfo->arraySubtitles[ iFirstNewSubtitleNumber ] )
    delete pInfo->arraySubtitles[ iFirstNewSubtitleNumber ];
  pInfo->arraySubtitles [ iFirstNewSubtitleNumber ] = pSubtitles;

  // And extract the subtitles in a background task ...
  MediaCreator::registerWithMediaScanner ( pOrigObject, pInfo, pSubtitles, fFPS );
}


