/*
** PHREL
** $Id: prefs.c,v 1.26 2006/10/27 03:17:54 sella Exp $
** Copyright (c) 2004 James M. Sella. All Rights Reserved.
** Released under the GPL Version 2 License.
** http://www.digitalgenesis.com
*/

static const char rcsid[] = "$Id: prefs.c,v 1.26 2006/10/27 03:17:54 sella Exp $";

#include "prefs.h"

#include "phrel.h"

#include <snmptrap.h>
#include <configuration.h>

#include <pcap.h>
#include <string.h>
#include <syslog.h>
#include <pthread.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <netinet/in.h>

#define _GNU_SOURCE
#include <getopt.h>

#include <errno.h>
#include <assert.h>

/* Global variables */
extern struct args_t args;
extern int errno;

char const short_options[] = "I:P:p:T:R:B:D:X:Y:A:i:c:s:S:tmj:x:C:u:g:r:o::z::n:d::vh";
struct option const long_options[] = {
	{"interface", required_argument, NULL, 'I'},
	{"protocol", required_argument, NULL, 'P'},
	{"port", required_argument, NULL, 'p'},
	{"threshold", required_argument, NULL, 'T'},
	{"rate", required_argument, NULL, 'R'},
	{"burst", required_argument, NULL, 'B'},
	{"decay", required_argument, NULL, 'D'},
	{"exclude", required_argument, NULL, 'X'},
	{"include", required_argument, NULL, 'Y'},
	{"algo", required_argument, NULL, 'A'},
	{"interval", required_argument, NULL, 'i'},
	{"check-interval", required_argument, NULL, 'c'},
	{"stats-interval", required_argument, NULL, 's'},
	{"snmptrap", required_argument, NULL, 'S'},
	{"test", no_argument, NULL, 't'},
	{"promiscuous", no_argument, NULL, 'm'},
	{"jump", required_argument, NULL, 'j'},
	{"iptables", required_argument, NULL, 'x'},
	{"config", required_argument, NULL, 'C'},
	{"user", required_argument, NULL, 'u'},
	{"group", required_argument, NULL, 'g'},
	{"chroot", required_argument, NULL, 'r'},
	{"stats", optional_argument, NULL, 'o'},
	{"dump", optional_argument, NULL, 'z'},
	{"nice", required_argument, NULL, 'n'},
	{"debug", optional_argument, NULL, 'd'},
	{"version", no_argument, NULL, 'v'},
	{"help", no_argument, NULL, 'h'},
	{NULL, 0, NULL, 0}
};

int get_args(int argc, char** argv) {
	int c, i, len, warn = 0, opt_index = 0;
	char *ptr, buf[512];
	struct threshold_t **t;
	struct prefix_t **p;
	snmptrap_peer_t **s;
	struct in_addr ip;

	/* Default our settings. */
	memset(&args, 0, sizeof(struct args_t));
	args.nice = DEF_NICENESS;
	args.rate = DEF_RATELIMIT;
	args.burst = DEF_BURST;
	args.decay = DEF_DECAY;
	args.algo = DEF_PPS_ALGO;
	args.interval = DEF_INTERVAL;
	args.check_interval = DEF_CHECK_INTERVAL;
	args.stats_interval = DEF_STATS_INTERVAL;
	args.stale_interval = (args.interval * STALE_INTERVAL_MLT);
	args.max_samples = (args.interval + 1);

	/* Set passed arguments. */
	while ((c = getopt_long(argc, argv, short_options, long_options, &opt_index)) != -1) {
		switch (c) {
			case 'P':
				if (args.expression == NULL) {
					snprintf(buf, sizeof(buf), "%s", optarg);
				} else {
					snprintf(buf, sizeof(buf), "%s and %s", args.expression, optarg);
					free(args.expression);
				}
				args.expression = malloc(strlen(buf) + 1);
				strncpy(args.expression, buf, strlen(buf));

				break;

			case 'p':
				if (args.expression == NULL) {
					snprintf(buf, sizeof(buf), "dst port %s", optarg);
				} else {
					snprintf(buf, sizeof(buf), "%s and dst port %s", args.expression, optarg);
					free(args.expression);
				}
				args.expression = malloc(strlen(buf) + 1);
				strncpy(args.expression, buf, strlen(buf));

				break;
				
			case 'I':
				if (args.interface == NULL) {
					args.interface = (char*) strdup(optarg);
				}

				break;

			case 'T': /* --threshold=PPS[:RATE[:BURST[:DECAY]]] */
				/* FIXME: If thresholds are specified by the user non-ascending, do they get used?? */
				t = &(args.threshold);
				while ((*t) != NULL) t = &((*t)->next); /* Move to the end of threshold list */
				*t = malloc(sizeof(struct threshold_t));
				memset(*t, 0, sizeof(struct threshold_t));

				(*t)->threshold = strtoul(optarg, NULL, 10);

				if ((*t)->threshold > MAX_THRESHOLD) {
					(*t)->threshold = MAX_THRESHOLD;
				}

				if ((ptr = strchr(optarg, ':')) != NULL && strlen(ptr) > 1) { /* RATE */
					ptr++;
					(*t)->rate = strtoul(ptr, NULL, 10);

					if ((*t)->rate > MAX_RATELIMIT) {
						(*t)->rate = MAX_RATELIMIT;
					}

					if ((ptr = strchr(ptr, ':')) != NULL && strlen(ptr) > 1) { /* BURST */
						ptr++;
						(*t)->burst = strtoul(ptr, NULL, 10);

						if ((*t)->burst > MAX_BURST) {
							(*t)->burst = MAX_BURST;
						}

						if ((ptr = strchr(ptr, ':')) != NULL && strlen(ptr) > 1) { /* DECAY */
							ptr++;
							(*t)->decay = strtoul(ptr, NULL, 10);

							if ((*t)->decay > MAX_DECAY) {
								(*t)->decay = MAX_DECAY;
							}
						} else {
							(*t)->decay = USE_DEFAULT;
						}
					} else {
						(*t)->burst = USE_DEFAULT;
						(*t)->decay = USE_DEFAULT;
					}
				} else {
					(*t)->rate = USE_DEFAULT;
					(*t)->burst = USE_DEFAULT;
					(*t)->decay = USE_DEFAULT;
				}

				break;
				
			case 'R':
				args.rate = strtoul(optarg, NULL, 10);

				if (args.rate > MAX_RATELIMIT) {
					args.rate = MAX_RATELIMIT;
				}

				break;
				
			case 'B':
				args.burst = strtoul(optarg, NULL, 10);

				if (args.burst > MAX_BURST) {
					args.burst = MAX_BURST;
				}

				break;
				
			case 'D':
				args.decay = strtoul(optarg, NULL, 10);

				if (args.decay > MAX_DECAY) {
					args.decay = MAX_DECAY;
				}

				break;
				
			case 'X':
				p = &(args.exclude);
				while ((*p) != NULL) p = &((*p)->next); /* Move to the end of exclude list */
				*p = malloc(sizeof(struct prefix_t));
				memset(*p, 0, sizeof(struct prefix_t));

				if ((ptr = strchr(optarg, '/')) != NULL && strlen(ptr) > 1) { /* Mask */
					*ptr = '\0'; /* Separate mask from ip */
					ptr++; /* Move past seperator */

					(*p)->cidr = (strtoul(ptr, NULL, 10) < 32) ? strtoul(ptr, NULL, 10) : 32;
					(*p)->mask.s_addr = ((*p)->cidr == 0) ? 0 : htonl(0xFFFFFFFF << (32 - (*p)->cidr));
				} else {
					(*p)->mask.s_addr = htonl(0xFFFFFFFF);
					(*p)->cidr = 32;
				}

				if (inet_aton(optarg, &ip) == 0) {
					fprintf(stderr, "excluded prefix '%s' has an invalid format\n", optarg);

					return EXIT_FAILURE;
				}

				(*p)->ip.s_addr = ip.s_addr & (*p)->mask.s_addr;

				break;
				
			case 'Y':
				p = &(args.include);
				while ((*p) != NULL) p = &((*p)->next); /* Move to the end of include list */
				*p = malloc(sizeof(struct prefix_t));
				memset(*p, 0, sizeof(struct prefix_t));

				if ((ptr = strchr(optarg, '/')) != NULL && strlen(ptr) > 1) { /* Mask */
					*ptr = '\0'; /* Separate mask from ip */
					ptr++; /* Move past seperator */

					(*p)->cidr = (strtoul(ptr, NULL, 10) < 32) ? strtoul(ptr, NULL, 10) : 32;
					(*p)->mask.s_addr = ((*p)->cidr == 0) ? 0 : htonl(0xFFFFFFFF << (32 - (*p)->cidr));
				} else {
					(*p)->mask.s_addr = htonl(0xFFFFFFFF);
					(*p)->cidr = 32;
				}

				if (inet_aton(optarg, &ip) == 0) {
					fprintf(stderr, "included prefix '%s' has an invalid format\n", optarg);

					return EXIT_FAILURE;
				}

				(*p)->ip.s_addr = ip.s_addr & (*p)->mask.s_addr;

				break;
				
			case 'A':
				for (i = 0; i < MAX_ALGO_NAMES; i++) {
					if (strncasecmp(optarg, algo_names[i], strlen(algo_names[i])) == 0) {
						args.algo = i;

						break;
					}
				}

				break;
				
			case 'i':
				args.interval = strtoul(optarg, NULL, 10);

				if (args.interval > MAX_INTERVAL) {
					args.interval = MAX_INTERVAL;
				} else if (args.interval < MIN_INTERVAL) {
					args.interval = MIN_INTERVAL;
				}

				args.stale_interval = (args.interval * STALE_INTERVAL_MLT);
				args.max_samples = (args.interval + 1);

				break;
				
			case 'c':
				args.check_interval = strtoul(optarg, NULL, 10);

				if (args.check_interval > MAX_CHECK_INTERVAL) {
					args.check_interval = MAX_CHECK_INTERVAL;
				} else if (args.check_interval < MIN_CHECK_INTERVAL) {
					args.check_interval = MIN_CHECK_INTERVAL;
				}

				break;
				
			case 's':
				args.stats_interval = strtoul(optarg, NULL, 10);

				if (args.stats_interval > MAX_STATS_INTERVAL) {
					args.stats_interval = MAX_STATS_INTERVAL;
				} else if (args.stats_interval < MIN_STATS_INTERVAL) {
					args.stats_interval = MIN_STATS_INTERVAL;
				}

				break;
				
			case 'S': /* --snmptrap=HOST[:PORT][:COMMUNITY[:VERSION[:INFORM]]] */
#ifdef HAVE_NET_SNMP
				s = &(args.peer);
				while ((*s) != NULL) s = &((*s)->next); /* Move to the end of peer list */
				*s = malloc(sizeof(snmptrap_peer_t));
				memset(*s, 0, sizeof(snmptrap_peer_t));
				(*s)->version = (*s)->inform = -1;

				ptr = strtok(optarg, ":");
				while (ptr != NULL) {
					if ((*s)->peername[0] == '\0') { /* HOST */
						strncpy((*s)->peername, ptr, sizeof((*s)->peername));

						/* Check if PORT is following hostname */
						if ((ptr = strtok(NULL, ":")) != NULL) { 
							/* Assume it's a PORT if its only numbers */
							snprintf(buf, sizeof(buf), "%u", atoi(ptr));
							if (strcmp(ptr, buf) == 0) { /* PORT */
								if (atoi(ptr) >= 1 && atoi(ptr) <= 65535) { /* Ignore invalid range */
									snprintf(buf, sizeof(buf), "%s:%u", (*s)->peername, atoi(ptr));
									strncpy((*s)->peername, buf, sizeof((*s)->peername));
								} else if (warn == 0) {
									fprintf(stderr, "Specified port '%s' for '--snmptrap' is invalid and will be ignored.\n", ptr);
									warn = 1;
								}
							} else {
								continue;
							}
						}
					} else if ((*s)->community[0] == '\0') { /* COMMUNITY */
						strncpy((*s)->community, ptr, sizeof((*s)->community));
					} else if ((*s)->version == -1) { /* VERSION */
						if (strcasecmp(ptr, "1") == 0 || strcasecmp(ptr, "v1") == 0) {
							(*s)->version = SNMP_VERSION_1;
						} else if (strcasecmp(ptr, "3") == 0 || strcasecmp(ptr, "v3") == 0) {
							(*s)->version = SNMP_VERSION_3;
						} else {
							(*s)->version = SNMP_VERSION_2c;
						}
					} else if ((*s)->inform == -1) { /* INFORM */
						if (strncasecmp(ptr, "y", 1) == 0 || strcasecmp(ptr, "on") == 0 || strcasecmp(ptr, "1") == 0) { /* Yes, On or 1 */
							(*s)->inform = 1;
						} else {
							(*s)->inform = 0;
						}
					} else {
						break;
					}

					/* Next token */
					ptr = strtok(NULL, ":");
				}

				/* Set COMMUNITY default, if needed */
				if ((*s)->community[0] == '\0') {
					strncpy((*s)->community, "public", sizeof((*s)->community));
				}

				/* Set VERSION default, if needed */
				if ((*s)->version == -1) {
					(*s)->version = SNMP_VERSION_2c;
				}

				/* Set INFORM default, if needed */
				if ((*s)->inform == -1) {
					(*s)->inform = 0;
				}
#else
				if (warn == 0) {
					fprintf(stderr, "Support for '--snmptrap' is not compiled into the binary.\n");
					warn = 1;
				}
#endif /* HAVE_NET_SNMP */

				break;
				
			case 't':
				args.test = 1;

				break;
				
			case 'm':
				args.promisc = 1;

				break;
				
			case 'j':
				if (args.jump == NULL) {
					args.jump = (char*) strdup(optarg);
				}

				break;

			case 'x':
				if (args.iptables == NULL) {
					args.iptables = (char*) strdup(optarg);
				}

				break;

			case 'C':
				if (args.config == NULL) {
					args.config = (char*) strdup(optarg);
				}

				break;

			case 'u':
				if (args.user == NULL) {
					args.user = (char*) strdup(optarg);
				}

				break;

			case 'g':
				if (args.group == NULL) {
					args.group = (char*) strdup(optarg);
				}

				break;

			case 'r':
				if (args.chroot == NULL) {
					args.chroot = (char*) strdup(optarg);
				}

				break;

			case 'o':
				if (args.stats == NULL) {
					if (optarg) {
						args.stats = (char*) strdup(optarg);
					} else {
						args.stats = (char*) strdup(DEF_STATS);
					}
				}

				break;

			case 'z':
				if (args.dump == NULL) {
					if (optarg) {
						args.dump = (char*) strdup(optarg);
					} else {
						args.dump = (char*) strdup(DEF_DUMP);
					}
				}

				break;

			case 'n':
				if (optarg) {
					args.nice = atoi(optarg);
				}

				if (args.nice < -19 || args.nice > 20) {
					args.nice = DEF_NICENESS;
				}

				break;
				
			case 'd':
				if (optarg) {
					args.debug = strtoul(optarg, NULL, 10);

					if (args.debug > DEBUG_MAXVAL) {
						args.debug = DEBUG_MAXVAL;
					}
				} else {
					args.debug = DEBUG_ERROR;
				}

				break;

			case 'v':
				version(0);

				return EXIT_FAILURE;

			default:
			case 'h':
				usage();
				version(1);

				return EXIT_FAILURE;
		}
	}

	/* Exclude interface addresses */
	exclude_interfaces(&(args.exclude));

	if (args.jump == NULL) {
		args.jump = (char*) strdup(DEF_JUMP);
	}

	if (args.iptables == NULL) {
		args.iptables = (char*) strdup(DEF_IPTABLES);
	}

	if (optind < argc) {
		if (args.expression != NULL) {
			fprintf(stderr, "expression has been overridden by specified port and/or protocol\n");
		} else {
			args.expression = malloc(PCAP_EXPRESSION_SIZE);
			args.expression[0] = '\0';
			while (optind < argc) {
				if ((len = PCAP_EXPRESSION_SIZE - strlen(args.expression) - strlen(argv[optind])) > 0) {
					strncat(args.expression, argv[optind++], len);
					strncat(args.expression, " ", 1);
				} else {
					fprintf(stderr, "expression's size exceeds the internal buffer of (%u bytes)\n", PCAP_EXPRESSION_SIZE);

					return EXIT_FAILURE;
				}
			}
		}
	}

	return EXIT_SUCCESS;
}

int process_config(struct config_node *node) {
	int i, warn = 0, valid;
	char buf[512], *ptr, *protocol = NULL, *port = NULL, *snmpport = NULL;
	struct threshold_t **t;
	struct prefix_t **p;
	snmptrap_peer_t **s;
	struct in_addr ip;
	struct config_node *n;

	while (node != NULL) {
		/* Interface */
		if (strcasecmp(node->var, "interface") == 0 && node->val != NULL) {
			if (args.interface == NULL) {
				args.interface = (char*) strdup(node->val);
			}

		/* Protocol - Used to build expression */
		} else if (strcasecmp(node->var, "protocol") == 0 && node->val != NULL) {
			if (protocol) free(protocol);
			protocol = strdup(node->val);

		/* Port - Used to build expression */
		} else if (strcasecmp(node->var, "port") == 0 && node->val != NULL) {
			if (port) free(port);
			port = strdup(node->val);

		/* Expression - tcpdump format */
		} else if (strcasecmp(node->var, "expression") == 0 && node->val != NULL) {
			if (args.expression) free(args.expression);
			args.expression = strdup(node->val);

		/* Threshold */
		} else if (strcasecmp(node->var, "threshold") == 0) {
			t = &(args.threshold);
			while ((*t) != NULL) t = &((*t)->next); /* Move to the end of threshold list */
			*t = malloc(sizeof(struct threshold_t));
			memset(*t, 0, sizeof(struct threshold_t));

			(*t)->rate = USE_DEFAULT;
			(*t)->burst = USE_DEFAULT;
			(*t)->decay = USE_DEFAULT;

			if (node->val != NULL) {
				(*t)->threshold = strtoul(node->val, NULL, 10);

				if ((*t)->threshold > MAX_THRESHOLD) {
					(*t)->threshold = MAX_THRESHOLD;
				}
			}

			n = node->sub;
			while (n != NULL) {
				if (strcasecmp(n->var, "pps") == 0 && n->val != NULL) {
					(*t)->threshold = strtoul(n->val, NULL, 10);

					if ((*t)->threshold > MAX_THRESHOLD) {
						(*t)->threshold = MAX_THRESHOLD;
					}
				} else if (strcasecmp(n->var, "rate") == 0 && n->val != NULL) {
					(*t)->rate = strtoul(n->val, NULL, 10);

					if ((*t)->rate > MAX_RATELIMIT) {
						(*t)->rate = MAX_RATELIMIT;
					}
				} else if (strcasecmp(n->var, "burst") == 0 && n->val != NULL) {
					(*t)->burst = strtoul(n->val, NULL, 10);

					if ((*t)->burst > MAX_BURST) {
						(*t)->burst = MAX_BURST;
					}
				} else if (strcasecmp(n->var, "decay") == 0 && n->val != NULL) {
					(*t)->decay = strtoul(n->val, NULL, 10);

					if ((*t)->decay > MAX_DECAY) {
						(*t)->decay = MAX_DECAY;
					}
				} else {
					syslog(LOG_INFO, "invalid option: <%s> %s", node->var, n->var);
				}

				n = n->next;
			}

			/* Remove threshold if not valid */
			if ((*t)->threshold == 0) {
				syslog(LOG_INFO, "invalid option: <%s> (missing 'pps')", node->var);
				free(*t);
				*t = NULL;
			}

		/* Rate - default value */
		} else if (strcasecmp(node->var, "rate") == 0 && node->val != NULL) {
			if (args.rate == DEF_RATELIMIT) {
				args.rate = strtoul(node->val, NULL, 10);

				if (args.rate > MAX_RATELIMIT) {
					args.rate = MAX_RATELIMIT;
				}
			}

		/* Burst - default value */
		} else if (strcasecmp(node->var, "burst") == 0 && node->val != NULL) {
			if (args.burst == DEF_BURST) {
				args.burst = strtoul(node->val, NULL, 10);

				if (args.burst > MAX_BURST) {
					args.burst = MAX_BURST;
				}
			}

		/* Decay - default value */
		} else if (strcasecmp(node->var, "decay") == 0 && node->val != NULL) {
			if (args.decay == DEF_DECAY) {
				args.decay = strtoul(node->val, NULL, 10);

				if (args.decay > MAX_DECAY) {
					args.decay = MAX_DECAY;
				}
			}

		/* Exclude - ignore traffic from exclusions */
		} else if (strcasecmp(node->var, "exclude") == 0) {
			p = &(args.exclude);
			while ((*p) != NULL) p = &((*p)->next); /* Move to the end of exclude list */

			n = node->sub;
			while (n != NULL) {
				*p = malloc(sizeof(struct prefix_t));
				memset(*p, 0, sizeof(struct prefix_t));
				valid = 0;

				if (strcasecmp(n->var, "prefix") == 0 && n->val != NULL) {
					if ((ptr = strchr(n->val, '/')) != NULL && strlen(ptr) > 1) { /* Mask */
						*ptr = '\0'; /* Separate mask from ip */
						ptr++; /* Move past seperator */

						(*p)->cidr = (strtoul(ptr, NULL, 10) < 32) ? strtoul(ptr, NULL, 10) : 32;
						(*p)->mask.s_addr = ((*p)->cidr == 0) ? 0 : htonl(0xFFFFFFFF << (32 - (*p)->cidr));
					} else {
						(*p)->mask.s_addr = htonl(0xFFFFFFFF);
						(*p)->cidr = 32;
					}

					if (inet_aton(n->val, &ip) == 0) {
						syslog(LOG_INFO, "excluded prefix '%s' has an invalid format\n", n->val);
					} else {
						(*p)->ip.s_addr = ip.s_addr & (*p)->mask.s_addr;

						valid = 1;
					}
				} else {
					syslog(LOG_INFO, "invalid option: <%s> %s", node->var, n->var);
				}

				if (valid == 1) { /* If valid, move to next prefix */
					p = &((*p)->next);
				} else { /* Remove invalid entry */
					syslog(LOG_INFO, "invalid option: <%s> (missing 'prefix')", node->var);
					free(*p);
					*p = NULL;
				}

				/* If end if sub list, append block value prefix */
				if (n->next == NULL && node->val != NULL) {
					n->next = malloc(sizeof(struct config_node));
					memset(n->next, 0, sizeof(struct config_node));

					n->next->var = strdup("prefix");
					n->next->val = strdup(node->val);
					free(node->val);
					node->val = NULL;
				}

				n = n->next;
			}
		/* Include - include specified prefix rather than all traffic */
		} else if (strcasecmp(node->var, "include") == 0) {
			p = &(args.include);
			while ((*p) != NULL) p = &((*p)->next); /* Move to the end of exclude list */

			n = node->sub;
			while (n != NULL) {
				*p = malloc(sizeof(struct prefix_t));
				memset(*p, 0, sizeof(struct prefix_t));
				valid = 0;

				if (strcasecmp(n->var, "prefix") == 0 && n->val != NULL) {
					if ((ptr = strchr(n->val, '/')) != NULL && strlen(ptr) > 1) { /* Mask */
						*ptr = '\0'; /* Separate mask from ip */
						ptr++; /* Move past seperator */

						(*p)->cidr = (strtoul(ptr, NULL, 10) < 32) ? strtoul(ptr, NULL, 10) : 32;
						(*p)->mask.s_addr = ((*p)->cidr == 0) ? 0 : htonl(0xFFFFFFFF << (32 - (*p)->cidr));
					} else {
						(*p)->mask.s_addr = htonl(0xFFFFFFFF);
						(*p)->cidr = 32;
					}

					if (inet_aton(n->val, &ip) == 0) {
						syslog(LOG_INFO, "included prefix '%s' has an invalid format\n", n->val);
					} else {
						(*p)->ip.s_addr = ip.s_addr & (*p)->mask.s_addr;

						valid = 1;
					}
				} else {
					syslog(LOG_INFO, "invalid option: <%s> %s", node->var, n->var);
				}

				if (valid == 1) { /* If valid, move to next prefix */
					p = &((*p)->next);
				} else { /* Remove invalid entry */
					syslog(LOG_INFO, "invalid option: <%s> (missing 'prefix')", node->var);
					free(*p);
					*p = NULL;
				}

				/* If end if sub list, append block value prefix */
				if (n->next == NULL && node->val != NULL) {
					n->next = malloc(sizeof(struct config_node));
					memset(n->next, 0, sizeof(struct config_node));

					n->next->var = strdup("prefix");
					n->next->val = strdup(node->val);
					free(node->val);
					node->val = NULL;
				}

				n = n->next;
			}
		/* Algorithim used to caculate pps */
		} else if (strcasecmp(node->var, "algo") == 0 && node->val != NULL) {
			if (args.algo == DEF_PPS_ALGO) {
				for (i = 0; i < MAX_ALGO_NAMES; i++) {
					if (strncasecmp(node->val, algo_names[i], strlen(algo_names[i])) == 0) {
						args.algo = i;

						break;
					}
				}
			}

		/* Interval - Length of time to calculate PPS over. */
		} else if (strcasecmp(node->var, "interval") == 0 && node->val != NULL) {
			if (args.interval == DEF_INTERVAL) {
				args.interval = strtoul(node->val, NULL, 10);

				if (args.interval > MAX_INTERVAL) {
					args.interval = MAX_INTERVAL;
				} else if (args.interval < MIN_INTERVAL) {
					args.interval = MIN_INTERVAL;
				}

				args.stale_interval = (args.interval * STALE_INTERVAL_MLT);
				args.max_samples = (args.interval + 1);
			}

		} else if (strcasecmp(node->var, "calc-interval") == 0 && node->val != NULL) {
			if (args.check_interval == DEF_CHECK_INTERVAL) {
				args.check_interval = strtoul(node->val, NULL, 10);

				if (args.check_interval > MAX_CHECK_INTERVAL) {
					args.check_interval = MAX_CHECK_INTERVAL;
				} else if (args.check_interval < MIN_CHECK_INTERVAL) {
					args.check_interval = MIN_CHECK_INTERVAL;
				}
			}

		} else if (strcasecmp(node->var, "stats-interval") == 0 && node->val != NULL) {
			if (args.stats_interval == DEF_STATS_INTERVAL) {
				args.stats_interval = strtoul(node->val, NULL, 10);

				if (args.stats_interval > MAX_STATS_INTERVAL) {
					args.stats_interval = MAX_STATS_INTERVAL;
				} else if (args.stats_interval < MIN_STATS_INTERVAL) {
					args.stats_interval = MIN_STATS_INTERVAL;
				}
			}

		} else if (strcasecmp(node->var, "snmptrap") == 0) {
#ifdef HAVE_NET_SNMP
			s = &(args.peer);
			while ((*s) != NULL) s = &((*s)->next); /* Move to the end of peer list */
			*s = malloc(sizeof(snmptrap_peer_t));
			memset(*s, 0, sizeof(snmptrap_peer_t));

			strncpy((*s)->community, "public", sizeof((*s)->community));
			(*s)->version = SNMP_VERSION_2c;

			/* Capture peername (if available) */
			if (node->val != NULL) {
				strncpy((*s)->peername, node->val, sizeof((*s)->peername));
			}

			n = node->sub;
			while (n != NULL) {
				if (strcasecmp(n->var, "host") == 0 && n->val != NULL) {
					strncpy((*s)->peername, n->val, sizeof((*s)->peername));
				} else if (strcasecmp(n->var, "port") == 0 && n->val != NULL) {
					if (atoi(n->val) >= 1 && atoi(n->val) <= 65535) { /* Ignore invalid range */
						if (snmpport) free(snmpport);
						snmpport = strdup(n->val);
					}
				} else if (strcasecmp(n->var, "community") == 0 && n->val != NULL) {
					strncpy((*s)->community, n->val, sizeof((*s)->community));
				} else if (strcasecmp(n->var, "version") == 0 && n->val != NULL) {
					if (strcasecmp(n->val, "1") == 0 || strcasecmp(n->val, "v1") == 0) {
						(*s)->version = SNMP_VERSION_1;
					} else if (strcasecmp(n->val, "3") == 0 || strcasecmp(n->val, "v3") == 0) {
						(*s)->version = SNMP_VERSION_3;
					} else {
						(*s)->version = SNMP_VERSION_2c;
					}
				} else if (strcasecmp(n->var, "inform") == 0) {
					if (n->val == NULL) { /* Keyword only - enabled */
						(*s)->inform = 1;
					} else if (strncasecmp(n->val, "y", 1) == 0 || strcasecmp(n->val, "on") == 0 || strcasecmp(n->val, "1") == 0) { /* Yes, On or 1 */
						(*s)->inform = 1;
					} else {
						(*s)->inform = 0;
					}
				} else {
					syslog(LOG_INFO, "invalid option: <%s> %s", node->var, n->var);
				}

				n = n->next;
			}

			/* Remove peer if not valid */
			if ((*s)->peername[0] == '\0') {
				syslog(LOG_INFO, "invalid option: <%s> (missing 'host')", node->var);
				free(*s);
				*s = NULL;
			} else if (snmpport != NULL && strchr((*s)->peername, ':') == NULL) {
				snprintf(buf, sizeof(buf), "%s:%u", (*s)->peername, atoi(snmpport));
				strncpy((*s)->peername, buf, sizeof((*s)->peername));
			}

			if (snmpport) {
				free(snmpport);
				snmpport = NULL;
			}
#else
			if (warn == 0) {
				fprintf(stderr, "Support for '--snmptrap' is not compiled into the binary.\n");
				warn = 1;
			}
#endif /* HAVE_NET_SNMP */
		/* Test mode */
		} else if (strcasecmp(node->var, "test") == 0) {
			args.test = 1;
		/* Promiscuous mode */
		} else if (strcasecmp(node->var, "promiscuous") == 0) {
			args.promisc = 1;

		} else if (strcasecmp(node->var, "jump") == 0 && node->val != NULL) {
			if (args.jump == NULL) {
				args.jump = strdup(node->val);
			}

		} else if (strcasecmp(node->var, "iptables") == 0 && node->val != NULL) {
			if (args.iptables == NULL) {
				args.iptables = strdup(node->val);
			}

		} else if (strcasecmp(node->var, "user") == 0 && node->val != NULL) {
			if (args.user == NULL) {
				args.user = strdup(node->val);
			}

		} else if (strcasecmp(node->var, "group") == 0 && node->val != NULL) {
			if (args.group == NULL) {
				args.group = strdup(node->val);
			}

		} else if (strcasecmp(node->var, "chroot") == 0 && node->val != NULL) {
			if (args.chroot == NULL) {
				args.chroot = strdup(node->val);
			}

		} else if (strcasecmp(node->var, "stats") == 0) {
			if (args.stats == NULL) {
				if (node->val) {
					args.stats = (char*) strdup(node->val);
				} else {
					args.stats = (char*) strdup(DEF_STATS);
				}
			}

		} else if (strcasecmp(node->var, "dump") == 0) {
			if (args.dump == NULL) {
				if (node->val) {
					args.dump = (char*) strdup(node->val);
				} else {
					args.dump = (char*) strdup(DEF_DUMP);
				}
			}

		} else if (strcasecmp(node->var, "nice") == 0 && node->val != NULL) {
			if (args.nice == DEF_NICENESS) {
				args.nice = atoi(node->val);

				if (args.nice < -19 || args.nice > 20) {
					args.nice = DEF_NICENESS;
				}
			}

		} else if (strcasecmp(node->var, "debug") == 0) {
			if (args.debug == 0) {
				if (node->val != NULL) {
					args.debug = strtoul(node->val, NULL, 10);

					if (args.debug > DEBUG_MAXVAL) {
						args.debug = DEBUG_MAXVAL;
					}
				} else {
					args.debug = DEBUG_ERROR;
				}
			}

		} else {
			syslog(LOG_INFO, "invalid option: %s", node->var);
		}

		node = node->next;
	}

	/* Build expression from protocol and port if not set. */
	if (args.expression == NULL) {
		if (protocol != NULL && port != NULL) {
			snprintf(buf, sizeof(buf), "%s and dst port %s", protocol, port);
			args.expression = malloc(strlen(buf) + 1);
			strncpy(args.expression, buf, strlen(buf));
		} else if (protocol != NULL) {
			args.expression = strdup(protocol);
		} else if (port != NULL) {
			snprintf(buf, sizeof(buf), "dst port %s", port);
			args.expression = malloc(strlen(buf) + 1);
			strncpy(args.expression, buf, strlen(buf));
		}
	}

	/* Cleanup */
	if (protocol) free(protocol);
	if (port) free(port);

	return 0;
}

int exclude_interfaces(struct prefix_t **exclude) {
	char errbuf[PCAP_ERRBUF_SIZE];
	char buf[32];
	pcap_if_t* pcap_if;
	pcap_if_t *alldevs = NULL;
	pcap_addr_t* pcap_addr;
	struct sockaddr_in *sin;
	struct prefix_t **p;

	if (pcap_findalldevs(&alldevs, errbuf) == -1) {
		fprintf(stderr, "pcap_findalldevs(): %s\n", errbuf);

		return 1;
	}

	p = exclude;
	while ((*p) != NULL) p = &((*p)->next); /* Move to the end of exclude list */

	pcap_if = alldevs;
	while (pcap_if != NULL) {
		pcap_addr = pcap_if->addresses;
		while (pcap_addr != NULL) {
			sin = (struct sockaddr_in*) pcap_addr->addr;
			if (pcap_addr->addr != NULL && pcap_addr->netmask != NULL && sin->sin_addr.s_addr != 0) {
				if (args.debug >= DEBUG_INTERNAL) {
					inet_ntop(AF_INET, &(sin->sin_addr), buf, sizeof(buf)); /* REMOVE */
					fprintf(stderr, "Excluding %s/32 (interface %s)\n", buf, pcap_if->name);
				}

				*p = malloc(sizeof(struct prefix_t));
				memset(*p, 0, sizeof(struct prefix_t));

				(*p)->ip.s_addr = sin->sin_addr.s_addr;
				(*p)->mask.s_addr = htonl(0xFFFFFFFF);
				(*p)->cidr = 32;
				(*p)->interface = 1;

				/* Update end of list */
				p = &((*p)->next);
			}
			pcap_addr = pcap_addr->next;
		}
		pcap_if = pcap_if->next;
	}

	if (alldevs) {
		pcap_freealldevs(alldevs);
	}

	return 0;
}

void usage() {
	printf("Usage: %s [OPTION] <-T PPS[:RATE[:BURST[:DECAY]]] ...> [expression]\n", PROGRAM_NAME);
	printf("Per Host RatE Limiter (%s) - Generates and updates iptables chains for hosts\nwhich cross the configured threshold(s).\n", PROGRAM_PKGN);
	printf("\nPreferences:\n");
	printf("  -I, --interface=INTERFACE           monitor traffic on interface (any)\n");
	printf("  -P, --protocol=PROTO                monitor traffic for protocol (any)\n");
	printf("  -p, --port=PORT                     monitor traffic on port (any)\n");
	printf("  -T, --threshold=PPS[:R[:B[:D]]]     user threshold (-:%d:%d:%d)\n", DEF_RATELIMIT, DEF_BURST, DEF_DECAY);
	printf("  -R, --rate=PPS                      use rate limit (%d)\n", DEF_RATELIMIT);
	printf("  -B, --burst=PACKETS                 use burst for rate limiter (%d)\n", DEF_BURST);
	printf("  -D, --decay=SECONDS                 use decay (%d)\n", DEF_DECAY);
	printf("  -X, --exclude=PREFIX[/CIDR]         exclude packets from prefix (none)\n");
	printf("  -Y, --include=PREFIX[/CIDR]         include packets from prefix (any)\n");
	printf("  -A, --algo=<avg|sum|max>            use algorithim to calculate PPS (avg)\n");
	printf("  -i, --interval=SEC                  interval to calculate PPS over (%d)\n", DEF_INTERVAL);
	printf("  -c, --check-interval=SEC            interval to check thresholds (%d)\n", DEF_CHECK_INTERVAL);
	printf("  -s, --stats-interval=SEC            interval to update stats file (%d)\n", DEF_STATS_INTERVAL);
	printf("  -S, --snmptrap=HOST[:P][:C[:V[:I]]] send snmptraps to host (none)\n");
	printf("  -t, --test                          test mode - do not update iptables (off)\n");
	printf("  -m, --promiscuous                   promiscuous mode (off)\n");
	printf("  -j, --jump=CHAIN                    jump to user defined chain (%s)\n", DEF_JUMP);
	printf("  -x, --iptables=PATH                 full path to iptables (%s)\n", DEF_IPTABLES);
	printf("\nProcess:\n");
	printf("  -C, --config=FILE                   use FILE for configuration (none)\n");
	printf("  -u, --user=USER                     run process as USER (current)\n");
	printf("  -g, --group=GROUP                   run process as GROUP (current)\n");
	printf("  -r, --chroot=DIR                    change root to DIR (none)\n");
	printf("  -o, --stats[=FILE]                  write stats to FILE (%s)\n", strlen(DEF_STATS) > 1 ? DEF_STATS : "none");
	printf("  -z, --dump[=FILE]                   write dump to FILE (%s)\n", strlen(DEF_DUMP) > 1 ? DEF_DUMP : "none");
	printf("  -n, --nice=[+/-]NUM                 set process niceness (%s%d)\n", (DEF_NICENESS) >= 0 ? "+" : "-", DEF_NICENESS);
	printf("  -d, --debug[=LEVEL]                 enable debug and remain in foreground (off)\n");
	printf("\nGeneral:\n");
	printf("  -v, --version                       displays version information\n");
	printf("  -h, --help                          displays this help information\n\n");
	printf("Examples:\n\n");
	printf("  %s -p 53 -T 50:5 -T 75\n", PROGRAM_NAME);
	printf("  %s -p 53 -T 25:20 -T 50:10 -T 75 --snmptrap=192.168.33.1:public:v2\n", PROGRAM_NAME);
	printf("  %s -I eth0 -P tcp -p 25 -T 25:20:10:300 -T 50:10:5 -T 75:0\n", PROGRAM_NAME);
	printf("  %s -I eth0 -p 53 -T 100 -B 10 -D 1800 -X 192.168.33.1 -X 192.168.55.0/24\n", PROGRAM_NAME);
	printf("  %s -T 50 'udp dst port 53 and not (dst net 192.168.33.1 or 192.168.55.0/24)'\n", PROGRAM_NAME);
	printf("\n");
}

void version(int flag) {
	if (flag == 0 || flag == 2) {
		printf("%s v%s\n", PROGRAM_NAME, PROGRAM_VERS);
	}
	printf("Written by James M. Sella <%s>\n", PROGRAM_BUGS);
	if (!flag) {
		printf("\nCopyright (c) 2005-2006 James M. Sella. All rights reserved.\n");
		printf("This is free software; see the source for copying conditions. There is NO\nwarranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\n");
	}
}

void display_args() {
	char buf[32], buf2[32];
	struct threshold_t *t;
	struct prefix_t *p;
	snmptrap_peer_t *s;

	syslog(LOG_DEBUG, "[Arguments]");

	if (args.interface) {
		syslog(LOG_DEBUG, " interface: %s", args.interface);
	} else {
		syslog(LOG_DEBUG, " interface: any");
	}

	if ((t = args.threshold) != NULL) {
		while (t != NULL) {
			syslog(LOG_DEBUG, " threshold: %u:%u:%u:%u (threshold:rate:burst:decay)", t->threshold, (t->rate == USE_DEFAULT) ? args.rate : t->rate, (t->burst == USE_DEFAULT) ? args.burst : t->burst, (t->decay == USE_DEFAULT) ? args.decay : t->decay);
			t = t->next;
		}
	} else {
		syslog(LOG_DEBUG, " threshold: none");
	}

	syslog(LOG_DEBUG, " rate: %u", args.rate);
	syslog(LOG_DEBUG, " burst: %u", args.burst);
	syslog(LOG_DEBUG, " decay: %u", args.decay);
	
	if ((p = args.exclude) != NULL) {
		while (p != NULL) {
			inet_ntop(AF_INET, &(p->mask), buf2, sizeof(buf2));
			inet_ntop(AF_INET, &(p->ip), buf, sizeof(buf));
			syslog(LOG_DEBUG, " exclude: %s/%u%s (/%u == %s)", buf, p->cidr, (p->interface) ? "*" : "", p->cidr, buf2);
			p = p->next;
		}
	} else {
		syslog(LOG_DEBUG, " exclude: none");
	}

	if ((p = args.include) != NULL) {
		while (p != NULL) {
			inet_ntop(AF_INET, &(p->mask), buf2, sizeof(buf2));
			inet_ntop(AF_INET, &(p->ip), buf, sizeof(buf));
			syslog(LOG_DEBUG, " include: %s/%u (/%u == %s)", buf, p->cidr, p->cidr, buf2);
			p = p->next;
		}
	} else {
		syslog(LOG_DEBUG, " include: 0.0.0.0/0 (any)");
	}

	syslog(LOG_DEBUG, " algo: %s", algo_names[args.algo]);

	syslog(LOG_DEBUG, " interval: %d", args.interval);
	syslog(LOG_DEBUG, " check interval: %d", args.check_interval);
	syslog(LOG_DEBUG, " stats interval: %d", args.stats_interval);

	if ((s = args.peer) != NULL) {
		while (s != NULL) {
			syslog(LOG_DEBUG, " snmptrap: %s:%s:%s:%d (host[:port]:community:version:inform)", s->peername, s->community, (s->version == SNMP_VERSION_1) ? "v1" : (s->version == SNMP_VERSION_3) ? "v3" : "v2c", s->inform);
			s = s->next;
		}
	} else {
		syslog(LOG_DEBUG, " snmptrap: none");
	}

	if (args.expression) {
		syslog(LOG_DEBUG, " expression: %s", args.expression);
	} else {
		syslog(LOG_DEBUG, " expression: none");
	}

	if (args.test > 0) {
		syslog(LOG_DEBUG, " test: enabled");
	} else {
		syslog(LOG_DEBUG, " test: disabled");
	}

	if (args.promisc > 0) {
		syslog(LOG_DEBUG, " promiscuous: enabled");
	} else {
		syslog(LOG_DEBUG, " promiscuous: disabled");
	}

	if (args.jump) {
		syslog(LOG_DEBUG, " jump: %s", args.jump);
	} else {
		syslog(LOG_DEBUG, " jump: none");
	}

	if (args.iptables) {
		syslog(LOG_DEBUG, " iptables: %s", args.iptables);
	} else {
		syslog(LOG_DEBUG, " iptables: none");
	}

	if (args.config) {
		syslog(LOG_DEBUG, " config: %s", args.config);
	} else {
		syslog(LOG_DEBUG, " config: none");
	}

	if (args.user) {
		syslog(LOG_DEBUG, " user: %s", args.user);
	} else {
		syslog(LOG_DEBUG, " user: current");
	}

	if (args.group) {
		syslog(LOG_DEBUG, " group: %s", args.group);
	} else {
		syslog(LOG_DEBUG, " group: current");
	}

	if (args.stats) {
		syslog(LOG_DEBUG, " stats: %s", args.stats);
	} else {
		syslog(LOG_DEBUG, " stats: disabled");
	}

	if (args.dump) {
		syslog(LOG_DEBUG, " dump: %s", args.dump);
	} else {
		syslog(LOG_DEBUG, " dump: disabled");
	}

	if (args.chroot) {
		syslog(LOG_DEBUG, " chroot: %s", args.chroot);
	} else {
		syslog(LOG_DEBUG, " chroot: disabled");
	}

	syslog(LOG_DEBUG, " nice: %d", args.nice);

	if (args.debug > 0) {
		syslog(LOG_DEBUG, " debug: %d", args.debug);
	} else {
		syslog(LOG_DEBUG, " debug: disabled");
	}
}

void free_args() {
	struct threshold_t *t;
	struct prefix_t *p;
	snmptrap_peer_t *s;

	if (args.interface) free(args.interface);
	if (args.expression) free(args.expression);
	if (args.jump) free(args.jump);
	if (args.iptables) free(args.iptables);
	if (args.config) free(args.config);
	if (args.user) free(args.user);
	if (args.group) free(args.group);
	if (args.chroot) free(args.chroot);
	if (args.stats) free(args.stats);
	while (args.threshold) {
		t = args.threshold;
		args.threshold = t->next;
		free(t);
	}
	while (args.exclude) {
		p = args.exclude;
		args.exclude = p->next;
		free(p);
	}
	while (args.include) {
		p = args.include;
		args.include = p->next;
		free(p);
	}
	while (args.peer) {
		s = args.peer;
		args.peer = s->next;
		free(s);
	}

	memset(&args, 0, sizeof(args));
}

/*
** Local Variables:
** c-basic-offset: 3
** tab-width: 3
** End:
** vim: noet ts=3 sw=3
*/
