/**
 * $Id: mod_estraier_search.c,v 1.22 2006/04/22 19:22:55 shinh Exp $
 *
 * Copyright (C) shinichiro.h <hamaji@nii.ac.jp>
 *  http://shinh.skr.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/**
 * The present copyright holders of this program have given permission,
 * as a special exception, to link this program with Apache Portable
 * Runtime (APR) and to include header files for APR components when
 * those header files are covered by the Apache licenses, as long as
 * the GNU GPL is followed for this program in all other ways. 
 */

#include <estraier.h>
#include <estnode.h>
#include <cabin.h>
#include <stdlib.h>

#include <assert.h>

/* apache */
#include <http_protocol.h>
#include <http_config.h>
#include <http_log.h>
#include <apr_strings.h>

module AP_MODULE_DECLARE_DATA estraier_search_module;

typedef struct {
  char *node;
  char *user;
  char *pass;

  char *proxy_host;
  int proxy_port;
  int timeout;

  int node_depth;

  char *template_head, *template_foot;
} estsearch_dir_config;

typedef struct {
  char *phrase;
  int max;
  int page;
} estsearch_args;

#define PTR_OR(a, b) (((a) != NULL) ? (a) : (b))
#define INT_OR(a, b) (((a) != -1) ? (a) : (b))

/* for debugging */
static void printfinfo(const char *format, ...){
  FILE* fp = fopen("/tmp/log", "a");
  va_list ap;
  va_start(ap, format);
  fprintf(fp, "%s: INFO: ", "mod_estraier_search");
  vfprintf(fp, format, ap);
  fputc('\n', fp);
  fflush(fp);
  va_end(ap);
  fclose(fp);
}

static void estsearch_add_html(request_rec *r, apr_bucket_brigade *bb,
                               const char *filename)
{
  apr_bucket *b;
  conn_rec *c = r->connection;
  apr_file_t *fd;
  apr_finfo_t finfo;

  apr_stat(&finfo, filename, APR_FINFO_MIN, r->pool);
  apr_file_open(&fd, filename,
                APR_READ | APR_BINARY | APR_BUFFERED, APR_OS_DEFAULT, r->pool);
  b = apr_bucket_file_create(fd, 0, finfo.size, r->pool, c->bucket_alloc);
  APR_BRIGADE_INSERT_TAIL(bb, b);
}

static void estsearch_set_phrase(ESTNODE *node, ESTCOND *cond, char *val) {
  char *phrase = NULL, *p, *word;
  int now_or = 0;

  for (word = p = val;; p++) {
    if (*p == ' ' || *p == '\0') {
      int done = (*p == '\0');
      *p = '\0';

      if (strncmp("site:", word, 4) == 0) {
        char *site = (char *)malloc(sizeof(char)*(strlen(word)+8));
        strcpy(site, "@uri STRINC ");
        strcat(site, word+5);
        est_cond_add_attr(cond, site);
        free(site);
      }
      else if (strncmp("related:", word, 8) == 0) {
        CBDATUM *datum;
        const char *kbuf, *vbuf;
        int ksiz, vsiz;
        CBMAP *svmap = est_node_etch_doc_by_uri(node, word+8);

        if (!svmap) return;

        datum = cbdatumopen(ESTOPSIMILAR, -1);
        cbmapiterinit(svmap);
        while((kbuf = cbmapiternext(svmap, &ksiz)) != NULL){
          vbuf = cbmapget(svmap, kbuf, ksiz, &vsiz);
          cbdatumcat(datum, " WITH ", -1);
          cbdatumcat(datum, vbuf, vsiz);
          cbdatumcat(datum, " ", 1);
          cbdatumcat(datum, kbuf, ksiz);
        }

        est_cond_set_phrase(cond, CB_DATUMPTR(datum));

        cbdatumclose(datum);
        free(phrase);

        return;
      }
      else if (strncmp("link:", word, 5) == 0) {
        char *link = (char *)malloc(sizeof(char)*(strlen(word)+8));
        strcpy(link, "link STRINC ");
        strcat(link, word+5);

        est_cond_add_attr(cond, link);
        free(link);
      }
      else if (phrase && strncmp("-", word, 1) == 0) {
        int nsize = strlen(phrase) + strlen(word) + 8;
        phrase = (char *)realloc(phrase, sizeof(char)*nsize);
        strcat(phrase, " ANDNOT ");
        strcat(phrase, word+1);
      }
      else if (strcmp(word, "OR") == 0 || strcmp(word, "|") == 0) {
        now_or = 1;
      }
      else {
        if (phrase) {
          if (now_or) {
            int nsize = strlen(phrase) + strlen(word) + 5;
            phrase = (char *)realloc(phrase, sizeof(char)*nsize);
            strcat(phrase, " OR ");
            now_or = 0;
          }
          else {
            int nsize = strlen(phrase) + strlen(word) + 6;
            phrase = (char *)realloc(phrase, sizeof(char)*nsize);
            strcat(phrase, " AND ");
          }
          strcat(phrase, word);
        }
        else {
          phrase = (char *)malloc(sizeof(char)*strlen(word)+1);
          strcpy(phrase, word);
        }
      }

      if (done) break;

      p++;
      while (*p == ' ') p++;
      word = p;
    }
    else if (*p == '|') {
      now_or = 1;
      *p = ' ';
      p--;
    }
  }

  if (phrase) {
    est_cond_set_phrase(cond, phrase);
    free(phrase);
  }
  else {
    est_cond_set_phrase(cond, "");
  }
}

static ESTCOND *estsearch_parse_args(const char *a, ESTNODE *node,
                                     estsearch_args *ap)
{
  ESTCOND *cond;
  char *args = (char *)malloc(strlen(a)+1);
  char *p, *key, *val;

  val = NULL;
  ap->max = 10;
  ap->page = 1;
  ap->phrase = "";

  strcpy(args, a);

  cond = est_cond_new();

  for (key = p = args;; p++) {
    if (*p == '=') {
      *p = '\0';
      val = p + 1;
      printfinfo("= %s", val);
    }
    else if (*p == '&' || *p == '\0') {
      int done = (*p == '\0');
      *p = '\0';

      if (!val) {
        if (done) break; else continue;
      }

      printfinfo("%s", val);
      if (strcmp(key, "phrase") == 0) {
        ap_unescape_url(val);
        ap->phrase = cbmemdup(val, -1);
        estsearch_set_phrase(node, cond, val);
      }
      else if (strcmp(key, "max") == 0) {
        ap->max = atoi(val);
      }
      else if (strcmp(key, "page") == 0) {
        ap->page = atoi(val);
      }

      key = p + 1;
      val = NULL;
      if (done) break;
    }
    /* decode + to space */
    else if (*p == '+') {
      *p = ' ';
    }
  }

  free(args);

  est_cond_set_max(cond, ap->max*ap->page+1);

  return cond;
}

static void estsearch_printf(apr_bucket_brigade *bb, const char *fmt, ...) {
  va_list ap;
  va_start(ap, fmt);
  apr_brigade_vprintf(bb, NULL, NULL, fmt, ap);
  va_end(ap);
}

static void estsearch_add_result(apr_pool_t *pool, apr_bucket_brigade *bb,
                                 ESTNODERES *nres, ESTCOND *cond,
                                 estsearch_args *ap)
{
  int i;
  CBMAP *hints;
  int nn = est_noderes_doc_num(nres);
  int n = ap->page * ap->max;
  int first = (ap->page-1)*ap->max;
  if (n > nn) n = nn;

  hints = est_noderes_hints(nres);

  estsearch_printf(bb, "<div id=\"estresult\" class=\"estresult\">\n");
  estsearch_printf(bb, "<div class=\"resinfo\">");
  /* @@@ reffering member of ESTCOND */
  estsearch_printf(bb, "Results <strong>%d</strong> - <strong>%d</strong> of about <strong>%s hits</strong> for <strong>%s</strong></div>\n",
                   first+1, n, cbmapget(hints, "HIT", -1, NULL), ap->phrase);

  for (i = first; i < n; i++) {
    ESTRESDOC *rdoc;
    CBLIST *lines;
    int j;
    const char *uri;

    estsearch_printf(bb, "<dl class=\"doc\">");

    rdoc = est_noderes_get_doc(nres, i);

    estsearch_printf(bb, "<dt class=\"title\"><a href=\"%s\">%s</a></dt>",
                     est_resdoc_uri(rdoc),
                     est_resdoc_attr(rdoc, "@title"));

    estsearch_printf(bb, "<dd class=\"doc_text\">");

    lines = cbsplit(est_resdoc_snippet(rdoc), -1, "\n");
    for(j = 0; j < cblistnum(lines); j++){
      char *pv, *str;
      const char *line = cblistval(lines, j, NULL);
      if(line[0] == '\0'){
        if(j < cblistnum(lines) - 1) estsearch_printf(bb, " ... ");
      } else if((pv = strchr(line, '\t')) != NULL){
        str = cbmemdup(line, pv - line);
        estsearch_printf(bb, "<strong class=\"key key1\">%s</strong>", str);
        free(str);
      } else {
        estsearch_printf(bb, "%s", line);
      }
    }
    cblistclose(lines);

    estsearch_printf(bb, "</dd>");

    uri = est_resdoc_uri(rdoc);
    estsearch_printf(bb, "<dd class=\"doc_attr\">%s - %s - %s hits - <a href=\"search?phrase=related%%3A%s\">[related]</a></dd>",
                     uri,
                     est_resdoc_attr(rdoc, "@size"),
                     est_resdoc_attr(rdoc, "viewcount"),
                     ap_escape_uri(pool, uri));

    estsearch_printf(bb, "</dl>\n");
  }
  estsearch_printf(bb, "<div class=\"paging\">");
  if (ap->page > 1)
    estsearch_printf(bb, "<a href=\"search?phrase=%s&amp;max=%d&amp;page=%d\" type=\"text/html; charset=UTF-8\" class=\"navi\" title=\"Go back one page\">PREV</a> ", ap->phrase, ap->max, ap->page-1);
  else
    estsearch_printf(bb, "<span class=\"void\">PREV</span> ");
  if (n < nn)
    estsearch_printf(bb, "<a href=\"search?phrase=%s&amp;max=%d&amp;page=%d\" type=\"text/html; charset=UTF-8\" class=\"navi\" title=\"Go forward one page\">NEXT</a>", ap->phrase, ap->max, ap->page+1);
  else
    estsearch_printf(bb, "<span class=\"void\">NEXT</span>");
  estsearch_printf(bb, "</div><hr><div class=\"logo\">Powered by <a href=\"http://hyperestraier.sourceforge.net/\">Hyper Estraier</a> %s.</div>", est_version);

  est_cond_delete(cond);
}

static int check_estsearch_config(request_rec *r, estsearch_dir_config *conf) {
  if (conf->node == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "EstraierNode is not set");
    return TRUE;
  }
  if (conf->user == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "EstraierUser is not set");
    return TRUE;
  }
  if (conf->pass == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "EstraierPass is not set");
    return TRUE;
  }
  if (conf->template_head == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "EstsearchTemplateHead is not set");
    return TRUE;
  }
  if (conf->template_foot == NULL) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r,
                  "EstsearchTemplateFoot is not set");
    return TRUE;
  }

  return FALSE;
}

static int estsearch_handler(request_rec *r) {
  static const char *form_str = "<div id=\"estform\" class=\"estform\"><form action=\"search\" method=\"get\" id=\"form_self\"><div class=\"form_basic\"><input type=\"text\" name=\"phrase\" value=\"%s\" size=\"50\" id=\"phrase\" class=\"text\" tabindex=\"1\" accesskey=\"0\" /><input type=\"submit\" value=\"Search\" id=\"search\" class=\"submit\" tabindex=\"2\" accesskey=\"1\" /></div></form></div>";

  apr_bucket_brigade *bb;
  apr_bucket *b;
  conn_rec *c = r->connection;
  estsearch_dir_config *conf;

  if (strcmp(r->handler, "estraier_search")) {
    return DECLINED;
  }

  /* get directory config */
  conf = (estsearch_dir_config *)ap_get_module_config(
    r->per_dir_config, &estraier_search_module);

  /* check conf */
  if (check_estsearch_config(r, conf)) {
    return DECLINED;
  }

  ap_set_content_type(r, "text/html; charset=UTF-8");

  bb = apr_brigade_create(r->pool, c->bucket_alloc);

  estsearch_add_html(r, bb, conf->template_head);

  if (r->args) {
    ESTNODE *node;
    ESTCOND *cond;
    ESTNODERES *nres;
    estsearch_args args;

    node = est_node_new(conf->node);
    if (conf->proxy_host) {
      est_node_set_proxy(node, conf->proxy_host, INT_OR(conf->proxy_port, 80));
    }
    est_node_set_timeout(node, INT_OR(conf->timeout, 5));
    est_node_set_auth(node, conf->user, conf->pass);

    /* create ESTCOND */
    cond = estsearch_parse_args(r->args, node, &args);

    nres = est_node_search(node, cond, INT_OR(conf->node_depth, 1));

    est_node_delete(node);

    estsearch_printf(bb, form_str, args.phrase);
    if (nres) estsearch_add_result(r->pool, bb, nres, cond, &args);

    if (*args.phrase) free(args.phrase);
  }
  else {
    estsearch_printf(bb, form_str, "");
  }

  estsearch_add_html(r, bb, conf->template_foot);

  b = apr_bucket_eos_create(c->bucket_alloc);
  APR_BRIGADE_INSERT_TAIL(bb, b);

  ap_pass_brigade(r->output_filters, bb);

  return OK;
}

static apr_status_t cleanup_estsearch(void *p) {
  est_free_net_env();
  return OK;
}

static int estsearch_post_config(apr_pool_t* p, apr_pool_t* p1, apr_pool_t* p2,
                                 server_rec* s) {
  if (!est_init_net_env()) {
    ap_log_error(APLOG_MARK, APLOG_ERR, 0, s, "est_init_net_env failed");
    return !OK;
  }
  apr_pool_cleanup_register(p, NULL, cleanup_estsearch, apr_pool_cleanup_null);

  return OK;
}

static void register_estsearch_hooks(apr_pool_t* p) {
  ap_hook_handler(estsearch_handler, NULL, NULL, APR_HOOK_MIDDLE);
  ap_hook_post_config(estsearch_post_config, NULL, NULL, APR_HOOK_MIDDLE);
}

static void *create_estsearch_dir_config(apr_pool_t *p, char *path) {
  estsearch_dir_config *conf;

  conf = (estsearch_dir_config *)apr_pcalloc(p, sizeof(estsearch_dir_config));

  conf->node = NULL;
  conf->user = NULL;
  conf->pass = NULL;

  conf->proxy_host = NULL;
  conf->proxy_port = -1;
  conf->timeout = -1;

  conf->node_depth = -1;

  conf->template_head = NULL;
  conf->template_foot = NULL;

  return (void *)conf;
}

static void *merge_estsearch_dir_configs(apr_pool_t *p, void *bp, void *ap) {
  estsearch_dir_config *base = (estsearch_dir_config *)bp;
  estsearch_dir_config *add = (estsearch_dir_config *)ap;
  estsearch_dir_config *conf;

  conf = (estsearch_dir_config *)apr_pcalloc(p, sizeof(estsearch_dir_config));

  conf->node = PTR_OR(add->node, base->node);
  conf->user = PTR_OR(add->user, base->user);
  conf->pass = PTR_OR(add->pass, base->pass);

  conf->proxy_host = PTR_OR(add->proxy_host, base->proxy_host);
  conf->proxy_port = INT_OR(add->proxy_port, base->proxy_port);
  conf->timeout = INT_OR(add->timeout, base->timeout);

  conf->node_depth = INT_OR(add->node_depth, base->node_depth);

  conf->template_head = PTR_OR(add->template_head, base->template_head);
  conf->template_foot = PTR_OR(add->template_foot, base->template_foot);

  return conf;
}

static const char *set_estsearch_node(cmd_parms *cmd,
                                      void *tmp, const char *arg)
{
  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;
  conf->node = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estsearch_user(cmd_parms *cmd,
                                      void *tmp, const char *arg)
{
  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;
  conf->user = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estsearch_pass(cmd_parms *cmd,
                                      void *tmp, const char *arg)
{
  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;
  conf->pass = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estsearch_proxy_host(cmd_parms *cmd,
                                            void *tmp, const char *arg)
{
  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;
  conf->proxy_host = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estsearch_proxy_port(cmd_parms *cmd,
                                            void *tmp, const char *arg)
{
  int port = atoi(arg);
  if (port < 1 || port > 65535) {
    return "EstsearchPort must be valid port";
  }

  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;
  conf->proxy_port = port;

  return NULL;
}

static const char *set_estsearch_timeout(cmd_parms *cmd,
                                         void *tmp, const char *arg)
{
  int timeout = atoi(arg);
  if (timeout < 1 || timeout > 30) {
    return "EstsearchTimeout must be between 1 and 30";
  }

  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;
  conf->timeout = timeout;

  return NULL;
}

static const char *set_estsearch_node_depth(cmd_parms *cmd,
                                            void *tmp, const char *arg)
{
  int node_depth = atoi(arg);
  if (node_depth < 0) {
    return "EstsearchNodeDepth must be positive integer";
  }

  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;
  conf->node_depth = node_depth;

  return NULL;
}

static int is_file(apr_pool_t *p, const char *filename) {
  apr_finfo_t finfo;
  /* @@@ can i use this pool? */
  apr_stat(&finfo, filename, APR_FINFO_MIN, p);
  return finfo.filetype == APR_REG;
}

static const char *set_estsearch_template_head(cmd_parms *cmd,
                                               void *tmp, const char *arg)
{
  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;

  if (!is_file(cmd->pool, arg)) {
    return "EstsearchTemplateHead must be regular file";
  }

  conf->template_head = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const char *set_estsearch_template_foot(cmd_parms *cmd,
                                               void *tmp, const char *arg)
{
  estsearch_dir_config *conf = (estsearch_dir_config *)tmp;

  if (!is_file(cmd->pool, arg)) {
    return "EstsearchTemplateFoot must be regular file";
  }

  conf->template_foot = apr_pstrdup(cmd->pool, arg);

  return NULL;
}

static const command_rec estsearch_cmds[] = {
  AP_INIT_TAKE1("EstsearchNode", set_estsearch_node, NULL, OR_OPTIONS,
                "url of node"),
  AP_INIT_TAKE1("EstsearchUser", set_estsearch_user, NULL, OR_OPTIONS,
                "user of node"),
  AP_INIT_TAKE1("EstsearchPass", set_estsearch_pass, NULL, OR_OPTIONS,
                "password of node"),
  AP_INIT_TAKE1("EstsearchProxyHost", set_estsearch_proxy_host, NULL,
                OR_OPTIONS, "proxy host for node"),
  AP_INIT_TAKE1("EstsearchProxyPort", set_estsearch_proxy_port, NULL,
                OR_OPTIONS, "proxy port for node"),
  AP_INIT_TAKE1("EstsearchTimeout", set_estsearch_timeout, NULL, OR_OPTIONS,
                "timeout seconds, integer number"),
  AP_INIT_TAKE1("EstsearchNodeDepth", set_estsearch_node_depth, NULL,
                OR_OPTIONS, "depth of meta search, integer number"),
  AP_INIT_TAKE1("EstsearchTemplateHead", set_estsearch_template_head, NULL,
                OR_OPTIONS, "head template html"),
  AP_INIT_TAKE1("EstsearchTemplateFoot", set_estsearch_template_foot, NULL,
                OR_OPTIONS, "foot template html"),
  {NULL}
};

module AP_MODULE_DECLARE_DATA estraier_search_module = {
  STANDARD20_MODULE_STUFF,
  create_estsearch_dir_config,
  merge_estsearch_dir_configs,
  NULL,
  NULL,
  estsearch_cmds,
  register_estsearch_hooks
};
