/***************************************************************************
   Copyright (C) 2007
   by Marco Gulino <marco@kmobiletools.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "serialdevice.h"
#include <qmutex.h>
#include <kdebug.h>
#include <qtimer.h>
#include <qstring.h>
#include <qregexp.h>
#include <kapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <kuser.h>
#include <qserial.h>
#include <weaver.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include "config.h"
#if defined (KBLUETOOTH) || defined (KBLUETOOTH_NEW)
    #define HAVE_KBLUETOOTH 1
#endif
#define MAXBUFSIZE 32

#ifdef HAVE_KBLUETOOTH
    #ifdef KBLUETOOTH
        #include <kdebluetooth/rfcommsocketdevice.h>
    #endif
    #ifdef KBLUETOOTH_NEW
        #include "rfcommsocketdevice.h"
    #endif
    #include <qsocketnotifier.h>
#endif

#include <iostream>

#include "engineslist.h"
#include "kmobiletools_devices.h"

using namespace std;
using namespace KMobileTools;

//TODO: Remove this define
// Define from at_jobs.cpp

class KMobileTools::SerialManagerPrivate {
    public:
        SerialManagerPrivate() : /*lockfilecreated(-1), modem(-1), locked(false), connectEmitted(false), disconnectEmitted(false), timeouts(0), */
            b_connected(false), gotData(0), serial(0L), m_baudrate(QSerial::BAUD_57600), bluetooth(false), errnum(0)
        {
            mutex=new QMutex(true);
#ifdef HAVE_KBLUETOOTH
            rfcomm=0L;
#endif
        }
        ~SerialManagerPrivate() { delete mutex; }

        bool b_connected;
//         QStringList commandQueueStack;
        QString buffer;
        QMutex *mutex;
        uint gotData;
        QSerial *serial;
#ifdef HAVE_KBLUETOOTH
        KBluetooth::RfcommSocketDevice *rfcomm;
#endif
        QIODevice *device;
        QString s_devicePath;
        QStringList deviceInitStrings;
        QSerial::Baud m_baudrate;
        bool bluetooth;
        bool log;
        int errnum;
        QFile logfile;
        QTextStream logstream;
};

void SerialManager::lockMutex() {
    if(!d || ! d->mutex) return;
    d->mutex->lock();
}
void SerialManager::unlockMutex() {
    if(!d || ! d->mutex) return;
    d->mutex->unlock();
}


//TODO: Remove this function
// Function from at_jobs.cpp
QString parseInfo( QString buffer )
{
    QString tmp = buffer.section("OK\r\n",0,0).remove( '\r' ).remove( '\n' );
    int i = tmp.find( ':' );
    if ( i>0 && i<=6 && tmp.at(0)=='+' )
        tmp = tmp.section( ":",1 );
    tmp = tmp.stripWhiteSpace();
    if ( tmp.at(0)=='"' && tmp.at(tmp.length()-1)=='"' )
        tmp = tmp.mid( 1, tmp.length()-2 );
    return tmp;
}

SerialManager::SerialManager(QObject * parent, const char* name, const QString &devicePath, const QStringList &initStrings)
    : QObject(parent, name)
{
    d=new SerialManagerPrivate;
    if(QString(name)!="nodevice")
        d->log=DeviceConfigInstance(name)->verbose();
    else d->log=DEFAULT_VERBOSE;
    if(devicePath.length() && (QFile::exists( devicePath ) || devicePath.contains( "bluetooth://")) ) d->s_devicePath=devicePath;
    if(initStrings.count() ) d->deviceInitStrings=initStrings;
}

bool SerialManager::isConnected() { return d->b_connected; }
SerialManager::~SerialManager()
{
    close();
    delete d;
}

void SerialManager::setDevicePath(const QString &path)
{
    if(d->device)
    {
        if(d->device->isOpen())
        {
            close();
            setDevicePath(path);
            open(0);
        } else d->s_devicePath=path;
    }
}

bool SerialManager::open(ThreadWeaver::Job *job)
{
    bool isOpen;
#ifdef HAVE_KBLUETOOTH
    QRegExp addr("bluetooth://\\[(([A-F\\d]{2,2}:*){6,6})\\]:([\\d]+)");
    if( addr.search( d->s_devicePath ) != -1 )
    {
        d->bluetooth=true;
        d->rfcomm=new KBluetooth::RfcommSocketDevice();
        uint i=0;
        do {
            if(job && ! d->rfcomm->isOpen())
                job->thread()->msleep(2);
            else usleep(200000);
            d->rfcomm->connect(KBluetooth::DeviceAddress(addr.cap(1)), addr.cap(3).toInt());
            i++;
        } while ( ! d->rfcomm->isOpen() && i<5 );
        d->device=(d->rfcomm);
        d->rfcomm->setBlocking(false);
        isOpen=d->rfcomm->isOpen();
//         kdDebug() << "RFComm socket now should be connected: isOpen is returning " << d->serial->isOpen() << endl;
    } else {
#endif
    d->bluetooth=false;
    d->serial=new QSerial(d->s_devicePath);
    d->device=(d->serial);
    d->serial->setBaud( d->m_baudrate );
    d->serial->setStopBits( QSerial::STOP_BITS_1 );
    d->serial->setDatabits( QSerial::DATABITS_8 );
    d->serial->setFlowControl( QSerial::FLOW_CONTROL_HARD );
    d->serial->setParity( QSerial::PARITY_NONE );
    connect(d->serial, SIGNAL(gotData()), this, SLOT(gotData()));
    isOpen= d->device->open(IO_ReadWrite/* | O_NOCTTY | O_NONBLOCK */);
#ifdef HAVE_KBLUETOOTH
}
#endif
    if(!isOpen) return false;

    if(d->log)
    {
        d->logfile.setName(KGlobal::dirs()->saveLocation("tmp", "kmobiletools", true) + name() + ".log" );
        kdDebug() << "Starting log to " << d->logfile.name() << endl;
        d->logfile.open(IO_WriteOnly);
        d->logstream.setDevice(&(d->logfile));
    }
//     d->serial->reset();
    d->buffer=sendATCommand(job, "ATZ\r", 300);
    if(ATError(d->buffer))
    {
        kdDebug() << "Error while sending ATZ. Device closed.\n";
        close();
        return false;
    }
//     sendATCommand(job, "AT\r", 100);

    for ( QStringList::Iterator it=d->deviceInitStrings.begin(); it!=d->deviceInitStrings.end(); ++it)
        if((*it).stripWhiteSpace().length()>1) {
            d->buffer=sendATCommand(job, *it + "\r");
            if(ATError(d->buffer))
            {
                kdDebug() << "Error while sending " << *it << ". Device closed.\n";
                close();
                return false;
            }
        }
    emit connected();
    d->b_connected=true;
    return true;
}

#include "serialdevice.moc"
void SerialManager::close()
{
    if( !d->device || !d->device->isOpen() ) return;
    d->device->close();
    delete d->device;
    d->device=0;
    d->serial=0;
    d->logfile.close();
#ifdef HAVE_KBLUETOOTH
    d->rfcomm=0;
#endif
    d->b_connected=false;
    emit disconnected();
}

QString SerialManager::devicePath() const {
    return d->s_devicePath;
}

QString SerialManager::sendATCommand(ThreadWeaver::Job *job, const QString &cmd, uint timeout, bool tryBreakingTimeout)
{
    if(!d || ! d->mutex) return QString("\rERROR\r");
    QTime timer;
//     kdDebug() << "Mutex is locked::" << d->mutex->locked() << endl;
    QMutexLocker mlocker(d->mutex);
    if(!cmd.length() || ! d->device || !d->device->isOpen()) return QString::null;
//     QString classcmd=cmd.section( QRegExp("^AT"), 1,1,QString::SectionCaseInsensitiveSeps);
//     classcmd=classcmd.left(classcmd.find(QRegExp("[^\\w]"), 1) );
//     d->commandQueueStack+=classcmd;
//     kdDebug() << "QueueStack: " << d->commandQueueStack << endl;
//     timeout=(timeout * 1000)+1;

    d->buffer=QString::null;
    long err;
#ifdef HAVE_KBLUETOOTH
    if(d->bluetooth)
        err=d->rfcomm->writeBlock(cmd.latin1(), cmd.length()); else
#endif
    err=d->serial->writeBlock(cmd.latin1(), cmd.length());
    if(err==-1)
    {
        kdDebug() << "Write error: closing device link\n";
        close();
        return QString::null;
    }
    if(d->device) d->device->flush(); else return QString::null;
//     kdDebug() << "Sent cmd: " << cmd.latin1() << endl;
//     std::cout << ">>>" << QString(cmd).replace("\r","\n").replace("\n\n", "\n") << endl;
//     std::cout << "<<<";
    log(false, cmd);
    QRegExp exitExp("(OK|ERROR)(\\n|\\r)");
    timer.start();
//     uint i_try=0;
    while ( d->device && !EnginesList::instance()->closing() )
    {
        if( d->gotData )
        {
            timer.restart();
//             kdDebug() << "Got Chars, resetting timer: " << timer.elapsed() << endl;
            d->gotData=0;
        }
//         if(cmd.contains( "CPBR" ) || cmd.contains( "MPBR" ) ) kdDebug() << "Time elapsed: " << timer.elapsed() << "; buflen:" << buflen << "; buflen==buffer:" << (buflen==d->buffer.length() ) << endl;
        if( timer.elapsed() >=3000 &&
            (timer.elapsed()%3000 == 0) && tryBreakingTimeout)
        {
#ifdef HAVE_KBLUETOOTH
            if(d->bluetooth)
                err=d->rfcomm->writeBlock("AT\r", 3); else
#endif
            err=d->serial->writeBlock("AT\r", 3);
            d->device->flush();
            if(err==-1)
            {
                kdDebug() << "Write error: closing device link\n";
                close();
                return QString::null;
            }
            kdDebug() << "****************** WARNING!!!! Sending AT\\r to unblock the phone.\n";
            kdDebug() << "****************** this can be a bug of the phone, or of kmobiletools.\n";
            kdDebug() << "****************** please report to marco AT kmobiletools.org: AT command=" << cmd << endl;
        }
        if(job
#ifdef HAVE_KBLUETOOTH
           && ! d->bluetooth
#endif
          )
            job->thread()->msleep( 1 );
        else if(!job)
        {
            usleep(1000);
        }
#ifdef HAVE_KBLUETOOTH
        if(d && d->bluetooth)
        {
//             kdDebug() << "Bluetooth reading: try " << i_try << endl;
//             i_try++;
            char *buf=new char[MAXBUFSIZE+1];
            memset(buf, 0, MAXBUFSIZE+1);
            err=d->rfcomm->readBlock(buf, MAXBUFSIZE);
//             if(err==-1)
//             {
//                 delete [] buf;
//                 continue;
//                 kdDebug() << "Read error: closing device link\n";
//                 close();
//                 return QString::null;
//             }
//             kdDebug() << "Reading " << err << " characters.\n";
            if(err>0) d->buffer+=buf;
            d->gotData=strlen(buf);
            delete [] buf;
            if(job)job->thread()->msleep( 1 ); else usleep(1000);
        }
#endif

//         if(job)
//             job->thread()->msleep( 1 );
//         else usleep(1000);
        if(d && timeout)
        {
            if( (uint) timer.elapsed() >timeout)
            {
                kdDebug() << "Timeout exit: max timeout was " << timeout << ", timer: " << timer.elapsed() << endl;
                break;
            }
            if(d->buffer.contains(exitExp) )
            {
//                 kdDebug() << "Regexp exit\n";
                break;
            }
        }
    }
//     std::cout << endl;
    if( timeout<10 // Don't generate warnings for _wanted_ timeout-exit commands
        && (uint) timer.elapsed() >=timeout )
    {
        kdDebug() << "****************** WARNING!!!! Phone seems to be locked.\n";
        kdDebug() << "****************** this can be a bug of the phone, or of kmobiletools.\n";
        kdDebug() << "****************** please report to marco AT kmobiletools.org: AT command=" << cmd << endl;
    }
    int found=d->buffer.find(cmd);
//     kdDebug() << "Got buffer: " << d->buffer << endl;
    if(found!=-1 && found < 2)
        d->buffer=d->buffer.remove( found, cmd.length() );
    /// @TODO handle also partial errors
    exitExp.setPattern( "ERROR(\\n|\\r)");
//     if( d->buffer.contains( exitExp ) ) d->buffer="ERROR";
//     d->commandQueueStack.remove(classcmd);
    log(true, d->buffer);
    return d->buffer;
}

KMobileTools::QSerial *SerialManager::qserial()
{
    return d->serial;
}

void SerialManager::gotData()
{
    uint availData;
#ifdef HAVE_KBLUETOOTH
    if(d->bluetooth)
        availData=d->rfcomm->size(); else
#endif
        availData=d->serial->size();

if(!availData) availData=MAXBUFSIZE; // fix for rfcomm wrong size
//     kdDebug() << "GotData: Size=" << availData << endl;
    char *buffer=new char [availData+1];
    memset(buffer, 0, availData+1);
    int readdata;
#ifdef HAVE_KBLUETOOTH
    if(d->bluetooth)
    readdata=d->rfcomm->readBlock(buffer, availData);
#endif
    readdata=d->serial->readBlock(buffer, availData);
    if(readdata==-1)
    {
        kdDebug() << "Write error: closing device link\n";
        close();
        return;
    }
    if(readdata>0)
    {
        d->buffer+=buffer;
        d->gotData=availData;
//         std::cout << QString(buffer).replace("\r", "\n").replace("\n\n", "\n");
    }
    delete [] buffer;
}




/*!
    \fn SerialManager::speed(int value)
 */
/// @TODO have to move this somewhere...
/// @TODO better baudrate management
void SerialManager::setSpeed(int value)
{
    switch( value ){
        case 0:
            d->m_baudrate=QSerial::BAUD_9600;
            break;
        case 1:
            d->m_baudrate= QSerial::BAUD_19200;
            break;
        case 2:
            d->m_baudrate= QSerial::BAUD_38400;
            break;
        case 3:
            d->m_baudrate= QSerial::BAUD_57600;
            break;
        case 4:
            d->m_baudrate= QSerial::BAUD_115200;
            break;
        case 5:
            d->m_baudrate= QSerial::BAUD_230400;
            break;
        default:
            d->m_baudrate= QSerial::BAUD_57600;
            break;
    }
}

QString SerialManager::decodePDU( QString text )
{
    QString decoded;
    for ( uint i=0; i<text.length(); i += 2 )
    {
        decoded.append( QChar( text.mid( i, 2 ).toInt( 0,16 ) ) );
    }
    return decoded;
}


/*!
    \fn KMobileTools::SerialManager::atError(const QString &buffer)
 */
bool KMobileTools::SerialManager::ATError(const QString &buffer)
{
    if(!buffer.length() ) return true;
    int i=buffer.findRev( "ERROR" );
    if(i==-1) return false;
    if((buffer.length()-i)==5) return true; // "ERROR" is the last part of the string
    if(buffer.mid(i+5).contains( "[^\\n\\r]" )) return false;
    return true;
}

void KMobileTools::SerialManager::gotError(int err)
{
    d->errnum=err;
}



/*!
    \fn KMobileTools::SerialManager::log(bool incoming, const QString &data)
 */
void KMobileTools::SerialManager::log(bool incoming, const QString &data)
{
    if(!d->log) return;
    if(incoming)
    {
        d->logstream << "<<<" << QString(data).replace("\r", "\n").replace("\n\n", "\n");
        return;
    }
    d->logstream << ">>>" << QString(data).replace("\r","\n").replace("\n\n", "\n") << endl;

}
