/***************************************************************************
   Copyright (C) 2007
   by Marco Gulino <marco@kmobiletools.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "kmobiletoolsat_engine.h"
#include <qregexp.h>
#include "engineloader.h"
#include "picksmscenter.h"
#include "config.h"
#include <qtimer.h>


kmobiletoolsAT_engine::kmobiletoolsAT_engine(QObject *parent, const char *name)
 : kmobiletoolsEngine(parent, name), p_lastJob(NULL)
{
    device=NULL;
//     p_smsList = new SMSList();
    queue_sms=false;
}


kmobiletoolsAT_engine::~kmobiletoolsAT_engine()
{
    kdDebug() << "kmobiletoolsAT_engine::~kmobiletoolsAT_engine()\n";
//      weaver->finish();
//     closeDevice();
//     delete weaver;

}
void kmobiletoolsAT_engine::queryClose()
{
        closeDevice();
        kmobiletoolsEngine::queryClose();
}

extern "C"
{
    void* init_libkmobiletools_at()
    {
/*    KGlobal::locale()->insertCatalogue("kmobiletools");*/
        return new kmobiletoolsAT_engineFactory;
    }
};

kmobiletoolsAT_engineFactory::kmobiletoolsAT_engineFactory()
{
}
kmobiletoolsAT_engineFactory::~kmobiletoolsAT_engineFactory()
{
}
kmobiletoolsAT_engine *kmobiletoolsAT_engineFactory::createObject(QObject *parent, const char *name,
                                            const char *classname, const QStringList &args )
{
    (void) classname;
    (void) args;
    return new kmobiletoolsAT_engine(parent, name);
}




/*!
    \fn kmobiletoolsAT_engine::probePhone()
 */
void kmobiletoolsAT_engine::probePhone()
{
    devicesList.probeDevices( DeviceConfigInstance(name())->atdevices(), engineLibName(), QStringList(DeviceConfigInstance(name())->at_initString())+=DeviceConfigInstance(name())->at_initString2(), false, 0, DeviceConfigInstance(name())->mobileimei() );
}


void kmobiletoolsAT_engine::initPhone()
{
    p_foundDevice=devicesList.findByIMEI(DeviceConfigInstance(name())->mobileimei());
    kdDebug() << "Was searching for imei=\"" << DeviceConfigInstance(name())->mobileimei() << "\"; list of devices found::\n";
    devicesList.dump();
    if(!p_foundDevice)
    {
        kdDebug() << "Device not found\n";
        emit disconnected();
        return;
    } else kdDebug() << "Device found on " << p_foundDevice->foundPath() << endl;
//     usleep(10000000);
    device=new SerialManager(this, this->name(), p_foundDevice->foundPath(), initStrings() );
//     kmobiletoolsEngine::initPhone();
//     connect(device, SIGNAL(connected()), this, SLOT(devConnected() ) );
//     emit connected();
    connect(device, SIGNAL(disconnected()), SIGNAL(disconnected() ) );
    connect(device, SIGNAL(error()), SIGNAL(error() ) );
    connect(device, SIGNAL(invalidLockFile( const QString& )), this, SIGNAL(invalidLockFile( const QString& )) );
//     connect(device, SIGNAL(invalidLockFile( const QString& )), this, SLOT(invalidLockFile( const QString& )) );
    p_lastJob=new initPhoneJob(device, this);
//     connect(tempjob, SIGNAL(connected()), SIGNAL(connected() ) );
    enqueueJob( p_lastJob );
}


using namespace ThreadWeaver;
void kmobiletoolsAT_engine::processSlot(Job* p_job)
{
    kmobiletoolsEngine::processSlot(p_job);
//     kdDebug() << "job Owner: " << p_job->jobOwner() << "; job class: " << p_job->className() << endl;
    if(p_job->jobOwner() != name()) return;
    p_lastJob=0;
//     kdDebug() << "is KMobileToolsJob: " << p_job->inherits("kmobiletoolsJob") << endl;
    kmobiletoolsJob *job=(kmobiletoolsJob *) p_job;
    KMobileTools::DevicesConfig *wconfig=KMobileTools::DevicesConfig::prefs(name() );
    switch( job->type() ){
        case kmobiletoolsJob::initPhone:
            kdDebug() << "Device is connected: " << device->isConnected() << endl;
            b_connected=device->isConnected();
            if( device->isConnected() ) emit connected(); break;
        case kmobiletoolsJob::pollStatus:
            emit signal( ((PollStatus*) job)->phoneSignal() );
            emit charge( ((PollStatus*) job)->phoneCharge() );
            emit chargeType( ((PollStatus*) job)->phoneChargeType() );
            emit isRinging( ((PollStatus*) job)->ringing() );
            break;
        case kmobiletoolsJob::fetchPhoneInfos:
            s_manufacturer=( (FetchPhoneInfos*) job )->rawManufacturer();
            s_model=( (FetchPhoneInfos*) job )->model();
            s_revision=( (FetchPhoneInfos*) job )->revision();
            s_imei=( (FetchPhoneInfos*) job )->imei();
            s_smscenter=( (FetchPhoneInfos*) job )->smsCenter();
            if(s_smscenter!=QString::null) emit networkName( i18n("Network: %1").arg(PickSMSCenter::smsCenterName (s_smscenter)) );
            if ( s_manufacturer.contains( "Siemens", false )>0 ) i_manufacturer = Siemens;
            if ( s_manufacturer.contains( "Motorola", false )>0 ) i_manufacturer = Motorola;
            if ( s_manufacturer.contains( "Ericsson", false )>0 ) i_manufacturer = SonyEricsson;
            wconfig->setRawdevicename( s_model);
            wconfig->setRawdevicevendor( s_manufacturer );
            wconfig->writeConfig();
            break;
        case kmobiletoolsJob::fetchAddressBook:
            suspendStatusJobs(false);
            p_addresseeList=(ContactPtrList*) &((FetchAddressee*) job )->fullAddresseeList();
//                 kdDebug() << "trying to call EnginesList::instance()->emitPhonebookUpdated();\n";
//                 EnginesList::instance()->emitPhonebookUpdated();
//                 if( ! ((FetchAddressee*) job )->partialUpdates() )
            emit phoneBookUpdated();
            break;
        case kmobiletoolsJob::fetchSMS:
            p_diffSMSList->append(  ((FetchSMS*) job)->smsList );
            if( ((FetchSMS*) job)->last()) {
                p_smsList->sync(p_diffSMSList);
                queue_sms=false;
//                  p_smsList->dump();
//                  p_diffSMSList->dump();
            }
//         emit smsListUpdated();
        break;
        case kmobiletoolsJob::testPhoneFeatures:
            wconfig->setFstype( ((TestPhoneFeatures *) job)->getAbilities().filesystem() );
        wconfig->writeConfig();
        break;
        case kmobiletoolsJob::addAddressee:
            emit addressBookToUpdate();
            if( ((EditAddressees*)job)->pbIsFull() ) emit fullPhonebook();
            suspendStatusJobs(false);
            break;
        case kmobiletoolsJob::editAddressee:
            emit addressBookToUpdate();
            if( ((EditAddressees*)job)->pbIsFull() ) emit fullPhonebook();
            suspendStatusJobs(false);
            break;
        case kmobiletoolsJob::delAddressee:
            emit addressBookToUpdate();
            suspendStatusJobs(false);
            break;
        case kmobiletoolsJob::selectSMSSlot:
            if( ((SelectSMSSlot*) job)->done() )
                setSMSSlot( DeviceConfigInstance(name())->at_smsslots().findIndex( ((SelectSMSSlot*) job)->getReadSlot() ) );
            break;
        case kmobiletoolsJob::delSMS:
            if( ! ((DeleteSMS*)job)->succeeded() ) break;
            emit smsDeleted( ((DeleteSMS*)job)->sms()->uid() );
            break;
        case kmobiletoolsJob::storeSMS:
        case kmobiletoolsJob::sendStoredSMS:
        case kmobiletoolsJob::sendSMS:
            suspendStatusJobs(false);
            break;
        case kmobiletoolsJob::fetchKCal:
#ifdef HAVE_KCAL
            suspendStatusJobs(false);
            emit calendarParsed();
            p_calendar->dump();
#endif
            break;
    }
}


void kmobiletoolsAT_engine::devConnected()
{
    kmobiletoolsEngine::devConnected();
    getPhoneInfos();
}

#include "kmobiletoolsat_engine.moc"

void kmobiletoolsAT_engine::pollPhoneStatus()
{
    if (i_suspendStatusJobs) return;
    if(!device) return;
    p_lastJob=( new PollStatus(p_lastJob, device, this, this->name() ) );
    enqueueJob(p_lastJob);
}

void kmobiletoolsAT_engine::getPhoneInfos()
{
    if(!device) return;
    enqueueJob ( new FetchPhoneInfos(device, this, this->name() ) );
    enqueueJob ( new TestPhoneFeatures(device, this, this->name() ) );
    if( DeviceConfigInstance(name())->sync_clock() )
    {
      p_lastJob=new SyncDateTime(p_lastJob, device, this, this->name() );
      enqueueJob ( p_lastJob );
    }
    p_lastJob=new SelectCharacterSet( p_lastJob, DeviceConfigInstance(name())->at_encoding(), device, this, this->name() );
    enqueueJob( p_lastJob );
    p_lastJob=new SelectSMSSlot( p_lastJob, "ME", device, this, this->name() );
    enqueueJob( p_lastJob );
//    enqueueJob ( new FetchAddresseeSiemens(device, this, this->name() ) );
//    enqueueJob ( new FetchSMS( SMS::All, device, this, this->name() ) );
}

bool kmobiletoolsAT_engine::pdu()
{
    return atAbilities.isPDU();
}


/*!
    \fn kmobiletoolsAT_engine::retrieveAddressBook()
 */
void kmobiletoolsAT_engine::retrieveAddressBook()
{
    if(!device) return;
    kmobiletoolsJob *job;
    if( atAbilities.canSiemensVCF() || atAbilities.canSDBR() )
        job=( new FetchAddresseeSiemens(p_lastJob, device, this, this->name() ) );
    else job= ( new FetchAddressee(p_lastJob, availPbSlots(), device, this, this->name() ) );
    connect(job, SIGNAL(gotAddresseeList(int, const ContactPtrList&) ), this, SIGNAL(phoneBookUpdated(int, const ContactPtrList& ) ) );
    p_lastJob=job;
    enqueueJob(job);
}

void kmobiletoolsAT_engine::setATAbilities( ATAbilities atAbilities )
{
    this->atAbilities = atAbilities;
}

ATAbilities kmobiletoolsAT_engine::getATAbilities() const
{
    return atAbilities;
}


/*!
    \fn kmobiletoolsAT_engine::retrieveSMSList()
 */
void kmobiletoolsAT_engine::retrieveSMSList()
{
    if(i_suspendStatusJobs || ! device ) return;
//     p_smsList->clear();
    p_diffSMSList->clear();
    if(queue_sms) return;
    QStringList sl_slots=DeviceConfigInstance(name())->at_smsslots();
    if( ! sl_slots.count() ) {
        kdDebug() << "**** WARNING - this phone is NOT reporting having SMS slots. Perhaps it can't provide SMS. I'm trying anyway to fetch them.\n";
        p_lastJob=new FetchSMS(p_lastJob, SMS::All, device, true, this, this->name() );
        enqueueJob(p_lastJob);
        queue_sms=true;
        return;
    }
    for (QStringList::iterator it=sl_slots.begin(); it!=sl_slots.end(); ++it)
    {
        p_lastJob=new SelectSMSSlot( p_lastJob, (*it), device, this, this->name() );
        enqueueJob( p_lastJob );
	p_lastJob=new FetchSMS( p_lastJob, SMS::All, device, *it==sl_slots.last() ,this, this->name() );
        enqueueJob( p_lastJob );
        queue_sms=true;
    }
}

/*!
    \fn kmobiletoolsAT_engine::addAddressee(KABC::Addressee *addressee)
 */
void kmobiletoolsAT_engine::slotAddAddressee(QValueList<KABC::Addressee>* abclist)
{
    EditAddressees *tempjob=new EditAddressees(p_lastJob, abclist, device, false, this, this->name() );
    connect(tempjob, SIGNAL(fullPhonebook() ), this, SIGNAL(fullPhonebook() ) );
    p_lastJob=tempjob;
    if(device) enqueueJob( tempjob ) ;
}
/*!
    \fn kmobiletoolsAT_engine::delAddressee(int index)
 */
void kmobiletoolsAT_engine::slotDelAddressee(QValueList<KABC::Addressee>* abclist)
{
    if(device) {
        p_lastJob=new EditAddressees(p_lastJob, abclist, device, true, this, this->name() );
	enqueueJob(p_lastJob);
    }
}

int kmobiletoolsAT_engine::availPbSlots()
{
    int retval=0;
    if ( atAbilities.getPBSlots().findIndex("ME") >=0 ) retval+= PB_Phone;
    if ( atAbilities.getPBSlots().findIndex("SM") >=0 ) retval+= PB_SIM;
    if ( atAbilities.getPBSlots().findIndex("TA") >=0 ) retval+= PB_DataCard;
    return retval;
}


/*!
    \fn kmobiletoolsAT_engine::getPBMemSlotString(int memslot)
 */
QString kmobiletoolsAT_engine::getPBMemSlotString(int memslot)
{
    switch( memslot ){
        case PB_SIM:
            return QString("\"SM\"");
        case PB_Phone:
            return QString("\"ME\"");
        case PB_DataCard:
            return QString("\"TA\"");
        default:
            return QString::null;
    }
}


/*!
    \fn kmobiletoolsAT_engine::slotEditAddressee(KABC::Addressee*, KABC::Addressee*)
 */
void kmobiletoolsAT_engine::slotEditAddressee(KABC::Addressee *p_oldAddressee, KABC::Addressee *p_newAddressee)
{
    EditAddressees *tempjob=new EditAddressees( p_lastJob, p_oldAddressee, p_newAddressee, device, this, this->name() );
    p_lastJob=tempjob;
    connect(tempjob, SIGNAL(fullPhonebook() ), this, SIGNAL(fullPhonebook() ) );
    if(device) enqueueJob( tempjob );
}


/*!
    \fn kmobiletoolsAT_engine::weaverSuspended()
 */
void kmobiletoolsAT_engine::weaverSuspended()
{
    kmobiletoolsEngine::weaverSuspended();
    /// @TODO Look if we can fix this, otherwise remove
//     device->stopDevice();
}


/*!
    \fn kmobiletoolsAT_engine::resumeDevice()
 */
void kmobiletoolsAT_engine::resumeDevice()
{
    kmobiletoolsEngine::resumeDevice();
    /// @TODO Look if we can fix this, otherwise remove
//     device->resumeDevice();
    initPhoneJob* tempjob=new initPhoneJob(device, this);
    enqueueJob( tempjob );
    p_lastJob=tempjob;
    suspendStatusJobs(false);
}


/*!
    \fn kmobiletoolsAT_engine::stopDevice()
 */
void kmobiletoolsAT_engine::stopDevice()
{
    if(! statusJobsSuspended() ) suspendStatusJobs(true);
    if(weaver->queueLength())
    {
        QTimer::singleShot( 500, this, SLOT(stopDevice()));
        return;
    }
    kmobiletoolsEngine::stopDevice();
    device->close();
}


/*!
    \fn kmobiletoolsAT_engine::setDevice ( const QString &deviceName)
 */
void kmobiletoolsAT_engine::setDevice ( const QString &deviceName)
{
//     delete device;
    device->setDevicePath(deviceName );
}

/*!
    \fn kmobiletoolsAT_engine::probeDevice ( const QString &deviceName, const QString &param1, const QString &param2)
 */

DeviceInfos kmobiletoolsAT_engine::probeDevice( ThreadWeaver::Job *job, bool fullprobe, const QString &deviceName, const QStringList &params ) const
{
    SerialManager *device=new SerialManager(0, "nodevice", deviceName, params );
    connect(device, SIGNAL(invalidLockFile( const QString& )), this, SIGNAL(invalidLockFile( const QString& )) );
//     kdDebug() << "Probing device " << deviceName << endl;
    if(! device->open(job) ) return DeviceInfos();
//     kdDebug() << deviceName << " opened" << endl;
    QString buffer, temp;
    DeviceInfos retval;
    retval.setFoundPath( deviceName );
    const int probeTimeout=600;
    buffer=device->sendATCommand(job, "AT+CGSN\r", probeTimeout);
    if( !KMobileTools::SerialManager::ATError(buffer) ) // Phone imei
        retval.setImei( kmobiletoolsATJob::parseInfo( buffer ) );
    else  { closeDevice(device); delete device; return retval; }
    if(! fullprobe )
    {
        closeDevice(device);
        delete device;
        return retval;
    }

    buffer=device->sendATCommand(job, "AT+CGMR\r", probeTimeout);
    if(!KMobileTools::SerialManager::ATError(buffer)) // Phone revision
        retval.setRevision( kmobiletoolsATJob::parseInfo( buffer ) );

    buffer=device->sendATCommand(job, "AT+CGMI\r", probeTimeout);
    if( !KMobileTools::SerialManager::ATError(buffer) ) // Phone manufacturer
        retval.setManufacturer( kmobiletoolsATJob::parseInfo( buffer ) );

    buffer=device->sendATCommand(job, "AT+CPMS=?\r", probeTimeout);
    if( !KMobileTools::SerialManager::ATError(buffer) ) // SMS Slots
        temp = kmobiletoolsATJob::parseInfo( buffer );
    else temp=QString::null;
    retval.setSMSSlots( kmobiletoolsATJob::parseList( temp.replace("AT+CPMS=?", "") ) );

    buffer=device->sendATCommand(job, "AT+CPBS=?\r", probeTimeout);
    if( !KMobileTools::SerialManager::ATError(buffer) ) // PhoneBook Slots
        temp = kmobiletoolsATJob::parseInfo( buffer );
    else temp=QString::null;
    retval.setPbSlots( kmobiletoolsATJob::parseList( temp.replace("AT+CPBS=?", "") ) );

    buffer=device->sendATCommand(job, "AT+CGMM\r", probeTimeout);
    if(!KMobileTools::SerialManager::ATError(buffer)) // Phone model
        retval.setModel( kmobiletoolsATJob::parseInfo( buffer ) );

    buffer=device->sendATCommand(job, "AT+CSCS=?\r", probeTimeout);
    if( !KMobileTools::SerialManager::ATError(buffer) ) // Charset
        temp = kmobiletoolsATJob::parseInfo( buffer );
    else temp=QString::null;
    retval.setCharsets( kmobiletoolsATJob::parseList( temp ) );

    buffer=device->sendATCommand(job, "AT+CSCA?\r", probeTimeout); // SMS Center
    if(!KMobileTools::SerialManager::ATError(buffer))
    {
        temp=kmobiletoolsATJob::parseInfo( buffer);
        QRegExp tempRegexp;
        tempRegexp.setPattern( ".*\"(.*)\".*");
        if(tempRegexp.search(temp)>=0) temp=tempRegexp.cap( 1 ); else temp=QString::null;
        retval.setSMSCenter( temp );
    }
    closeDevice(device);
    delete device;
    return retval;
}


/*!
    \fn kmobiletoolsAT_engine::slotDelSMS(SMS* sms);

 */
void kmobiletoolsAT_engine::slotDelSMS(SMS* sms)
{
    if(!device) return;
    p_lastJob=new SelectSMSSlot( p_lastJob, sms->rawSlot(), device, this, this->name() ) ;
    enqueueJob(p_lastJob);
    p_lastJob=new DeleteSMS( p_lastJob, sms, device, this, this->name() ) ;
}

void kmobiletoolsAT_engine::slotStoreSMS( const QString &number, const QString &text)
{
    if(!device) return;
    p_lastJob=( new StoreSMS(p_lastJob, number, text, device, this, this->name() ));
    enqueueJob(p_lastJob);
}


void kmobiletoolsAT_engine::slotSendSMS( const QString &number, const QString &text)
{
    if(!device) return;
    p_lastJob=( new SendSMS(p_lastJob, number, text, device, this, this->name() ));
    enqueueJob(p_lastJob);
}


/*!
    \fn kmobiletoolsAT_engine::slotSendSMS(SMS*)
 */
void kmobiletoolsAT_engine::slotSendSMS(SMS* sms)
{
    if(!device) return;
    p_lastJob=( new SendSMS(p_lastJob, sms, device, this, this->name() ) );
    enqueueJob(p_lastJob);
}

void kmobiletoolsAT_engine::slotSendStoredSMS(SMS* sms)
{
    if(!device) return;
    p_lastJob=( new SendStoredSMS(p_lastJob, sms, device, this, this->name() ) );
    enqueueJob(p_lastJob);
}


/*!
    \fn kmobiletoolsAT_engine::slotStoreSMS(SMS*)
 */
void kmobiletoolsAT_engine::slotStoreSMS(SMS* sms)
{
    if(!device) return;
    p_lastJob=( new StoreSMS(p_lastJob, sms, device, this, this->name() ) );
    enqueueJob(p_lastJob);
}


/*!
    \fn kmobiletoolsAT_engine::fetchCalendar()
 */
void kmobiletoolsAT_engine::fetchCalendar()
{
#ifdef HAVE_KCAL
    if(!device) return;
    p_lastJob=( new FetchCalendar(p_lastJob, device, this, this->name() ) );
    enqueueJob(p_lastJob);
#endif
}


/*!
    \fn kmobiletoolsAT_engine::dial(DialActions)
 */
void kmobiletoolsAT_engine::dial(DialActions action, const QString &gnumber)
{
    uint dialsystem=DeviceConfigInstance(name())->at_dialsystem();
    QString number=gnumber;
    switch( action ){
        case DIAL_DIAL:
            if(number.isNull()) return;
            switch( dialsystem ){
                case 0:
                    if( number.at(0)=='+' )
                        number=number.right( number.length() -1 ).prepend("AT+CKPD=\"0\",20;+CKPD=\"").append("s\"\r" );
                    else number=number.prepend( "AT+CKPD=\"" ).append( "s\"\r" );
                    break;
                case 1:
                    number=number.prepend( "ATD" ).append( ";\r" );
                    break;
            }
            device->sendATCommand(0, number, 1500);
            break;
        case DIAL_HANGUP:
            switch( dialsystem ){
                case 0:
                    device->sendATCommand(0, "AT+CKPD=\"e\"\r", 1500 );
                    break;
                case 1:
                    device->sendATCommand( 0, "ATH" );
                    device->sendATCommand( 0, "AT+CHUP" );
                    break;
            }
            break;
    }
}

QString kmobiletoolsAT_engine::currentDeviceName() const
{
    return device->devicePath();
}


QString kmobiletoolsAT_engine::engineLibName() const
{
    return QString("libkmobiletools_at");
}

/*!
    \fn kmobiletoolsAT_engine::switchToFSMode()
 */
void kmobiletoolsAT_engine::switchToFSMode()
{
    switch( DeviceConfigInstance(name())->fstype() ){
        case 1:
            device->sendATCommand(0, "AT+MODE=8\r");
            stopDevice();
            break;
        default:
            kmobiletoolsEngine::switchToFSMode();
            break;
    }
}
