/***************************************************************************
                          KickPimWidget.cpp  -  description
                             -------------------
    begin                : Fre Okt 18 2002
    copyright            : (C) 2002 by Bert Speckels
    email                : bert@speckels.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include <qapplication.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qcursor.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qtooltip.h>
#include <qtimer.h>

#include <kaboutapplication.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <krun.h>
#include <kpanelapplet.h>
#include <kaudioplayer.h>
#include <klocale.h>
#include <kkeydialog.h>

#include "dialogs/kpinfo.h"
#include "dialogs/kickpiminfodialog.h"
#include "dialogs/kickpimpreferencesdialog.h"
#include "dialogs/kickpimnotifydialog.h"

#include "mailing/kickpimmailmonitor.h"
#include "mailing/kickpimmaildialog.h"

#include "kickpim.h"
#include "kickpimmenu.h"
#include "LogService.h"

#include "kickpimwidget.h"

//#ifdef USE_DCOP
  #include <dcopclient.h>
//#endif



static const int POPUP_MISC_PREFS         = 101;
static const int POPUP_MISC_KEYS          = 102;
static const int POPUP_MISC_ABOUT         = 103;

static const int POPUP_ADDR_BOOK          = 201;
static const int POPUP_ORGANIZER          = 202;
static const int POPUP_ADDR_ADD           = 203;
static const int POPUP_ADDR_PREFS         = 204;

static const int POPUP_MAIL_CHECK         = 301;
static const int POPUP_MAIL_TOGGLE        = 302;
static const int POPUP_MAIL_PREFS         = 303;


KickPimWidget::KickPimWidget(QWidget *parent, const char *name ) :
//#ifdef USE_DCOP
        DCOPObject("KickPIM-Iface"),
//#endif
        QFrame(parent,name)
{
  LogService::construct("KickPimWidget");

  m_dcopClient=0;
  #ifdef USE_DCOP    
    m_dcopClient = KApplication::dcopClient();
    m_dcopClient->attach();
    m_dcopId = m_dcopClient->registerAs( "KickPIM-Applet" );
    LogService::logInfo(LogService::CAT_KICKPIM,"DCOP-ID: " + m_dcopId);
  #endif

  m_emailCount    = 0;
  m_eventCount    = 0;
  m_anniCount     = 0;

  m_blinkEvent    = false;
  m_blinkAnni     = false;
  m_blinkEmail    = false;
  m_blinkEmailVip = false;

  m_blinkToggle   = false;

  // widgets
  m_notifyWidget  = 0;
  m_menu          = 0;

  // icons
  m_icon          = new QPixmap( KickPIM::rep()->dirOfIcons() + "hi24-kickpim-icon.png" );
  m_iconEmail     = new QPixmap( KickPIM::rep()->dirOfIcons() + "hi24-kickpim-icon-email.png" );
  m_iconEmailVip  = new QPixmap( KickPIM::rep()->dirOfIcons() + "hi24-kickpim-icon-email-vip.png" );
  m_iconEvent     = new QPixmap( KickPIM::rep()->dirOfIcons() + "hi24-kickpim-icon-event.png" );
  m_iconAnni      = new QPixmap( KickPIM::rep()->dirOfIcons() + "hi24-kickpim-icon-anniversary.png");

  // misc popup
  m_popupMisc = new KPopupMenu(this,"RightClickPopupMisc");
  m_popupMisc->insertItem ( SmallIcon("korganizer"),    i18n("K&Organizer"),            POPUP_ORGANIZER);
  m_popupMisc->insertItem ( SmallIcon("kaddressbook"),  i18n("&KAddressBook"),          POPUP_ADDR_BOOK);
  m_popupMisc->insertItem ( SmallIcon("edit_add"),      i18n("Add &Contact"),           POPUP_ADDR_ADD);
  m_popupMisc->insertSeparator();
  m_popupMisc->insertItem ( SmallIcon("mail_get"),      i18n("Check Mails &Now!"),      POPUP_MAIL_CHECK);
  m_popupMisc->insertItem ( SmallIcon("mail_find"),     i18n("A&utocheck Mails"),       POPUP_MAIL_TOGGLE);
  m_popupMisc->insertSeparator();
  m_popupMisc->insertItem ( SmallIcon("configure"),     i18n("&Email Settings"),       POPUP_MAIL_PREFS);
  m_popupMisc->insertItem ( SmallIcon("configure"),     i18n("&Settings"),             POPUP_MISC_PREFS );
  m_popupMisc->insertItem ( SmallIcon("configure_shortcuts"), i18n("Define &Keys..."), POPUP_MISC_KEYS  );
  m_popupMisc->insertSeparator();
  m_popupMisc->insertItem ( *m_icon,                    i18n("&About")+" KickPIM",     POPUP_MISC_ABOUT );

  // timers
  m_emailTimer    = new QTimer( this );
  m_eventTimer    = new QTimer( this );
  m_blinkTimer    = new QTimer( this );

  m_eventTimer->start( 1000*60*1, FALSE ); // 1 minute
  m_emailTimer->start( 1000*60*5, FALSE ); // 5 minutes
}

void KickPimWidget::init() {
  m_menu = new KickPimMenu(this,"KickPimMenu");


  // connect
  connect( m_popupMisc,  SIGNAL(activated(int)), this, SLOT(popupMiscClick(int)) );
  connect( m_blinkTimer, SIGNAL(timeout()),      this, SLOT(onTimerBlink()) );
  connect( m_eventTimer, SIGNAL(timeout()),      this, SLOT(onTimerCheckEvents()) );
  connect( KickPIM::rep(), SIGNAL( eventListChanged() ),
           this,           SLOT  ( onEventsChanged() ));
  connect( KickPIM::rep(), SIGNAL(   emailsChanged() ),
           this,           SLOT  ( onEmailsChanged() ));

  // setAcceptDrops(TRUE);
}

KickPimWidget::~KickPimWidget()
{
  LogService::destruct("KickPimWidget");

  if (m_menu) m_menu->preOptionSave();
  if (KickPIM::rep()) KickPIM::opt()->save();
  if (m_dcopClient!=0) m_dcopClient->detach();

  delete m_menu;        m_menu        = 0;
  delete m_popupMisc;   m_popupMisc   = 0;

  delete m_icon;
  delete m_iconEvent;
  delete m_iconEmail;
  delete m_iconEmailVip;
  delete m_iconAnni;
}





/** event handler for mouse click on applet */
void KickPimWidget::mousePressEvent(QMouseEvent *e)
{
  if (e->button() == LeftButton) {
    if (KickPIM::opt()->isInfoActive(KickPimOptions::INFO_MENU)) {
      KickPIM::opt()->deactivateInfo(KickPimOptions::INFO_MENU);
      KickPimInfoDialog dlg;
      dlg.setInfoText( i18n(infoKickPimMenu) );
      dlg.setPicture( QPixmap( KickPIM::rep()->dirOfInfo() + "info-kickpim-menu.png") );
      dlg.exec();
    }

    displayMenu();
  }
  else if (e->button() == RightButton) {
    displayPopupMisc();
  }
}

bool KickPimWidget::setMenuVisible(bool visible)
{
  LogService::call("KickPimWidget","setMenuVisible(bool)");
  if (!m_menu) return false;

  if (m_menu->isActive()  &&  visible) return false;
  if (!m_menu->isActive() && !visible) return false;

  if (visible) {
    //displayMenu(true); // display centered on screen.
    displayMenu();
  }
  else {
    m_menu->forceHide();
  }

  return true;
}


/** show popup menu */
void KickPimWidget::displayMenu(bool displayAtCenter)
{
  LogService::call("KickPimWidget","displayMenu(bool displayAtCenter)");
  if (!m_menu) return;

  if (m_menu->isActive()) // toggle display of the menu (tricky but works)
  {
    m_menu->forceHide();
    return;
  }

  KickPIM* applet = dynamic_cast<KickPIM*>(parent());
  if (!applet) return;

  // now display the menu
  QDesktopWidget* desktop = QApplication::desktop();
  int xres = desktop->width();
  int yres = desktop->height();

  QSize menuSize;
  menuSize.setWidth ( m_menu->width()  );
  menuSize.setHeight( m_menu->height() );
  QRect rect = applet->rect();

  int x = 0;
  int y = 0;

  if (displayAtCenter)
  {
    x = (xres - menuSize.width()) / 2;
    y = (yres - menuSize.height()) / 2;
  }
  else
  {
    QPoint p = mapToGlobal( QPoint( 0, 0 ) );

    if (applet->isVertical())
    {
      x = p.x()+rect.width(); // pop RIGHT
      y = p.y();
      if (x+menuSize.width() >xres) x = p.x() - menuSize.width(); // pop LEFT
      if (y+menuSize.height()>yres) y = yres  - menuSize.height();
      if (x<0) x=0;
    }
    else
    {
      x = p.x();
      y = p.y()+rect.height(); // pop DOWN
      if (y+menuSize.height()>yres) y = p.y() - menuSize.height(); // pop UP
      if (x+menuSize.width() >xres) x = xres  - menuSize.width();
      if (y<0) y=0;
    }
  }

  // little hack to update the display date
  // should be connected so a signal (if such thing exists)
  m_menu->onDateChanged();
  
  m_menu->move( x, y );

  m_menu->show( );

  updateWidget();
}

void KickPimWidget::displayPopupMisc()
{
  LogService::call("KickPimWidget","displayPopupMisc");
  if (!m_popupMisc) return;
  KickPIM* applet = dynamic_cast<KickPIM*>(parent());
  QRect  rect = applet->rect();
  QPoint p    = mapToGlobal( QPoint( 0, 0 ) );
  int    y    = p.y() - m_popupMisc->sizeHint().height();  
  if (y<0) y = p.y()+rect.height();
  
  p.setY( y );
  
  bool autoMailCheck = KickPIM::opt()->mailCheckEnabled;
  m_popupMisc->setItemChecked(POPUP_MAIL_TOGGLE, autoMailCheck);

  m_popupMisc->exec( p );
}




void KickPimWidget::paintEvent(QPaintEvent*)
{
  QPainter painter(this);

  drawDefault  (&painter);

  /*
  if      (m_eventCount>0)     drawEvents   (&painter);
  else if (m_emailCount>0)     drawEmails   (&painter);
  else                         drawDefault  (&painter);
  */

  //painter.drawRect( rect );
}


void KickPimWidget::drawDefault(QPainter* painter)
{
  drawIcon(painter,m_icon);

  if ( (m_blinkEmail && m_blinkToggle) || !m_blinkEmail) {
    if (m_emailCount>0) drawEmails( painter, m_blinkEmailVip );
  }

  if ( (m_blinkAnni && m_blinkToggle) || !m_blinkAnni) {
    if (m_anniCount >0) drawAnniversaries( painter    );
  }

  if ( (m_blinkEvent && m_blinkToggle) || !m_blinkEvent) {
    if (m_eventCount>0) drawEvents( painter, 0 );
  }
}

void KickPimWidget::drawEmails(QPainter* painter, bool vip)
{
  if (vip) drawIcon(painter,m_iconEmailVip);
  else     drawIcon(painter,m_iconEmail);
}


void KickPimWidget::drawEvents(QPainter* painter, int /*priority*/)
{
  drawIcon(painter,m_iconEvent);
}

void KickPimWidget::drawAnniversaries(QPainter* painter)
{
  drawIcon(painter,m_iconAnni);
}


void KickPimWidget::drawText(QPainter* painter, QString text)
{
  KickPIM* applet = dynamic_cast<KickPIM*>(parent());
  if (!applet) return;
  QSize size = frameSize();

  int x=0,y=0,xd=size.width(),yd=size.height();

  QFont font = painter->font();
  font.setPixelSize(14);
  font.setBold(true);
  painter->setFont(font);
  painter->drawText (x,y,xd,yd, Qt::AlignCenter, text);
}

#include <kmessagebox.h>
void KickPimWidget::drawIcon(QPainter* painter, QPixmap* icon)
{
  QRect rect = painter->viewport();

  /*
  KickPIM* applet = dynamic_cast<KickPIM*>(parent());
  if (!applet) return;
  rect = applet->rect();
                     */
  int x = ( rect.width() - icon->width()  ) / 2;
  int y = ( rect.height()- icon->height() ) / 2;

  /*
  cout << x << "/" << y << "(" << rect.width() << "x" << rect.height() << ")" << endl;
  char buf[255];
  sprintf(buf,"%i / %i (%i x %i)",x,y,rect.width(),rect.height());
  int result = KMessageBox::questionYesNo(
                        0,
                        buf,
                        "Info", KStdGuiItem::yes());
  */

  painter->drawPixmap(x,y,*icon);
}


void KickPimWidget::displayKeyPreferences()
{
  KKeyDialog::configure(KickPIM::opt()->globalAccel);

  if (m_menu) m_menu->preOptionSave();
  if (KickPIM::rep()) KickPIM::opt()->save();
}

void KickPimWidget::prefsContacts()
{
}

void KickPimWidget::prefsEmailAccounts()
{
  KickPimMailDialog dlg;
  if (dlg.exec() == QDialog::Accepted)
  {
  }
}


/** show configuration dialog */
void KickPimWidget::prefsGeneral(void)
{
  KickPimPreferencesDialog dlg( 0, "peoplePrefs");
  dlg.setOptions( KickPIM::opt() );
  if( dlg.exec() == QDialog::Accepted)
  {
    dlg.getOptions( KickPIM::opt() );

    if (m_menu) m_menu->preOptionSave();
    if (KickPIM::rep()) KickPIM::opt()->save();

    onEmailsChanged();
    onTimerCheckEvents();

    // recreate the menu
    delete m_menu;
    m_menu = new KickPimMenu(this,"KickPimMenu");
    KickPIM::rep()->reload();
  }

}

/** event handler for popup menu click */
void KickPimWidget::popupMiscClick(int id)
{
  switch ( id )
  {
      //
      // general
      //

      case POPUP_MISC_PREFS:
        prefsGeneral();
        break;
      case POPUP_MISC_KEYS:
        displayKeyPreferences();
        break;
      case POPUP_MISC_ABOUT:
        displayAboutDialog();
        break;

      //
      // ADDRESS
      //

      case POPUP_ADDR_BOOK:
        KRun::runCommand("kaddressbook");
        break;
      case POPUP_ORGANIZER:
        KRun::runCommand("korganizer");
        break;
      case POPUP_ADDR_ADD:
        addContact();
        break;
      case POPUP_ADDR_PREFS:
        prefsContacts();
        break;

      //
      // MAIL
      //

      case POPUP_MAIL_CHECK:
        checkMailAccounts();
        break;
      case POPUP_MAIL_TOGGLE:
        toggleMailCheck();
        break;
      case POPUP_MAIL_PREFS:
        prefsEmailAccounts();
        break;
  }
}


void KickPimWidget::displayAboutDialog()
{
  const char* description = I18N_NOOP("KickPIM supports fast access to your <b>KDE Addressbook</b>\n"
                                      "and reminds you about important <b>Events</b>\n"
                                      "such as birthdays and anniversaries.\n"
                                      "It also monitors multiple <b>Email Accounts</b>.");

  // about data
  KAboutData aboutData(
    "kickpim",
    "KickPIM",
    VERSION,
    description,
    KAboutData::License_GPL,
    "(c)2003, Bert Speckels",
    "",
    "http://kickpim.sourceforge.net",
    "bert.speckels@web.de"
  );
  aboutData.addAuthor     ("Bert Speckels", 0, "bert.speckels@web.de");  
  aboutData.setTranslator (i18n("_: NAME OF TRANSLATORS\nYour names") ,I18N_NOOP("_: EMAIL OF TRANSLATORS\nYour emails"));

  KAboutApplication dlg( &aboutData, parentWidget() );

  /*
  KAboutDialog dlg(
          KAboutDialog::AbtKDEStandard,
          i18n("About KickPIM"),
          KDialogBase::Ok,
          KDialogBase::Ok,
          this);
          */

  QString iconPath = KickPIM::rep()->dirOfIcons() + "hi32-kickpim-icon.png";
  dlg.setLogo( QPixmap(iconPath) );
  dlg.setImage( iconPath );  

  dlg.addTextPage(
          i18n("&Thanks To"),
          "<qt>"
          "<b>Translators:</b><br>"
          "&nbsp;&nbsp; Michal Rudolf (pl)<br>"
          //"&nbsp;&nbsp; Arend van Beelen (nl)<br>"
          //"&nbsp;&nbsp; ?vind S?her (nb)<br>"
          //"&nbsp;&nbsp; Milos Sifalda (cz)<br>"
          "&nbsp;&nbsp; Davide Ferrari (it)<br>"
          //"&nbsp;&nbsp; Laurent Rathle (fr)<br>"
          "<br>"
          "<b>RPM builders</b><br>"
          "&nbsp;&nbsp; Matthias Hagen (Mandrake and Testing)<br>"
          "<br>"
          "<b>Developers</b><br>"
          "&nbsp;&nbsp; Kurt Granroth (KBiff)<br>"
          "<br>"
          "and all of you out there ...!<br>"
          "<br>"
          "</qt>",true);

  KAboutContainer* cont = dlg.addContainerPage( i18n("Version")+" '"+VERSION_TITLE+"'" );
  QString imagePath = KickPIM::rep()->dirOfInfo() + VERSION_IMAGE;
  QLabel* imageLabel = new QLabel(cont);
  imageLabel->setPixmap( QPixmap(imagePath) );
  
  dlg.exec();
}

void KickPimWidget::onTimerBlink()
{
  m_blinkToggle = !m_blinkToggle;
  
  repaint();
}

void KickPimWidget::onTimerCheckEvents()
{                       
  LogService::call("KickPimWidget","onTimerCheckEvents");

  KPEventList list;
  m_eventCount = KickPIM::rep()->getWaitingEvents(list);
  m_blinkEvent = m_eventCount>0;

  m_anniCount  = KickPIM::rep()->getWaitingAnniversaries(list);
  m_blinkAnni  = false;

  // check if the anni-Icons should blink
  if (m_anniCount>0)
  {
    QDate todayDate     = QDate().currentDate();
    QDate remindedDate  = KickPIM::opt()->lastReminded.date();
    if ( todayDate != remindedDate )
    {
      KPEvent* event = list.first();
      QDate    date  = event->date();
      m_blinkAnni    = ( date.day()   == todayDate.day()
                      && date.month() == todayDate.month() );
      if (m_blinkAnni) LogService::logInfo(LogService::CAT_EVENTS,"Blink anniversary: yes");
      else             LogService::logInfo(LogService::CAT_EVENTS,"Blink anniversary: no");
    }
  }

  updateWidget();
}

/*
void KickPimWidget::polish()
{
  FUNC_CALL(KickPimWidget::polish)
  QWidget::polish();

  repaint();
}
*/


void KickPimWidget::addContact()
{
  KickPIM::rep()->addContact();
}

void KickPimWidget::checkMailAccounts()
{
  // LogService::call("KickPimWidget","checkMailAccounts");
  KickPIM::rep()->mailMonitors_Check();
}

bool KickPimWidget::toggleMailCheck()
{
  LogService::call("KickPimWidget","toggleMailCheck");

  bool isOn = !KickPIM::opt()->mailCheckEnabled;
  KickPIM::opt()->mailCheckEnabled = isOn;

  if (isOn) LogService::logInfo(LogService::CAT_MAIL,"  AutoMailCheck is on");
  else      LogService::logInfo(LogService::CAT_MAIL,"  AutoMailCheck is off");  

  return true;
}



void KickPimWidget::blink(bool active)
{
  if (  active &&  m_blinkTimer->isActive() ) return;
  if ( !active && !m_blinkTimer->isActive() ) return;

  m_blinkToggle=false;
  if (active) {
    m_blinkTimer->start( 750, FALSE ); // 1/2 second
  }
  else {
    m_blinkTimer->stop( );
  }
}

void KickPimWidget::updateWidget()
{
  // tooltip
  QToolTip::remove( this );

  QString summary="";

  if  (m_eventCount>0) {
    summary+=i18n("%n Event","%n Events",m_eventCount);
  }
  if (m_anniCount>0) {
    if (!summary.isEmpty())  summary += "\n";
    summary+=i18n("%n Birthday/Anniversary","%n Birthdays/Anniversaries",m_anniCount);
  }
  if (m_emailCount>0) {
    if (!summary.isEmpty())  summary += "\n";
    summary+=i18n("%n Email","%n Emails",m_emailCount);
  }
  if (!summary.isEmpty()) {
    QToolTip::add( this, summary );
  }

  // blink?
  blink(m_blinkEvent || m_blinkAnni || m_blinkEmail);

  repaint();
}

void KickPimWidget::onEventsChanged()
{
  onTimerCheckEvents();
}

/**
 * Counting new mails.
 *
 * Beware of these different counts:
 * - All mails stored at the accounts
 * - All new mails since the last mailcount reset (by opening the KickPIM menu
 * - All new mails since the last mailcheck (the last notification)
 *
 * The mail count since of the last mail check is stored in m_emailCount. 
 */
void KickPimWidget::onEmailsChanged()
{
  LogService::call("KickPimWidget","onEmailsChanged");

  KPMailMonitorThreadList&  list = KickPIM::rep()->mailMonitorThreads();
  QPtrListIterator<KickPimMailMonitorThread> itor( list );
  KickPimMailMonitorThread* thread = 0;

  int mailsAll = 0; // Sum of all mails
  int mailsNew = 0; // Sum of all new mails since last reset
  while ( itor.current()!=0 ) {
    thread = (KickPimMailMonitorThread*)itor.current();
    if (thread->monitor()->account()->isActive()) {
      int accMailsAll = thread->monitor()->allMessageCount();
      int accMailsNew = thread->monitor()->newMessageCount();
      LogService::logInfo(LogService::CAT_MAIL,"KickPimWidget:   "+QString::number(accMailsNew)+" new eMails from '"+thread->monitor()->account()->name()+"'");
      mailsAll += accMailsAll;
      mailsNew += accMailsNew;
    }
    ++itor;
  }  
  LogService::logInfo(LogService::CAT_MAIL,"KickPimWidget: "+QString::number(mailsNew)+"/"+QString::number(mailsAll)+" Emails for all accounts.");

  // check actions if new mails arrived!
  m_blinkEmail = (mailsNew>0);
    
  // action if mail arrived since last mail check
  if ( mailsNew > 0 && mailsAll > m_emailCount ) {
    // beep
    if (KickPIM::opt()->mailNotifyBeep) {
      QApplication::beep();
    }
    // play audio
    if (KickPIM::opt()->mailNotifySoundFile.length()>0) {
      KAudioPlayer::play( KickPIM::opt()->mailNotifySoundFile );
    }
    // notification window
    if (KickPIM::opt()->mailNotifyPopup) {
      QString mailText = "";
      mailText+=i18n("<b>%n new</b> Email","<b>%n new</b> Emails",mailsNew);
      mailText+="<br><br>";
      mailText+=i18n("(%n waiting Email)","(%n waiting Emails)",mailsAll);

      if (m_notifyWidget!=0) {
        disconnect(this,SLOT(onNotifyWidgetDestroyed()));
        m_notifyWidget->close( true );
        // delete m_notifyWidget;m_notifyWidget=0;
      }
      LogService::logInfo( LogService::CAT_KICKPIM, "NotifyWindow" );
      m_notifyWidget = new KickPimNotifyDialog(0,mailText);
      connect( m_notifyWidget, SIGNAL(destroyed()), this, SLOT(onNotifyWidgetDestroyed()) );
      m_notifyWidget->show();
    }
    // start app
    if (KickPIM::opt()->mailNotifyAppPath.length()>0) {
      QString appPath = KickPIM::opt()->mailNotifyAppPath;
      KRun::run( appPath, QStringList(""), appPath, 0);
    }
  }
  
  m_emailCount = mailsAll;

  updateWidget();
}



void KickPimWidget::resetState()
{
  KickPIM::opt()->lastReminded = QDateTime().currentDateTime();
  m_blinkAnni  = false;
  m_blinkEvent = false;
  m_blinkEmail = false;

  KPMailMonitorThreadList& list = KickPIM::rep()->mailMonitorThreads();
  QPtrListIterator<KickPimMailMonitorThread> itor( list );
  KickPimMailMonitorThread* thread = 0;
  while ( itor.current()!=0 ) {
    thread = (KickPimMailMonitorThread*)itor.current();    
    thread->monitor()->resetMailCount();
    ++itor;
  }
}

void KickPimWidget::dragEnterEvent(QDragEnterEvent* event)
{
  event->accept(true);
  showMenu();
}

void KickPimWidget::dragLeaveEvent(QDragLeaveEvent*/* event*/)
{
  // hideMenu();
}



void KickPimWidget::showMenu()
{
  LogService::call("KickPimWidget","showMenu");
  setMenuVisible(true);
}

void KickPimWidget::hideMenu()
{
  LogService::call("KickPimWidget","hideMenu");
  setMenuVisible(false);
}

void KickPimWidget::toggleMenu()
{
  LogService::call("KickPimWidget","toggleMenu");
  if (!setMenuVisible(true)) setMenuVisible(false);
}

void KickPimWidget::showEvents()
{
  LogService::call("KickPimWidget","showEvents");
  if (m_menu) m_menu->showEvents(true);
}

void KickPimWidget::onNotifyWidgetDestroyed() {
  LogService::logInfo( LogService::CAT_KICKPIM, "NotifyWindow destroyed." );
  m_notifyWidget = 0;
}
