/***************************************************************************
                          kickpimmenu.cpp  -  description
                             -------------------
    begin                : Die Okt 22 2002
    copyright            : (C) 2002 by Bert Speckels
    email                : bert@speckels.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <algorithm>
#include <map>
#include <sys/types.h>
#include <unistd.h>
#include <pwd.h>

#include <qlayout.h>
#include <qlineedit.h>
#include <qtoolbutton.h>
#include <qlabel.h>
#include <qcursor.h>
#include <qtable.h>
#include <qtimer.h>
#include <qapplication.h> 
#include <qdesktopwidget.h>

#include <klocale.h>
#include <ktextbrowser.h>
#include <kglobalsettings.h>
#include <kcombobox.h>
#include <kmessagebox.h>
#include <krun.h>
#include <kpopupmenu.h>

#include "dialogs/kpinfo.h"
#include "dialogs/kickpiminfodialog.h"
#include "dialogs/kickpimcontactchangedialog.h"
#include "dialogs/kickpimemaildialog.h"
#include "dialogs/kickpimcard.h"

#include "kickpim.h"
#include "kickpimwidget.h"
#include "kickpimdatepicker.h"
#include "kickpimsidebar.h"

#include "kickpimrepository.h"
#include "kickpimcontactview.h"
#include "kickpimcontactviewitem.h"
#include "kickpimeventview.h"
#include "kickpimemailview.h"
#include "LogService.h"

// #include "tools/kpballoon.h"
#include "kickpimmenu.h"

#include "kpcontact.h"
#include "kpkabcontact.h"

static const int POPUP_CARD  = 101;
static const int POPUP_MAIL  = 201;

static const int POPUP_EDIT  = 301;
static const int POPUP_REMOVE= 302;
static const int POPUP_ADD   = 303;
static const int POPUP_MULTIEDIT = 304;


KickPimMenu::KickPimMenu(QWidget *parent, const char *name )
  : KPopupFrame(parent,name)
{
  if (LogService::doLogConstruct) LogService::construct( "KickPimMenu" );

  m_isActive     = false;
  m_card         = 0;

  m_titleWidget  = 0;
  m_sideBar      = 0;
  m_dateLabel    = 0;
    
  m_contactView  = 0;
  m_eventView    = 0;
  m_emailView    = 0;
  m_catSelect    = 0;

  m_datePicker   = 0;

  m_lastDate     = QDate().currentDate();

  initLayout();

  initMenues();

  connect( KickPIM::rep(), SIGNAL( contactListChanged()),
           this,           SLOT  ( onContactsChanged()) );

  connect( KickPIM::rep(), SIGNAL(emailAccountsChanged()),
           this,           SLOT  (onEmailAccountsChanged()) );

  connect( KickPIM::rep(), SIGNAL(eventListChanged()),
           this,           SLOT  (onEventsChanged()) );

  // initialize timer for once after start actions
  m_afterStartTimer = new QTimer(this);
  connect( m_afterStartTimer, SIGNAL(timeout()), this, SLOT(onShortlyAfterStart()) );
  m_afterStartTimer->start( 1000 , TRUE );

  setAcceptDrops(TRUE);
}


void KickPimMenu::updateLayout()
{
  if (!isVisible()) {
    QSize size = sizeHint();
    resize( size.width(), preferredHeight() );
  }
}


void KickPimMenu::initLayout()
{
  if (LogService::doLogCall) LogService::call("KickPimMenu","initLayout");

  // the main widget which will be the root widget for all content of
  // the popup menu. This is currently the menu itself!
  QWidget* main = this;

  setSizePolicy( QSizePolicy(QSizePolicy::Maximum,QSizePolicy::Fixed) );

  QFont  font = this->font();

  bool displayLeftPanel  = KickPIM::opt()->showContacts;

  bool displayRightPanel = KickPIM::opt()->showEmails ||
                           KickPIM::opt()->showEvents ||
                           KickPIM::opt()->showCalendar;

  // better do this somewhere else?!
  QString tipps = (KickPIM::opt()->enableTooltips) ? "yes" : "no";
  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_KICKPIM,"KickPimMenu: Tooltips enabled? "+tipps);

  QBoxLayout* layout  = 0; // for temporary use


  //
  // vertical layout for info bar
  //

  QBoxLayout* infoLayout = new QVBoxLayout(main);
  infoLayout->setSpacing(0);
  infoLayout->setMargin (0);

  if (KickPIM::opt()->showInfo)
  {
    m_titleWidget = new QWidget(main,"InfoWidget");
    //m_titleWidget->setFixedHeight( 24 );
    m_titleWidget->setPaletteBackgroundColor( KGlobalSettings::activeTitleColor() );
    m_titleWidget->setPaletteForegroundColor( KGlobalSettings::activeTextColor() );
    infoLayout->addWidget(m_titleWidget,0);

    // horizontal layout in title area
    layout = new QHBoxLayout(m_titleWidget);
    layout->setSpacing(0);
    layout->setMargin(0);

    // icon
    layout->addSpacing(2);
    QLabel* iconLabel = new QLabel(m_titleWidget);
    iconLabel->setPixmap( QPixmap( KickPIM::rep()->dirOfIcons() + "hi24-kickpim-icon.png") );
    layout->addWidget(iconLabel,0);
  
    passwd* userInfo = getpwuid(geteuid());
    if (userInfo)
    {
      layout->addSpacing(8);
      //QString user = i18n("User")+": "+userInfo->pw_name;
      QString user = userInfo->pw_name;
      QLabel* username = new QLabel(user,m_titleWidget, "Title_User" );
      username->setFont( KickPIM::opt()->fontBold );
      username->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
      layout->addWidget(username,10);
    }
    if (displayLeftPanel && displayRightPanel)
    {
      // date
      KLocale locale("");
      QDate today = QDate().currentDate();
      QString dateString = locale.formatDate( today );
      m_dateLabel = new QLabel(dateString,m_titleWidget, "Title_Date" );
      m_dateLabel->setFont( KickPIM::opt()->fontBold );
      m_dateLabel->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
      layout->addWidget(m_dateLabel,0);
      layout->addSpacing(8); // right margin
    }

    layout=0;
  }

  //
  // horizontal layout for sidebar
  //

  QBoxLayout* sideLayout = new QHBoxLayout(main);
  sideLayout->setSpacing(0);
  sideLayout->setMargin (0);
  infoLayout->addLayout(sideLayout);

  // sidebar
  if (KickPIM::opt()->showSidebar)
  {
    QColor colort = paletteForegroundColor();
    QColor color1 = paletteBackgroundColor().dark(110);
    QColor color2 = KGlobalSettings::activeTitleColor();
    if (!KickPIM::opt()->showInfo)
    {
      color1 = KGlobalSettings::activeTitleColor();
      color2 = KGlobalSettings::activeTitleColor().dark(135);
      colort = KGlobalSettings::activeTextColor();
    }
    m_sideBar = new KickPimSideBar(main,"SidebarWidget");
    m_sideBar->setPaletteBackgroundColor( color1 );
    m_sideBar->setPaletteForegroundColor( colort );
    m_sideBar->setGradientColors( color1, color2 );
    sideLayout->addWidget(m_sideBar,0);
  }

  //
  // horizontal layout for content (inside sideLayout)
  //

  QBoxLayout* contentLayout = new QHBoxLayout(main);
  contentLayout->setSpacing(4);
  contentLayout->setMargin (4);
  sideLayout->addLayout(contentLayout);

  // left panel: contact layout (list + search field)
  
  layout = new QVBoxLayout(main);
  layout->setSpacing(4);
  contentLayout->addLayout(layout);

  if (KickPIM::opt()->showContacts)
  {
    m_contactView = new KickPimContactView (main ,"AddressView" );
    layout->addWidget(m_contactView);

    m_contactView->setMinimumSize( 150, 350 );

    if (KickPIM::opt()->showContactSearch)
    {
      QLineEdit* searchEdit = new QLineEdit(main);
      layout->addWidget(searchEdit);
      connect( searchEdit, SIGNAL( textChanged         (const QString&) ),
               this,       SLOT  ( onSearchTextChanged (const QString&) ) );
      connect( searchEdit, SIGNAL( returnPressed () ),
               this,       SLOT  ( onSearchNext  () ) );
    }

    if (KickPIM::opt()->showContactCategories) {
      m_catSelect = new KComboBox(main);
      rebuildContactCategories();
      layout->addWidget(m_catSelect);

      connect( m_catSelect, SIGNAL( activated         (int) ),
               this,        SLOT  ( onCategoryChanged (int) ) );
    }
    else {
      KickPIM::opt()->contactCategory="";
    }
  }

  // right panel: event/emaillist and calendar

  if (displayRightPanel)
  {
    bool sep = false;

    // right panel layout (vertical)
    layout = new QVBoxLayout(main);
    layout->setSpacing(4);
    contentLayout->addLayout(layout);

    // right panel: emails
    if (KickPIM::opt()->showEmails)
    {
      sep=true;

      m_emailView = new KickPimEmailView(main,"KickPimEmailView");
      layout->addWidget( m_emailView, 0);
    }

    // right panel: events / calendar
    if (KickPIM::opt()->showEvents)
    {
      if (sep) layout->addSpacing(10);
      sep=true;

      m_eventView = new KickPimEventView(main,"KickPimEventView");
      layout->addWidget( m_eventView, 1);
    }

    // right panel: calendar
    if (KickPIM::opt()->showCalendar)
    {
      if (sep) layout->addSpacing(10);
      sep=true;

      m_datePicker = new KickPimDatePicker(main);
      m_datePicker->setMinimumWidth(150);
      layout->addWidget(m_datePicker);
    }
  }

  // connect
  if (m_contactView)
  {
    connect(m_contactView, SIGNAL(contactDoubleClicked(KickPimContactViewItem*) ),
            this,          SLOT(onContactsDoubleClicked(KickPimContactViewItem*)) );

    connect(m_contactView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
            this,          SLOT(onContactsContextMenu(KListView*, QListViewItem*,const QPoint&)) );
  }
}

void KickPimMenu::initMenues()
{
  m_addrPopup = new KPopupMenu(this,"AddrPopup");
  m_addrPopup->insertItem(i18n("Email"),        POPUP_MAIL);
  m_addrPopup->insertSeparator();
  m_addrPopup->insertItem(i18n("View"),         POPUP_CARD);
  m_addrPopup->insertItem(i18n("Edit"),         POPUP_EDIT);
  m_addrPopup->insertItem(i18n("Notes&&Categories"),       POPUP_MULTIEDIT);
  m_addrPopup->insertSeparator();
  m_addrPopup->insertItem(i18n("Add Contact"),  POPUP_ADD);
  m_addrPopup->insertSeparator();
  m_addrPopup->insertItem(i18n("Remove"),       POPUP_REMOVE);

  m_distPopup = new KPopupMenu(this,"DistPopup");
  m_distPopup->insertItem(i18n("Email"),        POPUP_MAIL);
}

KickPimMenu::~KickPimMenu()
{
  if (LogService::doLogConstruct) LogService::destruct("KickPimMenu");

  delete m_contactView;     m_contactView=0;
  delete m_eventView;       m_eventView=0;
  delete m_emailView;       m_emailView=0;
  delete m_card;            m_card=0;
}




void KickPimMenu::showAddrSheet(KPContact* contact)
{
  if (!m_card)
  {
    m_card = new KickPimCard(0,"card");
    m_card->setMinimumSize(100,100);
    m_card->setMaximumSize(640,480);
  }

  QDesktopWidget* desktop = QApplication::desktop();
  int xres = desktop->width();
  int yres = desktop->height();

  m_card->setBaseSize(300,200);
  m_card->setContact(contact);
  m_card->updateGeometry();
  m_card->move(QPoint(-1000,-1000)); // hide it
  m_card->show(); // show it: calculates new size
  // now calculate the correct position of the card on screen
  QPoint pt = QCursor::pos();
  int x = pt.x()-m_card->width()/2;
  int y = pt.y()-m_card->height()/2;

  if (x<0) x=0;
  if (y<0) y=0;
  if (x+m_card->width()  > xres) x = xres - m_card->width();
  if (y+m_card->height() > yres) y = yres - m_card->height();

  pt.setX(x);pt.setY(y);
  m_card->move(pt);
}

void KickPimMenu::fillContactList()
{

}


void KickPimMenu::onContactsDoubleClicked(KickPimContactViewItem* item)
{
  if (!item) return;

  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_CONTACTS, "Doubleclick on contact '"+item->text(0)+"'");

  switch (KickPIM::opt()->defaultAction)
  {
    case KickPimOptions::ADDR_ACTION_CARD:
      if (!item->isDistributionList()) {
        doShowContactCard(item->contact());
      }
      break;
    case KickPimOptions::ADDR_ACTION_MAIL:
      doNewEmail();
      break;
    case KickPimOptions::ADDR_ACTION_EDIT:
      if (!item->isDistributionList())
         doEditContact(item->contact());
      break;
    default:
      break;
  }

}

 
void KickPimMenu::doRemoveContacts()
{
  KickPimContactViewItem* item = (KickPimContactViewItem*)m_contactView->firstChild();
  int result = KMessageBox::questionYesNo(
                        0,
                        i18n("Do you really want to remove the contact(s) from your addressbook?"),
                        i18n("Remove contacts"), KStdGuiItem::yes(), KStdGuiItem::no());
  if (result!=KMessageBox::Yes) return;

  while (item) {
    if (item->isSelected() && !item->isDistributionList()) {
      item->contact()->remove();
    }
    item = (KickPimContactViewItem*)item->itemBelow();
  }
  KABC::StdAddressBook::save(); // TODO: make indenpendent from special address source  
}

void KickPimMenu::doShowContactCard(KPContact* contact)
{
  forceHide();
  showAddrSheet(contact);
}



void KickPimMenu::doNewEmail()
{
  KPContact*              contact = 0;
  KickPimContactViewItem* item    = 0;

  // check if there are emailaddresses missing
  KickPimEmailDialog dlg;
  item = (KickPimContactViewItem*)m_contactView->firstChild();
  while (item)
  {
    if (item->isSelected())
    {
      contact = item->contact();
      if (contact && contact->emailAddresses().isEmpty()) dlg.addContact(contact);
    }
    item = (KickPimContactViewItem*)item->itemBelow();
  }

  // handle missing email addresses
  if (dlg.count()>0)
  {
    // dlg.setText( i18n("There are contacts without an email address. You may insert them now ...") );
    dlg.exec();
  }

  // create mail list
  QStringList targetEmailList;
  item = (KickPimContactViewItem*)m_contactView->firstChild();
  while (item)
  {
    if (item->isSelected())
    {
      if (item->isDistributionList() && item->distributionList())
      {
        targetEmailList += item->distributionList()->emails();
      }
      else
      {
        contact = item->contact();
        QStringList emailList = contact->emailAddresses();
        if (!emailList.isEmpty())
        {
          QString email = emailList.first();
          if (email.find("<")<0)
          {
            QStringMap names = contact->names();
            email = names["givenName"] + " " + names["familyName"]+" <"+email+">";
          }
          targetEmailList.push_back( email );
        }
      }
    }
    item = (KickPimContactViewItem*)item->itemBelow();
  }  
    
  // Is the only one email in the list?
  // Then check if this one contact owns more than one email address.
  // Display popupmenu for address-selection
  if (contact && targetEmailList.count()==1)
  {
    QStringList emailList = contact->emailAddresses();
    if (emailList.count()>1)
    {
      KPopupMenu* emailPopup = new KPopupMenu(this,"EmailPopup");
      QString email=emailList.first();
      QStringList::iterator it;
      int count=1;
      for (it = emailList.begin();it!=emailList.end();it++)
      {        
        emailPopup->insertItem( (QString)*it ,count);
        count++;
      }
      emailPopup->insertSeparator();
      emailPopup->insertItem( i18n("To all of them"), 0 );

      QPoint pos = QCursor::pos();
      int selection = emailPopup->exec( pos );

      targetEmailList.clear();

      if (selection>=0)
      {
        if (selection==0) targetEmailList = emailList;
        else
        {
          targetEmailList.clear();
          QStringList::iterator it;
          int count=1;
          for (it = emailList.begin();it!=emailList.end();it++)
          {
            if (count==selection)
            {
              QString email = (QString)*it;
              if ( email.find("<")<0 )
              {
                QStringMap names = contact->names();
                email = names["givenName"] + " " + names["familyName"] + " <"+email+">";
              }
              targetEmailList.append( email );
            }
            count++;
          }
        }
      }
            
      delete emailPopup;
    }
  }

  // send email
  if (!targetEmailList.isEmpty())
  {
    forceHide();
    KURL url( "mailto:" + targetEmailList.join(",") );
    new KRun(url);
  }
}


void KickPimMenu::doEditContact(KPContact* contact)
{
  forceHide();
  contact->edit();
  //m_contactView->onContactsChanged();
  //m_eventView->onEventListChanged();
}

void KickPimMenu::doMultiEdit()
{
  QStringList notesList;
  QStringList catList;
  QStringList list;
  KPContact* contact = 0;
  // get data from all selected contacts
  KickPimContactViewItem* item = (KickPimContactViewItem*)m_contactView->firstChild();
  QString str;
  while (item)
  {
    if (item->isSelected())
    {
      contact = item->contact();
      str = contact->note();
      if ( str.length()>0 && !notesList.contains(str) ) {
        notesList += str;
      }
      list = contact->categories();
      for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) {
        str = (QString)*it;
        if  ( !catList.contains(str) ) catList += str;
      }
    }
    item = (KickPimContactViewItem*)item->itemBelow();
  }

  KickPimContactChangeDialog dlg;
  dlg.setNoteList( notesList );
  dlg.setCategories (catList);
  if (dlg.exec() == QDialog::Accepted)
  {
    if (dlg.changeCategories() || dlg.changeNotes())
    {
      // commit changes
      QString     note       = dlg.note();
      QStringList categories = dlg.categories();

      item = (KickPimContactViewItem*)m_contactView->firstChild();
      while (item)
      {
        if (item->isSelected())
        {
          contact = item->contact();
          if (dlg.changeNotes())      contact->setNote       ( note );
          if (dlg.changeCategories()) contact->setCategories ( categories );
          contact->changed();
        }
        item = (KickPimContactViewItem*)item->itemBelow();
      }
      KABC::StdAddressBook::save(); // TODO: make indenpendent from special address source      
    } // if change
  }; // if QDialog::Accepted
}

void KickPimMenu::onContactsContextMenu(KListView*, QListViewItem* qitem, const QPoint&)
{
  if (!m_addrPopup)                                            return;
  if (!qitem || !dynamic_cast<KickPimContactViewItem*>(qitem)) return;

  KickPimContactViewItem* item = static_cast<KickPimContactViewItem*>(qitem);

  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_CONTACTS, "Context menu on contact '"+item->text(0)+"'");

  QPoint p = QCursor::pos();
  KPopupMenu* popup = 0;

  bool multipleSelections = m_contactView->selectedItems().count()>1;

  if (item->isDistributionList()) {
    popup = m_distPopup;
  }
  else {
    popup = m_addrPopup;
    m_addrPopup->setItemEnabled(POPUP_EDIT,      !multipleSelections);
    m_addrPopup->setItemEnabled(POPUP_CARD,      !multipleSelections);
  }

  bool enabled = KPDynamicTip::isActive();
  if (enabled) KPDynamicTip::setActive(false);
  int id = popup->exec( p );
  if (enabled) KPDynamicTip::setActive(true);

  switch (id)
  {
    case POPUP_CARD:
      if (!item->isDistributionList()) {
        doShowContactCard(item->contact());
      }
      break;

    case POPUP_MULTIEDIT:
      if (!item->isDistributionList()) {
        doMultiEdit();
      }
      break;

    case POPUP_EDIT:
      if (!item->isDistributionList()) {
        doEditContact(item->contact());
      }
      break;

    case POPUP_MAIL:
      doNewEmail();
      break;

    case POPUP_ADD:
      KickPIM::rep()->addContact();
      break;

    case POPUP_REMOVE:
      doRemoveContacts();
      break;
  }
}


void KickPimMenu::onContactsPopupSelection (int)
{
}


/*
void KickPimMenu::fillEvents()
{
  if (!m_eventView) return;

  int nr=0;

  m_eventView->clear();

  KPEventList list = KickPIM::rep()->events();
  KPEventEntry* event = list.first();
  while ( event )
  {
    if (
         (event->type()==KPEventEntry::BIRTHDAY    && KickPIM::opt()->showBirthdays) ||
         (event->type()==KPEventEntry::ANNIVERSARY && KickPIM::opt()->showAnniversaries)
        )
    {
      m_eventView->addEvent( event );
    }

    event = list.next();
    nr++;
    // if (KickPIM::opt()->maxNumBirthdays>0 && KickPIM::opt()->maxNumBirthdays<=nr) break;
  }

}
*/

void KickPimMenu::paintEvent(QPaintEvent*)
{
  // QPainter painter(this);
}


void KickPimMenu::showEvent(QShowEvent*)
{
  m_isActive = true;

  // updates displayed date
  if (m_datePicker) m_datePicker->setDate( QDate().currentDate() );
  KPDynamicTip::setActive( KickPIM::opt()->enableTooltips );
}

void KickPimMenu::hideEvent(QHideEvent*)
{
  KickPimWidget* widget = dynamic_cast<KickPimWidget*>(parentWidget());

  // resets state of blinking icons and mail-monitors
  if (widget) widget->resetState();  
  KPDynamicTip::setActive( false );
}

void KickPimMenu::forceHide()
{
  hide();
  m_isActive = false;
}

void KickPimMenu::closeEvent(QCloseEvent* event)
{
  if (LogService::doLogCall) LogService::call("KickPimMenu","closeEvent");
  m_isActive = false;

  // KickPimWidget* widget = dynamic_cast<KickPimWidget*>(parentWidget());

  if (parentWidget())
  {
    QPoint pt = parentWidget()->mapFromGlobal( QCursor::pos() );
    if (parentWidget()->frameGeometry().contains(pt))
    {
      m_isActive=true;
    }
  }

  //
  // reset state of the panel widget
  //

  event->accept();
}


void KickPimMenu::showEvents( bool /*force*/ )
{
/*
  if (!force && !KickPIM::opt()->remindAnniversaries) return;

  int     countNew = 0;
  QString text     = "";
  QString eventText= "";

  KPEventList list = KickPIM::rep()->events();
  KPEvent* event = list.first();

  while ( event )
  {
    // date and/or time of event
    QString dateTime = "";
    if (event->date().isValid())
    {
      int distance = KickPIM::rep()->distanceToDate(event->date(),event->ignoreYear());
      if (distance > - KickPIM::rep()->options()->remindAnniPast    &&
          distance <   KickPIM::rep()->options()->remindAnniFuture    )
      {
        dateTime+=event->date().toString("dd.MM.");

        if (event->time().isValid())
        {
          if (!dateTime.isEmpty()) dateTime+=" ";
          dateTime+=event->time().toString("hh:mm");
        }
        QString text = event->text();

        eventText += dateTime;
        if (!dateTime.isEmpty()) eventText+=" ";
        eventText += text;
        eventText += "<br>";

        countNew++;
      }
    }

    event = list.next();
  }

  if (force || countNew>0)
  {
    if (!m_eventDialog) // create event dialog
    {
       m_eventDialog = new KickPimEventDialog(parentWidget());
       m_eventDialog->move   ( KickPIM::opt()->remindWindowPos  );
       m_eventDialog->resize ( KickPIM::opt()->remindWindowSize );
    }
    m_eventDialog->setText(eventText);
    m_eventDialog->show();
  }
*/
}

void KickPimMenu::preOptionSave()
{
  /*
  if (m_eventDialog)
  {
    KickPIM::opt()->remindWindowPos  = m_eventDialog->pos();
    KickPIM::opt()->remindWindowSize = m_eventDialog->size();
  }
  */
}


void KickPimMenu::onSearchNext() {
  LogService::logInfo( LogService::CAT_CONTACTS, "Search next hit for '"+m_lastSearch+"'");
  if (m_lastSearch.length()>0) {
    JumpToContact(m_lastSearch, true);
  }
}

void KickPimMenu::onSearchTextChanged(const QString& text)
{
  LogService::logInfo( LogService::CAT_CONTACTS, "Search Text '"+text+"'");
  m_lastSearch = text;
  JumpToContact(text);
}


bool KickPimMenu::JumpToContact(const QString& name, bool searchNext)
{
  KickPimContactViewItem* item = 0;
  // check if the first item is already a hit
  if (searchNext) {
    KickPimContactViewItem* checkItem = (KickPimContactViewItem*)m_contactView->currentItem();
    if (checkItem) {
      LogService::logInfo( LogService::CAT_CONTACTS, "Current Item found: "+checkItem->text(0));
      if ( checkItem->text(0).find(name,0,false)>=0 && checkItem->itemBelow())
          item = (KickPimContactViewItem*)checkItem->itemBelow();
    } 
  }
  else item = (KickPimContactViewItem*)m_contactView->firstChild();

  // search for the next
  while (item)
  {
    if (item->text(0).find(name,0,false)>=0)
    {
      m_contactView->ensureItemVisible(item);
      m_contactView->clearSelection();
      m_contactView->setSelected(item,true);
      m_contactView->setCurrentItem(item);

      return true;
    }
    item = (KickPimContactViewItem*)item->itemBelow();
  }
  return false;
}

void KickPimMenu::onEventsChanged() {
  if (LogService::doLogCall) LogService::call( "KickPimMenu","onEventsChanged");
  if (m_eventView!=0) m_eventView->updateEventList();
  updateLayout();
}

void KickPimMenu::onEmailAccountsChanged() {
  if (LogService::doLogCall) LogService::call( "KickPimMenu","onEmailAccountsChanged");
  if (m_emailView!=0) m_emailView->updateEmailAccountList();
  updateLayout();
}

void KickPimMenu::onContactsChanged() {
  if (LogService::doLogCall) LogService::call( "KickPimMenu","onContactsChanged");
  rebuildContactCategories();
  if (m_contactView!=0) m_contactView->updateContactList();
}

void KickPimMenu::rebuildContactCategories() {
  if (m_catSelect!=0) {
    if (LogService::doLogInfo) LogService::logInfo( LogService::CAT_CONTACTS,"KickPimMenu.rebuildContactCategories. current='"+KickPIM::opt()->contactCategory+"'");
    m_catSelect->clear();
    QStringList& list = KickPIM::rep()->contactCategories();
    list += i18n(KickPimContactView::CAT_DISTLISTS);
    list += i18n(KickPimContactView::CAT_REST);
    list += i18n(KickPimContactView::CAT_ALL);

    QString entry;
    int selectIndex = -1;
    int index=-1;
    for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) {
      ++index;
      entry = (QString)*it;
      m_catSelect->insertItem( entry, index );
      if (entry==KickPIM::opt()->contactCategory) {
        selectIndex = index;
      }
      if (LogService::doLogInfo) LogService::logInfo( LogService::CAT_CONTACTS,"KickPimMenu.rebuildContactCategories. insert '"+entry+"' (#"+QString::number(index)+")");
    }
    if (selectIndex < 0) {
      selectIndex = list.findIndex( i18n(KickPimContactView::CAT_ALL) );
    }
    if (LogService::doLogInfo) LogService::logInfo( LogService::CAT_CONTACTS,"KickPimMenu.rebuildContactCategories. selecting entry #"+QString::number(selectIndex));
    m_catSelect->setCurrentItem( selectIndex );
    KickPIM::opt()->contactCategory = m_catSelect->currentText();
  }
}

void KickPimMenu::onCategoryChanged( int index )
{
  QString text=i18n(KickPimContactView::CAT_ALL);
  if (index>=0) {
    text = m_catSelect->text(index);
    KickPIM::opt()->contactCategory = text;
  }
  m_contactView->updateContactList();
}


void KickPimMenu::onShortlyAfterStart()
{
  if (LogService::doLogCall) LogService::call("KickPimMenu","onShortlyAfterStart");
  if (KickPIM::opt()->isInfoActive(KickPimOptions::INFO_APPLET)) {
    KickPIM::opt()->deactivateInfo(KickPimOptions::INFO_APPLET);
    KickPimInfoDialog dlg;
    dlg.setInfoText( i18n(infoKickPimApplet) );
    dlg.setPicture( QPixmap( KickPIM::rep()->dirOfInfo() + "info-kickpim-applet.png") );
    dlg.exec();
  }
}


/** No descriptions */
void KickPimMenu::onDateChanged()
{
  if (LogService::doLogCall) LogService::call("KickPimMenu","onDateChanged");
  if (m_dateLabel==0) return;

  KLocale locale("");
  QDate today = QDate().currentDate();
  QString dateString = locale.formatDate( today );
  m_dateLabel->setText(dateString);

  if (m_lastDate!=today)
  {
    KickPIM::rep()->reload();
  }
  m_lastDate = today;
}

int KickPimMenu::preferredHeight()
{
  QDesktopWidget* desktop = QApplication::desktop();
  int yres = desktop->height();

  int heightLeft  = (m_contactView?m_contactView->preferredHeight():0)
                  + 50;

  int heightRight = (m_datePicker?m_datePicker->height():0)
                  + (m_emailView?m_emailView->preferredHeight():0)
                  + (m_eventView?m_eventView->preferredHeight():0)
                  + 30;

  int height = (int) ( (heightLeft + heightRight*2) / 3 );
  if (!m_contactView) height = heightRight;
  if (!m_emailView && !m_eventView && !m_datePicker) height = heightLeft;

  if (LogService::doLogInfo) LogService::logInfo(LogService::CAT_KICKPIM,"KickPimMenu: Height: " + QString::number(height)+" pixel");

  if (height < 200)        height = 200;
  if (height > (yres-100)) height = yres-100;

  return height;
}

void KickPimMenu::hideMe() {
  forceHide();
}

/** Crazy: Without this KickPIM may crash immediatly when hitting the ESCape key
 *
 */
void KickPimMenu::keyPressEvent(QKeyEvent* e)
{
  if (e->key() == Qt::Key_Escape) {
    LogService::logInfo( LogService::CAT_KICKPIM, "Escape Key pressed in KickPimMenu");
    hideMe();
  }
  e->ignore();
}

