/*  -*- C++ -*-
    This file is part of the KDE libraries
    Copyright (C) 1997 Tim D. Gilman (tdgilman@best.org)
              (C) 1998-2001 Mirko Boehm (mirko@kde.org)
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include "kickpimdatepicker.h"
#include <kglobal.h>
#include <kapplication.h>
#include <klocale.h>
#include <kiconloader.h>
#include <qframe.h>
#include <qpainter.h>
#include <qdialog.h>
#include <qstyle.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qvalidator.h>
#include <kdebug.h>
#include <knotifyclient.h>

#include "kpevent.h"
#include "kickpim.h"
#include "kickpimdatetable.h"
//#include "KickPimDatePicker.moc"

class KickPimDatePicker::KickPimDatePickerPrivate
{
public:
    KickPimDatePickerPrivate() : closeButton(0L), selectWeek(0L) {}

    QToolButton *closeButton;
    QToolButton *selectWeek;
};


KickPimDatePicker::KickPimDatePicker(QWidget *parent, QDate dt, const char *name)
  : QFrame(parent,name)
{
  init( dt );
}

KickPimDatePicker::KickPimDatePicker(QWidget *parent, QDate dt, const char *name, WFlags f)
  : QFrame(parent,name, f)
{
  init( dt );
}

KickPimDatePicker::KickPimDatePicker( QWidget *parent, const char *name )
  : QFrame(parent,name)
{
  init( QDate::currentDate() );
}

void KickPimDatePicker::init( const QDate &dt )
{
  QBoxLayout* vlayout = new QVBoxLayout(this);

  vlayout->addWidget( dateDisplay = new QLabel(this) );
  dateDisplay->setFont(KickPIM::opt()->fontBold);
  dateDisplay->setAlignment(Qt::AlignCenter);

  vlayout->addWidget( table = new KickPimDateTable(this) );
  table->setFont(KickPIM::opt()->fontNormal);

  QBoxLayout* hlayout = new QHBoxLayout();
  vlayout->addLayout(hlayout);

  hlayout->addWidget( yearBackward = new QToolButton(this)  );
  hlayout->addWidget( monthBackward = new QToolButton(this) );
  hlayout->addStretch(1);
  hlayout->addWidget( monthForward = new QToolButton(this)  );
  hlayout->addWidget( yearForward = new QToolButton(this)   );

  yearBackward  ->setFixedSize(30,24);
  monthBackward ->setFixedSize(30,24);
  monthForward  ->setFixedSize(30,24);
  yearForward   ->setFixedSize(30,24);

  QToolTip::add(yearForward, i18n("Next year"));
  QToolTip::add(yearBackward, i18n("Previous year"));
  QToolTip::add(monthForward, i18n("Next month"));
  QToolTip::add(monthBackward, i18n("Previous month"));

  val = new KDateValidator(this);

  d = new KickPimDatePickerPrivate();

  yearForward  ->setText(">>");
  yearBackward ->setText("<<");
  monthForward ->setText(">");
  monthBackward->setText("<");

  yearForward->setFont(KickPIM::opt()->fontBold);
  yearBackward->setFont(KickPIM::opt()->fontBold);
  monthForward->setFont(KickPIM::opt()->fontBold);
  monthBackward->setFont(KickPIM::opt()->fontBold);
  
  setDate(dt); // set button texts
  connect(table, SIGNAL(dateChanged(QDate)), SLOT(dateChangedSlot(QDate)));
  connect(table, SIGNAL(tableClicked()), SLOT(tableClickedSlot()));
  connect(monthForward, SIGNAL(clicked()), SLOT(monthForwardClicked()));
  connect(monthBackward, SIGNAL(clicked()), SLOT(monthBackwardClicked()));
  connect(yearForward, SIGNAL(clicked()), SLOT(yearForwardClicked()));
  connect(yearBackward, SIGNAL(clicked()), SLOT(yearBackwardClicked()));
  table->setFocus();
}

KickPimDatePicker::~KickPimDatePicker()
{
  delete d;
}

bool
KickPimDatePicker::eventFilter(QObject *o, QEvent *e )
{
   if ( e->type() == QEvent::KeyPress ) {
      QKeyEvent *k = (QKeyEvent *)e;

      if ( (k->key() == Qt::Key_Prior) ||
           (k->key() == Qt::Key_Next)  ||
           (k->key() == Qt::Key_Up)    ||
           (k->key() == Qt::Key_Down) )
       {
          QApplication::sendEvent( table, e );
          table->setFocus();
          return TRUE; // eat event
       }
   }
   return QFrame::eventFilter( o, e );
}

void
KickPimDatePicker::resizeEvent(QResizeEvent*)
{ 
}

void
KickPimDatePicker::dateChangedSlot(QDate date)
{
    kdDebug() << "KickPimDatePicker::dateChangedSlot: date changed (" << date.year() << "/" << date.month() << "/" << date.day() << ")." << endl;
    //dateDisplay->setText(KGlobal::locale()->formatDate(date, true));
    dateDisplay->setText(date.toString("ddd dd. MMM. yyyy"));
    emit(dateChanged(date));
}

void
KickPimDatePicker::tableClickedSlot()
{
  kdDebug() << "KickPimDatePicker::tableClickedSlot: table clicked." << endl;
  emit(dateSelected(table->getDate()));
  emit(tableClicked());
}

const QDate&
KickPimDatePicker::getDate() const
{
  return table->getDate();
}

const QDate &
KickPimDatePicker::date() const
{
    return table->getDate();
}

bool
KickPimDatePicker::setDate(const QDate& date)
{
  if(date.isValid())
  {
    QString temp;
    // -----
    table->setDate(date);
    //dateDisplay->setText(KGlobal::locale()->formatDate(date, true));
    dateDisplay->setText(date.toString("ddd dd. MMM. yyyy"));
    return true;
  }
  else
  {
    kdDebug() << "KickPimDatePicker::setDate: refusing to set invalid date." << endl;
    return false;
  }
}

void
KickPimDatePicker::monthForwardClicked()
{
    setDate( table->getDate().addMonths(1) );
}

void
KickPimDatePicker::monthBackwardClicked()
{
    setDate( table->getDate().addMonths(-1) );
}

void
KickPimDatePicker::yearForwardClicked()
{
    setDate( table->getDate().addYears(1) );
}

void
KickPimDatePicker::yearBackwardClicked()
{
    setDate( table->getDate().addYears(-1) );
}

void
KickPimDatePicker::selectWeekClicked()
{
  return;
  /*
  int week;
  KPopupFrame* popup = new KPopupFrame(this);
  KDateInternalWeekSelector* picker = new KDateInternalWeekSelector(fontsize, popup);
  // -----
  picker->resize(picker->sizeHint());
  popup->setMainWidget(picker);
  connect(picker, SIGNAL(closeMe(int)), popup, SLOT(close(int)));
  picker->setFocus();
  if(popup->exec(d->selectWeek->mapToGlobal(QPoint(0, d->selectWeek->height()))))
    {
      QDate date;
      int year;
      // -----
      week=picker->getWeek();
      date=table->getDate();
      year=date.year();
      // ----- find the first selectable day in this week (hacky solution :)
      date.setYMD(year, 1, 1);
      while (weekOfYear(date)>50)
          date=date.addDays(1);
      while (weekOfYear(date)<week && (week!=53 || (week==53 &&
            (weekOfYear(date)!=52 || weekOfYear(date.addDays(1))!=1))))
          date=date.addDays(1);
      if (week==53 && weekOfYear(date)==52)
          while (weekOfYear(date.addDays(-1))==52)
              date=date.addDays(-1);
      // ----- set this date
      setDate(date);
    } else {
         KNotifyClient::beep();
    }
  delete popup;
  */
}

void
KickPimDatePicker::selectMonthClicked()
{
/*
  int month;
  KPopupFrame* popup = new KPopupFrame(this);
  KDateInternalMonthPicker* picker = new KDateInternalMonthPicker(fontsize, popup);
  // -----
  picker->resize(picker->sizeHint());
  popup->setMainWidget(picker);
  picker->setFocus();
  connect(picker, SIGNAL(closeMe(int)), popup, SLOT(close(int)));
  if(popup->exec(selectMonth->mapToGlobal(QPoint(0, selectMonth->height()))))
    {
      QDate date;
      int day;
      // -----
      month=picker->getResult();
      date=table->getDate();
      day=date.day();
      // ----- construct a valid date in this month:
      date.setYMD(date.year(), month, 1);
      date.setYMD(date.year(), month, QMIN(day, date.daysInMonth()));
      // ----- set this month
      setDate(date);
    } else {
      KNotifyClient::beep();
    }
  delete popup;
  */
}

void
KickPimDatePicker::selectYearClicked()
{
/*
  int year;
  KPopupFrame* popup = new KPopupFrame(this);
  KDateInternalYearSelector* picker = new KDateInternalYearSelector(fontsize, popup);
  // -----
  picker->resize(picker->sizeHint());
  popup->setMainWidget(picker);
  connect(picker, SIGNAL(closeMe(int)), popup, SLOT(close(int)));
  picker->setFocus();
  if(popup->exec(selectYear->mapToGlobal(QPoint(0, selectMonth->height()))))
    {
      QDate date;
      int day;
      // -----
      year=picker->getYear();
      date=table->getDate();
      day=date.day();
      // ----- construct a valid date in this month:
      date.setYMD(year, date.month(), 1);
      date.setYMD(year, date.month(), QMIN(day, date.daysInMonth()));
      // ----- set this month
      setDate(date);
    } else {
      KNotifyClient::beep();
    }
  delete popup;
  */
}

void
KickPimDatePicker::setEnabled(bool enable)
{
  QWidget *widgets[]= {
    yearForward, yearBackward, monthForward, monthBackward,
    //selectMonth, selectYear,
    dateDisplay, table, d->selectWeek };
  const int Size=sizeof(widgets)/sizeof(widgets[0]);
  int count;
  // -----
  for(count=0; count<Size; ++count)
    {
      widgets[count]->setEnabled(enable);
    }
}


QSize
KickPimDatePicker::sizeHint() const
{
  QSize tableSize=table->sizeHint();
  QWidget *buttons[]={
    yearBackward,
    monthBackward,
    //selectMonth,
    //selectYear,
    monthForward,
    yearForward,
    d->closeButton
  };
  const int NoOfButtons=sizeof(buttons)/sizeof(buttons[0]);
  QSize sizes[NoOfButtons];
  int cx=0, cy=0, count;
  // ----- store the size hints:
  for(count=0; count<NoOfButtons; ++count)
    {
      if ( buttons[count] )
          sizes[count]=buttons[count]->sizeHint();
      else
          sizes[count] = QSize(0,0);
      /*
      if(buttons[count]==selectMonth)
      {
        QSize metricBound = style().sizeFromContents(QStyle::CT_ToolButton, selectMonth, maxMonthRect);
        cx+=QMAX(metricBound.width(), maxMonthRect.width()+2*QApplication::style().pixelMetric(QStyle::PM_ButtonMargin));
      }
      else
      */
      {
        cx+=sizes[count].width();
      }
      cy=QMAX(sizes[count].height(), cy);
    }
  // ----- calculate width hint:
  cx=QMAX(cx, tableSize.width()); // line edit ignored
  // ----- calculate height hint:
  cy+=tableSize.height()+dateDisplay->sizeHint().height();
  return QSize(cx, cy);
}


void
KickPimDatePicker::setCloseButton( bool enable )
{
    if ( enable == (d->closeButton != 0L) )
        return;

    if ( enable ) {
        d->closeButton = new QToolButton( this );
        QToolTip::add(d->closeButton, i18n("Close"));
        d->closeButton->setPixmap( SmallIcon("remove") );
        connect( d->closeButton, SIGNAL( clicked() ),
                 topLevelWidget(), SLOT( close() ) );
    }
    else {
        delete d->closeButton;
        d->closeButton = 0L;
    }

    updateGeometry();
}

bool KickPimDatePicker::hasCloseButton() const
{
    return (d->closeButton != 0L);
}

int KickPimDatePicker::weekOfYear(QDate date)
{
    // Calculate ISO 8601 week number (taken from glibc/Gnumeric)
    int year, week, wday, jan1wday, nextjan1wday;
    QDate jan1date, nextjan1date;

    year=date.year();
    wday=date.dayOfWeek();

    jan1date=QDate(year,1,1);
    jan1wday=jan1date.dayOfWeek();

    week = (date.dayOfYear()-1 + jan1wday-1)/7 + ((jan1wday-1) == 0 ? 1 : 0);

    /* Does date belong to last week of previous year? */
    if ((week == 0) && (jan1wday > 4 /*THURSDAY*/)) {
        QDate tmpdate=QDate(year-1,12,31);
        return weekOfYear(tmpdate);
    }

    if ((jan1wday <= 4 /*THURSDAY*/) && (jan1wday > 1 /*MONDAY*/))
        week++;

    if (week == 53) {
        nextjan1date=QDate(year+1, 1, 1);
        nextjan1wday = nextjan1date.dayOfWeek();
        if (nextjan1wday <= 4 /*THURSDAY*/)
            week = 1;
    }

    return week;
}

void KickPimDatePicker::virtual_hook( int /*id*/, void* /*data*/ )
{ /*BASE::virtual_hook( id, data );*/ }


void KickPimDatePicker::addEvent(KPEvent*)
{
  
}


