/***************************************************************************
 *   Copyright (C) 2005 praKsys                                            *
 *   gpernot@praksys.org                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.             *
 ***************************************************************************/

#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qmap.h>
#include <qpixmap.h>

#include <klocale.h>
#include <kdebug.h>
#include <ktempdir.h>
#include <kgenericfactory.h>

#include "settings.h"
#include "datastruct/aux.h"
#include "datastruct/DItem.h"
#include "datastruct/DDataControl.h"
#include "datastruct/DDataItem.h"

#include "kdissstx.h"

K_EXPORT_COMPONENT_FACTORY(libkdissstx, kdissstxFactory) 
	QObject *kdissstxFactory::createObject(QObject *, const char *, const char *, const QStringList &) { return new kdissstx; }

const QString kdissstx::identifikation()
{
	return "kdissstx";
}

const QString kdissstx::fullName()
{
	return "StructuredText";
}

const QString kdissstx::group()
{
	return "Plain Text";
}

const QString kdissstx::description()
{
	return i18n("This template generates a StructuredText document for use in Plone. The pictures will be included in the directory created.");
}

const QString kdissstx::quickstart()
{
	return i18n("Go into the directory created and view main.txt with a text editor (eg: Kate)\n");
}

static const QString indent_string = "\n          ";
QString indent(const QString &str, int level)
{
	QStringList strings( QStringList::split( "\n", str ) );
	QStringList::Iterator i;

	for ( i = strings.begin(); i != strings.end(); ++i )
	{
	     *i = indent_string.left( level ) + (*i).simplifyWhiteSpace() + '\n';
	}
	return strings.join("");
}

void outputPixParagraph(DDataItem* item, QTextStream & s, int level)
{
	if (item->m_pix.isNull())
		return;
	s << indent("</p><div align=\"center\"><img src=\"pics/" + item->m_picfilename +
		    "\" alt=\""  + item->m_piccaption + "\" /></div><p>", level)<<"\n";
}

void kdissstx::outputLinks(DDataItem* item, QTextStream & s, int level)
{
	for (unsigned int i=0; i<item->m_urllist.count(); i++)
	{
		if (!item->m_urllist[i].m_caption.length())
			s <<indent( item->m_urllist[i].m_url, level)<<"\n";
		else
                        s << indent( "\"" + item->m_urllist[i].m_caption + "\":" + item->m_urllist[i].m_url,
				     level)<<"\n";
	}
}

void kdissstx::outputComment(DDataItem* item, QTextStream & s, int level)
{
        if (item->m_comment.length()) {
                s<<indent("::", level)<<"\n";
                s<<indent(item->m_comment, level + 1)<<"\n";
	}
}

void kdissstx::writeItem(DDataItem* item, QTextStream & s, int level)
{
	DGenerator::writeItem(item, s, level);

        // print the abstract
	//s<<i18n("add a space after the:", "Abstract: ")<<wipeTags(item->m_text)<<"\n\n\n";
	s<<indent( item->m_summary, level );
	s<<indent( wipeTags(item->m_text), level + 1 );
	outputPixParagraph(item, s, level + 1 );
	outputComment(item, s, level + 1 );
	outputLinks(item, s, level + 1 );

	// write recursively the rest of the tree
	for (unsigned int i = 0; i<item->countChildren(); i++ )
	{
		writeItem( (DDataItem*) m_data->Item( item->childNum(i) ), s, level + 1 );
	}
}

bool kdissstx::writeMaindoc(DDataItem *root, const QString & path)
{
	QFile textfile(path+"/main.txt");
	if (!textfile.open(IO_WriteOnly | IO_Append) )
	{
		kdWarning()<<"could not open ascii file for append :-/"<<endl;
		return false;
	}

	QTextStream u(&textfile);
	if (Settings::templatesUTF8())
	{
		u.setEncoding(QTextStream::UnicodeUTF8);
	}

	// u<<i18n("add a space after the:", "Title: ")<<root->m_summary<<"\n";
	// u<<i18n("add a space after the:", "Author: ")<<m_data->m_fname<<" "<<m_data->m_sname<<"\n";
	// u<<i18n("add a space after the:", "Email: ")<<m_data->m_email<<" "<<m_data->m_sname<<"\n";
	// u<<i18n("add a space after the:", "Company: ")<<m_data->m_company<<" "<<m_data->m_sname<<"\n";

	// now the central part, with text, pictures, etc ..
	writeItem(root, u, 0);

	// close the file 
	textfile.close();

	notifyDocGenerated( path+"/main.txt" );
	
	return true;
}

#include "kdissstx.moc"
