/*
    This file is part of libqobex.

    Copyright (c) 2003-2004 Mathias Froehlich <Mathias.Froehlich@web.de>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Steet, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include <qobex/qobexclient.h>
#include "client.h"

client::client( QObexTransport * transport, QFile * f, QFile * df, bool verbose )
  : QObexClient( transport ), mFile( f ), mDumpStream( df ), mVerbose( verbose ) {
  if ( df->isOpen() ) {
    QObject::connect(transport, SIGNAL(signalPacketRecieved(QObexObject&)),
		     SLOT(slotPacketRecieved( QObexObject&)) );
    QObject::connect(transport, SIGNAL(signalWritingPacket(const QObexObject*)),
		     SLOT(slotWritingPacket(const QObexObject*)) );
  }
}

void client::response( const QObexObject& resp ) {
  if ( mVerbose ) {
    fprintf( stderr, "Got Response Packet\n   %s\n",
	     (const char *)QFile::encodeName( resp.stringCode() ) );
    QValueList<QObexHeader> hdrs = resp.getHeaders();
    QValueList<QObexHeader>::Iterator it;
    for ( it = hdrs.begin(); it != hdrs.end(); ++it )
      fprintf( stderr, "        %s\n",
	       (const char *)QFile::encodeName( (*it).stringHeaderId() ) );
  }
}

bool client::dataReq( QByteArray& data, size_t max ) {
  data.resize( max );
  int ret = mFile->readBlock( data.data(), max );
  data.resize( ret );
  return 0 <= ret;
}

bool client::data( const QValueList<QByteArray>& data ) {
  if ( !data.isEmpty() ) {
    QValueList<QByteArray>::ConstIterator it;
    for ( it = data.begin(); it != data.end(); ++it ) {
      if ( 0 < (*it).size() ) {
	if ( 0 > mFile->writeBlock( (*it).data(), (*it).size() ) )
	  return false;
      }
    }
  }
  return true;
}

void client::authenticationRequired( const QString& realm ) {
  if ( mVerbose ) {
    fprintf( stderr, "Got Authentication Required with realm \"%s\"\n",
	     (const char *)QFile::encodeName( realm ) );
  }
}

void client::slotPacketRecieved( QObexObject& obj ) {
  mDumpStream << "Read: ("
	      << QTime::currentTime().toString( "hh:mm:ss.zzz" )
	      << ")\n";
  mDumpStream << obj.toString( 0, true );
  mDumpStream << endl;
  mDumpStream.device()->flush();
}

void client::slotWritingPacket( const QObexObject* obj ) {
  mDumpStream << "Write: ("
	      << QTime::currentTime().toString( "hh:mm:ss.zzz" )
	      << ")\n";
  mDumpStream << obj->toString( 0, true );
  mDumpStream << endl;
  mDumpStream.device()->flush();
}

#include "client.moc"
