/***************************************************************************
*   Copyright (C) 2004-2007 by Thomas Fischer                             *
*   fischer@unix-ag.uni-kl.de                                             *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/
#include <file.h>
#include <element.h>
#include <entry.h>
#include <macro.h>
#include <preamble.h>
#include <value.h>
#include <comment.h>
#include <encoderlatex.h>

#include "fileexporterbibtex.h"

namespace BibTeX
{

    FileExporterBibTeX::FileExporterBibTeX() : FileExporter(),
            m_stringOpenDelimiter( '"' ), m_stringCloseDelimiter( '"' ), m_encoding( File::encLaTeX ), cancelFlag( FALSE )
    {
        // nothing
    }


    FileExporterBibTeX::~FileExporterBibTeX()
    {
        // nothing
    }


    bool FileExporterBibTeX::save( QIODevice* iodevice, File* bibtexfile, QStringList * /*errorLog*/ )
    {
        bool result = TRUE;
        QTextStream stream( iodevice );
        if ( m_encoding == File::encUTF8 )
            stream.setEncoding( QTextStream::UnicodeUTF8 );

        // first, write preambles and strings at the beginning
        for ( File::ElementList::iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !cancelFlag; it++ )
        {
            Preamble *preamble = dynamic_cast<Preamble*>( *it );
            if ( preamble != NULL )
                result &= writePreamble( stream, preamble );
        }

        for ( File::ElementList::iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !cancelFlag; it++ )
        {
            Macro *macro = dynamic_cast<Macro*>( *it );
            if ( macro != NULL )
                result &= writeMacro( stream, macro );
        }

        // second, write cross-referencing elements
        for ( File::ElementList::iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !cancelFlag; it++ )
        {
            Entry *entry = dynamic_cast<Entry*>( *it );
            if (( entry != NULL ) && ( entry->getField( EntryField::ftCrossRef ) != NULL ) )
                result &= writeEntry( stream, entry );
        }

        // third, write remaining elements
        for ( File::ElementList::iterator it = bibtexfile->elements.begin(); it != bibtexfile->elements.end() && result && !cancelFlag; it++ )
        {
            Entry *entry = dynamic_cast<Entry*>( *it );
            if ( entry != NULL )
            {
                if ( entry->getField( EntryField::ftCrossRef ) == NULL )
                    result &= writeEntry( stream, entry );
            }
            else
            {
                Comment *comment = dynamic_cast<Comment*>( *it );
                if ( comment != NULL )
                    result &= writeComment( stream, comment );
            }
        }

        return result && !cancelFlag;
    }

    bool FileExporterBibTeX::save( QIODevice* iodevice, Element* element, QStringList * /*errorLog*/ )
    {
        bool result = FALSE;
        QTextStream stream( iodevice );
        if ( m_encoding == File::encUTF8 )
            stream.setEncoding( QTextStream::UnicodeUTF8 );

        Entry *entry = dynamic_cast<Entry*>( element );
        if ( entry != NULL )
            result |= writeEntry( stream, entry );
        else
        {
            Macro * macro = dynamic_cast<Macro*>( element );
            if ( macro != NULL )
                result |= writeMacro( stream, macro );
            else
            {
                Comment * comment = dynamic_cast<Comment*>( element );
                if ( comment != NULL )
                    result |= writeComment( stream, comment );
            }
        }

        return result && !cancelFlag;
    }

    void FileExporterBibTeX::cancel()
    {
        cancelFlag = TRUE;
    }

    bool FileExporterBibTeX::writeEntry( QTextStream&stream, Entry* entry )
    {
        stream << "@" << applyKeywordCasing( entry->entryTypeString() ) << "{ " << entry->id();

        for ( Entry::EntryFields::ConstIterator it = entry->begin(); it != entry->end(); it++ )
        {
            EntryField *field = *it;
            stream << ',' << endl << '\t' << field->fieldTypeName() << " = " << valueToString( field->value(), field->fieldType() );
        }
        stream << endl << "}" << endl << endl;
        return TRUE;
    }

    bool FileExporterBibTeX::writeMacro( QTextStream&stream, Macro *macro )
    {
        stream << "@" << applyKeywordCasing( "String" ) << "{ " << macro->key() << " = " << valueToString( macro->value() ) << " }" << endl << endl;

        return TRUE;
    }

    bool FileExporterBibTeX::writeComment( QTextStream&stream, Comment *comment )
    {
        if ( !comment->useCommand() )
        {
            QString text = comment->text() ;
            escapeLaTeXChars( text );

            switch ( m_encoding )
            {
            case File::encLaTeX:
                text = EncoderLaTeX::currentEncoderLaTeX() ->encode( text );
                break;
            default:
                stream.setEncoding( QTextStream::UnicodeUTF8 );
            }
            QStringList commentLines = QStringList::split( '\n', text );
            for ( QStringList::Iterator it = commentLines.begin(); it != commentLines.end(); it++ )
            {
                stream << *it << endl;
            }
            stream << endl;
        }
        else
        {
            QString text = comment->text() ;
            escapeLaTeXChars( text );

            switch ( m_encoding )
            {
            case File::encLaTeX:
                text = EncoderLaTeX::currentEncoderLaTeX() ->encode( text );
                break;
            default:
                stream.setEncoding( QTextStream::UnicodeUTF8 );
            }
            stream << "@" << applyKeywordCasing( "Comment" ) << "{" << text << "}" << endl << endl;
        }
        return TRUE;
    }

    bool FileExporterBibTeX::writePreamble( QTextStream &stream, Preamble* preamble )
    {
        stream << "@" << applyKeywordCasing( "Preamble" ) << "{" << valueToString( preamble->value() )  << "}" << endl << endl;

        return TRUE;
    }

    void FileExporterBibTeX::setStringDelimiter( const QChar& stringOpenDelimiter, const QChar& stringCloseDelimiter )
    {
        m_stringOpenDelimiter = stringOpenDelimiter;
        m_stringCloseDelimiter = stringCloseDelimiter;
    }

    void FileExporterBibTeX::setKeywordCasing( const KeywordCasing keywordCasing )
    {
        m_keywordCasing = keywordCasing;
    }

    void FileExporterBibTeX::setEncoding( File::Encoding encoding )
    {
        m_encoding = encoding;
    }

    QString FileExporterBibTeX::valueToString( const Value *value, const EntryField::FieldType fieldType )
    {
        QString result;
        bool isFirst = TRUE;
        EncoderLaTeX *encoder = EncoderLaTeX::currentEncoderLaTeX();

        for ( QValueList<ValueItem*>::ConstIterator it = value->items.begin(); it != value->items.end(); ++it )
        {
            if ( !isFirst )
                result.append( " # " );
            else
                isFirst = FALSE;

            MacroKey *macroKey = dynamic_cast<MacroKey*>( *it );
            if ( macroKey != NULL )
                result.append( macroKey->text() );
            else
            {
                QString text;
                BibTeX::PersonContainer *personContainer = dynamic_cast<BibTeX::PersonContainer*>( *it );
                BibTeX::PlainText *plainText = dynamic_cast<BibTeX::PlainText*>( *it );
                BibTeX::KeywordContainer *keywordContainer = dynamic_cast<BibTeX::KeywordContainer*>( *it );

                if ( plainText != NULL )
                    text = plainText->text();
                else if ( keywordContainer != NULL )
                {
                    bool first = TRUE;
                    for ( QValueList<Keyword*>::Iterator it = keywordContainer->keywords.begin(); it != keywordContainer->keywords.end(); ++it )
                    {
                        if ( !first )
                            text.append( ", " );
                        else
                            first = FALSE;
                        text.append(( *it )->text() );
                    }
                }
                else if ( personContainer != NULL )
                {
                    bool first = TRUE;
                    for ( QValueList<Person*>::Iterator it = personContainer->persons.begin(); it != personContainer->persons.end(); ++it )
                    {
                        if ( !first )
                            text.append( " and " );
                        else
                            first = FALSE;

                        QString v = ( *it )->firstName();
                        if ( !v.isEmpty() )
                        {
                            bool requiresQuoting = requiresPersonQuoting( v, FALSE );//v.contains( " and " );
                            if ( requiresQuoting ) text.append( "{" );
                            text.append( v );
                            if ( requiresQuoting ) text.append( "}" );
                            text.append( " " );
                        }

                        v = ( *it )->lastName();
                        if ( !v.isEmpty() )
                        {
                            bool requiresQuoting = requiresPersonQuoting( v, TRUE );//v.contains( " " );
                            if ( requiresQuoting ) text.append( "{" );
                            text.append( v );
                            if ( requiresQuoting ) text.append( "}" );
                        }
                    }
                }

                escapeLaTeXChars( text );

                switch ( m_encoding )
                {
                case File::encLaTeX:
                    text = encoder->encodeSpecialized( text, fieldType );
                    break;
                default:
                    {
                        // nothing
                    }
                }

                // if the text to save contains a quote char ("),
                // force string delimiters to be curly brackets,
                // as quote chars as string delimiters would result
                // in parser failures
                QChar stringOpenDelimiter = m_stringOpenDelimiter;
                QChar stringCloseDelimiter = m_stringCloseDelimiter;
                if ( result.contains( '"' ) && ( m_stringOpenDelimiter == '"' || m_stringCloseDelimiter == '"' ) )
                {
                    stringOpenDelimiter = '{';
                    stringCloseDelimiter = '}';
                }

                result.append( stringOpenDelimiter ).append( text ).append( stringCloseDelimiter );
            }
        }

        return result;
    }

    void FileExporterBibTeX::escapeLaTeXChars( QString &text )
    {
        text.replace( "&", "\\&" );
    }

    QString FileExporterBibTeX::applyKeywordCasing( const QString &keyword )
    {
        switch ( m_keywordCasing )
        {
        case kcLowerCase: return keyword.lower();
        case kcInitialCapital: return keyword.at( 0 ) + keyword.lower().mid( 1 );
        case kcCapital: return keyword.upper();
        default: return keyword;
        }
    }

    bool FileExporterBibTeX::requiresPersonQuoting( const QString &text, bool isLastName )
    {
        if ( isLastName && !text.contains( " " ) )
            return FALSE;
        else if ( !isLastName && !text.contains( " and " ) )
            return FALSE;
        else if ( text[0] != '{' || text[text.length()-1] != '}' )
            return TRUE;

        int bracketCounter = 0;
        for ( int i = text.length() - 1;i >= 0;--i )
        {
            if ( text[i] == '{' )
                ++bracketCounter;
            else if ( text[i] == '}' )
                --bracketCounter;
            if ( bracketCounter == 0 && i > 0 )
                return TRUE;
        }
        return FALSE;
    }

}
