// K-3D
// Copyright (c) 1995-2006, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "fstream.h"
#include "i18n.h"
#include "options.h"
#include "result.h"
#include "shader_collection.h"
#include "string_cast.h"
#include "string_modifiers.h"
#include "system.h"

#include <boost/filesystem/path.hpp>
#include <boost/filesystem/operations.hpp>
#include <fstream>

namespace k3d
{

namespace detail
{

/// Inserts shaders into a collection
class shader_functor
{
public:
	shader_functor(k3d::shader_collection::message_signal_t& MessageSignal, k3d::sl::shaders_t& Shaders) :
		m_message_signal(MessageSignal),
		m_shaders(Shaders)
	{
	}

	bool operator()(const boost::filesystem::path& SourcePath)
	{
		// Skip files that aren't shader source code ...
		if(k3d::file_extension(SourcePath) != "sl")
			return true;

		const boost::filesystem::path metafile_path(SourcePath.native_file_string() + ".slmeta", boost::filesystem::native);
		if(!boost::filesystem::exists(metafile_path))
			{
				k3d::log() << warning << "Shader " << SourcePath.native_file_string() << " has no metafile and will be ignored" << std::endl;
				return true;
			}

		m_message_signal.emit(k3d::string_cast(boost::format(_("Loading shader %1%")) % SourcePath.leaf()));
		k3d::filesystem::ifstream metafile(metafile_path);
		const k3d::sl::shaders_t shaders(k3d::sl::parse_metafile(metafile, SourcePath.native_file_string(), metafile_path.native_file_string()));
		m_shaders.insert(m_shaders.end(), shaders.begin(), shaders.end());

		return true;
	}

private:
	k3d::shader_collection::message_signal_t& m_message_signal;
	k3d::sl::shaders_t& m_shaders;
};

/// Loads shaders from a series of delimited paths
void load_shaders(k3d::shader_collection::message_signal_t& MessageSignal, k3d::sl::shaders_t& Shaders, const std::string& Paths)
{
	const k3d::system::paths_t paths = k3d::system::decompose_path_list(Paths);
	for(k3d::system::paths_t::const_iterator path = paths.begin(); path != paths.end(); ++path)
	{
		MessageSignal.emit(k3d::string_cast(boost::format(_("Loading shaders from %1%")) % path->native_file_string()));

		shader_functor functor(MessageSignal, Shaders);

		for(boost::filesystem::directory_iterator file(*path); file != boost::filesystem::directory_iterator(); ++file)
		{
			if(boost::filesystem::is_directory(*file))
				continue;

			functor(*file);
		}
	}
}

} // namespace detail

/////////////////////////////////////////////////////////////////////////////
// shader_collection

shader_collection::shader_collection()
{
}

shader_collection::message_signal_t& shader_collection::message_signal()
{
	return m_message_signal;
}

void shader_collection::load_shaders(const std::string& SourcePaths)
{
	detail::load_shaders(m_message_signal, m_shaders, SourcePaths);
}

const sl::shaders_t& shader_collection::shaders()
{
	return m_shaders;
}

} // namespace k3d


