// Copyright (c) 2003, Google Inc.
// All rights reserved.
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
// 
//     * Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//     * Redistributions in binary form must reproduce the above
// copyright notice, this list of conditions and the following disclaimer
// in the documentation and/or other materials provided with the
// distribution.
//     * Neither the name of Google Inc. nor the names of its
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

// ---
// Author: Sanjay Ghemawat
//
// Test speed of handling fragmented heap

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <sys/time.h>           // for struct timeval
#include <sys/resource.h>       // for getrusage
#include <vector>
#include "base/logging.h"
#include <google/malloc_extension.h>

using std::vector;

int main(int argc, char** argv) {
  static const int kAllocSize = 36<<10; // Bigger than tcmalloc page size
  static const int kTotalAlloc = 400 << 20; // Allocate 400MB in total
  static const int kAllocIterations = kTotalAlloc / kAllocSize;

  // Allocate lots of objects
  vector<char*> saved(kAllocIterations);
  for (int i = 0; i < kAllocIterations; i++) {
    saved[i] = new char[kAllocSize];
  }

  // Free alternating ones to fragment heap
  size_t free_bytes = 0;
  for (int i = 0; i < saved.size(); i += 2) {
    delete[] saved[i];
    free_bytes += kAllocSize;
  }

  // Check that slack is within 10% of expected
  size_t slack;
  MallocExtension::instance()->GetNumericProperty("tcmalloc.slack_bytes",
                                                  &slack);
  CHECK_GT(double(slack), 0.9*free_bytes);
  CHECK_LT(double(slack), 1.1*free_bytes);

  // Dump malloc stats
  static const int kBufSize = 1<<20;
  char* buffer = new char[kBufSize];
  MallocExtension::instance()->GetStats(buffer, kBufSize);
  VLOG(1, "%s", buffer);
  delete[] buffer;

  // Now do timing tests
  for (int i = 0; i < 5; i++) {
    static const int kIterations = 100000;
    struct rusage r;
    getrusage(RUSAGE_SELF, &r);    // figure out user-time spent on this
    struct timeval tv_start = r.ru_utime;

    for (int i = 0; i < kIterations; i++) {
      size_t s;
      MallocExtension::instance()->GetNumericProperty("tcmalloc.slack_bytes",
                                                      &s);
    }

    getrusage(RUSAGE_SELF, &r);
    struct timeval tv_end = r.ru_utime;
    int64 sumsec = static_cast<int64>(tv_end.tv_sec) - tv_start.tv_sec;
    int64 sumusec = static_cast<int64>(tv_end.tv_usec) - tv_start.tv_usec;
    fprintf(stderr, "getproperty: %6.1f ns/call\n",
            (sumsec * 1e9 + sumusec * 1e3) / kIterations);
  }

  printf("PASS\n");
  return 0;
}
