/***************************************************************************
 *   Copyright (C) 2003 by Gav Wood                                        *
 *   gav@cs.york.ac.uk                                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU Library General Public License as       *
 *   published by the Free Software Foundation; either version 2 of the    *
 *   License, or (at your option) any later version.                       *
 ***************************************************************************/

#define __GEDDEI_BUILD

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "qfactoryexporter.h"

#include "signaltype.h"
#include "bufferdata.h"
#include "subprocessor.h"
#include "buffer.h"
using namespace Geddei;

#include "spectrum.h"
#include "wave.h"
using namespace SignalTypes;

#ifdef HAVE_FFTW

#include <fftw3.h>

class FFT: public SubProcessor
{
	int theSize, theStep;

	fftwf_plan thePlan;
	float *theIn, *theOut;

	virtual void processChunk(const BufferDatas &in, BufferDatas &out) const;
	virtual PropertiesInfo specifyProperties() const;
	virtual void initFromProperties(const Properties &p);
	virtual const bool verifyAndSpecifyTypes(const SignalTypeRefs &inTypes, SignalTypeRefs &outTypes);
public:
	FFT() : SubProcessor("FFT") {}
	~FFT();
};

void FFT::initFromProperties(const Properties &p)
{
	theStep = p["Step"].toInt();
	theSize = p["Size"].toInt();
	setupIO(1, 1, theSize, theStep, 1);
	theIn = (float *)fftwf_malloc(sizeof(float) * theSize);
	theOut = (float *)fftwf_malloc(sizeof(float) * theSize);
	thePlan = fftwf_plan_r2r_1d(theSize, theIn, theOut, FFTW_R2HC, p["Optimise"].toBool() ? FFTW_MEASURE : FFTW_ESTIMATE);
}

FFT::~FFT()
{
	fftwf_destroy_plan(thePlan);
	fftwf_free(theIn);
	fftwf_free(theOut);
}

void FFT::processChunk(const BufferDatas &in, BufferDatas &out) const
{
	in[0].copyTo(theIn);
	fftwf_execute(thePlan);
	for(int i = 0; i < theSize / 2; i++)
		theOut[i] /= theSize;
	out[0].copyFrom(theOut);
}

#else

#include <math.h>
using namespace std;

#define PI 3.1415926535898

class FFT : public SubProcessor
{
	uint theLogSize, theStep, theSize, theSizeMask;
	float *costable, *sintable;
	uint *bitReverse;
	float *real, *imag;

	virtual void processChunk(const BufferDatas &in, BufferDatas &out) const;
	virtual const bool verifyAndSpecifyTypes(const SignalTypeRefs &inTypes, SignalTypeRefs &outTypes);
	virtual PropertiesInfo specifyProperties() const;
	virtual void initFromProperties(const Properties &properties);

public:
	FFT();
	virtual ~FFT();
};

const int reverseBits(uint initial, const uint logSize)
{
	uint reversed = 0;
	for(uint loop = 0; loop < logSize; loop++)
	{
		reversed <<= 1;
		reversed += (initial & 1);
		initial >>= 1;
	}
	return reversed;
}

FFT::FFT() : SubProcessor("FFT")
{
	sintable = 0;
	costable = 0;
	bitReverse = 0;
}

FFT::~FFT()
{
	delete [] sintable;
	delete [] costable;
	delete [] bitReverse;
	delete [] real;
	delete [] imag;
}

void FFT::processChunk(const BufferDatas &in, BufferDatas &out) const
{
	for(uint i = 0; i < theSize; i++)
	{	real[i] = in[0][bitReverse[i] & theSizeMask];
		imag[i] = 0;
	}
	uint exchanges = 1, factfact = theSize / 2;
	float fact_real, fact_imag, tmp_real, tmp_imag;

	for(uint i = theLogSize; i != 0; i--)
	{
		for(uint j = 0; j != exchanges; j++)
		{
			fact_real = costable[j * factfact];
			fact_imag = sintable[j * factfact];

			for(uint k = j; k < theSize; k += exchanges << 1)
			{
				int k1 = k + exchanges;
				tmp_real = fact_real * real[k1] - fact_imag * imag[k1];
				tmp_imag = fact_real * imag[k1] + fact_imag * real[k1];
				real[k1] = real[k] - tmp_real;
				imag[k1] = imag[k] - tmp_imag;
				real[k]  += tmp_real;
				imag[k]  += tmp_imag;
			}
		}
		exchanges <<= 1;
		factfact >>= 1;
	}

	float theMagnitude = (theSize / 2);
	theMagnitude *= theMagnitude;

	for(uint i = 0; i < theSize / 2; i++)
		out[0][i] = ((real[i] * real[i]) + (imag[i] * imag[i])) / theMagnitude;
	out[0][0] /= 4;
}

void FFT::initFromProperties(const Properties &properties)
{
	theLogSize = int(floor(log(double(properties["Size"].toInt())) / log(2.0)));
	theStep = properties["Step"].toInt();
	theSize = 1 << theLogSize;
	if(theSize != (uint)properties["Size"].toInt())
		qDebug("*** WARNING: Using simple FFT as FFTW not found. Can only use FFT sizes that are\n"
		       "             powers of 2. Size truncated from %d (=2^%f) to %d (=2^%d). Fix this\n"
		       "             by installing libfftw and recompiling the FFT subprocessor.", properties["Size"].toInt(), log(double(properties["Size"].toInt())) / log(2.0), theSize, theLogSize);
	theSizeMask = theSize - 1;
	setupIO(1, 1, theSize, theStep, 1);

	// set up lookups
	delete [] sintable;
	delete [] costable;
	delete [] bitReverse;
	sintable = new float[theSize / 2];
	costable = new float[theSize / 2];
	for(uint i = 0; i < theSize / 2; i++)
	{	costable[i] = cos(2 * PI * i / theSize);
		sintable[i] = sin(2 * PI * i / theSize);
	}
	bitReverse = new uint[theSize];
	for(uint i = 0; i < theSize; i++)
		bitReverse[i] = reverseBits(i, theLogSize);

	real = new float[theSize];
	imag = new float[theSize];
}

#endif

const bool FFT::verifyAndSpecifyTypes(const SignalTypeRefs &inTypes, SignalTypeRefs &outTypes)
{
	if(!inTypes[0].isA<Wave>()) return false;
	// TODO: should output a "stepped spectrum".
	outTypes[0] = Spectrum(theSize / 2, inTypes[0].frequency() / float(theStep), inTypes[0].frequency() / float(theSize));
	return true;
}

PropertiesInfo FFT::specifyProperties() const
{
	return PropertiesInfo("Size", 2048, "The size of the block (in samples) from which to conduct a short time Fourier transform.")
	                     ("Step", 1024, "The number of samples between consequent sampling blocks.")
	                     ("Optimise", true, "True if time is taken to optimise the calculation.");
}

EXPORT_CLASS(FFT, 0,9,0, SubProcessor);
