// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 1999 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

/* ʸɽβ JISX4081-1986 + */

#include "honmon.h"

#include <cstdio>

ATTR::ATTR() : bold(false),italic(false),hidden(false),
	       hankaku(false),superscript(false),subscript(false),
	       widechar(false),underline(0),color(0),
	       notText(false),
	       //	       voice(false),pict(false),movie(false),
	       id(0) {};

ATTR::ATTR(const ATTR& a){
  bold=a.bold;
  italic=a.italic;
  hidden=a.hidden;
  hankaku=a.hankaku;
  superscript=a.superscript;
  subscript=a.subscript;
  notText=a.notText;
  widechar=a.widechar;
  underline=a.underline;
  color=a.color;
  
  //  voice=a.voice;
  //  pict=a.pict;
  //  movie=a.movie;
  id=a.id;
}

DICCHAR::DICCHAR(int ch,ATTR a,TAG t){
  code=ch;
  attr=a;
  tag=t;
};

DICCHAR::~DICCHAR() {
};

static void SetTag(DICSTRING& txt,const TAG& t){
  for(DICSTRING::reverse_iterator r=txt.rbegin();
      r!=txt.rend() && (*r).Tag()==TAG(-1,-1);++r){
    (*r).Tag(t);
  }
}

word DICCHAR::Hankaku(void) const{
  static byte table21[]=" @@,.\xa5:;?!@@@@@^~_@@@@@@@@@---/\\~@|@@`'\"\"()"
    "[][]{}<>@@@@@@@@+-@@@=@<>@@@@@@@'\"@\\$@@%#&*@@@@@@@";
  static byte table22[]="@@@@@@@@@@@@@@";

  int hi=(code >> 8) & 0xff;
  int lo=code & 0xff;
  if (hi==0x23) {
    return lo;
  } else if (hi==0x21) {
    if (lo<0x21 || lo>0x7e) return 0;
    if (table21[lo-0x21]=='@') return code;
    return (table21[lo-0x21]);
  } else if (hi==0x22) {
    if (lo<0x21 || lo>0x2e) return 0;
    if (table22[lo-0x21]=='@') return code;
    return (table22[lo-0x21]);
  } 
  return code;
};

std::string DICCHAR::Euc() const {
  static byte table21[]=" \xa4\xa1,.\xa5:;?!\xde\xdfZZZ^~_ZZZZZZZZZ"
    "\xb0--/\\ZZ|ZZ`'\"\"()"
    "ZZ[]{}<>ZZ\xa2\xa3ZZZZ+-ZZZ=Z<>ZZZZZZZ'\"Z\\$ZZ%#&*@ZZZZZZZ";
  static byte table25[]="\xa7\xb1\xa8\xb2\xa9\xb3\xaa\xb4\xab\xb5" // -
    "\xb6\x36\xb7\x37\xb8\x38\xb9\x39\xba\x3a" // -
    "\xbb\x3b\xbc\x3c\xbd\x3d\xbe\x3e\xbf\x3f" // -
    "\xc0\x40\xc1\x41\xaf\xc2\x42\xc3\x43\xc4\x44" // -
    "\xc5\xc6\xc7\xc8\xc9" // -
    "\xca\x4a\x0a\xcb\x4b\x0b\xcc\x4c\x0c\xcd\x4d\x0d\xce\x4e\x0e" // -
    "\xcf\xd0\xd1\xd2\xd3" "\xac\xd4\xad\xd5\xae\xd6" // -
    "\xd7\xd8\xd9\xda\xdb" " \xdc  \xa6\xdd   ";
  std::string result="";
  int high=255 & (code>>8);
  int low=255 & code;
  if (attr.hankaku) {
    if (high==0x23) {
      result+=char(low);
    } else if (high==0x21 && table21[low-0x21]!='Z') {
      if (table21[low-0x21] & 0x80) {
	result+='\x8e';
	result+=char(table21[low-0x21]);
      } else {
	result+=char(table21[low-0x21]);
      }
    } else if (high==0x25) {
      int x=table25[low-0x21];
      if (x==' ') {
	result+=char(high | 0x80);
	result+=char(low | 0x80);
      } else if (x & 0x80) {
	result+=char(0x8e);
	result+=char(x);
      } else if (x & 0x40) {
	result+=char(0x8e);
	result+=char(x+0x40);
	result+=char(0x8e);
	result+=char(0xde);
      } else {
	result+=char(0x8e);
	result+=char(x+0x80);
	result+=char(0x8e);
	result+=char(0xdf);
      }
    } else {
      result+=char(high | 0x80);
      result+=char(low | 0x80);
    }
  } else {
    result+=char(high | 0x80);
    result+=char(low | 0x80);
  }
  return result;
}

DICLINE::DICLINE(DICTIONARY *dct,const TAG& hed) {
  currentMargin=0;
  marginSplit=0;
  currentAttr=ATTR();
  currentTag=TAG(0,0);
  isNotEOF=true;
  dict=dct;
  if (dict==NULL) {
    file=NULL;
  } else {
    file=dct->BlockFile();
  }
  if (file!=NULL && hed!=TAG(0,0)) file->Seek(hed);
}


static DICSTRING MakeString(char *str,ATTR a=ATTR(),
					TAG t=TAG(0,0),TAG t2=TAG(0,0)){
  DICSTRING s;
  byte* bp=(byte*)str;
  while(*bp!=0){
    DICCHAR ch;
    if (*bp & 0x80){
      ch.Code(0x7f7f & (((bp[0])<<8)+bp[1]));
      a.hankaku=false;
      bp+=2;
    } else {
      ch.Code(('#'<<8)+bp[0]);
      a.hankaku=true;
      bp++;
    }
    ch.Attr(a);
    ch.Tag(t);
    s+=ch;
  }
  return s;
}

static DICSTRING MakeHexString(int n,ATTR a=ATTR(),TAG t=TAG(0,0)){
  const char hexString[]="0123456789ABCDEF";
  DICSTRING s;
  DICCHAR ch;
  a.hankaku=true;
  ch.Attr(a);
  ch.Tag(t);
  ch.Code(('#'<<8)+hexString[15 & (n>>4)]);
  s+=ch;
  ch.Code(('#'<<8)+hexString[n & 15]);
  s+=ch;
  return s;
}

bool DICLINE::DecodeDecorate0X(int ch2,DICCHAR& chr,
			       DICSTRING& txt,bool& lflag){
  switch(ch2){
  case 0x02:  // ɽϻ
    //    txt+=MakeString("<ɽ>",currentAttr,currentTag);
    txt+=DICCHAR(0x1f02);
    marginSplit=-1;  // ignore split at margin=1
    return true;
  case 0x03:  // ɽλ
    //  txt+=MakeString("<ɽλ>",currentAttr,currentTag);
    //    txt+=MakeString("[Next]",currentAttr,file->Tell());
    txt+=DICCHAR(0x1f03);
    lflag=false;
    isNotEOF=false;
    return true;
  case 0x04:  // Ⱦѳϻ
    currentAttr.hankaku=true;
    return true;
  case 0x05:  // Ⱦѽλ
    currentAttr.hankaku=false;
    return true;
  case 0x06:  // źϻ
    currentAttr.subscript=true;
    return true;
  case 0x07:  // źλ
    currentAttr.subscript=false;
    return true;
  case 0x08:  // ĥΰ 
    txt+=MakeString("[ĥΰ 08]",currentAttr,currentTag);
    return true;
  case 0x09:  // 
    {
      int m=file->GetWord();
      if (m==1){
#if 0
	if (marginSplit==1 && currentMargin>=1) {
	  TAG t=file->Tell();
	  int b=t.Block();
	  int o=t.Offset()-4;
	  if (o<0) {
	    b--;
	    o+=cBlockSize;
	  }
	  txt+=MakeString("[Next]",currentAttr,TAG(b,o));
	  isNotEOF=false;
	  lflag=false;
	}
#endif
	if (marginSplit==0) {
	  marginSplit=1;
	} else if (marginSplit==1) {
	  marginSplit=2;
	}
      }
      // *************** TEST ***************
      currentMargin=m;
      //      Debug::DebugOut(Debug::HONMON_CONTROL,"margin=%d %d\n",
      //		      currentMargin,marginSplit);
    }
    return true;
  case 0x0a:  // 
    #if 1
      lflag=false;
    #else
    if (currentTag==TAG(0,0)) {
      lflag=false;
    } else {
      txt+=DICCHAR(0x20,currentAttr,currentTag);
      // ignore Line Feed
    }
    #endif
    // ************ TEST ***********************
    // if (marginSplit==0) marginSplit=-1;  // ignore split at margin=1
    return true;
  case 0x0b:  // ɽɽϻ
    txt+=MakeString("<ɽɽ>",currentAttr,currentTag);
    return true;
  case 0x0c:  // ɽɽλ
    txt+=MakeString("<ɽɽλ>",currentAttr,currentTag);
    return true;
  case 0x0d:  // ĥΰ 
    txt+=MakeString("[ĥΰ 0d]",currentAttr,currentTag);
    return true;
  case 0x0e:  // źϻ
    currentAttr.superscript=true;
    return true;
  case 0x0f:  // źλ
    currentAttr.superscript=false;
    return true;
  }
  return false;
}

bool DICLINE::DecodeDecorate1X(int ch2,DICCHAR& chr,
			       DICSTRING& txt,bool& lflag){
  switch(ch2){
  case 0x10:  // ʬػ߳ϻ
    txt+=DICCHAR(0x1f10,currentAttr,currentTag);
    return true;
  case 0x11:  // ʬػ߽λ
    txt+=DICCHAR(0x1f11,currentAttr,currentTag);
    return true;
  case 0x12:  // ϻ
    currentAttr.underline=1;
    return true;
  case 0x13:  // λ
    currentAttr.underline=0;
    return true;
  case 0x14:  // ĥΰ Color Sample Start?
    {
      int colorCode=0;
      for(;;){
	int num=file->GetWord();
	if ((num & 0xff00)!=0x1e00) break; // 0x1f15..
	colorCode=colorCode*100+((num & 0xf0)>>4)*10+(num & 0xf);
      }
      char buf[100];
      snprintf(buf,100,"[ Code=%d]",colorCode);
      currentAttr.id=ch2;
      currentAttr.color=true;
      currentAttr.notText=true;
      currentTag=TAG(dict->IndexStart(0xe0).Block()+colorCode-1,0);
      txt+=MakeString(buf,currentAttr,currentTag);
      currentAttr.color=false;
      currentAttr.notText=false;
      currentAttr.id=0;
      currentTag=TAG(0,0);
    }
    return true;
  case 0x15:  // ĥΰ Color Sample End?  * unreachable
    txt+=MakeString("[ λ]",currentAttr,currentTag);
    break;
  case 0x16:  // ĥΰ Ĵϻ?
    currentAttr.bold=true;
    return true;
  case 0x17:  // ĥΰ Ĵλ
    currentAttr.bold=false;
    return true;
  case 0x18:  // ĥΰ ȿž
    currentAttr.color|=0x10;
    return true;
  case 0x19:  // ĥΰ ȿžλ
    currentAttr.color&=(~0x10);
    return true;
  case 0x1a:  // ĥΰ TAB?
    if (dict->isEB) {
      // Ignore
    } else {
      currentAttr.id=file->GetBCD(4);
      txt+=DICCHAR(0x1f1a,currentAttr,currentTag);
      currentAttr.id=0;
    }
    return true;
  case 0x1b:  // ĥΰ ???
    if (dict->isEB) {
      // Ignore
      return true;
    } 
  case 0x1c:  // ĥΰ ???  ޤ centering ?
    // Ignore
    file->GetWord();
    return true;
  case 0x1d:  // ĥΰ ???
  case 0x1e:  // ĥΰ ???
  case 0x1f:  // ĥΰ ???
    {	
      txt+=MakeString("[ĥΰ ",currentAttr,currentTag);
      txt+=MakeHexString(ch2,currentAttr,currentTag);
      txt+=MakeString(":",currentAttr,currentTag);
      // skip 2bytes
      txt+=MakeHexString(file->GetByte(),currentAttr,currentTag);
      txt+=MakeHexString(file->GetByte(),currentAttr,currentTag);
      txt+=MakeString("]",currentAttr,currentTag);
    }
    return true;
  }
  return false;
}

bool DICLINE::DecodeDecorate2X(int ch2,DICCHAR& chr,
			       DICSTRING& txt,bool& lflag){
  switch(ch2){
  case 0x20:
    txt+=MakeString("[Sound]",currentAttr,currentTag);
    return true;
  case 0x21:  // Ƭ§
  case 0x22:  // §
    //    file->GetWord();
    return true;
  case 0x23:  // ޤǷ֤...
    {
      int ch1=file->GetByte();
      int ch2=file->GetByte();
      DICCHAR chr=DICCHAR((ch1<<8)+ch2);
      chr.Attr(currentAttr);
      chr.Tag(currentTag);
      txt+=chr;  // Ȥꤢ ʸ餤Ǹ洪....
      txt+=chr;
      txt+=chr;
      txt+=chr;
      txt+=chr;
    }
    return true;
  case 0x24:  // ̤ ???
    {
      int typ=file->GetByte();  // 01: nʸ, 02: ޤ nʸ
      int len=file->GetByte();
      int ch1=file->GetByte();
      int ch2=file->GetByte();
      DICCHAR chr=DICCHAR((ch1<<8)+ch2);
      chr.Attr(currentAttr);
      chr.Tag(currentTag);
      if (typ==1){
	for(int i=0;i<len;i++){
	  txt+=chr;
	}
      } else {
	if (len>35) len=35;  // 
	for(int i=0;i<35-len;i++){
	  txt+=chr;
	}
      }
    }
    return true;
  case 0x25:  // S-EBXA ܳѳ
    currentAttr.widechar=true;
    return true;
  case 0x26:  // S-EBXA ܳѽλ
    currentAttr.widechar=false;
    return true;
  case 0x27:  // (S-EBXA)
    currentAttr.underline=2;
    return true;
  case 0x28:  // (S-EBXA)λ
    currentAttr.underline=0;
    return true;
  case 0x29:  // (S-EBXA)
    currentAttr.underline=3;
    return true;
  case 0x2a:  // (S-EBXA)λ
    currentAttr.underline=0;
    return true;
  case 0x2b:  // (S-EBXA)˵
    currentAttr.underline=4;
    return true;
  case 0x2c:  // (S-EBXA)˵λ
    currentAttr.underline=0;
    return true;
  case 0x2d:  // S-EBXA ϥե͡İ
    return true;
  case 0x2e:  // S-EBXA ISO-8859-1
    txt+=DICCHAR(file->GetWord(),currentAttr,currentTag);
    return true;
    /*
  case 0x2f:  // ̤ ???
  */
  default:
    txt+=MakeString("[Undef ",currentAttr,currentTag);
    txt+=MakeHexString(ch2,currentAttr,currentTag);
    txt+=MakeString("]",currentAttr,currentTag);

  };
  return true;
}

int menuCount=0;
bool pageFlag=false;

bool DICLINE::DecodeDecorate35X(int ch2,DICCHAR& chr,
				DICSTRING& txt,bool& lflag){
  switch(ch2){
  case 0x31:  // ̤ Figure Start?
    {
      currentAttr.id=0x31;
      currentAttr.notText=true;
      //      currentAttr.pict=true;
      currentTag=file->Tell();
      txt+=DICCHAR(0xffff,currentAttr,currentTag);
      file->Read(NULL,6);
      {
	TAG t=file->Tell();
	if (file->GetByte()>=0x1F){
	  file->Seek(t);
	} else {
	  file->GetByte();
	}
      }
      //      txt+=MakeString(" ",currentAttr,currentTag);
      return true;
    }
  case 0x51:  // Figure end
    //    txt+=MakeString(" ",currentAttr,currentTag);
    currentAttr.id=0;
    currentAttr.notText=false;
    //    currentAttr.pict=false;
    file->Read(NULL,6);  // Dummy Read
    currentTag=TAG(0,0);
    pageFlag=true;
    return true;
  case 0x32:  // Clickable Map? Dokuwa...
    {
      TAG t=file->Tell();
      currentTag=TAG(-1,-1);
      if (file->GetWord()<0x1000){
	file->Seek(t);
	currentAttr.id=file->GetBCD(8);
	txt+=DICCHAR(0x1f32,currentAttr,currentTag);
	currentAttr.id=0;
      } else {
	file->Seek(t);
      }
    }
    return true;
  case 0x52:
    {
      TAG t=file->GetBCDTag();
      SetTag(txt,t);
      currentTag=TAG(0,0);
      currentAttr.id=0;
    }
    return true;
  case 0x33:  // Sound Start EB?
    currentTag=TAG(-1,-1);
    currentAttr.id=file->GetBCD(8);
    currentAttr.notText=true;
    //    currentAttr.voice=true;
    //    txt+=DICCHAR(0x1f33,currentAttr,currentTag);
    return true;
  case 0x53:  // Sound end EB?
    currentAttr.id=0;
    currentAttr.notText=false;
    //    currentAttr.voice=false;
    SetTag(txt,file->Tell());
    file->Read(NULL,8);  // Dummy Read
    currentTag=TAG(0,0);
    return true;

  case 0x35:
    txt+=MakeString("[1f35]",currentAttr,currentTag);
    return true;

  case 0x55:
    txt+=MakeString("[1f55]",currentAttr,currentTag);
    return true;

  case 0x36:
    {
      int x=file->GetBCD(4);
      int y=file->GetBCD(4);
      int id=file->GetWord();
      if (id==0xff) {
	id+=0x100*menuCount;
	++menuCount;
      }
      if (isIndexKey(id,dict->isEB)) {
	currentAttr.id=id;
	txt+=DICCHAR(0x1f36,currentAttr,TAG(x,y));
	currentAttr.id=0;
      }
    }
    return true;
  case 0x56:
    return true;

  case 0x37:  // Stream Tag Jump
    currentAttr.id=file->GetWord(); // ATTR?
    txt+=DICCHAR(0x1f37,currentAttr,file->GetBCDTag());
    currentAttr.id=0;
    file->GetWord(); // ATTR?
    return true;
  case 0x57:
    return true;

  case 0x38: // Stream Nazo
    {
      txt+=MakeString("[1f38:",currentAttr,currentTag);
      for(int i=0;i<10;i++){
	txt+=MakeHexString(file->GetByte());
      }
      txt+=MakeString("]",currentAttr,currentTag);
    }
    return true;
  case 0x58:
    return true;

  case 0x39:  // Movie Start
    {
      currentAttr.id=0x39;
      file->GetWord(); // ATTR?
      file->GetWord(); // ?
      file->Read(NULL,16); // ? (zero)
      currentTag=file->Tell();
      file->Read(NULL,16); // File Name
      file->Read(NULL,8);  // ? (zero)
      currentAttr.notText=true;
      //      currentAttr.movie=true;
      //txt+=MakeString("[MOVIE=",currentAttr,currentTag);
      txt+=MakeString("MOVIE:",currentAttr,currentTag);
    }
    return true;
  case 0x59:  // Movie end
    {
      //      txt+=MakeString("]",currentAttr,currentTag);
      currentAttr.notText=false;
      //      currentAttr.movie=false;
      currentTag=TAG(0,0);
      currentAttr.id=0;
    }
    return true;
  case 0x3b:
  case 0x5b:
    return true;  // Ignore
  case 0x3c:  // Inline Figure
    {
      file->Read(NULL,12);
      currentAttr.id=0x3c;
      txt+=DICCHAR(0x1f3c,currentAttr,file->GetBCDTag());
      currentAttr.id=0;
    }
    return true;
  case 0x5c:
    return true;
  case 0x3f:
    {
      TAG t=file->Tell();
      currentTag=TAG(-1,-1);
      if (file->GetWord()<0x1000){
	file->Seek(t);
	currentAttr.id=file->GetBCD(8);
	txt+=DICCHAR(0x1f3f,currentAttr,currentTag);
	currentAttr.id=0;
      } else {
	file->Seek(t);
      }
    }
    return true;
  case 0x5f:
    currentTag=TAG(0,0);
    SetTag(txt,file->GetBCDTag());
    return true;
    
  case 0x3a:
  case 0x5a:
    return true;

  case 0x34:
  case 0x3d:
  case 0x3e:
    {	
      txt+=MakeString("[1F",currentAttr,currentTag);
      txt+=MakeHexString(ch2,currentAttr,currentTag);
      txt+=MakeString("/",currentAttr,currentTag);
      txt+=MakeHexString(ch2+0x20,currentAttr,currentTag);
      txt+=MakeString("]",currentAttr,currentTag);
    }
    do {
      while(file->GetByte()!=0x1f);
    } while(file->GetByte()!=ch2+0x20);
    break;
  case 0x54:
  case 0x5d:
  case 0x5e:
    break;   // Unreachable
  }
  return false;
}

bool DICLINE::DecodeDecorateEFX(int ch2,DICCHAR& chr,
			       DICSTRING& txt,bool& lflag){
  static ATTR prevAttr;
  switch(ch2){
  case 0xe0:  // ĥĴɽϻ
    {
      prevAttr=currentAttr;
      int ach1=file->GetWord();
      if (ach1&1) currentAttr.bold=true;
      if (ach1&2) currentAttr.italic=true;
      if (ach1&0x1000) currentAttr.italic=true;
      /*
      txt+=MakeString("[ĥĴɽ: ",currentAttr,currentTag);
      int ach1=file->GetByte();
      int ach2=file->GetByte();
      txt+=MakeHexString(ach1,currentAttr,currentTag);
      txt+=MakeHexString(ach2,currentAttr,currentTag);
      txt+=MakeString(":",currentAttr,currentTag);
      */
    }
    return true;
  case 0xe1:  // ĥĴɽλ
    currentAttr=prevAttr;
    // txt+=MakeString("]",currentAttr,currentTag);
    return true;

  case 0xfb: // S-EBXA NOP2 
    for(;;){
      while(file->GetByte()!=0x1f);
      if (file->GetByte()==0xfc) break;
    }
    return true;
    //  case 0xfc: // S-EBXA NOP2 λ
    //    return true;

  case 0xfe:
  case 0xff:
    return true; // Ignore
  default:
    if ((ch2 & 1)==0){
      txt+=MakeString("[ĥΰ ",currentAttr,currentTag);
      txt+=MakeHexString(ch2,currentAttr,currentTag);
      txt+=MakeString("/",currentAttr,currentTag);
      txt+=MakeHexString(ch2+1,currentAttr,currentTag);
      txt+=MakeString("]",currentAttr,currentTag);
      do {
	while(file->GetByte()!=0x1f);
      } while(file->GetByte()!=ch2+1);
      return true;
    } else {
      txt+=MakeString("[Error ",currentAttr,currentTag);
      txt+=MakeHexString(ch2,currentAttr,currentTag);
      txt+=MakeString("]",currentAttr,currentTag);
      return true;
    }
  }
  return false;
}


// ɽ浭һ (Nest )
bool DICLINE::DecodeDecorate(int ch2,DICCHAR& chr,
			     DICSTRING& txt,bool& lflag){
  switch((ch2 >> 4)&15){
  case 0:
    return DecodeDecorate0X(ch2,chr,txt,lflag);
  case 1:
    return DecodeDecorate1X(ch2,chr,txt,lflag);
  case 2:
    return DecodeDecorate2X(ch2,chr,txt,lflag);
  case 3:
  case 5:
    return DecodeDecorate35X(ch2,chr,txt,lflag);
  case 0xe:
  case 0xf:
    return DecodeDecorateEFX(ch2,chr,txt,lflag);
  }
  return false;
}

bool DICLINE::GetLine(DICTIONARY* dct,const TAG& t) {
  if (dct!=NULL) {
    dict=dct; 
    file=dict->BlockFile();
  } 
  if (!isNotEOF) {
    return false;
  }
  DICSTRING txt;
  if (t!=TAG(0,0)) {
    file->Seek(t);
    currentPoint=t;
  } 
  word ch;
  bool lflag=true;
  do { 
    DICCHAR chr;
    chr.Attr(currentAttr);
    chr.Tag(currentTag);
    ch=file->GetByte();
    if (ch==0) {
      txt+=DICCHAR(0);
      break;
    }

    // ʸǡ⵭һ β
    if (ch==0x1f) {
      int ch2=file->GetByte();
      // ޤ ɽ浭һҤ
      if (!DecodeDecorate(ch2,chr,txt,lflag)){ 
	switch(ch2) {
	case 0x40:  // ͽΰ (Ѷػ)
	  txt+=MakeString("<ͽΰ(Ѷػ)>",currentAttr,currentTag);
	  break;

	case 0x41:  // һ
	  file->GetWord(); // ATTR?
	  if (marginSplit==2 && currentMargin==1) {
	    /*
	    TAG t=file->Tell();
	    int b=t.Block();
	    int o=t.Offset()-4;
	    if (o<0) {
	      b--;
	      o+=cBlockSize;
	    }
	    txt+=MakeString("[Next]",currentAttr,TAG(b,o));
	    */
	    txt+=MakeString("[Next]",currentAttr,currentPoint);
	    isNotEOF=false;
	    lflag=false;
	  } else {
	    currentAttr.bold=true;
	  }
	  break;
	case 0x61:  // λһ
	  currentAttr.bold=false;
	  break;
	case 0x42:  // ̹ܻȵһ
	  currentTag=TAG(-1,-1);
	  {
	    TAG t=file->Tell();
	    if (file->GetWord()<0x1000){
	      file->Seek(t);
	      currentAttr.id=file->GetBCD(8);
	      txt+=DICCHAR(0x1f42,currentAttr,currentTag);
	    } else {
	      file->Seek(t);
	    }
	  }
	  break;
	case 0x62:  // ̹ܻȽλһ
	  {
	    currentAttr.id=0;
	    currentTag=TAG(0,0);
	    TAG t=file->GetBCDTag();
	    SetTag(txt,t);
	  }
	  break;
	case 0x43:  // ˥塼һ
	  currentTag=TAG(-1,-1);
	  break;
	case 0x63:  // ˥塼λһ
	  currentTag=TAG(0,0);
	  SetTag(txt,file->GetBCDTag());
	  break;
	case 0x44:  // Bitmap Start
	  {
	    currentAttr.id=0x44;
	    currentAttr.notText=true;
	    currentTag=file->Tell();
	    txt+=DICCHAR(0xffff,currentAttr,currentTag);
	    int type44=file->GetWord();
	    if (type44==1){
	      file->Read(NULL,8);
	      //	      currentAttr.pict=true;
	      char buf[128];
	      sprintf(buf," ");
	      txt+=MakeString(buf,currentAttr,currentTag);
	    }
	  }
	  break;
	case 0x64:  // Bitmap end
	  txt+=MakeString(" ",currentAttr,currentTag);
	  txt+=DICCHAR(0xffff,currentAttr,currentTag);
	  currentAttr.notText=false;
	  //	  currentAttr.pict=false;
	  currentAttr.id=0;
	  currentTag=TAG(0,0);
	  file->Read(NULL,6);
	  break;
	case 0x45:  // Start Figure??
	  pageFlag=false;
	  if (dict->isEB) {
	    menuCount=0;
	    txt+=DICCHAR(0x1f00+ch2,currentAttr,currentTag);
	  }
	  file->GetWord();
	  break;
	case 0x65:  // End Figure??
	  if (dict->isEB) {
	    if (pageFlag) {
	      isNotEOF=false;
	      lflag=false;
	    }
	    txt+=DICCHAR(0x1f00+ch2,currentAttr,currentTag);
	  }
	  break;
	case 0x4C:  // Picture List ?
	  {
	    menuCount=0;
	    file->GetWord();
	    txt+=DICCHAR(0x1f00+ch2,currentAttr,currentTag);
	  }
	  break;
	case 0x6C:  // Picture List End
	  {
	    // ====================================
	    if (!dict->isEB || marginSplit==0){
	      lflag=false;
	      isNotEOF=false;
	    }
	    txt+=DICCHAR(0x1f00+ch2,currentAttr,currentTag);
	  }
	  break;
	case 0x4A:  
	  if (dict->isEB){ // Random Jump
	    currentTag=TAG(-1,-1);
	    currentAttr.id=0x1f4A;
	  } else { // Audio (PCM) Start
	    currentTag=file->Tell();
	    currentAttr.id=0x4A;
	    currentAttr.notText=true;
	    //	    currentAttr.voice=true;
	    file->Read(NULL,4+6+6);
	    //	    txt+=MakeString("Snd:",currentAttr,currentTag);
	  }
	  break;
	case 0x6A:  
	  if (dict->isEB) { // Random Jump End
	    SetTag(txt,TAG(file->GetBCD(8),0));// ==== ⤷ DWord ?
	    currentTag=TAG(0,0); 
	    currentAttr.id=0;
	  } else {  // Audio (PCM) End
	    //	    txt+=MakeString("",currentAttr,currentTag);
	    currentAttr.id=0;
	    currentTag=TAG(0,0);
	    currentAttr.notText=false;
	    //	    currentAttr.voice=false;
	  }
	  break;
	case 0x4B:  // Jump?
	  if (dict->isEB){ // Jump Next
	    currentTag=TAG(-1,-1);
	    currentAttr.id=0x1f4B;
	  } else {
	    currentTag=file->GetBCDTag();
	    //	    currentAttr.pict=true;
	    //	    txt+=MakeString("Fig:",currentAttr,currentTag);
	  }
	  break;
	case 0x6B:
	  if (dict->isEB) { // Jump Next End
	    SetTag(txt,TAG(file->GetBCD(8),0));// ==== ⤷ DWord ?
	    currentTag=TAG(0,0); 
	    currentAttr.id=0;
	  } else  {
	    //	    txt+=MakeString("",currentAttr,currentTag);
	    //	    currentAttr.pict=false;
	    currentAttr.id=0;
	    currentTag=TAG(0,0);
	  }
	  break;
	case 0x4D:  // Figure (BMP,JPEG) Start
	  {
	    file->Read(NULL,12);
	    currentTag=file->GetBCDTag();
	    currentAttr.id=0x4D;
	    currentAttr.notText=true;
	    //	    currentAttr.pict=true;
	    txt+=DICCHAR(0xffff,currentAttr,currentTag);
	    //	    txt+=MakeString(" ",currentAttr,currentTag);
	  }
	  break;
	case 0x6D:  // Figure End
	  {
	    // txt+=MakeString(" ",currentAttr,currentTag);
	    currentTag=TAG(0,0);
	    currentAttr.id=0;
	    currentAttr.notText=false;
	    //	    currentAttr.pict=false;
	  }
	  break;
	case 0x4F: // Clickable Area
	  txt+=DICCHAR(0x1f4f,currentAttr,file->Tell());
	  file->Read(NULL,32);
	  break;
	case 0x6F:
	  break;

	case 0xa0: // S-EBXA ֤
	  currentAttr.underline=5;
	  break;
	case 0xa1: // S-EBXA ֤λ
	  currentAttr.underline=0;
	  break;

	case 0xa2:  // S-EBXA Color
	  currentAttr.color=(currentAttr.color &0xf0)|(file->GetWord() & 0xf);
	  //	  fprintf(stderr,"Color= %x\n",currentAttr.color);
	  txt+=DICCHAR(0x1fa2,currentAttr,currentTag);
	  break;
	case 0xa3:
	  currentAttr.color&=0xf0;
	  txt+=DICCHAR(0x1fa3,currentAttr,currentTag);
	  break;

	case 0xa4: // S-EBXA ?
	case 0xa5: // S-EBXA ?
	  break;

	case 0xaa: // S-EBXA ? 
	  break;

	default:
	  {	
	    txt+=MakeString("[1F",currentAttr,currentTag);
	    txt+=MakeHexString(ch2,currentAttr,currentTag);
	    txt+=MakeString("]",currentAttr,currentTag);
	  }
	  break;
	}
      }
    } else {
      DICCHAR chr=DICCHAR((ch<<8)+file->GetByte());
      chr.Attr(currentAttr);
      chr.Tag(currentTag);
      txt+=chr;
    }
  } while (lflag);
  text=txt;
  currentPoint=file->Tell();

  return true;
  //  return isNotEOF;
}

DICLINE::~DICLINE() {};

void DICCHAR::DicCharCopy(const DICCHAR& ch) {
  code=ch.code;
  attr=ch.attr;
  tag=ch.tag;
}

DICCHAR::DICCHAR(const DICCHAR& ch) {
  DicCharCopy(ch);
}

const DICCHAR& DICCHAR::operator= (const DICCHAR& ch) {
  DicCharCopy(ch);
  return *this;
}

