//  -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader "ebdic"
# Copyright (C) 1999-2000 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#include "codeconv.h"
#include <cctype>
#include <cstdio>

#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

#ifdef HAVE_LANGINFO_H
#include <langinfo.h>
#endif

const static byte tableHex[]="0123456789ABCDEF";
CODE_CONV_MODE convAll;

std::string CODECONV::Jis2Euc(const std::string& jisStr){
  int len=jisStr.length();
  std::string result="";
  for(int i=0;i<len;i++){
    if (jisStr[i]==0) break;
    if (static_cast<unsigned char>(jisStr[i])>=0xa1 && i<len-1){
      result+='\\';
      result+=tableHex[0xf & (jisStr[i]>>4)];
      result+=tableHex[0xf & jisStr[i]];
      result+=tableHex[0xf & (jisStr[i+1]>>4)];
      result+=tableHex[0xf & jisStr[i+1]];
      i++;
    } else {
      result+=char(jisStr[i++] | 0x80);
      result+=char(jisStr[i] | 0x80);
    }
  }
  return result;
};

std::string CODECONV::Jis2Euc2(const std::string& jisStr){
  const static byte table21[]=
    " @@,.\xa5:;?!@@@@@^~_@@@@@@@@@---/\\~@|@@`'\"\"()"
    "[][]{}<>@@@@@@@@+-@@@=@<>@@@@@@@'\"@\\$@@%#&*@@@@@@@";
  int len=jisStr.length();
  std::string result="";
  for(int i=0;i<len;i+=2){
    if (static_cast<unsigned char>(jisStr[i])>=0xa1){
      result+='\\';
      result+=tableHex[0xf & (jisStr[i]>>4)];
      result+=tableHex[0xf & jisStr[i]];
      result+=tableHex[0xf & (jisStr[i+1]>>4)];
      result+=tableHex[0xf & jisStr[i+1]];
    } else 
    switch(jisStr[i]){
    case 0x23:
      result+=jisStr[i+1];
      break;
    case 0x21:
      if (table21[jisStr[i+1]]!='@') {
	result+=table21[jisStr[i+1]];
	break;
      }
    default:
      result+=char(jisStr[i] | 0x80);
      result+=char(jisStr[i+1] | 0x80);
    }
  }
  return result;
};

int Hex2Dec(char hex){
  if ('0'<=hex && hex<='9') return hex-'0';
  if ('A'<=hex && hex<='F') return hex-'A'+10;
  if ('a'<=hex && hex<='f') return hex-'a'+10;
  return -1;
}

std::string CODECONV::Euc2Jis(const std::string& eucStr){
  std::string result="";
  int len=eucStr.length();
  for(int i=0;i<len;i++){
    if (eucStr[i]=='\\' && len>i+4){
      result+=Hex2Dec(eucStr[i+1])*16+Hex2Dec(eucStr[i+2]);
      result+=Hex2Dec(eucStr[i+3])*16+Hex2Dec(eucStr[i+4]);
      i+=4;
    } else if ((eucStr[i] & 0x80)==0){
      result+='\x23';
      result+=eucStr[i];
    } else {
      if ((eucStr[i+1] & 0x80)==0){
	result+=eucStr[i]; // Gaiji
      } else {
	result+=eucStr[i] & 0x7f;
      }
      result+=eucStr[i+1] & 0x7f;
      i++;
    }
  }
  return result;
};

std::string CODECONV::RevJis(const std::string& jisStr){
  std::string result="";
  int len=jisStr.length();
  for(int i=2;i<len;i+=2){
    result+=jisStr.substr(len-i,2);
  }
  return result;
}


static std::string ConvertKatakana(const std::string& jisStr){
  std::string result="";
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    char ch=jisStr[i];
    if (ch==0x25) {
      ch--;
    } 
    result+=ch;
    result+=jisStr[i+1];
  }
  return result;
};

static std::string ConvertEikomoji(const std::string& jisStr){
  std::string result="";
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    result+=jisStr[i];
    if (jisStr[i]==0x23 && isalpha(jisStr[i+1])){
      result+=toupper(jisStr[i+1]);
    } else {
      result+=jisStr[i+1];
    }
  }
  return result;
};

std::string CODECONV::JisUpcase(const std::string& jisStr){
  return ConvertEikomoji(jisStr);
}

static std::string ConvertKigou(const std::string& jisStr){
  std::string result="";
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    if (jisStr[i]!=0x21 ||
	( (jisStr[i+1]!=0x47) &&
	  (jisStr[i+1]!=0x5d) &&
	  (jisStr[i+1]!=0x26) &&
	  (jisStr[i+1]!=0x3e))) {
      result+=jisStr[i];
      result+=jisStr[i+1];
    }
  }
  return result;
};

static std::string ConvertChouon(const std::string& jisStr){
  const char* tbl="  "
                  ""
                  ""
                  ""
                  ""
                  "󤦤";
  std::string result="";
  char lastVowelH=0x24; // ''
  char lastVowelL=0x22;
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    if (jisStr[i]==0x24 || jisStr[i]==0x25){
      lastVowelH=jisStr[i];
      lastVowelL=tbl[(jisStr[i+1]-0x20)*2+1] & 0x7f;
    }
    if (jisStr[i]==0x21 && jisStr[i+1]==0x3c){
      result+=lastVowelH;
      result+=lastVowelL;
    } else {
      result+=jisStr[i];
      result+=jisStr[i+1];
    }
  }
  return result;
};

static std::string ConvertChouon2(const std::string& jisStr){
  std::string result="";
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    if (jisStr[i]!=0x21 || jisStr[i+1]!=0x4c){
      result+=jisStr[i];
      result+=jisStr[i+1];
    }
  }
  return result;
};

static std::string ConvertSokuon(const std::string& jisStr) {
  std::string result="";
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    result+=jisStr[i];
    if (jisStr[i]==0x24 && jisStr[i+1]==0x43){  //  
      result+='\x44';
    } else if (jisStr[i]==0x25 && jisStr[i+1]==0x43){ //  
      result+='\x44';
    } else {
      result+=jisStr[i+1];
    }
  }
  return result;
};

static std::string ConvertYouon(const std::string& jisStr){
  std::string result="";
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    result+=jisStr[i];
    if (jisStr[i]==0x24 || jisStr[i]==0x25){
      char ch=jisStr[i+1];
      if (ch==0x63 || ch==0x65 || ch==0x67 || ch==0x6e){  // 
	ch++;
      } else if (jisStr[i]==25 && ch==0x75){              // 
	ch=0x2b;
      } else if (jisStr[i]==25 && ch==0x76){              // 
	ch=0x31;
      }
      result+=ch;
    } else {
      result+=jisStr[i+1];
    }
  }
  return result;
};

static std::string ConvertKomoji(const std::string& jisStr){
  std::string result="";
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    result+=jisStr[i];
    if (jisStr[i]==0x24 || jisStr[i]==0x25){
      char ch=jisStr[i+1];
      if (ch==0x21 || ch==0x23 || ch==0x25 || ch==0x27 || ch==0x29){  
	//   =>  
	ch++;
      }
      result+=ch;
    } else {
      result+=jisStr[i+1];
    }
  }
  return result;
};

static std::string ConvertDakuon(const std::string& jisStr){
  static std::string src=
    CODECONV::Euc2Jis(std::string("¤ŤǤɤФӤ֤٤ܥ"));
  static std::string dest=
    CODECONV::Euc2Jis(std::string("ĤƤȤϤҤդؤۤ"));
  std::string result="";
  int len=jisStr.length();
  int len2=src.length();
  for(int i=0;i<len;i+=2){
    result+=jisStr[i];
    if (jisStr[i]==0x24 || jisStr[i]==0x25){
      char ch=jisStr[i+1];
      for(int j=1;j<len2;j+=2){
	if (ch==src[j]) {
	  ch=dest[j];
	  break;
	}
      }
      result+=ch;
    } else {
      result+=jisStr[i+1];
    }
  }
  return result;
};

static std::string ConvertHandakuon(const std::string& jisStr){
  static std::string src=CODECONV::Euc2Jis("ѤԤפڤ");
  static std::string dest=CODECONV::Euc2Jis("ϤҤդؤ");
  std::string result="";
  int len=jisStr.length();
  int len2=src.length();
  for(int i=0;i<len;i+=2){
    result+=jisStr[i];
    if (jisStr[i]==0x24 || jisStr[i]==0x25){
      char ch=jisStr[i+1];
      for(int j=1;j<len2;j+=2){
	if (ch==src[j]) {
	  ch=dest[j];
	  break;
	}
      }
      result+=ch;
    } else {
      result+=jisStr[i+1];
    }
  }
  return result;
};

std::string CODECONV::ConvertForCompare(const std::string& jisStr,
				   const CODE_CONV_MODE& convMode){
  std::string tmp1=jisStr;
  if (convMode.convertKatakana==0) tmp1=ConvertKatakana(tmp1);
  if (convMode.convertEikomoji==0) tmp1=ConvertEikomoji(tmp1);
  if (convMode.convertKigou==0)    tmp1=ConvertKigou(tmp1);
  if (convMode.convertChouon==0) {
    tmp1=ConvertChouon(tmp1);
  } else if (convMode.convertChouon==2) {
    tmp1=ConvertChouon2(tmp1);
  }
  if (convMode.convertSokuon==0)   tmp1=ConvertSokuon(tmp1);
  if (convMode.convertYouon==0)    tmp1=ConvertYouon(tmp1);
  if (convMode.convertKomoji==0)   tmp1=ConvertKomoji(tmp1);
  if (convMode.convertDakuon==0)   tmp1=ConvertDakuon(tmp1);
  if (convMode.convertHandakuon==0)tmp1=ConvertHandakuon(tmp1);
  return tmp1;
}

int CODECONV::CompareKana(const std::string& str1, const std::string& str2){
  // ================================
  convAll.Setup(NULL);

  std::string str_a=ConvertForCompare(str1,convAll);
  std::string str_b=ConvertForCompare(str2,convAll);

  if (str_a>str_b) return 1;
  if (str_a<str_b) return -1;
  return 0;
}

bool CODE_CONV_MODE::Setup(byte* buf) {
  if (buf!=NULL){
    convertKatakana=(buf[1]>>6)&3;
    convertEikomoji=(buf[1]>>4)&3;
    convertKigou   =(buf[1]>>2)&3;
    convertChouon  =buf[1]&3;
    convertSokuon  =(buf[2]>>6)&3;
    convertYouon   =(buf[2]>>4)&3;
    convertKomoji  =(buf[2]>>2)&3;
    convertDakuon  =buf[2]&3;
    convertHandakuon=(buf[3]>>6)&3;
    /*    Debug::DebugOut(Debug::TEMP,"CODE CONV MODE : %d %d %d %d %d %d %d %d %d \n",
	    convertKatakana,convertEikomoji,convertKigou,
	    convertChouon,convertSokuon,convertYouon,convertKomoji,
	    convertDakuon, convertHandakuon);
    */
    return true;
  } else {
    convertKatakana=0;
    convertEikomoji=0;
    convertKigou=0;
    convertChouon=0;
    convertSokuon=0;
    convertYouon=0;
    convertKomoji=0;
    convertDakuon=0;
    convertHandakuon=0;
  }
  return false;
}

bool CODECONV::isKana(const std::string& jisStr) {
  int len=jisStr.length();
  for(int i=0;i<len;i+=2){
    if (jisStr[i]==0) break;
    if (static_cast<unsigned char>(jisStr[i])>=0xa1){
      return false; // Gaiji
    } else if (jisStr[i]!=0x24 && jisStr[i]!=0x25) {
      if (jisStr[i]==0x21 && jisStr[i+1]==0x3c) continue; // ""
      return false;          // Not Hiragana && Katakana
    }
  }
  return true;
}

void CODECONV::Initialize(){
  if (initialized) return;
  initialized=true;

#if defined(HAVE_NL_LANGINFO) && defined(CODESET)
  char* buf=nl_langinfo(CODESET);
  if (buf!=NULL) {
    codeset=std::string(buf);
  } else {
    char* LC_MES0=setlocale(LC_MESSAGES,"");
    char* mes=strdup(setlocale(LC_MESSAGES,LC_MES0));
    while(*mes!=0 && *mes!='.') mes++;
    if (*mes=='.') {
      mes++;
      char* mp;
      for(mp=mes;*mp!=';' && *mp!=0; mp++);
      *mp=0;
      CODECONV::codeset=mes;
    }
  }
  Debug::DebugOut(Debug::CODE,"CODESET='%s'\n",CODECONV::codeset.c_str());
#endif

#ifdef USE_ICONV
  InitIConv();
#endif
}

#ifdef USE_ICONV
bool CODECONV::InitIConv(void){
  if (codeset.length()==0) codeset="EUC-JP";
  ec=iconv_open(codeset.c_str(),"EUC-JP");
  ce=iconv_open("EUC-JP",codeset.c_str());
  eu=iconv_open("UTF-8","EUC-JP");
  ue=iconv_open("EUC-JP","UTF-8");
  if (ec==(iconv_t)(-1) || ce==(iconv_t)(-1) ||
      eu==(iconv_t)(-1) || ue==(iconv_t)(-1)) return false;
  return true;
}

std::string CODECONV::IConvInternal(iconv_t xcd,const std::string& s){
  Debug::DebugOut(Debug::CODE,"IConv:From(%s) len=%d\n",s.c_str(),s.length());
//  Debug::DebugOut(Debug::CODE,"(%x:%x) \n",s[0],s[1]);

  std::string out;
  size_t ilen=s.length();
  char* ibuf0=new char[ilen];
  char* ibuf=ibuf0;
  memcpy(ibuf,s.c_str(),ilen);
  char buf[1025];
  size_t olen;
  while(ilen>0){
    char* obuf=buf;
    olen=1024;
    size_t ret = iconv(xcd,&ibuf,&ilen,&obuf,&olen);
    if (ret>=0) ret= iconv(xcd,&ibuf,&ilen,&obuf,&olen);
    *obuf=0;
    out.append(buf,1024-olen);
  }
  delete [] ibuf0;
  Debug::DebugOut(Debug::CODE,"IConv:From(%s) len=%d ok\n",s.c_str(),s.length());
  return out;
}

std::string CODECONV::EUCToCurrent(const std::string& s) {
  return IConvInternal(ec,s);
};

std::string CODECONV::CurrentToEUC(const std::string& s) {
  return IConvInternal(ce,s);
}

std::string CODECONV::EUCToUTF8(const std::string& s) {
  return IConvInternal(eu,s);
}

std::string CODECONV::UTF8ToEUC(const std::string& s) {
  return IConvInternal(ue,s);
}



iconv_t CODECONV::ec=(iconv_t)-1;
iconv_t CODECONV::ce=(iconv_t)-1;
iconv_t CODECONV::eu=(iconv_t)-1;
iconv_t CODECONV::ue=(iconv_t)-1;
#endif

bool CODECONV::initialized=false;
std::string CODECONV::codeset="EUC-JP";

