// -*- C++ -*-
/*
#
# This Program is part of Dictionary Reader
# Copyright (C) 2001 Takashi Nemoto
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details. 
#
#    Send bugs and comments to tnemoto@mvi.biglobe.ne.jp
#
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifndef HAVE_LIBXML

#include <cstdio>
#include <cstring>
#include <string>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_LANGINFO_H
#include <langinfo.h>
#endif
#include "codeconv.h"
#include "tinyxml.h"
#include "debug.h"



xmlNode::~xmlNode(void){
}

static char* ToUTF(iconv_t cd, const string& str){
  return strdup(CODECONV::IConvInternal(cd,str).c_str());
}

#ifdef USE_ICONV
static xmlNodePtr xmlParseFileSub(FILE* fp,int level=0,iconv_t cd=(iconv_t)-1)
#else
static xmlNodePtr xmlParseFileSub(FILE* fp,int level=0)
#endif
{
  xmlNodePtr top=new xmlNode;
  int ch;
  string val;
  while((ch=getc(fp))!=EOF){
    if (ch=='<'){
      ch=getc(fp);
      if (ch=='?'){
	string buf;
	while((ch=getc(fp))!=EOF){
	  buf+=ch;
	  if (ch!='?') continue;
	  if ((ch=getc(fp))!='>') {
	    if (ungetc(ch,fp)==ch) continue;
	  }
	  break;
	}
      } else if (ch=='!'){
	int ch1,ch2;
	ch1=getc(fp);
	ch2=getc(fp);
	if (ch1=='-' && ch2=='-'){
	  while((ch=getc(fp))!=EOF){
	    if (ch!='-') continue;
	    ch1=getc(fp);
	    ch2=getc(fp);
	    if (ch1=='-' && ch2=='>') break;
	    ungetc(ch2,fp);
	    ungetc(ch1,fp);
	  }
	} else {
	  ungetc(ch2,fp);
	  ungetc(ch1,fp);
	  while((ch=getc(fp))!=EOF && ch!='>');
	}
      } else if (ch=='/'){
	while((ch=getc(fp))!=EOF && ch!='>');
	//	if (top->xmlChildrenNode!=NULL) return top;
#ifdef USE_ICONV
	if (cd!=(iconv_t)-1){
	  top->val=ToUTF(cd,val);
	} else {
	  top->val=strdup(val.c_str()); 
	}
#else
	top->val=strdup(val.c_str()); 
#endif
	return top;
      } else {
	string nm;
	nm+=char(ch);
	while((ch=getc(fp))!=EOF && ch!='>'){
	  nm+=char(ch);
	}
	xmlNodePtr p=top;
	if (p->name!=NULL) {
	  while(p->next!=NULL) p=p->next;
	  p->next=new xmlNode;
	  p->next->parent=p;
	  p=p->next;
	}
	p->name=strdup(nm.c_str());
#ifdef USE_ICONV
	p->xmlChildrenNode=xmlParseFileSub(fp,level+1,cd);
#else
	p->xmlChildrenNode=xmlParseFileSub(fp,level+1);
#endif
	p->xmlChildrenNode->parent=p;
      }
    } else {
      val+=char(ch);
    }
  }
  if (top->name==NULL) {
    delete top;
    return NULL;
  }
 ok:
  return top;
}

xmlNodePtr xmlParseFile(char* fname){
  FILE* fp=fopen(fname,"r");
  char buf[1024];
#ifdef USE_ICONV
  iconv_t cd;
#endif
  if (fp==NULL) return NULL;
  fgets(buf,1000,fp);
  string enc=buf;
  if (enc.find("encoding")!=0) {
    enc=enc.substr(enc.find("encoding"));
    enc=enc.substr(enc.find('"'));
    enc=enc.substr(1,enc.find('"',1)-1);
#ifdef USE_ICONV
    cd=iconv_open("UTF-8",enc.c_str());
#endif
  }

#ifdef USE_ICONV
  xmlNodePtr p=xmlParseFileSub(fp,0,cd);
  if (cd!=(iconv_t)-1) iconv_close(cd);
#else
  xmlNodePtr p=xmlParseFileSub(fp,0,0);
#endif
  fclose(fp);
  return p;
}

void xmlFreeDoc(xmlNodePtr top){
  if (top->name!=NULL) free(top->name);
  if (top->val!=NULL) free(top->val);
  if (top->xmlChildrenNode!=NULL) xmlFreeDoc(top->xmlChildrenNode);
  if (top->next!=NULL) xmlFreeDoc(top->next);
  delete top;
}

void xmlUnlinkNode(xmlNodePtr top){
  if (top==NULL) return;
  if (top->parent==NULL) return;
  xmlNodePtr parent=top->parent;
  if (parent->xmlChildrenNode==top){
    parent->xmlChildrenNode=NULL;
    top->parent=NULL;
  } else if (parent->next==top){
    parent->next=top->next;
    top->parent=NULL;
  } else {
    Debug::DebugOut(Debug::TEMP,"xmlUnlinkError\n");
  }
};

void xmlFreeNode(xmlNodePtr top){ 
  xmlUnlinkNode(top);
  if (top->xmlChildrenNode!=NULL) delete (top->xmlChildrenNode);
  if (top->next!=NULL) delete (top->next);
  if (top->name!=NULL) free(top->name);
  if (top->val!=NULL) free(top->val);
  delete top;
}

xmlNodePtr xmlNewChild(xmlNodePtr ptr,void * dummy,
		       const xmlChar* name,const xmlChar* stringVal){
  xmlNodePtr cur;
  if (ptr->xmlChildrenNode==NULL){
    ptr->xmlChildrenNode=new xmlNode;
    cur=ptr->xmlChildrenNode;
    cur->parent=ptr;
  } else {
    cur=ptr->xmlChildrenNode;
    while(cur->next!=NULL) cur=cur->next;
    cur->next=new xmlNode;
    cur->next->parent=cur;
    cur=cur->next;
  }
  if (name!=NULL) cur->name=strdup(name);
  if (stringVal!=NULL) xmlNodeSetContent(cur,stringVal);
  return cur;
}

void xmlSaveFileEnc(const char* fileName,xmlDocPtr doc,char* encoding){
  xmlSaveFile(fileName,doc);
};

static void xmlSaveFileSub(FILE* fp,xmlNodePtr np){
  if (np==NULL) return;
  if (np->name!=NULL){
    fprintf(fp,"<%s>",np->name);
    if (np->xmlChildrenNode!=NULL){
      xmlSaveFileSub(fp,np->xmlChildrenNode);
    }
    fprintf(fp,"</%s>\n",np->name);
    if (np->next){
      xmlSaveFileSub(fp,np->next);
    }
  } else if (np->val!=NULL){
    fprintf(fp,"%s",_EC(_UE(np->val)).c_str());
  }
}

void xmlSaveFile(const char* fileName,xmlDocPtr doc){
  FILE* fp=fopen(fileName,"w");
  fprintf(fp,"<?xml version=\"1.0\" encoding=\"%s\"?>\n",CODECONV::codeset.c_str());
  fprintf(fp,"<!-- -*- XML -*- -->\n");
  fprintf(fp,"<!DOCTYPE ebdic SYSTEM \"%s\">\n","ebdic.dtd");
  xmlSaveFileSub(fp,doc);
  fclose(fp);
};

xmlDocPtr xmlParseDoc(const char* str){
  char fname[1024];
  if (str==NULL) return NULL;
  snprintf(fname,1000,"%s/XXXXXX","/tmp");
  int fd=mkstemp(fname);
  if (fd<0) return NULL;
  write(fd,str,strlen(str));
  close(fd);
  xmlNodePtr np=xmlParseFile(fname);
  unlink(fname);
  return np;
};

void xmlNodeSetContent(xmlNodePtr np,const xmlChar* str){
  if (str==NULL) return;
  if (np->xmlChildrenNode!=NULL) xmlFreeNode(np->xmlChildrenNode);
  np->xmlChildrenNode=new xmlNode;
  np=np->xmlChildrenNode;
  np->val=strdup(str);
}

char* xmlNodeGetContent(xmlNodePtr node)
{
  if (node==NULL || node->name==NULL 
      || node->xmlChildrenNode==NULL || node->xmlChildrenNode->val==NULL) {
    return NULL;
  }
  return strdup(node->xmlChildrenNode->val);
};


#endif
