/***************************************************************************
 * CT-API library for the REINER SCT cyberJack pinpad/e-com USB.
 * Copyright (C) 2005  REINER SCT
 * Author: Harald Welte
 * Support: support@reiner-sct.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * File: cjio.c
 * CVS: $Id: cjio_user.c 54 2007-03-13 22:16:21Z martin $
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <errno.h>
/* #include <sys/stat.h> */
#include <sys/time.h>
/* #include <sys/types.h> */
#include <time.h>
#include "../ausb/ausb_l.h"

#include "cj.h"
#include "cjio.h"



static void ecom_handle_interrupt(struct usbdevfs_urb *uurb, void *userdata){
  ausb_dev_handle *ah;
  CJ_INFO *ci = userdata;
  short old_rdtodo;
  unsigned short size;
  unsigned char *data;

  DEBUGP("%s entered\n", __FUNCTION__);

  if (!uurb) {
    DEBUGP("received interrupt signal, but no urb?\n");
    return;
  }

  if (uurb->status < 0) {
    DEBUGP("interupt urb with status %d, aborting\n", uurb->status);
    return;
  }

  data = uurb->buffer;
  ah = uurb->usercontext;

  /* Handle the interrupt URB */

  /* make sure this is a bulk_in signalling interrupt */
  /* FIXME: should this mean data[0]!=0x01?? */
  if ((uurb->actual_length != 4) || data[0] == 0x01)
    goto resubmit;

  size = ((unsigned short)data[3]<<8)+data[2]+3;
  old_rdtodo = ci->ll.libusb.rdtodo;

  if ((old_rdtodo+size) < old_rdtodo)
    goto resubmit;

  ci->ll.libusb.rdtodo += size;
  DEBUGP("Changing size from %d to %d\n", old_rdtodo, ci->ll.libusb.rdtodo);

  /* if somebody wants to receive data, he can now submit
   * the read URB on the bulk in pipe */

resubmit:
  /* we have to resubmit the interrupt urb */
  DEBUGP("resubmitting URB %p\n", uurb);
  if (ausb_submit_urb(ah, uurb))
    DEBUGP("unable to resubmit interrupt urb\n");

  return;
}



static int ecom_setup_inturb(CJ_INFO *ci){
  if (ausb_register_callback(ci->ll.libusb.dh,
                             USBDEVFS_URB_TYPE_INTERRUPT,
                             ecom_handle_interrupt, ci)) {
    DEBUGP("unable to register interrupt callback\n");
    return -1;
  }
  ausb_fill_int_urb(&ci->ll.libusb.uurb, 0x81, ci->ll.libusb.buffer,
                    sizeof(ci->ll.libusb.buffer));
  ci->ll.libusb.dh->intUrb=&ci->ll.libusb.uurb;
  DEBUGP("submitting URB %p\n", &ci->ll.libusb.uurb);
  if (ausb_submit_urb(ci->ll.libusb.dh, &ci->ll.libusb.uurb)) {
    DEBUGP("unable to submit interrupt urb\n");
    ci->ll.libusb.dh->intUrb=0;
    return -1;
  }

  return 0;
}



/* Open device.
 * ci is a pointer to a CJ_INFO struct.
 * Returns CJ_EXIT_*.
 */
int cjIoOpen_libusb(CJ_INFO *ci){
  BYTE buffer[7];
  int ret, len;

  if (!ci)
    return CJ_EXIT_BAD_PARAM;

  /* Currently only USB supported */
  if (ci->type != CJ_IO_TYPE_LIBUSB)
    return CJ_EXIT_BAD_PARAM;

  DEBUGP("trying to open device\n");
  ci->ll.libusb.dh = ausb_open(ci->ll.libusb.dev);
  if (!ci->ll.libusb.dh)
    return CJ_EXIT_FILE_ERROR;

  //ausb_set_configuration(ci->ll.libusb.dh, 0);

  DEBUGP("trying to claim interface\n");
  if (ausb_claim_interface(ci->ll.libusb.dh, 0) < 0) {
    char name[PATH_MAX+1];
    if (ausb_get_driver_np(ci->ll.libusb.dh, 0, name, PATH_MAX) < 0)
      return CJ_EXIT_FILE_ERROR;
    DEBUGP("driver `%s' using the interface\n", name);
    if (strcmp(name, "cyberjack")) {
      fprintf(stderr, "not detaching unknown driver `%s'\n",
              name);
      return CJ_EXIT_FILE_ERROR;
    }
    if (ausb_detach_kernel_driver_np(ci->ll.libusb.dh, 0) < 0)
      return CJ_EXIT_FILE_ERROR;
    if (ausb_claim_interface(ci->ll.libusb.dh, 0) < 0)
      return CJ_EXIT_FILE_ERROR;
  }

  DEBUGP("successfully claimed interface\n");

  /* Reset the device */
  ausb_reset(ci->ll.libusb.dh);

#if 0
  if (ausb_clear_halt(ci->ll.libusb.dh, 0x02)) {
    fprintf(stderr, "error during CLEAR_HALT\n");
    return CJ_EXIT_FILE_ERROR;
  }
#endif

  if (ecom_setup_inturb(ci) < 0)
    return CJ_EXIT_FILE_ERROR;

  /* always accept interrupts */
  ausb_start_accept_int(ci->ll.libusb.dh);

  /* Fill CJ_INFO */
  ci->type = CJ_IO_TYPE_LIBUSB;
  DEBUGP("reseting ns/nr\n");
  ci->t1.ns = 0;
  ci->t1.nr = 0;
  ci->t1.ifsc = 0xFF;
  ci->t1.ifsd = 0xFF;
  ci->t1.ifsreq = FALSE;
  ci->t1.bwt = 8000000L;
  ci->t1.wtx = 0L;
  ci->t1.cwt = 100000L;	/* Not really CWT */

  usleep(200000);

  ret = cjIoSendBlock(ci, (unsigned char *) "\xE2\xC1\x00\x23", 4);
  if (ret < 0)
    return ret;

  /* Send a S(RESYCH) to resychronise with reader and to test if there
   * is a appropriate reader at the other end of the line.
   */
  ret = cjIoSendBlock(ci, (unsigned char *) "\xE2\xC0\x00\x22", 4);
  if (ret < 0)
    return ret;

#if 1
  ret = cjIoReceiveBlock(ci, buffer, &len);
  if (ret < 0)
    return ret;
  if (len != 4)
    return CJ_EXIT_IO_ERROR;

  /* Correct response? */
  if (memcmp(buffer, "\x2E\xE0\x00\xCE", 4))
    return CJ_EXIT_IO_ERROR;
#endif
  return CJ_EXIT_OK;
}



/* Close device.
 * ci is a pointer to a CJ_INFO struct.
 * Returns CJ_EXIT_*.
 */
int cjIoClose_libusb(CJ_INFO *ci){
  int ret;

  if (!ci)
    return CJ_EXIT_BAD_PARAM;

  ret = cjIoSendBlock(ci, (unsigned char*) "\xE2\xC1\x00\x23", 4);
  if (ret < 0)
    return ret;

  ausb_end_accept_int(ci->ll.libusb.dh);

  ausb_discard_urb(ci->ll.libusb.dh, &ci->ll.libusb.uurb);
  ausb_reset(ci->ll.libusb.dh);
  ausb_release_interface(ci->ll.libusb.dh, 0);
  ausb_reattach_kernel_driver_np(ci->ll.libusb.dh, 0);
  ausb_close(ci->ll.libusb.dh);

  return CJ_EXIT_OK;
}



/* Send block to reader.
 * ci is a pointer to a CJ_INFO struct.
 * data contains a pointer to the data to transmit.
 * datalen contains the length of the data to transmit.
 * Returns CJ_EXIT_*.
 */
int cjIoSendBlock_libusb(CJ_INFO *ci, BYTE *data, int datalen){
  BYTE buffer[3+3+255+1];
  int ptr, size, ret;
  int timeout = 1000;

  buffer[0] = 0x00;
  /* Quick fix. 1-byte blocks are not possible with T=1. */
  if (datalen == 1) buffer[0]=0xFF;
  buffer[1] = datalen & 0xFF;
  buffer[2] = (datalen>>8) & 0xFF;

  memcpy(buffer+3, data, datalen);

  datalen += 3;

  rsct_log_bytes(CT_FLAGS_DEBUG_TRANSFER,
		 __FILE__, __LINE__, __FUNCTION__,
		 "PC->CYBJCK", datalen, data);

  for (ptr = 0; ptr < datalen; ptr += 64) {
    size = min(64, datalen-ptr);
    DEBUGP("write(buffer+%d,%d)\n", ptr, size);

    ret = ausb_bulk_write(ci->ll.libusb.dh, 0x02, (char *) buffer+ptr,
                          size, timeout);
    if (ret != size) {
      DEBUGP("write(buffer+%d,%d) sent %d bytes\n",
             ptr, size, ret);
      /* return CJ_EXIT_IO_ERROR; */
    }
  }

  return CJ_EXIT_OK;
}



/* Receive block from reader.
 * ci is a pointer to a CJ_INFO struct.
 * data contains a pointer to the buffer, which should receive the data.
 * datalen contains a pointer to an int, which must contain on calling the
 * size of the buffer and which returns the length of data received.
 * Returns CJ_EXIT_*.
 */
int cjIoReceiveBlock_libusb(CJ_INFO *ci, BYTE *data, int *datalen){
  int timeout;
  BYTE buffer[5*64];
  int ret, len=0;
  LONG tmpbwt;

  if (!ci || !data || !datalen)
    return CJ_EXIT_BAD_PARAM;

  /*
  if (ci->ll.libusb.rdtodo == 0)
    DEBUGP("read but no data available (yet?)\n");*/

  usleep(20);

  if (ci->t1.wtx)
    tmpbwt = ci->t1.bwt*ci->t1.wtx;
  else
    tmpbwt = ci->t1.bwt;
  ci->t1.wtx = 0;

  /* Wait for first data with BWT */
  timeout = tmpbwt/1000;
  DEBUGP("timeout=%d\n", timeout);

  /* Read */
  DEBUGP("read(buffer,1)\n");
  while (1) {
    int size = min(64, sizeof(buffer)-len);
    ret = ausb_bulk_read(ci->ll.libusb.dh, 0x82, (char *) buffer+len,
                         size, timeout);
    if (ret == -ERESTART)
      continue;
    if (ret < 0) {
      DEBUGP("ret = %d (%s)\n", ret, strerror(errno));
      return CJ_EXIT_PROTOCOL_ERROR;
    }

    len += ret;
    if (len > (3+3+255+1)) {
      DEBUGP("protocol error\n");
      return CJ_EXIT_PROTOCOL_ERROR;
    }

    if (len >= 3) {
      short size = ((short)buffer[2]<<8)+buffer[1]+3;
      if (len >= size)
        break;
    }

    /* Wait for next data with CWT (Is not really CWT.) */
    timeout = ci->t1.cwt/1000;

    ci->ll.libusb.rdtodo -= ret; /* FIXME! */

    /* Read rest of data */
    DEBUGP("read(buffer+%d,1)\n", len);
  }

  len += ret;

  if (len == 0) {
    DEBUGP("timeout!\n");
    return CJ_EXIT_TIMEOUT;
  }

  /* Data length */
  *datalen = ((int)buffer[2]<<8) + buffer[1];
  if ((*datalen)>(3+255+1)) {
    DEBUGP("datalength to big\n");
    return CJ_EXIT_PROTOCOL_ERROR;
  }

  /* Copy data */
  memcpy(data, buffer+3, *datalen);

  rsct_log_bytes(CT_FLAGS_DEBUG_TRANSFER,
		 __FILE__, __LINE__, __FUNCTION__,
		 "CYBJCK->PC", *datalen, data);

  return CJ_EXIT_OK;
}


