/***************************************************************************
 * CT-API library for the REINER SCT cyberJack pinpad/e-com USB.
 * Copyright (C) 2004  REINER SCT
 * Author: Harald Welte
 * Support: support@reiner-sct.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * File: cjctapi_switch.c
 * CVS: $Id: cjctapi_switch.c 54 2007-03-13 22:16:21Z martin $
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdarg.h>
#include <time.h>
#include <sys/time.h>
#include <string.h>
#include <usb.h>

#include "cyberjack_l.h"
#include "ausb/ausb_l.h"

#include "ecom/ctapi-ecom.h"
#include "include/ctapi.h"
#include "ppa/cjppa.h"

#include "ctn_list.h"
#include "cjctapi_beep.h"
#include "ctapi_config_l.h"

#define CJPPA_USB_VENDOR_ID	0x0c4b

#define CJECOM_USB_DEVICE_ID	0x0100
#define CJPPA_USB_DEVICE_ID	0x0300

#define cjctapi_enter() cjctapi_log("ENTERING\n")

#define DEBUGP(format, args...) \
  rsct_log(CT_FLAGS_DEBUG_CTAPI, __FILE__, __LINE__, __FUNCTION__, format, ## args)


static int _ctapi_init_count=0;

static struct beep_struct *beepstruct = NULL;


CYBERJACK_EXPORT
void rsct_log(unsigned int what, char *file, int line, const char *function,
	      const char *format, ...){
  if (config_get_flags() & what) {
    va_list ap;
    struct timeval tv;
    struct tm *tm;
    const char *s;
    const char *wstr;
    FILE *outfd;
    int pid;

    pid=getpid();
    s=config_get_debug_filename();
    if (s) {
      outfd=fopen(s, "a+");
      if (outfd==0) {
        outfd=stderr;
        fprintf(stderr, "CYBERJACK: Could not create outfd\n");
      }
    }
    else
      outfd=stderr;

    switch(what) {
    case CT_FLAGS_DEBUG_GENERIC:    wstr="generic:  "; break;
    case CT_FLAGS_DEBUG_READER:     wstr="reader:   "; break;
    case CT_FLAGS_DEBUG_CTAPI:      wstr="ctapi:    "; break;
    case CT_FLAGS_DEBUG_AUSB:       wstr="ausb:     "; break;
    case CT_FLAGS_DEBUG_CJPPA:      wstr="cjppa:    "; break;
    case CT_FLAGS_DEBUG_ECOM:       wstr="ecom:     "; break;
    case CT_FLAGS_DEBUG_TRANSFER:   wstr="transfer: "; break;
    case CT_FLAGS_DEBUG_USB:        wstr="usb:      "; break;
    case CT_FLAGS_DEBUG_IFD:        wstr="ifd:      "; break;
    default:                        wstr="unknown:  "; break;
    }

    va_start(ap, format);
    gettimeofday(&tv, NULL);
    tm=localtime(&tv.tv_sec);
    fprintf(outfd, "[%08x] %04d/%02d/%02d %02d:%02d:%02d:%06d: %s"
            "%s:%4d:%s(): ",
            pid,
            tm->tm_year+1900,
            tm->tm_mon,
            tm->tm_mday,
            tm->tm_hour,
            tm->tm_min,
            tm->tm_sec,
            (int)tv.tv_usec,
            wstr,
            file, line, function);
    vfprintf(outfd, format, ap);
    va_end(ap);
    fflush(outfd);
    if (outfd!=stderr)
      fclose(outfd);
  }
}



CYBERJACK_EXPORT
void rsct_log_bytes(unsigned int what, char *file, int line,
		    const char *function,
		    const char *hdr,
		    int datalen, const unsigned char *data){
  if (config_get_flags() & what) {
    struct timeval tv;
    struct tm *tm;
    const char *s;
    const char *wstr;
    FILE *outfd;
    int pid;

    pid=getpid();

    s=config_get_debug_filename();
    if (s) {
      outfd=fopen(s, "a+");
      if (outfd==0) {
        outfd=stderr;
        fprintf(stderr, "CYBERJACK: Could not create outfd\n");
      }
    }
    else
      outfd=stderr;

    switch(what) {
    case CT_FLAGS_DEBUG_GENERIC:    wstr="generic:  "; break;
    case CT_FLAGS_DEBUG_READER:     wstr="reader:   "; break;
    case CT_FLAGS_DEBUG_CTAPI:      wstr="ctapi:    "; break;
    case CT_FLAGS_DEBUG_AUSB:       wstr="ausb:     "; break;
    case CT_FLAGS_DEBUG_CJPPA:      wstr="cjppa:    "; break;
    case CT_FLAGS_DEBUG_ECOM:       wstr="ecom:     "; break;
    case CT_FLAGS_DEBUG_TRANSFER:   wstr="transfer: "; break;
    case CT_FLAGS_DEBUG_USB:        wstr="usb:      "; break;
    case CT_FLAGS_DEBUG_IFD:        wstr="ifd:      "; break;
    default:                        wstr="unknown:  "; break;
    }

    gettimeofday(&tv, NULL);
    tm=localtime(&tv.tv_sec);
    fprintf(outfd, "[%08x] %04d/%02d/%02d %02d:%02d:%02d:%06d: %s"
            "%s:%4d:%s(): ",
            pid,
            tm->tm_year+1900,
            tm->tm_mon,
            tm->tm_mday,
            tm->tm_hour,
            tm->tm_min,
            tm->tm_sec,
            (int)tv.tv_usec,
            wstr,
            file, line, function);
    fprintf(outfd, "%s: ", hdr);
    do {
      int i;

      for( i=0; i<(datalen); i++ )
        fprintf(outfd, " %.2X", (data)[i] );
    } while(0);
    fprintf(outfd, "\n");

    fflush(outfd);
    if (outfd!=stderr)
      fclose(outfd);
  }
}


static int usb_read = 0;

/* find an apropriate USB device */
static struct usb_device *find_rsct_usbdev(int num){
  struct usb_bus *busses, *bus;
  struct usb_device *dev;
  int found = 0;

  cjctapi_enter();

  if (!usb_read) {
    ausb_init();
    usb_read = 1;
  }

  busses = usb_get_busses();

  /* FIXME: this ignores topology changes after first device was opened */
  for (bus = busses; bus; bus = bus->next) {
    for (dev = bus->devices; dev; dev = dev->next) {
      if (dev->descriptor.idVendor == CJPPA_USB_VENDOR_ID &&
          (dev->descriptor.idProduct == CJPPA_USB_DEVICE_ID
           || dev->descriptor.idProduct == CJECOM_USB_DEVICE_ID)) {
        found++;
        if (found == num)
          return dev;
      }
    }
  }
  return NULL;
}



static struct usb_device *find_rsct_usbdev_by_name(const char *devName) {
  struct usb_bus *busses, *bus;
  struct usb_device *dev;
  char filename[PATH_MAX+1];
  int nlen;
  unsigned int device_vendor, device_product;
  const char *devDir;
  const char *devFile;
  char namebuf[256];
  int i;

  /* get name of libusb device */
  if (strncmp("usb:", devName, 4) != 0) {
    DEBUGP("Device name does not start with \"usb:\"\n");
    return 0;
  }

  if (sscanf(devName, "usb:%x/%x", &device_vendor, &device_product) != 2){
    DEBUGP("Unable to parse device name (1)\n");
    return 0;
  }

  devDir=strstr(devName, "libusb:");
  if (devDir==NULL) {
    DEBUGP("Unable to parse device name (2)\n");
    return 0;
  }
  devDir+=7;

  devFile=strchr(devDir, ':');
  if (devFile==NULL) {
    DEBUGP("Unable to parse device name (3)\n");
    return 0;
  }
  i=devFile-devDir;
  if (i>=sizeof(namebuf)) {
    DEBUGP("Device path too long (1)\n");
    return 0;
  }
  devFile++;

  strncpy(namebuf, devDir, i);
  namebuf[i]=0;
  i=strlen(devFile);
  if ((strlen(namebuf)+i+2)>=sizeof(namebuf)) {
    DEBUGP("Device path too long (2)\n");
    return 0;
  }
  strcat(namebuf, "/");
  strcat(namebuf, devFile);

  /* now we got the bus/device name */
  filename[PATH_MAX]=0;


  if (!usb_read) {
    ausb_init();
    usb_read = 1;
  }

  nlen=strlen(namebuf);

  usb_find_busses();
  usb_find_devices();
  busses = usb_get_busses();

  for (bus = busses; bus; bus = bus->next) {
    for (dev = bus->devices; dev; dev = dev->next) {
      int flen;

      strncpy( filename, bus->dirname, PATH_MAX );
      strncat( filename, "/", PATH_MAX );
      strncat( filename, dev->filename, PATH_MAX );
      flen=strlen(filename);
      if (flen>=nlen) {
	if (strncmp(filename+(flen-nlen), namebuf, nlen)==0)
	  return dev;
      }
    }
  }
  return NULL;
}




/* callback functions for ppa */
static void CJPP_CALLBACK_TYPE cjppCallbackKey(CCID_CTX ctx, 
                                               unsigned char status) {
  u_int16_t pn = (u_int16_t) (u_int32_t) ctx;
  int rv;

  cjctapi_log("%s called for port %u\n", __FUNCTION__, pn);
  rv=ctn_list_invoke_keycb(pn);
  cjctapi_log("callback returned %d\n", rv);
  if (rv!=1) {
    if (beepstruct)
      beep_whatever(beepstruct);
    else {
      cjctapi_log("no beep struct\n");
    }
  }
}

#if 0
static void CJPP_CALLBACK_TYPE cjppCallbackStatus(CCID_CTX ctx, 
						   unsigned char status){
  u_int16_t pn = (u_int16_t) (u_int32_t) ctx;

  cjctapi_log("%s called for port %u\n", __FUNCTION__, pn);
}
#endif



/* callback function for e-com */
static void cjecomCallbackKey(struct cj_info *dev){
  u_int16_t pn = ctn_list_lookup_bydev(dev);
  int rv;

  cjctapi_log("%s called for port %u\n", __FUNCTION__, pn);
  rv=ctn_list_invoke_keycb(pn);
  if (rv!=1) {
    if (beepstruct)
      beep_whatever(beepstruct);
    else {
      cjctapi_log("no beep struct\n");
    }
  }
}



static int init() {
  /* init CTAPI configuration */
  if (_ctapi_init_count==0) {
    if (config_init()) {
      cjctapi_log("not enough memory available\n");
      return -1;
    }
    /* only initialize beep_struct once */
    if (!(config_get_flags() & CT_FLAGS_NO_BEEP)) {
      cjctapi_log("beep init\n");
      beepstruct=beep_init();
    }
  }
  _ctapi_init_count++;

  return 0;
}



static void fini() {
  /* fini CTAPI configuration */
  if (_ctapi_init_count>0) {
    _ctapi_init_count--;
    if (_ctapi_init_count==0) {
      if (beepstruct) {
        beep_fini(beepstruct);
        beepstruct=0;
      }
      config_fini();
    }
  }
}



CYBERJACK_EXPORT IS8 rsct_init_name(IU16 ctn, const char *devName) {
  int ret = CT_API_RV_OK;
  int type = -1;
  void *dev;
  struct usb_device *udev;
  char filename[PATH_MAX+1];

  DEBUGP("Init ctn=%d, name=\"%s\"\n", ctn, devName);
  filename[PATH_MAX]=0;

  if (init()) {
    fprintf(stderr, "Could not init CTAPI driver\n");
    return CT_API_RV_ERR_MEMORY;
  }

  if (ctn_list_lookup(ctn, &dev) > 0) {
    cjctapi_log("ctn already exists, cannot reuse\n");
    fini();
    return CT_API_RV_ERR_INVALID;
  }

  /* Normal case: iterate over existing usb devices */
  udev = find_rsct_usbdev_by_name(devName);
  if (!udev) {
    cjctapi_log("no matching usb device found\n");
    fini();
    return CT_API_RV_ERR_INVALID;
  }
  else {
    strncpy( filename, udev->bus->dirname, PATH_MAX );
    strncat( filename, "/", PATH_MAX );
    strncat( filename, udev->filename, PATH_MAX );
  }

  switch (udev->descriptor.idProduct) {
  case CJPPA_USB_DEVICE_ID:
    cjctapi_log("detected pinpad_a at %s\n",
                filename);
    type = CJ_TYPE_PPA;
    /*dev = ctapiInit(filename, (void *)pn,
                    NULL, &cjppCallbackKey);*/
    dev = ctapiInit(filename, (void *)ctn,
                    NULL, &cjppCallbackKey);
    cjctapi_log("ctapiInit returned %p\n", dev);
    if (!dev)
      ret = CT_API_RV_ERR_HOST;
    break;

  case CJECOM_USB_DEVICE_ID:
    if (config_get_flags() & CT_FLAGS_ECOM_KERNEL) {
      cjctapi_log("detected e-com/pp at %s\n", devName);
      type = CJ_TYPE_ECOMPP;
      ret = cjecom_CT_initKernel(devName, (struct cj_info **)&dev);
    }
    else {
      cjctapi_log("detected e-com/pp at %s\n", devName);
      type = CJ_TYPE_ECOMPP_USER;
      ret = cjecom_CT_initUser(udev, (struct cj_info **)&dev);
    }
    cjctapi_log("cjecom_CT_init returned %d\n", ret);
    if (!dev || ret < 0)
      cjctapi_log("no device returned by "
                  "cjecom_CT_init\n");
    else
      ret=cjecom_CT_keycb(dev, &cjecomCallbackKey);
    break;
  default:
    cjctapi_log("unknown Device ID 0x%x found\n",
                udev->descriptor.idProduct);
    ret = CT_API_RV_ERR_INVALID;
    break;
  }

  if (ret == 0) {
    if (ctn_list_add(ctn, dev, type) < 0) {
      switch (type) {
      case CJ_TYPE_PPA:
	ctapiClose(dev);
	break;
      case CJ_TYPE_ECOMPP:
      case CJ_TYPE_ECOMPP_USER:
	cjecom_CT_close(dev);
	break;
      }
      cjctapi_log("unable to add ctn %u to list\n", ctn);
      fini();
      return CT_API_RV_ERR_HOST;
    }
  }

  if (ret!=0)
    fini();

  return ret;

}



CYBERJACK_EXPORT IS8 CT_init(IU16 ctn, IU16 pn){
  int ret = CT_API_RV_OK;
  int type = -1;
  void *dev;
  struct usb_device *udev;
  struct usb_device fake_ecom_udev;
  char filename[PATH_MAX+1];

  cjctapi_enter();
  if (init()) {
    fprintf(stderr, "Could not init CTAPI driver\n");
    return CT_API_RV_ERR_MEMORY;
  }

  cjctapi_log("Called CT_init() with ctn=%d, pn=%d)\n", ctn, pn);

  if (ctn_list_lookup(ctn, &dev) > 0) {
    cjctapi_log("ctn already exists, cannot reuse\n");
    fini();
    return CT_API_RV_ERR_INVALID;
  }

  if (pn & 0x8000) {
    /* Special case for direct opening of ttyUSB* by libchipcard */
    cjctapi_log("Using fake_ecom_udev ttyUSB hack\n");
    pn = pn & ~0x8000;
    memset(&fake_ecom_udev, 0, sizeof(fake_ecom_udev));
    fake_ecom_udev.next = fake_ecom_udev.prev = &fake_ecom_udev;
    fake_ecom_udev.descriptor.idVendor = CJPPA_USB_VENDOR_ID;
    fake_ecom_udev.descriptor.idProduct = CJECOM_USB_DEVICE_ID;
    strncpy(fake_ecom_udev.filename, "/dev/fake_ecom_udev",
	    PATH_MAX);
    udev = &fake_ecom_udev;
  }
  else {
    /* Normal case: iterate over existing usb devices */
    udev = find_rsct_usbdev(pn);
    if (!udev) {
      cjctapi_log("no matching usb device found\n");
      fini();
      return CT_API_RV_ERR_INVALID;
    }
    else {
      strncpy( filename, udev->bus->dirname, PATH_MAX );
      strncat( filename, "/", PATH_MAX );
      strncat( filename, udev->filename, PATH_MAX );
    }
  }

  switch (udev->descriptor.idProduct) {
  case CJPPA_USB_DEVICE_ID:
    cjctapi_log("detected pinpad_a at %s\n",
                filename);
    type = CJ_TYPE_PPA;
    /*dev = ctapiInit(filename, (void *)pn,
                    NULL, &cjppCallbackKey);*/
    dev = ctapiInit(filename, (void *)ctn,
                    NULL, &cjppCallbackKey);
    cjctapi_log("ctapiInit returned %p\n", dev);
    if (!dev)
      ret = CT_API_RV_ERR_HOST;
    break;

  case CJECOM_USB_DEVICE_ID:
    if (config_get_flags() & CT_FLAGS_ECOM_KERNEL) {
      snprintf(filename, PATH_MAX, "/dev/ttyUSB%u", pn-1);
      cjctapi_log("detected e-com/pp at %s, assuming %s\n",
                  filename, filename);
      type = CJ_TYPE_ECOMPP;
      ret = cjecom_CT_initKernel(filename, (struct cj_info **)&dev);
    }
    else {
      cjctapi_log("detected e-com/pp at %s\n", filename);
      type = CJ_TYPE_ECOMPP_USER;
      ret = cjecom_CT_initUser(udev, (struct cj_info **)&dev);
    }
    cjctapi_log("cjecom_CT_init returned %d\n", ret);
    if (!dev || ret < 0)
      cjctapi_log("no device returned by "
                  "cjecom_CT_init\n");
    else
      ret=cjecom_CT_keycb(dev, &cjecomCallbackKey);
    break;
  default:
    cjctapi_log("unknown Device ID 0x%x found\n",
                udev->descriptor.idProduct);
    ret = CT_API_RV_ERR_INVALID;
    break;
  }

  if (ret == 0) {
    if (ctn_list_add(ctn, dev, type) < 0) {
      switch (type) {
      case CJ_TYPE_PPA:
	ctapiClose(dev);
	break;
      case CJ_TYPE_ECOMPP:
      case CJ_TYPE_ECOMPP_USER:
	cjecom_CT_close(dev);
	break;
      }
      cjctapi_log("unable to add ctn %u to list\n", ctn);
      fini();
      return CT_API_RV_ERR_HOST;
    }
  }

  if (ret!=0)
    fini();

  return ret;
}



static IS8 CT__data(IU16 ctn, IU8 *dad, IU8 *sad,
                    IU16 lenc, IU8 *command,
                    IU16 *lenr, IU8 *response){
  int ret;
  void *dev;
  int type = ctn_list_lookup(ctn, &dev);

  if (lenc<4) {
    cjctapi_log("CT_data: Too few bytes for APDU\n");
    return CT_API_RV_ERR_INVALID;
  }

  if (*lenr<2) {
    cjctapi_log("CT_data: Too few bytes for response\n");
    return CT_API_RV_ERR_INVALID;
  }

  /* filter out command INPUT */
  if (command[0]==0x20 &&
      command[1]==0x16 &&
      !(config_get_flags() & CT_FLAGS_ALLOW_INPUT)) {
    cjctapi_log("CT_data: INPUT command not allowed\n");
    response[0]=0x69;
    response[1]=0x00;
    *lenr=2;
    return CT_API_RV_OK;
  }

  switch (type) {
  case CJ_TYPE_ECOMPP:
  case CJ_TYPE_ECOMPP_USER:
    ret = cjecom_CT_data(dev, dad, sad, lenc, command, lenr,
			 response);
    break;
  case CJ_TYPE_PPA:
    ret = ctapiData(dev, dad, sad, lenc, command, lenr, response);
    break;
  default:
    ret = CT_API_RV_ERR_INVALID;
    break;
  }

  return ret;
}




CYBERJACK_EXPORT IS8 CT_data(IU16 ctn, IU8 *dad, IU8 *sad,
                             IU16 lenc, IU8 *command,
                             IU16 *lenr, IU8 *response){
  IS8 ret;

  cjctapi_log("Sending: CTN=%04x, SAD=%02x, DAD=%02x\n",
              ctn, *sad, *dad);
  rsct_log_bytes(CT_FLAGS_DEBUG_CTAPI,
		 __FILE__, __LINE__, __FUNCTION__,
		 "Sending", lenc, command);
  ret=CT__data(ctn, dad, sad, lenc, command, lenr, response);
  rsct_log(CT_FLAGS_DEBUG_CTAPI,
	   __FILE__, __LINE__, __FUNCTION__,
	   "Response: CTN=%04x, SAD=%02x, DAD=%02x, res=%d\n",
	   ctn, *sad, *dad, ret);
  if (ret==0) {
    if (*lenr) {
      rsct_log_bytes(CT_FLAGS_DEBUG_CTAPI,
		     __FILE__, __LINE__, __FUNCTION__,
		     "Response:", *lenr, response);
    }
  }
  return ret;
}


CYBERJACK_EXPORT IS8 CT_close(IU16 ctn){
  int ret;
  void *dev;
  int type = ctn_list_lookup(ctn, &dev);

  IU8 dad=CT_API_AD_CT, sad=CT_API_AD_HOST, response[2];
  IU16 lenr=sizeof(response);

  if (type < 0)
    return CT_API_RV_ERR_INVALID;

  cjctapi_log("Closing CTN %04x\n", ctn);

  /* EJECT ICC */
  CT_data(ctn, &dad, &sad, 4, (unsigned char *) "\x20\x15\x01\x07",
          &lenr, response);

  switch(type) {
  case CJ_TYPE_ECOMPP:
  case CJ_TYPE_ECOMPP_USER:
    ret = cjecom_CT_close(dev);
    break;
  case CJ_TYPE_PPA:
    ret = ctapiClose(dev);
    break;
  default:
    ret = CT_API_RV_ERR_INVALID;
    break;
  }
  ctn_list_del(ctn);

  cjctapi_log("CTN %04x closed\n", ctn);

  fini();
  return ret;
}


/* Proprietary extension */
CYBERJACK_EXPORT IS8 rsct_setkeycb(IU16 ctn, CT_KEY_CB cb, void *user_data){
  if (ctn_list_set_keycb(ctn, cb, user_data) < 0)
    return CT_API_RV_ERR_INVALID;

  return CT_API_RV_OK;
}



CYBERJACK_EXPORT void rsct_version(IU8 *vmajor,
				   IU8 *vminor,
				   IU8 *vpatchlevel,
				   IU16 *vbuild){
  if (vmajor)
    *vmajor=CYBERJACK_VERSION_MAJOR;
  if (vminor)
    *vminor=CYBERJACK_VERSION_MINOR;
  if (vpatchlevel)
    *vpatchlevel=CYBERJACK_VERSION_PATCHLEVEL;
  if (vbuild)
    *vbuild=CYBERJACK_VERSION_BUILD;
}





