/* Wrapper/Extension code to libusb-0.1 to support asynchronous requests
 * on Linux platforns 
 *
 * (C) 2004-2005 by Harald Welte <laforge@gnumonks.org>
 *
 * Distributed and licensed under the terms of GNU LGPL, Version 2.1
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <sys/utsname.h>
#include <sys/ioctl.h>

#include "ausb_l.h"
#include "usb_l.h"
#include "ctapi_config_l.h"
#include "cyberjack_l.h"

#define DEBUGP(format, args...) \
  rsct_log(CT_FLAGS_DEBUG_AUSB, __FILE__, __LINE__, __FUNCTION__, format, ## args)

#define ausb_log_bytes(tag, length, value) \
  rsct_log_bytes(CT_FLAGS_DEBUG_AUSB, __FILE__, __LINE__, __FUNCTION__, \
                 tag, length, value);

#define SIGAUSB		(SIGRTMIN+1)

static int ausb_get_fd(ausb_dev_handle *ah)
{
	return *((int *)ah->uh);
}


static int kernel_2_5;

static struct ausb_dev_handle *handle_array[CT_MAX_DEVICES];
static int ausb_was_init=0;

static int is_kernel_2_5(){
  struct utsname uts;
  int ret;

  uname(&uts);

  ret = (strncmp(uts.release, "2.5.", 4) == 0) ||
    (strncmp(uts.release, "2.6.", 4) == 0);

  return ret;
}



void ausb_dump_urb(struct usbdevfs_urb *uurb){
  if (uurb->buffer && uurb->actual_length) {
    DEBUGP("urb(%p): type=%u, endpoint=0x%x, status=%d, flags=0x%x, number_of_packets=%d, error_count=%d\n",
           uurb, uurb->type, uurb->endpoint, uurb->status, uurb->flags, uurb->number_of_packets, uurb->error_count);
    ausb_log_bytes("Data:", uurb->actual_length, uurb->buffer);
  }
  else {
    DEBUGP("urb(%p): type=%u, endpoint=0x%x, status=%d, flags=0x%x, number_of_packets=%d, error_count=%d [no data]\n",
           uurb, uurb->type, uurb->endpoint, uurb->status, uurb->flags, uurb->number_of_packets, uurb->error_count);
  }
}



void ausb_fill_int_urb(struct usbdevfs_urb *uurb, unsigned char endpoint,
		      void *buffer, int buffer_length){
  uurb->type=USBDEVFS_URB_TYPE_INTERRUPT;
  uurb->endpoint = endpoint; /* | USB_DIR_IN; */
  //	uurb->endpoint = endpoint | USB_ENDPOINT_IN;
  uurb->flags = 0;
  uurb->buffer = buffer;
  uurb->buffer_length = buffer_length;
  uurb->actual_length=0;
  uurb->signr = SIGAUSB;
  uurb->start_frame = -1;
}



void ausb_fill_bulk_urb(struct usbdevfs_urb *uurb, unsigned char endpoint,
                        void *buffer, int buffer_length){
  uurb->type=USBDEVFS_URB_TYPE_BULK;
  uurb->endpoint = endpoint; /* | USB_DIR_IN; */
  //	uurb->endpoint = endpoint | USB_ENDPOINT_IN;
  uurb->flags = kernel_2_5 ? 0 : 1 ; /* USBDEVFS_URB_QUEUE_BULK; */
  uurb->buffer = buffer;
  uurb->buffer_length = buffer_length;
  uurb->actual_length=0;
  uurb->signr = 0; //SIGAUSB;
  uurb->start_frame = -1;
}



int ausb_submit_urb(ausb_dev_handle *ah, struct usbdevfs_urb *uurb){
  int ret;

  DEBUGP("ah=%p\n", ah);
  uurb->actual_length=0;
  ausb_dump_urb(uurb);

  /* save ausb_dev_handle in opaque usercontext field */
  uurb->usercontext = ah;

  do {
    ret = ioctl(ausb_get_fd(ah), USBDEVFS_SUBMITURB, uurb);
  } while (ret < 0 && errno == EINTR);

  return ret;
}



int ausb_discard_urb(ausb_dev_handle *ah, struct usbdevfs_urb *uurb){
  int ret;

  DEBUGP("ah=%p, uurb=%p\n");
  ausb_dump_urb(uurb);

  do {
    ret = ioctl(ausb_get_fd(ah), USBDEVFS_DISCARDURB, uurb);
  } while (ret < 0 && errno == EINTR);

  return ret;
}



struct usbdevfs_urb *ausb_get_urb(ausb_dev_handle *ah){
  int ret;
  struct usbdevfs_urb *uurb;

  DEBUGP("entering\n");

  do {
    uurb=NULL;
    ret = ioctl(ausb_get_fd(ah), USBDEVFS_REAPURBNDELAY, &uurb);
  } while (ret < 0 && errno == EINTR);

  if (ret < 0 && errno == EAGAIN) {
    DEBUGP("ioctl returned %d (errno=%d:%s)\n", ret,
           errno, strerror(errno));
    return NULL;
  }

  DEBUGP("returning %p\n", uurb);
  return uurb;
}



static void handle_urb(struct usbdevfs_urb *uurb){
  struct ausb_dev_handle *ah = uurb->usercontext;

  DEBUGP("called, ah=%p\n", ah);
  ausb_dump_urb(uurb);

  if (uurb->type >= AUSB_USBDEVFS_URB_TYPES) {
    DEBUGP("unknown uurb type %u\n", uurb->type);
    return;
  }

  if (!ah) {
    DEBUGP("cant't call handler because missing ah ptr\n");
    return;
  }

  if (!ah->cb[uurb->type].handler) {
    DEBUGP("received URB type %u, but no handler\n", uurb->type);
    return;
  }
  ah->cb[uurb->type].handler(uurb, ah->cb[uurb->type].userdata);
}



/* This is the signal handler for asynchronous USB request signals.
 * Interrrupt URBs are sent by the reader whenever a key has been
 * pressed. However, we only accept keypress events while the reader
 * is executing a command, in which case this driver is in the function
 * ausb_bulk_read(). Only there keypress events are handled.
 * However, the reader does not make this determination: It always sends
 * interrupt urb's whether the driver expects them or not. So we need
 * to reap all interrupt urbs here and simply re-submit them (thus
 * effectively pulling all pending interrupt urbs).
 *
 */
static void sigact_rtmin(int sig, siginfo_t *siginfo, void *v){
  int idx;

  if (sig != SIGAUSB)
    return;

  //DEBUGP("errno=%d, si_addr=%p\n", siginfo->errno, siginfo->si_addr);

  for (idx=0; idx<CT_MAX_DEVICES; idx++) {
    ausb_dev_handle *ah;

    ah=handle_array[idx];
    if (ah && !(ah->inBulkRead)) {
      int count;

      for (count = 0; ; count++) {
        struct usbdevfs_urb *urb;

        /* this handle is not reading bulk, so we have to dismiss any
         * URBs and re-submit them.
         */
        urb=ausb_get_urb(ah);
        if (urb) {
          /* simply resubmit interrupt */
          DEBUGP("got an unexpected interrupt urb, discarding\n");
          ausb_dump_urb(urb);
          if (ausb_submit_urb(ah, urb))
            DEBUGP("unable to resubmit interrupt urb\n");
        }
        else
          /* no more urbs waiting */
          break;
      }
    } /* if not in bulk read */
  } /* for every handle */
}



int ausb_init(void){
  struct sigaction act;
  int i;

  DEBUGP("entering\n");

  memset(&act, 0, sizeof(act));
  act.sa_sigaction = sigact_rtmin;
  sigemptyset(&act.sa_mask);
  act.sa_flags = SA_SIGINFO;

  sigaction(SIGAUSB, &act, NULL);

  kernel_2_5 = is_kernel_2_5();

  DEBUGP("kernel 2.5+ = %d\n", kernel_2_5);

  /* clear handle array */
  for (i=0; i<CT_MAX_DEVICES; i++)
    handle_array[i]=0;
  ausb_was_init=1;

  usb_init();
  usb_find_busses();
  usb_find_devices();
  return 1;
}



ausb_dev_handle *ausb_open(struct usb_device *dev){
  int idx=-1;
  ausb_dev_handle *dh=0;

  DEBUGP("entering, dh=%p\n", dh);
  for (idx=0; idx<CT_MAX_DEVICES; idx++)
    if (handle_array[idx]==0)
      break;
  if (idx>=CT_MAX_DEVICES) {
    DEBUGP("too many open devices\n");
    return 0;
  }

  dh=malloc(sizeof *dh);
  if (dh==0) {
    DEBUGP("memory full\n");
    return 0;
  }
  memset(dh, 0, sizeof(*dh));

  dh->uh=usb_open(dev);
  if (!dh->uh) {
    DEBUGP("usb_open() failed\n");
    free(dh);
    return NULL;
  }

  handle_array[idx]=dh;
  DEBUGP("last handle = %p\n", dh);

  return dh;
}



int ausb_register_callback(ausb_dev_handle *ah, unsigned char type,
                           void (*callback)(struct usbdevfs_urb *uurb,
                                            void *userdata),
                           void *userdata){
  DEBUGP("registering callback for type %u:%p\n", type, callback);

  if (type >= AUSB_USBDEVFS_URB_TYPES) {
    errno = EINVAL;
    return -1;
  }

  if (!kernel_2_5 && type == USBDEVFS_URB_TYPE_INTERRUPT)
    type = USBDEVFS_URB_TYPE_BULK;

  ah->cb[type].handler = callback;
  ah->cb[type].userdata = userdata;

  return 0;
}



int ausb_claim_interface(ausb_dev_handle *ah, int interface){
  DEBUGP("entering\n");
  return usb_claim_interface(ah->uh, interface);
}



int ausb_release_interface(ausb_dev_handle *ah, int interface){
  DEBUGP("entering\n");
  if (ausb_was_init<1) {
    DEBUGP("ausb subsystem has not been initialized!");
    return -1;
  }
  return usb_release_interface(ah->uh, interface);
}



int ausb_set_configuration(ausb_dev_handle *ah, int configuration){
  int ret;

  DEBUGP("entering\n");
  ret=usb_set_configuration(ah->uh, configuration);
  if (ret<0) {
    DEBUGP("usb_set_configuration returned %d (errno=%d:%s)\n", ret,
           errno, strerror(errno));
  }
  return ret;
}



int ausb_bulk_write(ausb_dev_handle *ah, int ep, char *bytes, int size, int timeout){
  DEBUGP("entering (ah=%p, ep=0x%x, bytes=%p, size=%d, timeout=%d\n",
         ah, ep, bytes, size, timeout);
  return __usb_bulk_write(ah->uh, ep, bytes, size, timeout);
}



void ausb_start_accept_int(ausb_dev_handle *ah) {
  ah->inBulkRead++;
}



void ausb_end_accept_int(ausb_dev_handle *ah) {
  if (ah->inBulkRead>0)
    ah->inBulkRead--;
}



int ausb_bulk_read(ausb_dev_handle *ah, int ep, char *bytes, int size, int timeout){
  int ret;
  struct usbdevfs_urb *uurb;
  struct usbdevfs_urb *rurb;

  DEBUGP("entering (ah=%p, ep=0x%x, bytes=%p, size=%d, timeout=%d\n",
         ah, ep, bytes, size, timeout);

  uurb=malloc(sizeof(*uurb));
  if (!uurb) {
    DEBUGP("unable to allocate urb\n");
    return -1;
  }

  ausb_fill_bulk_urb(uurb, ep, bytes, size);
  if (ausb_submit_urb(ah, uurb)) {
    DEBUGP("unable to submit interrupt urb\n");
    return -1;
  }

  for (;;) {
    do {
      rurb=NULL;
      //ret = ioctl(ausb_get_fd(ah), USBDEVFS_REAPURBNDELAY, &rurb);
      DEBUGP("sending ioctl...\n");
      ret = ioctl(ausb_get_fd(ah), USBDEVFS_REAPURB, &rurb);
    } while (ret < 0 && (errno == EINTR || errno==EAGAIN));

    if (ret<0) {
      DEBUGP("ioctl returned %d (errno=%d:%s)\n", ret,
             errno, strerror(errno));
      ausb_discard_urb(ah, uurb);
      return -1;
    }

    DEBUGP("ioctl returned urb %p\n", rurb);
    if (rurb==ah->intUrb) {
      /* handle interrupt urb */
      DEBUGP("answer for interrupt urb\n");
      handle_urb(rurb);
    }
    else if (rurb==uurb) {
      DEBUGP("answer for bulk urb\n");
      ret=rurb->actual_length;
      free(uurb);
      return ret;
    }
    else {
      DEBUGP("returned unknown uurb %p\n", rurb);
      ausb_dump_urb(uurb);
      ausb_discard_urb(ah, uurb);
      return -1;
    }
  }

}



int ausb_clear_halt(ausb_dev_handle *ah, unsigned int ep){
  DEBUGP("entering (ah=%p, ep=0x%x)\n", ah, ep);
  return usb_clear_halt(ah->uh, ep);
}



int ausb_reset(ausb_dev_handle *ah){
  DEBUGP("entering (ah=%p)\n", ah);
  return usb_reset(ah->uh);
}



int ausb_resetep(ausb_dev_handle *ah, int ep){
  DEBUGP("entering (ah=%pm ep=0x%x)\n", ah, ep);
  return usb_resetep(ah->uh, ep);
}

#ifdef LIBUSB_HAS_GET_DRIVER_NP
int ausb_get_driver_np(ausb_dev_handle *ah, int interface, char *name,
		       unsigned int namelen)
{
	DEBUGP("entering\n");
	return usb_get_driver_np(ah->uh, interface, name, namelen);
}
#endif

#ifdef LIBUSB_HAS_DETACH_KERNEL_DRIVER_NP
int ausb_detach_kernel_driver_np(ausb_dev_handle *ah, int interface){
  DEBUGP("entering\n");
  return usb_detach_kernel_driver_np(ah->uh, interface);
}



int ausb_reattach_kernel_driver_np(ausb_dev_handle *ah, int interface){
  DEBUGP("entering\n");
  return __usb_reattach_kernel_driver_np(ah->uh, interface);
}
#endif



int ausb_close(struct ausb_dev_handle *ah){
  int idx;
  int rv;

  DEBUGP("entering\n");
  for (idx=0; idx<CT_MAX_DEVICES; idx++) {
    if (handle_array[idx]==ah)
      handle_array[idx]=0;
  }

  rv=usb_close(ah->uh);
  free(ah);
  return rv;
}



void ausb_fini(void){
  DEBUGP("entering\n");
  sigaction(SIGAUSB, NULL, NULL);
}



