/* 
 * Copyright (C) 2002-2004 Benoit Poulot-Cazajous
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *   
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *   
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define _POSIX_C_SOURCE 199506L
#define _XOPEN_SOURCE 500
#define _GNU_SOURCE

#include <sys/types.h>
#include <sys/param.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/time.h>
#include <sys/select.h>
#include <alloca.h>
#include <signal.h>
#include <limits.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <poll.h>
#include <dlfcn.h>

#include "cachecc1.h"

/*
 * Trampolines
 */
int real_execv(const char *name, char *const argv[])
{
    static int (*real_fun)() = 0;
    if (!real_fun) real_fun = (int(*)())dlsym(RTLD_NEXT, "execv");
    return real_fun(name, argv);
}

int real_execve(const char *name, char *const argv[], char *const envp[])
{
    static int (*real_fun)() = 0;
#ifdef __linux
    char *execve_name = "execve";
#endif
#ifdef __sun
    char *execve_name = "_execve";
#endif
    if (!real_fun) real_fun = (int(*)())dlsym(RTLD_NEXT, execve_name);
    return real_fun(name, argv, envp);
}

static int follow_allexec(const char *name0, char *const argv0[], char *const envp0[], int envsize)
{
    int i, j, l;
    char *name = (char*)name0;
    char **argv = (char**)argv0;
    char **envp = (char**)envp0;
    
    l = strlen(name);
        /* backends */
    do {
        if (l < 7) break;
        if (strcmp(&name[l-4], "/cc1") &&
            strcmp(&name[l-3], "/as") &&
            strcmp(&name[l-8], "/cc1plus")) break;
        if (envsize == 0) envp = 0;
        return exec_cachecc1(name, argv, envp);
    } while (0);

        /* frontends */
    do {
        struct stat st;
        char **_argv;
        int nointegratedcpp;

        if (l < 7) break;
        if (strcmp(&name[l-3], "gcc") &&
#ifdef linux
            strcmp(&name[l-7], "/bin/cc") &&
#endif
            strcmp(&name[l-3], "g++")) break;
        if (stat(name, &st)) break;

        envp = read_env(envp);
        if (!env_dir) break;

            /* copy argv */
        for (i = 0; argv[i]; i++) continue;
        _argv = alloca((i+2)*sizeof(char*));
        
            /*
             * strip -pipe, because temporary files are faster (fewer
             * context switches) and easier for cachecc1 to handle
             * check if -no-integrated-cpp is used
             */
        nointegratedcpp = 0;
        for (i = j = 0; argv[i]; ++i) {
            if (!strcmp(argv[i], "-pipe")) continue;
            if (!strcmp(argv[i], "-no-integrated-cpp")) nointegratedcpp++;
            _argv[j++] = argv[i];
        }
            /*
             * try to turn off the integrated cpp on gcc-3.x
             * we don't have enough information here.
             * We will later, when the backend is called
             */
        if (!nointegratedcpp) {
            static char ifgcc3[PATH_MAX];
            char *fname;
            strcpy(ifgcc3, "CACHECC1_ifgcc3=");
            fname = &ifgcc3[strlen(ifgcc3)];
            sprintf(fname, "%s/gcc3_%lx_%lx_%lx_%lx",
                    env_dir,
                    st.st_mtime, st.st_size, st.st_ino, (long)st.st_dev);
            if (!access(fname, F_OK)) {
                    /* gcc-3.x detected */
                _argv[j++] = "-no-integrated-cpp";
            } else {
                    /* gcc-2.x assumed. ask cachecc1 */
                char **_envp;
                for (i = 0; envp[i]; i++) continue;
                _envp = alloca((i+2)*sizeof(char*));
                for (i = 0; envp[i]; i++) _envp[i] = envp[i];
                _envp[i++] = ifgcc3;
                _envp[i] = 0;
                envp = _envp;
                envsize = i;
            }
        }
        _argv[j] = 0;
        argv = _argv;
    } while (0);
    if (envsize == 0) {
        return real_execv(name, argv);
    } else {
        return real_execve(name, argv, envp);
    }
}

static USED int follow_execve(const char *name, char *const argv[], char *const envp[])
{
    int i = 0;
    while (envp[i]) i++;
    return follow_allexec(name, argv, envp, i);
}

static USED int follow_execv(const char *name, char *const argv[])
{
    return follow_allexec(name, argv, 0, 0);
}

/*
 * Alias definitions
 * Inlined assembler can be more portable than C.
 */
static USED void __followcc1_alias_execve()
{
    asm("\t .global execve");
    asm("\t .global _execve");
    asm("\t .global __execve");
    asm("\t execve = follow_execve");
    asm("\t _execve = follow_execve");
    asm("\t __execve = follow_execve");
 #ifdef __linux__
    asm("\t .global execv");
    asm("\t .global _execv");
    asm("\t .global __execv");
    asm("\t execv = follow_execv");
    asm("\t _execv = follow_execv");
    asm("\t __execv = follow_execv");
#endif
}

/*
 * On Linux, execvp() does not call execve() from the libc, but
 * directly calls the kernel.
 * Not necessary on Solaris.
 */
#ifdef __linux

extern char **__environ;

static int real_execvp(const char *name, char *const argv[])
{
    static int (*real_fun)() = 0;
    if (!real_fun) real_fun = (int(*)())dlsym(RTLD_NEXT, "execvp");
    return real_fun(name, argv);
}

static USED int follow_execvp(char *file, char *argv[])
{
    char *path, *p, *name;
    int len, pathlen, doit;
    if (!file) goto direct;
    if (!file[0]) goto direct;
    len = strlen(file);
    doit = 0;
    doit = doit || (len >= 2 && !strcmp(&file[len-2], "as"));
    doit = doit || (len >= 3 && !strcmp(&file[len-3], "gcc"));
    doit = doit || (len >= 3 && !strcmp(&file[len-3], "g++"));
    if (!doit) goto direct;
    if (strchr(file, '/')) {
        follow_execve(file, argv, __environ);
        goto direct;
    }
    path = getenv("PATH");
    if (!path) goto direct;
    pathlen = strlen(path);
    name = alloca(pathlen+1+len+1);
    name = (char*)memcpy(name+pathlen+1, file, len+1);
    *--name = '/';
    p = path;
    do {
        char *file;
        path = p;
        while (*p && *p != ':') p++;
        if (p == path) {
            file = name+1;
        } else {
            file = (char *) memcpy(name-(p-path), path, p-path);
        }
        if (!access(file, X_OK)) {
            follow_execve(file, argv, __environ);
            goto direct;
        }
    } while (*p++ != '\0');
  direct:
    return real_execvp(file, argv);
}

static USED void __followcc1_alias_execvp()
{
    asm("\t .global execvp");
    asm("\t .global _execvp");
    asm("\t .global __execvp");
    asm("\t execvp = follow_execvp");
    asm("\t _execvp = follow_execvp");
    asm("\t __execvp = follow_execvp");
}
#endif

/* arch-tag: ddd5a5d9-3e47-47b3-8bb6-1b22713ad4e4 */
