/* archive-version.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "commands/cmd.h"
#include "commands/archive-version.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [archive]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit."))


t_uchar arch_cmd_archive_version_help[] = N_(
        "list the archive-version in an archive\n"
        "Print the archive format identifier string of an arch archive.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_archive_version (t_uchar *program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_archive_version_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, _("unhandled option `%s'\n"), option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, _("ill-formed argument for `%s' (`%s')\n"), option->opt_string, option->arg_string);
          goto usage_error;

        }
    }

  if (argc < 1 || argc > 2)
    goto usage_error;

  {
    t_uchar * version;
    struct arch_archive * arch = 0;

    /* First, see if they gave us an archive. If so, we can use that
     * to look up meta-info
     */
    if ( argc == 2 )
      {
	arch = arch_archive_connect_branch (argv[1], NULL);
      }

    /* Nope. so lets try for a tree_root
     */
    if (! arch)
      {
        arch_project_tree_t * tree = arch_project_tree_new (talloc_context, ".");

        if (tree)
          {
            arch_project_tree_check_name (tree, &arch, NULL, "unused-category");
	    arch_project_tree_delete (tree);
          }
      }
    
    /* We dont have a tree_root either! So maybe the default archive?)
     */
    if (! arch)
      {
        t_uchar * default_archive = arch_my_default_archive (NULL);
        arch = arch_archive_connect_branch (default_archive, NULL);
        lim_free (0, default_archive);
      }
    
    if (!arch)
      {
        safe_printfmt (2, _("%s: unable to connect to archive\n"), argv[0]);
        exit(1);
      }

    version = arch_archive_version (arch);
    safe_printfmt (1, "%s\n", version);
    lim_free (0, version);
    
    arch_archive_close (arch);
  
  }

return 0;

}




/* tag: Tom Lord Thu Jun  5 16:30:06 2003 (archive-version.c)
 */
