/* -*- Mode: C++ -*-
 * Worldvisions Weaver Software:
 *   Copyright (C) 1997-2002 Net Integration Technologies, Inc.
 * 
 * UniConf low-level tree storage abstraction.
 */
#ifndef __UNICONFTREE_H
#define __UNICONFTREE_H

#include "uniconfkey.h"
#include "wvvector.h"
#include "wvcallback.h"
#include "unihashtree.h"

/**
 * A recursively composed dictionary for ordered tree-structured
 * data indexed by UniConfKey with logarithmic lookup time for
 * each 1-level search, linear insertion and removal.
 *
 * This container maintains a sorted vector of children which
 * greatly facilitates tree comparison and merging.  The
 * underlying collection guarantees fast lookups, but insertion
 * and removal may be quite slow.  If this becomes a problem,
 * then a different (more complex) data structure should be used,
 * such as a binary search tree.  However, for the moment, the
 * use of a vector keeps down memory footprint.
 *
 * Someday this could be further abstracted into a generic WvTreeDict.
 *
 * "Sub" is the name of the concrete subclass of UniConfTree
 */

template<class Sub, class Base = UniHashTreeBase>
class UniConfTree : public Base
{
   
public:
    typedef WvCallback<bool, const Sub *, const Sub *, void *> Comparator;

    /** Creates a node and links it to a subtree, if parent is non-NULL */
    UniConfTree(Sub *parent, const UniConfKey &key) :
        Base(parent, key)
        { }

    /** Destroy this node's contents and children. */
    ~UniConfTree()
        { zap(); }

    /** Returns a pointer to the parent node, or NULL if there is none. */
    Sub *parent() const
        { return static_cast<Sub*>(xparent); }

    /** Reparents this node. */
    void setparent(Sub *parent)
        { Base::_setparent(parent); }
    
    /** Returns a pointer to the root node of the tree. */
    Sub *root() const
        { return static_cast<Sub*>(Base::_root()); }
    
    /**
     * Returns full path of this node relative to an ancestor.
     * If ancestor is NULL, returns the root.
     */
    UniConfKey fullkey(const Sub *ancestor = NULL) const
        { return Base::_fullkey(ancestor); }

    /**
     * Finds the sub-node with the specified key.
     * If key.isempty(), returns this node.
     */
    Sub *find(const UniConfKey &key) const
        { return static_cast<Sub*>(Base::_find(key)); }
    
    /**
     * Finds the direct child node with the specified key.
     *
     * If key.numsegments() == 1, then performs the same task
     * as find(key), but a little faster.  Otherwise returns NULL.
     */
    Sub *findchild(const UniConfKey &key) const
        { return static_cast<Sub*>(Base::_findchild(key)); }

    /**
     * Removes the node for the specified key from the tree
     * and deletes it along with any of its children.
     *
     * If the key is UniConfKey::EMPTY, deletes this object.
     */
    void remove(const UniConfKey &key)
        { delete find(key); }
    
    /** Removes and deletes all children of this node. */
    void zap()
    {
        if (!xchildren)
            return;
        // set xchildren to NULL first so that the zap() will happen faster
        // otherwise, each child will attempt to unlink itself uselessly

        typename Base::Container *oldchildren = xchildren;
        xchildren = NULL;

        // delete all children
        typename Base::Container::Iter i(*oldchildren);
        for (i.rewind(); i.next();)
            delete static_cast<Sub*>(i.ptr());

        delete oldchildren;
    }

    /**
     * Compares this tree with another using the specified comparator
     * function.
     * Comparison of a subtree ends when the comparator returns false.
     * "comparator" is the value compare function
     * "userdata" is userdata for the compare function
     * Returns: true if the comparison function returned true each time
     */
    void compare(const Sub *other, const Comparator &comparator,
        void *userdata)
    {
        _recursivecompare(this, other, reinterpret_cast<
            const typename Base::BaseComparator&>(comparator), userdata);
    }

    /**
     * An iterator that walks over all elements on one level of a
     * UniConfTree.
     */
    class Iter : public Base::Iter
    {
    public:
        typedef typename Base::Iter MyBase;

        /** Creates an iterator over the specified tree. */
        Iter(Sub &tree) : Base::Iter(tree)
	    { }

        /** Returns a pointer to the current node. */
        Sub *ptr() const
            { return static_cast<Sub*>(MyBase::ptr()); }
        WvIterStuff(Sub);
    };
};


/** A plain UniConfTree that holds keys and values. */
class UniConfValueTree : public UniConfTree<UniConfValueTree>
{
    WvString xvalue;  /*!< the value of this entry */
    
public:
    UniConfValueTree(UniConfValueTree *parent,
        const UniConfKey &key, WvStringParm value) :
        UniConfTree<UniConfValueTree>(parent, key), xvalue(value)
    {
    }
    
    /** Returns the value field. */
    const WvString &value() const
        { return xvalue; }

    /** Sets the value field. */
    void setvalue(WvStringParm value)
        { xvalue = value; }
};


#endif // __UNICONFTREE_H
