/*
 * Relay calls helper routines
 *
 * Copyright 1993 Robert J. Amstadt
 * Copyright 1995 Martin von Loewis
 * Copyright 1995, 1996, 1997 Alexandre Julliard
 * Copyright 1997 Eric Youngdale
 * Copyright 1999 Ulrich Weigand
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "config.h"
#include "wine/port.h"

#include <ctype.h>

#include "thread.h"
#include "wine/winbase16.h"

#include "build.h"

/* offset of a structure field relative to the start of the struct */
#define STRUCTOFFSET(type,field) ((int)FIELD_OFFSET(type,field))

/* offset of register relative to the start of the CONTEXT struct */
#define CONTEXTOFFSET(reg)  STRUCTOFFSET(CONTEXT86,reg)

/* offset of register relative to the start of the STACK16FRAME struct */
#define STACK16OFFSET(reg)  STRUCTOFFSET(STACK16FRAME,reg)

/* offset of register relative to the start of the STACK32FRAME struct */
#define STACK32OFFSET(reg)  STRUCTOFFSET(STACK32FRAME,reg)

/* offset of the stack pointer relative to %fs:(0) */
#define STACKOFFSET 0xc0  /* STRUCTOFFSET(TEB,WOW32Reserved) */

/* fix this if the ntdll_thread_regs structure is changed */
#define GS_OFFSET  0x1b0  /* STRUCTOFFSET(TEB,SpareBytes1) + STRUCTOFFSET(ntdll_thread_regs,gs) */

static void function_header( FILE *outfile, const char *name )
{
    fprintf( outfile, "\n\t.align %d\n", get_alignment(4) );
    fprintf( outfile, "\t%s\n", func_declaration(name) );
    fprintf( outfile, "%s\n", asm_globl(name) );
}


/*******************************************************************
 *         BuildCallFrom16Core
 *
 * This routine builds the core routines used in 16->32 thunks:
 * CallFrom16Word, CallFrom16Long, CallFrom16Register, and CallFrom16Thunk.
 *
 * These routines are intended to be called via a far call (with 32-bit
 * operand size) from 16-bit code.  The 16-bit code stub must push %bp,
 * the 32-bit entry point to be called, and the argument conversion
 * routine to be used (see stack layout below).
 *
 * The core routine completes the STACK16FRAME on the 16-bit stack and
 * switches to the 32-bit stack.  Then, the argument conversion routine
 * is called; it gets passed the 32-bit entry point and a pointer to the
 * 16-bit arguments (on the 16-bit stack) as parameters. (You can either
 * use conversion routines automatically generated by BuildCallFrom16,
 * or write your own for special purposes.)
 *
 * The conversion routine must call the 32-bit entry point, passing it
 * the converted arguments, and return its return value to the core.
 * After the conversion routine has returned, the core switches back
 * to the 16-bit stack, converts the return value to the DX:AX format
 * (CallFrom16Long), and returns to the 16-bit call stub.  All parameters,
 * including %bp, are popped off the stack.
 *
 * The 16-bit call stub now returns to the caller, popping the 16-bit
 * arguments if necessary (pascal calling convention).
 *
 * In the case of a 'register' function, CallFrom16Register fills a
 * CONTEXT86 structure with the values all registers had at the point
 * the first instruction of the 16-bit call stub was about to be
 * executed.  A pointer to this CONTEXT86 is passed as third parameter
 * to the argument conversion routine, which typically passes it on
 * to the called 32-bit entry point.
 *
 * CallFrom16Thunk is a special variant used by the implementation of
 * the Win95 16->32 thunk functions C16ThkSL and C16ThkSL01 and is
 * implemented as follows:
 * On entry, the EBX register is set up to contain a flat pointer to the
 * 16-bit stack such that EBX+22 points to the first argument.
 * Then, the entry point is called, while EBP is set up to point
 * to the return address (on the 32-bit stack).
 * The called function returns with CX set to the number of bytes
 * to be popped of the caller's stack.
 *
 * Stack layout upon entry to the core routine (STACK16FRAME):
 *  ...           ...
 * (sp+24) word   first 16-bit arg
 * (sp+22) word   cs
 * (sp+20) word   ip
 * (sp+18) word   bp
 * (sp+14) long   32-bit entry point (reused for Win16 mutex recursion count)
 * (sp+12) word   ip of actual entry point (necessary for relay debugging)
 * (sp+8)  long   relay (argument conversion) function entry point
 * (sp+4)  long   cs of 16-bit entry point
 * (sp)    long   ip of 16-bit entry point
 *
 * Added on the stack:
 * (sp-2)  word   saved gs
 * (sp-4)  word   saved fs
 * (sp-6)  word   saved es
 * (sp-8)  word   saved ds
 * (sp-12) long   saved ebp
 * (sp-16) long   saved ecx
 * (sp-20) long   saved edx
 * (sp-24) long   saved previous stack
 */
static void BuildCallFrom16Core( FILE *outfile, int reg_func, int thunk )
{
    /* Function header */
    if (thunk) function_header( outfile, "__wine_call_from_16_thunk" );
    else if (reg_func) function_header( outfile, "__wine_call_from_16_regs" );
    else function_header( outfile, "__wine_call_from_16" );

    /* Create STACK16FRAME (except STACK32FRAME link) */
    fprintf( outfile, "\tpushw %%gs\n" );
    fprintf( outfile, "\tpushw %%fs\n" );
    fprintf( outfile, "\tpushw %%es\n" );
    fprintf( outfile, "\tpushw %%ds\n" );
    fprintf( outfile, "\tpushl %%ebp\n" );
    fprintf( outfile, "\tpushl %%ecx\n" );
    fprintf( outfile, "\tpushl %%edx\n" );

    /* Save original EFlags register */
    if (reg_func) fprintf( outfile, "\tpushfl\n" );

    if ( UsePIC )
    {
        fprintf( outfile, "\tcall 1f\n" );
        fprintf( outfile, "1:\tpopl %%ecx\n" );
        fprintf( outfile, "\t.byte 0x2e\n\tmovl %s-1b(%%ecx),%%edx\n",
                 asm_name("CallTo16_DataSelector") );
    }
    else
        fprintf( outfile, "\t.byte 0x2e\n\tmovl %s,%%edx\n", asm_name("CallTo16_DataSelector") );

    /* Load 32-bit segment registers */
    fprintf( outfile, "\tmovw %%dx, %%ds\n" );
    fprintf( outfile, "\tmovw %%dx, %%es\n" );

    if ( UsePIC )
        fprintf( outfile, "\tmovw %s-1b(%%ecx), %%fs\n", asm_name("CallTo16_TebSelector") );
    else
        fprintf( outfile, "\tmovw %s, %%fs\n", asm_name("CallTo16_TebSelector") );

    fprintf( outfile, "\t.byte 0x64\n\tmov (%d),%%gs\n", GS_OFFSET );

    /* Translate STACK16FRAME base to flat offset in %edx */
    fprintf( outfile, "\tmovw %%ss, %%dx\n" );
    fprintf( outfile, "\tandl $0xfff8, %%edx\n" );
    fprintf( outfile, "\tshrl $1, %%edx\n" );
    if (UsePIC)
    {
        fprintf( outfile, "\taddl wine_ldt_copy_ptr-1b(%%ecx),%%edx\n" );
        fprintf( outfile, "\tmovl (%%edx), %%edx\n" );
    }
    else
        fprintf( outfile, "\tmovl %s(%%edx), %%edx\n", asm_name("wine_ldt_copy") );
    fprintf( outfile, "\tmovzwl %%sp, %%ebp\n" );
    fprintf( outfile, "\tleal %d(%%ebp,%%edx), %%edx\n", reg_func ? 0 : -4 );

    /* Get saved flags into %ecx */
    if (reg_func) fprintf( outfile, "\tpopl %%ecx\n" );

    /* Get the 32-bit stack pointer from the TEB and complete STACK16FRAME */
    fprintf( outfile, "\t.byte 0x64\n\tmovl (%d), %%ebp\n", STACKOFFSET );
    fprintf( outfile, "\tpushl %%ebp\n" );

    /* Switch stacks */
    fprintf( outfile, "\t.byte 0x64\n\tmovw %%ss, (%d)\n", STACKOFFSET + 2 );
    fprintf( outfile, "\t.byte 0x64\n\tmovw %%sp, (%d)\n", STACKOFFSET );
    fprintf( outfile, "\tpushl %%ds\n" );
    fprintf( outfile, "\tpopl %%ss\n" );
    fprintf( outfile, "\tmovl %%ebp, %%esp\n" );
    fprintf( outfile, "\taddl $%d, %%ebp\n", STACK32OFFSET(ebp) );


    /* At this point:
       STACK16FRAME is completely set up
       DS, ES, SS: flat data segment
       FS: current TEB
       ESP: points to last STACK32FRAME
       EBP: points to ebp member of last STACK32FRAME
       EDX: points to current STACK16FRAME
       ECX: contains saved flags
       all other registers: unchanged */

    /* Special case: C16ThkSL stub */
    if ( thunk )
    {
        /* Set up registers as expected and call thunk */
        fprintf( outfile, "\tleal %d(%%edx), %%ebx\n", (int)sizeof(STACK16FRAME)-22 );
        fprintf( outfile, "\tleal -4(%%esp), %%ebp\n" );

        fprintf( outfile, "\tcall *%d(%%edx)\n", STACK16OFFSET(entry_point) );

        /* Switch stack back */
        fprintf( outfile, "\t.byte 0x64\n\tmovw (%d), %%ss\n", STACKOFFSET+2 );
        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%esp\n", STACKOFFSET );
        fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STACKOFFSET );

        /* Restore registers and return directly to caller */
        fprintf( outfile, "\taddl $8, %%esp\n" );
        fprintf( outfile, "\tpopl %%ebp\n" );
        fprintf( outfile, "\tpopw %%ds\n" );
        fprintf( outfile, "\tpopw %%es\n" );
        fprintf( outfile, "\tpopw %%fs\n" );
        fprintf( outfile, "\tpopw %%gs\n" );
        fprintf( outfile, "\taddl $20, %%esp\n" );

        fprintf( outfile, "\txorb %%ch, %%ch\n" );
        fprintf( outfile, "\tpopl %%ebx\n" );
        fprintf( outfile, "\taddw %%cx, %%sp\n" );
        fprintf( outfile, "\tpush %%ebx\n" );

        fprintf( outfile, "\t.byte 0x66\n" );
        fprintf( outfile, "\tlret\n" );

        return;
    }


    /* Build register CONTEXT */
    if ( reg_func )
    {
        fprintf( outfile, "\tsubl $%d, %%esp\n", (int)sizeof(CONTEXT86) );

        fprintf( outfile, "\tmovl %%ecx, %d(%%esp)\n", CONTEXTOFFSET(EFlags) );

        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Eax) );
        fprintf( outfile, "\tmovl %%ebx, %d(%%esp)\n", CONTEXTOFFSET(Ebx) );
        fprintf( outfile, "\tmovl %%esi, %d(%%esp)\n", CONTEXTOFFSET(Esi) );
        fprintf( outfile, "\tmovl %%edi, %d(%%esp)\n", CONTEXTOFFSET(Edi) );

        fprintf( outfile, "\tmovl %d(%%edx), %%eax\n", STACK16OFFSET(ebp) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Ebp) );
        fprintf( outfile, "\tmovl %d(%%edx), %%eax\n", STACK16OFFSET(ecx) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Ecx) );
        fprintf( outfile, "\tmovl %d(%%edx), %%eax\n", STACK16OFFSET(edx) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Edx) );

        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(ds) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegDs) );
        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(es) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegEs) );
        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(fs) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegFs) );
        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(gs) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegGs) );

        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(cs) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegCs) );
        fprintf( outfile, "\tmovzwl %d(%%edx), %%eax\n", STACK16OFFSET(ip) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Eip) );

        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%eax\n", STACKOFFSET+2 );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(SegSs) );
        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%eax\n", STACKOFFSET );
        fprintf( outfile, "\taddl $%d, %%eax\n", STACK16OFFSET(ip) );
        fprintf( outfile, "\tmovl %%eax, %d(%%esp)\n", CONTEXTOFFSET(Esp) );
#if 0
        fprintf( outfile, "\tfsave %d(%%esp)\n", CONTEXTOFFSET(FloatSave) );
#endif

        /* Push address of CONTEXT86 structure -- popped by the relay routine */
        fprintf( outfile, "\tmovl %%esp,%%eax\n" );
        fprintf( outfile, "\tandl $~15,%%esp\n" );
        fprintf( outfile, "\tsubl $4,%%esp\n" );
        fprintf( outfile, "\tpushl %%eax\n" );
    }
    else
    {
        fprintf( outfile, "\tsubl $8,%%esp\n" );
        fprintf( outfile, "\tandl $~15,%%esp\n" );
        fprintf( outfile, "\taddl $8,%%esp\n" );
    }

    /* Call relay routine (which will call the API entry point) */
    fprintf( outfile, "\tleal %d(%%edx), %%eax\n", (int)sizeof(STACK16FRAME) );
    fprintf( outfile, "\tpushl %%eax\n" );
    fprintf( outfile, "\tpushl %d(%%edx)\n", STACK16OFFSET(entry_point) );
    fprintf( outfile, "\tcall *%d(%%edx)\n", STACK16OFFSET(relay) );

    if ( reg_func )
    {
        fprintf( outfile, "\tleal -%d(%%ebp), %%ebx\n",
                 (int)sizeof(CONTEXT) + STACK32OFFSET(ebp) );

        /* Switch stack back */
        fprintf( outfile, "\t.byte 0x64\n\tmovw (%d), %%ss\n", STACKOFFSET+2 );
        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%esp\n", STACKOFFSET );
        fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STACKOFFSET );

        /* Get return address to CallFrom16 stub */
        fprintf( outfile, "\taddw $%d, %%sp\n", STACK16OFFSET(callfrom_ip)-4 );
        fprintf( outfile, "\tpopl %%eax\n" );
        fprintf( outfile, "\tpopl %%edx\n" );

        /* Restore all registers from CONTEXT */
        fprintf( outfile, "\tmovw %d(%%ebx), %%ss\n", CONTEXTOFFSET(SegSs) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%esp\n", CONTEXTOFFSET(Esp) );
        fprintf( outfile, "\taddl $4, %%esp\n" );  /* room for final return address */

        fprintf( outfile, "\tpushw %d(%%ebx)\n", CONTEXTOFFSET(SegCs) );
        fprintf( outfile, "\tpushw %d(%%ebx)\n", CONTEXTOFFSET(Eip) );
        fprintf( outfile, "\tpushl %%edx\n" );
        fprintf( outfile, "\tpushl %%eax\n" );
        fprintf( outfile, "\tpushl %d(%%ebx)\n", CONTEXTOFFSET(EFlags) );
        fprintf( outfile, "\tpushl %d(%%ebx)\n", CONTEXTOFFSET(SegDs) );

        fprintf( outfile, "\tpushl %d(%%ebx)\n", CONTEXTOFFSET(SegEs) );
        fprintf( outfile, "\tpopl %%es\n" );
        fprintf( outfile, "\tpushl %d(%%ebx)\n", CONTEXTOFFSET(SegFs) );
        fprintf( outfile, "\tpopl %%fs\n" );
        fprintf( outfile, "\tpushl %d(%%ebx)\n", CONTEXTOFFSET(SegGs) );
        fprintf( outfile, "\tpopl %%gs\n" );

        fprintf( outfile, "\tmovl %d(%%ebx), %%ebp\n", CONTEXTOFFSET(Ebp) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%esi\n", CONTEXTOFFSET(Esi) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%edi\n", CONTEXTOFFSET(Edi) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%eax\n", CONTEXTOFFSET(Eax) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%edx\n", CONTEXTOFFSET(Edx) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%ecx\n", CONTEXTOFFSET(Ecx) );
        fprintf( outfile, "\tmovl %d(%%ebx), %%ebx\n", CONTEXTOFFSET(Ebx) );

        fprintf( outfile, "\tpopl %%ds\n" );
        fprintf( outfile, "\tpopfl\n" );
        fprintf( outfile, "\tlret\n" );
    }
    else
    {
        /* Switch stack back */
        fprintf( outfile, "\t.byte 0x64\n\tmovw (%d), %%ss\n", STACKOFFSET+2 );
        fprintf( outfile, "\t.byte 0x64\n\tmovzwl (%d), %%esp\n", STACKOFFSET );
        fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STACKOFFSET );

        /* Restore registers */
        fprintf( outfile, "\tpopl %%edx\n" );
        fprintf( outfile, "\tpopl %%ecx\n" );
        fprintf( outfile, "\tpopl %%ebp\n" );
        fprintf( outfile, "\tpopw %%ds\n" );
        fprintf( outfile, "\tpopw %%es\n" );
        fprintf( outfile, "\tpopw %%fs\n" );
        fprintf( outfile, "\tpopw %%gs\n" );

        /* Return to return stub which will return to caller */
        fprintf( outfile, "\tlret $12\n" );
    }
    if (thunk) output_function_size( outfile, "__wine_call_from_16_thunk" );
    else if (reg_func) output_function_size( outfile, "__wine_call_from_16_regs" );
    else output_function_size( outfile, "__wine_call_from_16" );
}


/*******************************************************************
 *         BuildCallTo16Core
 *
 * This routine builds the core routines used in 32->16 thunks:
 *
 * extern DWORD WINAPI wine_call_to_16( FARPROC16 target, DWORD cbArgs, PEXCEPTION_HANDLER handler );
 * extern void WINAPI wine_call_to_16_regs( CONTEXT86 *context, DWORD cbArgs, PEXCEPTION_HANDLER handler );
 *
 * These routines can be called directly from 32-bit code.
 *
 * All routines expect that the 16-bit stack contents (arguments) and the
 * return address (segptr to CallTo16_Ret) were already set up by the
 * caller; nb_args must contain the number of bytes to be conserved.  The
 * 16-bit SS:SP will be set accordinly.
 *
 * All other registers are either taken from the CONTEXT86 structure
 * or else set to default values.  The target routine address is either
 * given directly or taken from the CONTEXT86.
 */
static void BuildCallTo16Core( FILE *outfile, int reg_func )
{
    const char *name = reg_func ? "wine_call_to_16_regs" : "wine_call_to_16";

    /* Function header */
    function_header( outfile, name );

    /* Function entry sequence */
    fprintf( outfile, "\tpushl %%ebp\n" );
    fprintf( outfile, "\tmovl %%esp, %%ebp\n" );

    /* Save the 32-bit registers */
    fprintf( outfile, "\tpushl %%ebx\n" );
    fprintf( outfile, "\tpushl %%esi\n" );
    fprintf( outfile, "\tpushl %%edi\n" );
    fprintf( outfile, "\t.byte 0x64\n\tmov %%gs,(%d)\n", GS_OFFSET );

    /* Setup exception frame */
    fprintf( outfile, "\t.byte 0x64\n\tpushl (%d)\n", STACKOFFSET );
    fprintf( outfile, "\tpushl 16(%%ebp)\n" ); /* handler */
    fprintf( outfile, "\t.byte 0x64\n\tpushl (0)\n" );
    fprintf( outfile, "\t.byte 0x64\n\tmovl %%esp,(0)\n" );

    /* Call the actual CallTo16 routine (simulate a lcall) */
    fprintf( outfile, "\tpushl %%cs\n" );
    fprintf( outfile, "\tcall .L%s\n", name );

    /* Remove exception frame */
    fprintf( outfile, "\t.byte 0x64\n\tpopl (0)\n" );
    fprintf( outfile, "\taddl $4, %%esp\n" );
    fprintf( outfile, "\t.byte 0x64\n\tpopl (%d)\n", STACKOFFSET );

    if ( !reg_func )
    {
        /* Convert return value */
        fprintf( outfile, "\tandl $0xffff,%%eax\n" );
        fprintf( outfile, "\tshll $16,%%edx\n" );
        fprintf( outfile, "\torl %%edx,%%eax\n" );
    }
    else
    {
        /*
         * Modify CONTEXT86 structure to contain new values
         *
         * NOTE:  We restore only EAX, EBX, EDX, EDX, EBP, and ESP.
         *        The segment registers as well as ESI and EDI should
         *        not be modified by a well-behaved 16-bit routine in
         *        any case.  [If necessary, we could restore them as well,
         *        at the cost of a somewhat less efficient return path.]
         */

        fprintf( outfile, "\tmovl %d(%%esp), %%edi\n", STACK32OFFSET(target) - STACK32OFFSET(edi));
                /* everything above edi has been popped already */

        fprintf( outfile, "\tmovl %%eax, %d(%%edi)\n", CONTEXTOFFSET(Eax) );
        fprintf( outfile, "\tmovl %%ebx, %d(%%edi)\n", CONTEXTOFFSET(Ebx) );
        fprintf( outfile, "\tmovl %%ecx, %d(%%edi)\n", CONTEXTOFFSET(Ecx) );
        fprintf( outfile, "\tmovl %%edx, %d(%%edi)\n", CONTEXTOFFSET(Edx) );
        fprintf( outfile, "\tmovl %%ebp, %d(%%edi)\n", CONTEXTOFFSET(Ebp) );
        fprintf( outfile, "\tmovl %%esi, %d(%%edi)\n", CONTEXTOFFSET(Esp) );
                 /* The return glue code saved %esp into %esi */
    }

    /* Restore the 32-bit registers */
    fprintf( outfile, "\tpopl %%edi\n" );
    fprintf( outfile, "\tpopl %%esi\n" );
    fprintf( outfile, "\tpopl %%ebx\n" );

    /* Function exit sequence */
    fprintf( outfile, "\tpopl %%ebp\n" );
    fprintf( outfile, "\tret $12\n" );


    /* Start of the actual CallTo16 routine */

    fprintf( outfile, ".L%s:\n", name );

    /* Switch to the 16-bit stack */
    fprintf( outfile, "\tmovl %%esp,%%edx\n" );
    fprintf( outfile, "\t.byte 0x64\n\tmovw (%d),%%ss\n", STACKOFFSET + 2);
    fprintf( outfile, "\t.byte 0x64\n\tmovw (%d),%%sp\n", STACKOFFSET );
    fprintf( outfile, "\t.byte 0x64\n\tmovl %%edx,(%d)\n", STACKOFFSET );

    /* Make %bp point to the previous stackframe (built by CallFrom16) */
    fprintf( outfile, "\tmovzwl %%sp,%%ebp\n" );
    fprintf( outfile, "\tleal %d(%%ebp),%%ebp\n", STACK16OFFSET(bp) );

    /* Add the specified offset to the new sp */
    fprintf( outfile, "\tsubw %d(%%edx), %%sp\n", STACK32OFFSET(nb_args) );

    if (reg_func)
    {
        /* Push the called routine address */
        fprintf( outfile, "\tmovl %d(%%edx),%%edx\n", STACK32OFFSET(target) );
        fprintf( outfile, "\tpushw %d(%%edx)\n", CONTEXTOFFSET(SegCs) );
        fprintf( outfile, "\tpushw %d(%%edx)\n", CONTEXTOFFSET(Eip) );

        /* Get the registers */
        fprintf( outfile, "\tpushw %d(%%edx)\n", CONTEXTOFFSET(SegDs) );
        fprintf( outfile, "\tpushl %d(%%edx)\n", CONTEXTOFFSET(SegEs) );
        fprintf( outfile, "\tpopl %%es\n" );
        fprintf( outfile, "\tpushl %d(%%edx)\n", CONTEXTOFFSET(SegFs) );
        fprintf( outfile, "\tpopl %%fs\n" );
        fprintf( outfile, "\tpushl %d(%%edx)\n", CONTEXTOFFSET(SegGs) );
        fprintf( outfile, "\tpopl %%gs\n" );
        fprintf( outfile, "\tmovl %d(%%edx),%%ebp\n", CONTEXTOFFSET(Ebp) );
        fprintf( outfile, "\tmovl %d(%%edx),%%esi\n", CONTEXTOFFSET(Esi) );
        fprintf( outfile, "\tmovl %d(%%edx),%%edi\n", CONTEXTOFFSET(Edi) );
        fprintf( outfile, "\tmovl %d(%%edx),%%eax\n", CONTEXTOFFSET(Eax) );
        fprintf( outfile, "\tmovl %d(%%edx),%%ebx\n", CONTEXTOFFSET(Ebx) );
        fprintf( outfile, "\tmovl %d(%%edx),%%ecx\n", CONTEXTOFFSET(Ecx) );
        fprintf( outfile, "\tmovl %d(%%edx),%%edx\n", CONTEXTOFFSET(Edx) );

        /* Get the 16-bit ds */
        fprintf( outfile, "\tpopw %%ds\n" );
    }
    else  /* not a register function */
    {
        /* Push the called routine address */
        fprintf( outfile, "\tpushl %d(%%edx)\n", STACK32OFFSET(target) );

        /* Set %fs and %gs to the value saved by the last CallFrom16 */
        fprintf( outfile, "\tpushw %d(%%ebp)\n", STACK16OFFSET(fs)-STACK16OFFSET(bp) );
        fprintf( outfile, "\tpopw %%fs\n" );
        fprintf( outfile, "\tpushw %d(%%ebp)\n", STACK16OFFSET(gs)-STACK16OFFSET(bp) );
        fprintf( outfile, "\tpopw %%gs\n" );

        /* Set %ds and %es (and %ax just in case) equal to %ss */
        fprintf( outfile, "\tmovw %%ss,%%ax\n" );
        fprintf( outfile, "\tmovw %%ax,%%ds\n" );
        fprintf( outfile, "\tmovw %%ax,%%es\n" );
    }

    /* Jump to the called routine */
    fprintf( outfile, "\t.byte 0x66\n" );
    fprintf( outfile, "\tlret\n" );

    /* Function footer */
    output_function_size( outfile, name );
}


/*******************************************************************
 *         BuildRet16Func
 *
 * Build the return code for 16-bit callbacks
 */
static void BuildRet16Func( FILE *outfile )
{
    function_header( outfile, "__wine_call_to_16_ret" );

    /* Save %esp into %esi */
    fprintf( outfile, "\tmovl %%esp,%%esi\n" );

    /* Restore 32-bit segment registers */

    fprintf( outfile, "\t.byte 0x2e\n\tmovl %s", asm_name("CallTo16_DataSelector") );
    fprintf( outfile, "-%s,%%edi\n", asm_name("__wine_call16_start") );
    fprintf( outfile, "\tmovw %%di,%%ds\n" );
    fprintf( outfile, "\tmovw %%di,%%es\n" );

    fprintf( outfile, "\t.byte 0x2e\n\tmov %s", asm_name("CallTo16_TebSelector") );
    fprintf( outfile, "-%s,%%fs\n", asm_name("__wine_call16_start") );

    fprintf( outfile, "\t.byte 0x64\n\tmov (%d),%%gs\n", GS_OFFSET );

    /* Restore the 32-bit stack */

    fprintf( outfile, "\tmovw %%di,%%ss\n" );
    fprintf( outfile, "\t.byte 0x64\n\tmovl (%d),%%esp\n", STACKOFFSET );

    /* Return to caller */

    fprintf( outfile, "\tlret\n" );
    output_function_size( outfile, "__wine_call_to_16_ret" );
}


/*******************************************************************
 *         BuildCallTo32CBClient
 *
 * Call a CBClient relay stub from 32-bit code (KERNEL.620).
 *
 * Since the relay stub is itself 32-bit, this should not be a problem;
 * unfortunately, the relay stubs are expected to switch back to a
 * 16-bit stack (and 16-bit code) after completion :-(
 *
 * This would conflict with our 16- vs. 32-bit stack handling, so
 * we simply switch *back* to our 32-bit stack before returning to
 * the caller ...
 *
 * The CBClient relay stub expects to be called with the following
 * 16-bit stack layout, and with ebp and ebx pointing into the 16-bit
 * stack at the designated places:
 *
 *    ...
 *  (ebp+14) original arguments to the callback routine
 *  (ebp+10) far return address to original caller
 *  (ebp+6)  Thunklet target address
 *  (ebp+2)  Thunklet relay ID code
 *  (ebp)    BP (saved by CBClientGlueSL)
 *  (ebp-2)  SI (saved by CBClientGlueSL)
 *  (ebp-4)  DI (saved by CBClientGlueSL)
 *  (ebp-6)  DS (saved by CBClientGlueSL)
 *
 *   ...     buffer space used by the 16-bit side glue for temp copies
 *
 *  (ebx+4)  far return address to 16-bit side glue code
 *  (ebx)    saved 16-bit ss:sp (pointing to ebx+4)
 *
 * The 32-bit side glue code accesses both the original arguments (via ebp)
 * and the temporary copies prepared by the 16-bit side glue (via ebx).
 * After completion, the stub will load ss:sp from the buffer at ebx
 * and perform a far return to 16-bit code.
 *
 * To trick the relay stub into returning to us, we replace the 16-bit
 * return address to the glue code by a cs:ip pair pointing to our
 * return entry point (the original return address is saved first).
 * Our return stub thus called will then reload the 32-bit ss:esp and
 * return to 32-bit code (by using and ss:esp value that we have also
 * pushed onto the 16-bit stack before and a cs:eip values found at
 * that position on the 32-bit stack).  The ss:esp to be restored is
 * found relative to the 16-bit stack pointer at:
 *
 *  (ebx-4)   ss  (flat)
 *  (ebx-8)   sp  (32-bit stack pointer)
 *
 * The second variant of this routine, CALL32_CBClientEx, which is used
 * to implement KERNEL.621, has to cope with yet another problem: Here,
 * the 32-bit side directly returns to the caller of the CBClient thunklet,
 * restoring registers saved by CBClientGlueSL and cleaning up the stack.
 * As we have to return to our 32-bit code first, we have to adapt the
 * layout of our temporary area so as to include values for the registers
 * that are to be restored, and later (in the implementation of KERNEL.621)
 * we *really* restore them. The return stub restores DS, DI, SI, and BP
 * from the stack, skips the next 8 bytes (CBClient relay code / target),
 * and then performs a lret NN, where NN is the number of arguments to be
 * removed. Thus, we prepare our temporary area as follows:
 *
 *     (ebx+22) 16-bit cs  (this segment)
 *     (ebx+20) 16-bit ip  ('16-bit' return entry point)
 *     (ebx+16) 32-bit ss  (flat)
 *     (ebx+12) 32-bit sp  (32-bit stack pointer)
 *     (ebx+10) 16-bit bp  (points to ebx+24)
 *     (ebx+8)  16-bit si  (ignored)
 *     (ebx+6)  16-bit di  (ignored)
 *     (ebx+4)  16-bit ds  (we actually use the flat DS here)
 *     (ebx+2)  16-bit ss  (16-bit stack segment)
 *     (ebx+0)  16-bit sp  (points to ebx+4)
 *
 * Note that we ensure that DS is not changed and remains the flat segment,
 * and the 32-bit stack pointer our own return stub needs fits just
 * perfectly into the 8 bytes that are skipped by the Windows stub.
 * One problem is that we have to determine the number of removed arguments,
 * as these have to be really removed in KERNEL.621. Thus, the BP value
 * that we place in the temporary area to be restored, contains the value
 * that SP would have if no arguments were removed. By comparing the actual
 * value of SP with this value in our return stub we can compute the number
 * of removed arguments. This is then returned to KERNEL.621.
 *
 * The stack layout of this function:
 * (ebp+20)  nArgs     pointer to variable receiving nr. of args (Ex only)
 * (ebp+16)  esi       pointer to caller's esi value
 * (ebp+12)  arg       ebp value to be set for relay stub
 * (ebp+8)   func      CBClient relay stub address
 * (ebp+4)   ret addr
 * (ebp)     ebp
 */
static void BuildCallTo32CBClient( FILE *outfile, BOOL isEx )
{
    function_header( outfile, isEx ? "CALL32_CBClientEx" : "CALL32_CBClient" );

    /* Entry code */

    fprintf( outfile, "\tpushl %%ebp\n" );
    fprintf( outfile, "\tmovl %%esp,%%ebp\n" );
    fprintf( outfile, "\tpushl %%edi\n" );
    fprintf( outfile, "\tpushl %%esi\n" );
    fprintf( outfile, "\tpushl %%ebx\n" );

    /* Get pointer to temporary area and save the 32-bit stack pointer */

    fprintf( outfile, "\tmovl 16(%%ebp), %%ebx\n" );
    fprintf( outfile, "\tleal -8(%%esp), %%eax\n" );

    if ( !isEx )
        fprintf( outfile, "\tmovl %%eax, -8(%%ebx)\n" );
    else
        fprintf( outfile, "\tmovl %%eax, 12(%%ebx)\n" );

    /* Set up registers and call CBClient relay stub (simulating a far call) */

    fprintf( outfile, "\tmovl 20(%%ebp), %%esi\n" );
    fprintf( outfile, "\tmovl (%%esi), %%esi\n" );

    fprintf( outfile, "\tmovl 8(%%ebp), %%eax\n" );
    fprintf( outfile, "\tmovl 12(%%ebp), %%ebp\n" );

    fprintf( outfile, "\tpushl %%cs\n" );
    fprintf( outfile, "\tcall *%%eax\n" );

    /* Return new esi value to caller */

    fprintf( outfile, "\tmovl 32(%%esp), %%edi\n" );
    fprintf( outfile, "\tmovl %%esi, (%%edi)\n" );

    /* Return argument size to caller */
    if ( isEx )
    {
        fprintf( outfile, "\tmovl 36(%%esp), %%ebx\n" );
        fprintf( outfile, "\tmovl %%ebp, (%%ebx)\n" );
    }

    /* Restore registers and return */

    fprintf( outfile, "\tpopl %%ebx\n" );
    fprintf( outfile, "\tpopl %%esi\n" );
    fprintf( outfile, "\tpopl %%edi\n" );
    fprintf( outfile, "\tpopl %%ebp\n" );
    fprintf( outfile, "\tret\n" );
    output_function_size( outfile, isEx ? "CALL32_CBClientEx" : "CALL32_CBClient" );

    /* '16-bit' return stub */

    function_header( outfile, isEx ? "CALL32_CBClientEx_Ret" : "CALL32_CBClient_Ret" );
    if ( !isEx )
    {
        fprintf( outfile, "\tmovzwl %%sp, %%ebx\n" );
        fprintf( outfile, "\tlssl %%ss:-16(%%ebx), %%esp\n" );
    }
    else
    {
        fprintf( outfile, "\tmovzwl %%bp, %%ebx\n" );
        fprintf( outfile, "\tsubw %%bp, %%sp\n" );
        fprintf( outfile, "\tmovzwl %%sp, %%ebp\n" );
        fprintf( outfile, "\tlssl %%ss:-12(%%ebx), %%esp\n" );
    }
    fprintf( outfile, "\tlret\n" );
    output_function_size( outfile, isEx ? "CALL32_CBClientEx_Ret" : "CALL32_CBClient_Ret" );
}


/*******************************************************************
 *         BuildCallFrom32Regs
 *
 * Build a 32-bit-to-Wine call-back function for a 'register' function.
 * 'args' is the number of dword arguments.
 *
 * Stack layout:
 *   ...
 * (ebp+16)  first arg
 * (ebp+12)  ret addr to user code
 * (ebp+8)   eax saved by relay code
 * (ebp+4)   ret addr to relay code
 * (ebp+0)   saved ebp
 * (ebp-128) buffer area to allow stack frame manipulation
 * (ebp-332) CONTEXT86 struct
 * (ebp-336) padding for stack alignment
 * (ebp-336-n) CONTEXT86 *argument
 *  ....     other arguments copied from (ebp+12)
 *
 * The entry point routine is called with a CONTEXT* extra argument,
 * following the normal args. In this context structure, EIP_reg
 * contains the return address to user code, and ESP_reg the stack
 * pointer on return (with the return address and arguments already
 * removed).
 */
static void BuildCallFrom32Regs( FILE *outfile )
{
    static const int STACK_SPACE = 128 + sizeof(CONTEXT86);

    /* Function header */

    function_header( outfile, "__wine_call_from_32_regs" );

    /* Allocate some buffer space on the stack */

    fprintf( outfile, "\tpushl %%ebp\n" );
    fprintf( outfile, "\tmovl %%esp,%%ebp\n ");
    fprintf( outfile, "\tleal -%d(%%esp), %%esp\n", STACK_SPACE + 4 /* for context arg */);

    /* Build the context structure */

    fprintf( outfile, "\tpushfl\n" );
    fprintf( outfile, "\tpopl %%eax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(EFlags) - STACK_SPACE );
    fprintf( outfile, "\tmovl 0(%%ebp),%%eax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(Ebp) - STACK_SPACE );
    fprintf( outfile, "\tmovl 8(%%ebp),%%eax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(Eax) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%ebx,%d(%%ebp)\n", CONTEXTOFFSET(Ebx) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%ecx,%d(%%ebp)\n", CONTEXTOFFSET(Ecx) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%edx,%d(%%ebp)\n", CONTEXTOFFSET(Edx) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%esi,%d(%%ebp)\n", CONTEXTOFFSET(Esi) - STACK_SPACE );
    fprintf( outfile, "\tmovl %%edi,%d(%%ebp)\n", CONTEXTOFFSET(Edi) - STACK_SPACE );

    fprintf( outfile, "\txorl %%eax,%%eax\n" );
    fprintf( outfile, "\tmovw %%cs,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegCs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%es,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegEs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%fs,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegFs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%gs,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegGs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ss,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegSs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ds,%%ax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(SegDs) - STACK_SPACE );
    fprintf( outfile, "\tmovw %%ax,%%es\n" );  /* set %es equal to %ds just in case */

    fprintf( outfile, "\tmovl $0x%x,%%eax\n", CONTEXT86_FULL );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(ContextFlags) - STACK_SPACE );

    fprintf( outfile, "\tmovl 12(%%ebp),%%eax\n" ); /* Get %eip at time of call */
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(Eip) - STACK_SPACE );

    /* Transfer the arguments */

    fprintf( outfile, "\tmovl 4(%%ebp),%%ebx\n" );   /* get relay code addr */
    fprintf( outfile, "\tmovzbl 4(%%ebx),%%ecx\n" ); /* fetch number of args to copy */
    fprintf( outfile, "\tsubl %%ecx,%%esp\n" );
    fprintf( outfile, "\tandl $~15,%%esp\n" );
    fprintf( outfile, "\tleal 16(%%ebp),%%esi\n" );  /* get %esp at time of call */
    fprintf( outfile, "\tmovl %%esp,%%edi\n" );
    fprintf( outfile, "\tshrl $2,%%ecx\n" );
    fprintf( outfile, "\tjz 1f\n" );
    fprintf( outfile, "\tcld\n" );
    fprintf( outfile, "\trep\n\tmovsl\n" );  /* copy args */
    fprintf( outfile, "1:\tleal %d(%%ebp),%%eax\n", -STACK_SPACE );  /* get addr of context struct */
    fprintf( outfile, "\tmovl %%eax,(%%edi)\n" );    /* and pass it as extra arg */
    fprintf( outfile, "\tmovzbl 5(%%ebx),%%eax\n" ); /* fetch number of args to remove */
    fprintf( outfile, "\tleal 16(%%ebp,%%eax),%%eax\n" );
    fprintf( outfile, "\tmovl %%eax,%d(%%ebp)\n", CONTEXTOFFSET(Esp) - STACK_SPACE );

    /* Call the entry point */

    fprintf( outfile, "\taddl (%%ebx),%%ebx\n" );
    fprintf( outfile, "\tcall *%%ebx\n" );
    fprintf( outfile, "\tleal -%d(%%ebp),%%ecx\n", STACK_SPACE );

    /* Restore the context structure */

    fprintf( outfile, "2:\tpushl %d(%%ecx)\n", CONTEXTOFFSET(SegEs) );
    fprintf( outfile, "\tpopl %%es\n" );
    fprintf( outfile, "\tpushl %d(%%ecx)\n", CONTEXTOFFSET(SegFs) );
    fprintf( outfile, "\tpopl %%fs\n" );
    fprintf( outfile, "\tpushl %d(%%ecx)\n", CONTEXTOFFSET(SegGs) );
    fprintf( outfile, "\tpopl %%gs\n" );

    fprintf( outfile, "\tmovl %d(%%ecx),%%edi\n", CONTEXTOFFSET(Edi) );
    fprintf( outfile, "\tmovl %d(%%ecx),%%esi\n", CONTEXTOFFSET(Esi) );
    fprintf( outfile, "\tmovl %d(%%ecx),%%edx\n", CONTEXTOFFSET(Edx) );
    fprintf( outfile, "\tmovl %d(%%ecx),%%ebx\n", CONTEXTOFFSET(Ebx) );
    fprintf( outfile, "\tmovl %d(%%ecx),%%eax\n", CONTEXTOFFSET(Eax) );
    fprintf( outfile, "\tmovl %d(%%ecx),%%ebp\n", CONTEXTOFFSET(Ebp) );

    fprintf( outfile, "\tpushl %d(%%ecx)\n", CONTEXTOFFSET(SegSs) );
    fprintf( outfile, "\tpopl %%ss\n" );
    fprintf( outfile, "\tmovl %d(%%ecx),%%esp\n", CONTEXTOFFSET(Esp) );

    fprintf( outfile, "\tpushl %d(%%ecx)\n", CONTEXTOFFSET(EFlags) );
    fprintf( outfile, "\tpushl %d(%%ecx)\n", CONTEXTOFFSET(SegCs) );
    fprintf( outfile, "\tpushl %d(%%ecx)\n", CONTEXTOFFSET(Eip) );
    fprintf( outfile, "\tpushl %d(%%ecx)\n", CONTEXTOFFSET(SegDs) );
    fprintf( outfile, "\tmovl %d(%%ecx),%%ecx\n", CONTEXTOFFSET(Ecx) );

    fprintf( outfile, "\tpopl %%ds\n" );
    fprintf( outfile, "\tiret\n" );
    output_function_size( outfile, "__wine_call_from_32_regs" );

    function_header( outfile, "__wine_call_from_32_restore_regs" );
    fprintf( outfile, "\tmovl 4(%%esp),%%ecx\n" );
    fprintf( outfile, "\tjmp 2b\n" );
    output_function_size( outfile, "__wine_call_from_32_restore_regs" );
}


/*******************************************************************
 *         BuildPendingEventCheck
 *
 * Build a function that checks whether there are any
 * pending DPMI events.
 *
 * Stack layout:
 *   
 * (sp+12) long   eflags
 * (sp+6)  long   cs
 * (sp+2)  long   ip
 * (sp)    word   fs
 *
 * On entry to function, fs register points to a valid TEB.
 * On exit from function, stack will be popped.
 */
static void BuildPendingEventCheck( FILE *outfile )
{
    /* Function header */

    function_header( outfile, "DPMI_PendingEventCheck" );

    /* Check for pending events. */

    fprintf( outfile, "\t.byte 0x64\n\ttestl $0xffffffff,(%d)\n",
             STRUCTOFFSET(TEB,vm86_pending) );
    fprintf( outfile, "\tje %s\n", asm_name("DPMI_PendingEventCheck_Cleanup") );

    fprintf( outfile, "\t.byte 0x64\n\ttestl $0xffffffff,(%d)\n",
             STRUCTOFFSET(TEB,dpmi_vif) );

    fprintf( outfile, "\tje %s\n", asm_name("DPMI_PendingEventCheck_Cleanup") );

    /* Process pending events. */

    fprintf( outfile, "\tsti\n" );

    /* Start cleanup. Restore fs register. */

    fprintf( outfile, "%s\n", asm_globl("DPMI_PendingEventCheck_Cleanup") );
    fprintf( outfile, "\tpopw %%fs\n" );

    /* Return from function. */

    fprintf( outfile, "%s\n", asm_globl("DPMI_PendingEventCheck_Return") );
    fprintf( outfile, "\tiret\n" );

    output_function_size( outfile, "DPMI_PendingEventCheck" );
}


/*******************************************************************
 *         BuildRelays16
 *
 * Build all the 16-bit relay callbacks
 */
void BuildRelays16( FILE *outfile )
{
    if (target_cpu != CPU_x86)
    {
        fprintf( outfile, "/* File not used with this architecture. Do not edit! */\n\n" );
        return;
    }

    /* File header */

    fprintf( outfile, "/* File generated automatically. Do not edit! */\n\n" );
    fprintf( outfile, "\t.text\n" );

    fprintf( outfile, "%s:\n\n", asm_name("__wine_spec_thunk_text_16") );

    fprintf( outfile, "%s\n", asm_globl("__wine_call16_start") );

    /* Standard CallFrom16 routine */
    BuildCallFrom16Core( outfile, FALSE, FALSE );

    /* Register CallFrom16 routine */
    BuildCallFrom16Core( outfile, TRUE, FALSE );

    /* C16ThkSL CallFrom16 routine */
    BuildCallFrom16Core( outfile, FALSE, TRUE );

    /* Standard CallTo16 routine */
    BuildCallTo16Core( outfile, 0 );

    /* Register CallTo16 routine */
    BuildCallTo16Core( outfile, 1 );

    /* Standard CallTo16 return stub */
    BuildRet16Func( outfile );

    /* CBClientThunkSL routine */
    BuildCallTo32CBClient( outfile, FALSE );

    /* CBClientThunkSLEx routine */
    BuildCallTo32CBClient( outfile, TRUE  );

    /* Pending DPMI events check stub */
    BuildPendingEventCheck( outfile );

    fprintf( outfile, "%s\n", asm_globl("__wine_call16_end") );
    output_function_size( outfile, "__wine_spec_thunk_text_16" );

    /* Declare the return address and data selector variables */
    fprintf( outfile, "\n\t.data\n\t.align %d\n", get_alignment(4) );
    fprintf( outfile, "%s\n\t.long 0\n", asm_globl("CallTo16_DataSelector") );
    fprintf( outfile, "%s\n\t.long 0\n", asm_globl("CallTo16_TebSelector") );
    if (UsePIC) fprintf( outfile, "wine_ldt_copy_ptr:\t.long %s\n", asm_name("wine_ldt_copy") );
    output_gnu_stack_note( outfile );
}

/*******************************************************************
 *         BuildRelays32
 *
 * Build all the 32-bit relay callbacks
 */
void BuildRelays32( FILE *outfile )
{
    if (target_cpu != CPU_x86)
    {
        fprintf( outfile, "/* File not used with this architecture. Do not edit! */\n\n" );
        return;
    }

    /* File header */

    fprintf( outfile, "/* File generated automatically. Do not edit! */\n\n" );
    fprintf( outfile, "\t.text\n" );
    fprintf( outfile, "%s:\n\n", asm_name("__wine_spec_thunk_text_32") );

    /* 32-bit register entry point */
    BuildCallFrom32Regs( outfile );

    output_function_size( outfile, "__wine_spec_thunk_text_32" );
    output_gnu_stack_note( outfile );
}
