/*
 * terminal.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include "defines.h"
#include <ncurses.h>
#include "terminal.h"
#include "xmem.h"

#define OKSTR     "[  OK  ]"
#define YESSTR    "[ YES  ]"
#define NOSTR     "[  NO  ]"
#define CANCELSTR "[CANCEL]"


static ColorDefinitions * color_defs = NULL;
static int                max_colors = 0;
static int                uses_color;


/* Color option's legal values */
static char *colors[] = {
        "Default",
        "Black",
        "Red",
        "Green",
        "Yellow",
        "Blue",
        "Magenta",
        "Cyan",
        "White",
        NULL
};




static int get_color_value(int name)
{
        int value;

        switch (name)
        {
                case NoColor: value = -1;               break;
                case Black:   value = COLOR_BLACK;      break;
                case Red:     value = COLOR_RED;        break;
                case Green:   value = COLOR_GREEN;      break;
                case Yellow:  value = COLOR_YELLOW;     break;
                case Blue:    value = COLOR_BLUE;       break;
                case Magenta: value = COLOR_MAGENTA;    break;
                case Cyan:    value = COLOR_CYAN;       break;
                case White:
                default:      value = COLOR_WHITE;      break;
        }
        return value;
}


static void init_color_element(int element, int fg_color, int bg_color)
{
        int fg = COLOR_WHITE, bg = -1;

        fg = get_color_value(fg_color);
        bg = get_color_value(bg_color);
        if ((use_default_colors() != OK) && (bg == -1))
        {
                if (fg != COLOR_BLACK)
                {
                        bg = COLOR_BLACK;
                }
                else
                {
                        bg = COLOR_WHITE;
                }
        }

        init_pair(element, fg, bg);
}


static void init_colors()
{
        if (has_colors() != TRUE)
        {
                uses_color = FALSE;
        }
        if (uses_color == TRUE)
        {
                int c;

                start_color();
                for (c = 0 ; c < max_colors ; c++)
                {
                        init_color_element(c,
                                        color_defs[c].fg,
                                        color_defs[c].bg);
                }
        }
}


void init_curses(int usecolors, int nbcolors, ColorDefinitions *coldefs)
{
        /* Ncurses initialization */
        use_env(TRUE);
        initscr();

#ifdef COLORIZED
        uses_color = usecolors;
        color_defs = coldefs;
        max_colors = nbcolors;
        init_colors();
#endif

        clear();
        keypad(stdscr, TRUE);
        cbreak();
        //nonl();
        noecho();
        curs_set(0);
        refresh();
}


static int convert_color(char *colorname, int *value)
{
        int status = 0;
        int i;

        if (! colorname)
        {
                return status;
        }
        if (strcasecmp(colorname, "hl") == 0)
        {
                *value = 1;
                status = 2;
        }
        if (strcasecmp(colorname, "ll") == 0)
        {
                *value = 0;
                status = 2;
        }
        for (i = 0 ; colors[i] ; i++)
        {
                if (strcasecmp(colorname, colors[i]) == 0)
                {
                        *value = i;
                        status = 1;
                        break;
                }
        }
        return status;
}


static void convert_color_element(int elt)
{
        char * data   = NULL;
        char * token  = NULL;
        char * backup = NULL;
        int    value;
        int    status;
        int    fg = color_defs[elt].fg;
        int    bg = color_defs[elt].bg;
        int    hl = color_defs[elt].hl;

        if (! color_defs[elt].str)
        {
                return;
        }
        data = xstrdup(color_defs[elt].str);
        token = strtok_r(data, "/", &backup);

        status = convert_color(token, &value);
        if ((status == 1) && value)
        {
                fg = value - 1;
        }

        token = strtok_r(NULL, "/", &backup);
        status = convert_color(token, &value);
        if ((status == 1) && value)
        {
                bg = value - 1;
        }

        token = strtok_r(NULL, "/", &backup);
        status = convert_color(token, &value);
        if (status == 2)
        {
                hl = value;
        }
        FREE(data);
        color_defs[elt].fg = fg;
        color_defs[elt].bg = bg;
        color_defs[elt].hl = hl;
}


void convert_colors()
{
        int i;

        for (i = 0 ; i < max_colors ; i++)
        {
                convert_color_element(i);
        }
}


void set_color(WINDOW *win, int element, int switch_hl)
{
        if (uses_color)
        {
                int do_hl = 1;
                wattron(win, COLOR_PAIR(element));
                if (switch_hl)
                {
                        do_hl = 0;
                }
                if (color_defs[element].hl == do_hl)
                {
                        wattron(win, A_BOLD);
                }
                else
                {
                        wattroff(win, A_BOLD);
                }
        }
}


int command_loop(Cmdslst *cmds)
{
        int i, cmd, found = -1;

        /* Lecture d'une commande au clavier */
        cmd = wgetch(stdscr);

        for (i = 0 ; cmds[i].key != 0 ; i++)
        {
                if (cmds[i].key == cmd)
                {
                        found = i;
                        break;
                }
        }

        if (found == -1)
        {
#if DEBUG_LEVEL!=0
                printlog(1, "Touche reue : %d\n", cmd);
#else
                fprintf(stderr, "");
#endif
                return CONT;
        }

        return cmds[found].fct();
}


WINDOW *create_newwin(const char *title,
                      int height, int width,
                      int starty, int startx)
{
        WINDOW *local_win;

        local_win = newwin(height, width, starty, startx);
        set_color(local_win, BORDERCOLOR, FALSE);
        box(local_win, 0 , 0);

        if (title != NULL)
        {
                wmove(local_win, 0, 2);
                set_color(local_win, BORDERCOLOR, FALSE);
                waddch(local_win, ACS_RTEE);
                wmove(local_win, 0, 2 + strlen(title) + 3);
                waddch(local_win, ACS_LTEE);
                set_color(local_win, TITLECOLOR, FALSE);
                wmove(local_win, 0, 3);
                wprintw(local_win, " %s ", title);
        }

        wrefresh(local_win);

        return local_win;
}


WINDOW *create_dialog(const char *title, int height)
{
        int x, y;

        getmaxyx(stdscr, y, x);
        return create_newwin(title, height, x - 6, (y - height) / 2, 3);
}


void destroy_win(WINDOW *local_win)
{
        wborder(local_win, ' ', ' ', ' ', ' ', ' ', ' ', ' ', ' ');
        wrefresh(local_win);
        delwin(local_win);
}


static void display_buttons(WINDOW *dialog, int buttons, int def)
{
        set_color(dialog, NORMALCOLOR, FALSE);
        switch (buttons)
        {
                case 1:
                        wattron(dialog, A_REVERSE);
                        mvwprintw(dialog, 4, 2, OKSTR);
                        wattroff(dialog, A_REVERSE);
                        break;
                case 3:
                        if (def == 3)
                        {
                                wattron(dialog, A_REVERSE);
                        }
                        mvwprintw(dialog, 4, 34, CANCELSTR);
                        if (def == 3)
                        {
                                wattroff(dialog, A_REVERSE);
                        }
                case 2:
                        if (def == 2)
                        {
                                wattron(dialog, A_REVERSE);
                        }
                        mvwprintw(dialog, 4, 18, NOSTR);
                        if (def == 2)
                        {
                                wattroff(dialog, A_REVERSE);
                        }
                        if (def == 1)
                        {
                                wattron(dialog, A_REVERSE);
                        }
                        mvwprintw(dialog, 4, 2, YESSTR);
                        if (def == 1)
                        {
                                wattroff(dialog, A_REVERSE);
                        }
                        break;
                default:
                        break;
        }
        wattroff(dialog, A_BOLD);
}


static int dialog_cmd(WINDOW *dialog, int datas, int def)
{
        unsigned int key;
        int cont = CONT, button = def;

        while (cont == CONT)
        {
                display_buttons(dialog, datas, button);
                wrefresh(dialog);
                key = wgetch(stdscr);
                switch (key)
                {
                        case KEY_UP:
                        case KEY_LEFT:
                                button--;
                                if (button == 0)
                                {
                                        button = datas;
                                }
                                break;
                        case KEY_DOWN:
                        case KEY_RIGHT:
                                button++;
                                if (button > datas)
                                {
                                        button = 1;
                                }
                                break;
                        case 10:
                        case ' ':
                                cont = STOP;
                                break;
                        default:
                                fprintf(stderr, "");
                }
        }
        return button;
}


int yes_or_no(const char *title, const char *reason, int def)
{
        WINDOW *dwin = create_dialog(title, 10);
        int res = def;

        set_color(dwin, BORDERCOLOR, FALSE);
        wrefresh(dwin);
        set_color(dwin, NORMALCOLOR, FALSE);
        mvwprintw(dwin, 2, 2, "%s", reason);
        wrefresh(dwin);
        res = dialog_cmd(dwin, 2, def);

        destroy_win(dwin);
        return res;
}


int yes_no_cancel(const char *title, const char *reason, int def)
{
        WINDOW *dwin = create_dialog(title, 10);
        int res = def;

        set_color(dwin, BORDERCOLOR, FALSE);
        wrefresh(dwin);
        set_color(dwin, NORMALCOLOR, FALSE);
        mvwprintw(dwin, 2, 2, "%s", reason);
        wrefresh(dwin);
        res = dialog_cmd(dwin, 3, def);

        destroy_win(dwin);
        return res;
}

