# Copyright (C) 2005  Network Applied Communication Laboratory Co., Ltd.
#
# This file is part of Rast.
# See the file COPYING for redistribution information.
#

require "test/unit"
require "bdb"

require "rast/xmlrpc-server-testable"

module Rast
  class DocumentTest < Test::Unit::TestCase
    include XMLRPCServerTestable

    def test_add_text
      add_text_test_simple
      add_text_test_xmlrpc_client
    end

    def add_text_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name, DB::RDWR) do |db|
        doc = db.create_document
        assert_equal(1, doc.doc_id)

        doc.add_text("本日は晴天なり")
        result = db.search("本日")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)
        result = db.search("雨天")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)

        doc.add_text("昨日は雨天なり")
        result = db.search("本日")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)
        result = db.search("昨日")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)

        doc.commit
        db.sync

        result = db.search("本日")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("本日", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)

        result = db.search("昨日")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("昨日", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)
      end
    end

    def add_text_test_xmlrpc_client
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, options)
      kick_xmlrpc_client("--read-write", db_name) do |db|
        doc = db.create_document
        assert_raise(RastError) do
          doc.doc_id
        end

        doc.add_text("本日は晴天なり")
        result = db.search("本日")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)
        result = db.search("雨天")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)

        doc.add_text("昨日は雨天なり")
        result = db.search("本日")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)
        result = db.search("昨日")
        assert_equal(0, result.num_docs)
        assert_equal(0, result.hit_count)

        doc.commit
        # db.sync

        result = db.search("本日")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("本日", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)

        result = db.search("昨日")
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("昨日", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal(nil, result.items[0].properties)
        assert_equal(1, result.items.length)
      end
    end

    def test_set_property
      set_property_test_simple
      set_property_test_default_property_value
      set_property_test_with_search_flag
      set_property_test_with_text_search_flag
      set_property_test_with_full_text_search_flag
      set_property_test_with_omit_property_flag
      set_property_test_xmlrpc_client
    end

    def set_property_test_simple
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
         ],
      }

      date = Date.new(2005, 6, 23)
      datetime = DateTime.new(2005, 6, 8, 13, 58, 4)

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      LocalDB.open(db_name, DB::RDWR) do |db|
        doc = db.create_document
        assert_equal(1, doc.doc_id)

        doc.add_text("本日は晴天なり")
        doc.set_property("title", "天気")
        doc.set_property("count", 1)
        doc.set_property("date", date)
        doc.set_property("datetime", datetime)
        doc.commit
        db.sync

        result = db.search("本日",
                           {
                             "properties" => [
                               "title", "count", "date", "datetime"
                             ]
                           })
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("本日", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal("天気", result.items[0].properties[0])
        assert_equal(1, result.items[0].properties[1])
        assert_equal(date.to_s, result.items[0].properties[2])
        assert_equal(datetime.strftime("%FT%T"), result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
        assert_equal(1, result.items.length)

        doc = db.create_document
        assert_equal(2, doc.doc_id)

        doc.add_text("本日は雨天なり")
        doc.set_property("title", "今日の天気")
        doc.set_property("count", 2)
        doc.set_property("date", date.strftime("%F"))
        doc.set_property("datetime", datetime.strftime("%FT%T"))
        doc.commit
        db.sync

        result = db.search("雨天",
                           {
                             "properties" => [
                               "title", "count", "date", "datetime"
                             ]
                           })
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("雨天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal("今日の天気", result.items[0].properties[0])
        assert_equal(2, result.items[0].properties[1])
        assert_equal(date.to_s, result.items[0].properties[2])
        assert_equal(datetime.strftime("%FT%T"), result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
        assert_equal(1, result.items.length)
      end
    end

    def set_property_test_default_property_value
      create_options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, create_options)

      text = "abc"
      title = "天気"
      date = Date.new(2005, 7, 7)
      datetime = DateTime.new(2005, 7, 7, 11, 17, 41)
      count = 10

      LocalDB.open(db_name, Rast::DB::RDWR) do |db|
        doc = db.create_document
        doc.add_text(text)
        doc.set_property("date", date)
        doc.set_property("datetime", datetime)
        doc.set_property("count", count)
        doc.commit

        doc = db.create_document
        doc.add_text(text)
        doc.set_property("title", title)
        doc.set_property("datetime", datetime)
        doc.set_property("count", count)
        doc.commit

        doc = db.create_document
        doc.add_text(text)
        doc.set_property("title", title)
        doc.set_property("date", date)
        doc.set_property("count", count)
        doc.commit

        doc = db.create_document
        doc.add_text(text)
        doc.set_property("title", title)
        doc.set_property("date", date)
        doc.set_property("datetime", datetime)
        doc.commit

        doc = db.create_document
        doc.add_text(text)
        doc.commit
      end
      LocalDB.open(db_name) do |db|
        result = db.search(text,
                           {
                             "properties" => [
                               "title", "date", "datetime", "count",
                             ]
                           })

        assert_equal(1, result.items[0].doc_id)
        assert_equal("", result.items[0].properties.title)
        assert_equal(date.to_s, result.items[0].properties.date)
        assert_equal(datetime.strftime("%FT%T"),
                     result.items[0].properties.datetime)
        assert_equal(count, result.items[0].properties.count)
        assert_equal(4, result.items[0].properties.length)

        assert_equal(2, result.items[1].doc_id)
        assert_equal(title, result.items[1].properties.title)
        assert_equal("", result.items[1].properties.date)
        assert_equal(datetime.strftime("%FT%T"),
                     result.items[1].properties.datetime)
        assert_equal(count, result.items[1].properties.count)
        assert_equal(4, result.items[1].properties.length)

        assert_equal(3, result.items[2].doc_id)
        assert_equal(title, result.items[2].properties.title)
        assert_equal(date.to_s, result.items[2].properties.date)
        assert_equal("", result.items[2].properties.datetime)
        assert_equal(count, result.items[2].properties.count)
        assert_equal(4, result.items[2].properties.length)

        assert_equal(4, result.items[3].doc_id)
        assert_equal(title, result.items[3].properties.title)
        assert_equal(date.to_s, result.items[3].properties.date)
        assert_equal(datetime.strftime("%FT%T"),
                     result.items[3].properties.datetime)
        assert_equal(0, result.items[3].properties.count)
        assert_equal(4, result.items[3].properties.length)

        assert_equal(5, result.items[4].doc_id)
        assert_equal("", result.items[4].properties.title)
        assert_equal("", result.items[4].properties.date)
        assert_equal("", result.items[4].properties.datetime)
        assert_equal(0, result.items[4].properties.count)
        assert_equal(4, result.items[4].properties.length)

        assert_equal(5, result.items.length)
      end
    end

    def set_property_test_with_search_flag
      create_options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, create_options)

      text = "abc"
      title = "天気"
      date = Date.new(2005, 7, 7)
      datetime = DateTime.new(2005, 7, 7, 11, 17, 41)
      count = 10

      LocalDB.open(db_name, Rast::DB::RDWR) do |db|
        doc = db.create_document
        doc.add_text(text)
        doc.set_property("title", title)
        doc.set_property("date", date)
        doc.set_property("datetime", datetime)
        doc.set_property("count", count)
        doc.commit
      end
      LocalDB.open(db_name) do |db|
        result = db.search("title = #{title}")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("date = #{date.strftime('%F')}")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("datetime = #{datetime.strftime('%FT%T')}")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("count = #{count}")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)
      end
    end

    def set_property_test_with_text_search_flag
      create_options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => true,
            "full_text_search" => false,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, create_options)

      text = "abc"
      title = "天気"

      LocalDB.open(db_name, Rast::DB::RDWR) do |db|
        doc = db.create_document
        doc.add_text(text)
        doc.set_property("title", title)
        doc.commit
      end
      LocalDB.open(db_name) do |db|
        result = db.search("title : 気")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)
      end
    end

    def set_property_test_with_full_text_search_flag
      create_options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => true,
            "unique" => false,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, create_options)

      text = "abc"
      title = "天気"

      LocalDB.open(db_name, Rast::DB::RDWR) do |db|
        doc = db.create_document
        doc.add_text(text)
        doc.set_property("title", title)
        doc.commit
      end
      LocalDB.open(db_name) do |db|
        result = db.search("abc")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("天気")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        assert_equal(text, db.get_text(1))
      end
    end

    def set_property_test_with_omit_property_flag
      create_options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
            "omit_property" => true,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
            "omit_property" => true,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
            "omit_property" => true,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => true,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
            "omit_property" => true,
          },
        ],
      }
      db_name = generate_db_name
      LocalDB.create(db_name, create_options)

      text = "abc"
      title = "天気"
      date = Date.new(2005, 7, 7)
      datetime = DateTime.new(2005, 7, 7, 11, 17, 41)
      count = 10

      LocalDB.open(db_name, Rast::DB::RDWR) do |db|
        doc = db.create_document
        doc.add_text(text)
        doc.set_property("title", title)
        doc.set_property("date", date)
        doc.set_property("datetime", datetime)
        doc.set_property("count", count)
        doc.commit
      end
      properties_db_name = File.join(db_name, "properties.db")
      BDB::Btree.open(properties_db_name, nil, 0) do |db|
        assert_equal(1 + 4, db[[1].pack("I")].length)
      end
      LocalDB.open(db_name) do |db|
        result = db.search("title = #{title}")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("date = #{date.strftime('%F')}")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("datetime = #{datetime.strftime('%FT%T')}")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)

        result = db.search("count = #{count}")
        assert_equal(1, result.items[0].doc_id)
        assert_equal(1, result.items.length)
      end
    end

    def set_property_test_xmlrpc_client
      options = {
        "encoding" => "utf8",
        "preserve_text" => true,
        "properties" => [
          {
            "name" => "title",
            "type" => Rast::PROPERTY_TYPE_STRING,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "date",
            "type" => Rast::PROPERTY_TYPE_DATE,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "datetime",
            "type" => Rast::PROPERTY_TYPE_DATETIME,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
          {
            "name" => "count",
            "type" => Rast::PROPERTY_TYPE_UINT,
            "search" => false,
            "text_search" => false,
            "full_text_search" => false,
            "unique" => false,
          },
         ],
      }

      date = Date.new(2005, 6, 23)
      datetime = DateTime.new(2005, 6, 8, 13, 58, 4)

      db_name = generate_db_name
      LocalDB.create(db_name, options)
      kick_xmlrpc_client("--read-write", db_name) do |db|
        doc = db.create_document

        doc.add_text("本日は晴天なり")
        doc.set_property("title", "天気")
        doc.set_property("count", 1)
        doc.set_property("date", date)
        doc.set_property("datetime", datetime)
        doc.commit
        # db.sync

        result = db.search("本日",
                           {
                             "properties" => [
                               "title", "count", "date", "datetime"
                             ]
                           })
        assert_equal(1, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("本日", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(1, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal("天気", result.items[0].properties[0])
        assert_equal(1, result.items[0].properties[1])
        assert_equal(date.to_s, result.items[0].properties[2])
        assert_equal(datetime.strftime("%FT%T"), result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
        assert_equal(1, result.items.length)

        doc = db.create_document

        doc.add_text("本日は雨天なり")
        doc.set_property("title", "今日の天気")
        doc.set_property("count", 2)
        doc.set_property("date", date.strftime("%F"))
        doc.set_property("datetime", datetime.strftime("%FT%T"))
        doc.commit
        # db.sync

        result = db.search("雨天",
                           {
                             "properties" => [
                               "title", "count", "date", "datetime"
                             ]
                           })
        assert_equal(2, result.num_docs)
        assert_equal(1, result.hit_count)
        assert_equal("雨天", result.terms[0].term)
        assert_equal(1, result.terms[0].doc_count)
        assert_equal(1, result.terms.length)
        assert_equal(2, result.items[0].doc_id)
        assert_equal(0, result.items[0].db_index)
        assert_equal("今日の天気", result.items[0].properties[0])
        assert_equal(2, result.items[0].properties[1])
        assert_equal(date.to_s, result.items[0].properties[2])
        assert_equal(datetime.strftime("%FT%T"), result.items[0].properties[3])
        assert_equal(4, result.items[0].properties.length)
        assert_equal(1, result.items.length)
      end
    end
  end
end
