/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_bulletin_board.c,v 1.37.2.7 2005/08/15 17:15:07 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha_bulletin_board.h"
#include "ochusha_bbs_thread.h"
#include "utils.h"

#include "marshal.h"

#include <glib.h>
#include <libxml/SAX.h>
#include <libxml/parser.h>

#include <errno.h>
#include <fcntl.h>

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <unistd.h>

#include <zlib.h>


static void ochusha_bulletin_board_class_init(OchushaBulletinBoardClass *klass);
static void ochusha_bulletin_board_init(OchushaBulletinBoard *board);
static void ochusha_bulletin_board_finalize(GObject *object);

static void ochusha_bulletin_board_set_property(GObject *object, guint prop_id,
						const GValue *value,
						GParamSpec *pspec);
static void ochusha_bulletin_board_get_property(GObject *object, guint prop_id,
						GValue *value,
						GParamSpec *pspec);
static char *ochusha_bulletin_board_generate_board_id(
						OchushaBulletinBoard *board,
						const char *url);

static void ochusha_bulletin_board_read_boardlist_element(
						OchushaBulletinBoard *board,
						GHashTable *board_attributes);
static void ochusha_bulletin_board_write_boardlist_element(
						OchushaBulletinBoard *board,
						gzFile boardlist_xml);

GType
ochusha_bulletin_board_get_type(void)
{
  static GType bulletin_board_type = 0;

  if (bulletin_board_type == 0)
    {
      static const GTypeInfo bulletin_board_info =
	{
	  sizeof(OchushaBulletinBoardClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_bulletin_board_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaBulletinBoard),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_bulletin_board_init,
	};

      bulletin_board_type = g_type_register_static(G_TYPE_OBJECT,
						   "OchushaBulletinBoard",
						   &bulletin_board_info, 0);
    }

  return bulletin_board_type;
}


enum {
  PROP_0,
  PROP_NAME,
  PROP_BASE_URL
};


enum {
  READ_BOARDLIST_ELEMENT_SIGNAL,
  WRITE_BOARDLIST_ELEMENT_SIGNAL,
  THREADLIST_READ_THREAD_ELEMENT_SIGNAL,
  THREADLIST_WRITE_THREAD_ELEMENT_SIGNAL,
  LAST_SIGNAL
};


static GObjectClass *parent_class = NULL;
static int bulletin_board_signals[LAST_SIGNAL] = { 0, 0 };


static void
ochusha_bulletin_board_class_init(OchushaBulletinBoardClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->set_property = ochusha_bulletin_board_set_property;
  o_class->get_property = ochusha_bulletin_board_get_property;
  o_class->finalize = ochusha_bulletin_board_finalize;

  g_object_class_install_property(o_class,
				  PROP_NAME,
				  g_param_spec_string("name",
						      _("Name"),
						      _("The Name of the Board"),
						      _("Unknown Board"),
						      G_PARAM_READWRITE));
  g_object_class_install_property(o_class,
				  PROP_BASE_URL,
				  g_param_spec_string("base_url",
						      _("Name"),
						      _("The URL of the Board"),
						      "http://www.2ch.net/",
						      G_PARAM_READWRITE));


  bulletin_board_signals[READ_BOARDLIST_ELEMENT_SIGNAL] =
    g_signal_new("read_boardlist_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(OchushaBulletinBoardClass,
				 read_boardlist_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  bulletin_board_signals[WRITE_BOARDLIST_ELEMENT_SIGNAL] =
    g_signal_new("write_boardlist_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_FIRST,
		 G_STRUCT_OFFSET(OchushaBulletinBoardClass,
				 write_boardlist_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__POINTER,
		 G_TYPE_NONE, 1,
		 G_TYPE_POINTER);
  bulletin_board_signals[THREADLIST_READ_THREAD_ELEMENT_SIGNAL] =
    g_signal_new("threadlist_read_thread_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBulletinBoardClass,
				 threadlist_read_thread_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BBS_THREAD,
		 G_TYPE_POINTER);
  bulletin_board_signals[THREADLIST_WRITE_THREAD_ELEMENT_SIGNAL] =
    g_signal_new("threadlist_write_thread_element",
		 G_TYPE_FROM_CLASS(klass),
		 G_SIGNAL_RUN_LAST,
		 G_STRUCT_OFFSET(OchushaBulletinBoardClass,
				 threadlist_write_thread_element),
		 NULL, NULL,
		 libochusha_marshal_VOID__OBJECT_POINTER,
		 G_TYPE_NONE, 2,
		 OCHUSHA_TYPE_BBS_THREAD,
		 G_TYPE_POINTER);

  klass->read_boardlist_element
    = ochusha_bulletin_board_read_boardlist_element;
  klass->write_boardlist_element
    = ochusha_bulletin_board_write_boardlist_element;
  klass->threadlist_read_thread_element = NULL;
  klass->threadlist_write_thread_element = NULL;

  klass->recover_threadlist = NULL;
  klass->generate_base_path = NULL;
  klass->generate_board_id = ochusha_bulletin_board_generate_board_id;
  klass->server_changed = NULL;
  klass->thread_new = NULL;
  klass->lookup_thread_by_url = NULL;
  klass->lookup_kako_thread_by_url = NULL;
  klass->get_threadlist_source = NULL;
  klass->refresh_threadlist = NULL;
  klass->preview_new_thread = NULL;
  klass->new_thread_supported = NULL;
  klass->create_new_thread = NULL;
}


static void
ochusha_bulletin_board_init(OchushaBulletinBoard *board)
{
  board->name = NULL;
  board->base_url = NULL;
  board->server = NULL;
  board->base_path = NULL;
  board->id = NULL;

  board->thread_list = NULL;
  board->thread_table
    = g_hash_table_new_full(g_str_hash, g_str_equal, NULL,
			    (GDestroyNotify)OCHU_OBJECT_UNREF_FUNC);
  board->dropped_list = NULL;

  board->bbs_type = OCHUSHA_BBS_TYPE_UNKNOWN;

  board->monitor = ochusha_monitor_new(NULL);
}


static void
ochusha_bulletin_board_finalize(GObject *object)
{
  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(object);

  if (board->name != NULL)
    {
      G_FREE(board->name);
      board->name = NULL;
    }

  if (board->base_url != NULL)
    {
      G_FREE(board->base_url);
      board->base_url = NULL;
    }

  if (board->server != NULL)
    {
      G_FREE(board->server);
      board->server = NULL;
    }

  if (board->base_path != NULL)
    {
      G_FREE(board->base_path);
      board->base_path = NULL;
    }

  if (board->id != NULL)
    {
      G_FREE(board->id);
      board->id = NULL;
    }

  if (board->thread_list != NULL)
    {
      g_slist_foreach(board->thread_list, (GFunc)OCHU_OBJECT_UNREF_FUNC, NULL);
      g_slist_free(board->thread_list);
      board->thread_list = NULL;
    }

  if (board->thread_table != NULL)
    {
      g_hash_table_destroy(board->thread_table);
      board->thread_table = NULL;
    }

  if (board->dropped_list != NULL)
    {
      g_slist_foreach(board->dropped_list,
		      (GFunc)OCHU_OBJECT_UNREF_FUNC, NULL);
      g_slist_free(board->dropped_list);
      board->dropped_list = NULL;
    }

  if (board->monitor != NULL)
    {
      ochusha_monitor_free(board->monitor);
      board->monitor = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
ochusha_bulletin_board_set_property(GObject *object, guint prop_id,
				    const GValue *value, GParamSpec *pspec)
{
  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(object);

  switch (prop_id)
    {
    case PROP_NAME:
      ochusha_bulletin_board_set_name(board, g_value_get_string(value));
      break;

    case PROP_BASE_URL:
      ochusha_bulletin_board_set_base_url(board, g_value_get_string(value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}


static void
ochusha_bulletin_board_get_property(GObject *object, guint prop_id,
				    GValue *value, GParamSpec *pspec)
{
  OchushaBulletinBoard *board = OCHUSHA_BULLETIN_BOARD(object);

  switch (prop_id)
    {
    case PROP_NAME:
      g_value_set_string(value, board->name);
      break;

    case PROP_BASE_URL:
      g_value_set_string(value, board->base_url);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}


OchushaBulletinBoard *
ochusha_bulletin_board_new(const gchar *name, const char *base_url)
{
  g_assert(name != NULL && base_url != NULL);

  return OCHUSHA_BULLETIN_BOARD(g_object_new(OCHUSHA_TYPE_BULLETIN_BOARD,
					     "name", name,
					     "base_url", base_url,
					     NULL));
}


void
ochusha_bulletin_board_set_name(OchushaBulletinBoard *board,
				const gchar *name)
{
  g_return_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && name != NULL);

  if (board->name != NULL)
    g_free(board->name);

  board->name = wipe_string(name);
  g_object_notify(G_OBJECT(board), "name");
}


const gchar *
ochusha_bulletin_board_get_name(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), NULL);

  return board->name;
}


void
ochusha_bulletin_board_set_base_url(OchushaBulletinBoard *board,
				    const char *url)
{
  OchushaBulletinBoardClass *klass;
  char *prev_server;
  char *server;
  char *base_path;

  g_return_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && url != NULL);

  server = ochusha_utils_url_extract_http_server(url);
  g_return_if_fail(server != NULL);

  if (board->base_url != NULL)
    G_FREE(board->base_url);
  board->base_url = G_STRDUP(url);

  prev_server = board->server;
  board->server = server;

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);

  if (prev_server != NULL && strcmp(prev_server, server) != 0
      && *klass->server_changed != NULL)
    (*klass->server_changed)(board, prev_server);

  if (prev_server != NULL)
    G_FREE(prev_server);

  if (*klass->generate_base_path)
    base_path = (*klass->generate_base_path)(board, url);
  else
    base_path = ochusha_utils_url_extract_http_absolute_path(url);
  g_return_if_fail(base_path != NULL);

  if (board->base_path != NULL)
    G_FREE(board->base_path);
  board->base_path = base_path;

  if (board->id != NULL)
    G_FREE(board->id);

  board->id = (*klass->generate_board_id)(board, url);
  g_object_notify(G_OBJECT(board), "base_url");
}


void
ochusha_bulletin_board_process_board_move(OchushaBulletinBoard *board,
					  const char *new_url,
					  OchushaConfig *config)
{
  char *old_base_url = G_STRDUP(board->base_url);
  char *old_path = NULL;
  char *new_path = NULL;
  char tmp_path[PATH_MAX];
  char *tmp_pos;
  struct stat sb;

  /* åǥ쥯ȥΰư */
  /* 0.5ϤǤϤad-hocˤ롣*/
  snprintf(tmp_path, PATH_MAX, "cache/%s%s%s/" OCHUSHA_THREADLIST_XML,
	   ochusha_bulletin_board_get_server(board),
	   ochusha_bulletin_board_get_base_path(board),
	   ochusha_bulletin_board_get_id(board));
  old_path = ochusha_config_find_file(config, tmp_path, NULL);

  ochusha_bulletin_board_set_base_url(board, new_url);

  snprintf(tmp_path, PATH_MAX, "cache/%s%s%s/" OCHUSHA_THREADLIST_XML,
	   ochusha_bulletin_board_get_server(board),
	   ochusha_bulletin_board_get_base_path(board),
	   ochusha_bulletin_board_get_id(board));
  new_path = ochusha_config_find_file(config, tmp_path, NULL);

  if (new_path == NULL && old_path != NULL)
    {
      snprintf(tmp_path, PATH_MAX, "%s/cache/%s%s%s/" OCHUSHA_THREADLIST_XML,
	       config->home,
	       ochusha_bulletin_board_get_server(board),
	       ochusha_bulletin_board_get_base_path(board),
	       ochusha_bulletin_board_get_id(board));
      tmp_pos = strstr(tmp_path, "/threadlist.xml");
      *tmp_pos = '\0';
      if (mkdir_p(tmp_path))
	{
	  *tmp_pos = '/';
	  rename(old_path, tmp_path);
#if 0
	  fprintf(stderr, "%s is moved to %s\n", old_path, tmp_path);
#endif
	}
      new_path = G_STRDUP(tmp_path);
    }

  if (old_path != NULL)
    {
      tmp_pos = strstr(old_path, "/threadlist.xml");
      tmp_pos[1] = 'd';
      tmp_pos[2] = 'a';
      tmp_pos[3] = 't';
      tmp_pos[4] = '\0';
      if (stat(old_path, &sb) == 0)
	{
	  tmp_pos = strstr(new_path, "/threadlist.xml");
	  tmp_pos[1] = 'd';
	  tmp_pos[2] = 'a';
	  tmp_pos[3] = 't';
	  tmp_pos[4] = '\0';
	  if (stat(new_path, &sb) != 0 && errno == ENOENT)
	    {
	      rename(old_path, new_path);
#if 0
	      fprintf(stderr, "%s is moved to %s\n", old_path, new_path);
#endif
	    }
	}
    }

  if (old_path != NULL)
    G_FREE(old_path);

  if (new_path != NULL)
    G_FREE(new_path);

  G_FREE(old_base_url);
}


void
ochusha_bulletin_board_set_bbs_type(OchushaBulletinBoard *board,
				    int bbs_type)
{
  OchushaBulletinBoardClass *klass;
  char *url = board->base_url;
  char *server;
  char *base_path;

  g_return_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board));

  board->bbs_type = bbs_type;

  server = ochusha_utils_url_extract_http_server(url);
  g_return_if_fail(server != NULL);

  if (board->server != NULL)
    G_FREE(board->server);
  board->server = server;

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);

  if (*klass->generate_base_path)
    base_path = (*klass->generate_base_path)(board, url);
  else
    base_path = ochusha_utils_url_extract_http_absolute_path(url);
  g_return_if_fail(base_path != NULL);

  if (board->base_path != NULL)
    G_FREE(board->base_path);
  board->base_path = base_path;

  if (board->id != NULL)
    G_FREE(board->id);

  board->id = (*klass->generate_board_id)(board, url);
}


int
ochusha_bulletin_board_get_bbs_type(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board),
		       OCHUSHA_BBS_TYPE_UNKNOWN);
  return board->bbs_type;
}


static char *
ochusha_bulletin_board_generate_board_id(OchushaBulletinBoard *board,
					 const char *url)
{
  g_warning("A sub class of OchushaBulletinBoard didn't implement generate_board_id()\n");
  return NULL;
}


const char *
ochusha_bulletin_board_get_base_url(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), NULL);

  return board->base_url;
}


const char *
ochusha_bulletin_board_get_server(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), NULL);

  return board->server;
}


const char *
ochusha_bulletin_board_get_base_path(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), NULL);

  return board->base_path;
}


const char *
ochusha_bulletin_board_get_id(OchushaBulletinBoard *board)
{
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), NULL);

  return board->id;
}


void
ochusha_bulletin_board_lock_thread_list(OchushaBulletinBoard *board)
{
  g_assert(OCHUSHA_IS_BULLETIN_BOARD(board));

  ochusha_monitor_enter(board->monitor);
}


gboolean
ochusha_bulletin_board_trylock_thread_list(OchushaBulletinBoard *board)
{
  g_assert(OCHUSHA_IS_BULLETIN_BOARD(board));

  return ochusha_monitor_try_enter(board->monitor);
}


void
ochusha_bulletin_board_unlock_thread_list(OchushaBulletinBoard *board)
{
  g_assert(OCHUSHA_IS_BULLETIN_BOARD(board));

  ochusha_monitor_exit(board->monitor);
}


OchushaBBSThread *
ochusha_bulletin_board_bbs_thread_new(OchushaBulletinBoard *board,
				      const char *id, const gchar *title)
{
  OchushaBBSThread *thread;
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && id != NULL,
		       NULL);

  thread = g_hash_table_lookup(board->thread_table, id);
  if (thread != NULL)
    return thread;

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  g_return_val_if_fail(klass->thread_new != NULL, NULL);

  thread = (*klass->thread_new)(board, id, title);

  OCHU_OBJECT_REF(thread);
  if (thread != NULL)
    g_hash_table_insert(board->thread_table, thread->id, thread);
    
  return thread;
}


OchushaBBSThread *
ochusha_bulletin_board_lookup_bbs_thread_by_id(OchushaBulletinBoard *board,
					       const char *id)
{
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && id != NULL,
		       NULL);

  return g_hash_table_lookup(board->thread_table, id);
}


OchushaBBSThread *
ochusha_bulletin_board_lookup_bbs_thread_by_url(OchushaBulletinBoard *board,
						OchushaNetworkBroker *broker,
						const char *url)
{
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && url != NULL,
		       NULL);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  g_return_val_if_fail(klass->lookup_thread_by_url != NULL, NULL);

  return (*klass->lookup_thread_by_url)(board, broker, url);
}


OchushaBBSThread *
ochusha_bulletin_board_lookup_kako_thread_by_url(OchushaBulletinBoard *board,
						 OchushaNetworkBroker *broker,
						 const char *url)
{
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && url != NULL,
		       NULL);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  g_return_val_if_fail(klass->lookup_kako_thread_by_url != NULL, NULL);

  return (*klass->lookup_kako_thread_by_url)(board, broker, url);
}


OchushaAsyncBuffer *
ochusha_bulletin_board_get_threadlist_source(OchushaBulletinBoard *board,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode)
{
  OchushaBulletinBoardClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board)
		       && OCHUSHA_IS_NETWORK_BROKER(broker), NULL);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  g_return_val_if_fail(klass->get_threadlist_source != NULL, NULL);

  return (*klass->get_threadlist_source)(board, broker, buffer, mode);
}


gboolean
ochusha_bulletin_board_refresh_threadlist(OchushaBulletinBoard *board,
					  OchushaAsyncBuffer *buffer,
					  EachThreadCallback *each_thread_cb,
					  StartParsingCallback *start_parsing_cb,
					  BeforeWaitCallback *before_wait_cb,
					  AfterWaitCallback *after_wait_cb,
					  EndParsingCallback *end_parsing_cb,
					  gpointer callback_data)
{
  OchushaBulletinBoardClass *klass;
  gboolean result;
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && buffer != NULL,
		       FALSE);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  g_return_val_if_fail(klass->get_threadlist_source != NULL, FALSE);

  ochusha_bulletin_board_lock_thread_list(board);
  result = (*klass->refresh_threadlist)(board, buffer, each_thread_cb,
					start_parsing_cb, before_wait_cb,
					after_wait_cb, end_parsing_cb,
					callback_data);
  ochusha_bulletin_board_unlock_thread_list(board);

  return result;
}


const char *
ochusha_bulletin_board_get_response_character_encoding(
						OchushaBulletinBoard *board)
{
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_BULLETIN_BOARD(board), NULL);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  g_return_val_if_fail(klass->get_response_character_encoding != NULL, NULL);

  return (*klass->get_response_character_encoding)(board);
}


iconv_helper *
ochusha_bulletin_board_get_response_iconv_helper(OchushaBulletinBoard *board)
{
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_BULLETIN_BOARD(board), NULL);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  g_return_val_if_fail(klass->get_response_iconv_helper != NULL, NULL);

  return (*klass->get_response_iconv_helper)(board);
}


gboolean
ochusha_bulletin_board_is_new_thread_preview_supported(
						OchushaBulletinBoard *board)
{
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), FALSE);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  return klass->preview_new_thread != NULL;
}


gboolean
ochusha_bulletin_board_preview_new_thread(OchushaBulletinBoard *board,
					  const gchar *title,
					  const OchushaBBSResponse *response,
					  StartThreadCallback *start_cb,
					  EachResponseCallback *response_cb,
					  EndThreadCallback *end_cb,
					  gpointer callback_data)
{
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board)
		       && title != NULL && response != NULL, FALSE);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);

  return (*klass->preview_new_thread)(board, title, response,
				      start_cb, response_cb, end_cb,
				      callback_data);
}


gboolean
ochusha_bulletin_board_is_new_thread_supported(OchushaBulletinBoard *board)
{
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), FALSE);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
  if (klass->new_thread_supported != NULL)
    return (*klass->new_thread_supported)(board);

  return FALSE;
}


gboolean
ochusha_bulletin_board_create_new_thread(OchushaBulletinBoard *board,
					 OchushaNetworkBroker *broker,
					 const gchar *title,
					 const OchushaBBSResponse *response)
{
  OchushaBulletinBoardClass *klass;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board)
		       && OCHUSHA_IS_NETWORK_BROKER(broker)
		       && title != NULL && response != NULL,
		       FALSE);

  klass = OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);

  g_return_val_if_fail(klass->create_new_thread != NULL, FALSE);

  return (*klass->create_new_thread)(board, broker, title, response);
}


/*
 * boardlist.xmlϢ
 */
#define OUTPUT_BOARD_ATTRIBUTE_BOOLEAN(gzfile, board, attribute)	\
  do									\
    {									\
      if ((board)->attribute)						\
	{								\
	  gzprintf(gzfile,						\
		  "        <attribute name=\"" #attribute "\">\n"	\
		  "          <boolean val=\"%s\"/>\n"			\
		  "        </attribute>\n",				\
		  (board)->attribute ? "true" : "false");		\
	}								\
    } while (0)


#define OUTPUT_BOARD_ATTRIBUTE_INT(gzfile, board, attribute)		\
  do									\
    {									\
      if ((board)->attribute)						\
	{								\
	  gzprintf(gzfile,						\
		  "        <attribute name=\"" #attribute "\">\n"	\
		  "          <int val=\"%d\"/>\n"			\
		  "        </attribute>\n", (board)->attribute);	\
	}								\
    } while (0)


#define OUTPUT_BOARD_ATTRIBUTE_STRING(gzfile, board, attribute)		\
  do									\
    {									\
      if ((board)->attribute)						\
	{								\
	  gchar *text = g_markup_escape_text((board)->attribute, -1);	\
	  gzprintf(gzfile,						\
		  "        <attribute name=\"" #attribute "\">\n"	\
		  "          <string>%s</string>\n"			\
		  "        </attribute>\n", text);			\
	  g_free(text);							\
	}								\
  } while (0)


static void
ochusha_bulletin_board_read_boardlist_element(OchushaBulletinBoard *board,
					      GHashTable *board_attributes)
{
  board->hidden = ochusha_utils_get_attribute_boolean(board_attributes,
						      "hidden");
  board->post_mode = ochusha_utils_get_attribute_int(board_attributes,
						     "post_mode");
}


static void
ochusha_bulletin_board_write_boardlist_element(OchushaBulletinBoard *board,
					       gzFile boardlist_xml)
{
  OUTPUT_BOARD_ATTRIBUTE_INT(boardlist_xml, board, bbs_type);
  OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, board, name);
  OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, board, base_url);
  OUTPUT_BOARD_ATTRIBUTE_BOOLEAN(boardlist_xml, board, hidden);
  OUTPUT_BOARD_ATTRIBUTE_INT(boardlist_xml, board, post_mode);
}


typedef enum
{
  SAX_INITIAL,
  SAX_OCHUSHA,
  SAX_THREADLIST,
  SAX_THREAD,
  SAX_THREAD_ATTRIBUTE,
  SAX_THREAD_ATTRIBUTE_BOOLEAN,
  SAX_THREAD_ATTRIBUTE_INT,
  SAX_THREAD_ATTRIBUTE_STRING,
  SAX_ACCEPTED,
  SAX_ERROR
} SAXState;


typedef struct _SAXContext
{
  SAXState state;

  OchushaBulletinBoard *board;

  char *current_attr_name;
  char *current_attr_val;

  GHashTable *thread_attributes;
} SAXContext;


static void
cleanup_sax_context(SAXContext *context)
{
  if (context->current_attr_name != NULL)
    {
      G_FREE(context->current_attr_name);
      context->current_attr_name = NULL;
    }

  if (context->current_attr_val != NULL)
    {
      G_FREE(context->current_attr_val);
      context->current_attr_val = NULL;
    }

  if (context->thread_attributes != NULL)
    {
      g_hash_table_destroy(context->thread_attributes);
      context->thread_attributes = NULL;
    }
}


#if TRACE_MEMORY_USAGE
static void
trace_free(gpointer pointer)
{
  G_FREE(pointer);
}
#define TRACE_FREE	trace_free
#else
#define TRACE_FREE	g_free
#endif


static void
startElementHandler(void *context, const xmlChar *name, const xmlChar **atts)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_INITIAL:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_OCHUSHA; return; }
      break;	/* 顼 */

    case SAX_OCHUSHA:
      if (strcmp(name, "threadlist") == 0)
	{
	  sax_context->thread_attributes
	    = g_hash_table_new_full(g_str_hash, g_str_equal,
				    TRACE_FREE, TRACE_FREE);
	  sax_context->state = SAX_THREADLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_THREAD:
      if (strcmp(name, "attribute") == 0
	  && atts != NULL && strcmp(atts[0], "name") == 0)
	{
	  sax_context->state = SAX_THREAD_ATTRIBUTE;
	  if (sax_context->current_attr_val != NULL)
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "unexpected attribute found: %s=%s\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val);
#endif
	      G_FREE(sax_context->current_attr_name);
	      G_FREE(sax_context->current_attr_val);
	      sax_context->current_attr_name = NULL;
	      sax_context->current_attr_val = NULL;
	      break;	/* 顼 */
	    }

	  sax_context->current_attr_name = G_STRDUP(atts[1]);
	  return;
	}
      break;	/* 顼 */

    case SAX_THREAD_ATTRIBUTE:
      if (atts != NULL && strcmp(atts[0], "val") == 0)
	{
	  /* int/booleanβǽ */
	  if (strcmp(name, "int") == 0)
	    sax_context->state = SAX_THREAD_ATTRIBUTE_INT;
	  else if (strcmp(name, "boolean") == 0)
	    sax_context->state = SAX_THREAD_ATTRIBUTE_BOOLEAN;
	  else
	    {
#if DEBUG_SAX_HANDLER
	      fprintf(stderr, "element unexpected in state(%d) found: %s\n",
		      sax_context->state, name);
#endif
	      break;	/* 顼 */
	    }

	  if (sax_context->current_attr_val != NULL)
	    {
	      /* 顼ǤɤΤġ̯ */
#if DEBUG_SAX_HANDLER
	      fprintf(stderr,
		      "attribute %s=\"%s\" is overwritten by \"%s\"!\n",
		      sax_context->current_attr_name,
		      sax_context->current_attr_val, atts[1]);
#endif
	      G_FREE(sax_context->current_attr_val);
	    }

	  sax_context->current_attr_val = G_STRDUP(atts[1]);
	  return;
	}
      else if (strcmp(name, "string") == 0)
	{
	  sax_context->state = SAX_THREAD_ATTRIBUTE_STRING;
	  return;
	}
      break;	/* 顼 */

    case SAX_THREADLIST:
      if (strcmp(name, "thread") == 0)
	{ sax_context->state = SAX_THREAD; return; }
      break;	/* 顼 */

    case SAX_THREAD_ATTRIBUTE_INT:
    case SAX_THREAD_ATTRIBUTE_STRING:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
      fprintf(stderr, "startHandler is called in unknown state: %d\n",
	      sax_context->state);
    }
  sax_context->state = SAX_ERROR;
}


static gboolean
hash_table_cleanup_func(gpointer key, gpointer value, gpointer unused)
{
  return TRUE;
}


static void
hash_table_cleanup(GHashTable *hash_table)
{
  g_hash_table_foreach_remove(hash_table, hash_table_cleanup_func, NULL);
}


static void
endElementHandler(void *context, const xmlChar *name)
{
  SAXContext *sax_context = (SAXContext *)context;

  switch (sax_context->state)
    {
    case SAX_OCHUSHA:
      if (strcmp(name, "ochusha") == 0)
	{ sax_context->state = SAX_ACCEPTED; return; }
      break;	/* 顼 */

    case SAX_THREAD_ATTRIBUTE:
      if (strcmp(name, "attribute") == 0)
	{
	  GHashTable *hash_table = sax_context->thread_attributes;
	  sax_context->state = SAX_THREAD;
#if DEBUG_SAX_HANDLER_NEW
	  fprintf(stderr, "%s = \"%s\"\n", sax_context->current_attr_name,
		  sax_context->current_attr_val);
#endif
	  g_hash_table_insert(hash_table,
			      sax_context->current_attr_name,
			      sax_context->current_attr_val);
	  sax_context->current_attr_name = NULL;
	  sax_context->current_attr_val = NULL;
	  return;
	}
      break;	/* 顼 */

    case SAX_THREAD_ATTRIBUTE_STRING:
      if (strcmp(name, "string") == 0)
	{
	  sax_context->state = SAX_THREAD_ATTRIBUTE;
	  if (sax_context->current_attr_val == NULL)
	    sax_context->current_attr_val = G_STRDUP("");
	  return;
	}
      break;	/* 顼 */

    case SAX_THREADLIST:
      if (strcmp(name, "threadlist") == 0)
	{
	  g_hash_table_destroy(sax_context->thread_attributes);
	  sax_context->thread_attributes = NULL;

	  sax_context->state = SAX_OCHUSHA;
	  return;
	}
      break;	/* 顼 */

    case SAX_THREAD:
      if (strcmp(name, "thread") == 0)
	{
	  OchushaBulletinBoard *board = sax_context->board;
	  GHashTable *thread_attrs = sax_context->thread_attributes;

	  char *thread_id = g_hash_table_lookup(thread_attrs, "id");

	  /* for backward compatibility */
	  if (thread_id == NULL)
	    thread_id = g_hash_table_lookup(thread_attrs, "dat_filename");

	  if (thread_id != NULL)
	    {
	      OchushaBBSThread *thread;

	      /* for backward compatibility */
	      char *tmp_id;
	      char *tmp_pos;
	      if ((tmp_pos = strstr(thread_id, ".dat")) != NULL)
		{
		  tmp_id = G_STRNDUP(thread_id, tmp_pos - thread_id);
		  thread_id = tmp_id;
		}
	      else
		tmp_id = NULL;

	      thread = ochusha_bulletin_board_lookup_bbs_thread_by_id(board,
								thread_id);

	      if (thread == NULL)
		{
		  const gchar *title = g_hash_table_lookup(thread_attrs,
							   "title");
		  if (title != NULL)
		    {
		      thread
			= ochusha_bulletin_board_bbs_thread_new(board,
								thread_id,
								title);
		      if (thread != NULL)
			{
			  g_signal_emit_by_name(G_OBJECT(thread),
						"read_threadlist_element",
						thread_attrs);
			  g_signal_emit(G_OBJECT(board),
					bulletin_board_signals[THREADLIST_READ_THREAD_ELEMENT_SIGNAL],
					0,
					thread,
					thread_attrs);
			  board->thread_list
			    = g_slist_prepend(board->thread_list, thread);
			}
		    }
		}
	      if (tmp_id != NULL)
		G_FREE(tmp_id);
	    }
	   /* ꡼󥢥å */
	  hash_table_cleanup(thread_attrs);
	  sax_context->state = SAX_THREADLIST;
	  return;
	}
      break;	/* 顼 */

    case SAX_THREAD_ATTRIBUTE_BOOLEAN:
      if (strcmp(name, "boolean") == 0)
	{ sax_context->state = SAX_THREAD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_THREAD_ATTRIBUTE_INT:
      if (strcmp(name, "int") == 0)
	{ sax_context->state = SAX_THREAD_ATTRIBUTE; return; }
      break;	/* 顼 */

    case SAX_INITIAL:
    case SAX_ACCEPTED:
    case SAX_ERROR:
      break;	/* 顼 */

    default:
#if DEBUG_SAX_HANDLER
      fprintf(stderr, "endHandler called in unknown state: %d.\n",
	      sax_context->state);
#endif
      break;	/* 顼 */
    }
#if DEBUG_SAX_HANDLER
  fprintf(stderr, "Invalid document: </%s> appeared in state=%d\n",
	  name, sax_context->state);
#endif
  sax_context->state = SAX_ERROR;
}


static void
charactersHandler(void *context, const xmlChar *ch, int len)
{
  SAXContext *sax_context = (SAXContext *)context;

  if (sax_context->state == SAX_THREAD_ATTRIBUTE_STRING)
    {
      if (sax_context->current_attr_val == NULL)
	sax_context->current_attr_val = G_STRNDUP(ch, len);
      else
	{
	  int old_len = strlen(sax_context->current_attr_val);
	  sax_context->current_attr_val
	    = G_REALLOC(sax_context->current_attr_val, old_len + len + 1);
	  strncat(sax_context->current_attr_val + old_len, ch, len);
	}
    }
}


static void
nopHandler(void *context)
{
}


static xmlEntityPtr
getEntityHandler(void *context, const xmlChar *name)
{
  return NULL;
}


gboolean
ochusha_bulletin_board_read_threadlist_xml(OchushaBulletinBoard *board,
					   OchushaConfig *config,
					   const char *subdir,
					   gboolean recover_mode)
{
  SAXContext context =
    {
      SAX_INITIAL,	/* state */
      board,		/* board */
      NULL, NULL,	/* current_attr_name, current_attr_val */
      NULL		/* thread_attributes */
    };
  xmlSAXHandler sax_handler;
  char pathname[PATH_MAX];
  char *threadlist_xml;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && config != NULL,
		       FALSE);

#if 0
  fprintf(stderr, "server: %s\n", ochusha_bulletin_board_get_server(board));
  fprintf(stderr, "base_path: %s\n", ochusha_bulletin_board_get_base_path(board));
  fprintf(stderr, "id: %s\n", ochusha_bulletin_board_get_id(board));
#endif

  if (snprintf(pathname, PATH_MAX, "%s%s%s/%s",
	       ochusha_bulletin_board_get_server(board),
	       ochusha_bulletin_board_get_base_path(board),
	       ochusha_bulletin_board_get_id(board),
	       OCHUSHA_THREADLIST_XML) >= PATH_MAX)
    return FALSE;

  threadlist_xml = ochusha_config_find_file(config, pathname, subdir);
  if (threadlist_xml == NULL)
    {
      if (snprintf(pathname, PATH_MAX, "cache/%s%s%s/%s",
		   ochusha_bulletin_board_get_server(board),
		   ochusha_bulletin_board_get_base_path(board),
		   ochusha_bulletin_board_get_id(board),
		   OCHUSHA_THREADLIST_XML) >= PATH_MAX)
	return FALSE;

      threadlist_xml = ochusha_config_find_file(config, pathname, subdir);
      if (threadlist_xml == NULL)
	return FALSE;
    }

  memset(&sax_handler, 0, sizeof(xmlSAXHandler));
#if LIBXML_VERSION >= 20600
  xmlSAX2InitDefaultSAXHandler(&sax_handler, TRUE);
#else
  initxmlDefaultSAXHandler(&sax_handler, TRUE);
#endif

  sax_handler.getEntity = getEntityHandler;
  sax_handler.startDocument = nopHandler;
  sax_handler.endDocument = nopHandler;
  sax_handler.startElement = startElementHandler;
  sax_handler.endElement = endElementHandler;
#if LIBXML_VERSION >= 20600
  sax_handler.startElementNs = NULL;
  sax_handler.endElementNs = NULL;
#endif
  sax_handler.characters = charactersHandler;

  ochusha_bulletin_board_lock_thread_list(board);

  xmlSAXUserParseFile(&sax_handler, &context, threadlist_xml);

  cleanup_sax_context(&context);

  if (recover_mode)
    {
      OchushaBulletinBoardClass *board_class
	= OCHUSHA_BULLETIN_BOARD_GET_CLASS(board);
      if (board_class->recover_threadlist)
	(*board_class->recover_threadlist)(board, config);
    }

  ochusha_bulletin_board_unlock_thread_list(board);

  if (context.state == SAX_ACCEPTED)
    {
      G_FREE(threadlist_xml);
      return TRUE;
    }

  fprintf(stderr, "%s is unacceptable as ochusha's threadlist.\n",
	  threadlist_xml);

  G_FREE(threadlist_xml);

  return FALSE;
}


typedef struct _WriteThreadlistArgs
{
  OchushaBulletinBoard *board;
  gzFile threadlist_xml;
} WriteThreadlistArgs;


static void
write_bbs_thread(gpointer data, gpointer user_data)
{
  OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(data);
  WriteThreadlistArgs *args = (WriteThreadlistArgs *)user_data;
  gzFile threadlist_xml = args->threadlist_xml;

  if (thread->flags & OCHUSHA_BBS_THREAD_DAT_DROPPED
      && ochusha_bbs_thread_get_number_of_responses_read(thread) == 0)
    return;

  gzprintf(threadlist_xml, "    <thread>\n");

  g_signal_emit_by_name(G_OBJECT(thread),
			"write_threadlist_element",
			threadlist_xml);
  g_signal_emit(G_OBJECT(args->board),
		bulletin_board_signals[THREADLIST_WRITE_THREAD_ELEMENT_SIGNAL],
		0,
		thread,
		threadlist_xml);

  gzprintf(threadlist_xml, "    </thread>\n");
}


gboolean
ochusha_bulletin_board_write_threadlist_xml(OchushaBulletinBoard *board,
					    OchushaConfig *config,
					    const char *subdir)
{
  WriteThreadlistArgs args;
  char pathname[PATH_MAX];
  int fd;
  gzFile threadlist_xml;

  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board) && config != NULL,
		       FALSE);
  g_return_val_if_fail(config->home != NULL, FALSE);

  if (board->thread_list == NULL)
    {
#if 0
      fprintf(stderr, "board(%s)'s thread_list is NULL!\n",
	      board->base_url);
#endif
      return TRUE;
    }

  if (subdir != NULL)
    {
      if (snprintf(pathname, PATH_MAX, "%s%s%s/%s",
		   ochusha_bulletin_board_get_server(board),
		   ochusha_bulletin_board_get_base_path(board),
		   ochusha_bulletin_board_get_id(board),
		   OCHUSHA_THREADLIST_XML) >= PATH_MAX)
	return FALSE;
    }
  else
    {
      if (snprintf(pathname, PATH_MAX, "cache/%s%s%s/%s",
		   ochusha_bulletin_board_get_server(board),
		   ochusha_bulletin_board_get_base_path(board),
		   ochusha_bulletin_board_get_id(board),
		   OCHUSHA_THREADLIST_XML) >= PATH_MAX)
	return FALSE;
    }

  fd = ochusha_config_open_file(config, pathname, subdir,
				O_WRONLY | O_TRUNC | O_CREAT);
  if (fd < 0)
    {
      fprintf(stderr, "Couldn't open \"%s/%s\" to write.\n",
	      config->home, pathname);
      return FALSE;
    }

  threadlist_xml = gzdopen(fd, "w");
  if (threadlist_xml == NULL)
    {
      close(fd);
      fprintf(stderr, "Couldn't open fd to write.\n");
      return FALSE;
    }

  gzprintf(threadlist_xml, "<?xml version=\"1.0\"?>\n");
  gzprintf(threadlist_xml, "<ochusha>\n");
  gzprintf(threadlist_xml, "  <threadlist>\n");

  args.board = board;
  args.threadlist_xml = threadlist_xml;

  g_slist_foreach(board->thread_list, write_bbs_thread, &args);

  gzprintf(threadlist_xml, "  </threadlist>\n");
  gzprintf(threadlist_xml, "</ochusha>\n");

  return gzclose(threadlist_xml) == 0;
}


gboolean
ochusha_bulletin_board_get_post_use_2ch_be(OchushaBulletinBoard *board,
					   const OchushaConfig *config)
{
  int mode;
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), FALSE);

  if (board->bbs_type != OCHUSHA_BBS_TYPE_2CH
      && board->bbs_type != OCHUSHA_BBS_TYPE_2CH_BE)
    return FALSE;

  mode = POST_MODE_2CH_BE(board->post_mode);
  return mode == POST_USE_2CH_BE_YES
    || (mode == POST_USE_2CH_BE_DEFAULT && config->use_2ch_be_id_for_posting);
}


void
ochusha_bulletin_board_set_post_use_2ch_be(OchushaBulletinBoard *board,
					   gboolean use_2ch_be)
{
  g_return_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board));

  board->post_mode &= ~POST_USE_2CH_BE_MASK;

  if (board->bbs_type != OCHUSHA_BBS_TYPE_2CH
      && board->bbs_type != OCHUSHA_BBS_TYPE_2CH_BE)
    return;

  if (use_2ch_be)
    board->post_mode |= POST_USE_2CH_BE_YES;
  else
    board->post_mode |= POST_USE_2CH_BE_NO;
}


gboolean
ochusha_bulletin_board_get_post_use_2ch_viewer(OchushaBulletinBoard *board,
					       const OchushaConfig *config)
{
  int mode;
  g_return_val_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board), FALSE);

  if (board->bbs_type != OCHUSHA_BBS_TYPE_2CH
      && board->bbs_type != OCHUSHA_BBS_TYPE_2CH_BE)
    return FALSE;

  mode = POST_MODE_2CH_VIEWER(board->post_mode);
  return mode == POST_USE_2CH_VIEWER_YES
    || (mode == POST_USE_2CH_VIEWER_DEFAULT
	&& config->use_2ch_viewer_for_posting);
}


void
ochusha_bulletin_board_set_post_use_2ch_viewer(OchushaBulletinBoard *board,
					       gboolean use_2ch_viewer)
{
  g_return_if_fail(OCHUSHA_IS_BULLETIN_BOARD(board));

  board->post_mode &= ~POST_USE_2CH_VIEWER_MASK;

  if (board->bbs_type != OCHUSHA_BBS_TYPE_2CH
      && board->bbs_type != OCHUSHA_BBS_TYPE_2CH_BE)
    return;

  if (use_2ch_viewer)
    board->post_mode |= POST_USE_2CH_VIEWER_YES;
  else
    board->post_mode |= POST_USE_2CH_VIEWER_NO;
}

