/*
 * the_nilfs.c - the_nilfs shared structure.
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * the_nilfs.c,v 1.19 2006/07/12 07:50:22 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 *
 */

#include <linux/buffer_head.h>
#include <linux/slab.h>
#include <linux/blkdev.h>
#include <linux/kobject.h>
#include "nilfs.h"

/*
 * nilfs kobject interface
 */

/**
 * nilfs_attr_show - get kobject attribute of the_nilfs structure
 * @kobj: kobject
 * @attr: attribute
 * @page: data page
 *
 * nilfs_attr_show() dispatches show() method calls of sysfs to
 * corresponding handlers with a pointer to the_nilfs structure.
 * The handlers are picked out from the struct nilfs_attribute,
 * whose pointers are registered in nilfs_default_attrs[].
 *
 * Return Value: On success, byte count written to the given page is
 * returned. On error, a negative error code is returned.
 */
static ssize_t
nilfs_attr_show(struct kobject *kobj, struct attribute *attr, char *page)
{
	struct the_nilfs *nilfs =
		container_of(kobj, struct the_nilfs, ns_kobj);
	struct nilfs_attribute *nilfs_attr = 
		container_of(attr, struct nilfs_attribute, attr);
	ssize_t ret = 0;

	if (nilfs_attr->show)
		ret = nilfs_attr->show(nilfs, page);
	return ret;
}

/**
 * nilfs_attr_store - store kobject attribute of the_nilfs structure
 * @kobj: kobject
 * @attr: attribute
 * @page: data page
 * @length: length of a string stored in the data page
 *
 * nilfs_attr_store() dispatches store() method calls of sysfs to
 * corresponding handlers with a pointer to the_nilfs structure.
 * The handlers are picked out from the struct nilfs_attribute.
 *
 * Return Value: On success, byte count read from the given page is
 * returned. On error, a negative error code is returned.
 */
static ssize_t
nilfs_attr_store(struct kobject *kobj, struct attribute *attr,
		 const char *page, size_t length)
{
	struct the_nilfs *nilfs =
		container_of(kobj, struct the_nilfs, ns_kobj);
	struct nilfs_attribute *nilfs_attr = 
		container_of(attr, struct nilfs_attribute, attr);
	ssize_t ret = -EINVAL;

	if (nilfs_attr->store) {
#if NEED_SYSFS_TERMINATOR_CHECK
		const char *cp = page, *ep = cp + PAGE_SIZE;

		for (;;) {
			if (cp == ep)
				return ret;
			if (*cp++ == '\0')
				break;
		}
#endif		
		ret = nilfs_attr->store(nilfs, page, length);
	}
	return ret;
}

static struct attribute *nilfs_default_attrs[] = {
	&nilfs_attr_sc_interval.attr,
	&nilfs_attr_sc_threshold.attr,
	&nilfs_attr_sc_max_bio.attr,
	&nilfs_attr_sc_mjcp_freq.attr,
	&nilfs_attr_sc_ctime.attr,
	&nilfs_attr_max_sketch_size.attr,
	&nilfs_attr_sync.attr,
	NULL,
};

static struct sysfs_ops nilfs_sysfs_ops = {
	.show = nilfs_attr_show,
	.store = nilfs_attr_store,
};

static struct kobj_type nilfs_ktype = {
	.sysfs_ops   = &nilfs_sysfs_ops,
	.default_attrs = nilfs_default_attrs,
};

/**
 * nilfs_register_kobject - register kobject of the_nilfs structure
 * @nilfs: the_nilfs structure
 *
 * nilfs_register_kobject() initializes a kobject member of @nilfs
 * and calls kobject_register() to register it as a sysfs entry.
 *
 * Return Value: On success, 0 is returned. On error, a negative error
 * code is returned.
 */
static int nilfs_register_kobject(struct the_nilfs *nilfs)
{
	struct block_device *bdev = nilfs->ns_bdev;
	int err;

	if (!bdev->bd_disk)
		return 0;

	nilfs->ns_kobj.parent =
		bdev->bd_part ? &bdev->bd_part->kobj : &bdev->bd_disk->kobj;
	nilfs->ns_kobj.ktype = &nilfs_ktype;
	kobject_set_name(&nilfs->ns_kobj, "nilfs");
	err = kobject_register(&nilfs->ns_kobj);
	return err;
}

/**
 * nilfs_unregister_kobject - unregister kobject of the_nilfs structure
 * @nilfs: the_nilfs structure
 *
 * nilfs_unregister_kobject() calls kobject_unregister() to unregister
 * the sysfs entry.
 */
static void nilfs_unregister_kobject(struct the_nilfs *nilfs)
{
	if (!nilfs->ns_bdev->bd_disk)
		return;

	kobject_unregister(&nilfs->ns_kobj);
}

/**
 * alloc_nilfs - allocate the_nilfs structure
 * @bdev: block device to which the_nilfs is related
 *
 * alloc_nilfs() allocates memory for the_nilfs and
 * initializes its reference count and locks.
 *
 * Return Value: On success, pointer to the_nilfs is returned.
 * On error, NULL is returned.
 */
struct the_nilfs *alloc_nilfs(struct block_device *bdev)
{
	struct the_nilfs *nilfs;

	nilfs = kmalloc(sizeof(struct the_nilfs), GFP_KERNEL);
	if (!nilfs)
		return NULL;

	memset(nilfs, 0, sizeof(*nilfs));
	nilfs->ns_bdev = bdev;
	atomic_set(&nilfs->ns_count, 1);
	spin_lock_init(&nilfs->ns_lock);
	init_rwsem(&nilfs->ns_sem);
	init_rwsem(&nilfs->ns_writer_sem);
	return nilfs;
}

/**
 * put_nilfs - release a reference to the_nilfs
 * @nilfs: the_nilfs structure to be released
 *
 * put_nilfs() decrements a reference counter of the_nilfs.
 * If the reference count reaches zero, the_nilfs is freed.
 */
void put_nilfs(struct the_nilfs *nilfs)
{
#ifdef CONFIG_NILFS_DEBUG
	char b[BDEVNAME_SIZE];
	struct block_device *bdev = nilfs->ns_bdev;
#endif
	if (!atomic_dec_and_test(&nilfs->ns_count))
		return;
	/*
	 * Increment of ns_count never occur below because the caller
	 * of get_nilfs() holds at least one reference to the_nilfs.
	 * Thus its exclusion control is not required here.
	 */
	might_sleep();
	if (test_bit(THE_NILFS_KOBJECT, &nilfs->ns_flags))
		nilfs_unregister_kobject(nilfs);
	if (test_bit(THE_NILFS_INIT, &nilfs->ns_flags))
		nilfs_brelse(nilfs->ns_sbh);

	kfree(nilfs);
	nilfs_debug(1, "the_nilfs on bdev %s was freed\n",
		    bdevname(bdev, b));
}

/**
 * init_nilfs - initialize a NILFS instance.
 * @nilfs: the_nilfs structure
 * @sbi: nilfs_sb_info
 * @sb: super block
 * @data: mount options
 *
 * init_nilfs() performs common initialization per block device (e.g.
 * reading the super block, getting disk layout information, initializing
 * shared fields in the_nilfs). It takes on some portion of the jobs 
 * typically done by a fill_super() routine. This division arises from
 * the nature that multiple NILFS instances may be simultaneously
 * mounted on a device.
 * For multiple mounts on the same device, only the first mount
 * invokes these tasks.
 *
 * Return Value: On success, 0 is returned. On error, a negative error
 * code is returned.
 */
int init_nilfs(struct the_nilfs *nilfs, struct nilfs_sb_info *sbi,
	       struct super_block *sb, char *data)
{
	struct buffer_head *sbh;
	struct nilfs_super_block *sbp;
	int blocksize;
	int err = 0;

	down_write(&nilfs->ns_sem);
	if (test_and_set_bit(THE_NILFS_INIT, &nilfs->ns_flags)) {
		/* Load values from existing the_nilfs */
		sbp = nilfs->ns_sbp;
		err = nilfs_store_magic_and_option(sb, sbp, data);
		if (err)
			goto out;

		blocksize = BLOCK_SIZE << le32_to_cpu(sbp->s_log_block_size);
		if (sb->s_blocksize != blocksize &&
		    !sb_set_blocksize(sb, blocksize)) {
			printk("NILFS: blocksize %d unfit to device\n",
			       blocksize);
			err = -EINVAL;
		} else
			err = nilfs_store_disk_layout(sb, sbp);
		goto out;
	}

	sbp = nilfs_load_super_block(sb, &sbh);
	if (!sbp) {
		err = -EINVAL;
		goto failed;
	}
	err = nilfs_store_magic_and_option(sb, sbp, data);
	if (err)
		goto failed_sbh;

	blocksize = BLOCK_SIZE << le32_to_cpu(sbp->s_log_block_size);
	if (sb->s_blocksize != blocksize) {
		sbp = nilfs_reload_super_block(sb, &sbh, blocksize);
		if (!sbp) {
			err = -EINVAL;
			goto failed; 
                        /* not failed_sbh; sbh is released automatically
			   when reloading fails. */
		}
	}

	err = nilfs_store_disk_layout(sb, sbp);
	if (err)
		goto failed_sbh;

	nilfs->ns_mount_state = le16_to_cpu(sbp->s_state);
	nilfs->ns_sbh = sbh;
	nilfs->ns_sbp = sbp;

	/* Finding last segment */
	nilfs->ns_last_pseg = le64_to_cpu(sbp->s_last_segment);
	nilfs->ns_seg_seq = nilfs->ns_last_seq = le64_to_cpu(sbp->s_last_seq);
	nilfs->ns_segnum = nilfs->ns_last_pseg / sbi->s_blocks_per_segment;
	if (nilfs->ns_segnum >= sbi->s_nsegment) {
		printk(KERN_ERR "NILFS invalid last segment number. "
		       "running fsck is required\n");
		err = -EINVAL;
		goto failed_sbh;
	}
	/* Dummy values  */
	nilfs->ns_free_segments_count =
		sbi->s_nsegment - (nilfs->ns_segnum + 1);

	if (!test_and_set_bit(THE_NILFS_KOBJECT, &nilfs->ns_flags)) {
		err = nilfs_register_kobject(nilfs);
		if (err) {
			printk(KERN_ERR
			       "NILFS error registering kobject.\n");
			clear_bit(THE_NILFS_KOBJECT, &nilfs->ns_flags);
			goto failed_sbh;
		}
	}
	err = 0;
 out:
	up_write(&nilfs->ns_sem);
	return err;

 failed_sbh:
	nilfs_brelse(sbh);

 failed:
	clear_bit(THE_NILFS_INIT, &nilfs->ns_flags);
	goto out;
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
