/*
 * sketch.c - NILFS sketch file handling primitives
 *
 * Copyright (C) 2006 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * sketch.c,v 1.10 2006/05/16 16:40:46 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 */

#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/writeback.h>
#include <linux/module.h>
#include "nilfs.h"

static ssize_t
sketch_file_write(struct file *file, const char __user *buf, size_t count, loff_t *ppos)
{
	struct inode *inode = file->f_dentry->d_inode;
	struct nilfs_sb_info *sbi = NILFS_SB(inode->i_sb);
	struct nilfs_sc_info *sci;
	ssize_t ret = 0;

	if (sbi->s_snapshot_cp)
		return -EROFS;
	if (unlikely(*ppos < 0))
		return -EINVAL;
	if (file->f_flags & O_APPEND)
		*ppos = i_size_read(inode);

	might_sleep();
#if NEED_INODE_SEMAPHORE
	down(&inode->i_sem);
#else
	mutex_lock(&inode->i_mutex);
#endif
	down_write(&sbi->s_segctor.sem);
	sci = NILFS_SC(sbi);
	if (sci && sci->sc_cp_bh) {
		unsigned long max_bytes = sci->sc_cp_bh->b_size - NILFS_CP_BYTES;
		
		if (*ppos < max_bytes) {
			unsigned long bytes = min_t(unsigned long, count, max_bytes - *ppos), left;

			left = copy_from_user(sci->sc_cp_bh->b_data + NILFS_CP_BYTES + *ppos, buf, bytes);
			*ppos += (ret = bytes - left);

			if (*ppos > nilfs_sketch_size(sci)) {
				i_size_write(inode, *ppos);
				if (!sci->sc_sketch_inode)
					sci->sc_sketch_inode = igrab(inode);
			} else
				BUG_ON(inode->i_size > 0 && !sci->sc_sketch_inode);

			inode->i_mtime = CURRENT_TIME;
		} else
			ret = -EFBIG;
		/*
		 * If *ppos > current sketch_bytes, it creates a hole.
		 * We don't clear it here because the region in the hole was
		 * zero-filled when the file was truncated.
		 */
	}
	up_write(&sbi->s_segctor.sem);
#if NEED_INODE_SEMAPHORE
	up(&inode->i_sem);
#else
	mutex_unlock(&inode->i_mutex);
#endif
	return ret;
}

static ssize_t
copy_sketch_from_cp_buffer(char __user *buf, size_t count, loff_t *ppos, 
			   struct buffer_head *bh_cp, unsigned int cp_bytes, unsigned long max_bytes)
{
	unsigned long bytes, left;
	ssize_t ret;

	if (*ppos >= max_bytes)
		return 0;

	bytes = min_t(unsigned long, count, max_bytes - *ppos);
	left = copy_to_user(buf, bh_cp->b_data + cp_bytes + *ppos, bytes);
	*ppos += (ret = bytes - left);
	return ret;
}

static ssize_t
sketch_file_read(struct file *file, char __user *buf, size_t count, loff_t *ppos)
{
	struct inode *inode = file->f_dentry->d_inode;
	struct nilfs_sb_info *sbi = NILFS_SB(inode->i_sb);
	ssize_t ret;

	if (unlikely(*ppos < 0))
		return -EINVAL;

	might_sleep();
#if NEED_INODE_SEMAPHORE
	down(&inode->i_sem);
#else
	mutex_lock(&inode->i_mutex);
#endif
	if (sbi->s_snapshot_cp) { /* Read from a snapshot */
		struct buffer_head *bh_cp = nilfs_bread(inode->i_sb, sbi->s_snapshot_cp);

		ret = -EIO;
		if (bh_cp) {
			struct nilfs_checkpoint *raw_cp
				= (struct nilfs_checkpoint *)bh_cp->b_data;

			ret = copy_sketch_from_cp_buffer(buf, count, ppos, bh_cp,
							 le16_to_cpu(raw_cp->cp_bytes),
							 le32_to_cpu(raw_cp->cp_sketch_size));
			nilfs_brelse(bh_cp);
		}
	} else { /* Read from a buffer of the segment constructor */
		struct nilfs_sc_info *sci;

		ret = 0;
		down_read(&sbi->s_segctor.sem);
		sci = NILFS_SC(sbi);
		if (sci && sci->sc_cp_bh)
			ret = copy_sketch_from_cp_buffer(buf, count, ppos,
							 sci->sc_cp_bh, NILFS_CP_BYTES,
							 nilfs_sketch_size(sci));
		up_read(&sbi->s_segctor.sem);
	}
#if NEED_INODE_SEMAPHORE
	up(&inode->i_sem);
#else
	mutex_unlock(&inode->i_mutex);
#endif
	return ret;
}

static struct file_operations nilfs_sketch_file_operations = {
	.llseek		= generic_file_llseek,
	.read		= sketch_file_read,
	.write		= sketch_file_write,
	.open		= generic_file_open,
};

static int sketch_truncate(struct nilfs_inode_info *ii, loff_t offset)
{
	struct inode *inode = &ii->vfs_inode;
	struct nilfs_sb_info *sbi = NILFS_SB(inode->i_sb);
	struct nilfs_sc_info *sci;
	struct inode *iprev = NULL;
	int err = 0;

	if (sbi->s_snapshot_cp)
		return -EROFS;
	
	down_write(&sbi->s_segctor.sem);
	sci = NILFS_SC(sbi);
	if (sci && sci->sc_cp_bh) {
		unsigned long max_bytes = sci->sc_cp_bh->b_size - NILFS_CP_BYTES;
		unsigned long sketch_bytes = nilfs_sketch_size(sci);
		
		if (offset <= max_bytes) {
			i_size_write(inode, offset);
			if (offset < sketch_bytes)
				memset(sci->sc_cp_bh->b_data + NILFS_CP_BYTES + offset, 0, 
				       sketch_bytes - offset);
			iprev = sci->sc_sketch_inode;
			sci->sc_sketch_inode = offset ? igrab(inode) : NULL;
		} else
			err = -EFBIG;
	}
	up_write(&sbi->s_segctor.sem);
	if (iprev)
		iput(iprev);
	return err;
}

static int sketch_setattr(struct dentry *dentry, struct iattr *iattr)
{
	struct inode *inode = dentry->d_inode;
	unsigned int ia_valid;
	int err;

	err = inode_change_ok(inode, iattr);
	if (err)
		return err;
	
	ia_valid = iattr->ia_valid;
	if (ia_valid & ATTR_SIZE) {
		if (iattr->ia_size != i_size_read(inode)) {
			struct nilfs_inode_info *ii = NILFS_I(inode);

			err = sketch_truncate(ii, iattr->ia_size);
			if (err || (ia_valid == ATTR_SIZE))
				goto out;
		} else
			ia_valid |= ATTR_MTIME | ATTR_CTIME;
	}
	if (ia_valid & ATTR_ATIME)
		inode->i_atime = timespec_trunc(iattr->ia_atime,
						inode->i_sb->s_time_gran);
	if (ia_valid & ATTR_MTIME)
		inode->i_mtime = timespec_trunc(iattr->ia_mtime,
						inode->i_sb->s_time_gran);
	if (ia_valid & ATTR_CTIME)
		inode->i_ctime = timespec_trunc(iattr->ia_ctime,
						inode->i_sb->s_time_gran);

	ia_valid &= ~(ATTR_ATIME | ATTR_MTIME | ATTR_CTIME | ATTR_SIZE);
	if (!ia_valid)
		goto out;

	if (ia_valid & ATTR_UID)
		inode->i_uid = iattr->ia_uid;
	if (ia_valid & ATTR_GID)
		inode->i_gid = iattr->ia_gid;
	if (ia_valid & ATTR_MODE) {
		umode_t mode = iattr->ia_mode;

		if (!in_group_p(inode->i_gid) && !capable(CAP_FSETID))
			mode &= ~S_ISGID;
		inode->i_mode = mode;
	}
	mark_inode_dirty(inode);
 out:
	return err;
}

static struct inode_operations nilfs_sketch_file_inode_operations = {
	.setattr	= sketch_setattr,
};

int nilfs_read_sketch_inode(struct nilfs_inode_info *ii)
{
	struct inode *inode = &ii->vfs_inode;
	struct nilfs_sb_info *sbi = NILFS_SB(inode->i_sb);

	if (!S_ISREG(inode->i_mode)) {
		nilfs_warning(inode->i_sb, __FUNCTION__,
			      "Invalid sketch inode (ino=%lu, type=0x%x)\n",
			      inode->i_ino, (inode->i_mode & S_IFMT) >> 12);
		return -EINVAL;
	}
	inode->i_fop = &nilfs_sketch_file_operations;
	inode->i_op = &nilfs_sketch_file_inode_operations;

	if (sbi->s_snapshot_cp) {
		struct buffer_head *bh_cp;
		struct nilfs_checkpoint *raw_cp;

		bh_cp = nilfs_bread(inode->i_sb, sbi->s_snapshot_cp);
		if (unlikely(!bh_cp))
			return -EIO;
		raw_cp = (struct nilfs_checkpoint *)bh_cp->b_data;
		inode->i_size = le32_to_cpu(raw_cp->cp_sketch_size);
		nilfs_brelse(bh_cp);
	} else
		inode->i_size = 0;
	        /* At this time, the segment constructor never bind
		   inode of the sketch file; the sketch file whose
		   size is greater than zero, is always bound to the
		   nilfs_sc_info struct. it can be found in the inode
		   cache (e.g. protected from the shrinker). */

	return 0;
}

/*
 * Sysfs interface
 */
static ssize_t nilfs_show_max_sketch_size(struct the_nilfs *nilfs, char *page)
{
	struct nilfs_sb_info *sbi = nilfs_get_writer(nilfs);
	ssize_t ret = 0;

	if (sbi)
		ret = sprintf(page, "%lu\n", sbi->s_super->s_blocksize - NILFS_CP_BYTES);
	nilfs_put_writer(nilfs);
	return ret;
}

define_nilfs_ro_attr(max_sketch_size, 0444)

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
