/*
 * recovery.c - NILFS recovery logic
 *
 * Copyright (C) 2005, 2006 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * recovery.c,v 1.9 2006/07/12 13:00:17 ryusuke Exp
 *
 * Written by Ryusuke Konishi <ryusuke@osrg.net>
 */

#include <linux/buffer_head.h>
#include <linux/blkdev.h>
#include <linux/swap.h>
#include <linux/crc32c.h>
#include "nilfs.h"


#ifdef NILFS_CP_BARRIER
#define CHECK_PAYLOAD_ON_CP_SEARCH   0
#define CHECK_HEADER_AND_TRAILER     1  /* Ensure completeness of each segment by checking
					   a segment summary and a checkpoint block. */
#else
#define CHECK_PAYLOAD_ON_CP_SEARCH   1  /* The segment is checked fully */
#define CHECK_HEADER_AND_TRAILER     0
#endif

/* work structure for recovery */
struct nilfs_recovery_block {
	ino_t ino;              /* Inode number of the file that this block belongs to */
	dbn_t dbn;              /* Disk block number of the block */
	fbn_t offset;           /* File offset of the data block (per block) */
	struct list_head list;
};

struct nilfs_recovery_info {
	u64 last_seq;
	int need_recovery;
	dbn_t inode_root;       /* inode root of the latest checkpoint */
	dbn_t last_pseg;
	dbn_t last_cp;          /* Pointers of the latest valid checkpoint */
	time_t last_ctime;
	unsigned long inodes_count;
	unsigned long blocks_count;

	dbn_t lsegs_start;
	dbn_t lsegs_end;
	u64 lsegs_start_seq;      /* sequence value of the segment at lsegs_start */
};


static inline void 
store_segsum_info(struct nilfs_segsum_info *ssi, struct nilfs_seg_summary *sum)
{
	ssi->nblocks = le16_to_cpu(sum->ss_nblocks);
	ssi->nfinfo = le16_to_cpu(sum->ss_nfinfo);
	ssi->nfbinfo = le16_to_cpu(sum->ss_nfbinfo);
	ssi->nblk_file = le16_to_cpu(sum->ss_nfblk);
	ssi->nblk_fbt = le16_to_cpu(sum->ss_nfbblk);
	ssi->nblk_inode = le16_to_cpu(sum->ss_niblk);
	ssi->nblk_ibt = le16_to_cpu(sum->ss_nibblk);
	ssi->flags = le16_to_cpu(sum->ss_flags);
	ssi->ctime = le64_to_cpu(sum->ss_create);
	ssi->prev_pseg = le16_to_cpu(sum->ss_prev);
	ssi->seg_seq = le64_to_cpu(sum->ss_seq);
}

static inline void
store_cp_info(struct nilfs_cp_info *cpi, struct nilfs_checkpoint *cp)
{
	cpi->flags = le16_to_cpu(cp->cp_flags);
	cpi->inode_root = le64_to_cpu(cp->cp_inode_root);
	cpi->inodes_count = le64_to_cpu(cp->cp_inodes_count);
	cpi->blocks_count = le64_to_cpu(cp->cp_blocks_count);
}

static inline void
get_nilfs_cp(struct the_nilfs *nilfs, struct nilfs_recovery_info *ri)
{
	spin_lock_irq(&nilfs->ns_lock);
	ri->inode_root = nilfs->ns_inode_root;
	ri->last_seq = nilfs->ns_last_seq;
	ri->last_pseg = nilfs->ns_last_pseg;
	ri->last_cp = nilfs->ns_last_cp;
	ri->last_ctime = nilfs->ns_last_ctime;
	ri->inodes_count = nilfs->ns_inodes_count;
	ri->blocks_count = nilfs->ns_blocks_count;
	spin_unlock_irq(&nilfs->ns_lock);
}

/**
 * calc_crc_cont - check CRC of blocks continuously
 * @sbi: nilfs_sb_info
 * @bhs: buffer head of start block
 * @sum: place to store result
 * @offset: offset bytes in the first block
 * @check_bytes: number of bytes to be checked
 * @start: DBN of start block
 * @nblock: number of blocks to be checked
 */
static int calc_crc_cont(struct nilfs_sb_info *sbi, struct buffer_head *bhs,
			 u32 *sum, unsigned long offset, u64 check_bytes,
			 dbn_t start, unsigned long nblock)
{
	unsigned long blocksize = sbi->s_super->s_blocksize;
	unsigned long size;
	u32 crc;

	BUG_ON(offset >= blocksize);
	check_bytes -= offset;
	size = min_t(u64, check_bytes, blocksize - offset);
	crc = crc32c_le(sbi->s_crc_seed,
			(unsigned char *)bhs->b_data + offset, size);
	if (--nblock > 0) {
		do {
			struct buffer_head *bh
				= sb_bread(sbi->s_super, ++start);
			if (!bh)
				return -EIO;
			check_bytes -= size;
			size = min_t(u64, check_bytes, blocksize);
			crc = crc32c_le(crc, bh->b_data, size);
			brelse(bh);
		} while (--nblock > 0);
	}
	*sum = crc;
	return 0;
}

/**
 * load_segment_summary - read segment summary of the specified partial segment
 * @sbi: nilfs_sb_info
 * @pseg_start: start disk block number of partial segment
 * @ssi: pointer to nilfs_segsum_info struct to store information
 * @full_check: full check flag
 *              (0: only checks segment summary CRC, 1: data CRC)
 */
static int
load_segment_summary(struct nilfs_sb_info *sbi, dbn_t pseg_start,
		     struct nilfs_segsum_info *ssi, int full_check)
{
	struct buffer_head *bh_sum;
	struct nilfs_seg_summary *sum;
	unsigned long offset, nblock;
	u64 check_bytes;
	u32 crc, crc_sum;
	int ret = NILFS_SEG_FAIL_IO;

	recovery_debug(2, "checking segment (pseg_start=%lu, full_check=%d)\n",
		       pseg_start, full_check);
	bh_sum = sb_bread(sbi->s_super, pseg_start);
	if(!bh_sum)
		goto out;

	sum = (struct nilfs_seg_summary *)bh_sum->b_data;

	/* Check consistency of segment summary */
	if (le16_to_cpu(sum->ss_magic) != NILFS_SEGSUM_MAGIC) {
		ret = NILFS_SEG_FAIL_MAGIC;
		goto failed;
	}
	store_segsum_info(ssi, sum);
	check_bytes = nilfs_recalc_segsum_size(ssi, sbi->s_super->s_blocksize);

	/*
	 * Code for checking consistency of the segment summary should be
	 * inserted here.
	 */
	if (nilfs_seg_blocks(ssi, !!NILFS_SEG_HAS_CP(ssi)) > ssi->nblocks) {
		ret = NILFS_SEG_FAIL_CONSISTENCY;
		goto failed;
	}

	if (full_check) {
		offset = sizeof(sum->ss_datasum);
		check_bytes = ((u64)ssi->nblocks << sbi->s_super->s_blocksize_bits);
		nblock = ssi->nblocks;
		crc_sum = le32_to_cpu(sum->ss_datasum);
		ret = NILFS_SEG_FAIL_CHECKSUM_FULL;
	} else { /* only checks segment summary */
		offset = sizeof(sum->ss_datasum) + sizeof(sum->ss_sumsum);
		nblock = ssi->nblk_sum;
		crc_sum = le32_to_cpu(sum->ss_sumsum);
		ret = NILFS_SEG_FAIL_CHECKSUM_SEGSUM;
	}
	if (calc_crc_cont(sbi, bh_sum, &crc, offset, check_bytes,
			  pseg_start, nblock)) {
		ret = NILFS_SEG_FAIL_IO;
		goto failed;
	}
	if (crc == crc_sum)
		ret = 0;
 failed:
	brelse(bh_sum);
 out:
	recovery_debug(2, "done (ret=%d)\n", ret);
	return ret;
}

/**
 * load_checkpoint - read a checkpoint block
 * @sb: super block
 * @cp_block: disk block number the checkpoint block to be read
 * @cpi: pointer to nilfs_cp_info struct to store information
 */
static int
load_checkpoint(struct super_block *sb, dbn_t cp_block, 
		struct nilfs_cp_info *cpi)
{
	struct buffer_head *bh_cp;
	struct nilfs_checkpoint *cp;
#if CHECK_HEADER_AND_TRAILER
	u32 crc;
#endif
	int ret;
  
	bh_cp = sb_bread(sb, cp_block);
	if(!bh_cp)
		return NILFS_SEG_FAIL_IO;

	cp = (struct nilfs_checkpoint *)bh_cp->b_data;
	store_cp_info(cpi, cp);

	if(!(cpi->flags & NILFS_CP_MAJOR)) {
		ret = NILFS_SEG_HAS_NO_MAJOR_CP;
		goto fail;
	}
#if CHECK_HEADER_AND_TRAILER
	if (calc_crc_cont(NILFS_SB(sb), bh_cp, &crc, sizeof(cp->cp_sum),
			  le16_to_cpu(cp->cp_bytes), cp_block, 1)) {
		ret = NILFS_SEG_FAIL_IO;
		goto fail;
	}
	if(crc != le32_to_cpu(cp->cp_sum)) {
		ret = NILFS_SEG_FAIL_CHECKSUM_CP;
		goto fail;
	}
#endif
	ret = 0;
 fail:
	brelse(bh_cp);
	return ret;
}

int nilfs_warn_segment_error(struct nilfs_sb_info *sbi, int err)
{
	switch (err) {
	case NILFS_SEG_FAIL_IO:
		printk(KERN_WARNING
		       "NILFS warning: I/O error on loading last segment\n");
		return -EIO;
	case NILFS_SEG_FAIL_MAGIC:
		printk(KERN_WARNING
		       "NILFS warning: Invalid magic number at last segment\n");
		break;
	case NILFS_SEG_FAIL_SEQ:
		printk(KERN_WARNING
		       "NILFS warning: Sequence number mismatch\n");
		break;
	case NILFS_SEG_FAIL_CHECKSUM_SEGSUM:
		printk(KERN_WARNING
		       "NILFS warning: Checksum error in segment summary\n");
		break;
	case NILFS_SEG_FAIL_CHECKSUM_CP:
		printk(KERN_WARNING
		       "NILFS warning: Checksum error in checkpoint\n");
		break;
	case NILFS_SEG_FAIL_CHECKSUM_FULL:
		printk(KERN_WARNING
		       "NILFS warning: Checksum error in whole segment check\n");
		break;
	case NILFS_SEG_FAIL_CONSISTENCY:
		printk(KERN_WARNING
		       "NILFS warning: Inconsistent segment\n");
		break;
	case NILFS_SEG_HAS_NO_MAJOR_CP:
		printk(KERN_WARNING
		       "NILFS warning: No major checkpoint in the last "
		       "segment\n");
		break;
	case NILFS_SEG_HAS_NO_CP:
		printk(KERN_WARNING
		       "NILFS warning: No checkpoint in the last segment\n");
		break;
	case NILFS_SEG_BAD_SNAPSHOT:
		printk(KERN_WARNING
		       "NILFS warning: Bad snapshot segment (block number=%lu)\n",
		       sbi->s_snapshot_pseg);
		break;
	case NILFS_SEG_VALID:
	default:
		BUG();
	}
	return -EINVAL;
}

/**
 * read_from_segsum - read a data item from segment summary blocks.
 * sbi: nilfs_sb_info
 * pbh: address to store a pointer to buffer_head struct.
 * offset: offset position on the buffer.
 * out: address to store data
 * bytes: size of date to be read
 *
 * read_from_segsum() copy a data item from segment summary blocks.
 * On success, @offset is forwarded automatically.
 * If the data item strides over two summary blocks, two fragment
 * of data are read from each block and are concatinated on @out
 * In that case, the buffer_head pointed by @pbh is also shifted
 * so that the caller can read data continuously in successive calls.
 */
static int
read_from_segsum(struct nilfs_sb_info *sbi, struct buffer_head **pbh,
		 unsigned int *offset, void *out, unsigned int bytes)
{
	unsigned int bcnt;
	dbn_t dbn;

	BUG_ON((*pbh)->b_size < *offset);
 retry:
	bcnt = min_t(unsigned int, bytes, (*pbh)->b_size - *offset);
	memcpy(out, (*pbh)->b_data + *offset, bcnt);
	if (likely(bcnt == bytes)) {
		*offset += bcnt;
		return 0;
	}
	dbn = (*pbh)->b_blocknr;
	brelse(*pbh);
	*pbh = sb_bread(sbi->s_super, ++dbn);
	if (!*pbh)
		return -EIO;

	*offset = 0;
	out += bcnt;
	bytes -= bcnt; 
	goto retry;
}

static int
collect_blocks_from_segsum(struct nilfs_sb_info *sbi, dbn_t sum_block,
			   struct nilfs_segsum_info *ssi, struct list_head *head)
{
	struct buffer_head *bh;
	struct nilfs_finfo finfo;
	unsigned long nblocks, nfinfo;
	unsigned int offset = sizeof(struct nilfs_seg_summary);
	dbn_t dbn = sum_block + ssi->nblk_sum;
	ino_t ino;
	int err = -EIO;

	recovery_debug(2, "reading summary (sum_block=%lu)\n", sum_block);
	bh = sb_bread(sbi->s_super, sum_block);
	if (unlikely(!bh))
		goto out;

	for (nfinfo = ssi->nfinfo; nfinfo > 0; nfinfo--) {
		err = read_from_segsum(sbi, &bh, &offset, &finfo, sizeof(finfo));
		if (unlikely(err))
			goto out;

		nblocks = le32_to_cpu(finfo.fi_nblocks);
		ino = le64_to_cpu(finfo.fi_ino);
		while (nblocks-- > 0) {
			struct nilfs_recovery_block *rb;
			__le64 raw_fbn;

			err = read_from_segsum(sbi, &bh, &offset, &raw_fbn, sizeof(raw_fbn));
			if (unlikely(err))
				goto out;

			rb = kmalloc(sizeof(*rb), GFP_NOFS);
			if (unlikely(!rb)) {
				err = -ENOMEM;
				goto out;
			}
			rb->ino = ino;
			rb->dbn = dbn;
			rb->offset = le64_to_cpu(raw_fbn);
			/* INIT_LIST_HEAD(&rb->list); */
			list_add_tail(&rb->list, head);
		}
	}
	err = 0;
 out:
	brelse(bh);   /* brelse(NULL) is just ignored */
	recovery_debug(2, "done (err=%d)\n", err);
	return err;
}

static int
recover_dsync_blocks(struct nilfs_sb_info *sbi, struct list_head *head,
		     unsigned long *nr_recovery_block)
{
	struct inode *inode;
	struct nilfs_inode_info *ii;
	struct nilfs_recovery_block *rb, *n;
	unsigned blocksize = sbi->s_super->s_blocksize;
	struct page *page;
	struct buffer_head *bh;
	unsigned long index, nth;
	int err, err2;

	err = err2 = 0;
	list_for_each_entry_safe(rb, n, head, list) {

		recovery_debug(2, "looking block (ino=%lu, dbn=%lu, offset=%lu)\n",
			       rb->ino, rb->dbn, rb->offset);
		err = -EINVAL;
		inode = iget(sbi->s_super, rb->ino);
		if (!inode || is_bad_inode(inode))
			goto failed_inode;
		recovery_debug(3, "iget done\n");

		ii = NILFS_I(inode);

		index = rb->offset >> (PAGE_CACHE_SHIFT - inode->i_blkbits);
		nth = (rb->offset - (index << (PAGE_CACHE_SHIFT - inode->i_blkbits)));
		
		page = find_or_create_page(inode->i_mapping, index, GFP_NOFS);
		if (!page) {
			recovery_debug(2, "find_or_create_page() failed\n");
			goto failed_inode;
		}
		recovery_debug(3, "find_or_create_page done\n");

		/* Page is locked */
		if (!page_has_buffers(page))
			create_empty_buffers(page, blocksize, 0);
		
		bh = nilfs_page_get_nth_block(page, nth);
		recovery_debug(3, "nilfs_page_get_nth_block done\n");
		BUG_ON(!bh);

		if (!buffer_mapped(bh)) {
			err = nilfs_get_block(inode, rb->offset, bh, 1);
			recovery_debug(3, "nilfs_get_block done\n");
			if (unlikely(err)) {
				recovery_debug(2, "nilfs_get_block() failed (err=%d)\n", err);
				if (err > 0)
					err = -EINVAL;
				goto failed_bh;
			}
		}
		err = nilfs_btree_mark_from_data(&ii->i_block_root, bh);
		recovery_debug(3, "btree_mark_from_data done (err=%d)\n", err);
		if (unlikely(err < 0)) {
			err = nilfs_handle_fbt_error(err, __FUNCTION__, inode);
			goto failed_bh;
		}
		err = nilfs_btree_update_data_dbn(&ii->i_block_root, bh, rb->dbn);
		recovery_debug(3, "btree_update_data_dbn done (err=%d)\n", err);
		if (unlikely(err < 0)) {
			err = nilfs_handle_fbt_error(err, __FUNCTION__, inode);
			goto failed_bh;
		}
		bh->b_blocknr = rb->dbn;
		
		nilfs_set_file_dirty(sbi, inode);
		recovery_debug(3, "nilfs_set_file_dirty done\n");
		brelse(bh);
		unlock_page(page);
		recovery_debug(3, "unlock_page done\n");
		page_cache_release(page);

		(*nr_recovery_block)++;
		goto next;

	failed_bh:
		brelse(bh);

		ClearPageUptodate(page);
		unlock_page(page);
		page_cache_release(page);

	failed_inode:
		printk(KERN_WARNING
		       "NILFS warning: error recovering data block "
		       "(err=%d, ino=%lu, block-offset=%lu)\n",
		       err, rb->ino, rb->offset);
		if (!err2)
			err2 = err;
	next:
		iput(inode); /* iput(NULL) is just ignored */
		list_del_init(&rb->list);
		kfree(rb);
	}
	return err2;
}

static void dispose_recovery_list(struct list_head *head)
{
	while (!list_empty(head)) {
		struct nilfs_recovery_block *rb
			= list_entry(head->next, struct nilfs_recovery_block, list);
		list_del(&rb->list);
		kfree(rb);
	}
}

/**
 * nilfs_do_roll_forward - salvage logical segments newer than the latest checkpoint
 * @sbi: nilfs_sb_info
 * @nilfs: the_nilfs
 * @ri: pointer to a nilfs_recovery_info
 */
static int
nilfs_do_roll_forward(struct nilfs_sb_info *sbi, struct the_nilfs *nilfs, 
		      struct nilfs_recovery_info *ri)
{
	struct nilfs_segsum_info ssi;
	dbn_t pseg_start;
	dbn_t seg_start, seg_end;   /* Starting/ending DBN of full segment */
	unsigned long nr_salvaged_block = 0;
	u64 seg_seq;
	segnum_t next_segnum, segnum;
	int err, ret, alt_seg = 0;
	LIST_HEAD(dsync_blocks);  /* list of data blocks to be recovered */
	enum {
		RF_INIT_ST,
		RF_DSYNC_ST,   /* scanning data-sync segments */
	};
	int state = RF_INIT_ST;

	recovery_debug(1, "started (lsegs_start=%lu, lsegs_end=%lu)\n)",
		       ri->lsegs_start, ri->lsegs_end);

	pseg_start = ri->lsegs_start;
	seg_seq = ri->lsegs_start_seq;
	segnum = pseg_start / sbi->s_blocks_per_segment;

	nilfs_get_segment_range(sbi, segnum, &seg_start, &seg_end);

	for (;;) {
		if (unlikely(segnum == nilfs->ns_segnum &&
			     pseg_start >= seg_start + nilfs->ns_pseg_offset))
			break;

		ret = load_segment_summary(sbi, pseg_start, &ssi, 1);
		if (ret) {
			if (ret == NILFS_SEG_FAIL_IO) {
				err = -EIO;
				goto failed;
			}
			goto strayed;
		}
		if (seg_seq != ssi.seg_seq)
			goto strayed;

		if (unlikely(NILFS_SEG_HAS_MJCP(&ssi))) {
			err = -EINVAL; /* confused */
			goto failed;
		}
		    
		/* Found a valid partial segment; do recovery actions */
		alt_seg = 0;
		switch (state) {
		case RF_INIT_ST:
			if (!NILFS_SEG_LOGBGN(&ssi) || !NILFS_SEG_DSYNC(&ssi))
				goto try_next_pseg;
			state = RF_DSYNC_ST;
			/* Fall through */
		case RF_DSYNC_ST:
			if (!NILFS_SEG_DSYNC(&ssi)) {
				err = -EINVAL; /* confused */
				goto failed;
			}
			err = collect_blocks_from_segsum(sbi, pseg_start, &ssi, &dsync_blocks);
			if (unlikely(err))
				goto failed;
			if (NILFS_SEG_LOGEND(&ssi)) {
				err = recover_dsync_blocks(sbi, &dsync_blocks, &nr_salvaged_block);
				if (unlikely(err))
					goto failed;
				state = RF_INIT_ST;
			}
			break; /* Fall through to try_next_pseg */
		}

	try_next_pseg:
		recovery_debug(2, "try_next_pseg: state=%d\n", state);
		if (pseg_start == ri->lsegs_end)
			break;
		pseg_start += ssi.nblocks;
		if (pseg_start < seg_end)
			continue;
		goto feed_segment;

	strayed:
		recovery_debug(2, "strayed: state=%d\n", state);
		if (pseg_start == ri->lsegs_end || alt_seg++)
			break;

	feed_segment:
		/* Looking to the next full segment */
		err = nilfs_find_next_segment(sbi, segnum, &next_segnum);
		if (unlikely(err))
			break;

		seg_seq++;
		segnum = next_segnum;
		nilfs_get_segment_range(sbi, segnum, &seg_start, &seg_end);
		pseg_start = seg_start;
		recovery_debug(2, "looking segment (seg_start=%lu, "
			       "seg_end=%lu, segnum=%lu, seg_seq=%llu)\n",
			       seg_start, seg_end, segnum, seg_seq);
	}

	if (nr_salvaged_block) {
		printk(KERN_INFO "NILFS (device %s): salvaged %lu data blocks\n",
		       sbi->s_super->s_id, nr_salvaged_block);
		ri->need_recovery = RECOVERY_ROLLFORWARD_DONE;
	}
	dispose_recovery_list(&dsync_blocks);
	return 0;

 failed:
	printk(KERN_ERR
	       "NILFS (device %s): Error roll-forwarding (err=%d, pseg block=%lu). ",
	       sbi->s_super->s_id, err, pseg_start);
	dispose_recovery_list(&dsync_blocks);
	return err;
}

/**
 * nilfs_search_checkpoint - search the latest checkpoint
 * @sbi: nilfs_sb_info
 * @nilfs: the_nilfs
 * @ri: pointer to a nilfs_recovery_info struct to store search results.
 *
 * nilfs_search_checkpoint() looks for the latest checkpoint from a partial 
 * segment pointed by the superblock.  It sets up struct the_nilfs through
 * this search.  If the_nilfs struct has already been loaded, it skips the
 * search.
 * On success, nilfs_recovery_info (ri) is filled in as follows:
 * 
 * - A DBN of inode root block is stored in ri->inode_root.
 * - In ri->need_recovery, it stores whether if the superblock has pointed
 *   the latest segment or not.
 * - Information needed for roll-forwarding is stored in
 *   ri->lsegs_start, ri->lsegs_end, ri->lsegs_start_seq, and so on.
 *
 * Return Value: On success, 0 is returned.  On error, a following negative
 * error code is returned.
 *
 * %-EINVAL - No valid segment found
 *
 * %-EIO - I/O error
 */
static int
nilfs_search_checkpoint(struct nilfs_sb_info *sbi, struct the_nilfs *nilfs,
			struct nilfs_recovery_info *ri)
{
	struct nilfs_segsum_info ssi;
	struct nilfs_cp_info cpi;
	dbn_t pseg_start, pseg_end;
	dbn_t seg_start, seg_end;   /* Starting/ending DBN of full segment */
	dbn_t cp_block;
	dbn_t end_block;
	u64 seg_seq;
	segnum_t next_segnum, segnum;
	int ret, err, alt_seg = 0;
	enum {
		LAST_PSEG_ST,      /* Looking the partial segment pointed by superblock */
		SEEK_OLDCP_REV_ST, /* Scanning older CP (seeking backward in a segment) */
		SEEK_OLDCP_FWD_ST, /* Scanning older CP (seeking forward in a segment) */
		SEEK_NEWCP_ST,     /* Scanning newer CP */
	};
	int search = LAST_PSEG_ST;   /* search mode */

	if (test_bit(THE_NILFS_LOADED, &nilfs->ns_flags)) {
		get_nilfs_cp(nilfs, ri);
		return 0;
	} 

	/*
	 * Copying context of the segment constructor into work variables.
	 * These variables are used to track the latest checkpoint.
	 * ns_last_pseg, ns_last_seq, ns_segnum are initialized at init_nilfs().
	 */
	pseg_start = nilfs->ns_last_pseg;
	seg_seq = nilfs->ns_last_seq;
	segnum = pseg_start / sbi->s_blocks_per_segment;

	/* Calculate range of segment */
	nilfs_get_segment_range(sbi, segnum, &seg_start, &seg_end);
	end_block = seg_end + 1;
	recovery_debug(2, 
		       "looking segment (seg_start=%lu, seg_end=%lu, "
		       "segnum=%lu, seg_seq=%llu)\n",
		       seg_start, seg_end, segnum, seg_seq);
	for(;;) {
		/* Load segment summary */
		ret = load_segment_summary(sbi, pseg_start, &ssi, CHECK_PAYLOAD_ON_CP_SEARCH);
		if (ret) {
			if (ret == NILFS_SEG_FAIL_IO)
				goto io_failure;
			goto strayed;
		}
		/* Checks depending on the context */
		if (seg_seq != ssi.seg_seq) {
			ret = NILFS_SEG_FAIL_SEQ;
			goto strayed;
		}
		pseg_end = pseg_start + ssi.nblocks - 1;
		if (unlikely(pseg_end > seg_end ||
			     pseg_start - ssi.prev_pseg < seg_start)) {
			ret = NILFS_SEG_FAIL_CONSISTENCY;
			goto strayed;
		}

		/* A valid pseg found .. Move the write position forward */
		alt_seg = 0;
		if (search == LAST_PSEG_ST || search == SEEK_NEWCP_ST) {
			nilfs->ns_pseg_offset = pseg_start + ssi.nblocks - seg_start;
			nilfs->ns_prev_pseg = ssi.nblocks;
			nilfs->ns_seg_seq = seg_seq;
			nilfs->ns_segnum = segnum;
		}

		if (!NILFS_SEG_HAS_MJCP(&ssi)) {
			switch (search) {
			case LAST_PSEG_ST:
				/* This will never happen because a superblock 
				   (last_segment) always points to a pseg having
				   a checkpoint. */
				ret = NILFS_SEG_FAIL_CONSISTENCY;
				break;
			case SEEK_NEWCP_ST:
				if (!ri->lsegs_start && NILFS_SEG_LOGBGN(&ssi)) {
					ri->lsegs_start = pseg_start;
					ri->lsegs_start_seq = seg_seq;
				}
				if (NILFS_SEG_LOGEND(&ssi))
					ri->lsegs_end = pseg_start;
				break;
			case SEEK_OLDCP_REV_ST:
				if (!ri->lsegs_end && NILFS_SEG_LOGEND(&ssi))
					ri->lsegs_end = pseg_start;
				if (NILFS_SEG_LOGBGN(&ssi)) {
					ri->lsegs_start = pseg_start;
					ri->lsegs_start_seq = seg_seq;
				}
				break;
			case SEEK_OLDCP_FWD_ST:
				break;
			}
			goto try_next_pseg;
		}

		/* Load checkpoint */
		cp_block = pseg_start + nilfs_seg_blocks(&ssi, 0);
		BUG_ON(pseg_end < cp_block);

		ret = load_checkpoint(sbi->s_super, cp_block, &cpi);
		recovery_debug(2, "done load_checkpoint (ret=%d)\n", ret);
		if (unlikely(ret)) {
			if (ret == NILFS_SEG_FAIL_IO)
				goto io_failure;
			goto try_next_pseg;
		}

		/* A valid checkpoint was found.  saving the context. */
		ri->inode_root = cpi.inode_root;
		ri->last_pseg = pseg_start;
		ri->last_seq = seg_seq;
		ri->last_cp = cp_block;
		ri->last_ctime = ssi.ctime;
		ri->inodes_count = cpi.inodes_count;
		ri->blocks_count = cpi.blocks_count;
		if (search != LAST_PSEG_ST)
			ri->need_recovery = RECOVERY_CP_UPDATED;

		/* Decide what to do next. */
		switch (search) {
		case LAST_PSEG_ST:
			if (nilfs->ns_mount_state & NILFS_VALID_FS)
				goto cp_found;  /* can skip roll-foward because the
						   device was unmounted cleanly */
			search = SEEK_NEWCP_ST;
			/* Fall through */
		case SEEK_NEWCP_ST:
			ri->lsegs_start = ri->lsegs_end = 0;
			    /* reset region for roll-forward */
			pseg_start += ssi.nblocks;
			if (pseg_start < seg_end)
				continue;
			goto feed_segment;
		case SEEK_OLDCP_REV_ST:
			goto cp_found;
		case SEEK_OLDCP_FWD_ST:
			search = SEEK_NEWCP_ST;
			break;
		}  /* fall through to try_next_pseg */

	try_next_pseg:
		/* Standing on a course, or met an inconsistent state */
		recovery_debug(2, "try_next_pseg: search=%d, ret=%d\n",
			       search, ret);
		switch (search) {
		case LAST_PSEG_ST:
			if (ret) {
				err = nilfs_warn_segment_error(sbi, ret);
				if (!nilfs_test_opt(sbi, RECOVERY))
					goto failed;
			}
			end_block = pseg_start;
			search = SEEK_OLDCP_REV_ST;
			/* Fall through */
		case SEEK_OLDCP_REV_ST:
			if (ssi.prev_pseg > 0) {
				pseg_start -= ssi.prev_pseg;
				continue;  /* Look previous partial segment */
			}
			break;
		case SEEK_OLDCP_FWD_ST:
			pseg_start += ssi.nblocks;
			if (pseg_start < end_block)
				continue;
			break;
		case SEEK_NEWCP_ST:
			pseg_start += ssi.nblocks;
			if (ret)  /* CP is incomplete */
				alt_seg++;
			else if (pseg_start < seg_end)
				continue;
			nilfs_invalidate_segment(nilfs, seg_start, seg_end);
			goto feed_segment;
		}  /* Fall through to strayed */

	strayed:
		/* Off the trail */
		recovery_debug(2, "strayed: search=%d, ret=%d\n", search, ret);
		switch (search) {
		case SEEK_NEWCP_ST:
			if (alt_seg++)
				goto cp_found;
			goto feed_segment;
		case LAST_PSEG_ST:
			/*
			 * This can happen if a checkpoint was written without
			 * barriers, or as a result of an I/O failure.
			 * We would shrink this fall back code in a future
			 * release; fsck should take over the job.
			 */
			nilfs_invalidate_segment(nilfs, seg_start, seg_end);
			/* Write will be resumed from the next full segment */
			if (ret) {
				err = nilfs_warn_segment_error(sbi, ret);
				if (!nilfs_test_opt(sbi, RECOVERY))
					goto failed;
			}  /* Fall through */
		case SEEK_OLDCP_REV_ST:
			if (pseg_start > seg_start) {
				end_block = pseg_start;
				pseg_start = seg_start;
				ri->lsegs_start = ri->lsegs_end = 0;
				search = SEEK_OLDCP_FWD_ST;
				continue;
			}
			break;
		case SEEK_OLDCP_FWD_ST:
			break;
		}

		/* Looking to the previous full segment */
		err = nilfs_find_prev_segment(sbi, segnum, &next_segnum);
		if (unlikely(err)) {
			if (err == -ESRCH) {
				printk(KERN_ERR
				       "NILFS: no valid segment found\n");
				err = -EINVAL;
			}
			goto failed;
		}
		seg_seq--;
		ri->lsegs_start = ri->lsegs_end = 0;
		search = SEEK_OLDCP_FWD_ST;
		goto switch_segment;

	feed_segment:
		/* Looking to the next full segment */
		err = nilfs_find_next_segment(sbi, segnum, &next_segnum);
		if (unlikely(err))
			break;  /* search == SEEK_NEWCP_ST. 
				   So, we have valid cp here. */
		seg_seq++;

	switch_segment:
		segnum = next_segnum;
		nilfs_get_segment_range(sbi, segnum, &seg_start, &seg_end);
		end_block = seg_end + 1;
		pseg_start = seg_start;
		recovery_debug(2, "looking segment (seg_start=%lu, "
			       "seg_end=%lu, segnum=%lu, seg_seq=%llu)\n",
			       seg_start, seg_end, segnum, seg_seq);
	}

 cp_found:
	/* Updating pointers relating to the latest checkpoint */
	set_nilfs_cp(nilfs, ri->inode_root,
		     ri->last_seq, ri->last_pseg, ri->last_cp, ri->last_ctime,
		     ri->inodes_count, ri->blocks_count);
	set_bit(THE_NILFS_LOADED, &nilfs->ns_flags);

	recovery_debug(1, "cp_found: ns_segnum=%lu, ns_last_seq=%llu, "
		       "ns_last_pseg=%lu (segnum from ns_last_pseg=%lu), "
		       "ns_pseg_offset=%lu, inode_root=%lu\n", 
		       nilfs->ns_segnum, nilfs->ns_last_seq,
		       nilfs->ns_last_pseg,
		       nilfs->ns_last_pseg / sbi->s_blocks_per_segment,
		       nilfs->ns_pseg_offset, ri->inode_root);
	return 0;

 io_failure:
	err = nilfs_warn_segment_error(sbi, ret);
 failed:
	return err;
}

int nilfs_load_last_segment(struct nilfs_sb_info *sbi, struct the_nilfs *nilfs)
{
	struct nilfs_sc_info *sci;
	struct nilfs_recovery_info ri;
	int err;

	memset(&ri, 0, sizeof(ri));
	down_write(&nilfs->ns_sem);   /* Guarantees atomicity of this recovery process */
	err = nilfs_search_checkpoint(sbi, nilfs, &ri);
	if (err) {
		printk(KERN_ERR "NILFS: error loading checkpoint.\n");
		goto failed;
	}
	/* End of search; initialize the inode b-tree with the root blknum */
	BUG_ON(ri.inode_root == 0);
	err = nilfs_btree_init(&sbi->s_inode_root, NILFS_BTREE_INODE,
			       ri.inode_root, sbi->s_super);
	if (unlikely(err)) {
		BUG_ON(err == -NILFS_BTREE_EINVAL);
		err = nilfs_convert_btree_error(err);
		printk(KERN_ERR "NILFS: error initializing inode B-tree.\n");
		goto failed;
	}

	atomic_set(&sbi->s_inodes_count, ri.inodes_count);
	atomic_set(&sbi->s_blocks_count, ri.blocks_count);

	if (ri.lsegs_start && ri.lsegs_end) {
		err = nilfs_do_roll_forward(sbi, nilfs, &ri);
		if (err)
			goto failed_btree;
	}

	sci = NILFS_SC(sbi);
	if (sci) {
		sci->sc_ctime = ri.last_ctime;
		nilfs_segctor_reset(sci);
	}
	up_write(&nilfs->ns_sem);
	return ri.need_recovery;

 failed_btree:
	nilfs_btree_clear(&sbi->s_inode_root);

 failed:
	up_write(&nilfs->ns_sem);
	return err;
}

int nilfs_load_last_segment_for_remount(struct nilfs_sb_info *sbi, struct the_nilfs *nilfs)
{
	struct nilfs_sc_info *sci;
	struct nilfs_recovery_info ri;
	int err;

	memset(&ri, 0, sizeof(ri));
	down_write(&nilfs->ns_sem);   /* Guarantees atomicity of this recovery process */
	err = nilfs_search_checkpoint(sbi, nilfs, &ri);
	if (err) {
		printk(KERN_ERR "NILFS (device %s): error loading the latest segment.\n",
		       sbi->s_super->s_id);
		goto failed;
	}

	if (ri.inode_root != NILFS_BTREE_ROOT_DBN(&sbi->s_inode_root) ||
	    ri.last_pseg != sbi->s_snapshot_pseg) {
		printk(KERN_WARNING "NILFS (device %s): couldn't "
		       "remount because the current RO-mount is not the latest one.\n",
		       sbi->s_super->s_id);
		err = -EINVAL;
		goto failed;
	}

	if (ri.lsegs_start && ri.lsegs_end) {
		err = nilfs_do_roll_forward(sbi, nilfs, &ri);
		if (err)
			goto failed;
	}

	sci = NILFS_SC(sbi);
	BUG_ON(!sci);
	sci->sc_ctime = ri.last_ctime;
	nilfs_segctor_reset(sci);
	return 0;
 failed:
	up_write(&nilfs->ns_sem);
	return err;
}

int nilfs_load_snapshot(struct nilfs_sb_info *sbi)
{
	struct nilfs_segsum_info ssi;
	struct nilfs_cp_info cpi;
	dbn_t seg_start, seg_end; /* start and end DBN of full segment */
	dbn_t pseg_end, cp_block;
	segnum_t snapshot_segnum;
	int ret;

	snapshot_segnum = sbi->s_snapshot_pseg / sbi->s_blocks_per_segment;
	if (snapshot_segnum >= sbi->s_nsegment) {
		ret = NILFS_SEG_BAD_SNAPSHOT;
		goto failed;
	}
	ret = load_segment_summary(sbi, sbi->s_snapshot_pseg, &ssi, CHECK_PAYLOAD_ON_CP_SEARCH);
	if (ret)
		goto failed;

	pseg_end = sbi->s_snapshot_pseg + ssi.nblocks - 1;
	nilfs_get_segment_range(sbi, snapshot_segnum, &seg_start, &seg_end);
	if (pseg_end > seg_end) {
		ret = NILFS_SEG_FAIL_CONSISTENCY;
		goto failed;
	}

	cp_block = sbi->s_snapshot_cp = sbi->s_snapshot_pseg + nilfs_seg_blocks(&ssi, 0);
	if (pseg_end < cp_block) {
		ret = ((cp_block == pseg_end + 1) ?
		       NILFS_SEG_HAS_NO_CP : NILFS_SEG_FAIL_CONSISTENCY);
		goto failed;
	}

	ret = load_checkpoint(sbi->s_super, cp_block, &cpi);
	if (ret)
		goto failed;

	atomic_set(&sbi->s_inodes_count, cpi.inodes_count);
	atomic_set(&sbi->s_blocks_count, cpi.blocks_count);

	ret = nilfs_btree_init(&sbi->s_inode_root, NILFS_BTREE_INODE, cpi.inode_root,
			       sbi->s_super);
	if (unlikely(ret < 0)) {
		BUG_ON(ret == -NILFS_BTREE_EINVAL);
		ret = nilfs_convert_btree_error(ret);
		goto failed;
	}

	recovery_debug(1, "got btree inode root (dbn=%lu)\n", cpi.inode_root);
	ret = 0;
 failed:
	return ret;
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
