/*
 * page.c - buffer/page management for NILFS
 *
 * Copyright (C) 2005, 2006 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * page.c,v 1.240 2006/07/13 05:29:19 ryusuke Exp
 *
 * Modified for NILFS by Amagai Yoshiji <amagai@osrg.net>,
 *                       Ryusuke Konishi <ryusuke@osrg.net>
 */
/*
 *  linux/fs/buffer.c
 *
 *  Copyright (C) 1991, 1992, 2002  Linus Torvalds
 */

#include <linux/pagemap.h>
#include <linux/writeback.h>
#include <linux/swap.h>
#include "nilfs.h"

/*
 * static functions copied from fs/buffer.c
 *	link_dev_buffers()
 *	init_page_buffers()
 *	end_buffer_read_sync()
 *	__bread_slow()
 */

static inline void
link_dev_buffers(struct page *page, struct buffer_head *head)
{
	struct buffer_head *bh, *tail;

	bh = head;
	do {
		tail = bh;
		bh = bh->b_this_page;
	} while (bh);
	tail->b_this_page = head;
	attach_page_buffers(page, head);
}

/*
 * Initialise the state of a blockdev page's buffers.
 */ 
static void
init_page_buffers(struct page *page, struct block_device *bdev,
			sector_t block, int size)
{
	struct buffer_head *head = page_buffers(page);
	struct buffer_head *bh = head;
	int uptodate = PageUptodate(page);

	do {
		if (!buffer_mapped(bh)) {
			init_buffer(bh, NULL, NULL);
			bh->b_bdev = bdev;
			bh->b_blocknr = block;
			if (uptodate)
				set_buffer_uptodate(bh);
			set_buffer_mapped(bh);
		}
		block++;
		bh = bh->b_this_page;
	} while (bh != head);
}

struct buffer_head *nilfs_bread_slow(struct buffer_head *bh)
{
	lock_buffer(bh);
	if (buffer_uptodate(bh)) {
		unlock_buffer(bh);
		return bh;
	} else {
		get_bh(bh);
		page_debug(2, "try to read block (dbn=%llu)\n", bh->b_blocknr);
		bh->b_end_io = end_buffer_read_sync;
		submit_bh(READ, bh);
		wait_on_buffer(bh);
		if (buffer_uptodate(bh))
			return bh;
	}
	nilfs_brelse(bh);
	return NULL;
	/*
	 * __bread_slow() releases buffer count when it fails to read.
	 * The caller must not release buffer_head if NULL is returned.
	 * Note that an increment by get_bh() in __bread_slow() is consumed by
	 * the BIO submission.
	 */
}

/* end copied functions from fs/buffer.c */

/**
 * __nilfs_alloc_buffer_page - allocate a private page with buffer heads
 *
 * Return Value: On success, a pointer to the allocated page is returned.
 * On error, NULL is returned.
 */
static struct page *
__nilfs_alloc_buffer_page(struct block_device *bdev, sector_t block, int size)
{
	struct buffer_head *bufs;
	struct page *page;

	might_sleep();

	page = alloc_page(GFP_NOFS); /* return a page whose page_count is 1 */
	if (unlikely(!page))
		return NULL;

	BUG_ON(PageLRU(page));

	lock_page(page);
#if 0 /* page LRU debug */
	INIT_LIST_HEAD(&page->lru);
#endif
	bufs = alloc_page_buffers(page, size, 0);
	if (unlikely(!bufs)) {
		unlock_page(page);
		__free_page(page);
		return NULL;
	}
	link_dev_buffers(page, bufs);
	init_page_buffers(page, bdev, block, size);
#ifdef CONFIG_NILFS_DEBUG
	/*
	 * Count only the head of buffers here.
	 * Callers that uses multiple blocks should increment
	 * getblkbh counters severally.
	 */
	atomic_inc(&nilfs_getblkbh_n);
#endif
	return page;
}

/**
 * nilfs_getblkbh - allocate private buffer(s)
 *
 * Return Value: On success, a pointer to the allocated buffer heads
 * is returned.  On error, NULL is returned.
 */
struct buffer_head *
nilfs_getblkbh(struct block_device *bdev, sector_t block, int size)
{
	struct page *page = __nilfs_alloc_buffer_page(bdev, block, size);

	if (likely(page)) {
		struct buffer_head *bufs = page_buffers(page);

		unlock_page(page);
		return bufs;
	}
	return NULL;
}

/**
 * __nilfs_free_page - deallocate a private page.
 * page: page to be freed
 */
static inline void __nilfs_free_page(struct page *page)
{
#if 0 /* page LRU debug */
	if (unlikely(!list_empty(&page->lru))) {
		unlock_page(page);
		page_warn("*** PAGE LRU WRONG **** (page=%p)\n", page);
		return;
	}
#endif
	if (unlikely(page_count(page) != 1)) {
		page_warn("*** INVALID PAGE COUNT *** (page=%p, count=%d)\n",
			  page, page_count(page));
		return;
	}
	unlock_page(page);
	__free_page(page);
}

/**
 * __nilfs_free_buffer_page - try to free a private buffer page.
 * page: page to be freed
 *
 * This function should be called when the reference count of
 * each buffer becomes zero.
 * The page is freed only when it has no active buffers.
 * In that case, buffer heads attached to the page are destroyed.
 */
static void __nilfs_free_buffer_page(struct page *page)
{
#ifdef CONFIG_NILFS_DEBUG
	nilfs_count_freeing_bh(page_buffers(page));
#endif
	if (unlikely(PageLocked(page))) {
		page_warn("*** PAGE LOCKED *** page=%p\n", page);
		return;
	}
	page_debug(3, "locking page %p\n", page);
	lock_page(page);
	page_debug(3, "calling try_to_free_buffers (page=%p)\n", page);
	if (!try_to_free_buffers(page)) {
		unlock_page(page);
		page_debug(3, "try_to_free_buffers failed (page=%p)\n", page);
		return;
	}
	__nilfs_free_page(page);
}

void nilfs_putblkbh(struct buffer_head *bh)
{
	struct page *page = bh->b_page;

	page_debug(3, "called (bh=%p, page=%p, b_count=%d, page_count=%d)\n",
		   bh, page, atomic_read(&bh->b_count), page_count(page));

	if (unlikely(atomic_read(&bh->b_count))) {
		page_warn("*** BUFFER COUNT WRONG *** "
			  "(page=%p, bh=%p, count=%d)\n",
			  page, bh, atomic_read(&bh->b_count));
		if (atomic_read(&bh->b_count) > 0) {
			page_warn("return because b_count positive\n");
			return;
		}
	}
#if 0  /* left for exhaustive check of page */
	BH_DEBUG(bh, "bfr free");
#endif
	__nilfs_free_buffer_page(page);
}

int __nilfs_brelse(struct super_block *sb, struct buffer_head *bh)
{
	struct page *page = bh->b_page;
	struct buffer_head *bufs = page_buffers(page);

	if (buffer_nilfs_allocated(bufs)) {
		__brelse(bh);
		if (!atomic_read(&bh->b_count)) {
#if 0  /* left for exhaustive check of page */
			BH_DEBUG(bh, "rls allocated bh");
#endif
			if (!sb)
				__nilfs_free_buffer_page(page);
			else {
#ifdef CONFIG_NILFS_DEBUG
				/*
				 * Functions around the inactive node page list 
				 * assumes the page consists of a single buffer
				 * head.  The following is a work-around.
				 */
				if (bh != bufs)
					nilfs_count_freeing_bh(bufs);
#endif /* CONFIG_NILFS_DEBUG */
				nilfs_put_allocated_page(page, sb);
			}
		}
		return 1;
	}
        if (buffer_nilfs_node(bufs)) {
		if (buffer_nilfs_bbt_node(bufs)) {
			nilfs_put_file_node_blk(bh);
			return 1;
		}
		if (buffer_nilfs_ibt_node(bufs)) {
			nilfs_put_inode_node_blk(bh);
			return 1;
		}
	}
	return 0;
}

/**
 * nilfs_copy_buffers - make a clone copy of page with buffers
 * @bh: buffer head of start block
 * @nblock: number of blocks to be copied
 *
 * On success, nilfs_copy_buffers() returns buffers whose page is locked.
 */
struct buffer_head *nilfs_copy_buffers(struct buffer_head *bh, int nblock)
{
	struct page *page;
	struct buffer_head *bufs;
	void *kaddr;

	BUG_ON(nblock <= 0);
	page = __nilfs_alloc_buffer_page(bh->b_bdev, bh->b_blocknr, bh->b_size);
	if (unlikely(!page))
		return NULL;

	bufs = page_buffers(page);
	kaddr = kmap_atomic(page, KM_USER0);
	memcpy(bufs->b_data, kaddr + bh_offset(bh), bh->b_size * nblock);
	kunmap_atomic(kaddr, KM_USER0);

	return bufs;
}

int nilfs_page_buffers_dirty(struct page *page)
{
	struct buffer_head *bh, *head;
	
	if (!page_has_buffers(page))
		return 0;
	bh = head = page_buffers(page);
	do {
		if (buffer_dirty(bh))
			return 1;
		bh = bh->b_this_page;
	} while (bh != head);
	return 0;
}

/*
 * inode map
 */

#if NEED_FIND_TRYLOCK_PAGE
static struct page *
nilfs_find_trylock_page(struct address_space *mapping, unsigned long offset)
{
	struct page *page;

	read_lock_irq(&mapping->tree_lock);
	page = radix_tree_lookup(&mapping->page_tree, offset);
	if (page && TestSetPageLocked(page))
		page = NULL;
	read_unlock_irq(&mapping->tree_lock);
	return page;
}
#else
#define nilfs_find_trylock_page  find_trylock_page
#endif

/*
 * static inline struct buffer_head *
 * nilfs_page_get_nth_block(struct page *page, unsigned int count);
 * now defined in nilfs.h.
 */
struct buffer_head *
nilfs_find_get_inode_block(struct super_block *sb, ino_t ino)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	struct buffer_head *ret = NULL;
	struct page *page;
	sector_t block = ino / sbi->s_inodes_per_block;
	pgoff_t offset;

	offset = block >> (PAGE_CACHE_SHIFT - sb->s_blocksize_bits);
	page = nilfs_find_trylock_page(sbi->s_mapping, offset);
	page_debug(2, "page=%p, ino=%lu, offset=%lu\n", page, ino, offset);
	if (!page)
		return NULL;

	if (likely(page_has_buffers(page))) {
		unsigned int bcnt;

		bcnt = (unsigned int)(block -
				      ((sector_t)offset <<
				       (PAGE_CACHE_SHIFT -
					sb->s_blocksize_bits)));
		ret = nilfs_page_get_nth_block(page, bcnt);
	}
	unlock_page(page); /* call tree of unlock_page() should be verified */
	return ret;
}

static struct buffer_head *
nilfs_init_inode_blocks(struct nilfs_sb_info *sbi, struct buffer_head *bh,
			sector_t first_block,  sector_t block)
{
	struct buffer_head *b = bh, *bh_result = NULL;
	ino_t ino = first_block * sbi->s_inodes_per_block;
	/* first inode number of this page */
	do {
		b->b_blocknr = 0; /* re-initialize disk block number */
		nilfs_init_iblock(b, ino, sbi);
		if (block == first_block++) {
			bh_result = b;
			get_bh(b);
		}
		ino += sbi->s_inodes_per_block;
		b = b->b_this_page;
	} while(b != bh);
	
	return bh_result;
}

struct buffer_head *
nilfs_create_inode_block(struct super_block *sb, ino_t ino)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	struct page *page;
	sector_t block = ino / sbi->s_inodes_per_block;
	sector_t first_block;
	pgoff_t offset;
	struct buffer_head *bh;

	offset = block >> (PAGE_CACHE_SHIFT - sb->s_blocksize_bits);
	
	page_debug(2, "0: ino=%lu, block=%llu, offset=%lu\n",
		   ino, block, offset);
	
	page = find_or_create_page(sbi->s_mapping, offset, GFP_NOFS);
	if (!page)
		return NULL;
	
	BUG_ON(!PageLocked(page));
	
	first_block = (sector_t)offset <<
		(PAGE_CACHE_SHIFT - sb->s_blocksize_bits);
	
	if (page_has_buffers(page)) {
		bh = page_buffers(page);
		if (bh->b_size == sb->s_blocksize) {
			bh = nilfs_page_get_nth_block(page,
						      block - first_block);
			unlock_page(page);
			page_cache_release(page);
			return bh;
		}
		if (!try_to_free_buffers(page))
			goto failed;
		BUG();
	}

	bh = alloc_page_buffers(page, sb->s_blocksize, 0);
	if (unlikely(!bh))
		goto failed;

	/* Link the page to the buffers and initialize them. */
	link_dev_buffers(page, bh);
	init_page_buffers(page, sb->s_bdev, 0, sb->s_blocksize);
	bh = nilfs_init_inode_blocks(sbi, bh, first_block, block);
#ifdef NILFS_PAGE_COUNT
	atomic_inc(&nilfs_allocated_inode_page_n);
#endif
	unlock_page(page);
	page_cache_release(page);

#ifdef NILFS_BH_DEBUG
	BH_DEBUG(bh, "linked");
#endif
	return bh;

 failed:
	unlock_page(page);
	page_cache_release(page);
	return NULL;
}

struct buffer_head *
nilfs_inode_bread(struct super_block *sb, ino_t ino)
{
	struct buffer_head *bh;
	struct nilfs_inode_hdr *hdr;
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	int nf;
	int err;

	bh = nilfs_find_get_inode_block(sb, ino);
	if (!bh) { /* page was locked or no page was found in a page cache */
		sector_t block = ino / sbi->s_inodes_per_block;
		dbn_t dbn = 0;
		int res;
		
		/* 
		 * btree_lookup may return buffer_head pointer in dbn (res == 1).
		 * when page exists in radix-tree but locked.
		 */
		res = nilfs_btree_lookup(&sbi->s_inode_root, block, &dbn);
		page_debug(2, "lookup res=%d, block=%llu, dbn=%lu\n",
			   res, block, dbn);
		if (unlikely(res < 0)) {
			err = nilfs_handle_ibt_error(res, __FUNCTION__, sb);
			/* -NILFS_BTREE_ENOKEY is mapped to -EINVAL.
			   In that case, the btree is deemed broken. */
			goto failed;
		}

		bh = nilfs_create_inode_block(sb, ino);
		if (unlikely(!bh))
			goto failed_mem;

		if (res == 0)   /* a dbn is a dbn */
			bh->b_blocknr = (sector_t)dbn;
	}
	touch_buffer(bh);
	if (!bh->b_blocknr)
		page_debug(2, "Trying to read inode with dbn=0 (ino=%lu); "
			   "may be on memory new inode.\n", ino);

	wait_on_buffer(bh);
	if (bh->b_blocknr && !buffer_uptodate(bh)) {
		err = -EIO;
		bh = nilfs_bread_slow(bh);
		if (unlikely(!bh))
			goto failed;
			/* We don't have to release buffer_head here.
			 * For details, read comments on nilfs_bread_slow() */

		if (!(sb->s_flags & MS_RDONLY)) {
			/* care for new page block (not yet read) */
			hdr = (struct nilfs_inode_hdr *)bh->b_data;
			down(&sbi->s_free_inodes_sem);
			if (nilfs_few_free_inodes((nf = le32_to_cpu(hdr->ih_nfree)), sbi) &&
			    !test_set_buffer_nilfs_inode_on_list(bh)) {
				struct nilfs_free_inode_list *fi;
				struct nilfs_sb_info *sbi = NILFS_SB(sb);

				fi = kmalloc(sizeof(struct nilfs_free_inode_list),
					     GFP_NOFS);
				if (unlikely(fi == NULL)) {
					/* freeing newly allocated bh */
					clear_buffer_nilfs_inode_on_list(bh);
					up(&sbi->s_free_inodes_sem);
					brelse(bh);
					goto failed_mem;
				}
				fi->free_bh = bh;
				get_bh(bh); /* link to nilfs_free_inode_list */
				sbi->s_free_inodes_len++;
				sbi->s_free_inodes_count += nf;
				list_add_tail(&fi->buffers, &sbi->s_free_inodes);
				page_debug(2, "list_add_tail %ld %p\n", ino, bh);
			}
			up(&sbi->s_free_inodes_sem);
		}
	}
	return bh;

 failed_mem:
	err = -ENOMEM;

 failed:
	return ERR_PTR(err);
}

/* free all entries on sbi->s_free_inodes */
void
nilfs_free_free_inode_list(struct super_block *sb)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	struct nilfs_free_inode_list *fi;
	struct buffer_head *bh;
	struct nilfs_inode_hdr *hdr;
	ino_t ino;
	int err;

	/* must care for memory nilfs_inode */
	BUG_ON(sbi == NULL);
	nilfs_lock_segment(sb);
	down(&sbi->s_free_inodes_sem);
	inode_debug(2, "begin\n");
	while (!list_empty(&sbi->s_free_inodes)) {
		fi = list_entry(sbi->s_free_inodes.next,
				struct nilfs_free_inode_list, buffers);
		bh = fi->free_bh;
		clear_buffer_nilfs_inode_on_list(bh);
		hdr = (struct nilfs_inode_hdr *)bh->b_data;
		fi->free_bh = NULL;
		list_del_init(&fi->buffers);
		sbi->s_free_inodes_len--;
		sbi->s_free_inodes_count -= le32_to_cpu(hdr->ih_nfree);
		kfree(fi);

		inode_debug(2, "ino %llu nfree %u\n", le64_to_cpu(hdr->ih_ino),
			    le32_to_cpu(hdr->ih_nfree));
		if (le32_to_cpu(hdr->ih_nfree) == sbi->s_inodes_per_block) {
			ino = le64_to_cpu(hdr->ih_ino);
			inode_debug(2, "delete unused inode from btree ino %lu\n",
				    ino);
			err = nilfs_btree_delete(&sbi->s_inode_root,
						 ino
						 / sbi->s_inodes_per_block);
			if (err < 0) {
				inode_debug(0, "cannot delete inode buffer "
					    "from btree, err=%d\n", err);
				nilfs_handle_ibt_error(err, __FUNCTION__, sb);
			}
		}
		put_bh(bh);
	}
	inode_debug(1, "end sbi->s_free_inodes_len/count %ld %ld\n",
		    sbi->s_free_inodes_len, sbi->s_free_inodes_count);
	up(&sbi->s_free_inodes_sem);
	nilfs_unlock_segment(sb);
}

/*
 * Note:
 *  this is almost a copy of nilfs_free_free_inode_list().
 *  these may be unified.
 */
void
nilfs_clean_free_inode_list(struct super_block *sb)
{
	struct nilfs_sb_info *sbi;
	struct nilfs_free_inode_list *fi;	
	struct buffer_head *bh;
	struct nilfs_inode_hdr *hdr;

	/* must care for memory nilfs_inode */

	if (sb->s_fs_info == NULL) {
		inode_debug(2, "null sb->s_fs_info\n");
		return;
	}
	sbi = NILFS_SB(sb);
	down(&sbi->s_free_inodes_sem);
	inode_debug(2, "begin\n");
	while (!list_empty(&sbi->s_free_inodes)) {
		fi = list_entry(sbi->s_free_inodes.next,
				struct nilfs_free_inode_list, buffers);
		bh = fi->free_bh;

		if (bh != NULL) {
			nilfs_warning(sb, __FUNCTION__,
				      "put buffer %p\n", bh);
			clear_buffer_nilfs_inode_on_list(bh);
			hdr = (struct nilfs_inode_hdr *)bh->b_data;
			sbi->s_free_inodes_len--;
			sbi->s_free_inodes_count -= le32_to_cpu(hdr->ih_nfree);
			fi->free_bh = NULL;
			put_bh(bh);
		}
		list_del_init(&fi->buffers);
		inode_debug(2, "free %p\n", fi);
		kfree(fi);
	}
	inode_debug(1, "end s_free_inodes_len/count %ld %ld\n",
		    sbi->s_free_inodes_len, sbi->s_free_inodes_count);
	up(&sbi->s_free_inodes_sem);
}

void
nilfs_free_inode_buffers(struct super_block *sb)
{
	struct nilfs_sb_info *sbi = NILFS_SB(sb);

	page_debug(2, "Trying to invalidate inode pages (s_mapping=%p)\n",
		   sbi->s_mapping);
	invalidate_inode_pages(sbi->s_mapping);
	page_debug(2, "Trying to truncate inode pages (s_mapping=%p)\n",
		   sbi->s_mapping);
	truncate_inode_pages(sbi->s_mapping, 0);
}

/*
 * nilfs_release_allocated_page()
 *  called from nilfs_releasepage()
 *  must fail (return 0)
 *  this function should be never called,
 *  because PageLRU(page) should be 0 for
 *  nilfs_allocated pages.
 */
int
nilfs_release_allocated_page(struct page *page, gfp_t gfp_mask)
{
	page_warn("page %p, mask %d\n", page, gfp_mask);
	return 0;
}

/*
 * nilfs_invalidate_allocated_page()
 *  same as nilfs_release_allocated_page()
 */
int
nilfs_invalidate_allocated_page(struct page *page, unsigned long offset)
{
	page_warn("page %p, offset %lu\n", page, offset);
	return 0;
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
