/*
 * nilfs_fs.h - NILFS on-disk structures and common declarations
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * nilfs_fs.h,v 1.29 2006/07/18 08:49:18 ryusuke Exp
 */
/*
 *  linux/include/linux/ext2_fs.h
 *
 * Copyright (C) 1992, 1993, 1994, 1995
 * Remy Card (card@masi.ibp.fr)
 * Laboratoire MASI - Institut Blaise Pascal
 * Universite Pierre et Marie Curie (Paris VI)
 *
 *  from
 *
 *  linux/include/linux/minix_fs.h
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 */

#ifndef _LINUX_NILFS_FS_H
#define _LINUX_NILFS_FS_H

#include <linux/types.h>

/*
 * Linux VFS sector size (used when submitting BIO)
 */
#define NILFS_SECTOR_BITS       9
#define NILFS_SECTOR_SIZE       (1 << SECTOR_BITS)

/*
 * NILFS filesystem version
 */
#define NILFS_VERSION           "1.0.10"

/*
 * Inode flags stored in nilfs_inode and on-memory nilfs inode
 *
 * We define these flags based on ext2-fs because of the 
 * compatibility reason; to avoid problems in chattr(1)
 */
#define NILFS_SECRM_FL          0x00000001 /* Secure deletion */
#define NILFS_UNRM_FL           0x00000002 /* Undelete */
#define NILFS_SYNC_FL           0x00000008 /* Synchronous updates */
#define NILFS_IMMUTABLE_FL      0x00000010 /* Immutable file */
#define NILFS_APPEND_FL         0x00000020 /* writes to file may only append */
#define NILFS_NODUMP_FL         0x00000040 /* do not dump file */
#define NILFS_NOATIME_FL        0x00000080 /* do not update atime */
/* Reserved for compression usage... */
#define NILFS_NOTAIL_FL         0x00008000 /* file tail should not be merged */
#define NILFS_DIRSYNC_FL        0x00010000 /* dirsync behaviour (directories only) */

#define NILFS_FL_USER_VISIBLE	0x0003DFFF /* User visible flags */
#define NILFS_FL_USER_MODIFIABLE 0x000380FF /* User modifiable flags */

/* Flags for inode allocators */
#define NILFS_INODE_NEW		0x10000000 /* i_mode has free link chain */
#define NILFS_INODE_UNUSED	0x20000000 /* i_mode has free link chain */

/*
 * Structure of an inode on the disk
 */
struct nilfs_inode {
	__le64	i_blocks;	/* Blocks count */	/*  8 */
	__le64	i_size;		/* Size in bytes */	/* 16 */
	__le64	i_ctime;	/* Creation time */	/* 24 */
	__le64	i_mtime;	/* Modification time */	/* 32 */
	__le64	i_dtime;	/* Deletion Time */	/* 40 */
	__le32	i_uid;		/* Owner Uid */		/* 44 */
	__le32	i_gid;		/* Group Id */		/* 48 */
	__le16	i_mode;		/* File mode */		/* 50 */
	__le16	i_links_count;	/* Links count */	/* 52 */
	__le32	i_flags;	/* File flags */	/* 56 */
	__le32	i_version;	/* inode version for GC */ /* 60 */

	__le64  i_block_root;	/* Data Block Btree Root */ /* 68 */

	__le32	i_generation;	/* File version (for NFS) */	/* 72 */
	__le32	i_file_acl;	/* File ACL */		/* 76 */
	__le32	i_dir_acl;	/* Directory ACL */	/* 80 */
} __attribute__((__packed__));
/*
 * For special files, we use i_block_root to store a device code.
 * The device code is encoded/decoded through 
 * new_encode_dev()/new_decode_dev().
 * (We don't have to support old encoding method.)
 */

/* inode header on disk inode block */
struct nilfs_inode_hdr {
	__le64	ih_ino;		/* first inode number in this block */
	__le16	ih_nfree;	/* # of free inode in this block */
	__le16	ih_flags;
        __le32  ih_free;        /* first free inode in this block if 0 < ih_nfree */
};

/*
 * Maximal mount counts
 */
#define NILFS_DFL_MAX_MNT_COUNT         50      /* 50 mounts */

/*
 * Retry count for IO errors
 *
 * Basically retry for the same block is entrusted to device drivers.
 * Recovery process gives up any further attempts to find a last valid
 * segment when IO errors repeatedly occur.
 */
#define NILFS_EIO_RETRY_COUNT           4

/*
 * File system states (sbp->s_state, nilfs->ns_mount_state)
 */
#define NILFS_VALID_FS                  0x0001  /* Unmounted cleanly */
#define NILFS_ERROR_FS                  0x0002  /* Errors detected */

/*
 * Mount flags (sbi->s_mount_opt)
 */
#define NILFS_MOUNT_FULL_CHECK          0x0001  /* Do full check */
#define NILFS_MOUNT_ERROR_MODE          0x0070  /* Error mode mask */
#define NILFS_MOUNT_ERRORS_CONT         0x0010  /* Continue on errors */
#define NILFS_MOUNT_ERRORS_RO           0x0020  /* Remount fs ro on errors */
#define NILFS_MOUNT_ERRORS_PANIC        0x0040  /* Panic on errors */
#define NILFS_MOUNT_RECOVERY            0x0300  /* Recovery mode flags */
#define NILFS_MOUNT_RECOVERY_QUICK      0x0100  /* Quick recovery mode */
#define NILFS_MOUNT_RECOVERY_FULL       0x0200  /* Full scan recovery mode */
#define NILFS_MOUNT_SYNC_MODE           0x0c00  /* Sync mode flags */
#define NILFS_MOUNT_SYNC_BIO            0x0400  /* BIO sync mode */
#define NILFS_MOUNT_SYNC_SEGMENT        0x0800  /* Segment sync mode */
#define NILFS_MOUNT_BARRIER             0x1000  /* Use block barriers */
#define NILFS_MOUNT_SNAPSHOT            0x0080  /* Snapshot flag */

#define NILFS_MOUNT_STRICT_ORDER        0x2000  /* Apply strict in-order
                                                   semantics also for data */
#define NILFS_MOUNT_PASSIVE             0x4000  /* Passive mode mount */

/*
 * Structure of the super block on disk
 */
struct nilfs_super_block {
	__le32	s_rev_level;		/* Revision level */
	__le16	s_minor_rev_level; 	/* minor revision level */
	__le16	s_magic;		/* Magic signature */
	__le16  s_bytes;                /* Bytes count of CRC calculation
					   for this structure. s_reserved
					   is excluded. */
	__le16  s_pad1[1];
	__le32  s_crc_seed;             /* Seed value of CRC calculation */
	__le32	s_sum;	                /* Check sum of super block */

	__le32  s_nsegment;		/* number of segment in filesystem */
	__le64  s_dev_size;             /* block device size in bytes */
	__le64	s_first_data_block;	/* 1st seg disk block number */
        __le32	s_log_block_size;	/* Block size represented as follows
					   blocksize =
					      1 << (s_log_block_size + 10) */
        __le32  s_blocks_per_segment;   /* number of blocks per full segment */
	__le32	s_r_segments_count;	/* Reserved segments count */ /* or __le16 */

        __le64  s_last_segment;		/* disk block addr seg written last */
	__le64  s_last_seq;             /* seq. number of seg written last */
	__le64	s_free_blocks_count;	/* Free blocks count */

	__le64	s_ctime;		/* Creation time (execution time of
					   newfs) */
	__le64	s_mtime;		/* Mount time */
	__le64	s_wtime;		/* Write time */
	__le16	s_mnt_count;		/* Mount count */
	__le16	s_max_mnt_count;	/* Maximal mount count */
	__le16	s_state;		/* File system state */
	__le16	s_errors;		/* Behaviour when detecting errors */
	__le64	s_lastcheck;		/* time of last check */

	__le32	s_checkinterval;	/* max. time between checks */
	__le32	s_creator_os;		/* OS */
	__le16	s_def_resuid;		/* Default uid for reserved blocks */
	__le16	s_def_resgid;		/* Default gid for reserved blocks */
	__le32	s_first_ino; 		/* First non-reserved inode */ /* or __le16 */
	__le16  s_inode_size; 		/* size of inode structure */
	__le16  s_pad2[1];
	/*
	 * 34 words (in 4-byte word) are consumed above, and
	 * 26 words below except for s_reserved.
	 * Number of s_reserved = 256 - (34 + 26) = 196
	 */
	__u8	s_uuid[16];		/* 128-bit uuid for volume */
	char	s_volume_name[16]; 	/* volume name */
	char	s_last_mounted[64]; 	/* directory where last mounted */

	__le32  s_c_interval;           /* Commit interval of segment */
	__le32  s_c_block_max;          /* Threshold of data amount for 
					   the segment construction */
	__u32	s_reserved[196];        /* Padding to the end of the block */
} __attribute__((__packed__));

/*
 * Revision levels
 */
#define NILFS_CURRENT_REV      1        /* current major revision */
#define NILFS_MINOR_REV        0        /* minor revision */

/*
 * Bytes count of super_block for CRC-calculation
 */
#define NILFS_SB_BYTES   ((int)(((struct nilfs_super_block *)0)->s_reserved))

/*
 * Special inode number
 */
#define NILFS_ROOT_INO         2        /* Root inode */
#define NILFS_ATIME_INO        4        /* .atime file */
#define NILFS_SKETCH_INO       10       /* .sketch file */
#define NILFS_USER_INO         11       /* Fisrt user's file inode number */

#define NILFS_SB_OFFSET_BYTES  1024     /* byte offset of nilfs superblock */
#define NILFS_SUPER_MAGIC      0x3434   /* NILFS filesystem  magic number */
#define NILFS_SEG_MIN_BLOCKS   16       /* Minimum number of blocks in
					   a full segment */
#define NILFS_PSEG_MIN_BLOCKS  2        /* Minimum number of blocks in
					   a partial segment */

/*
 * Maximal count of links to a file
 */
#define NILFS_LINK_MAX		32000

/*
 * Value for the sync control (sysfs attribute or ioctl)
 */
#define NILFS_SYNC_MJCP        1        /* Requesting a logical segment
					   having a major CP */
/* Other positive values are reserved for future use */

/*
 * Result value for the sync control
 * On success,
 * - A positive value is returned for ioctl().
 * - A negative value is written into the sysfs attribute.
 */
#define NILFS_SYNC_DONE        0        /* Sync request was completed.
					   (error returns are included) */
#define NILFS_SYNC_SKIPPED     1        /* Sync request was skipped (no updates) */


/*
 * Structure of a directory entry
 *  (Same as ext2)
 */

#define NILFS_NAME_LEN 255

/*
 * The new version of the directory entry.  Since V0 structures are
 * stored in intel byte order, and the name_len field could never be
 * bigger than 255 chars, it's safe to reclaim the extra byte for the
 * file_type field.
 */
struct nilfs_dir_entry {
	__le64	inode;			/* Inode number */
	__le16	rec_len;		/* Directory entry length */
	__u8	name_len;		/* Name length */
	__u8	file_type;
	char	name[NILFS_NAME_LEN];	/* File name */
	char    pad;
} __attribute__((__packed__));

/*
 * NILFS directory file types.  Only the low 3 bits are used.  The
 * other bits are reserved for now.
 */
enum {
	NILFS_FT_UNKNOWN,
	NILFS_FT_REG_FILE,
	NILFS_FT_DIR,
	NILFS_FT_CHRDEV,
	NILFS_FT_BLKDEV,
	NILFS_FT_FIFO,
	NILFS_FT_SOCK,
	NILFS_FT_SYMLINK,
	NILFS_FT_MAX
};

/*
 * NILFS_DIR_PAD defines the directory entries boundaries
 *
 * NOTE: It must be a multiple of 4
 */
#define NILFS_DIR_PAD		 	4
#define NILFS_DIR_ROUND 		(NILFS_DIR_PAD - 1)
#define NILFS_DIR_REC_LEN(name_len)	(((name_len) + 12 + NILFS_DIR_ROUND) & \
					~NILFS_DIR_ROUND)

/*
 * pertial segment summary, on disk
 */
struct nilfs_seg_summary {
	__le32	ss_datasum;	/* check sum of data */
	__le32	ss_sumsum;	/* check sum of summary block */
        __le64	ss_seq;		/* sequential number per full segment */
	__le16	ss_magic;	/* segment summary magic number */
	__le16	ss_nblocks;	/* number of blocks in seg */
	__le16	ss_nfinfo;	/* number of file info structures */
	__le16  ss_nfbinfo;     /* number of fbinfo structures */
	__le16	ss_nfblk;       /* number of file blocks in seg */
	__le16	ss_nfbblk;	/* number of file B-tree blocks in seg */
                                /* = number of file block info structures */
	/* __le16 ss_ndinfo;	   number of dir block in seg */
        __le16	ss_niblk;	/* number of inode block in seg */
                                /* = number of inode block info structures */
	__le16	ss_nibblk;	/* number of inode B-tree block in seg */
	__le64	ss_create;	/* creation time stamp */
	__le16	ss_prev;	/* previous partial segment (distance, in blocks) */
#define SS_FJCP 0x1		/* segment has major checkpoint */
#define SS_FNCP 0x2		/* segment has minor checkpoint */
#define SS_FSBLK 0x4		/* segment has super block copy */
#define SS_DDIR  0x8		/* segment has dirty directories */
#define SS_LOGBGN 0x0010        /* begin mark of logical segment */
#define SS_LOGEND 0x0020        /* end mark of logical segment */
#define SS_SYNDT 0x0040         /* segment has synchronized data blocks */
	__le16	ss_flags;	/* segment summary's flags */
	/* FINFO's */
} __attribute__((__packed__));

#define NILFS_SEGSUM_MAGIC      0x3672      /* segment summary magic number */

/*
 * segment summary per file block information, on disk
 */
struct nilfs_finfo {
	__le32	fi_nblocks;	/* number of blocks */
	__le32	fi_version;	/* version number */
	__le64	fi_ino;		/* inode number */
};

/*
 * segment summary per file block B-Tree block information
 */
struct nilfs_fbinfo {
	__le32	fbi_nblocks;	/* number of blocks (same ino) */
        __le64	fbi_ino;	/* */
} __attribute__((__packed__));

/*
 * segment summary per inode block information
 */
struct nilfs_iinfo {
        __le64	ii_ino;		 /* first inode number */
};

struct nilfs_checkpoint {
        __le32	cp_sum;	            /* checksum of checkpoint */

#define NILFS_CP_MAJOR	       0x1  /* major checkpoint (has inode root) */
#define NILFS_CP_HAS_SNAPSHOT  0x2  /* snapshot root */
        __le16	cp_flags;
	__le16	cp_bytes;           /* byte count of this structure 
				       except for reserved fields */
	__le64	cp_inode_root;	    /* current inode root */
	__le64  cp_nblk_lseg;       /* logical segment size in block count */
	__le64	cp_inodes_count;    /* Inodes count */
	__le64	cp_blocks_count;    /* Blocks count */
	__le32  cp_sketch_size;     /* Sketch data size */

	/*
	 * 11 words (in 4-byte word) are consumed above.
	 * Number of s_reserved = 128 - 11 = 117
	 */
	__u32	cp_reserved[117];   /* Padding to the end of the segment */
};

/*
 * Bytes count of checkpoint for CRC-calculation
 */
#define NILFS_CP_BYTES   ((int)(((struct nilfs_checkpoint *)0)->cp_reserved))

/* 
 * segusage flags
 */

#define SEGUSE_GC_MASK	   0x3  /* GC count */
#define SEGUSE_USED	   0x4	/* segment has valid user data */
#define SEGUSE_HAS_HEAD    0x8  /* logical segment start */
#define SEGUSE_HAS_TAIL	   0x10 /* logical segment end */
#define SEGUSE_ACTIVE	   0x20	/* segment is currently being written */
#define SEGUSE_SHORTLEN    0x40 /* short segment block length */
#define SEGUSE_DMASK	   0x7F

/* following flags are on segusage (on memory) only */
#define SEGUSE_DIRTY	   0x100  /* memory segusage is dirty */
#define SEGUSE_GCMARK	   0x200  /* gc marking in progress */
#define SEGUSE_GCMVSRC	   0x400  /* segment data move from this segment */
#define SEGUSE_GCMVDST	   0x800  /* segment data move to this segment */

/* 
 * full segment usage structure
 */
#define NILFS_MAX_SU_NCHUNK 50
#define NILFS_N_SEG_USEAGE (NILFS_MAX_SU_NCHUNK * 2)

#define SU_NOSEG 0xffffffff		/* end mark for prev/next full segment */
struct segusage {
	__le32	su_prev_full_seg;	/* logically previous full segment */
	__le32	su_next_full_seg;	/* logically next full segment */
	__le64	su_lastmod;		/* last modified timestamp */
	__le16	su_aux;			/* currently not uesed */
	__le16	su_flags;		/* see above */
} __attribute__((__packed__));

/* header, one for block */
struct segusage_hdr {
	__le32 suh_sum;			/* checksum header & all usage */
	__le32 suh_aux;			/* currently not uesed */
	__le64 suh_free_blocks_count;	/* copy of superblock */
	__le64 suh_last_segment;	/* copy of superblock */
	__le64 suh_wtime;		/* copy of superblock */
	__le64 suh_tm;			/* segusage write time */
};

#endif /* _LINUX_NILFS_FS_H */
/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
