/*
 * nilfs.h - NILFS local header file
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * nilfs.h,v 1.44 2006/07/14 07:04:56 ryusuke Exp
 */

#include <linux/fs.h>
#include <linux/buffer_head.h>
#include "nilfs_fs.h"
#include "sb.h"
#include "kern_feature.h"

/*
 * nilfs inode data in memory
 */
struct nilfs_inode_info {
	__u32	i_flags;
	__u32	i_state;
	struct nilfs_btree   i_block_root;	/* Data Block B-Tree Root */
	__u32	i_file_acl;
	__u32	i_dir_acl;
	__u32	i_dtime;
	__u32	i_dir_start_lookup;
	struct rw_semaphore  i_block_sem;	/* Data Block B-Tree Giant Lock */
	struct radix_tree_64_root i_block_ntree;/* Block B-Tree Node Block radix tree */
	spinlock_t i_block_ntree_lock;

        struct list_head i_dirty;               /* List for connecting dirty files */

#ifdef CONFIG_NILFS_XATTR
	/*
	 * Extended attributes can be read independently of the main file
	 * data. Taking i_mutex even when reading would cause contention
	 * between readers of EAs and writers of regular file data, so
	 * instead we synchronize on xattr_sem when reading or changing
	 * EAs.
	 */
	struct rw_semaphore xattr_sem;
#endif
#ifdef CONFIG_NILFS_POSIX_ACL
	struct posix_acl	*i_acl;
	struct posix_acl	*i_default_acl;
#endif
	rwlock_t i_meta_lock;
        struct buffer_head *i_bh;	/* i_bh contains a new or dirty
					   disk inode */
	struct semaphore i_node_pages_sem;	/* for i_*_node_pages */
        struct list_head i_partial_node_pages;	/* partial B-Tree node pages */
	struct inode	vfs_inode;
};

static inline struct nilfs_inode_info *
NILFS_I(struct inode *inode)
{
	return container_of(inode, struct nilfs_inode_info, vfs_inode);
}

/*
 * Inode dynamic state flags
 */
#define NILFS_STATE_NEW         0x0001     /* Inode is newly created */
#define NILFS_STATE_DIRTY       0x0002     /* The file is dirty */
#define NILFS_STATE_COLLECTED   0x0004     /* All dirty blocks are collected 
					      by the segment constructor */
#define NILFS_STATE_UPDATED     0x0008     /* The file has been written
					      back */
#define NILFS_STATE_INODE_DIRTY 0x0010     /* write_inode is requested */

/*
 * Macros to check inode numbers
 */
#define NILFS_SYS_INO_BITS   \
  (unsigned int)(1 << NILFS_ROOT_INO | 1 << NILFS_ATIME_INO | 1 << NILFS_SKETCH_INO)

#define NILFS_VALID_INODE(sb, ino) \
  ((ino) >= NILFS_FIRST_INO(sb) || (NILFS_SYS_INO_BITS & (1 << (ino))))

/*
 * free inode list, the head is super block memory structure s_free_inodes
 */
struct nilfs_free_inode_list {
	struct list_head buffers;
	struct buffer_head *free_bh;
};

/*
 * Extended buffer state bits
 */
enum {
	BH_Prepare_Dirty = BH_PrivateStart,
	BH_NILFS_Allocated,
	BH_NILFS_Node,
	BH_NILFS_BBT_Node,
	BH_NILFS_IBT_Node,
	BH_NILFS_New_Node,
	BH_NILFS_Inactive_Node,
	BH_NILFS_Partial_Node,
	BH_NILFS_Freeze,
	BH_NILFS_InodeOnList,
};

BUFFER_FNS(Prepare_Dirty, prepare_dirty)  /* prepare-dirty flag */
TAS_BUFFER_FNS(Prepare_Dirty, prepare_dirty)  /* prepare-dirty flag */
BUFFER_FNS(NILFS_Allocated, nilfs_allocated) /* raw page used by superblock block,
					   segment construction data, etc. */
BUFFER_FNS(NILFS_Node, nilfs_node)  	/* B-Tree node */
BUFFER_FNS(NILFS_BBT_Node, nilfs_bbt_node)	/* B-Tree node for data block */
BUFFER_FNS(NILFS_IBT_Node, nilfs_ibt_node)	/* B-Tree node for inode block */
BUFFER_FNS(NILFS_New_Node, nilfs_new_node) /* newly created B-Tree node */
BUFFER_FNS(NILFS_Inactive_Node, nilfs_inactive_node)
				 	/* page for B-Tree node with
					   no reference */
BUFFER_FNS(NILFS_Partial_Node, nilfs_partial_node)
					/* partial B-Tree node page */
BUFFER_FNS(NILFS_Freeze, nilfs_freeze)   /* needed to freeze during writeback */
BUFFER_FNS(NILFS_InodeOnList, nilfs_inode_on_list) /* inode block on free list */
TAS_BUFFER_FNS(NILFS_InodeOnList, nilfs_inode_on_list) /* inode block on free list */

/*
 * On-memory segment summary
 */
struct nilfs_segsum_info {
        unsigned int            nfinfo;       /* number of file information
					         structures */
	unsigned int            nfbinfo;      /* number of fbinfo structures */
        unsigned int            nblocks;      /* number of blocks in this
					         partial segment */
        unsigned int            nblk_sum;     /* number of summary blocks */
        unsigned int            nblk_file;    /* number of file blocks */
        unsigned int            nblk_fbt;     /* number of file B-tree blocks */
        unsigned int            nblk_inode;   /* number of inode blocks */
        unsigned int            nblk_ibt;     /* number of inode B-tree blocks */
	unsigned int            flags;
	unsigned int            prev_pseg;
	u64                     seg_seq;
	time_t                  ctime;
};

#define NILFS_SEG_HAS_CP(sum)    ((sum)->flags & (SS_FJCP | SS_FNCP))
#define NILFS_SEG_HAS_MJCP(sum)  ((sum)->flags & SS_FJCP)
#define NILFS_SEG_LOGBGN(sum)    ((sum)->flags & SS_LOGBGN)
#define NILFS_SEG_LOGEND(sum)    ((sum)->flags & SS_LOGEND)
#define NILFS_SEG_SIMPLEX(sum)   (((sum)->flags & (SS_LOGBGN | SS_LOGEND)) == (SS_LOGBGN | SS_LOGEND))
#define NILFS_SEG_DSYNC(sum)     ((sum)->flags & SS_SYNDT)

/*
 * On-memory checkpoint information
 */
struct nilfs_cp_info {
	unsigned int            flags;        /* checkpoint flags */
	dbn_t                   inode_root;
	unsigned long           inodes_count;
	unsigned long           blocks_count;
};

/*
 * Transaction info
 */
struct nilfs_transaction_info {
	struct super_block      *ti_super;
	void                    *ti_save;     /*
					       * Backup of journal_info field of
					       * task_struct. This should never
					       * used. If this will happen, some
					       * other filesystem has a bug.
					       */
#define NILFS_TI_DYNAMIC_ALLOC   0x0001
#define NILFS_TI_SYNC            0x0002       /* 
					       * Force to construct segment
					       * at the end of a transaction.
					       * (only for toplevel)
					       */
	unsigned short           ti_flags;
	unsigned short           ti_count;
};

/*
 * Segment constructor information
 */
struct nilfs_sc_info {
	struct super_block *sc_sb;      /* back pointer to super_block struct */

	dbn_t sc_pseg_start;            /* disk block number of partial segment */
	unsigned long sc_nblk_lseg;     /* block count of current logical segment */
	unsigned long sc_residual_blocks; /* residual blocks */

	short sc_stage, sc_prev_stage;  /* collection stage */
	struct nilfs_inode_info *sc_dirty_file_ptr;
	                                /* pointer on dirty_files list, or
					   inode of a target file for data only segment */
	struct buffer_head **sc_bh_arr; /* buffer head array */
	__u32 sc_ba_max;                /* size of bh_arr */

	/* Checkpoint buffer */
	struct buffer_head *sc_cp_bh;

	unsigned int sc_ba_idx;         /* current index on buffer head array */
	unsigned int sc_ba_sum_idx;     /* index for summary blocks */
	unsigned int sc_ba_cp_idx;      /* index for checkpoint blocks */

	unsigned long sc_blk_cnt;       /* data or node block count of current file */
	unsigned long sc_info_offset;   /* offset bytes in summary current file */
	unsigned long sc_sum_bytes;     /* byte count for the segment summary */

	unsigned long sc_flags;         /* Internal flags defined below */

	/*
	 * Pointer to an inode of the sketch.
	 * This pointer is kept only while it contains data.
	 * We protect it with a semaphore of the segment constructor.
	 */
	struct inode *sc_sketch_inode;

	struct nilfs_segsum_info sc_sum;/* configuration of current segment */
	struct nilfs_cp_info sc_cp;     /* checkpoint information */

	int sc_nbio;                    /* number of submitting bio */
	struct completion sc_bio_event;	/* completion event of segment write */
	atomic_t sc_max_bio;            /* maximum number of flying bios */
	atomic_t sc_bio_blk_cnt;	/* block count for bio completion routine */

	spinlock_t sc_state_lock;

#define NILFS_SEGCTOR_QUIT    0x0001 /* segctord is being destroyed */
#define NILFS_SEGCTOR_INIT    0x0002 /* segctord is being started */
#define NILFS_SEGCTOR_COMMIT  0x0004 /* at least one transaction has been committed */
#define NILFS_SEGCTOR_FLUSH_DATA  0x0010
#define NILFS_SEGCTOR_FLUSH_IBT   0x0020
#define NILFS_SEGCTOR_FLUSH  \
          (NILFS_SEGCTOR_FLUSH_DATA | NILFS_SEGCTOR_FLUSH_IBT)

	unsigned long sc_state;	        /* segctord state flags */

	int sc_errno;                   /* return value of segment constructor thread */

	wait_queue_head_t sc_wait_done; /* daemon wait queue */
	wait_queue_head_t sc_wait_request; /* client wait queue */

	__u32 sc_seq_request;           /* request counter to thread */
	__u32 sc_seq_done;              /* completion counter of thread */
	__u32 sc_seq_ack;               /* acknowledge counter to thread */

	int sc_max_buffers;             /* maximum number of segment buffers */
	int sc_sync;                    /* Request of explicit sync operation */
	unsigned long sc_interval;      /* Timeout value of segment construction */
	unsigned long sc_mjcp_freq;     /* Frequency of creating major checkpoints. */
	unsigned long sc_lseg_stime;    /* The time when the latest logical segment
					   started to construct (in 1/HZ seconds) */
	time_t sc_ctime;                /* The last construction time or timestamp
					   used for passive mode */
	struct timer_list *sc_timer;

	unsigned long sc_nr_dirty;	/* number of dirty buffers (in block) */
	unsigned long sc_block_max;	/* Threshold of dirty buffers */

	struct task_struct *sc_task;	/* current constructor thread */
};

/* sc_flags */
enum {
	NILFS_SC_DIRTY,            /* One or more dirty meta-data blocks exist */
	NILFS_SC_EIO,              /* BIO reports I/O error(s) */
	NILFS_SC_LAST_CP,          /* Last checkpoint was changed */
	NILFS_SC_UNCLOSED,         /* Logical segment is not closed */
	NILFS_SC_PASSIVE,          /* Passive mode */
	NILFS_SC_RETRY,            /* I/O error retry */
};

/* Flush mode */
enum {
	NILFS_SC_FLUSH_DATA = 1,   /* Flush current dirty data blocks and make
				      partial segments without CP and inode b-tree */
	NILFS_SC_FLUSH_IBT,        /* Flush current dirty data blocks and inode-btree;
				      make partial segments without CP */
	NILFS_SC_FLUSH_CP,         /* Flush dirty blocks and make a checkpoint */
	NILFS_SC_FLUSH_FDATA,      /* Flush data blocks of a given file and make
				      a logical segment without CP */
};

static inline struct nilfs_sc_info *NILFS_SC(struct nilfs_sb_info *sbi)
{
	return sbi->s_segctor.info;
}

/*
 * Default values of timeout, in seconds.
 */
#define NILFS_SC_DEFAULT_TIMEOUT      5   /* Timeout value of buffered dirty blocks.
					     This timeout triggers construction of a
					     logical segment having a major CP. */
#define NILFS_SC_DEFAULT_MJCP_FREQ    30  /* Maximum frequency of major CP creation */
#define NILFS_SC_DEFAULT_SUPER_FREQ   30  /* Minimum interval of periodical update of
					     superblock (reserved) */

/*
 * The default threshold amount of data, in block counts.
 */
#define NILFS_SC_DEFAULT_THRESHOLD_DATA_AMOUNT    1800

/*
 * The default maximum number of flying bios.
 */
#define NILFS_SC_DEFAULT_MAX_BIO      4

/*
 * Segment check result
 */
enum {
	NILFS_SEG_VALID,
	NILFS_SEG_HAS_NO_CP,
	NILFS_SEG_HAS_NO_MAJOR_CP,
	NILFS_SEG_FAIL_IO,
	NILFS_SEG_FAIL_MAGIC,
	NILFS_SEG_FAIL_SEQ,
	NILFS_SEG_FAIL_CHECKSUM_SEGSUM,
	NILFS_SEG_FAIL_CHECKSUM_CP,
	NILFS_SEG_FAIL_CHECKSUM_FULL,
	NILFS_SEG_FAIL_CONSISTENCY,
	NILFS_SEG_BAD_SNAPSHOT,
};

/*
 * Recovery status
 */
#define RECOVERY_CP_UPDATED         1  /* The latest checkpoint was updated */
#define RECOVERY_ROLLFORWARD_DONE   2  /* Rollforward was carried out */

/*
 * tags for the 64-bit page cache
 *   change RADIX_TREE_64_TAGS in radix-tree-64.c if you add more tags.
 */
#define PAGECACHE64_TAG_DIRTY		0
#define PAGECACHE64_TAG_PREPARE_DIRTY	1

/*
 * Return codes of the segment constructor
 */
#define NILFS_SEG_EMPTY                 1  /* skipped (= NILFS_SYNC_SKIPPED) */
#define NILFS_SEG_IGNORED               2  /* ignored */
#define NILFS_SEG_ERR(c)                (unlikely((c) < 0))  /* Error codes */

/*
 * nilfs kobject attribute
 */
struct nilfs_attribute {
	struct attribute attr;
	ssize_t (*show)(struct the_nilfs *, char *);
	ssize_t (*store)(struct the_nilfs *, const char *, size_t);
};

#define define_nilfs_rw_attr(_name, _mode) \
struct nilfs_attribute nilfs_attr_##_name = \
  __ATTR(_name, _mode, nilfs_show_##_name, nilfs_store_##_name);

#define define_nilfs_ro_attr(_name, _mode) \
struct nilfs_attribute nilfs_attr_##_name = \
  __ATTR(_name, _mode, nilfs_show_##_name, NULL);

/*
 * debug primitives
 */
#include "debug.h"

/*
 * function prototype
 */
#ifdef CONFIG_NILFS_POSIX_ACL
#error "NILFS: not yet supported POSIX ACL"
extern int nilfs_permission(struct inode *, int, struct nameidata *);
extern int nilfs_acl_chmod(struct inode *);
extern int nilfs_init_acl(struct inode *, struct inode *);

/* Value for i_acl and i_default_acl if the ACL has not been cached */
#define NILFS_ACL_NOT_CACHED  ((void *)-1)
#else
#include <linux/sched.h>
#define nilfs_permission   NULL

static inline int nilfs_acl_chmod(struct inode *inode)
{
	return 0;
}

static inline int nilfs_init_acl(struct inode *inode, struct inode *dir)
{
	inode->i_mode &= ~current->fs->umask;
	return 0;
}
#endif

/*
 * Error code conversion for B-tree routines
 */
static inline int nilfs_convert_btree_error(int btree_error)
{
	BUG_ON(btree_error >= 0);
	
	if (btree_error == -NILFS_BTREE_ENOMEM)
		return -ENOMEM;
	if (btree_error == -NILFS_BTREE_EIO)
		return -EIO;
	return -EINVAL;
}

/* dir.c */
extern int nilfs_add_link (struct dentry *, struct inode *);
extern ino_t nilfs_inode_by_name(struct inode *, struct dentry *);
extern int nilfs_make_empty(struct inode *, struct inode *);
extern struct nilfs_dir_entry * nilfs_find_entry (struct inode *,struct dentry *, struct page **);
extern int nilfs_delete_entry (struct nilfs_dir_entry *, struct page *);
extern int nilfs_empty_dir (struct inode *);
extern struct nilfs_dir_entry * nilfs_dotdot (struct inode *, struct page **);
extern void nilfs_set_link(struct inode *, struct nilfs_dir_entry *, struct page *, struct inode *);

/* ialloc.c */
extern int nilfs_handle_ibt_error(int, const char *, struct super_block *);
extern int nilfs_few_free_inodes(int, struct nilfs_sb_info *);
extern struct inode * nilfs_new_inode (struct inode *, int);
extern void nilfs_free_inode (struct inode *);
extern unsigned long nilfs_count_free_inodes (struct super_block *);
extern unsigned long nilfs_count_free (struct buffer_head *, unsigned);

extern void nilfs_init_iblock(struct buffer_head *, ino_t, struct nilfs_sb_info *);

/* file.c */
#define nilfs_ioctl          NULL   /* dummy code */
#define nilfs_release_file   NULL
extern int nilfs_sync_file(struct file *, struct dentry *, int);

/* inode.c */
extern int nilfs_handle_fbt_error(int, const char *, struct inode *);
extern int nilfs_get_block(struct inode *, sector_t, struct buffer_head *, int);
extern struct nilfs_inode *nilfs_get_inode(struct super_block *, ino_t,
				     struct buffer_head **);
extern void nilfs_set_inode_flags(struct inode *);
extern void nilfs_read_inode(struct inode *);
extern void nilfs_update_inode(struct inode *, struct nilfs_inode *,
			    struct buffer_head *);
extern void nilfs_truncate(struct inode *);
extern void nilfs_delete_inode(struct inode *);
extern int nilfs_setattr(struct dentry *, struct iattr *);
extern struct nilfs_inode *__nilfs_load_inode_block_nolock(struct nilfs_sb_info *, struct inode *, struct buffer_head **);
extern void nilfs_put_inode(struct inode *);

/* debug.c */
#ifdef CONFIG_NILFS_DEBUG
extern int nilfs_init_proc_entries(void);
extern void nilfs_remove_proc_entries(void);
extern void nilfs_init_counters(void);
extern void nilfs_fill_debug_info(int);
#define nilfs_init_debug_info()  do { nilfs_fill_debug_info(1); } while(0)
extern void nilfs_print_seginfo(struct nilfs_sc_info *, dbn_t, dbn_t);
extern void nilfs_check_radix_tree(const char *, struct address_space *, int);
extern void nilfs_check_radix_tree_64(const char *, struct radix_tree_64_root *, spinlock_t *, int);
extern int nilfs_release_inode_page(struct page *, gfp_t);
extern int nilfs_releasepage(struct page *, gfp_t);
#else
#define nilfs_init_proc_entries()  (0)
#define nilfs_remove_proc_entries()  do {} while(0)
#define nilfs_init_counters()  do {} while(0)
#define nilfs_init_debug_info()  do {} while(0)
#define nilfs_print_seginfo(sci, start, end)  do {} while(0)
#define nilfs_check_radix_tree(fname, mapping, blocksize_bits)  do {} while(0)
#define nilfs_check_radix_tree_64(fname, tree, tree_lock, blocksize_bits)  do {} while(0)
#define nilfs_release_inode_page  NULL
#define nilfs_releasepage   NULL
#endif /* CONFIG_NILFS_DEBUG*/

/* gc.c */
extern int nilfs_alloc_segment(struct nilfs_sb_info *, segnum_t *, int);
extern int nilfs_find_prev_segment(struct nilfs_sb_info *, segnum_t, segnum_t *);
extern int nilfs_find_next_segment(struct nilfs_sb_info *, segnum_t, segnum_t *);
extern unsigned long nilfs_count_free_blocks(struct nilfs_sb_info *);
extern dbn_t nilfs_sb_pcopy_blk(struct nilfs_sb_info *, int, int);

/* page.c */
extern struct buffer_head *nilfs_bread_slow(struct buffer_head *);
extern struct buffer_head *nilfs_getblkbh(struct block_device *, sector_t, int);
extern void nilfs_putblkbh(struct buffer_head *);
extern struct buffer_head *nilfs_copy_buffers(struct buffer_head *, int);
extern int __nilfs_brelse(struct super_block *, struct buffer_head *);
extern int nilfs_page_buffers_dirty(struct page *);
extern void nilfs_free_free_inode_list(struct super_block *);
extern void nilfs_clean_free_inode_list(struct super_block *);

extern struct buffer_head *nilfs_find_get_inode_block(struct super_block *, ino_t);
extern struct buffer_head *nilfs_create_inode_block(struct super_block *, ino_t);
extern struct buffer_head *nilfs_inode_bread(struct super_block *, ino_t);
extern void nilfs_free_inode_buffers(struct super_block *);
extern int nilfs_release_allocated_page(struct page *, gfp_t);
extern int nilfs_invalidate_allocated_page(struct page *, unsigned long);

/* btnode.c */
extern void nilfs_clear_inactive_node_page(struct page *);
extern int nilfs_release_node_page(struct page *, gfp_t);
extern int nilfs_invalidate_node_page(struct page *, unsigned long);
extern void nilfs_put_allocated_page(struct page *, struct super_block *);
extern int nilfs_shrink_inactive_node_pages(struct super_block *, int);
extern int nilfs_clean_inactive_node_pages(struct super_block *);
extern struct buffer_head *nilfs_get_file_node_blk(struct inode *, dbn_t);
extern struct buffer_head *nilfs_get_new_file_node_blk(struct inode *);
extern void nilfs_put_file_node_blk(struct buffer_head *);
extern void nilfs_delete_file_node_blk(struct buffer_head *);
extern void nilfs_delete_all_file_node_blk(struct inode *);
extern void nilfs_clean_all_file_node_blk(struct inode *);
extern void nilfs_mark_file_node_blk_dirty(struct buffer_head *);
extern void nilfs_mark_file_node_blk_prepare_dirty(struct buffer_head *);
extern struct buffer_head *nilfs_get_inode_node_blk(struct super_block *, dbn_t);
extern struct buffer_head *nilfs_get_new_inode_node_blk(struct super_block *);
extern void nilfs_put_inode_node_blk(struct buffer_head *);
extern void nilfs_delete_inode_node_blk(struct buffer_head *);
extern void nilfs_delete_all_inode_node_blk(struct super_block *);
extern void nilfs_clean_all_inode_node_blk(struct super_block *);
extern void nilfs_mark_inode_node_blk_dirty(struct buffer_head *);
extern void nilfs_mark_inode_node_blk_prepare_dirty(struct buffer_head *);
extern void nilfs_clear_node_page_dirty(struct page *);
extern int nilfs_move_dirty_node_blk(struct buffer_head **, dbn_t, unsigned int);

/* recovery.c */
extern int nilfs_warn_segment_error(struct nilfs_sb_info *, int);
extern int nilfs_load_last_segment(struct nilfs_sb_info *, struct the_nilfs *);
extern int nilfs_load_last_segment_for_remount(struct nilfs_sb_info *, struct the_nilfs *);
extern int nilfs_load_snapshot(struct nilfs_sb_info *);

/* segment.c */
extern int nilfs_init_segment(struct nilfs_sb_info *, unsigned int);
extern void nilfs_clear_segment(struct nilfs_sb_info *);

extern int nilfs_init_transaction_cache(void);
extern void nilfs_destroy_transaction_cache(void);
extern int nilfs_set_file_dirty(struct nilfs_sb_info *, struct inode *);
extern int nilfs_commit_dirty_file(struct inode *);
extern void nilfs_dirty_inode(struct inode *);
extern int nilfs_construct_segment(struct super_block *);
extern int nilfs_construct_fdata_segment(struct super_block *, struct inode *);
extern int nilfs_transaction_begin(struct super_block *, struct nilfs_transaction_info *);
extern int nilfs_transaction_end(struct super_block *);
extern void nilfs_flush_segment(struct nilfs_sb_info *, int);

extern void nilfs_segctor_add_dirty(struct nilfs_sc_info *, unsigned);
extern int nilfs_segctor_reset(struct nilfs_sc_info *);

/* sketch.c */
extern int nilfs_read_sketch_inode(struct nilfs_inode_info *);

/* super.c */
extern void nilfs_error (struct super_block *, const char *, const char *, ...)
	__attribute__ ((format (printf, 3, 4)));
extern void nilfs_warning(struct super_block *, const char *, const char *, ...)
       __attribute__ ((format (printf, 3, 4)));			      
extern struct nilfs_super_block *nilfs_load_super_block(struct super_block *, struct buffer_head **);
extern struct nilfs_super_block *nilfs_reload_super_block(struct super_block *, struct buffer_head **, int);
extern int nilfs_store_magic_and_option(struct super_block *, struct nilfs_super_block *, char *);
extern int nilfs_store_disk_layout(struct super_block *, struct nilfs_super_block *);
extern void nilfs_update_last_segment(struct nilfs_sb_info *);

/* the_nilfs.c */
extern struct the_nilfs *alloc_nilfs(struct block_device *);
extern void put_nilfs(struct the_nilfs *);
extern int init_nilfs(struct the_nilfs *, struct nilfs_sb_info *, struct super_block *, char *);

/* inode.c */
static inline struct nilfs_inode *
nilfs_raw_inode(struct nilfs_inode_info *ii)
{
	struct nilfs_inode_hdr *ih;

	if (unlikely(ii->i_bh == NULL)) {
		nilfs_debug(0, "tried to refer invalid i_bh (ii=%p, ino=%lu)\n", 
			    ii, ii->vfs_inode.i_ino);
		BUG();
	}
	ih = (struct nilfs_inode_hdr *)ii->i_bh->b_data;
	return ((struct nilfs_inode *)(ih + 1) + ii->vfs_inode.i_ino -
		ih->ih_ino);
}

static inline struct nilfs_inode *
nilfs_load_inode_block(struct nilfs_sb_info *sbi, struct inode *inode,
		       struct buffer_head **pbh)
{
	struct nilfs_inode *raw_inode;

	spin_lock_irq(&sbi->s_segctor.dirty_files_lock);
	raw_inode = __nilfs_load_inode_block_nolock(sbi, inode, pbh);
	spin_unlock_irq(&sbi->s_segctor.dirty_files_lock);
	return raw_inode;
}

/* the_nilfs.c */
static inline void get_nilfs(struct the_nilfs *nilfs)
{
	/* Caller must have at least one reference of the_nilfs. */
	atomic_inc(&nilfs->ns_count);
}

static inline struct nilfs_sb_info *nilfs_get_writer(struct the_nilfs *nilfs)
{
	down_read(&nilfs->ns_writer_sem);
	return nilfs->ns_writer;
}

static inline void nilfs_put_writer(struct the_nilfs *nilfs)
{
	up_read(&nilfs->ns_writer_sem);
}

static inline void
nilfs_attach_writer(struct the_nilfs *nilfs, struct nilfs_sb_info *sbi)
{
	down_write(&nilfs->ns_writer_sem);
	nilfs->ns_writer = sbi;
	up_write(&nilfs->ns_writer_sem);
}

static inline void
nilfs_detach_writer(struct the_nilfs *nilfs, struct nilfs_sb_info *sbi)
{
	down_write(&nilfs->ns_writer_sem);
	if (sbi == nilfs->ns_writer)
		nilfs->ns_writer = NULL;
	up_write(&nilfs->ns_writer_sem);
}

static inline void
set_nilfs_cp(struct the_nilfs *nilfs, dbn_t inode_root,
	     u64 last_seq, dbn_t last_pseg, dbn_t last_cp, time_t last_ctime,
	     unsigned long inodes_count, unsigned long blocks_count)
{
	unsigned long flags;

	spin_lock_irqsave(&nilfs->ns_lock, flags);
	nilfs->ns_inode_root = inode_root;
	nilfs->ns_last_seq = last_seq;
	nilfs->ns_last_pseg = last_pseg;
	nilfs->ns_last_cp = last_cp;
	nilfs->ns_last_ctime = last_ctime;
	nilfs->ns_inodes_count = inodes_count;
	nilfs->ns_blocks_count = blocks_count;
	spin_unlock_irqrestore(&nilfs->ns_lock, flags);
}

/* page.c */
static inline struct buffer_head *
nilfs_getblk(struct super_block *sb, sector_t block)
{
	struct buffer_head *bh = nilfs_getblkbh(sb->s_bdev, block,
						sb->s_blocksize);
	if (likely(bh)) {
		set_buffer_nilfs_allocated(bh);
		/*
		 * By nature, 'nilfs_allocated' SHOULD be set for all 
		 * nilfs-allocated buffers including node buffers,
		 * whereas current node buffers never have it.
		 */
		get_bh(bh);
	}
	return bh;
}

static inline struct buffer_head *
nilfs_bread(struct super_block *sb, sector_t block)
{
	struct buffer_head *bh = nilfs_getblk(sb, block);

	/*
	 * Because nilfs_getblk() just allocates a new private 
	 * buffer, nilfs_bread() always reads data from disk.
	 * This function should be rewritten if buffer cache
	 * is brought in.
	 */
	if (likely(bh))
		bh = nilfs_bread_slow(bh);
		/* We don't have to release buffer_head (bh) here.
		 * Read comments on nilfs_bread_slow() */
	return bh;
}

static inline void
nilfs_brelse(struct buffer_head *bh)
{
	if (!bh || __nilfs_brelse(NULL, bh))
		return;
#ifdef NILFS_BH_DEBUG
        BH_DEBUG((bh), "bfr _rs");
#endif
	__brelse(bh);
}

static inline void
nilfs_brelse_isr(struct super_block *sb, struct buffer_head *bh)
{
	if (!bh || __nilfs_brelse(sb, bh))
		return;
#ifdef NILFS_BH_DEBUG
        BH_DEBUG((bh), "bfr _rs");
#endif
	__brelse(bh);
}

static inline struct buffer_head *
nilfs_page_get_nth_block(struct page *page, unsigned int count)
{
	struct buffer_head *bh, *head;
	
	bh = head = page_buffers(page);
	do {
		if (count-- == 0) {
			get_bh(bh);
			return bh;
		}
		bh = bh->b_this_page;
	} while (bh != head);
	return NULL;
}

static inline int nilfs_node_page_new(struct page *page)
{
	if (unlikely(!PagePrivate(page)))
		return 0;	/* ERROR */
	return buffer_nilfs_new_node(page_buffers(page));
}

static inline u64 nilfs_node_page_index(struct page *page)
{
	if (unlikely(nilfs_node_page_new(page)))
		return NILFS_BTREE_BH_TO_PTR(page_buffers(page));
	return page_index(page);
}

static inline int nilfs_page_for_node(struct page *page)
{
	if (unlikely(!PagePrivate(page)))
		return 0;	/* ERROR */
	return buffer_nilfs_node(page_buffers(page));
}

static inline int nilfs_page_for_ibt_node(struct page *page)
{
	if (unlikely(!PagePrivate(page)))
		return 0;	/* ERROR */
	return buffer_nilfs_ibt_node(page_buffers(page));
}

static inline int nilfs_page_for_bbt_node(struct page *page)
{
	if (unlikely(!PagePrivate(page)))
		return 0;	/* ERROR */
	return buffer_nilfs_bbt_node(page_buffers(page));
}

static inline int
nilfs_inactive_node_page(struct page *page)
{
	return buffer_nilfs_inactive_node(page_buffers(page));
}

static inline int nilfs_page_to_be_frozen(struct page *page)
{
	if (unlikely(!PagePrivate(page)))
		return 0;
	return buffer_nilfs_freeze(page_buffers(page));
}

static inline void nilfs_set_page_to_be_frozen(struct page *page)
{
	BUG_ON(!PagePrivate(page));
	set_buffer_nilfs_freeze(page_buffers(page));
}

static inline void nilfs_clear_page_to_be_frozen(struct page *page)
{
	BUG_ON(!PagePrivate(page));
	clear_buffer_nilfs_freeze(page_buffers(page));
}

static inline void nilfs_set_page_writeback(struct page *page)
{
	BUG_ON(!PagePrivate(page));

	if (buffer_nilfs_node(page_buffers(page)) ||
	    buffer_nilfs_allocated(page_buffers(page)))
		SetPageWriteback(page);
	else
		set_page_writeback(page);
}

static inline void nilfs_end_page_writeback(struct page *page)
{
	BUG_ON(!PagePrivate(page));

	if (buffer_nilfs_node(page_buffers(page)) ||
	    buffer_nilfs_allocated(page_buffers(page)))
		ClearPageWriteback(page);
	else
		end_page_writeback(page);
}

static inline void nilfs_clear_page_dirty(struct page *page)
{
	/*
	 * Page index must be fixed before calling this function.
	 */
	if (nilfs_page_for_node(page))
		nilfs_clear_node_page_dirty(page);
	else
		clear_page_dirty(page);
}

static inline int
nilfs_move_dirty_file_node_blk(struct inode *inode, struct buffer_head **pbh,
			       dbn_t dbn)
{
	return nilfs_move_dirty_node_blk(pbh, dbn, inode->i_blkbits);
}

static inline int
nilfs_move_dirty_inode_node_blk(struct super_block *sb, struct buffer_head **pbh,
				dbn_t dbn)
{
	return nilfs_move_dirty_node_blk(pbh, dbn, sb->s_blocksize_bits);
}

/* sketch.c */
static inline unsigned long
nilfs_sketch_size(struct nilfs_sc_info *sci)
{
	/* This function must be used in the section locking the segment
	   semaphore. */
	return (sci->sc_sketch_inode ? sci->sc_sketch_inode->i_size : 0);
}

/* segment.c */
static inline int nilfs_file_dirty(struct inode *inode)
{
	struct nilfs_inode_info *ii = NILFS_I(inode);
	struct nilfs_sb_info *sbi = NILFS_SB(inode->i_sb);
	int ret = 0;

	if (!list_empty(&ii->i_dirty)) {
		spin_lock_irq(&sbi->s_segctor.dirty_files_lock);
		if (ii->i_state & (NILFS_STATE_DIRTY | NILFS_STATE_COLLECTED))
			ret = 1;
		spin_unlock_irq(&sbi->s_segctor.dirty_files_lock);
	}
	return ret;
}

static inline void
nilfs_mark_inode_buffer_dirty(struct nilfs_sb_info *sbi, struct buffer_head *ibh)
{
	/*
	 * Caller of this function must be locking the segment semaphore.
	 */
	mark_buffer_dirty(ibh);
	set_bit(NILFS_SB_IBLK_DIRTY, &sbi->s_flags);
}

static inline int nilfs_prepare_file_dirty(struct inode *inode)
{
	if (is_bad_inode(inode)) {
		inode_debug(1, "tried to register bad_inode. ignored.\n");
		nilfs_dump_stack(NILFS_VERBOSE_INODE, 2);
		return -EIO;
	}
	return nilfs_transaction_begin(inode->i_sb, NULL);
}

static inline void nilfs_cancel_file_dirty(struct inode *inode)
{
	nilfs_transaction_end(inode->i_sb);
}

static inline unsigned long
nilfs_recalc_segsum_size(struct nilfs_segsum_info *ssi, unsigned int blocksize)
{
	unsigned long sum_bytes;

	sum_bytes = sizeof(struct nilfs_seg_summary) +
		sizeof(struct nilfs_finfo) * ssi->nfinfo +
		sizeof(__le64) * ssi->nblk_file +
		sizeof(struct nilfs_fbinfo) * ssi->nfbinfo +
		sizeof(struct nilfs_iinfo) * ssi->nblk_inode;
	ssi->nblk_sum = (sum_bytes - 1) / blocksize + 1;
	return sum_bytes;
}

static inline unsigned long
nilfs_seg_blocks(const struct nilfs_segsum_info *ssi, unsigned int nblk_cp)
{
	return ssi->nblk_sum + ssi->nblk_file + ssi->nblk_fbt +
		ssi->nblk_inode + ssi->nblk_ibt + nblk_cp;
}

/*
 * The segment lock given by following functions makes directory
 * operations atomic; it prevents a (logical) segment from being
 * broken up.
 *
 * The segment lock for foreground processes is managed by 
 * nilfs_transaction_begin()/nilfs_transaction_end().
 * The following functions are remained for the background processes. 
 * Note that the segment lock given below is not re-entrant.
 */
static inline void nilfs_lock_segment(struct super_block *sb)
{
	might_sleep();
	down_read(&NILFS_SB(sb)->s_segctor.sem);
}

static inline int nilfs_trylock_segment(struct super_block *sb)
{
	return down_read_trylock(&NILFS_SB(sb)->s_segctor.sem);
}

static inline void nilfs_unlock_segment(struct super_block *sb)
{
	up_read(&NILFS_SB(sb)->s_segctor.sem);
}

static inline void nilfs_set_transaction_flag(unsigned int flag)
{
	struct nilfs_transaction_info *ti = current->journal_info;

	BUG_ON(!ti);
	ti->ti_flags |= flag;
}

/*
 * nilfs_get_segment_range - return start DBN and end DBN of the full segment
 *    specified by a segment number
 */
extern void
nilfs_get_segment_range(struct nilfs_sb_info *, segnum_t, dbn_t *, dbn_t *);

/*
 * nilfs_invalidate_segment - mark the end of current full segment which means
 *       that new partial segments cannot be allocated on the segment.
 */
static inline void
nilfs_invalidate_segment(struct the_nilfs *nilfs, dbn_t seg_start, dbn_t seg_end)
{
	unsigned long seg_len = seg_end - seg_start + 1;

	/* nilfs->ns_prev_pseg += seg_len - nilfs->ns_pseg_offset; */
	nilfs->ns_pseg_offset = seg_len;
}

/* super.c */
static inline loff_t nilfs_fbn_max_bits(void)
{
	return sizeof(fbn_t) * 8 /* CHAR_BIT */;
}

static inline void nilfs_mark_buffer_dirty(struct buffer_head *bh)
{
	set_buffer_dirty(bh);
}

static inline int nilfs_sync_dirty_buffer(struct buffer_head *bh)
{
	return sync_dirty_buffer(bh);
}

/*
 * Inodes and files operations
 */

/* dir.c */
extern struct file_operations nilfs_dir_operations;

/* file.c */
extern struct inode_operations nilfs_file_inode_operations;
extern struct file_operations nilfs_file_operations;

/* inode.c */
extern struct address_space_operations nilfs_aops;

/* namei.c */
extern struct inode_operations nilfs_dir_inode_operations;
extern struct inode_operations nilfs_special_inode_operations;

/* symlink.c */
extern struct inode_operations nilfs_symlink_inode_operations;


/*
 * proc entry
 */
extern struct proc_dir_entry *nilfs_proc_root;

/*
 * nilfs kobject attributes
 */
extern struct nilfs_attribute nilfs_attr_sc_interval;
extern struct nilfs_attribute nilfs_attr_sc_threshold;
extern struct nilfs_attribute nilfs_attr_sc_max_bio;
extern struct nilfs_attribute nilfs_attr_sc_mjcp_freq;
extern struct nilfs_attribute nilfs_attr_sc_ctime;
extern struct nilfs_attribute nilfs_attr_max_sketch_size;
extern struct nilfs_attribute nilfs_attr_sync;

/*
 * filesystem type
 */
extern struct file_system_type nilfs_fs_type;


/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
