/*
 * file.c - NILFS regular file handling primitives including fsync().
 *
 * Copyright (C) 2005, 2006 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * file.c,v 1.28 2006/06/12 01:01:07 ryusuke Exp
 *
 * Written by Amagai Yoshiji <amagai@osrg.net>,
 *            Ryusuke Konishi <ryusuke@osrg.net>
 */

#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/writeback.h>
#include "nilfs.h"

int nilfs_sync_file(struct file *file, struct dentry *dentry, int datasync)
{
	/* 
	 * Called from fsync() system call
	 * This is the only entry point that can catch write and synch
	 * timing for both data blocks and intermediate blocks.
	 * 
	 * This function should be implemented when the writeback function
	 * will be implemented.
	 */
	struct inode *inode = dentry->d_inode;
	int ret;

	if (!nilfs_file_dirty(inode)) {
		inode_debug(2, "called for non-dirty files (ino=%lu)\n",
			    inode->i_ino);
		return 0;
	}
	inode_debug(2, "calling nilfs_construct_segment (ino=%lu, datasync=%d)\n",
		    inode->i_ino, datasync);
	if (datasync)
		ret = nilfs_construct_fdata_segment(inode->i_sb, inode);
	else
		ret = nilfs_construct_segment(inode->i_sb);

	if (NILFS_SEG_ERR(ret))
		return ret;
	return 0;
}

static ssize_t
nilfs_file_aio_write(struct kiocb *iocb, const char __user *buf, size_t count, loff_t pos)
{
	struct file *file = iocb->ki_filp;
	struct inode *inode = file->f_dentry->d_inode;
	ssize_t ret;

	ret = generic_file_aio_write(iocb, buf, count, pos);
	if (ret <= 0)
		return ret;

	if ((file->f_flags & O_SYNC) || IS_SYNC(inode)) {
		int err;

		inode_debug(2, "calling nilfs_construct_fdata_segment (ino=%lu)\n",
			    inode->i_ino);
		err = nilfs_construct_fdata_segment(inode->i_sb, inode);
		if (NILFS_SEG_ERR(err))
			return err;
	}
	return ret;
}

/*
 * We have mostly NULL's here: the current defaults are ok for
 * the nilfs filesystem.
 */
struct file_operations nilfs_file_operations = {
	.llseek		= generic_file_llseek,
	.read		= do_sync_read,
	.write		= do_sync_write,
	.aio_read	= generic_file_aio_read,
	.aio_write	= nilfs_file_aio_write,
	.ioctl		= nilfs_ioctl,
	.mmap		= generic_file_mmap,
	.open		= generic_file_open,
	.release	= nilfs_release_file,
	.fsync		= nilfs_sync_file,
	.readv		= generic_file_readv,
	.writev		= generic_file_writev,
	.sendfile	= generic_file_sendfile,
};

struct inode_operations nilfs_file_inode_operations = {
	.truncate	= nilfs_truncate,
	.setattr	= nilfs_setattr,
	.permission     = nilfs_permission,
};

/* end of file */
