/*
 * debug.h - NILFS debug primitives
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * debug.h,v 1.6 2006/05/17 01:22:29 ryusuke Exp
 *
 * Written by Amagai Yoshiji <amagai@osrg.net>,
 *            Ryusuke Konishi <ryusuke@osrg.net>
 */

#ifndef _NILFS_DEBUG_H
#define _NILFS_DEBUG_H

#include <linux/bio.h>
#include <asm/kmap_types.h>
#include <linux/highmem.h>


/*
 * Macros to printks
 */
#ifdef CONFIG_NILFS_DEBUG
#define nilfs_debug(l, f, a...)  \
	do {  \
		if ((l) <= (int)nilfs_debug_info.verbose[0])  \
			printk(KERN_DEBUG "NILFS %s: " f,  \
			       __FUNCTION__, ## a);  \
        } while(0)

#define nilfs_debug_verbose(v, l, c, f, a...)  \
	do {  \
                if ((l) <= (int)nilfs_debug_info.verbose[v])  \
			printk(KERN_DEBUG "NILFS(" c ") %s: " f, \
			       __FUNCTION__, ## a);              \
        } while(0)

#define nilfs_dump_stack(v, l)  \
        do {  \
                if ((l) <= (int)nilfs_debug_info.verbose[v])  \
			dump_stack();   \
	} while(0)

#define page_debug(l, f, a...)  \
        nilfs_debug_verbose(NILFS_VERBOSE_PAGE, l, "page", f, ## a)
#define shrink_debug(l, f, a...)  \
        nilfs_debug_verbose(NILFS_VERBOSE_SHRINKER, l, "shrinker", f, ## a)
#define seg_debug(l, f, a...)  \
        nilfs_debug_verbose(NILFS_VERBOSE_SEGMENT, l, "segment", f, ## a)
#define inode_debug(l, f, a...)  \
        nilfs_debug_verbose(NILFS_VERBOSE_INODE, l, "inode", f, ## a)
#define recovery_debug(l, f, a...)  \
        nilfs_debug_verbose(NILFS_VERBOSE_RECOVERY, l, "recovery", f, ## a)
#define gc_debug(l, f, a...)  \
        nilfs_debug_verbose(NILFS_VERBOSE_GC, l, "gc", f, ## a)

#else /* CONFIG_NILFS_DEBUG */
#define nilfs_debug(l, f, a...)  do {} while(0)
#define nilfs_debug_verbose(v, l, c, f, a...)  do {} while(0)
#define nilfs_dump_stack(v, l)  do {} while(0)

#define page_debug(l, f, a...)  do {} while (0)
#define shrink_debug(l, f, a...)  do {} while(0)
#define seg_debug(l, f, a...)  do {} while(0)
#define inode_debug(l, f, a...)   do {} while(0)
#define recovery_debug(l, f, a...)   do {} while(0)
#define gc_debug(l, f, a...)   do {} while(0)
#endif /* CONFIG_NILFS_DEBUG */

#define page_warn(f, a...)					\
	do {							\
		printk(KERN_ERR "NILFS(PAGE): %s: "  \
		       f, __FUNCTION__, ## a);  \
	} while (0)

/*
 * VINODE_DEBUG() & BH_DEBUG()
 */
#ifdef CONFIG_NILFS_DEBUG
extern void nilfs_bh_debug(const char *, int, char *, struct buffer_head *);
extern void nilfs_count_freeing_bh(struct buffer_head *);
#define BH_DEBUG(bh, m)  nilfs_bh_debug(__FUNCTION__, __LINE__, (m), (bh))

/*
 * To enable VINODE debug message, set debug level of inode 3 as follows:
 *
 * # echo "`cat /proc/fs/nilfs/debug_option` -vvv /proc/fs/nilfs/" > 
 *    /proc/fs/nilfs/debug_option
 */
extern void nilfs_vinode_debug(const char *, int, char *, struct inode *);
#define VINODE_DEBUG(inode, m)  \
        do {  \
                if (nilfs_debug_info.verbose[NILFS_VERBOSE_INODE] >= 3)  \
                        nilfs_vinode_debug(__FUNCTION__, __LINE__, (m), (inode));  \
        } while(0)

#else /* CONFIG_NILFS_DEBUG */
#define BH_DEBUG(bh, m)  do {} while(0)
#define VINODE_DEBUG(inode, m)  do {} while(0)
#endif /* CONFIG_NILFS_DEBUG */


#ifdef CONFIG_NILFS_DEBUG

/*
 * debug switches
 */
enum {
	NILFS_VERBOSE_FS = 0,    /* Generic switches */
	NILFS_VERBOSE_PAGE,      /* Page operations */
	NILFS_VERBOSE_SHRINKER,  /* Shrinker */
	NILFS_VERBOSE_SEGMENT,   /* Segment construction */
	NILFS_VERBOSE_SEGINFO,   /* Segment summary information */
	NILFS_VERBOSE_INODE,     /* Inode operations */
	NILFS_VERBOSE_RECOVERY,  /* Recovery logic */
	NILFS_VERBOSE_GC,        /* Segment manager (Cleaner) */
	NILFS_VERBOSE_LIMIT
};

struct nilfs_debug_info {
	char  verbose[NILFS_VERBOSE_LIMIT];  /* message switches */
};

extern struct nilfs_debug_info nilfs_debug_info;

/*
 * NILFS_BH_DEBUG - trace get_bh, put_bh and brelse
 */
#ifdef NILFS_BH_DEBUG
#define get_bh(bh)	\
do {	\
        atomic_inc(&(bh)->b_count);	\
        BH_DEBUG((bh), "aft get");	\
} while(0)

#define put_bh(bh)	\
do {	\
        BH_DEBUG((bh), "bfr put");	\
        smp_mb__before_atomic_dec();	\
        atomic_dec(&(bh)->b_count);	\
} while(0)

#define brelse(bh)	\
do {	\
	if (bh) {	\
                BH_DEBUG((bh), "bfr rls");	\
		__brelse(bh);	\
        }	\
} while(0)

#define bforget(bh)	\
{	\
	if (bh)	\
                BH_DEBUG((bh), "bfr fgt");	\
		__bforget(bh);	\
} while (0)
#endif /* NILFS_BH_DEBUG */

#if defined(NILFS_SEMAPHORE_DEBUG) || defined(NILFS_SPINLOCK_DEBUG) || defined(NILFS_LOCK_BUFFER_DEBUG)

struct nilfs_sem_holders {
	struct task_struct *task;
	void *sem; 	/* locking object */
};


#ifdef NILFS_LOCK_BUFFER_DEBUG
#define HOLDERS_N 1024
#else
#define HOLDERS_N 24
#endif

extern struct nilfs_sem_holders holders_a[];
extern spinlock_t nilfs_sem_h_lock;

/*
 * CHECK_SEM_DOWN(), SEM_ADD() and CHECK_SEM_UP() can be called in
 * any context (interrupt may be enabled or may be disabled).
 * So, we MUST use _irqsave variant of spinlock instead of spin_lock().
 * The _irqsave variant is the safest and can cover other spinlock
 * variants.
 *
 * Note that inappropriate use of the spinlock causes deadlock.
 * For more detail, please refer to the following articles.
 *
 * o Documentation/DocBook/kernel-locking.pdf  -- written by Rusty Russell
 *   (can be made by 'make pdfdocs')
 * o Documentation/spinlocks.txt  -- written by Linus
 * o "spin_lock_irq vs. spin_lock_irqsave." -- Linux-Kernel ML discussion.
 */
static inline void
CHECK_SEM_DOWN(char *message, void *sem)
{
	struct task_struct *tsk = current;
	struct task_struct *w;
	int i;
	unsigned long flags;

	spin_lock_irqsave(&nilfs_sem_h_lock, flags);
	for (i = 0; i < HOLDERS_N; i++) {
		if (holders_a[i].task == tsk && holders_a[i].sem == sem) {
			spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
			printk("%s double down sem=%p task=%p\n", message, sem, tsk);
			return;
		}
		if (holders_a[i].sem == sem && (w = holders_a[i].task)) {
			spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
			printk("%s %p locked by %p state=%ld flags=0x%lx\n",
			       message, sem, w, w->state, w->flags);
			spin_lock_irqsave(&nilfs_sem_h_lock, flags);
		}
	}
	spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
}

static inline void
SEM_ADD(void *sem)
{
	struct task_struct *tsk = current;
	int i;
	unsigned long flags;

	spin_lock_irqsave(&nilfs_sem_h_lock, flags);
	for (i = 0; i < HOLDERS_N; i++)
		if (holders_a[i].task == (void *)0) {
			holders_a[i].task = tsk;
			spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
			holders_a[i].sem = sem;
			return;
		}
	spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
	printk("SEM_ADD lock holders overflow!\n");
}

static inline void
CHECK_SEM_UP(char *message, void *sem)
{
	struct task_struct *tsk = current;
	int i;
	unsigned long flags;

	spin_lock_irqsave(&nilfs_sem_h_lock, flags);
	for (i = 0; i < HOLDERS_N; i++) {
		if (holders_a[i].task == tsk && holders_a[i].sem == sem) {
			holders_a[i].task = (void *)0;
			spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
			return;
		}
	}
	spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
	printk("%s not found sem=%p task=%p\n", message, sem, tsk);
}
#endif /* defined(NILFS_SEMAPHORE_DEBUG) || defined(NILFS_SPINLOCK_DEBUG) ||
	  defined(NILFS_LOCK_BUFFER_DEBUG) */

#ifdef NILFS_SEMAPHORE_DEBUG

static inline void
SEM_DOWN(const char *fname, int line, struct semaphore *sem)
{
	struct task_struct *tsk = current;

	printk("SEM_DOWN %s(%d) sem=%p task=%p count=%d\n",
	       fname, line, sem, tsk, atomic_read(&sem->count));
	CHECK_SEM_DOWN("SEM_DOWN", (void *)sem);

	down(sem);
	SEM_ADD((void *)sem);
}

static inline void
SEM_UP(const char *fname, int line, struct semaphore *sem)
{
	struct task_struct *tsk = current;

	CHECK_SEM_UP("SEM_UP", (void *)sem);
	up(sem);
	printk("SEM_UP %s(%d) sem=%p task=%p count=%d\n",
	       fname, line, sem, tsk, atomic_read(&sem->count));
}

#define down(sem) SEM_DOWN(__FUNCTION__, __LINE__, (sem))
#define up(sem) SEM_UP(__FUNCTION__, __LINE__, (sem))

static inline int
SEM_TASK_CHECK(const char *fname, int line, char *message, struct rw_semaphore *sem, int try)
{
	struct task_struct *tsk = current;
	int ret = 1;
	
	printk("%s %s(%d) sem=%p task=%p count=%ld\n",
	       message, fname, line, sem, tsk, sem->count);
	CHECK_SEM_DOWN(message, (void *)sem);
	switch(try) {
	case 0:
		down_write(sem);
		break;
	case 1:
		ret = down_write_trylock(sem);
		break;
	case 2:
		down_read(sem);
		break;
	case 3:
		ret = down_read_trylock(sem);
		break;
	}
	if (ret)
		SEM_ADD((void *)sem);

	return ret;
}

static inline void
RW_WDOWN(const char *fname, int line, struct rw_semaphore *sem)
{
	SEM_TASK_CHECK(fname, line, "Down_Write", sem, 0);
}

static inline int
RW_WDOWNL(const char *fname, int line, struct rw_semaphore *sem)
{
	return SEM_TASK_CHECK(fname, line, "Down_Write_Trylock", sem, 1);
}

static inline void
RW_WUP(const char *fname, int line, struct rw_semaphore *sem)
{
	struct task_struct *tsk = current;
	long count = sem->count;

	CHECK_SEM_UP("Up_Write", (void *)sem);
	up_write(sem);
	printk("Up_Write %s(%d) sem=%p task=%p count=%ld\n", fname, line, sem, tsk, count-1);
}
#define down_write(sem) RW_WDOWN(__FUNCTION__, __LINE__, (sem))
#define down_write_trylock(sem) RW_WDOWNL(__FUNCTION__, __LINE__, (sem))
#define up_write(sem) RW_WUP(__FUNCTION__, __LINE__, (sem))

static inline void
RW_RDOWN(const char *fname, int line, struct rw_semaphore *sem)
{
	SEM_TASK_CHECK(fname, line, "Down_Read", sem, 2);
}

static inline int
RW_RDOWNL(const char *fname, int line, struct rw_semaphore *sem)
{
	return SEM_TASK_CHECK(fname, line, "Down_Read_Trylock", sem, 3);
}

static inline void
RW_RUP(const char *fname, int line, struct rw_semaphore *sem)
{
	struct task_struct *tsk = current;
	long count = sem->count;

	CHECK_SEM_UP("Up_Read", (void *)sem);
	up_read(sem);
	printk("Up_Read %s(%d) sem=%p task=%p count=%ld\n", fname, line, sem, tsk, count-1);
}

static inline void
RW_RUP_CHK(const char *fname, int line, struct rw_semaphore *sem)
{
	struct task_struct *tsk = current;
	long count = sem->count;

	CHECK_SEM_UP("Up_Read", (void *)sem);
	printk("Up_Read %s(%d) sem=%p task=%p count=%ld\n", fname, line, sem, tsk, count);
}

#define down_read(sem) RW_RDOWN(__FUNCTION__, __LINE__, (sem))
#define down_read_trylock(sem) RW_RDOWNL(__FUNCTION__, __LINE__, (sem))
#define up_read(sem) RW_RUP(__FUNCTION__, __LINE__, (sem))

#define drop_super(sb) \
{	\
	RW_RUP_CHK(__FUNCTION__, __LINE__, (&(sb)->s_umount));	\
	drop_super(sb);	\
}

#endif /* NILFS_SEMAPHORE_DEBUG */

#ifdef NILFS_LOCK_BUFFER_DEBUG
static inline void
CHECK_BH_LOCK(char *message, void *sem)
{
	int i;

	spin_lock_irq(&nilfs_sem_h_lock);
	for (i = 0; i < HOLDERS_N; i++) {
		if (holders_a[i].sem == sem && holders_a[i].task) {
			spin_unlock_irq(&nilfs_sem_h_lock);
			printk("%s double lock bh=%p\n", message, sem);
			return;
		}
	}
	spin_unlock_irq(&nilfs_sem_h_lock);
}

static inline void
BH_ADD(void *sem)
{
	struct task_struct *tsk = current;
	int i;

	spin_lock_irq(&nilfs_sem_h_lock);
	for (i = 0; i < HOLDERS_N; i++)
		if (holders_a[i].task == (void *)0) {
			holders_a[i].task = tsk;
			spin_unlock_irq(&nilfs_sem_h_lock);
			holders_a[i].sem = sem;
			return;
		}
	spin_unlock_irq(&nilfs_sem_h_lock);
	printk("SEM_ADD lock holders overflow!\n");
}

static inline void
CHECK_BH_UNLOCK(char *message, void *sem)
{
	struct task_struct *tsk = current;
	int i;
	unsigned long flags;

	spin_lock_irqsave(&nilfs_sem_h_lock, flags);
	for (i = 0; i < HOLDERS_N; i++) {
		if (holders_a[i].sem == sem && holders_a[i].task) {
			holders_a[i].task = (void *)0;
			spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
			return;
		}
	}
	spin_unlock_irqrestore(&nilfs_sem_h_lock, flags);
	printk("%s not found sem=%p task=%p\n", message, sem, tsk);
}

static inline void
LOCK_BH(const char *fname, int line, struct buffer_head *lock)
{
	struct task_struct *tsk = current;
	CHECK_BH_LOCK("LOCK_BUFFER", (void *)lock);
	lock_buffer(lock);
	BH_ADD((void *)lock);
	printk("LOCK_BUFFER %s(%d) bh=%p, task=%p\n",
	       fname, line, lock, tsk);
}
static inline void
UNLOCK_BH(const char *fname, int line, struct buffer_head *lock)
{
	struct task_struct *tsk = current;
	CHECK_BH_UNLOCK("UNLOCK_BUFFER", (void *)lock);
	unlock_buffer(lock);
	printk("UNLOCK_BUFFER %s(%d) bh=%p, task=%p\n",
	       fname, line, lock, tsk);
}

#define lock_buffer(bh) LOCK_BH(__FUNCTION__, __LINE__, (bh))
#define unlock_buffer(bh) UNLOCK_BH(__FUNCTION__, __LINE__, (bh))

#endif /* NILFS_LOCK_BUFFER_DEBUG */

#ifdef NILFS_LOCK_PAGE_DEBUG
static inline void
LOCK_PAGE(const char *fname, int line, struct page *page)
{
	printk("LOCK_PAGE %s(%d) page=%p\n",
	       fname, line, page);
	might_sleep();
	if (TestSetPageLocked(page)) {
		__lock_page(page);
		printk("LOCK_PAGE waited and locked (page=%p)\n", page);
	} else {
		printk("LOCK_PAGE locked (page=%p)\n", page);
	}
}

static inline void
UNLOCK_PAGE(const char *fname, int line, struct page *page)
{
	unlock_page(page);
	printk("UNLOCK_PAGE %s(%d) page=%p\n",
	       fname, line, page);
}

#define lock_page(page) LOCK_PAGE(__FUNCTION__, __LINE__, (page))
#define unlock_page(page) UNLOCK_PAGE(__FUNCTION__, __LINE__, (page))
#endif /* NILFS_LOCK_PAGE_DEBUG */

#ifdef NILFS_SPINLOCK_DEBUG
static inline void
SPLOCK_LOCK(const char *fname, int line, spinlock_t *lock)
{
	struct task_struct *tsk = current;
	CHECK_SEM_DOWN("SPIN_LOCK", (void *)lock);
	_spin_lock(lock);
	SEM_ADD((void *)lock);
	printk("SPIN_LOCK %s(%d) lock=%p, task=%p\n",
	       fname, line, lock, tsk);
}
static inline void
SPLOCK_UNLOCK(const char *fname, int line, spinlock_t *lock)
{
	struct task_struct *tsk = current;
	CHECK_SEM_UP("SPIN_UNLOCK", (void *)lock);
	_spin_unlock(lock);
	printk("SPIN_UNLOCK %s(%d) lock=%p, task=%p\n",
	       fname, line, lock, tsk);
}

static inline void
SPLOCK_LOCK_IRQ(const char *fname, int line, spinlock_t *lock)
{
	struct task_struct *tsk = current;
	CHECK_SEM_DOWN("SPIN_LOCK_IRQ", (void *)lock);
	_spin_lock_irq(lock);
	SEM_ADD((void *)lock);
	printk("SPIN_LOCK_IRQ %s(%d) lock=%p, task=%p\n",
	       fname, line, lock, tsk);
}
static inline void
SPLOCK_UNLOCK_IRQ(const char *fname, int line, spinlock_t *lock)
{
	struct task_struct *tsk = current;
	CHECK_SEM_UP("SPIN_UNLOCK_IRQ", (void *)lock);
	_spin_unlock_irq(lock);
	printk("SPIN_UNLOCK_IRQ %s(%d) lock=%p, task=%p\n",
	       fname, line, lock, tsk);
}

#ifdef CONFIG_SMP
#define nilfs_spin_lock_irqsave(lock, flags)	flags = _spin_lock_irqsave(lock)
#else
#define nilfs_spin_lock_irqsave(lock, flags)	_spin_lock_irqsave(lock, flags)
#endif

#define SPLOCK_LOCK_IRQSAVE(fname, line, lock, flags) \
do { \
	CHECK_SEM_DOWN("SPIN_LOCK_IRQSAVE", (void *)(lock)); \
	nilfs_spin_lock_irqsave((lock), (flags)); \
	SEM_ADD((void *)(lock)); \
	printk("SPIN_LOCK_IRQSAVE %s(%d) lock=%p, task=%p\n", \
	       (fname), (line), (lock), (current)); \
} while(0)

#define SPLOCK_UNLOCK_IRQRESTORE(fname, line, lock, flags) \
do { \
	CHECK_SEM_UP("SPIN_UNLOCK_IRQRESTORE", (void *)(lock)); \
	_spin_unlock_irqrestore((lock), (flags)); \
	printk("SPIN_UNLOCK_IRQRESTORE %s(%d) lock=%p, task=%p\n", \
	       (fname), (line), (lock), (current)); \
} while (0)
#undef spin_lock
#undef spin_unlock
#undef spin_lock_irq
#undef spin_unlock_irq
#undef spin_lock_irqsave
#undef spin_unlock_irqrestore

#define spin_lock(lock) SPLOCK_LOCK(__FUNCTION__, __LINE__, (lock))
#define spin_unlock(lock) SPLOCK_UNLOCK(__FUNCTION__, __LINE__, (lock))
#define spin_lock_irq(lock) SPLOCK_LOCK_IRQ(__FUNCTION__, __LINE__, (lock))
#define spin_unlock_irq(lock) SPLOCK_UNLOCK_IRQ(__FUNCTION__, __LINE__, (lock))
#define spin_lock_irqsave(lock, flags) SPLOCK_LOCK_IRQSAVE(__FUNCTION__, __LINE__, (lock), (flags))
#define spin_unlock_irqrestore(lock, flags) SPLOCK_UNLOCK_IRQRESTORE(__FUNCTION__, __LINE__, (lock), (flags))

#endif /* NILFS_SPINLOCK_DEBUG */

#ifdef NILFS_KMALLOC_DEBUG
static inline void *
NILFS_KMALLOC(const char *fname, int line, size_t size, int flags)
{
	void *ptr = kmalloc(size, flags);
	printk("KMALLOC %s(%d) ptr=%p, size=%u\n",
	       fname, line, ptr, size);
	return ptr;
}

static inline void
NILFS_KFREE(const char *fname, int line, const void *ptr)
{
	printk("KFREE %s(%d) ptr=%p\n", fname, line, ptr);
	kfree(ptr);
}

#define kmalloc(size, flags) NILFS_KMALLOC(__FUNCTION__, __LINE__, (size), (flags))
#define kfree(ptr) NILFS_KFREE(__FUNCTION__, __LINE__, (ptr))

#elif defined(NILFS_KMALLOC_COUNT)
static inline void *NILFS_KMALLOC(size_t size, int flags)
{
	extern atomic_t nilfs_kmalloc_n;
	void *ptr = kmalloc(size, flags);
	if (ptr)
		atomic_inc(&nilfs_kmalloc_n);
	return ptr;
}

static inline void NILFS_KFREE(const void *ptr)
{
	extern atomic_t nilfs_kfree_n;
	kfree(ptr);
	atomic_inc(&nilfs_kfree_n);
}

#define kmalloc(size, flags) NILFS_KMALLOC((size), (flags))
#define kfree(ptr) NILFS_KFREE(ptr)
#endif /* NILFS_KMALLOC_DEBUG */

#ifdef NILFS_BIO_COUNT
static inline struct bio *
NILFS_BIO_ALLOC(gfp_t gfp_mask, int nr_iovecs)
{
	extern atomic_t nilfs_get_bio_n;
	struct bio *bio = bio_alloc(gfp_mask, nr_iovecs);
	if (bio)
		atomic_inc(&nilfs_get_bio_n);
	return bio;
}

static inline void NILFS_BIO_PUT(struct bio *bio)
{
	extern atomic_t nilfs_put_bio_n;
	bio_put(bio);
	atomic_inc(&nilfs_put_bio_n);
}
#define bio_alloc(flags, nr_vecs) NILFS_BIO_ALLOC((flags), (nr_vecs))
#define bio_put(ptr) NILFS_BIO_PUT(ptr)
#endif /* NILFS_BIO_COUNT */

#ifdef NILFS_SLAB_COUNT
static inline void *
NILFS_KMEM_CACHE_ALLOC(kmem_cache_t *cachep, int flags)
{
	extern atomic_t nilfs_cache_alloc_n;
	void *ptr = kmem_cache_alloc(cachep, flags);
	if (ptr)
		atomic_inc(&nilfs_cache_alloc_n);
	return ptr;
}

static inline void
NILFS_KMEM_CACHE_FREE(kmem_cache_t *cachep, void *ptr)
{
	extern atomic_t nilfs_cache_free_n;
	kmem_cache_free(cachep, ptr);
	atomic_inc(&nilfs_cache_free_n);
}
#define kmem_cache_alloc(cachep, flags) NILFS_KMEM_CACHE_ALLOC((cachep), (flags))
#define kmem_cache_free(cachep, ptr) NILFS_KMEM_CACHE_FREE((cachep), (ptr))
#endif /* NILFS_SLAB_COUNT */

#ifdef NILFS_KMAP_COUNT
static inline void *NILFS_KMAP(struct page *page)
{
	extern atomic_t nilfs_kmap_n;
#ifdef CONFIG_HIGHMEM
	void *ptr = kmap(page);
#else
	might_sleep();
	void *ptr = page_address(page);
#endif
	if (ptr)
		atomic_inc(&nilfs_kmap_n);
	return ptr;
}

static inline void NILFS_KUNMAP(struct page *page)
{
	extern atomic_t nilfs_kunmap_n;
#ifdef CONFIG_HIGHMEM
	kunmap(page);
#else
	(void)(page);
#endif
	atomic_inc(&nilfs_kunmap_n);
}

static inline void *
NILFS_KMAP_ATOMIC(struct page *page, enum km_type type)
{
	extern atomic_t nilfs_kmap_n;
#ifdef CONFIG_HIGHMEM
	void *ptr = kmap_atomic(page, type);
#else
	void *ptr = page_address(page);
#endif
	if (ptr)
		atomic_inc(&nilfs_kmap_n);
	return ptr;
}

static inline void
NILFS_KUNMAP_ATOMIC(void *kvaddr, enum km_type type)
{
	extern atomic_t nilfs_kunmap_n;
#ifdef CONFIG_HIGHMEM
	kunmap_atomic(kvaddr, type);
#endif
	atomic_inc(&nilfs_kunmap_n);
}

#ifndef CONFIG_HIGHMEM
#undef kunmap
#undef kmap_atomic
#undef kunmap_atomic
#endif

#define kmap(page) NILFS_KMAP(page)
#define kunmap(page) NILFS_KUNMAP(page)
#define kmap_atomic(page, type) NILFS_KMAP_ATOMIC((page), (type))
#define kunmap_atomic(addr, type) NILFS_KUNMAP_ATOMIC((addr), (type))
#endif /* NILFS_KMAP_COUNT */

#ifdef NILFS_ALLOCPAGE_COUNT
static inline struct page *
NILFS_ALLOC_PAGE(unsigned gfp_mask)
{
	extern atomic_t nilfs_alloc_page_n;
	struct page *ret = alloc_pages(gfp_mask, 0);
	if (ret)
		atomic_inc(&nilfs_alloc_page_n);
	return ret;
}

static inline void
NILFS_FREE_PAGE(struct page *page)
{
	extern atomic_t nilfs_free_page_n;
	if (page_count(page) > 1) {
		page_debug(2, "page %p count %d\n", page, page_count(page));
		return;
	}
	if (PageReserved(page)) {
		page_debug(2, "PageReserved(%p)\n", page);
		return;
	}
	__free_pages(page, 0);
	atomic_inc(&nilfs_free_page_n);
}
#undef alloc_page
#define alloc_page(mask) NILFS_ALLOC_PAGE((mask))
#undef __free_page
#define __free_page(addr) NILFS_FREE_PAGE((addr))
#endif /* NILFS_ALLOCPAGE_COUNT */


/*
 * for debug, module static variables
 */
extern atomic_t nilfs_getblkbh_n;
extern atomic_t nilfs_putblkbh_n;
#ifdef NILFS_BH_COUNT_DETAIL
extern atomic_t nilfs_putblkbh_nilfs_allocated_n;
extern atomic_t nilfs_putblkbh_bbt_node_n;
extern atomic_t nilfs_putblkbh_ibt_node_n;
extern atomic_t nilfs_putblkbh_new_node_n;
extern atomic_t nilfs_putblkbh_inactive_node_n;
extern atomic_t nilfs_putblkbh_partial_node_n;
#endif
#ifdef NILFS_PAGE_COUNT
extern atomic_t nilfs_allocated_inode_page_n;
extern atomic_t nilfs_released_inode_page_n;
extern atomic_t nilfs_released_file_page_n;
#endif
extern atomic_t nilfs_released_inode_n;
extern atomic_t nilfs_allocated_inode_n;
#ifdef NILFS_KMALLOC_COUNT
extern atomic_t nilfs_kmalloc_n;
extern atomic_t nilfs_kfree_n;
#endif
#ifdef NILFS_BIO_COUNT
extern atomic_t nilfs_get_bio_n;
extern atomic_t nilfs_put_bio_n;
#endif
#ifdef NILFS_SLAB_COUNT
extern atomic_t nilfs_cache_alloc_n;
extern atomic_t nilfs_cache_free_n;
#endif
#ifdef NILFS_KMAP_COUNT
extern atomic_t nilfs_kmap_n;
extern atomic_t nilfs_kunmap_n;
#endif
#ifdef NILFS_ALLOCPAGE_COUNT
extern atomic_t nilfs_alloc_page_n;
extern atomic_t nilfs_free_page_n;
#endif

#endif /* CONFIG_NILFS_DEBUG */

#endif /* _NILFS_DEBUG_H */
/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
