#!/usr/bin/perl
#
# seekcp - seek a specified checkpoint in NILFS
#
# Copyright (C) 2006 Nippon Telegraph and Telephone Corporation.
# This file is part of NILFS and is licensed under the GPL.
#
# seekcp.pl,v 1.4 2006/05/15 06:47:15 amagai Exp
#
# Written by Satoshi Moriai <moriai@osrg.net>
#

use POSIX qw(mktime);

use constant {SIMPLE => 0, COMPACT => 1, LONG => 2, HTML => 3};
my $mode = SIMPLE;

my %MM = (
    Jan => 1, Feb => 2, Mar => 3, Apr => 4, May => 5, Jun => 6,
    Jul => 7, Aug => 8, Sep => 9, Oct => 10, Nov => 11, Dec => 12
);

sub usage {
    my ($code) = @_;

    print <<"EOF";
Usage: seekcp [option] device [date]
Options:
  -h, --help			show this help
  -l, --long			use a long listing format
  -c, --compact			use a compact listing format
  -x, --html=DIR		use a HTML listing format
				DIR is the mount point at the client side

Acceptable date formats:
  YYYY.MM.DD[.hh[.mm[.ss]]]	e.g. 2006.03.02.12.34.56
  YYYY_Mmm_DD_[hh:mm[:ss]]	e.g. 2006_Mar_02_12:34:56
  Mmm DD [YYYY] [hh:mm[:ss]]	e.g. Mar 02 2006 12:34:56
  Mmm DD [hh:mm[:ss]] [YYYY]	e.g. Mar 02 12:34:56 2006
  DD Mmm [YYYY] [hh:mm[:ss]]	e.g. 02 Mar 2006 12:34:56
  YYYY-MM-DD [hh:mm[:ss]]	e.g. 2006-03-02 12:34:56
  YYYY/MM/DD [hh:mm[:ss]]	e.g. 2006/03/02 12:34:56
EOF

    exit $code;
}

sub maketime {
    my ($str) = @_;
    my ($Y, $M, $D, $h, $m, $s, $t);
    my ($G, $T);

    if ($str =~ /^\d{4}\./) {				# YYYY.MM.DD.hh.mm.ss
	($Y, $M, $D, $h, $m, $s) = split /\./, $str;
    } else {
	for $tk (split /[_ ]+/, $str) {
	    if ($tk =~ /^[A-Z][a-z]{2}$/) {			# Mmm
		$M = $MM{$tk};
	    } elsif ($tk =~ /^\d{4}-\d{1,2}-\d{1,2}$/) {	# YYYY-MM-DD
		($Y, $M, $D) = split /-/, $tk;
	    } elsif ($tk =~ /^\d{4}\/\d{1,2}\/\d{1,2}$/) {	# YYYY/MM/DD
		($Y, $M, $D) = split '/', $tk;
	    } elsif ($tk =~ /^\d{1,2}:\d{1,2}/) {		# hh:mm:ss
		($h, $m, $s) = split /:/, $tk;
	    } elsif ($tk =~ /^\d{4}$/) {			# YYYY
		$Y = $tk;
	    } elsif ($tk =~ /^\d{1,2}$/) {			# DD
		$D = $tk;
	    } else {
		usage(1);
	    }
	}
    }

    $Y = ((localtime)[5] + 1900) unless defined $Y;
    if (!defined $M && !defined $D) {
	($D, $M) = (localtime)[3,4];
	$M++;
    }
    $M = 12 unless defined $M;
#   $D = 31 unless defined $D;
    usage(1) unless defined $D;
    $h = 23 unless defined $h;
    $m = 59 unless defined $m;
    $s = 59 unless defined $s;
    $t = mktime($s, $m, $h, $D, $M - 1, $Y - 1900);
#   print STDERR "$Y-$M-$D $h:$m:$s   $t\n"; $t;	# DEBUG
}

sub printdata {
    printf "%04d.%02d.%02d.%02d.%02d.%02d %d %d %d\n", @_;
}

sub printhead {
    print <<"EOF";
<html>
<table border="0" cellpadding="0">
<col span="1" align="left"><col span="1" align="right">
EOF
}

sub printentry {
    my ($W, $Mmm, $M, $D, $h, $m, $s, $Y, $ssize, $dir) = @_;
    my ($date);

    printf "<tr><td><a href=\"$dir/";
    printf "%04d.%02d.%02d.%02d.%02d.%02d", $Y, $M, $D, $h, $m, $s;
    printf "\" target=\"Content\">";
    $date = sprintf("%s %s %2d %02d:%02d:%02d %d", $W, $Mmm, $D, $h, $m, $s, $Y);
    $date =~ s/ /&nbsp;/g;
    printf "<tt>%s</tt></a></td><td><tt>&nbsp;%d KB</tt></td></tr>\n", $date, $ssize * 4;
}

sub printtail {
    my ($dir) = @_;

    print <<EOF;
<tr><td><a href="$dir/cur" target="Content"><tt>Current</tt></a></td></tr>
</table>
</html>
EOF
}

#
# main routine
#

while ($_ = $ARGV[0], /^-/) {
    shift;
    if (/-h/ || /--help/) { usage(0); }
    if (/-l$/ || /--long/) { $mode = LONG; last; }
    if (/-c$/ || /--compact/) { $mode = COMPACT; last; }
    if (/-x$/) { $mode = HTML; $rootdir = $ARGV[0]; shift; last; }
    if (/--html=(\S*)/) { $mode = HTML; $rootdir = $1; last; }
}

if ($#ARGV < 0) {
    usage(1);
}

if ($> != 0) {
    print STDERR "must be root!\n";
    exit 1;
}

open LISTCP, "printf 'listcp\nquit\n' | inspect $ARGV[0] |"
    || die "Can't popen listcp $!";

if ($#ARGV >= 1) {
    shift;
    $stime = maketime(join ' ', @ARGV);
#   printf STDERR "%d\n", $stime;			# DEBUG
}

if ($mode == HTML) {
    printhead();
}

while (<LISTCP>) {
    chop;
    s/nilfs> //;
    next unless /MajorCP/;

    ($str = $_) =~ s/^ +//;
    ($sno, $ssize, $W, $Mmm, $D, $T, $Y, $sstat, $sksize) = split / +/, $str;
    $M = $MM{$Mmm};
    ($h, $m, $s) = split /:/, $T;

    if (defined $stime) {
	last if ($matched);
	next if (mktime($s, $m, $h, $D, $M - 1, $Y - 1900) < $stime);
	$matched = 1;
    }

    if ($mode == LONG) {
	print "$_\n";
    } elsif ($mode == COMPACT) {
	printdata($Y, $M, $D, $h, $m, $s, $sno, $ssize, $sksize, $sstat);
    } elsif ($mode == HTML) {
	printentry($W, $Mmm, $M, $D, $h, $m, $s, $Y, $ssize, $rootdir);
    } else {
	if (defined $delim) {
	    print $delim;
	} else {
	    $delim = ' ';
	}
	print $sno;
    }
}

if ($mode == SIMPLE && defined $delim) {
    print "\n";
}

if ($mode == HTML) {
    printtail($rootdir);
}

if (defined $stime && !defined $matched) {
    exit 1;
}

exit 0;
