#!/bin/sh
# listcp - A checkpoint viewer
# Copyright (C) 2006 Nippon Telegraph and Telephone Corporation.
# This file is part of NILFS and is licensed under the GPL.
#
# listcp.sh,v 1.1 2006/03/20 08:53:46 ryusuke Exp
#
# Written by Ryusuke Konishi <ryusuke@osrg.net>
#
cmd=`basename $0`

function show_help() {
    echo "Usage: $cmd [options..] <device>"
    echo "       $cmd [options..] <directory>"
    echo "Options:"
    echo "  -h            show this help"
    echo "  -l            use long listing format"
    echo "  -e            print the number of seconds from the epoch instead"
    echo "                of formatted date string"
    echo "  -s            list only the checkpoint having sketch data"
    echo "  -t            show sketch (only text data is displayed)"
    echo "  -b            browse checkpoints (applying autofs is recommended,"
    echo "                which is described in misc/autofs )"
}

inspect=/sbin/inspect
tmpdir=/tmp
while getopts hlstbe o; do
    case $o in
	h)
	    show_help 1>&2
	    exit 0 ;;
	l)
	    verbose=1 ;;
	s)
	    cp_with_sketch=1 ;;
	t)
	    max_line=${TAG_LINES:-10}
	    show_sketch=1 ;;
	b)
	    browser=${BROWSER:-w3m}
	    browse_cp=1 ;;
	e)
	    epoch=1 ;;
	*)
	    show_help 1>&2
	    exit 1 ;;
    esac
done
shift `expr $OPTIND - 1`

function fs_dir2dev() {
    df -t $2 "$1" | sed -ne '2{s/^\(\S\+\)\s.*$/\1/;p;q}'
}

function is_autofs_dir() {
    sed -ne '\|^automount(\S\+)\s\+'"$1"'\s\+autofs\s|{x;s/^/y/;p;q}' /etc/mtab
}

if [ -z "$1" ]; then 
    show_help 1>&2; exit 1
elif [ -b "$1" ]; then
    device="$1"
elif [ -d "$1" ]; then
    device=`fs_dir2dev "$1" nilfs`
    if [ -z "$device" ]; then
	autofs=`is_autofs_dir "$1"`
	if [ "$autofs" = y -a -d $1/cur ]; then
	    device=`fs_dir2dev "$1/cur" nilfs`
	    mntdir="$1/cur"
	    autofs_dir="$1"
	fi
    else
	mntdir=`df -t nilfs "$1" | sed -ne '2{s/^.*\s\(\S\+\)$/\1/;p;q}'`
	autofs_dir=`dirname $mntdir`
	autofs=`is_autofs_dir "$autofs_dir"`
    fi
    if [ -z "$device" ]; then
	echo "Not NILFS directory" 1>&2; exit 1
    fi
    test "$autofs" = y || unset -v autofs_dir
else
    echo "Not directory nor device: $1" 1>&2; exit 1
fi

if [ ! -x $inspect ]; then echo "Please install inspect" 1>&2; exit 1; fi
if [ ! -r $device ]; then echo "Device not readable" 1>&2; exit 1; fi

function getcp() {
    (echo "listcp"; echo "quit") | $inspect ${epoch:+-e} $1 | sed -n ${filter:+-e "$filter"} -e '/MajorCP/{s/^nilfs> //;p}'
}

function print_file_type() {
    case "$1" in
	'J') echo jpeg ;;
	'G') echo gif ;;
	'P') echo png ;;
	'H') echo html ;;
	'X') echo xml ;;
	'W') echo wiki ;;
	'E') echo executable ;;
	'T') echo text ;;
	'U') echo uri ;;
	'O'|?) echo other ;;
    esac
}

function sketch_abort() { rm -f $tf; }

function print_sketch() {
    if expr "$3" : ".*\s0$" > /dev/null; then echo; return; fi
    tf=$tmpdir/sketch.$PPID.$2
    trap sketch_abort SIGINT
    $inspect -s $2 -o $tf $1 > /dev/null
    if [ -f $tf ]; then
	if [ -s $tf ]; then
	    data_size=`stat -c '%s' $tf`
	    offset=0
	    p=""
	    while [ $offset -lt $data_size ]; do
		h=`dd if=$tf bs=1 skip=$offset count=6 2> /dev/null`
		case `expr substr "$h" 1 1` in
		    '!')
			t=`expr substr "$h" 2 1`;  bytes=`expr substr "$h" 3 4 + 0`
			case "$t" in
			    T|U|W|X|H)
				dd if=$tf bs=1 skip=`expr $offset + 6` count=$bytes 2> /dev/null |
				sed -n -e '1{s/^\(.\{0,'$lc'\}\).*$/'"$p"' \1\n/};:loop;s/\n\n$/\n/;P;$q;'$max_line'q;n;s/^\(.\{0,'$lc'\}\).*$/'"$indent"'| \1\n/;b loop' ;;
			    *)
				echo "$p"' <'$bytes' bytes '`print_file_type $t`' data>' ;;
			esac
			offset=`expr $offset + $bytes + 6`
			p="$indent|"
			;;
		    *)
			rest=`expr $data_size - $offset`
			dd if=$tf bs=1 skip=$offset count=$rest 2> /dev/null |
			sed -n -e ':loop;/[[:cntrl:]]/{x;s/^/ <'$rest' bytes binary data>/;p;q};1{s/^\(.\{0,'$lc'\}\).*$/'"$p"' \1/;b next};s/^\(.\{0,'$lc'\}\).*$/'"$indent"'| \1/;:next;p;$q;'$max_line'q;n;b loop'
			break ;;
		esac
	    done
	else
	    echo
	fi
	rm -f $tf
    fi
    trap "" SIGINT
}

function m2i() {
    local m
    case $1 in
	Jan) m=1;; Feb) m=2;; Mar) m=3;; Apr) m=4;; May) m=5;; Jun) m=6;;
	Jul) m=7;; Aug) m=8;; Sep) m=9;; Oct) m=10;; Nov) m=11;; Dec) m=12;;
    esac
    echo $m
}

function expand_wiki_to_html() {
    sed -ne 's|\[\[\([^:]\+\):\(.*\+\?\)\]\]|<a href="\2">\1</a> |;s|^$|<p>|;P'
    echo '<br>'
}

function convert_sketch_to_html() {
    local td=$1 b=$2 dat=$1/$2.dat offset=0 i=1 data_size t h ft
    [ -f $dat -a -s $dat ] || return
    echo '<p>'
    data_size=`stat -c '%s' $dat`
    while [ $offset -lt $data_size ]; do
	h=`dd if=$dat bs=1 skip=$offset count=6 2> /dev/null`
	case `expr substr "$h" 1 1` in
	    '!')
		t=`expr substr "$h" 2 1`;  bytes=`expr substr "$h" 3 4 + 0`
		ft=`print_file_type $t`
		case "$t" in
		    T|U|W)
			dd if=$dat bs=1 skip=`expr $offset + 6` count=$bytes 2> /dev/null |
			case "$t" in
			    T) sed 's|^\(.*\)$|\1<br>|' ;;
			    W) expand_wiki_to_html ;;
			    U) sed 's|^\(.*\)$|<a href="\1">\1</a><br>|' ;;
			esac ;;

		    X|H)
			dd if=$dat bs=1 of="$td/${b}_$i.$ft" skip=`expr $offset + 6` count=$bytes 2> /dev/null
			echo '<a href="'"${b}_$i.$ft"'>'"${b}_$i.$ft"'</a><br>'
			;;
		    J|G|P)
			dd if=$dat bs=1 of="$td/${b}_$i.$ft" skip=`expr $offset + 6` count=$bytes 2> /dev/null
			echo '<img src="'$2_$i.$ft'" alt="'"$bytes $ft data"'">'
			;;
		    *)
			echo '<p>&lt;'$bytes' bytes '$ft' data&gt;</p>' ;;
		esac
		offset=`expr $offset + $bytes + 6`
		t2=$t ;;
	    *)
		rest=`expr $data_size - $offset`
		echo '<pre>'
		dd if=$dat bs=1 skip=$offset count=$rest 2> /dev/null |
		sed -ne ':loop;/[[:cntrl:]]/{x;s/^/ <'$rest' bytes binary data>/;p;q};:loop:s|^\(.*\)$|<p>\1</p>|;P;$q;'$max_line'q;n;b loop'
		echo '</pre>'
		break ;;
	esac
	i=`expr $i + 1`
    done
    echo '</p>'
}

function generate_html() {
    local l=`expr "$LANG" : "\(ja\)_JP"` cs of='cat'
    local b c d1 d2 d3 d4 d5 f s cpdir sketch=$3/.sketch
    : ${l:=en}
    case "$l" in
	ja) cs="EUC-JP";  test -x "`which nkf`" && of='nkf -e' ;;
	*)  cs="iso-8859-1";;
    esac
    echo '<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01//EN" "http://www.w3.org/TR/html4/strict.dtd">'
    echo '<html lang="'$l'">'
    echo '<head>'
    echo '<meta http-equiv="content-Type" content="text/html; charset='$cs'">'
    echo "<title> NILFS checkpoints ($3) </title>"
    echo '</head>'

    echo '<body>'
    echo "<H2> NILFS checkpoints ($3 on $2)</H2>"
    test -n "$4" || echo '<p>Introduce autofs to browse snapshots. See misc/autofs/README.</p>'
    echo '<H3><a href="file://'"$3"'/.">Current State</a></H3>'
    if [ -f $sketch ]; then
	cp $sketch $1/cur.dat
	convert_sketch_to_html $1 cur | $of
    fi
    echo '<HR>'
    while read b c d1 d2 d3 d4 d5 f s; do
	if [ -n "$4" ]; then
	    cpdir="$4/$d5.`m2i $d2`.$d3.${d4//:/.}"
	    echo '<H3><a href="file://'"$cpdir"'/.">'"$d1 $d2 $d3 $d4 $d5"' ('$c' blocks) </a></H3>'
	else
	    echo "<H3>$d1 $d2 $d3 $d4 $d5 ($c blocks)</H3>"
	fi
	if [ "$s" != 0 ]; then
	    $inspect -s $b -o "$1/$b.dat" "$2" > /dev/null
	    convert_sketch_to_html $1 $b | $of
	    echo '<HR>'
	fi
    done
    echo '<HR>'
    echo '</body>'
    echo '</html>'
}

function abort_browse() { rm -rf $td; }

function list_cp_with_browser() {
    td="$tmpdir/$cmd.$PPID"
    mkdir -p $td; chmod go-rxw $td
    trap abort_browse SIGINT

    local of="$td/nilfs.html"
    generate_html $td $@ > $of
    $browser $of
    wait

    rm -rf "$td"
    trap "" SIGINT
}

function list_sketch() {
    test -x "`which resize`" && eval `resize -u`
    export LANG=C
    if [ -n "$verbose" ]; then
	if [ -n "$epoch" ]; then
	    lc=`expr ${COLUMNS:-80} - 57`
	    indent='                          '
	    if [ $lc -le 0 ]; then lc=1; fi
	    while read b d; do
		e=`expr "$d" : "\([0-9]\+\s\+[0-9]\+\).*"`
		printf "% 10s % 16s" $b "$e"
		print_sketch "$1" $b "$d"
	    done
	else
	    lc=`expr ${COLUMNS:-80} - 43`
	    indent='                                        '
	    if [ $lc -le 0 ]; then lc=1; fi
	    while read b d; do
		e=`expr "$d" : "\([0-9]\+[[:print:]]\+[0-9]\{2\}:[0-9]\{2\}:[0-9]\{2\}\s[0-9]\{4\}\).*"`
		printf "% 10s % 30s" $b "$e"
		print_sketch "$1" $b "$d"
	    done
	fi
    else
	lc=`expr ${COLUMNS:-80} - 12`
	indent='         '
	if [ $lc -le 0 ]; then lc=1; fi
	while read b d; do
	    printf "% 10s" $b
	    print_sketch "$1" $b "$d"
	done
    fi
}

if [ -n "$cp_with_sketch" ]; then
    filter='/\s\+0$/{d}'
fi

if [ -n "$browse_cp" ]; then
    if [ ! -x "`which $browser`" ]; then
	echo "Cannot find valid web browser. " 1>&2
	echo "Please set path to your browser in BROWSER variable." 1>&2
	exit 1
    elif [ -z "$mntdir" ]; then
	echo "Specify a directory in NILFS." 1>&2
	exit 1
    elif [ -n "$epoch" ]; then
	echo "Epoch option conflicted with the browse option." 1>&2
	exit 1
    fi
    getcp $device | list_cp_with_browser $device "$mntdir" "$autofs_dir"
elif [ -n "$show_sketch" ]; then
    getcp $device | list_sketch $device
elif [ -z "$verbose" ]; then
    getcp $device | awk '{printf("%d ", $1)} END {print ""}'
else
    getcp $device
fi
