/*
 * ialloc.c - NILFS inode allocator and free inode management
 *
 * Copyright (C) 2005 Nippon Telegraph and Telephone Corporation.
 *
 * This file is part of NILFS.
 *
 * NILFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * NILFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with NILFS; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * ialloc.c,v 1.80 2006/07/12 13:00:16 ryusuke Exp
 *
 * Written by Amagai Yoshiji <amagai@osrg.net>
 */

#include <linux/config.h>
#include <linux/quotaops.h>
#include <linux/sched.h>
#include <linux/backing-dev.h>
#include <linux/buffer_head.h>
#include <linux/random.h>
#include <linux/pagemap.h>

#include "nilfs.h"

static struct buffer_head *
nilfs_an_inode(ino_t *, struct nilfs_sb_info *);


int nilfs_handle_ibt_error(int err, const char *fname, struct super_block *sb)
{
	err = nilfs_convert_btree_error(err);
	if (err == -EINVAL)
		nilfs_error(sb, fname, "broken inode btree\n");
	return err;
}

/*
 * The free inodes are managed by B-Tree and cache
 */

int
nilfs_few_free_inodes(int nf, struct nilfs_sb_info *sbi)
{
	if (nf <= 0)
		return 0;
	if (sbi->s_free_inodes_len < S_FREE_INODES_LEN_WMARK)
		return 1;
	if (sbi->s_free_inodes_len < S_FREE_INODES_LEN_WMARK*2 &&
	    sbi->s_inodes_per_block/2 <= nf)
		return 1;
	return 0;
}

struct inode *
nilfs_new_inode(struct inode *dir, int mode)
{
	struct super_block *sb;
	struct nilfs_sb_info *sbi;
	struct nilfs_inode_info *vi;
	struct nilfs_free_inode_list *fi = NULL;
	struct nilfs_inode_hdr *hdr;

	struct inode *inode;
	ino_t ino = 0;
	struct buffer_head *nbh;
	int err = -ENOMEM;
	int ret; 
	unsigned long btkey;

	sb = dir->i_sb;
	inode = new_inode(sb);
	if (unlikely(!inode))
		return ERR_PTR(-ENOMEM);

	/* cannot call nilfs_set_inode_flags() here. */
	mapping_set_gfp_mask(inode->i_mapping,
			     mapping_gfp_mask(inode->i_mapping) & ~__GFP_FS);
	sbi = NILFS_SB(sb);

	down(&sbi->s_free_inodes_sem);
	if (sbi->s_free_inodes_count < sbi->s_inodes_per_block/3) {
		fi = kmalloc(sizeof(struct nilfs_free_inode_list), GFP_NOFS);
		if (unlikely(fi == NULL))
			goto failed_sem;

		ret = nilfs_btree_find_unused_key(&sbi->s_inode_root, &btkey);
		if (unlikely(ret < 0)) {
			inode_debug(1, "nilfs_btree_find_unused_key failed (ret=%d)\n", ret);
			err = nilfs_handle_ibt_error(ret, __FUNCTION__, sb);
			goto failed_fi;
		}
		/*
		 * inode_debug(2, "nilfs_btree_find_unused_key return %lu, inodes_per_block %lu\n",
		 *             btkey, sbi->s_inodes_per_block);
		 */
		ino = btkey * sbi->s_inodes_per_block;
		nbh = nilfs_create_inode_block(sb, ino);
		if (unlikely(nbh == NULL)) {
			inode_debug(1, "inode block creation failed\n");
			goto failed_fi;
		}
		ret = nilfs_btree_insert(&sbi->s_inode_root, btkey, nbh);
		if (unlikely(ret < 0)) {
			inode_debug(1, "B-tree insertion failed (ret=%d)\n", ret);
			if (ret == -NILFS_BTREE_EEXIST)
				inode_debug(1, "unused key was not unused. (key=%lu)\n", btkey);

			err = nilfs_handle_ibt_error(ret, __FUNCTION__, sb);
			goto failed_nbh;
		}
		hdr = (struct nilfs_inode_hdr *)nbh->b_data;
		set_buffer_nilfs_inode_on_list(nbh);
		fi->free_bh = nbh;
		list_add_tail(&fi->buffers, &sbi->s_free_inodes);
		sbi->s_free_inodes_len++;
		sbi->s_free_inodes_count += sbi->s_inodes_per_block;

#if 0		/* Following lines can be omitted */
		fi = NULL;  /* prevent fi from being freed in error conditions */
		get_bh(nbh); /* link to nilfs_free_inode_list */
		brelse(nbh); /* correspond to nilfs_create_inode_block() */
		inode_debug(2, "list_add_tail0 %ld %p\n", ino, nbh);
		nbh = NULL;
#endif
	}

	nbh = nilfs_an_inode(&ino, sbi);
	up(&sbi->s_free_inodes_sem);

	inode->i_uid = current->fsuid;
	if (dir->i_mode & S_ISGID) {
		inode->i_gid = dir->i_gid;
		if (S_ISDIR(mode))
			mode |= S_ISGID;
	} else
		inode->i_gid = current->fsgid;

	inode->i_mode = mode;
	inode->i_ino = ino;
	inode->i_blksize = PAGE_SIZE;	/* This is the optimal IO size (for stat), not fs block size */
	inode->i_blocks = 0;
	inode->i_mtime = inode->i_atime = inode->i_ctime = CURRENT_TIME;
	inode->i_sb = sb;
	inode->i_mapping = &inode->i_data;

	vi = NILFS_I(inode);
	vi->i_bh = nbh;		/* get_bh()-ed in nilfs_an_inode() */

	ret = nilfs_btree_init(&vi->i_block_root, NILFS_BTREE_BLOCK, 0, inode);
	BUG_ON(ret < 0);

	vi->i_state = NILFS_STATE_NEW;
	vi->i_flags = NILFS_I(dir)->i_flags;
	if (S_ISLNK(mode))
		vi->i_flags &= ~(NILFS_IMMUTABLE_FL | NILFS_APPEND_FL);
	if (!S_ISDIR(mode))
		vi->i_flags &= ~NILFS_DIRSYNC_FL;

	vi->i_file_acl = 0;
	vi->i_dir_acl = 0;
	vi->i_dtime = 0;
	vi->i_dir_start_lookup = 0;
#ifdef CONFIG_NILFS_FS_POSIX_ACL
	vi->i_acl = NULL;
	vi->i_default_acl = NULL;
#endif
	nilfs_set_inode_flags(inode);
	spin_lock(&sbi->s_next_gen_lock);
	inode->i_generation = sbi->s_next_generation++;
	spin_unlock(&sbi->s_next_gen_lock);
	insert_inode_hash(inode);

	err = nilfs_init_acl(inode, dir);
	if (unlikely(err))
		goto failed; /* never occur. When implementing
				nilfs_init_acl(), cancelation of nilfs_btree_insert()
				must be handled correctly. */

	mark_inode_dirty(inode);
	atomic_inc(&sbi->s_inodes_count);
#ifdef NILFS_PAGE_COUNT
	atomic_inc(&nilfs_allocated_inode_n);
#endif
	return inode;

 failed:
	inode->i_nlink = 0;
	iput(inode);
	return ERR_PTR(err);

 failed_nbh:
	brelse(nbh);

 failed_fi:
	if (fi)
		kfree(fi);

 failed_sem:
	up(&sbi->s_free_inodes_sem);
	make_bad_inode(inode);
	iput(inode);
	return ERR_PTR(err);
}

void
nilfs_free_inode(struct inode *inode)
{
	struct nilfs_inode *dvi;
	struct super_block *sb = inode->i_sb;
	struct nilfs_sb_info *sbi = NILFS_SB(sb);
	struct buffer_head *nbh;
	struct nilfs_inode_hdr *hdr;
	struct nilfs_free_inode_list *fi = NULL;
	unsigned long hino;
	int hfree;
	int rino;
	int nfree;

	VINODE_DEBUG(inode,"");
	dvi = nilfs_load_inode_block(sbi, inode, &nbh);
	if (IS_ERR(dvi)) {
		nilfs_warning(sb, __FUNCTION__,
			      "unable to read inode (ino=%lu)",
			      inode->i_ino);
		goto failed;
	}
	clear_inode(inode);

	hdr = (struct nilfs_inode_hdr *)nbh->b_data;
	down(&sbi->s_free_inodes_sem);
	nfree = le32_to_cpu(hdr->ih_nfree) + 1;
	hino = le64_to_cpu(hdr->ih_ino);
	hfree = le32_to_cpu(hdr->ih_free);
	rino = inode->i_ino - hino;

	hdr->ih_nfree = cpu_to_le32(nfree);
	dvi->i_dtime = cpu_to_le64(get_seconds());
        /* setting NILFS_UNUSED_FL and clear all other flags */
	dvi->i_flags = cpu_to_le32(NILFS_INODE_UNUSED); 

	if (nfree == 1) {
		dvi->i_mode = cpu_to_le16(sbi->s_inodes_per_block - rino - 1);
		hdr->ih_free = cpu_to_le32(rino);
	} else if (rino < hfree) {
		dvi->i_mode = cpu_to_le16(hfree - rino - 1);
		hdr->ih_free = cpu_to_le32(rino);
	} else {
		int n = hfree;
		struct nilfs_inode *d = (struct nilfs_inode *)(nbh->b_data + sizeof(struct nilfs_inode_hdr));
		int x;

		d += hfree;
		for (;;) {
			x = le16_to_cpu(d->i_mode) + 1;
			if (rino <= n + x) break;
			n += x;
			d += x;
		}
		dvi->i_mode = le16_to_cpu(n + x - rino - 1);
		d->i_mode = le16_to_cpu(rino - n - 1);
	}

	if (nilfs_few_free_inodes(nfree, sbi) &&
	    (fi = kmalloc(sizeof(struct nilfs_free_inode_list), GFP_NOFS)) != NULL &&
	    !test_set_buffer_nilfs_inode_on_list(nbh)) {
		fi->free_bh = nbh;
		list_add_tail(&fi->buffers, &sbi->s_free_inodes);
		sbi->s_free_inodes_len++;
		sbi->s_free_inodes_count += nfree;
		get_bh(nbh);	/* link to nilfs_free_inode_list */
		inode_debug(2, "list_add_tail1 %ld %p\n", hino, nbh);
	} else {
		if (fi)
			kfree(fi);
		if (buffer_nilfs_inode_on_list(nbh))
		    sbi->s_free_inodes_count++;
		if (nfree == sbi->s_inodes_per_block) {
			/* 
			 * all inodes in this buffer are unused
			 * no writeback needed
			 */
			inode_debug(2, "all inodes are free in the buffer\n");
			clear_buffer_dirty(nbh);
			if (!buffer_nilfs_inode_on_list(nbh)) {
				int err;
				err = nilfs_btree_delete(&sbi->s_inode_root, hino / sbi->s_inodes_per_block);
				if (unlikely(err < 0)) {
					inode_debug(0, "cannot delete inode buffer "
						    "from btree, err=%d\n", err);
					nilfs_handle_ibt_error(err, __FUNCTION__, sb);
				}
			}
			goto out;
		}
	}
	nilfs_mark_inode_buffer_dirty(sbi, nbh);
 out:
	atomic_dec(&sbi->s_inodes_count);
	brelse(nbh);
	up(&sbi->s_free_inodes_sem);
	return;

 failed:
	clear_inode(inode);
	return;
}

void 
nilfs_init_iblock(struct buffer_head *bh, ino_t ino, struct nilfs_sb_info *sbi)
{
	struct nilfs_inode_hdr *hdr = (struct nilfs_inode_hdr *)bh->b_data;
	struct nilfs_inode *vi = (struct nilfs_inode *)(bh->b_data + sizeof(struct nilfs_inode_hdr));
	int i;

	hdr->ih_ino = cpu_to_le64(ino);
	hdr->ih_nfree = cpu_to_le32(sbi->s_inodes_per_block);
	hdr->ih_free = 0;
	for (i = 0; i < sbi->s_inodes_per_block; i++) {
		vi->i_flags = cpu_to_le32(NILFS_INODE_NEW);
		vi->i_mode = 0;
		vi++;
	}
}

static struct buffer_head *
nilfs_an_inode(ino_t *ret, struct nilfs_sb_info *sbi)
{
	struct nilfs_free_inode_list *fi = list_entry(sbi->s_free_inodes.next, struct nilfs_free_inode_list, buffers);
	struct buffer_head *bh;
	struct nilfs_inode_hdr *hdr;
	int nfree, n;
	struct nilfs_inode *vi;

	bh = fi->free_bh;
	get_bh(bh);	/* link to vi->i_bh */

	hdr = (struct nilfs_inode_hdr *)bh->b_data;
	nfree = le32_to_cpu(hdr->ih_nfree);
	BUG_ON(nfree == 0);

	sbi->s_free_inodes_count--;
	vi = (struct nilfs_inode *)(bh->b_data + sizeof(struct nilfs_inode_hdr));
	n = le32_to_cpu(hdr->ih_free);

	*ret = le64_to_cpu(hdr->ih_ino) + n;

	vi += n;
	n += le16_to_cpu(vi->i_mode) + 1; /* vi->i_mode stores a distance to
					     the next free inode. */
	hdr->ih_free = cpu_to_le32(n);
	--nfree;
	hdr->ih_nfree = cpu_to_le32(nfree);

	if (nfree == 0) {
		/* inode_debug(2, "list_del %ld %p\n", *ret, fi->free_bh); */
		clear_buffer_nilfs_inode_on_list(fi->free_bh);
		brelse(fi->free_bh);	/* unlink to nilfs_free_inode_list */
		fi->free_bh = NULL;
		list_del_init(&fi->buffers);
		sbi->s_free_inodes_len--;
		kfree(fi);
	}
	nilfs_mark_inode_buffer_dirty(sbi, bh);
	return bh;
}

/* Local Variables:	*/
/* eval: (c-set-style "linux")	*/
/* End:			*/
