//
// PKCS12Test.cs - NUnit tests for PKCS12
//
// Author:
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// (C) 2003 Motus Technologies Inc. (http://www.motus.com)
// Copyright (C) 2005,2006 Novell, Inc (http://www.novell.com)
//

using NUnit.Framework;

using System;
using System.Collections;
using System.IO;
using System.Security.Cryptography;
using System.Text;

using Mono.Security.Cryptography;
using Mono.Security.X509;

namespace MonoTests.Mono.Security.X509 {

	// References:
	// a.	Dr Stephen N Henson's PKCS#12 Key generation test vectors
	//	http://www.drh-consultancy.demon.co.uk/test.txt

	[TestFixture]
	public class PKCS12Test : Assertion {

		static public byte[] farscape_pfx = { 0x30, 0x82, 0x06, 0xA3, 0x02, 0x01, 0x03, 0x30, 0x82, 0x06, 0x63, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x06, 0x54, 0x04, 0x82, 0x06, 0x50, 0x30, 0x82, 0x06, 0x4C, 0x30, 0x82, 0x03, 0x8D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0x7E, 0x04, 0x82, 0x03, 0x7A, 0x30, 0x82, 0x03, 0x76, 0x30, 0x82, 0x03, 0x72, 0x06, 0x0B, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x0A, 0x01, 0x02, 0xA0, 0x82, 0x02, 0xB6, 0x30, 0x82, 0x02, 0xB2, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 
			0x0E, 0x04, 0x08, 0x67, 0xFE, 0x3A, 0x52, 0x75, 0xF3, 0x82, 0x1F, 0x02, 0x02, 0x07, 0xD0, 0x04, 0x82, 0x02, 0x90, 0x31, 0x6B, 0x00, 0xFA, 0x73, 0xE6, 0x8D, 0x3D, 0x62, 0x93, 0x41, 0xA1, 0x44, 0x04, 0x17, 0x8D, 0x66, 0x7A, 0x75, 0x14, 0x89, 0xA8, 0xD1, 0x4D, 0x2A, 0xD7, 0x20, 0x27, 0x71, 0x58, 0x81, 0x16, 0xB5, 0xA6, 0x41, 0x75, 0x92, 0xB2, 0xF4, 0x0C, 0xAA, 0x9B, 0x00, 0x46, 0x85, 0x85, 0x3B, 0x09, 0x2A, 0x62, 0x33, 0x3F, 0x3D, 0x06, 0xC7, 0xE7, 0x16, 0x0C, 0xA7, 0x1D, 0x9C, 0xDA, 0x9D, 0xD3, 0xC9, 0x05, 0x60, 0xA5, 0xBE, 0xF0, 0x07, 0xD5, 0xA9, 0x4F, 0x8A, 0x80, 0xF8, 0x55, 0x7B, 0x7B, 0x3C, 
			0xA0, 0x7C, 0x29, 0x29, 0xAB, 0xB1, 0xE1, 0x5A, 0x25, 0xE3, 0x23, 0x6A, 0x56, 0x98, 0x37, 0x68, 0xAF, 0x9C, 0x87, 0xBB, 0x21, 0x6E, 0x68, 0xBE, 0xAE, 0x65, 0x0C, 0x41, 0x8F, 0x5C, 0x3A, 0xB8, 0xB1, 0x9D, 0x42, 0x37, 0xE4, 0xA0, 0x37, 0xA6, 0xB8, 0xAC, 0x85, 0xD7, 0x85, 0x27, 0x68, 0xD0, 0xB6, 0x3D, 0xC7, 0x39, 0x92, 0x41, 0x46, 0x24, 0xDD, 0x08, 0x57, 0x22, 0x6A, 0xC0, 0xB7, 0xAD, 0x52, 0xC6, 0x7F, 0xE5, 0x74, 0x6A, 0x5E, 0x28, 0xA3, 0x85, 0xBD, 0xE8, 0xAD, 0x5D, 0xA3, 0x55, 0xE6, 0x63, 0x15, 0x56, 0x7B, 0x01, 0x26, 0x68, 0x5F, 0x11, 0xA3, 0x12, 0x37, 0x02, 0xA5, 0xD0, 0xB7, 0x73, 0x0C, 0x7C, 
			0x97, 0xE1, 0xC6, 0x2F, 0x98, 0x82, 0x67, 0x2F, 0x5F, 0x3F, 0xBE, 0x32, 0x16, 0x25, 0x9D, 0x51, 0x48, 0x32, 0xCB, 0x42, 0xD1, 0x31, 0x07, 0xBE, 0x5D, 0xF8, 0xCD, 0x2C, 0x38, 0x0A, 0x33, 0x3B, 0x7B, 0x04, 0x84, 0xAE, 0x9C, 0xA7, 0x6B, 0x36, 0x39, 0x12, 0x87, 0x9D, 0x5B, 0x56, 0x00, 0x44, 0x11, 0xB1, 0xE2, 0x78, 0x14, 0x60, 0xF3, 0xE4, 0x1A, 0x08, 0x14, 0xC0, 0x9E, 0x49, 0x9F, 0xE0, 0x4C, 0xEC, 0x95, 0x15, 0x18, 0x48, 0x0E, 0xB9, 0x0B, 0x3A, 0xFE, 0x45, 0xB0, 0x2D, 0x0D, 0x4F, 0x94, 0x5A, 0x3C, 0x43, 0xB7, 0x40, 0x8E, 0x7B, 0xA2, 0x8E, 0x23, 0x9F, 0x75, 0x97, 0xE7, 0x21, 0x0D, 0xEB, 0xA3, 0x9D, 
			0x6C, 0xC0, 0xDC, 0x73, 0xED, 0x15, 0x98, 0xE3, 0xE8, 0x32, 0x2C, 0x12, 0x92, 0x45, 0x25, 0x45, 0x76, 0x18, 0xF5, 0x97, 0x7F, 0xAC, 0xCE, 0xCF, 0x23, 0xF7, 0xD1, 0xCF, 0x06, 0xAB, 0x82, 0x96, 0x1F, 0xF8, 0x68, 0x4F, 0x5D, 0xE1, 0x09, 0xAA, 0xCB, 0xB3, 0x50, 0x85, 0x46, 0x72, 0x14, 0x6C, 0x49, 0x84, 0x57, 0x55, 0x00, 0x78, 0x3E, 0xD9, 0xAA, 0xBD, 0xCC, 0xE2, 0x7B, 0x18, 0xAA, 0x2E, 0x5D, 0xB9, 0x28, 0xEA, 0x8F, 0x8C, 0xFA, 0xB7, 0x06, 0x27, 0x07, 0x89, 0x41, 0x3F, 0x66, 0x1A, 0x91, 0xCA, 0xE9, 0xEC, 0x09, 0x12, 0x1C, 0x67, 0xB2, 0x2A, 0x8B, 0x4A, 0xF0, 0x97, 0x17, 0xDC, 0x3E, 0xCD, 0x9F, 0x03, 
			0x15, 0xEF, 0x03, 0x84, 0x08, 0x4A, 0x73, 0xAE, 0xE4, 0x07, 0x30, 0x27, 0xF7, 0x25, 0x69, 0x9D, 0x6C, 0x7D, 0x81, 0x88, 0xCC, 0xFA, 0xD4, 0xC7, 0x64, 0x11, 0xC0, 0xC8, 0x2C, 0x23, 0xF6, 0xFF, 0x9B, 0xE3, 0xC8, 0x89, 0x85, 0x0B, 0x3E, 0x81, 0xD8, 0x9C, 0xBD, 0xD0, 0x2D, 0xCD, 0x15, 0xA9, 0x30, 0x84, 0xF7, 0x6D, 0xEF, 0x62, 0x3B, 0xA7, 0x8C, 0xC2, 0x93, 0x90, 0x6F, 0x91, 0xB4, 0x8A, 0x71, 0x4E, 0x41, 0x4E, 0x5C, 0x67, 0xB5, 0x49, 0xF8, 0x56, 0x3A, 0x83, 0x03, 0x4F, 0xB1, 0xF6, 0xB7, 0x31, 0x5B, 0x68, 0x26, 0x70, 0x89, 0xB1, 0x1E, 0x67, 0x4F, 0xBA, 0xE7, 0xD9, 0xDF, 0x91, 0xD8, 0xFB, 0x8A, 0xDD, 
			0xB2, 0xD3, 0x4B, 0xBB, 0x9F, 0x5C, 0xA3, 0x04, 0x2C, 0x87, 0xBC, 0xD5, 0xBE, 0x8C, 0xD7, 0xCF, 0x9B, 0x72, 0x82, 0xA6, 0x99, 0xDA, 0xD7, 0x66, 0x48, 0xE7, 0x8F, 0xE9, 0x48, 0x56, 0x9D, 0xD2, 0xB9, 0x28, 0x84, 0x4F, 0x6A, 0x83, 0xB2, 0xB9, 0x4D, 0x91, 0x10, 0x58, 0x22, 0x4C, 0xE7, 0x9D, 0xC6, 0x0C, 0x74, 0xF4, 0x16, 0x58, 0x30, 0xB7, 0xB7, 0x96, 0x39, 0x6C, 0x5D, 0xFA, 0xB2, 0x03, 0x8C, 0x98, 0xD2, 0xC0, 0x64, 0xB8, 0x05, 0x29, 0x4F, 0xF0, 0x4C, 0x43, 0x48, 0xD3, 0xD8, 0xBD, 0xC7, 0xC1, 0xEA, 0x39, 0x2A, 0xDF, 0xD4, 0xDA, 0x79, 0x7C, 0xB9, 0x06, 0xC7, 0x10, 0x8D, 0x8B, 0xF1, 0xA8, 0x8E, 0x44, 
			0x9E, 0x99, 0xFF, 0x81, 0x84, 0x8F, 0xD0, 0x38, 0xE1, 0xF0, 0x5A, 0x12, 0x5F, 0xC5, 0xA6, 0xED, 0x6D, 0xEE, 0xE7, 0x69, 0xC0, 0xA2, 0xB4, 0x13, 0xCA, 0x7A, 0x5D, 0xDE, 0x88, 0x75, 0xE7, 0xE2, 0x6D, 0x8A, 0xEC, 0x0F, 0x88, 0x3F, 0xE2, 0xCB, 0x60, 0xF0, 0x6A, 0xEC, 0xD0, 0xF4, 0x0D, 0x11, 0xC2, 0x84, 0x19, 0x67, 0x52, 0xAD, 0xC0, 0xC0, 0x20, 0x84, 0x6D, 0x7D, 0xEA, 0xD2, 0xF9, 0x3F, 0xE5, 0x58, 0x00, 0xED, 0x24, 0xD6, 0x50, 0x9B, 0x80, 0x80, 0x0A, 0x31, 0x81, 0xA8, 0x30, 0x0D, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x02, 0x31, 0x00, 0x30, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 
			0x86, 0xF7, 0x0D, 0x01, 0x09, 0x15, 0x31, 0x06, 0x04, 0x04, 0x01, 0x00, 0x00, 0x00, 0x30, 0x17, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x14, 0x31, 0x0A, 0x1E, 0x08, 0x00, 0x4D, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x6F, 0x30, 0x69, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x01, 0x31, 0x5C, 0x1E, 0x5A, 0x00, 0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20, 0x00, 0x52, 0x00, 0x53, 0x00, 0x41, 0x00, 0x20, 0x00, 0x53, 0x00, 0x43, 0x00, 0x68, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x6E, 0x00, 0x65, 0x00, 0x6C, 
			0x00, 0x20, 0x00, 0x43, 0x00, 0x72, 0x00, 0x79, 0x00, 0x70, 0x00, 0x74, 0x00, 0x6F, 0x00, 0x67, 0x00, 0x72, 0x00, 0x61, 0x00, 0x70, 0x00, 0x68, 0x00, 0x69, 0x00, 0x63, 0x00, 0x20, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x76, 0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x72, 0x30, 0x82, 0x02, 0xB7, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x06, 0xA0, 0x82, 0x02, 0xA8, 0x30, 0x82, 0x02, 0xA4, 0x02, 0x01, 0x00, 0x30, 0x82, 0x02, 0x9D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 
			0x06, 0x30, 0x0E, 0x04, 0x08, 0xB8, 0x22, 0xEA, 0x3C, 0x70, 0x6A, 0xFC, 0x39, 0x02, 0x02, 0x07, 0xD0, 0x80, 0x82, 0x02, 0x70, 0x76, 0xBE, 0x5B, 0xD5, 0x3D, 0x05, 0xC1, 0xDB, 0x10, 0xA3, 0x02, 0xBB, 0x7F, 0x0A, 0x8B, 0x54, 0xC1, 0x7D, 0x19, 0xDA, 0x7E, 0x82, 0xDF, 0xAD, 0x6B, 0x42, 0xC2, 0x95, 0x95, 0x00, 0x6E, 0x82, 0x77, 0xD5, 0x42, 0x6E, 0x21, 0xA2, 0x95, 0xB4, 0x98, 0xF5, 0xDD, 0x18, 0x6F, 0xC4, 0xF3, 0xB6, 0x93, 0xA0, 0x6C, 0xF4, 0x34, 0x7A, 0x48, 0x72, 0x08, 0xB1, 0x28, 0x51, 0x54, 0x10, 0x7F, 0x35, 0xB2, 0xE5, 0x89, 0x5C, 0x0A, 0x14, 0x31, 0x1C, 0x9D, 0xA9, 0xE4, 0x94, 0x91, 0x28, 0x65, 
			0xC4, 0xE7, 0x5E, 0xA9, 0x37, 0x08, 0x3D, 0xB1, 0x16, 0x61, 0x9D, 0xA9, 0x44, 0x6F, 0x20, 0x0C, 0x55, 0xD7, 0xCC, 0x48, 0x82, 0x13, 0x5D, 0xE1, 0xBD, 0x9D, 0xCE, 0x64, 0x28, 0x6D, 0x69, 0x4E, 0x08, 0x53, 0x09, 0xE0, 0xCC, 0xA8, 0x79, 0x04, 0xCF, 0xFA, 0x35, 0x1C, 0xA6, 0x70, 0x37, 0x64, 0x70, 0x74, 0xF8, 0xD0, 0xC4, 0x34, 0x0F, 0x71, 0xEF, 0x57, 0xC2, 0x43, 0x7D, 0xFA, 0xE5, 0x1B, 0x8C, 0x15, 0xA5, 0x08, 0x60, 0x78, 0xAF, 0xDA, 0x36, 0xDF, 0x79, 0x2D, 0xD7, 0x54, 0x35, 0xD7, 0x8D, 0x99, 0xD5, 0x81, 0xEC, 0x6D, 0x9F, 0x2D, 0x5E, 0xF8, 0x48, 0x85, 0x50, 0x20, 0x7D, 0xBB, 0x16, 0x4E, 0x39, 0x64, 
			0xB7, 0xBC, 0xED, 0xA9, 0x6A, 0x7A, 0x06, 0x09, 0x6B, 0xBC, 0x2C, 0x5A, 0xE1, 0x4F, 0xD4, 0xA9, 0x82, 0x83, 0x5B, 0xBD, 0xCE, 0x14, 0x31, 0x89, 0x66, 0xB3, 0x9C, 0x31, 0x23, 0x00, 0x4B, 0x02, 0x34, 0x85, 0x30, 0x39, 0x77, 0x80, 0x5D, 0x72, 0x0A, 0xCE, 0x43, 0x2A, 0x1F, 0x02, 0x09, 0xAB, 0x2D, 0x46, 0x3A, 0x1C, 0xD2, 0x7B, 0xF6, 0x02, 0x92, 0xCA, 0xDA, 0x26, 0x0C, 0xF8, 0xE2, 0x67, 0x7E, 0xE2, 0x55, 0xB1, 0x3F, 0x6A, 0x06, 0x65, 0x6D, 0x74, 0x98, 0x59, 0xE2, 0x8A, 0x1E, 0x61, 0x03, 0x4D, 0xFC, 0x68, 0x31, 0x6A, 0xE7, 0xCF, 0x52, 0x88, 0x8E, 0x06, 0x97, 0x77, 0xB3, 0x20, 0x7E, 0x09, 0x5D, 0x3B, 
			0xAF, 0x56, 0xF4, 0xE8, 0x4C, 0x69, 0x09, 0xB9, 0x80, 0x38, 0xDC, 0x66, 0x2E, 0x06, 0xF6, 0xCB, 0x1F, 0x1B, 0xAD, 0x51, 0xFF, 0xFD, 0x38, 0x8D, 0x03, 0x90, 0xCF, 0x31, 0x01, 0x30, 0xEA, 0x48, 0x4C, 0xBB, 0x40, 0x87, 0x1D, 0x97, 0x6A, 0x56, 0x4C, 0xED, 0x07, 0x23, 0x45, 0x50, 0x2F, 0x56, 0xC9, 0x90, 0x79, 0x09, 0xC5, 0x45, 0xB9, 0xAD, 0x58, 0x2B, 0x4C, 0xA3, 0x01, 0xE0, 0x2D, 0xE5, 0x30, 0xBC, 0x54, 0xEC, 0x65, 0xB4, 0x79, 0x22, 0x7D, 0x15, 0xF6, 0x28, 0xCD, 0x84, 0x7E, 0x27, 0x95, 0xA1, 0xC7, 0x82, 0x6D, 0xFB, 0xDF, 0x03, 0xD9, 0x14, 0xFE, 0x0A, 0x06, 0x6F, 0x14, 0xFF, 0x8A, 0x27, 0x80, 0x36, 
			0xDC, 0xBA, 0xAE, 0xDD, 0x44, 0x15, 0xA5, 0x6E, 0x64, 0x73, 0xBD, 0xFB, 0xAE, 0x6D, 0x6F, 0x42, 0x96, 0xDF, 0x90, 0xE5, 0x6A, 0x9B, 0x05, 0xAE, 0xD5, 0x0A, 0x22, 0x88, 0xD6, 0x5D, 0x4C, 0x7B, 0xB1, 0x3A, 0xFC, 0x0C, 0x32, 0x02, 0xB1, 0x18, 0x0D, 0xAF, 0xE0, 0xFE, 0x7E, 0x07, 0x96, 0x85, 0xBB, 0xC8, 0x21, 0x68, 0x12, 0xD4, 0xC8, 0xBF, 0x91, 0x47, 0xE2, 0xF3, 0xA5, 0xA3, 0x86, 0xE6, 0x30, 0x42, 0xF5, 0xA9, 0xB9, 0x48, 0xCB, 0x18, 0xE6, 0x64, 0x3B, 0xE0, 0x8E, 0xC3, 0x03, 0x45, 0xA0, 0xED, 0x1A, 0x09, 0xFF, 0xB3, 0x99, 0x14, 0x5F, 0xDA, 0x90, 0x58, 0x61, 0x8E, 0xF7, 0x0A, 0x00, 0xC7, 0x44, 0xE7, 
			0x73, 0x78, 0xC4, 0x8B, 0x39, 0xCE, 0x70, 0x0E, 0x24, 0x03, 0x95, 0x94, 0x73, 0x76, 0x10, 0x7E, 0x4C, 0xFF, 0xCA, 0x49, 0x93, 0x89, 0xD4, 0x3E, 0x1A, 0x88, 0xCC, 0x48, 0xA7, 0x78, 0x2F, 0x83, 0x4F, 0x6C, 0x33, 0x55, 0xDD, 0x7F, 0x7D, 0x4D, 0xE5, 0xCD, 0x9C, 0x3D, 0x04, 0x1E, 0xC1, 0x9B, 0x6D, 0x7E, 0x7A, 0xAC, 0x93, 0x5E, 0x2B, 0xC3, 0x85, 0x36, 0x07, 0x66, 0xE8, 0xC9, 0xC0, 0xD1, 0x54, 0xF4, 0x4C, 0x6A, 0x02, 0x24, 0x9A, 0x7D, 0x10, 0xD9, 0x79, 0x94, 0x00, 0x64, 0x63, 0x36, 0xDC, 0x35, 0x0C, 0x8F, 0x79, 0xBA, 0xC7, 0x10, 0x76, 0xF8, 0x4A, 0xD3, 0x69, 0x95, 0x23, 0x89, 0x66, 0xC4, 0x5A, 0xE7, 
			0xCE, 0x21, 0xBC, 0xCB, 0xF2, 0x4F, 0x92, 0x33, 0xE7, 0x89, 0xD6, 0x23, 0xF7, 0x67, 0x5B, 0x20, 0xD9, 0xDA, 0x1A, 0xD1, 0xF6, 0x9E, 0x01, 0x83, 0x51, 0xAF, 0x35, 0x43, 0xDD, 0x3A, 0xAB, 0xCA, 0x0E, 0xED, 0x2E, 0x4D, 0x1E, 0x91, 0xCF, 0x2E, 0xA9, 0x4D, 0x08, 0xD9, 0x48, 0x30, 0x37, 0x30, 0x1F, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x04, 0x14, 0xA2, 0xED, 0x05, 0x50, 0x89, 0x91, 0x1D, 0xEB, 0xF6, 0x57, 0x66, 0xAF, 0x70, 0x15, 0xDD, 0x1A, 0xA1, 0x94, 0xB7, 0xB2, 0x04, 0x14, 0x09, 0xE4, 0x0B, 0xEC, 0x1D, 0x93, 0x3E, 0x32, 0x94, 0x6A, 0x95, 0x36, 0xDD, 0xBA, 0x93, 0x9D, 0x75, 0xB6, 
			0x3E, 0xF5 };

		static public byte[] farscape_nopwd_pfx = { 0x30, 0x82, 0x06, 0xA3, 0x02, 0x01, 0x03, 0x30, 0x82, 0x06, 0x63, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x06, 0x54, 0x04, 0x82, 0x06, 0x50, 0x30, 0x82, 0x06, 0x4C, 0x30, 0x82, 0x03, 0x8D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0x7E, 0x04, 0x82, 0x03, 0x7A, 0x30, 0x82, 0x03, 0x76, 0x30, 0x82, 0x03, 0x72, 0x06, 0x0B, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x0A, 0x01, 0x02, 0xA0, 0x82, 0x02, 0xB6, 0x30, 0x82, 0x02, 0xB2, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 
			0x0E, 0x04, 0x08, 0x31, 0xB9, 0x22, 0x7A, 0x73, 0xB6, 0x67, 0x3E, 0x02, 0x02, 0x07, 0xD0, 0x04, 0x82, 0x02, 0x90, 0x05, 0x3F, 0x9B, 0x6F, 0x4D, 0xE2, 0x97, 0xC0, 0x71, 0x61, 0xDC, 0x39, 0x33, 0x9B, 0x45, 0x36, 0xD1, 0xC2, 0xC1, 0x2E, 0xE3, 0x22, 0x88, 0xE2, 0x54, 0x18, 0xE8, 0xC9, 0x0E, 0xA7, 0xBB, 0x1B, 0xC6, 0xC8, 0x32, 0xD9, 0x47, 0x64, 0x40, 0xC2, 0x40, 0xDC, 0x34, 0xB5, 0x34, 0x5D, 0x8A, 0x56, 0xD9, 0xF6, 0x0A, 0x03, 0x93, 0x5D, 0xE5, 0x04, 0xDC, 0x5B, 0xBA, 0x49, 0x22, 0x0A, 0x51, 0x33, 0xFF, 0xF0, 0xAF, 0x5D, 0x1F, 0x97, 0x6A, 0x11, 0x1C, 0x6B, 0x1A, 0xCF, 0x2E, 0x41, 0xA1, 0xD0, 0x31, 
			0xC2, 0x2D, 0xDD, 0x83, 0xAA, 0x21, 0x0C, 0x0E, 0x78, 0xEE, 0x9C, 0x25, 0x74, 0xC5, 0x4F, 0xE4, 0x94, 0x84, 0xA8, 0xD9, 0x2F, 0x96, 0xF5, 0x06, 0x05, 0xAE, 0x99, 0xBF, 0x8B, 0xD6, 0x67, 0x5E, 0xCB, 0x61, 0x03, 0xCC, 0x5A, 0x5F, 0xAB, 0x82, 0x55, 0xB1, 0x8D, 0xCD, 0xFE, 0x1C, 0x25, 0x48, 0xA7, 0x1D, 0xFF, 0x2E, 0xC0, 0x23, 0x80, 0xF7, 0xE4, 0x22, 0x68, 0x07, 0xFF, 0x58, 0xA5, 0xAA, 0x71, 0x7A, 0xAB, 0x48, 0x2D, 0xE6, 0xDF, 0xB5, 0x3C, 0x90, 0x15, 0xE3, 0x55, 0x4A, 0xB4, 0x37, 0xFE, 0x7F, 0xE1, 0x5B, 0x0C, 0xF1, 0x01, 0x4C, 0x60, 0x2F, 0x6F, 0x59, 0x09, 0x2B, 0x96, 0xDC, 0xE2, 0x2C, 0xF0, 0xB9, 
			0xF3, 0x3E, 0x46, 0x5B, 0x68, 0xA9, 0xBB, 0x42, 0x8B, 0xAB, 0xA9, 0x68, 0x56, 0xF9, 0xB2, 0x2E, 0x93, 0xDD, 0xE9, 0xBB, 0x70, 0x9E, 0x2E, 0x48, 0xB9, 0xDB, 0x1C, 0x95, 0x0F, 0x67, 0xD4, 0x13, 0x02, 0x62, 0xE0, 0xFA, 0x18, 0x48, 0xAE, 0x31, 0xB6, 0x1F, 0x68, 0x7D, 0xB2, 0x16, 0x61, 0xCD, 0x04, 0x91, 0x50, 0xBF, 0x35, 0xBF, 0x76, 0xA3, 0x5B, 0x76, 0xFE, 0x3F, 0xAB, 0xB2, 0x59, 0x8B, 0xD0, 0xB7, 0xC6, 0x36, 0x0E, 0x2C, 0x31, 0x48, 0xFB, 0x69, 0x6F, 0x90, 0x37, 0x3F, 0xE1, 0x53, 0x36, 0x5A, 0x60, 0x53, 0x93, 0x46, 0xC4, 0x31, 0x92, 0x3B, 0x11, 0x9F, 0x67, 0xC3, 0xD0, 0x2E, 0x9F, 0x7D, 0xA8, 0xBE, 
			0xA3, 0xB2, 0xCF, 0x60, 0xA3, 0xCE, 0x9F, 0x4B, 0x72, 0xCD, 0x44, 0x26, 0x4C, 0x66, 0xF8, 0x75, 0x80, 0xFC, 0x23, 0xBC, 0xA1, 0x3A, 0xCA, 0xC9, 0xE7, 0x50, 0xA3, 0x79, 0x21, 0x2B, 0x2D, 0x09, 0x8C, 0x45, 0x89, 0xB6, 0xAF, 0x66, 0x3E, 0xF7, 0xFD, 0xA5, 0x69, 0x96, 0xB4, 0x65, 0xB5, 0xFE, 0x35, 0x1F, 0x80, 0xA7, 0x41, 0x90, 0xBA, 0x92, 0x8D, 0x3B, 0xC0, 0x37, 0xDE, 0x95, 0xA8, 0x0D, 0xF1, 0x1A, 0x9F, 0xD2, 0x70, 0xED, 0x38, 0x1E, 0xA2, 0xF1, 0x2B, 0x63, 0x62, 0xC5, 0xAE, 0x5D, 0x0F, 0xFC, 0x80, 0xFA, 0x0E, 0xE4, 0xE7, 0x6C, 0x62, 0x3B, 0x19, 0xBB, 0xA8, 0xE5, 0x1D, 0x3E, 0x06, 0x30, 0x0B, 0xE1, 
			0xCF, 0xB6, 0xB4, 0x87, 0x96, 0xA2, 0x5E, 0xF8, 0x0F, 0x13, 0xAE, 0x04, 0xAF, 0xB2, 0x6C, 0x9E, 0xA0, 0x28, 0x1C, 0x46, 0xE5, 0xA8, 0x25, 0x62, 0x51, 0x95, 0xB0, 0x70, 0x60, 0xB6, 0xD9, 0xBB, 0xE3, 0xD1, 0xF0, 0x1D, 0x25, 0xBD, 0x93, 0x5E, 0xB6, 0x47, 0x50, 0xCD, 0x77, 0x7A, 0xFF, 0xC5, 0xFF, 0x4A, 0x7A, 0x9A, 0x27, 0x22, 0xEB, 0x7C, 0x12, 0xE5, 0x59, 0x1F, 0x60, 0xEA, 0xC3, 0x93, 0x4D, 0x28, 0x49, 0x2D, 0xF9, 0xC0, 0x13, 0x12, 0x89, 0x96, 0xED, 0x78, 0xB0, 0x1C, 0x82, 0xDE, 0xEE, 0x40, 0xDE, 0x68, 0x2B, 0x45, 0x16, 0xBE, 0xBF, 0xD5, 0x85, 0x6A, 0xDB, 0xD9, 0x1E, 0xEE, 0xFA, 0x6C, 0x95, 0x19, 
			0xF3, 0x76, 0x61, 0x72, 0x21, 0x69, 0x77, 0x18, 0x2C, 0xFA, 0x99, 0x7A, 0xD7, 0x58, 0xC4, 0xD6, 0x1D, 0x8B, 0xE8, 0x0B, 0xEC, 0x0F, 0x0F, 0xFA, 0xCE, 0xE2, 0x6F, 0xB1, 0xF5, 0x4F, 0xC3, 0xF7, 0x4A, 0xE4, 0x79, 0xB0, 0xFC, 0x62, 0x88, 0xC0, 0x49, 0xEC, 0xDB, 0xC8, 0xCD, 0xBE, 0x25, 0x00, 0x68, 0xB6, 0x5E, 0x89, 0x78, 0xE6, 0x92, 0xA5, 0x5D, 0x55, 0xA7, 0xAD, 0xFF, 0x3D, 0xC7, 0xF9, 0x95, 0x8D, 0xCF, 0x6E, 0x37, 0x1D, 0x79, 0x74, 0xE1, 0xDE, 0x22, 0x07, 0x6B, 0xE6, 0xB7, 0x7D, 0xD1, 0x0F, 0xB6, 0xA4, 0x3F, 0x0F, 0x31, 0x81, 0x09, 0xAD, 0xFD, 0x5F, 0xA4, 0xF4, 0x8F, 0x3C, 0x02, 0xB8, 0xB0, 0x04, 
			0x70, 0x44, 0x2C, 0x73, 0x42, 0xEE, 0xFF, 0xBA, 0x45, 0x50, 0xC0, 0x95, 0xFF, 0x62, 0x14, 0x91, 0x23, 0xF2, 0x8A, 0x65, 0x40, 0x20, 0xEB, 0x4B, 0x7B, 0x66, 0xF2, 0xC2, 0xC8, 0xD7, 0x16, 0x93, 0x0A, 0xBD, 0x5C, 0xCC, 0x11, 0x38, 0xEA, 0x90, 0x9C, 0x37, 0xDA, 0xB2, 0x80, 0xBF, 0x5C, 0x41, 0xC8, 0x3B, 0x16, 0x81, 0x83, 0xF7, 0xE4, 0x16, 0x12, 0x6C, 0x5F, 0x05, 0xBE, 0x2B, 0x04, 0x62, 0x36, 0x13, 0x8F, 0xF1, 0xC2, 0x5A, 0xCB, 0xFB, 0x26, 0x04, 0xE0, 0x31, 0x81, 0xA8, 0x30, 0x0D, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x02, 0x31, 0x00, 0x30, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 
			0x86, 0xF7, 0x0D, 0x01, 0x09, 0x15, 0x31, 0x06, 0x04, 0x04, 0x01, 0x00, 0x00, 0x00, 0x30, 0x17, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x14, 0x31, 0x0A, 0x1E, 0x08, 0x00, 0x4D, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x6F, 0x30, 0x69, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x01, 0x31, 0x5C, 0x1E, 0x5A, 0x00, 0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20, 0x00, 0x52, 0x00, 0x53, 0x00, 0x41, 0x00, 0x20, 0x00, 0x53, 0x00, 0x43, 0x00, 0x68, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x6E, 0x00, 0x65, 0x00, 0x6C, 
			0x00, 0x20, 0x00, 0x43, 0x00, 0x72, 0x00, 0x79, 0x00, 0x70, 0x00, 0x74, 0x00, 0x6F, 0x00, 0x67, 0x00, 0x72, 0x00, 0x61, 0x00, 0x70, 0x00, 0x68, 0x00, 0x69, 0x00, 0x63, 0x00, 0x20, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x76, 0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x72, 0x30, 0x82, 0x02, 0xB7, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x06, 0xA0, 0x82, 0x02, 0xA8, 0x30, 0x82, 0x02, 0xA4, 0x02, 0x01, 0x00, 0x30, 0x82, 0x02, 0x9D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 
			0x06, 0x30, 0x0E, 0x04, 0x08, 0x37, 0xAE, 0x94, 0x2A, 0x4C, 0x78, 0xA2, 0x9A, 0x02, 0x02, 0x07, 0xD0, 0x80, 0x82, 0x02, 0x70, 0x49, 0xB9, 0xA3, 0x6E, 0xC7, 0x96, 0xCF, 0x92, 0x12, 0x43, 0x69, 0x57, 0xAD, 0x4B, 0x88, 0xA8, 0x3F, 0xEA, 0x25, 0xB6, 0xE4, 0x16, 0x74, 0x4E, 0xF5, 0xF8, 0xF2, 0xEC, 0xC0, 0xB7, 0xC2, 0x6A, 0x6E, 0xC0, 0x67, 0x5A, 0x5D, 0xFE, 0x0A, 0x7C, 0xBD, 0x06, 0xFF, 0x2F, 0x34, 0xFD, 0xE4, 0x06, 0x70, 0x23, 0xA3, 0x28, 0x27, 0xCA, 0x91, 0xD0, 0xC7, 0xA1, 0x08, 0x4F, 0x78, 0x0E, 0x89, 0xED, 0x29, 0x8F, 0xD6, 0x8E, 0x1C, 0xE0, 0x30, 0x08, 0x77, 0xA0, 0x3F, 0x18, 0xF1, 0x81, 0xD5, 
			0x73, 0xD7, 0x1A, 0xCA, 0xD4, 0x6D, 0x56, 0x7D, 0xFD, 0x30, 0xB5, 0xA0, 0x5D, 0x59, 0x82, 0xB9, 0xF7, 0x02, 0x19, 0x83, 0x68, 0x19, 0x08, 0x5E, 0x26, 0xCF, 0x06, 0xFA, 0xA0, 0xB4, 0x85, 0x95, 0x10, 0x6F, 0x91, 0x82, 0x89, 0xE8, 0x46, 0xEE, 0x51, 0xEB, 0x2A, 0x45, 0xAC, 0x93, 0x87, 0x48, 0x8C, 0xB6, 0x02, 0xB4, 0x4D, 0xC6, 0xFC, 0x51, 0x4C, 0x75, 0x9D, 0x5A, 0xE7, 0x46, 0x5B, 0x0A, 0x9D, 0x75, 0xA3, 0x0C, 0xB2, 0x54, 0x2A, 0x3E, 0x3A, 0x25, 0xA3, 0x75, 0x66, 0x52, 0x61, 0x7A, 0x78, 0xED, 0xDD, 0x7E, 0xF2, 0x4A, 0xA6, 0xB6, 0x3D, 0xEA, 0x62, 0xE4, 0x68, 0x95, 0x74, 0x3D, 0x45, 0xC1, 0x6E, 0x6B, 
			0xB6, 0x6E, 0x8F, 0x97, 0x39, 0xB5, 0x4F, 0xAA, 0x8E, 0xBB, 0x55, 0x10, 0x19, 0xCB, 0x66, 0xA0, 0xBF, 0xAE, 0x8B, 0xE6, 0xBC, 0x92, 0x8D, 0x2D, 0xC1, 0x83, 0x87, 0x53, 0x81, 0x32, 0x3B, 0x8E, 0x80, 0x76, 0xF9, 0xDE, 0x60, 0x8F, 0x99, 0x02, 0x4F, 0x97, 0x73, 0x3D, 0xE3, 0xC7, 0xBA, 0xBD, 0x4C, 0x3F, 0x8A, 0x9B, 0xE3, 0xFE, 0x24, 0xC3, 0x3E, 0xDE, 0x02, 0x0F, 0x46, 0x84, 0x79, 0xDF, 0x5E, 0xC9, 0xA3, 0x7C, 0x58, 0x62, 0xFC, 0x1D, 0x9F, 0x5E, 0x9A, 0xDB, 0x3C, 0x45, 0x96, 0x91, 0xFD, 0xD9, 0xD0, 0xE7, 0x7F, 0x72, 0xBA, 0x2D, 0xC5, 0x3A, 0x54, 0xBC, 0xA0, 0xAE, 0xAA, 0xFF, 0xE9, 0x18, 0x0C, 0x1B, 
			0x9A, 0xD4, 0xDA, 0x82, 0xBF, 0x51, 0x23, 0xB3, 0x6E, 0xEF, 0xDB, 0x85, 0xE5, 0xBF, 0x02, 0xCC, 0xFB, 0x79, 0xA6, 0x45, 0x86, 0xDC, 0xDF, 0xF0, 0x2C, 0x15, 0x0B, 0xD1, 0xE5, 0x80, 0xBB, 0x3F, 0x65, 0x94, 0xE5, 0xAB, 0x76, 0xE4, 0xA5, 0x92, 0x7D, 0x0E, 0x8C, 0xC0, 0x92, 0x83, 0x40, 0x9D, 0x2F, 0xBD, 0x30, 0xE1, 0x7B, 0xB5, 0x91, 0xB2, 0x5E, 0xD9, 0xC6, 0xB7, 0xA4, 0x30, 0x06, 0x18, 0xED, 0x33, 0x95, 0x7B, 0xA6, 0xE3, 0xE5, 0xC0, 0x4B, 0xF5, 0x0B, 0x6A, 0x3A, 0xF5, 0xAC, 0x77, 0x22, 0xC0, 0x84, 0x3C, 0x5B, 0xE5, 0x55, 0xD5, 0xDC, 0x7E, 0xFE, 0x08, 0x02, 0x37, 0x69, 0x52, 0xB8, 0x44, 0x29, 0x16, 
			0xB5, 0xE8, 0x8A, 0xA4, 0xAC, 0x24, 0x58, 0xC3, 0x53, 0xAC, 0x37, 0xE2, 0xD4, 0x0F, 0x21, 0xC1, 0x54, 0x62, 0x28, 0xCA, 0xA3, 0x8C, 0x01, 0x26, 0x97, 0xFF, 0xAD, 0x0E, 0x5F, 0xB1, 0x86, 0x96, 0xD1, 0xFA, 0xE5, 0x9F, 0x38, 0x42, 0x4D, 0x32, 0xEB, 0xC8, 0x4B, 0x4A, 0x01, 0x91, 0x5C, 0xCE, 0xC8, 0x89, 0x0A, 0x7C, 0x32, 0x6D, 0x08, 0x3E, 0x7D, 0xB0, 0x3D, 0x16, 0x99, 0x52, 0xB0, 0xE0, 0xBE, 0xFF, 0x42, 0x61, 0xC3, 0x56, 0xE1, 0x9A, 0xA3, 0xFB, 0x72, 0xBB, 0x3B, 0x4C, 0xA3, 0xFC, 0x5E, 0xFE, 0xC7, 0xF2, 0xBB, 0x17, 0x96, 0x00, 0xB6, 0x02, 0xD1, 0x58, 0xF0, 0xDA, 0x63, 0xD2, 0x4C, 0x91, 0xDF, 0xFA, 
			0xB5, 0xAF, 0x1E, 0xDA, 0xD7, 0x02, 0x85, 0xFE, 0x80, 0x94, 0x77, 0x92, 0x84, 0x9A, 0x2F, 0x1C, 0xC2, 0x71, 0xA7, 0x3F, 0xFA, 0x00, 0xFC, 0x7E, 0x4B, 0xE2, 0xD2, 0x7B, 0xC8, 0xB9, 0x26, 0xEC, 0xD4, 0x7A, 0x3D, 0x6F, 0x89, 0xB4, 0x22, 0x2F, 0xE9, 0x41, 0xA9, 0x97, 0x8C, 0x76, 0xCE, 0xCD, 0xA6, 0x94, 0xA9, 0x1D, 0x25, 0x7C, 0x4D, 0xCF, 0x2E, 0x51, 0x59, 0xE9, 0xE3, 0xDB, 0x84, 0x28, 0x2E, 0x31, 0x24, 0xF9, 0xA7, 0xC0, 0xA7, 0x77, 0xD0, 0xB5, 0x19, 0x1C, 0xC9, 0x22, 0x28, 0x94, 0x39, 0xF5, 0xC3, 0xAA, 0x78, 0x3A, 0xE6, 0x1D, 0xB3, 0xCA, 0x95, 0x7F, 0x7D, 0xBD, 0xFA, 0x7F, 0xCD, 0x09, 0xA5, 0x77, 
			0x8E, 0xC8, 0xEB, 0x03, 0x26, 0xAF, 0x38, 0x5A, 0x9A, 0xFB, 0xDC, 0x90, 0xBD, 0xD7, 0x46, 0xA7, 0xB4, 0x71, 0x8F, 0xF7, 0x66, 0x4A, 0x07, 0x66, 0xE4, 0xD7, 0x3E, 0xC4, 0xD4, 0x2B, 0x15, 0x1F, 0xC8, 0x9C, 0x3A, 0x47, 0x5E, 0x6F, 0x84, 0xE3, 0x02, 0x62, 0x05, 0x86, 0x63, 0x30, 0x37, 0x30, 0x1F, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x04, 0x14, 0x62, 0x54, 0xAE, 0x53, 0x8C, 0x33, 0xEC, 0x3E, 0x2D, 0x73, 0xE6, 0xEB, 0x9A, 0xDD, 0x31, 0xEE, 0x06, 0x83, 0x4B, 0xBA, 0x04, 0x14, 0x60, 0x9B, 0x73, 0xDD, 0x3F, 0x8F, 0x2E, 0x52, 0x1C, 0x4C, 0xB9, 0x8E, 0x7A, 0xC0, 0xCD, 0x52, 0xB4, 0xBA, 
			0xBD, 0x8C };

		static public byte[] farscape_path_pfx = { 0x30, 0x82, 0x0B, 0x53, 0x02, 0x01, 0x03, 0x30, 0x82, 0x0B, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x0B, 0x04, 0x04, 0x82, 0x0B, 0x00, 0x30, 0x82, 0x0A, 0xFC, 0x30, 0x82, 0x03, 0x8D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0x7E, 0x04, 0x82, 0x03, 0x7A, 0x30, 0x82, 0x03, 0x76, 0x30, 0x82, 0x03, 0x72, 0x06, 0x0B, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x0A, 0x01, 0x02, 0xA0, 0x82, 0x02, 0xB6, 0x30, 0x82, 0x02, 0xB2, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 
			0x0E, 0x04, 0x08, 0x38, 0xE0, 0x5B, 0xD2, 0xF6, 0x07, 0x19, 0xE1, 0x02, 0x02, 0x07, 0xD0, 0x04, 0x82, 0x02, 0x90, 0x1B, 0xCB, 0x34, 0x7D, 0x3A, 0x5F, 0x9A, 0xA9, 0x87, 0xD7, 0x17, 0xCF, 0x5D, 0xAD, 0x94, 0xB8, 0x36, 0xCD, 0xEC, 0x1B, 0xDF, 0x58, 0xA9, 0x99, 0x18, 0x75, 0xC8, 0x9D, 0x4F, 0x2D, 0xED, 0x82, 0x10, 0x18, 0xFC, 0x9A, 0x89, 0x3D, 0xAC, 0x3C, 0xC3, 0x26, 0x5F, 0x2B, 0x72, 0x48, 0xD5, 0xA3, 0xFA, 0x12, 0x57, 0x7E, 0x60, 0x05, 0xA1, 0x2B, 0x20, 0xAF, 0xB4, 0x3B, 0x1A, 0x38, 0xA4, 0xED, 0x0A, 0x25, 0xA3, 0x4B, 0xBF, 0xAB, 0x64, 0x26, 0xE9, 0xB5, 0x6E, 0xE1, 0x94, 0x44, 0xFA, 0x13, 0x14, 
			0x0B, 0x49, 0x76, 0xCB, 0xB8, 0x68, 0xD0, 0x64, 0xEE, 0xB6, 0xBC, 0xA4, 0xD1, 0x30, 0x8C, 0xEF, 0xC9, 0xA7, 0x07, 0x6F, 0x0E, 0xAA, 0xF7, 0x48, 0xA7, 0xE1, 0x7A, 0x3B, 0x85, 0x75, 0x79, 0x65, 0x51, 0xE9, 0xA7, 0x1A, 0x6C, 0xE3, 0x08, 0x99, 0x48, 0x4B, 0xC5, 0x66, 0xC5, 0x73, 0xB3, 0x5B, 0x99, 0x60, 0xDF, 0x8E, 0x3D, 0xB8, 0x04, 0x09, 0xA2, 0x2D, 0x70, 0xBF, 0x6F, 0x00, 0x9D, 0x1B, 0x3F, 0xE0, 0xE7, 0xDE, 0x05, 0x71, 0x54, 0x3F, 0x2D, 0x29, 0x3A, 0xFD, 0x44, 0xE6, 0x7F, 0x30, 0x9B, 0x71, 0x43, 0x55, 0x78, 0x40, 0x74, 0x6D, 0x11, 0x64, 0xBB, 0x04, 0xB7, 0x36, 0xE0, 0xA4, 0x45, 0x74, 0xBC, 0xC4, 
			0xC2, 0x40, 0xA5, 0xF1, 0x62, 0x9D, 0x59, 0x18, 0xAD, 0x3D, 0xD6, 0xDB, 0x9A, 0xF3, 0x34, 0x36, 0x8B, 0x0C, 0xAF, 0xC8, 0x72, 0x82, 0x9F, 0x34, 0xC2, 0xE0, 0x96, 0xB4, 0x57, 0xB9, 0x02, 0x8E, 0x00, 0x3F, 0xC6, 0x8A, 0x62, 0x02, 0xFE, 0x15, 0xCB, 0xAE, 0x49, 0xD7, 0x06, 0x4F, 0xFC, 0xF3, 0x93, 0xC1, 0x0F, 0x78, 0x8F, 0xD6, 0xEE, 0xD7, 0xB4, 0x6F, 0xA4, 0xD8, 0x62, 0x13, 0x27, 0x81, 0x58, 0xEC, 0x04, 0x44, 0xB6, 0x47, 0xBB, 0xAA, 0xEA, 0x51, 0x44, 0xF1, 0xD3, 0x2D, 0x7E, 0x79, 0xED, 0xF0, 0x3D, 0x05, 0xA0, 0xC5, 0x61, 0x05, 0x2A, 0xE3, 0x08, 0xF1, 0xD9, 0xFC, 0x85, 0x18, 0xF7, 0x49, 0xEE, 0x33, 
			0x01, 0xD8, 0xE1, 0x3C, 0x69, 0xC5, 0x99, 0xE4, 0xA3, 0xCE, 0xBE, 0xA8, 0x19, 0x6A, 0x76, 0x63, 0x09, 0xD3, 0xD3, 0x3B, 0x56, 0x5A, 0xEA, 0x90, 0x6F, 0x2A, 0xA9, 0x24, 0x6C, 0x30, 0x2F, 0x89, 0x8C, 0x77, 0x79, 0xC1, 0x5A, 0x6A, 0xFD, 0xEC, 0x42, 0x5E, 0xAF, 0x75, 0xDE, 0xC9, 0x08, 0x26, 0xCE, 0x2C, 0xF2, 0xBD, 0x86, 0x92, 0x1B, 0xBB, 0x51, 0x84, 0x56, 0xB7, 0x9D, 0x1C, 0x21, 0x31, 0x02, 0xFC, 0xCA, 0x6F, 0xDB, 0x71, 0xC2, 0x5B, 0x13, 0x5C, 0x96, 0x50, 0x14, 0x3B, 0x80, 0x0B, 0x23, 0xB1, 0x21, 0xCE, 0x63, 0x76, 0x7F, 0x0C, 0x00, 0x32, 0xE7, 0x7E, 0x60, 0x6E, 0x76, 0x61, 0xF5, 0xAC, 0xCF, 0xA2, 
			0xDC, 0xE6, 0x65, 0xD3, 0x01, 0xD4, 0x9C, 0x26, 0xDA, 0xF3, 0xAD, 0x00, 0xE2, 0x56, 0x24, 0xF2, 0x3F, 0x75, 0x7B, 0xEE, 0x2D, 0xB2, 0xCC, 0xF2, 0x0A, 0x14, 0x90, 0xC4, 0x2A, 0x79, 0xDE, 0x2A, 0x35, 0x66, 0xBC, 0xC2, 0x55, 0xDC, 0xB9, 0x0F, 0xC8, 0x08, 0x51, 0xCE, 0x40, 0xAD, 0x5E, 0xB8, 0xDD, 0xA0, 0x2C, 0x4C, 0x96, 0x99, 0x3B, 0x8F, 0x0B, 0xCC, 0x00, 0xA1, 0x37, 0xF5, 0xDB, 0xE2, 0xF8, 0x15, 0xA1, 0xC3, 0xCE, 0x6B, 0xDE, 0x3F, 0xE9, 0x52, 0x9D, 0x07, 0x2E, 0xBC, 0x7A, 0x67, 0x94, 0x19, 0x50, 0x32, 0xFC, 0x1E, 0x6E, 0x9A, 0x94, 0xDD, 0xB6, 0x87, 0x3D, 0x44, 0xA1, 0xA4, 0x95, 0x7A, 0x10, 0x3E, 
			0xAD, 0x73, 0x5E, 0xA8, 0xCF, 0x35, 0x0E, 0x6D, 0x48, 0xE7, 0xC5, 0x2D, 0x91, 0xAB, 0x56, 0xF5, 0x6B, 0x4B, 0xBC, 0x65, 0xB9, 0x3A, 0x7D, 0x46, 0x8E, 0xD0, 0xC0, 0x71, 0x0F, 0x12, 0x0D, 0xFD, 0x50, 0x49, 0xFF, 0x23, 0x0A, 0xC2, 0xA0, 0xB1, 0xD9, 0x5D, 0x08, 0xFE, 0xDB, 0x52, 0x91, 0xD0, 0x8D, 0x60, 0x7D, 0xDC, 0x77, 0x0A, 0x99, 0xBF, 0xC5, 0x42, 0xFB, 0x0D, 0x44, 0x38, 0xF4, 0xC4, 0xB1, 0xF2, 0x87, 0x8B, 0x26, 0x3C, 0x2D, 0x1C, 0xE2, 0x67, 0xEE, 0xF6, 0x84, 0xB8, 0x04, 0xE8, 0x2B, 0x8D, 0x7C, 0x59, 0x0F, 0x70, 0xF5, 0x60, 0xEA, 0x44, 0x1F, 0xB8, 0xB8, 0xCB, 0xEB, 0xFC, 0xBE, 0xBE, 0x82, 0xA6, 
			0xBB, 0x1B, 0x77, 0x26, 0x58, 0x45, 0xA6, 0x4B, 0x0A, 0x92, 0x86, 0xC4, 0x52, 0xEA, 0x72, 0xA4, 0xCF, 0x72, 0xB8, 0x98, 0x40, 0x76, 0x2D, 0x2D, 0xB2, 0xCD, 0xC0, 0x0D, 0x0F, 0x96, 0x8A, 0xA0, 0x39, 0xD2, 0xF0, 0x79, 0xD9, 0x19, 0xE3, 0x12, 0x40, 0xAB, 0x49, 0x3F, 0xC9, 0x97, 0x3D, 0x25, 0xFF, 0xCF, 0xEC, 0x01, 0xBF, 0xE3, 0x4E, 0xB4, 0x3B, 0x8F, 0x64, 0xDC, 0x70, 0xDA, 0xA0, 0xE6, 0x1C, 0x09, 0x95, 0x6B, 0xCF, 0x16, 0xBF, 0x44, 0xA5, 0xFA, 0x36, 0x31, 0x81, 0xA8, 0x30, 0x0D, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x02, 0x31, 0x00, 0x30, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 
			0x86, 0xF7, 0x0D, 0x01, 0x09, 0x15, 0x31, 0x06, 0x04, 0x04, 0x01, 0x00, 0x00, 0x00, 0x30, 0x17, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x14, 0x31, 0x0A, 0x1E, 0x08, 0x00, 0x4D, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x6F, 0x30, 0x69, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x01, 0x31, 0x5C, 0x1E, 0x5A, 0x00, 0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20, 0x00, 0x52, 0x00, 0x53, 0x00, 0x41, 0x00, 0x20, 0x00, 0x53, 0x00, 0x43, 0x00, 0x68, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x6E, 0x00, 0x65, 0x00, 0x6C, 
			0x00, 0x20, 0x00, 0x43, 0x00, 0x72, 0x00, 0x79, 0x00, 0x70, 0x00, 0x74, 0x00, 0x6F, 0x00, 0x67, 0x00, 0x72, 0x00, 0x61, 0x00, 0x70, 0x00, 0x68, 0x00, 0x69, 0x00, 0x63, 0x00, 0x20, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x76, 0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x72, 0x30, 0x82, 0x07, 0x67, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x06, 0xA0, 0x82, 0x07, 0x58, 0x30, 0x82, 0x07, 0x54, 0x02, 0x01, 0x00, 0x30, 0x82, 0x07, 0x4D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 
			0x06, 0x30, 0x0E, 0x04, 0x08, 0xB7, 0xF8, 0xA2, 0x70, 0xCF, 0xB6, 0x51, 0x48, 0x02, 0x02, 0x07, 0xD0, 0x80, 0x82, 0x07, 0x20, 0x5E, 0x23, 0x13, 0x6E, 0x0A, 0x69, 0x00, 0x0C, 0xDA, 0xAD, 0xA8, 0xC8, 0xF8, 0xAC, 0xDC, 0x31, 0x49, 0xB8, 0x96, 0xF3, 0x75, 0x0F, 0x1C, 0xEF, 0x68, 0xA2, 0x9F, 0xB3, 0xE3, 0xB0, 0x39, 0x81, 0x61, 0x95, 0x86, 0x5A, 0x99, 0x6C, 0x6C, 0x84, 0xE6, 0x8D, 0x8D, 0xD9, 0x2E, 0xA8, 0xE9, 0x3B, 0xC6, 0x74, 0xB1, 0xD1, 0xCA, 0xC2, 0x38, 0x8C, 0x0C, 0xB4, 0x06, 0x29, 0x26, 0x52, 0x35, 0x68, 0x28, 0xA6, 0xF4, 0x7B, 0x0A, 0xE0, 0xBC, 0xA9, 0xBD, 0xAC, 0x5F, 0xFF, 0x06, 0x56, 0x70, 
			0xE1, 0xB0, 0x11, 0x2B, 0x17, 0x4B, 0x0A, 0x8C, 0x42, 0x77, 0xD1, 0xF8, 0x08, 0x28, 0x73, 0x60, 0x5A, 0xC1, 0xD9, 0x3F, 0x60, 0x40, 0x21, 0xC6, 0x33, 0x9D, 0x44, 0xF6, 0x98, 0xF4, 0x52, 0xFB, 0x57, 0x6D, 0x1E, 0x9A, 0x52, 0xF6, 0x11, 0xAD, 0xA0, 0xE4, 0xFE, 0x0E, 0xEB, 0x3F, 0xDE, 0x7C, 0xB7, 0x82, 0x04, 0x1C, 0x78, 0xFD, 0xEB, 0x99, 0x4D, 0x8E, 0xEE, 0xCA, 0xB2, 0x4F, 0xD1, 0x47, 0x5B, 0xAA, 0x4A, 0xEB, 0x16, 0xBF, 0x50, 0x93, 0xBF, 0xDF, 0x57, 0x6A, 0x8A, 0xBD, 0x4E, 0xED, 0x72, 0x5D, 0x50, 0x59, 0xC7, 0x4F, 0x4F, 0xB5, 0x7E, 0x4F, 0xA3, 0xC2, 0x51, 0x3B, 0x31, 0x12, 0x14, 0xCC, 0x2E, 0x1C, 
			0xC4, 0xD3, 0xC2, 0x89, 0xD3, 0x6A, 0xC4, 0x16, 0xAA, 0x13, 0x83, 0xD6, 0x30, 0xD6, 0x76, 0xE0, 0x16, 0xCF, 0x28, 0x25, 0xA8, 0xFB, 0x25, 0xBC, 0x82, 0xEF, 0xD4, 0xC4, 0x64, 0x3B, 0xE0, 0xC0, 0x91, 0xC0, 0x21, 0x17, 0x43, 0x8F, 0xCB, 0xCE, 0xD3, 0x2B, 0x7B, 0x11, 0x5E, 0x73, 0x62, 0x17, 0x9E, 0xB4, 0x00, 0xF7, 0xB3, 0x7E, 0x24, 0xC9, 0x21, 0xB8, 0xF7, 0x33, 0x02, 0x04, 0xB9, 0xDE, 0x6C, 0x97, 0xC4, 0x3B, 0xCF, 0xDE, 0xC4, 0xF7, 0x2A, 0x68, 0x8A, 0x82, 0x22, 0xA1, 0x95, 0xBB, 0x9E, 0xF6, 0x6C, 0xBB, 0x0E, 0x69, 0x83, 0x55, 0xE2, 0x5A, 0x5F, 0x60, 0x2E, 0x43, 0x41, 0xD1, 0xB8, 0xAB, 0x17, 0x3A, 
			0x4C, 0xA8, 0xB9, 0x81, 0xFF, 0x40, 0x28, 0x50, 0x67, 0xCF, 0x3A, 0x01, 0x5D, 0xCC, 0x09, 0xD2, 0x2D, 0x2A, 0x7D, 0x4E, 0xDA, 0x88, 0x85, 0x26, 0x03, 0x05, 0x41, 0x2B, 0x7A, 0x4B, 0xEC, 0x1A, 0xBC, 0xDF, 0xD6, 0x98, 0xFA, 0x22, 0x9D, 0xFA, 0xE3, 0x7D, 0x37, 0x5B, 0x5E, 0xFF, 0x54, 0xB3, 0x48, 0x50, 0x0D, 0xDC, 0x93, 0xD4, 0xF8, 0xC7, 0x05, 0x66, 0xC7, 0xAC, 0x9C, 0x87, 0x4A, 0x90, 0xCE, 0x2E, 0xF3, 0x4C, 0x7E, 0x90, 0xC5, 0xE5, 0x20, 0x2C, 0xD9, 0x24, 0xB4, 0x2C, 0x51, 0x57, 0x1F, 0xD9, 0x24, 0x3C, 0xFA, 0x94, 0x0E, 0xAB, 0xAA, 0x6A, 0x42, 0xFE, 0x9D, 0xE7, 0x53, 0xD6, 0xC2, 0x1D, 0x5F, 0x2E, 
			0x79, 0x86, 0x02, 0xDB, 0xD0, 0x8B, 0x0C, 0xD7, 0x43, 0x96, 0x60, 0x2B, 0x4A, 0x8F, 0xEE, 0xE3, 0x19, 0x61, 0x47, 0x82, 0x87, 0x6F, 0x70, 0xB6, 0xE2, 0x3F, 0x17, 0x66, 0x21, 0xE1, 0x53, 0x61, 0xC2, 0xD3, 0xE8, 0x26, 0x02, 0x68, 0x2A, 0x80, 0xC8, 0xFD, 0x8A, 0xA8, 0xCC, 0xD5, 0xAA, 0xCB, 0x14, 0xDD, 0x5A, 0x7D, 0x4A, 0xCA, 0x39, 0xE1, 0x51, 0x8A, 0x26, 0x8A, 0xDA, 0xE2, 0x54, 0x92, 0x28, 0x6E, 0x63, 0x3F, 0x9A, 0x50, 0xC0, 0x5C, 0x76, 0xE2, 0x98, 0x94, 0xFB, 0xBC, 0xAD, 0x93, 0x2C, 0x97, 0xDF, 0x15, 0x20, 0x7D, 0x4F, 0x28, 0xD5, 0xD1, 0x5D, 0x35, 0x8B, 0x5B, 0x96, 0x77, 0x34, 0xDD, 0xA3, 0x07, 
			0x44, 0xFB, 0xCD, 0x72, 0x2B, 0x12, 0xD5, 0x1D, 0x23, 0xA1, 0x51, 0x04, 0xF8, 0x9D, 0xF8, 0xFF, 0x0F, 0x39, 0x0F, 0x05, 0x33, 0x24, 0x0A, 0x1F, 0xD1, 0xB5, 0xA8, 0x75, 0x39, 0x41, 0xE8, 0xD5, 0x99, 0x1D, 0x9B, 0x45, 0xFC, 0x32, 0x20, 0x6E, 0xFA, 0x96, 0x80, 0xDD, 0x88, 0x5D, 0x3B, 0x12, 0x1C, 0xC6, 0x86, 0xFA, 0x27, 0x4A, 0x78, 0x46, 0xE2, 0xC0, 0xCC, 0x66, 0x01, 0xE8, 0x30, 0x32, 0x26, 0x42, 0xDE, 0x8B, 0x71, 0xE8, 0x09, 0x9C, 0xF4, 0x90, 0x71, 0x17, 0x66, 0xFE, 0x0A, 0x0F, 0xD4, 0x84, 0x13, 0xFB, 0x66, 0x99, 0x98, 0xAF, 0xD8, 0x16, 0x13, 0x15, 0xAD, 0x18, 0xBE, 0x49, 0x02, 0xAB, 0x42, 0x50, 
			0x14, 0xB7, 0x3B, 0x33, 0x3C, 0x22, 0xA9, 0xE1, 0xC5, 0x32, 0xCD, 0x8D, 0xB4, 0x31, 0xF9, 0x27, 0x6C, 0x50, 0x0B, 0x40, 0x5E, 0xBE, 0x02, 0xB5, 0x27, 0xC4, 0x0D, 0x2C, 0xD5, 0xAB, 0x8A, 0x58, 0x5E, 0x53, 0xAD, 0x9C, 0x0D, 0x5B, 0x06, 0xE8, 0x77, 0x24, 0x94, 0xAD, 0xC0, 0x2C, 0x17, 0x8F, 0x20, 0x70, 0xC8, 0x29, 0xEB, 0x9B, 0x84, 0x4A, 0x91, 0xA5, 0xEC, 0x85, 0x2C, 0xCF, 0x68, 0xC9, 0x72, 0xEB, 0x3D, 0x63, 0xE9, 0xDB, 0x7D, 0xBB, 0x3D, 0xBE, 0x9A, 0x37, 0x91, 0x07, 0xE5, 0x95, 0x39, 0x63, 0xB0, 0x3F, 0x46, 0xEA, 0xB7, 0x76, 0x82, 0x05, 0xE1, 0xFF, 0x7C, 0x14, 0x41, 0x0F, 0x74, 0x3E, 0x43, 0xCD, 
			0x87, 0x5A, 0xA7, 0x30, 0x61, 0x29, 0xE2, 0xBA, 0x59, 0x7F, 0xF3, 0xBF, 0x78, 0x1C, 0x4A, 0xB8, 0x59, 0xEF, 0x9D, 0xE3, 0xFD, 0xEB, 0xE9, 0x39, 0xA9, 0xDF, 0x3D, 0xC9, 0xD4, 0x42, 0x5F, 0x79, 0xCA, 0xFC, 0xEE, 0xFE, 0xA0, 0x88, 0x30, 0x78, 0x69, 0x37, 0xA0, 0xC2, 0x64, 0xBC, 0x41, 0x65, 0xDB, 0xB8, 0xB0, 0x99, 0xC7, 0xD5, 0xF9, 0x03, 0x05, 0x98, 0x5A, 0xAB, 0x76, 0xB5, 0x0D, 0x7B, 0xDF, 0x9B, 0xFA, 0x92, 0x5D, 0xBD, 0x52, 0x30, 0xB7, 0x0F, 0xE8, 0x60, 0x97, 0xCC, 0xF5, 0x66, 0xA2, 0xD0, 0x77, 0x9E, 0x6B, 0x5C, 0xF2, 0x41, 0x20, 0x21, 0x6D, 0x0C, 0xE1, 0xB6, 0x26, 0xD8, 0xD9, 0x66, 0x94, 0xC0, 
			0x55, 0x4E, 0xCC, 0xBC, 0x2D, 0xDC, 0xC3, 0x23, 0x17, 0x9C, 0x7F, 0xA6, 0x75, 0x38, 0x33, 0xBB, 0x49, 0xBE, 0xF0, 0x77, 0xCB, 0xE7, 0xB2, 0xF5, 0x40, 0xE3, 0xD5, 0xCC, 0x1F, 0x83, 0x50, 0xA5, 0x2E, 0x1B, 0x17, 0xBD, 0x63, 0x8C, 0x9F, 0x99, 0x30, 0x41, 0xC8, 0x0D, 0x8B, 0x05, 0x80, 0x67, 0x03, 0x60, 0xCF, 0x2A, 0xA0, 0x55, 0x14, 0x74, 0xA7, 0xEA, 0x43, 0x3F, 0x5A, 0x03, 0x0B, 0xEF, 0x3F, 0xC8, 0xAA, 0x03, 0xB9, 0x1B, 0xC6, 0xB9, 0x63, 0xD4, 0x9C, 0xA1, 0x42, 0x18, 0x32, 0x12, 0x87, 0xEA, 0x44, 0xB9, 0xA8, 0x72, 0xA8, 0x75, 0xB4, 0x09, 0xEB, 0x90, 0x7F, 0xB5, 0x00, 0x39, 0xDE, 0x2C, 0x07, 0x87, 
			0xB5, 0x35, 0xDD, 0x44, 0x03, 0x09, 0xA9, 0xA0, 0xD6, 0x40, 0x69, 0x88, 0xC9, 0x2A, 0xDB, 0x3E, 0x56, 0xEF, 0x42, 0x23, 0xB7, 0x4D, 0x62, 0xDA, 0x56, 0x1C, 0xC5, 0xC1, 0x4D, 0x07, 0x02, 0x14, 0x7E, 0xE7, 0x6D, 0x38, 0x05, 0x70, 0x19, 0x77, 0xA6, 0x46, 0xE9, 0x18, 0x3E, 0x06, 0xEE, 0x5A, 0xCA, 0x35, 0xE3, 0xCD, 0xAA, 0x4C, 0x89, 0x5D, 0x41, 0x85, 0x17, 0x53, 0x18, 0x86, 0x3D, 0x20, 0x18, 0x53, 0x1D, 0xF8, 0x4C, 0x09, 0x2C, 0x48, 0x75, 0x00, 0xF8, 0xD1, 0xF4, 0xA5, 0xD7, 0x29, 0xB5, 0xFF, 0x42, 0xFA, 0x6B, 0xCB, 0x47, 0x0F, 0x33, 0x00, 0x59, 0x54, 0x07, 0x62, 0x5C, 0xC7, 0x57, 0x63, 0xB4, 0x32, 
			0x5C, 0xF1, 0x8E, 0xBA, 0x6D, 0x0E, 0x85, 0x3B, 0x3D, 0xE7, 0xAA, 0x31, 0x61, 0x77, 0x3F, 0x77, 0xF4, 0x1F, 0xD7, 0x6C, 0x5F, 0x13, 0xCF, 0xB5, 0x9A, 0xAC, 0x6F, 0x2C, 0x49, 0x2D, 0xDD, 0xED, 0x27, 0xA9, 0x77, 0x1F, 0x38, 0x78, 0x16, 0x82, 0x00, 0xB3, 0xD9, 0x8E, 0xFA, 0x43, 0xE2, 0x54, 0xB7, 0x36, 0xED, 0x20, 0x33, 0x1A, 0x7E, 0x0E, 0xF2, 0xEE, 0x20, 0xE7, 0x10, 0xC4, 0xB6, 0x07, 0x0A, 0x42, 0x26, 0xCA, 0xC1, 0x6C, 0x29, 0xA1, 0xBD, 0x05, 0xE0, 0x70, 0xED, 0x17, 0x7C, 0x21, 0xD3, 0x9F, 0x72, 0xE7, 0x47, 0xA3, 0x5F, 0xCA, 0x50, 0x4C, 0x3C, 0x16, 0x9D, 0x20, 0x49, 0x7B, 0x9A, 0x53, 0xCF, 0x7C, 
			0x9B, 0x69, 0x5E, 0xE0, 0x56, 0xB0, 0x60, 0xFB, 0xC6, 0xE8, 0x67, 0x31, 0x18, 0xDB, 0x67, 0xA2, 0x61, 0xE2, 0x8E, 0x35, 0xFA, 0xBD, 0xED, 0x43, 0x32, 0xCA, 0x99, 0xAC, 0xF7, 0xCB, 0xB8, 0x1E, 0x57, 0x2D, 0xC7, 0xB4, 0x2A, 0xEA, 0x53, 0xCE, 0xDB, 0xD7, 0xE6, 0x8F, 0x9A, 0x98, 0x12, 0xE3, 0xB0, 0x7E, 0x85, 0x1B, 0x6B, 0x2F, 0xE3, 0xD0, 0x54, 0x5C, 0x31, 0xC0, 0x96, 0xE8, 0x04, 0xD9, 0x87, 0x8F, 0x8A, 0xC9, 0xC1, 0x95, 0x31, 0x11, 0x66, 0xC3, 0x86, 0x06, 0x93, 0xC4, 0xDC, 0x6F, 0xD7, 0x7C, 0x65, 0xF3, 0xF9, 0xB1, 0x6F, 0x34, 0xD8, 0x25, 0x21, 0x02, 0x1B, 0x12, 0x52, 0x19, 0xD0, 0xED, 0xB8, 0x4A, 
			0x67, 0x1E, 0x49, 0x2C, 0xDD, 0x47, 0xA7, 0x0A, 0x1F, 0x21, 0x67, 0x5A, 0x0F, 0x34, 0xDB, 0x6D, 0x1E, 0xFB, 0x71, 0xBC, 0xB9, 0x8F, 0x84, 0xE4, 0x3A, 0xE4, 0x47, 0x93, 0x48, 0xC2, 0xCA, 0xDA, 0x5B, 0x7B, 0xA0, 0x3F, 0x2F, 0x9C, 0x68, 0xDC, 0x72, 0x15, 0x18, 0x57, 0xBA, 0x32, 0x6B, 0x92, 0x08, 0x08, 0x65, 0xA5, 0x30, 0xA4, 0xED, 0x35, 0x80, 0xC5, 0xD3, 0x93, 0x01, 0x36, 0x53, 0x8E, 0xC9, 0xF6, 0x2D, 0xCA, 0xB9, 0xE6, 0xE9, 0x94, 0x89, 0xE3, 0x05, 0x81, 0x9B, 0x69, 0x0B, 0xD7, 0xD2, 0x55, 0x0A, 0xFE, 0xAF, 0x98, 0x97, 0xB9, 0xCB, 0xB8, 0xCB, 0xB6, 0x48, 0xD1, 0xB4, 0xF1, 0xD2, 0x19, 0x53, 0xB1, 
			0xCE, 0x4E, 0x3D, 0x11, 0xF3, 0xC6, 0x89, 0xFC, 0x68, 0x48, 0x4C, 0xF9, 0x11, 0x1B, 0xDF, 0x10, 0x21, 0x44, 0x2C, 0x43, 0xCC, 0x00, 0x0D, 0xC8, 0x40, 0x6D, 0xDB, 0xCE, 0xC7, 0xBF, 0xA4, 0xCF, 0x6C, 0xAF, 0x71, 0xF4, 0x72, 0xB8, 0xF4, 0x32, 0x19, 0xD2, 0xFC, 0xFE, 0x28, 0xC8, 0x00, 0x44, 0xB9, 0xEC, 0x70, 0x5B, 0xF5, 0x39, 0xDC, 0xBD, 0xF4, 0xEF, 0xC3, 0x8A, 0xF1, 0x06, 0x52, 0x90, 0xAF, 0x18, 0x45, 0xF1, 0xC7, 0x10, 0x0C, 0xD2, 0xB3, 0xBC, 0xE5, 0xF1, 0x32, 0x6A, 0x0C, 0x55, 0x3E, 0x28, 0x62, 0xC9, 0xC1, 0x12, 0x1F, 0x6E, 0x30, 0x6A, 0x4C, 0x9B, 0x6D, 0x32, 0x18, 0x2D, 0xDD, 0xE6, 0x0F, 0xA5, 
			0x0B, 0xBE, 0xBB, 0xF8, 0x8D, 0xF7, 0x40, 0x9F, 0x24, 0xB1, 0x85, 0x17, 0x9F, 0x38, 0x2C, 0xEE, 0x2F, 0x50, 0x99, 0x30, 0x75, 0x95, 0x85, 0xA2, 0xB7, 0xA2, 0x7C, 0x42, 0x58, 0xD9, 0x7D, 0x52, 0xB8, 0x0D, 0xF2, 0x68, 0x57, 0x75, 0xEC, 0x5A, 0xFF, 0xBE, 0xC3, 0xB4, 0x21, 0xE2, 0xA1, 0xCB, 0x60, 0x5F, 0x2A, 0x3A, 0xEE, 0xE0, 0xC3, 0xFB, 0x9A, 0x21, 0x02, 0x86, 0x10, 0xA1, 0xCF, 0x2F, 0x6D, 0x83, 0xBC, 0x09, 0xCE, 0x97, 0x08, 0xDB, 0x6E, 0x17, 0x90, 0xEF, 0x06, 0x2E, 0x19, 0x27, 0xC2, 0x1A, 0x9E, 0xB4, 0xB7, 0x16, 0x1A, 0xF6, 0x92, 0x81, 0x5C, 0xF3, 0x8D, 0x52, 0x87, 0xD1, 0x50, 0x23, 0x78, 0xE5, 
			0x81, 0x16, 0x24, 0x12, 0x21, 0xD8, 0x70, 0x30, 0xB4, 0x38, 0x40, 0x7B, 0xBC, 0x21, 0x51, 0xDB, 0x66, 0x15, 0x4C, 0x21, 0x8E, 0xF5, 0x5B, 0x78, 0x65, 0x29, 0x7C, 0xEF, 0x5D, 0xE0, 0x7E, 0x4D, 0xF9, 0x35, 0xC6, 0x97, 0xEA, 0x4A, 0x56, 0xB5, 0xA2, 0x38, 0xAE, 0xF1, 0x50, 0x1A, 0xBE, 0xE8, 0xB1, 0x68, 0xBA, 0xB6, 0x4D, 0x67, 0x63, 0x52, 0x07, 0xBC, 0x4E, 0x68, 0x7C, 0x30, 0x75, 0x16, 0x97, 0x99, 0xD5, 0x8E, 0x14, 0xD9, 0x35, 0x80, 0x35, 0x48, 0xA3, 0x42, 0x3B, 0x60, 0xC0, 0xF7, 0xC5, 0x4E, 0x40, 0xEF, 0x1D, 0xA4, 0x2A, 0x43, 0xD3, 0xA9, 0xB5, 0x3D, 0x72, 0x4D, 0x51, 0x17, 0xFC, 0x92, 0x2F, 0xAC, 
			0xC8, 0x81, 0xE1, 0x01, 0x25, 0xBE, 0x89, 0xA9, 0x13, 0x9B, 0xE6, 0xBF, 0x60, 0x9E, 0x43, 0x68, 0x65, 0x34, 0x7A, 0x9C, 0x01, 0x9F, 0xA5, 0x32, 0xFC, 0x06, 0x38, 0x93, 0x4F, 0x15, 0x35, 0x42, 0xAB, 0xB0, 0x36, 0x67, 0x06, 0x4F, 0x2C, 0x70, 0x64, 0x41, 0x31, 0x20, 0x90, 0xA6, 0x3B, 0xC9, 0x99, 0xD8, 0x47, 0x42, 0x64, 0xB6, 0x2B, 0x08, 0xD3, 0x0E, 0x41, 0x19, 0x77, 0xF3, 0x5D, 0x56, 0x05, 0xF2, 0xBC, 0xF9, 0x9A, 0x92, 0x43, 0xD9, 0x18, 0x83, 0x90, 0x13, 0xD8, 0xA1, 0x42, 0x2A, 0x18, 0x2D, 0xD8, 0x1D, 0xE8, 0x50, 0xC5, 0xE8, 0x6F, 0x4F, 0x05, 0x45, 0x74, 0x06, 0x2A, 0x8B, 0x24, 0x33, 0xC0, 0xAA, 
			0xAA, 0x84, 0x8A, 0xE2, 0xE8, 0xB0, 0x7E, 0x2D, 0xBD, 0x0C, 0x20, 0xD1, 0x44, 0xAA, 0xCA, 0xC4, 0x83, 0x8C, 0x08, 0xA3, 0x17, 0x36, 0xA1, 0xA1, 0xC2, 0x04, 0xA7, 0x06, 0xCB, 0xB7, 0xF2, 0x64, 0x28, 0x0C, 0xC4, 0x91, 0x0B, 0xDF, 0x93, 0xCB, 0xBE, 0x85, 0x07, 0x11, 0xBB, 0xD8, 0x8A, 0xB1, 0xAB, 0xDC, 0x91, 0x02, 0xDE, 0x4F, 0xE5, 0x6E, 0xC7, 0x96, 0xAB, 0x7E, 0x4A, 0xAA, 0xCF, 0x05, 0x46, 0xE2, 0xB3, 0x52, 0xCD, 0x93, 0x7E, 0xAE, 0xB1, 0x40, 0x61, 0x7E, 0xCB, 0x43, 0x7F, 0xF4, 0x32, 0x8C, 0xA5, 0x0D, 0x52, 0xF7, 0x7A, 0x76, 0xBA, 0x05, 0x14, 0x65, 0xAD, 0x36, 0x85, 0x46, 0x15, 0x00, 0x01, 0xD7, 
			0x04, 0xA3, 0xD3, 0x1A, 0xAE, 0x59, 0xEF, 0x6E, 0xBE, 0x39, 0xDF, 0x0D, 0x78, 0xEF, 0x6F, 0x14, 0x02, 0xEC, 0x04, 0xB9, 0xE7, 0x15, 0x54, 0x51, 0x83, 0x75, 0xE2, 0x06, 0x1F, 0x07, 0xD3, 0x82, 0xCB, 0x01, 0x03, 0xE5, 0x43, 0xAC, 0x62, 0x66, 0xF7, 0xCF, 0x74, 0x1B, 0xCD, 0x30, 0x37, 0x30, 0x1F, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x04, 0x14, 0x84, 0xA8, 0x88, 0x5F, 0xF6, 0x8E, 0xEC, 0x49, 0x94, 0x28, 0xDD, 0xC1, 0xA3, 0xCB, 0xBD, 0x73, 0x4A, 0x76, 0x0E, 0x6B, 0x04, 0x14, 0x7B, 0x11, 0xA6, 0xD2, 0x27, 0x3F, 0x96, 0xC9, 0xA7, 0xEF, 0x06, 0x6D, 0x03, 0xF9, 0xEA, 0x04, 0x2C, 0xC7, 
			0xF7, 0x11 };

		static public byte[] farscape_strong_pfx = { 0x30, 0x82, 0x06, 0xA7, 0x02, 0x01, 0x03, 0x30, 0x82, 0x06, 0x63, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x06, 0x54, 0x04, 0x82, 0x06, 0x50, 0x30, 0x82, 0x06, 0x4C, 0x30, 0x82, 0x03, 0x8D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0x7E, 0x04, 0x82, 0x03, 0x7A, 0x30, 0x82, 0x03, 0x76, 0x30, 0x82, 0x03, 0x72, 0x06, 0x0B, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x0A, 0x01, 0x02, 0xA0, 0x82, 0x02, 0xB6, 0x30, 0x82, 0x02, 0xB2, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 
			0x0E, 0x04, 0x08, 0x21, 0x53, 0x51, 0xD2, 0xB9, 0x79, 0xA4, 0x4F, 0x02, 0x02, 0x07, 0xD0, 0x04, 0x82, 0x02, 0x90, 0xA2, 0xBA, 0x47, 0x19, 0xE3, 0x35, 0x0B, 0xA1, 0x71, 0x60, 0x65, 0x1A, 0xB0, 0x94, 0xF2, 0xF1, 0xDE, 0x8E, 0xA4, 0x59, 0xF5, 0x07, 0x1B, 0xDF, 0xD2, 0x9C, 0xF4, 0xA2, 0x5C, 0x8C, 0xA7, 0x3F, 0x7E, 0xD3, 0x93, 0x1B, 0x87, 0xA8, 0x1A, 0xA0, 0xDD, 0xF4, 0x0B, 0x76, 0x30, 0xDF, 0xE5, 0x56, 0xAE, 0x26, 0xF7, 0x6C, 0x72, 0x0A, 0x41, 0x0E, 0xD7, 0x42, 0xC3, 0x99, 0x8D, 0x8B, 0xC2, 0x18, 0xBE, 0x30, 0x12, 0xED, 0x20, 0xD3, 0xB3, 0xAF, 0xCD, 0xBE, 0x62, 0xED, 0x6F, 0x55, 0x79, 0x33, 0xCD, 
			0x8F, 0x2C, 0x1E, 0x1C, 0xB8, 0xDC, 0x1C, 0x4D, 0x45, 0xCE, 0xE6, 0x1E, 0x81, 0x46, 0xC1, 0x93, 0x1B, 0x7A, 0x41, 0x19, 0x82, 0xED, 0xBF, 0x8F, 0x0F, 0xB1, 0x39, 0x50, 0x58, 0xA2, 0xD2, 0xFB, 0xE6, 0x9A, 0x86, 0x68, 0x2C, 0x58, 0xF9, 0xB3, 0x7C, 0xAC, 0xCC, 0x02, 0xB0, 0x70, 0xC2, 0x5D, 0x0B, 0x7F, 0xC0, 0xA7, 0x60, 0x66, 0x41, 0xBA, 0x4F, 0xE6, 0x89, 0x16, 0x5C, 0x0D, 0x4B, 0x4F, 0x43, 0xF5, 0xFD, 0x5E, 0x22, 0x93, 0x4B, 0x34, 0xC3, 0xEC, 0xA9, 0xCF, 0x92, 0x6F, 0x0E, 0xD6, 0x01, 0x21, 0x48, 0xF4, 0x35, 0xF2, 0xA0, 0xF9, 0x84, 0xD4, 0x30, 0xE0, 0x69, 0xD2, 0xB2, 0x03, 0x3F, 0x3A, 0x5E, 0xFA, 
			0x06, 0xA2, 0x30, 0x1B, 0x93, 0xAC, 0x0D, 0x5D, 0x9E, 0xFA, 0x49, 0x29, 0x82, 0xE9, 0x91, 0x40, 0x87, 0x52, 0xF1, 0xED, 0xC3, 0x00, 0x1F, 0x00, 0x54, 0x9F, 0xB6, 0xC7, 0xDE, 0xF4, 0xFB, 0xCF, 0xE7, 0x9B, 0x40, 0x8B, 0x51, 0xEF, 0xFD, 0x33, 0x1C, 0xD9, 0x64, 0x19, 0x45, 0x89, 0x6C, 0x46, 0xA1, 0x9B, 0xA9, 0x93, 0xFE, 0xF8, 0x4D, 0x20, 0x63, 0x34, 0xFB, 0x1B, 0xB9, 0xC8, 0xFE, 0xD4, 0x63, 0xD1, 0x03, 0x57, 0x63, 0xC2, 0xEA, 0xC2, 0xA7, 0x9A, 0x03, 0xEB, 0xC0, 0xF1, 0xAF, 0xB8, 0xD7, 0x37, 0x76, 0x0C, 0xD3, 0x98, 0x21, 0xD2, 0x84, 0x11, 0xD6, 0xBF, 0x29, 0xC8, 0x03, 0xA1, 0xAA, 0xCB, 0xA6, 0x51, 
			0xF5, 0xBB, 0xA8, 0xFA, 0x43, 0xFB, 0x59, 0x51, 0x41, 0x1D, 0x2F, 0xF3, 0xFC, 0xF7, 0xA2, 0x4C, 0xEC, 0x47, 0x20, 0x14, 0xF3, 0x46, 0x8B, 0x41, 0x5E, 0x65, 0x2F, 0x18, 0x99, 0x07, 0x23, 0x6B, 0xEA, 0x55, 0x16, 0xE3, 0xA8, 0xB7, 0x31, 0x5C, 0x23, 0xF3, 0xD4, 0x1E, 0xC3, 0xB6, 0xF8, 0x53, 0x26, 0x7D, 0x42, 0x2D, 0x3E, 0xC2, 0xAE, 0x9F, 0x01, 0x83, 0x90, 0x0D, 0xA6, 0xB0, 0x2D, 0xBD, 0x86, 0x31, 0x5A, 0x15, 0x07, 0x6F, 0xA6, 0xFC, 0x35, 0x30, 0x6A, 0xEF, 0xF6, 0xC1, 0xEB, 0x18, 0x01, 0xE9, 0xE7, 0xAA, 0x87, 0x54, 0x4C, 0xAE, 0x98, 0x67, 0xDB, 0xDF, 0xFF, 0x4E, 0xA5, 0xA9, 0xA4, 0x05, 0x11, 0xE9, 
			0xA4, 0x1D, 0xB2, 0xB1, 0xD3, 0xF6, 0x77, 0xB6, 0xB5, 0x01, 0xC3, 0x3B, 0xAF, 0x6A, 0xA1, 0x60, 0xB9, 0x14, 0x62, 0x5E, 0x2D, 0x70, 0x25, 0xDF, 0xD3, 0x21, 0x02, 0x3C, 0xD7, 0xF1, 0xF6, 0x0E, 0xD9, 0x4D, 0x59, 0x1A, 0x41, 0x2E, 0x82, 0x28, 0x5C, 0x12, 0x26, 0x68, 0x77, 0xE0, 0x46, 0xC1, 0x0B, 0x9F, 0x15, 0x75, 0x87, 0xD7, 0x87, 0x3A, 0xF6, 0xB4, 0x57, 0x4B, 0x2C, 0x66, 0xD2, 0xCF, 0x47, 0x28, 0x31, 0x8A, 0xDF, 0x54, 0xC5, 0x16, 0x58, 0x23, 0x4F, 0x9E, 0x8D, 0x31, 0xFC, 0x9C, 0xF5, 0x07, 0x03, 0x0F, 0x63, 0xBD, 0xD7, 0x38, 0x65, 0x85, 0x7B, 0xB1, 0xBC, 0x19, 0x3C, 0xEF, 0x41, 0x55, 0xA6, 0x0D, 
			0x50, 0x61, 0xB6, 0xA5, 0x9F, 0xF8, 0xD6, 0x15, 0xBF, 0x40, 0x44, 0x2E, 0xD5, 0x12, 0xAA, 0x29, 0xFE, 0x8A, 0x27, 0x68, 0xC4, 0x83, 0xD5, 0x75, 0x78, 0xF7, 0x38, 0x8F, 0x0D, 0xD3, 0xC9, 0x25, 0xAC, 0xF5, 0x3E, 0x7B, 0x11, 0xCB, 0x45, 0xE0, 0x4E, 0x6A, 0x99, 0xD9, 0x24, 0xBD, 0x68, 0x59, 0x02, 0xF9, 0x0D, 0x32, 0x99, 0x87, 0xEA, 0xFA, 0x4D, 0x73, 0x72, 0x44, 0x10, 0x1B, 0xEC, 0x38, 0x04, 0xD4, 0xCD, 0x13, 0x98, 0x5F, 0x0A, 0x3F, 0x76, 0x80, 0x1E, 0x30, 0x89, 0xFF, 0x5D, 0x03, 0x38, 0x1C, 0x25, 0x9E, 0x79, 0x0A, 0xA8, 0x08, 0xAE, 0xAE, 0x28, 0x7A, 0x74, 0x68, 0x60, 0x13, 0x97, 0x48, 0x4C, 0xFA, 
			0xB3, 0x63, 0x3A, 0x66, 0xD0, 0xAB, 0x42, 0x35, 0x9B, 0xE6, 0x9D, 0x5D, 0xED, 0x05, 0xC5, 0x29, 0x11, 0xB2, 0x36, 0x21, 0x21, 0x86, 0x16, 0xFD, 0x81, 0xF6, 0x5D, 0x26, 0x75, 0xB2, 0x74, 0xD4, 0xF6, 0x59, 0x62, 0x7B, 0xC2, 0x6D, 0xB9, 0xF0, 0xA6, 0x5A, 0x47, 0xEC, 0x55, 0xEA, 0x35, 0xE0, 0xCE, 0x68, 0x63, 0x88, 0x2D, 0x9A, 0xCC, 0xDC, 0x0E, 0x02, 0xAB, 0x69, 0xFE, 0x22, 0xEC, 0xF5, 0xED, 0x36, 0x7B, 0xE9, 0xC2, 0xA6, 0xBF, 0x0C, 0x1C, 0x19, 0xE7, 0x31, 0x81, 0xA8, 0x30, 0x0D, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x02, 0x31, 0x00, 0x30, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 
			0x86, 0xF7, 0x0D, 0x01, 0x09, 0x15, 0x31, 0x06, 0x04, 0x04, 0x01, 0x00, 0x00, 0x00, 0x30, 0x17, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x14, 0x31, 0x0A, 0x1E, 0x08, 0x00, 0x4D, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x6F, 0x30, 0x69, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x01, 0x31, 0x5C, 0x1E, 0x5A, 0x00, 0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20, 0x00, 0x52, 0x00, 0x53, 0x00, 0x41, 0x00, 0x20, 0x00, 0x53, 0x00, 0x43, 0x00, 0x68, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x6E, 0x00, 0x65, 0x00, 0x6C, 
			0x00, 0x20, 0x00, 0x43, 0x00, 0x72, 0x00, 0x79, 0x00, 0x70, 0x00, 0x74, 0x00, 0x6F, 0x00, 0x67, 0x00, 0x72, 0x00, 0x61, 0x00, 0x70, 0x00, 0x68, 0x00, 0x69, 0x00, 0x63, 0x00, 0x20, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x76, 0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x72, 0x30, 0x82, 0x02, 0xB7, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x06, 0xA0, 0x82, 0x02, 0xA8, 0x30, 0x82, 0x02, 0xA4, 0x02, 0x01, 0x00, 0x30, 0x82, 0x02, 0x9D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 
			0x06, 0x30, 0x0E, 0x04, 0x08, 0x02, 0x77, 0x79, 0xCA, 0x9F, 0x4E, 0x06, 0xE3, 0x02, 0x02, 0x07, 0xD0, 0x80, 0x82, 0x02, 0x70, 0x82, 0xE1, 0x66, 0x48, 0xDB, 0x24, 0x9E, 0x33, 0x70, 0x43, 0xFB, 0x6D, 0xD0, 0x47, 0x90, 0xF6, 0xEF, 0xAD, 0x61, 0x28, 0xBF, 0x71, 0x9F, 0x63, 0x2A, 0x69, 0x6C, 0x04, 0x14, 0x34, 0x8A, 0x92, 0xB6, 0x7E, 0x67, 0x50, 0x82, 0x6D, 0x16, 0x8B, 0xB2, 0x18, 0xA4, 0x74, 0x7C, 0xC4, 0x97, 0x50, 0xCE, 0x0B, 0xDB, 0x76, 0xFF, 0xCB, 0x8F, 0x21, 0x4F, 0x50, 0x52, 0x41, 0x0C, 0xA0, 0x98, 0xFB, 0xA8, 0x7F, 0xC0, 0xB4, 0xE3, 0x81, 0x37, 0x12, 0x55, 0x53, 0x5C, 0x3E, 0x95, 0x53, 0x40, 
			0x39, 0xAB, 0xD6, 0x5A, 0xBA, 0xB5, 0x87, 0xCF, 0x60, 0x2A, 0x07, 0x2E, 0xAC, 0x56, 0xC0, 0xD1, 0x75, 0xE2, 0xFC, 0xE3, 0x24, 0xDA, 0x32, 0xED, 0x11, 0xC9, 0x53, 0xED, 0x54, 0xB4, 0xE4, 0x36, 0x1F, 0x72, 0x8B, 0xA2, 0xB1, 0x8D, 0x4B, 0x4C, 0x3F, 0x75, 0x4C, 0x2F, 0xE9, 0x78, 0x97, 0xFC, 0xBA, 0x77, 0x68, 0xE4, 0x65, 0x22, 0x67, 0xE5, 0x4D, 0xE9, 0x6A, 0x63, 0x0C, 0x89, 0x91, 0x12, 0xD5, 0x3B, 0x0D, 0xF4, 0x16, 0xA7, 0xE8, 0x57, 0xA8, 0x88, 0xF4, 0x48, 0x95, 0x71, 0x91, 0xAC, 0x69, 0x67, 0xEB, 0xF0, 0xF3, 0x8C, 0xFB, 0x35, 0x4F, 0x0F, 0xAD, 0x8B, 0xFE, 0x8A, 0xCE, 0xE2, 0x75, 0xDF, 0x63, 0x38, 
			0xB2, 0xD0, 0xAD, 0x71, 0x5F, 0x97, 0x8B, 0x0C, 0xA7, 0x43, 0xC6, 0x62, 0xC0, 0x60, 0x0F, 0xE6, 0x6A, 0xE9, 0x23, 0x47, 0xD2, 0xC4, 0x1D, 0x90, 0x75, 0x17, 0x14, 0xF3, 0x1B, 0xB9, 0x3A, 0x11, 0xA6, 0x04, 0x52, 0x2B, 0x6B, 0xCB, 0x01, 0xE4, 0x62, 0xB8, 0xD7, 0xE3, 0xB7, 0x1B, 0x6B, 0xF3, 0x20, 0x37, 0x24, 0x87, 0xFB, 0xF0, 0x46, 0x87, 0xC4, 0xAB, 0x88, 0xAC, 0x53, 0x37, 0xD8, 0x87, 0x0E, 0xBB, 0x39, 0x91, 0x1A, 0x8C, 0x90, 0xF8, 0xB4, 0x2A, 0xE2, 0xA9, 0x25, 0xAD, 0xE9, 0x8A, 0x0E, 0xEA, 0x4B, 0x49, 0xC2, 0xE9, 0x5B, 0x5A, 0x3C, 0xAC, 0xE2, 0xE9, 0x3B, 0x27, 0xE6, 0x60, 0x00, 0x49, 0x62, 0xAF, 
			0xED, 0x41, 0x7A, 0x8E, 0x95, 0xD2, 0xC2, 0x27, 0x04, 0x9F, 0xE9, 0xEE, 0x50, 0xB5, 0xDF, 0x17, 0x59, 0x46, 0x83, 0xED, 0xB7, 0xFB, 0xCE, 0xE4, 0x35, 0x6E, 0x85, 0x66, 0xA4, 0xF9, 0x11, 0xB9, 0x33, 0xAE, 0x84, 0x9B, 0x73, 0xCC, 0x5E, 0x1A, 0xBE, 0xC6, 0xF6, 0x71, 0x41, 0x07, 0x3F, 0xAF, 0x24, 0xA7, 0x37, 0x12, 0x65, 0x30, 0x87, 0x23, 0xAB, 0x6D, 0xB1, 0xF1, 0xE4, 0x71, 0xEA, 0x87, 0xF4, 0x9E, 0xC9, 0xCA, 0x4E, 0x34, 0xA4, 0x60, 0x80, 0xE5, 0x5E, 0x4A, 0xDF, 0x2B, 0xA5, 0x12, 0x5E, 0x7A, 0x76, 0x7E, 0x82, 0x8B, 0xF7, 0x55, 0x72, 0x79, 0xDC, 0x2A, 0x4C, 0xC1, 0x2B, 0xCA, 0xA5, 0x9A, 0x2D, 0x98, 
			0x6C, 0x51, 0x00, 0x95, 0x1F, 0x4C, 0x04, 0x6A, 0x44, 0x59, 0x7C, 0x5F, 0xE4, 0xA8, 0xAF, 0x66, 0x32, 0x29, 0x1D, 0x07, 0xD4, 0xAE, 0xC1, 0x9E, 0x45, 0x35, 0x6D, 0xE9, 0x7A, 0x92, 0x22, 0x4B, 0x0D, 0x4A, 0xA6, 0x53, 0x17, 0x31, 0x76, 0x1F, 0x60, 0x32, 0x84, 0x18, 0x6B, 0x90, 0xD3, 0x77, 0x9B, 0xF1, 0xF0, 0x95, 0x4D, 0x23, 0x95, 0x90, 0xF5, 0xDD, 0xFA, 0xC1, 0x3F, 0xB3, 0xDD, 0x5E, 0x54, 0xFC, 0x03, 0xFD, 0xDA, 0xC8, 0xEF, 0x78, 0x2E, 0x5F, 0xFF, 0x1D, 0xDB, 0xB7, 0xF8, 0xBF, 0x1A, 0x2A, 0x98, 0xD0, 0x06, 0x04, 0x9B, 0xE0, 0xFD, 0xFB, 0xA2, 0x2E, 0x83, 0x67, 0x38, 0x77, 0x9E, 0x2D, 0x7C, 0xB4, 
			0x95, 0xF2, 0x5F, 0xF8, 0x3B, 0x62, 0x41, 0x9C, 0xAA, 0xAA, 0xAB, 0xEB, 0x46, 0x78, 0x5D, 0xF6, 0x88, 0x6C, 0xED, 0x0F, 0x76, 0x3B, 0xD6, 0xFD, 0x13, 0xCE, 0x68, 0xDE, 0x0A, 0x33, 0xA6, 0x68, 0xB5, 0xCA, 0x64, 0x90, 0x16, 0x00, 0xCE, 0x97, 0xC5, 0xED, 0x57, 0xE1, 0x1D, 0xF2, 0xD3, 0xDC, 0xA7, 0x1B, 0xC3, 0xEE, 0x07, 0xBE, 0xE8, 0xC3, 0xE5, 0x77, 0x10, 0x00, 0x48, 0x7D, 0x4F, 0xEC, 0x3B, 0xEC, 0xEF, 0x0F, 0x86, 0xD6, 0x5E, 0xF0, 0x32, 0xE4, 0x3D, 0x4E, 0xB5, 0x48, 0xA7, 0x0E, 0x2C, 0xCA, 0x1C, 0x9E, 0xDA, 0x6F, 0xDD, 0x53, 0x12, 0x0A, 0xBF, 0x65, 0x61, 0xDD, 0xB3, 0xAE, 0x3E, 0xAA, 0x59, 0x8F, 
			0xFC, 0xCD, 0x0E, 0xD0, 0x7D, 0xB0, 0x49, 0x5C, 0x85, 0xF4, 0xEF, 0x77, 0xDA, 0x39, 0x9E, 0x21, 0xCE, 0x9B, 0x26, 0x81, 0x6C, 0x13, 0x09, 0x54, 0xBF, 0x2D, 0x20, 0xF6, 0x57, 0x74, 0x94, 0xE0, 0x99, 0x81, 0x08, 0x9E, 0x72, 0xDB, 0x95, 0xAC, 0xB3, 0x4E, 0x08, 0x66, 0x7C, 0x30, 0x3B, 0x30, 0x1F, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x04, 0x14, 0x08, 0x95, 0xCA, 0x53, 0x3F, 0xBA, 0xFC, 0x96, 0xB1, 0x48, 0x21, 0x06, 0xA4, 0x9E, 0x5A, 0xD6, 0x23, 0x60, 0x4F, 0x3C, 0x04, 0x14, 0x24, 0x98, 0x3B, 0xFC, 0x6E, 0x4B, 0x07, 0xB7, 0xFA, 0xC1, 0xFE, 0xD3, 0x15, 0xBB, 0x7B, 0x70, 0xA7, 0x97, 
			0xBC, 0x0A, 0x02, 0x02, 0x07, 0xD0 };

		static public byte[] farscape_strong_path_pfx = { 0x30, 0x82, 0x0B, 0x57, 0x02, 0x01, 0x03, 0x30, 0x82, 0x0B, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x0B, 0x04, 0x04, 0x82, 0x0B, 0x00, 0x30, 0x82, 0x0A, 0xFC, 0x30, 0x82, 0x03, 0x8D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0x7E, 0x04, 0x82, 0x03, 0x7A, 0x30, 0x82, 0x03, 0x76, 0x30, 0x82, 0x03, 0x72, 0x06, 0x0B, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x0A, 0x01, 0x02, 0xA0, 0x82, 0x02, 0xB6, 0x30, 0x82, 0x02, 0xB2, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 
			0x0E, 0x04, 0x08, 0x32, 0xC5, 0xBA, 0x13, 0xE8, 0x8D, 0xE0, 0x25, 0x02, 0x02, 0x07, 0xD0, 0x04, 0x82, 0x02, 0x90, 0x3C, 0xD2, 0x8F, 0x0F, 0x52, 0x6A, 0x97, 0xA2, 0x5D, 0xDC, 0xFC, 0x82, 0x3D, 0x95, 0xE1, 0x19, 0x02, 0x8E, 0x64, 0xC5, 0x7C, 0x64, 0xFF, 0xA3, 0x4E, 0x00, 0x5B, 0xC0, 0x3C, 0x7C, 0x53, 0xEA, 0xEA, 0xF9, 0xC8, 0xF6, 0xD5, 0xDF, 0x62, 0x94, 0x39, 0xA8, 0xF9, 0xAF, 0x46, 0xAE, 0x40, 0xF8, 0xBE, 0x5E, 0xF5, 0xAB, 0x7A, 0xB2, 0x5B, 0x9F, 0x2D, 0x6E, 0xBA, 0xF9, 0xBE, 0x99, 0x32, 0x4C, 0xBF, 0xC0, 0x2C, 0xFB, 0xC6, 0xBD, 0x24, 0x6E, 0x1B, 0x69, 0xF8, 0xB8, 0xA2, 0x3F, 0x0F, 0x7F, 0x03, 
			0x60, 0xC6, 0xA8, 0x10, 0x80, 0x7A, 0xEA, 0x16, 0xE2, 0x2E, 0xBA, 0x60, 0x0A, 0x76, 0x9A, 0x60, 0x5A, 0x50, 0x59, 0x06, 0xD6, 0x76, 0x2C, 0x73, 0x7B, 0x50, 0x6F, 0x55, 0x6E, 0x86, 0xED, 0xC5, 0x35, 0x2C, 0xEF, 0x9F, 0xC8, 0x81, 0xE5, 0xFF, 0xBE, 0xDD, 0x48, 0x20, 0x35, 0x1B, 0x9B, 0xCF, 0x77, 0x13, 0x21, 0x9D, 0xD6, 0x19, 0xFC, 0x08, 0x94, 0xB6, 0xB3, 0xF5, 0x03, 0xAE, 0x8F, 0x10, 0x14, 0x9C, 0x7E, 0x86, 0x43, 0x1D, 0x3D, 0x9D, 0xEA, 0x3A, 0xDD, 0x62, 0x53, 0xFE, 0x92, 0x1F, 0xD1, 0x9B, 0x0D, 0x18, 0x0E, 0xCD, 0x5F, 0xF7, 0xE9, 0x92, 0x5F, 0xFE, 0xF2, 0x14, 0x20, 0x46, 0x9C, 0xED, 0x93, 0x0B, 
			0x65, 0xE3, 0xDE, 0x41, 0x53, 0x55, 0x83, 0xEE, 0x83, 0x59, 0x61, 0x74, 0x96, 0x67, 0xDB, 0x26, 0x22, 0x0B, 0x19, 0x01, 0x53, 0xF6, 0xDC, 0x13, 0xD9, 0xF5, 0xCD, 0xB3, 0x98, 0xA0, 0x52, 0x65, 0x69, 0x7B, 0xED, 0x76, 0xED, 0xFA, 0xE0, 0xAD, 0x2F, 0xDC, 0xCA, 0x0D, 0xBA, 0x25, 0xBD, 0x13, 0x5D, 0xCA, 0xFC, 0x4A, 0x22, 0xD9, 0x1D, 0xAC, 0x7F, 0x25, 0x35, 0x0D, 0xF7, 0xB0, 0xD9, 0x36, 0x9F, 0x4F, 0x0D, 0x7C, 0x92, 0x93, 0xD0, 0xA5, 0x3F, 0x64, 0xC1, 0x26, 0x41, 0x73, 0xDC, 0x3E, 0x01, 0xCA, 0xB0, 0xD8, 0xF6, 0x6D, 0x83, 0x08, 0xA3, 0x3D, 0xCE, 0xEF, 0x29, 0x6B, 0x25, 0x3D, 0x65, 0xD9, 0xB0, 0xA3, 
			0xA4, 0xDC, 0xC8, 0x21, 0xDF, 0x5E, 0xE8, 0x5B, 0x5D, 0xCF, 0x41, 0x81, 0x28, 0xEE, 0x80, 0xDF, 0x87, 0x0B, 0x06, 0x09, 0x46, 0x61, 0xDC, 0x99, 0xF8, 0xC6, 0xD7, 0xA8, 0x4C, 0x59, 0x37, 0xB3, 0x54, 0x82, 0xE7, 0x4F, 0x36, 0x5D, 0x58, 0x8B, 0x26, 0xE8, 0xF7, 0x76, 0x87, 0x3A, 0x43, 0x61, 0x8B, 0x93, 0xAD, 0xDE, 0x35, 0xED, 0xC5, 0x5B, 0xEC, 0x49, 0x9B, 0xA3, 0x5A, 0xC7, 0x94, 0x50, 0x0E, 0x06, 0x2F, 0xC2, 0xF2, 0x28, 0x71, 0x52, 0xE1, 0x4B, 0x24, 0x97, 0x62, 0x2A, 0x28, 0x25, 0xC5, 0x43, 0x35, 0xF1, 0x9C, 0xDF, 0xFD, 0xAD, 0x7C, 0x7D, 0xED, 0x6F, 0xCC, 0x96, 0xBF, 0x93, 0x97, 0xBA, 0x2C, 0x44, 
			0xB7, 0xCD, 0x9C, 0x6E, 0x0D, 0x29, 0x7C, 0x19, 0xC9, 0xA0, 0x40, 0x94, 0x35, 0x66, 0xB6, 0x4C, 0xFB, 0xA8, 0x91, 0xC1, 0x89, 0x31, 0x6E, 0x76, 0x03, 0x94, 0x65, 0xF1, 0xA3, 0x2A, 0x1A, 0xD6, 0x99, 0xB9, 0xFF, 0x2E, 0xCF, 0xC3, 0xE7, 0x80, 0x28, 0x7B, 0xD5, 0x18, 0xA6, 0xD1, 0xEC, 0x1F, 0x28, 0x65, 0xB5, 0xB7, 0x32, 0xC9, 0x3A, 0x22, 0x56, 0x87, 0x61, 0x5C, 0xB0, 0xEC, 0x87, 0xA6, 0xA5, 0x23, 0x14, 0xBA, 0x0E, 0xD1, 0xE7, 0xA2, 0x74, 0x94, 0xC9, 0x74, 0x65, 0x9C, 0x5D, 0xAF, 0x5B, 0xDD, 0x3A, 0x8B, 0x2E, 0x44, 0xDF, 0x8C, 0x7F, 0x10, 0x83, 0x3C, 0xFC, 0x2C, 0x48, 0x4D, 0xF0, 0x20, 0xB8, 0xF8, 
			0x63, 0x37, 0x83, 0xB1, 0xDB, 0xA3, 0x51, 0x75, 0xF3, 0xA6, 0x77, 0xAD, 0x4A, 0xAF, 0xF6, 0x0F, 0x57, 0xAB, 0x18, 0x68, 0xD8, 0x42, 0xB1, 0xC3, 0x4D, 0xE9, 0xAA, 0x1A, 0xD5, 0xB9, 0xBD, 0xEB, 0xDB, 0x5B, 0x04, 0x95, 0xC2, 0x4E, 0xE2, 0x74, 0x99, 0x6E, 0x42, 0xF0, 0xCA, 0xC5, 0xA1, 0x71, 0x90, 0xFF, 0xF7, 0x08, 0x1E, 0xDA, 0xD7, 0x53, 0x33, 0x12, 0x0A, 0xC9, 0x10, 0x04, 0x60, 0xDA, 0xC2, 0xC8, 0x73, 0x9C, 0x10, 0x03, 0x20, 0x5A, 0xC2, 0xC5, 0x2E, 0x83, 0x88, 0x72, 0x25, 0xA3, 0x98, 0xC0, 0x35, 0x56, 0x16, 0x7B, 0x8F, 0x17, 0x92, 0x72, 0xF8, 0x8A, 0x4C, 0x65, 0x79, 0x7A, 0xA5, 0xAF, 0xCB, 0x03, 
			0x0A, 0xAA, 0xBC, 0xBB, 0xF8, 0x30, 0x16, 0xBA, 0x1D, 0x42, 0x6F, 0xF8, 0xAA, 0xC4, 0x3A, 0x89, 0x29, 0x22, 0x7F, 0x22, 0x1F, 0x87, 0xFE, 0x34, 0x06, 0x6C, 0x0E, 0x13, 0xD0, 0x43, 0x43, 0x59, 0x4F, 0x59, 0xAA, 0x2B, 0x1E, 0xDA, 0x96, 0x06, 0x38, 0x3E, 0x18, 0x97, 0xCF, 0x1D, 0xFE, 0x43, 0xA7, 0x8C, 0xB1, 0x0D, 0x40, 0x9B, 0xC6, 0xDD, 0x79, 0x97, 0xD0, 0x0B, 0x67, 0x0E, 0x2C, 0x2D, 0xBE, 0x4F, 0x9F, 0xBC, 0x6D, 0xD1, 0xAC, 0xA3, 0xB6, 0x38, 0x53, 0x31, 0x81, 0xA8, 0x30, 0x0D, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x02, 0x31, 0x00, 0x30, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 
			0x86, 0xF7, 0x0D, 0x01, 0x09, 0x15, 0x31, 0x06, 0x04, 0x04, 0x01, 0x00, 0x00, 0x00, 0x30, 0x17, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x14, 0x31, 0x0A, 0x1E, 0x08, 0x00, 0x4D, 0x00, 0x6F, 0x00, 0x6E, 0x00, 0x6F, 0x30, 0x69, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x01, 0x31, 0x5C, 0x1E, 0x5A, 0x00, 0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20, 0x00, 0x52, 0x00, 0x53, 0x00, 0x41, 0x00, 0x20, 0x00, 0x53, 0x00, 0x43, 0x00, 0x68, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x6E, 0x00, 0x65, 0x00, 0x6C, 
			0x00, 0x20, 0x00, 0x43, 0x00, 0x72, 0x00, 0x79, 0x00, 0x70, 0x00, 0x74, 0x00, 0x6F, 0x00, 0x67, 0x00, 0x72, 0x00, 0x61, 0x00, 0x70, 0x00, 0x68, 0x00, 0x69, 0x00, 0x63, 0x00, 0x20, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x76, 0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x72, 0x30, 0x82, 0x07, 0x67, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x06, 0xA0, 0x82, 0x07, 0x58, 0x30, 0x82, 0x07, 0x54, 0x02, 0x01, 0x00, 0x30, 0x82, 0x07, 0x4D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 
			0x06, 0x30, 0x0E, 0x04, 0x08, 0x4E, 0xEC, 0x16, 0x2C, 0x55, 0x4B, 0x1B, 0x3E, 0x02, 0x02, 0x07, 0xD0, 0x80, 0x82, 0x07, 0x20, 0x8A, 0xE2, 0x16, 0x78, 0xEB, 0x9F, 0xF6, 0x25, 0x67, 0x15, 0x21, 0x01, 0x86, 0xC3, 0x2E, 0x27, 0x02, 0xF4, 0x0B, 0x98, 0xD8, 0xF2, 0x44, 0xBB, 0x1A, 0xD4, 0xF2, 0xA6, 0xB8, 0x6B, 0x7B, 0x05, 0xD4, 0x57, 0x3C, 0x77, 0xD5, 0x0C, 0xDC, 0x6B, 0x1A, 0xC4, 0xEA, 0x32, 0xFC, 0x28, 0x48, 0x67, 0xA2, 0xC2, 0x20, 0xCA, 0x58, 0x95, 0x45, 0xAA, 0xE1, 0x69, 0x4B, 0x7D, 0xFD, 0x33, 0x43, 0xF3, 0xD2, 0xDF, 0xEA, 0xF3, 0xAD, 0x7D, 0x8F, 0x05, 0x5A, 0xAB, 0x7B, 0x6E, 0x6C, 0x6F, 0xD7, 
			0x49, 0xFC, 0x16, 0x1F, 0x30, 0x5D, 0x68, 0xB9, 0x78, 0x65, 0x12, 0x7C, 0x16, 0x14, 0x95, 0x3C, 0xA3, 0xE9, 0xEF, 0x56, 0x72, 0x07, 0x9B, 0x39, 0xA7, 0xA9, 0x52, 0x68, 0x47, 0x18, 0xED, 0x83, 0xC5, 0xFC, 0x4F, 0xC8, 0xFF, 0x86, 0xA5, 0x85, 0xBF, 0xBF, 0x38, 0xBA, 0x88, 0xA5, 0x91, 0x85, 0xE8, 0x68, 0x04, 0x71, 0x89, 0x8A, 0x0E, 0x33, 0x5A, 0x57, 0x65, 0xCC, 0x3A, 0x1D, 0xE8, 0x29, 0x1F, 0xF3, 0x5E, 0xAE, 0xBE, 0x94, 0x60, 0x79, 0x54, 0x40, 0xF6, 0x15, 0x32, 0xD6, 0x6C, 0xEC, 0x8D, 0x1A, 0x63, 0x98, 0x69, 0x6F, 0x93, 0xB2, 0x6C, 0x3E, 0xF3, 0x35, 0x42, 0x08, 0x2E, 0x7E, 0x44, 0x73, 0xC6, 0xE8, 
			0x74, 0x07, 0xBF, 0xD3, 0x75, 0xDE, 0x87, 0x93, 0x8D, 0x4C, 0x3B, 0xB8, 0x23, 0x91, 0x2C, 0x62, 0x99, 0xA8, 0xD6, 0xA6, 0x05, 0xB9, 0xE9, 0x8F, 0x9F, 0x86, 0xD0, 0xE0, 0xF5, 0xE6, 0x1C, 0x92, 0x8D, 0xC9, 0x7A, 0xBA, 0xF8, 0x05, 0x7C, 0x6F, 0xFB, 0x7F, 0x62, 0x4B, 0x8C, 0x9B, 0xC5, 0xDB, 0x24, 0xD0, 0x7D, 0xC6, 0xDF, 0xBB, 0x43, 0xB8, 0x31, 0x3C, 0x52, 0x69, 0xC3, 0xA9, 0x4B, 0x6C, 0xA5, 0x5D, 0xF0, 0x25, 0x90, 0x9B, 0xD8, 0xE4, 0xC2, 0x4C, 0x71, 0x68, 0x17, 0x79, 0x21, 0xB7, 0x14, 0x56, 0xC2, 0x31, 0xB5, 0xEF, 0x4D, 0x86, 0x83, 0x24, 0x78, 0x01, 0x6D, 0x41, 0x19, 0x5B, 0xD3, 0x78, 0xC0, 0xE7, 
			0xF4, 0xEE, 0x86, 0xEC, 0x4D, 0x19, 0xB4, 0x2E, 0x65, 0x4C, 0x47, 0xFC, 0x7D, 0xDF, 0x56, 0xE1, 0xC3, 0x65, 0xF2, 0x30, 0xB5, 0x13, 0x93, 0xAD, 0x31, 0x56, 0xF3, 0x77, 0x4D, 0x9B, 0x6A, 0xA6, 0xCC, 0xF0, 0x20, 0x34, 0x11, 0xE7, 0xBD, 0xB1, 0xAD, 0xB9, 0xCB, 0xF3, 0x48, 0x6B, 0xB9, 0xF6, 0xB1, 0x7F, 0x9B, 0x61, 0x4C, 0x09, 0x05, 0xC1, 0x8D, 0xA7, 0x2B, 0xF0, 0xE5, 0x6C, 0x7D, 0xEB, 0xE3, 0x83, 0x6B, 0xA0, 0x77, 0x98, 0x6C, 0x5D, 0x59, 0x5B, 0xE2, 0x69, 0xC6, 0x97, 0x15, 0x60, 0x7A, 0x69, 0x7F, 0xC2, 0xF6, 0xF6, 0xAD, 0x33, 0xAE, 0x7D, 0x06, 0xA1, 0xF4, 0x5B, 0xA4, 0x17, 0xC3, 0xA5, 0xB5, 0xFA, 
			0x60, 0xD1, 0x58, 0x30, 0x1F, 0x81, 0x15, 0x49, 0x05, 0x6D, 0x54, 0xD8, 0xE8, 0x9E, 0xBF, 0x61, 0xD2, 0xD0, 0x8A, 0x95, 0xBD, 0x1C, 0x05, 0x6B, 0x15, 0xDC, 0x29, 0x80, 0xB1, 0x91, 0x66, 0xCE, 0x57, 0xCE, 0xFD, 0x5A, 0x3F, 0x18, 0xB9, 0x87, 0x32, 0x92, 0x6E, 0xA3, 0x42, 0x20, 0xC4, 0x36, 0xBD, 0x7E, 0xE2, 0x38, 0xF3, 0xDD, 0x50, 0x00, 0x28, 0x8C, 0x78, 0x2C, 0x94, 0xEC, 0x0F, 0xFE, 0xC2, 0x6F, 0x4D, 0x09, 0xC6, 0xE0, 0xCD, 0x50, 0xC6, 0x3B, 0x1D, 0x81, 0x2C, 0xE5, 0xEA, 0xF8, 0x31, 0x17, 0xCC, 0x37, 0x09, 0xBC, 0x7F, 0x19, 0x46, 0x4D, 0x83, 0x4A, 0xB6, 0x6E, 0x02, 0x10, 0x03, 0xAF, 0xEF, 0xE4, 
			0x02, 0x4A, 0x11, 0xC6, 0x11, 0xA6, 0xFC, 0xC3, 0x0E, 0x62, 0xAB, 0x98, 0x71, 0x32, 0x86, 0xA6, 0x22, 0x1B, 0x5F, 0x2C, 0x18, 0x78, 0x0E, 0x21, 0xA5, 0x5C, 0xC4, 0x3D, 0x06, 0x9A, 0x1E, 0x03, 0xAA, 0x7A, 0x49, 0x04, 0x0D, 0x82, 0xAF, 0xDD, 0x02, 0x70, 0x2E, 0x33, 0x10, 0x6F, 0x6A, 0xBF, 0xE4, 0x2F, 0xAA, 0x54, 0x8A, 0x0F, 0xCC, 0x85, 0xE9, 0xD9, 0x7E, 0x43, 0xCF, 0xDC, 0x32, 0x35, 0xA8, 0xEC, 0x87, 0x3D, 0x08, 0xB1, 0x63, 0xE3, 0x48, 0xDC, 0xB9, 0xDB, 0x2C, 0xA6, 0x9F, 0xB9, 0x7D, 0x90, 0xCA, 0xD1, 0x03, 0x76, 0xE6, 0x6D, 0xAD, 0x51, 0x10, 0xED, 0xA1, 0x7E, 0x06, 0x72, 0xC2, 0x12, 0x93, 0x7E, 
			0x7A, 0xF0, 0x84, 0xFE, 0x1C, 0x2A, 0xB0, 0xF1, 0x13, 0x2F, 0x68, 0x2B, 0x66, 0xB8, 0xAD, 0x14, 0x28, 0x65, 0x85, 0xC4, 0x92, 0x0F, 0x10, 0x3F, 0x1A, 0xCD, 0x6D, 0x74, 0x39, 0x5E, 0x9D, 0x6F, 0x11, 0xD3, 0xB4, 0xDA, 0xB6, 0x1A, 0x1D, 0x73, 0x29, 0xF3, 0x4A, 0x4F, 0x77, 0x19, 0x53, 0x59, 0x38, 0xF5, 0x58, 0x77, 0xE5, 0x45, 0xBA, 0x03, 0xD4, 0x9C, 0x1B, 0x11, 0x9C, 0xA2, 0x91, 0x80, 0xC7, 0x2C, 0xEB, 0x0C, 0x8D, 0x7E, 0xA1, 0x94, 0x1B, 0x7B, 0x74, 0x0F, 0x21, 0xB5, 0xE5, 0xBA, 0xEA, 0x14, 0x65, 0xAA, 0xFB, 0x5F, 0x91, 0x7A, 0x73, 0xBB, 0x04, 0x4E, 0x74, 0xFE, 0x24, 0xDF, 0x63, 0x18, 0xDF, 0xC2, 
			0x74, 0x5F, 0x4C, 0x9C, 0x9D, 0x0C, 0x94, 0x58, 0x89, 0x3F, 0x77, 0x63, 0x03, 0x77, 0x9C, 0x6E, 0xDE, 0x01, 0x13, 0x62, 0xEF, 0xCF, 0x57, 0xAB, 0x15, 0x22, 0xB6, 0x40, 0xFF, 0x0A, 0xFC, 0x91, 0xB8, 0x52, 0xFA, 0xCC, 0x7C, 0xEE, 0xB8, 0xA3, 0x7C, 0x8B, 0x3C, 0x04, 0x2E, 0xAB, 0x00, 0x78, 0x95, 0xD5, 0x0B, 0xC0, 0xF1, 0x5D, 0x52, 0x25, 0x92, 0x6B, 0x99, 0x71, 0x2E, 0x2B, 0x3A, 0x15, 0x2A, 0x15, 0xE8, 0xAC, 0x79, 0x71, 0xAE, 0xC7, 0xDA, 0x7C, 0x9D, 0x7E, 0xD5, 0x75, 0x34, 0xA6, 0x76, 0x46, 0xB1, 0x87, 0x94, 0xAD, 0x1F, 0x75, 0x74, 0x69, 0xE7, 0x26, 0xD6, 0xB1, 0x32, 0xFE, 0x0A, 0xCA, 0x3C, 0x54, 
			0x7D, 0x24, 0x95, 0xE1, 0xFD, 0x37, 0xE6, 0x34, 0xA5, 0x8B, 0x42, 0x88, 0x71, 0x22, 0xFE, 0x94, 0x42, 0xD5, 0xF8, 0xEE, 0x1E, 0x57, 0x2A, 0xD4, 0xB7, 0xC8, 0x90, 0xA2, 0x4A, 0xE0, 0xB7, 0xB5, 0x37, 0x72, 0x59, 0xF5, 0xA4, 0x8D, 0x3C, 0xFE, 0x3B, 0xD2, 0x8F, 0xEA, 0xBB, 0x83, 0x77, 0x78, 0xDA, 0x6E, 0xF3, 0x35, 0x03, 0x2E, 0xF1, 0x95, 0x6E, 0x91, 0x25, 0x9B, 0x11, 0x35, 0xBE, 0x08, 0xBB, 0xCB, 0x5C, 0x34, 0x1F, 0x22, 0x02, 0xE3, 0x0E, 0x5E, 0xC4, 0xF8, 0xD1, 0x6C, 0xF9, 0x87, 0x67, 0x0B, 0x33, 0xBD, 0xD3, 0xF0, 0x25, 0x5C, 0x68, 0xD5, 0xD0, 0xA7, 0x4E, 0x2E, 0x8B, 0xB6, 0x45, 0xD9, 0x8A, 0x33, 
			0x80, 0x08, 0xBD, 0x8D, 0x17, 0xFD, 0x4C, 0x4C, 0xFD, 0x06, 0x85, 0x5A, 0x4A, 0xDF, 0x4C, 0xD7, 0x16, 0x78, 0xB0, 0x0F, 0xA9, 0x35, 0x4A, 0x71, 0x9F, 0x10, 0xE1, 0x26, 0x09, 0x03, 0x29, 0xE3, 0x11, 0x2A, 0xF4, 0x4F, 0x02, 0xBD, 0x29, 0x54, 0xA8, 0x8F, 0xFE, 0xC4, 0x98, 0xD4, 0x28, 0x86, 0x00, 0x37, 0xC9, 0x2C, 0xCF, 0xB6, 0xA6, 0x0B, 0x98, 0x22, 0xE6, 0xAA, 0x8A, 0x51, 0x45, 0xAF, 0xBB, 0x63, 0x25, 0x42, 0x7F, 0x3E, 0x8C, 0x34, 0x30, 0x86, 0x39, 0x9D, 0x58, 0xC1, 0xC0, 0x1F, 0xA8, 0xEE, 0x2A, 0x42, 0x4C, 0x0C, 0xCA, 0x62, 0xCA, 0xD1, 0xA6, 0x26, 0xA1, 0xB9, 0x3D, 0x6C, 0xDE, 0x72, 0x71, 0xC5, 
			0xAB, 0xE2, 0xD6, 0x67, 0x36, 0xDD, 0xD9, 0xFE, 0x32, 0x43, 0x3A, 0x33, 0x0E, 0x4C, 0x25, 0x5A, 0x25, 0x77, 0xFD, 0x0C, 0x7E, 0xE3, 0x8F, 0x9A, 0x12, 0x7F, 0x09, 0x15, 0x92, 0x5D, 0x21, 0xB8, 0x6F, 0x6C, 0xBC, 0xCC, 0x37, 0xEF, 0x32, 0x7D, 0xAF, 0x90, 0x47, 0xAF, 0x54, 0xD9, 0xD6, 0x9E, 0x50, 0xB6, 0x7D, 0x1E, 0x1C, 0xD8, 0x8D, 0x72, 0x64, 0x86, 0x93, 0xDB, 0x2E, 0xCE, 0x1E, 0x95, 0xA8, 0xBF, 0x6B, 0x22, 0x9B, 0x16, 0x72, 0x2A, 0xA2, 0x53, 0x37, 0x8F, 0x81, 0xCE, 0x0B, 0x8E, 0x27, 0x23, 0x73, 0x7C, 0xE7, 0x15, 0x32, 0x20, 0xA1, 0xD4, 0xE9, 0xCD, 0x10, 0xC2, 0xC0, 0xB9, 0x92, 0x8E, 0x8E, 0x0B, 
			0x50, 0x79, 0x15, 0xFE, 0x64, 0x90, 0x85, 0xB7, 0x65, 0xAD, 0x6A, 0x80, 0xF3, 0x43, 0x3D, 0x1C, 0xCC, 0xFB, 0xC6, 0x8C, 0x83, 0x23, 0xE3, 0xA7, 0x4C, 0xCE, 0x88, 0x8C, 0x99, 0xED, 0xCE, 0xC8, 0x29, 0xA7, 0x45, 0xA7, 0x3A, 0xC2, 0x24, 0x6C, 0xD7, 0x47, 0x37, 0x06, 0xEE, 0xAE, 0xD7, 0xC7, 0xE6, 0x09, 0x2D, 0xAC, 0x13, 0x5A, 0xE1, 0x83, 0x72, 0x65, 0x60, 0x21, 0x97, 0x1A, 0x56, 0x6B, 0xFA, 0xE4, 0x4B, 0x71, 0x6E, 0x99, 0x1F, 0x09, 0xBF, 0x97, 0x23, 0xC2, 0x1A, 0xEE, 0xEA, 0x82, 0x07, 0x9C, 0x8F, 0x7C, 0x80, 0xBA, 0xC3, 0x68, 0xFC, 0x42, 0xE4, 0x50, 0x80, 0x71, 0x48, 0x74, 0x6A, 0x10, 0x23, 0x7E, 
			0x44, 0xD6, 0xA2, 0xB9, 0x3E, 0x95, 0xA0, 0x0E, 0x0B, 0xA6, 0xA8, 0x45, 0xD8, 0xE5, 0x18, 0x17, 0x63, 0xB9, 0xE2, 0x66, 0x88, 0xA4, 0xEC, 0x4A, 0x88, 0x3F, 0x54, 0x2B, 0x7D, 0x89, 0xC3, 0x73, 0x84, 0x83, 0x76, 0x62, 0x6B, 0xD6, 0xDA, 0x76, 0x43, 0xD7, 0xCA, 0xE3, 0x78, 0x45, 0x5A, 0x46, 0xA7, 0x6D, 0x0E, 0x42, 0x41, 0x80, 0xDB, 0x03, 0xDD, 0x11, 0x5F, 0xD1, 0x0F, 0x3F, 0x9C, 0xD3, 0xBB, 0xEF, 0xCC, 0x8C, 0x10, 0x96, 0x6B, 0xA0, 0x86, 0xB9, 0x32, 0xFD, 0xB4, 0xA8, 0x55, 0x90, 0x57, 0x80, 0x30, 0x9A, 0xB3, 0xC0, 0xE1, 0xDE, 0xEC, 0x56, 0x3A, 0x16, 0x2F, 0x9B, 0x04, 0x00, 0x4C, 0x84, 0xC1, 0x42, 
			0xBE, 0x3D, 0xC8, 0xD4, 0x8B, 0x86, 0x0B, 0x0A, 0x73, 0x32, 0x50, 0x18, 0x73, 0x3E, 0x96, 0x70, 0xB1, 0x58, 0x96, 0x67, 0x60, 0x36, 0xFA, 0x9F, 0xC5, 0x22, 0xBD, 0x29, 0xEB, 0xFB, 0xA9, 0xDA, 0x16, 0xF0, 0x1C, 0xDE, 0xEB, 0x6E, 0xB7, 0x7D, 0x2F, 0x13, 0xBE, 0x13, 0x34, 0xB3, 0x70, 0x49, 0xEF, 0x15, 0x96, 0xEF, 0x55, 0xF4, 0x39, 0x34, 0x2F, 0xB4, 0x3C, 0x26, 0x2B, 0xEC, 0x66, 0x20, 0x29, 0x14, 0xB6, 0x70, 0xCC, 0x40, 0x11, 0xC1, 0xC4, 0x55, 0xDF, 0x44, 0x62, 0xEA, 0x86, 0x26, 0x7F, 0x70, 0x91, 0x37, 0x1E, 0x35, 0x6A, 0x29, 0x29, 0x1D, 0x4A, 0x38, 0x74, 0x4D, 0x19, 0xF2, 0x8B, 0x02, 0x51, 0xF2, 
			0x89, 0xE7, 0xBE, 0xEA, 0xCC, 0x3C, 0x7F, 0x94, 0x14, 0xB1, 0x99, 0x99, 0xCB, 0x19, 0xA8, 0x20, 0x1B, 0x5C, 0xEE, 0xAD, 0x5E, 0x41, 0xA3, 0x13, 0x63, 0xD9, 0x3C, 0x6B, 0x7E, 0x60, 0xBB, 0x26, 0x59, 0x11, 0xD1, 0x85, 0x00, 0x08, 0x68, 0xDC, 0xF3, 0xCD, 0x55, 0xAC, 0x0E, 0xC3, 0x77, 0xDC, 0xED, 0x5A, 0x83, 0x51, 0x4C, 0xCC, 0xAE, 0xBA, 0xF7, 0x60, 0x47, 0x61, 0x01, 0xCC, 0x31, 0x46, 0xF5, 0x2A, 0x06, 0x39, 0x15, 0xCD, 0x1C, 0x2C, 0x32, 0x79, 0x2E, 0x1E, 0x73, 0x80, 0xA6, 0x91, 0x7B, 0xCD, 0xFE, 0x0F, 0x8B, 0x86, 0x91, 0xD3, 0xD2, 0x07, 0xB8, 0x48, 0x1A, 0x63, 0x41, 0xE7, 0x62, 0xDF, 0x1E, 0x59, 
			0x9B, 0xFA, 0xC8, 0x25, 0xC3, 0x70, 0x96, 0x8E, 0xEB, 0xA7, 0x5D, 0x0D, 0x6C, 0xBA, 0xB1, 0xDD, 0x8F, 0xB7, 0xA8, 0xBF, 0xBD, 0xBE, 0xFF, 0x86, 0x8A, 0x31, 0x11, 0xAF, 0x33, 0x6A, 0x1E, 0x11, 0xDF, 0x06, 0x0C, 0x16, 0x98, 0x9E, 0x5B, 0xE3, 0x4C, 0x31, 0xF4, 0x5F, 0x61, 0xC0, 0x22, 0xAC, 0x80, 0xA7, 0x05, 0xAA, 0x48, 0x77, 0xCE, 0x2F, 0x2F, 0x93, 0xA6, 0x19, 0x12, 0x6C, 0x40, 0x24, 0xC0, 0xF1, 0xF9, 0xC1, 0x13, 0x99, 0x2A, 0xE7, 0x12, 0xB2, 0xA4, 0x67, 0xCD, 0xC6, 0xA7, 0xE2, 0x59, 0x20, 0x71, 0xB8, 0x5D, 0x3C, 0xAC, 0x8A, 0x30, 0x00, 0xDE, 0xAB, 0x97, 0x94, 0x7E, 0x72, 0xF0, 0xDA, 0x6D, 0x9D, 
			0xDF, 0x89, 0x53, 0x06, 0x93, 0x6F, 0xA4, 0x90, 0x30, 0xFA, 0xD3, 0x26, 0xE6, 0x24, 0xBF, 0x3E, 0x22, 0xA2, 0x46, 0x55, 0xF6, 0x6A, 0x16, 0x5D, 0x3E, 0x17, 0x7E, 0x13, 0x02, 0x9C, 0x2D, 0x07, 0x79, 0x58, 0x00, 0x32, 0xD2, 0x10, 0x93, 0x85, 0x98, 0x2D, 0xF8, 0x5D, 0x5B, 0x74, 0xB0, 0x7E, 0x4B, 0x57, 0x64, 0xEA, 0x66, 0xDF, 0xE1, 0x04, 0xCC, 0x35, 0x9F, 0x0E, 0x46, 0x8A, 0xFB, 0x1C, 0x0D, 0x37, 0x8D, 0xC0, 0x13, 0x67, 0xAE, 0xC0, 0xD2, 0x2E, 0x9B, 0xB4, 0x16, 0x1B, 0xA4, 0x9A, 0x23, 0x18, 0xD2, 0x58, 0x9E, 0x3F, 0x4B, 0x83, 0x0D, 0x4A, 0xE4, 0xF3, 0xB3, 0xFC, 0x0A, 0xF9, 0xA5, 0x01, 0x9C, 0x44, 
			0x89, 0x53, 0xBD, 0x10, 0xAA, 0x1E, 0xC6, 0x41, 0x7C, 0x71, 0xCA, 0x47, 0x79, 0x60, 0x95, 0x2E, 0xAB, 0xF6, 0x17, 0xBE, 0x1D, 0xF1, 0x05, 0xAD, 0x76, 0xD8, 0xBB, 0xF2, 0xBB, 0x36, 0xEE, 0xCD, 0xAA, 0xBC, 0xBA, 0xBE, 0x94, 0x42, 0x9A, 0xA8, 0xEE, 0xE5, 0x93, 0x5C, 0x3C, 0x23, 0xE2, 0x75, 0xD3, 0x59, 0xB4, 0xB3, 0xC4, 0x01, 0xBB, 0xC9, 0x05, 0xFE, 0xFA, 0x1B, 0x31, 0x7A, 0x15, 0x29, 0x91, 0xE6, 0xB9, 0xB2, 0x1E, 0xF9, 0x36, 0xC5, 0x17, 0xB9, 0x9D, 0x32, 0x0D, 0x0D, 0x80, 0x4A, 0x0D, 0x14, 0xB4, 0xDF, 0x89, 0x3A, 0x4B, 0xCA, 0x4B, 0xDE, 0xF8, 0xC0, 0x1B, 0xA6, 0x6D, 0xD9, 0x19, 0xFB, 0x0E, 0x02, 
			0x22, 0x82, 0x8E, 0x25, 0x8E, 0x9F, 0xD5, 0x7B, 0xB6, 0x14, 0x15, 0x08, 0x61, 0xCD, 0xA6, 0xB3, 0x0F, 0xC8, 0xE8, 0xF9, 0xEE, 0xDB, 0xAF, 0x25, 0x40, 0x01, 0x85, 0x0E, 0x89, 0x0D, 0x0D, 0x85, 0xA6, 0xA3, 0xC9, 0x65, 0x64, 0x8A, 0xD1, 0xF7, 0x55, 0xF9, 0x31, 0x24, 0x41, 0x30, 0x3B, 0x30, 0x1F, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x04, 0x14, 0xD4, 0x77, 0x3F, 0x92, 0xB6, 0xA6, 0x1B, 0xD5, 0x7F, 0xC1, 0x1B, 0x93, 0xB3, 0x6E, 0x87, 0x67, 0x80, 0xE7, 0x4C, 0x7C, 0x04, 0x14, 0x58, 0x4C, 0x92, 0xDD, 0x0B, 0xEF, 0x09, 0x35, 0xE0, 0x06, 0x33, 0x54, 0xB2, 0xA8, 0x8E, 0x0C, 0x60, 0x67, 
			0xB5, 0x0F, 0x02, 0x02, 0x07, 0xD0 };

		static public byte[] cert = {0x30,0x82,0x01,0xFF,0x30,0x82,0x01,0x6C,0x02,0x05,0x02,0x72,0x00,0x06,0xE8,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x30,0x5F,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x0A,0x13,0x17,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x2E,0x30,0x2C,0x06,0x03,0x55,0x04,0x0B,0x13,0x25,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,
			0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x37,0x5A,0x17,0x0D,0x39,0x37,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x36,0x5A,0x30,0x61,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x43,0x61,0x6C,0x69,0x66,0x6F,0x72,0x6E,0x69,0x61,0x31,0x14,0x30,0x12,0x06,0x03,0x55,0x04,0x0A,0x13,0x0B,0x43,0x6F,0x6D,0x6D,0x65,0x72,0x63,0x65,0x4E,0x65,0x74,0x31,0x27,0x30,0x25,0x06,0x03,0x55,0x04,0x0B,0x13,0x1E,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,
			0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x70,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x5F,0x00,0x30,0x5C,0x02,0x55,0x2D,0x58,0xE9,0xBF,0xF0,0x31,0xCD,0x79,0x06,0x50,0x5A,0xD5,0x9E,0x0E,0x2C,0xE6,0xC2,0xF7,0xF9,0xD2,0xCE,0x55,0x64,0x85,0xB1,0x90,0x9A,0x92,0xB3,0x36,0xC1,0xBC,0xEA,0xC8,0x23,0xB7,0xAB,0x3A,0xA7,0x64,0x63,0x77,0x5F,0x84,0x22,0x8E,0xE5,0xB6,0x45,0xDD,0x46,0xAE,0x0A,0xDD,0x00,0xC2,0x1F,0xBA,0xD9,0xAD,0xC0,0x75,0x62,0xF8,0x95,0x82,0xA2,0x80,0xB1,0x82,0x69,0xFA,0xE1,0xAF,
			0x7F,0xBC,0x7D,0xE2,0x7C,0x76,0xD5,0xBC,0x2A,0x80,0xFB,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x03,0x7E,0x00,0x54,0x20,0x67,0x12,0xBB,0x66,0x14,0xC3,0x26,0x6B,0x7F,0xDA,0x4A,0x25,0x4D,0x8B,0xE0,0xFD,0x1E,0x53,0x6D,0xAC,0xA2,0xD0,0x89,0xB8,0x2E,0x90,0xA0,0x27,0x43,0xA4,0xEE,0x4A,0x26,0x86,0x40,0xFF,0xB8,0x72,0x8D,0x1E,0xE7,0xB7,0x77,0xDC,0x7D,0xD8,0x3F,0x3A,0x6E,0x55,0x10,0xA6,0x1D,0xB5,0x58,0xF2,0xF9,0x0F,0x2E,0xB4,0x10,0x55,0x48,0xDC,0x13,0x5F,0x0D,0x08,0x26,0x88,0xC9,0xAF,0x66,0xF2,0x2C,0x9C,0x6F,0x3D,0xC3,0x2B,0x69,0x28,0x89,0x40,
			0x6F,0x8F,0x35,0x3B,0x9E,0xF6,0x8E,0xF1,0x11,0x17,0xFB,0x0C,0x98,0x95,0xA1,0xC2,0xBA,0x89,0x48,0xEB,0xB4,0x06,0x6A,0x22,0x54,0xD7,0xBA,0x18,0x3A,0x48,0xA6,0xCB,0xC2,0xFD,0x20,0x57,0xBC,0x63,0x1C };

		private byte[] GetFile (string filename) 
		{
			byte[] data = null;
			using (FileStream fs = File.OpenRead (filename)) {
				data = new byte [fs.Length];
				fs.Read (data, 0, data.Length);
				fs.Close ();
			}
			return data;
		}

		[TearDown]
		public void TearDown ()
		{
			PKCS12.MaximumPasswordLength = Int32.MaxValue;
		}

		[Test]
		public void Farscape_NoPassword () 
		{
			// exported from MS certificate manager WITHOUT password
			PKCS12 p12 = new PKCS12 (farscape_nopwd_pfx);
			RSA rsa = (RSA) p12.Keys [0];
			X509Certificate x509 = p12.Certificates [0];
			AssertEquals (rsa.ToXmlString (false), x509.RSA.ToXmlString (false));
		}

		[Test]
		public void Farscape_NoPassword_Explicit_Null () 
		{
			// exported from MS certificate manager WITHOUT password
			PKCS12 p12 = new PKCS12 (farscape_nopwd_pfx, (string)null);
			RSA rsa = (RSA) p12.Keys [0];
			X509Certificate x509 = p12.Certificates [0];
			AssertEquals (rsa.ToXmlString (false), x509.RSA.ToXmlString (false));
		}

		[Test]
		public void Farscape_NoPassword_Explicit_Empty () 
		{
			// exported from MS certificate manager WITHOUT password
			PKCS12 p12 = new PKCS12 (farscape_nopwd_pfx, String.Empty);
			RSA rsa = (RSA) p12.Keys [0];
			X509Certificate x509 = p12.Certificates [0];
			AssertEquals (rsa.ToXmlString (false), x509.RSA.ToXmlString (false));
		}

		[Test]
		public void Farscape_Weak () 
		{
			// exported from MS certificate manager WITHOUT strong encryption
			PKCS12 p12 = new PKCS12 (farscape_pfx, "farscape");
			RSA rsa = (RSA) p12.Keys [0];
			X509Certificate x509 = p12.Certificates [0];
			AssertEquals (rsa.ToXmlString (false), x509.RSA.ToXmlString (false));
		}

		[Test]
		public void Farscape_Weak_Path () 
		{
			// exported from MS certificate manager WITHOUT strong encryption
			// and including the certificate path
			PKCS12 p12 = new PKCS12 (farscape_path_pfx, "farscape");
			RSA rsa = (RSA) p12.Keys [0];
			X509Certificate x509 = p12.Certificates [0];
			AssertEquals (rsa.ToXmlString (false), x509.RSA.ToXmlString (false));
		}

		[Test]
		public void Farscape_Strong () 
		{
			// exported from MS certificate manager with strong encryption
			PKCS12 p12 = new PKCS12 (farscape_strong_pfx, "farscape");
			RSA rsa = (RSA) p12.Keys [0];
			X509Certificate x509 = p12.Certificates [0];
			AssertEquals (rsa.ToXmlString (false), x509.RSA.ToXmlString (false));
		}

		[Test]
		public void Farscape_Strong_Path () 
		{
			// exported from MS certificate manager with strong encryption
			// and including the certificate path
			PKCS12 p12 = new PKCS12 (farscape_strong_path_pfx, "farscape");
			RSA rsa = (RSA) p12.Keys [0];
			X509Certificate x509 = p12.Certificates [0];
			AssertEquals (rsa.ToXmlString (false), x509.RSA.ToXmlString (false));
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void BadFarscape () 
		{
			byte[] p12data = (byte[]) farscape_pfx.Clone ();
			p12data [38] = 0xFF;
			PKCS12 p12 = new PKCS12 (p12data, "farscape");
		}

		static private byte[] password1 = { 0x00, 0x73, 0x00, 0x6D, 0x00, 0x65, 0x00, 0x67, 0x00, 0x00 }; // smeg 
		static private byte[] password2 = { 0x00, 0x71, 0x00, 0x75, 0x00, 0x65, 0x00, 0x65, 0x00, 0x67, 0x00, 0x00 }; // queeg

		static private byte[] salt1 = { 0x0A, 0x58, 0xCF, 0x64, 0x53, 0x0D, 0x82, 0x3F };
		static private byte[] salt2 = { 0x64, 0x2B, 0x99, 0xAB, 0x44, 0xFB, 0x4B, 0x1F };
		static private byte[] salt3 = { 0x3D, 0x83, 0xC0, 0xE4, 0x54, 0x6A, 0xC1, 0x40 };
		static private byte[] salt4 = { 0x05, 0xDE, 0xC9, 0x59, 0xAC, 0xFF, 0x72, 0xF7 };
		static private byte[] salt5 = { 0x16, 0x82, 0xC0, 0xFC, 0x5B, 0x3F, 0x7E, 0xC5 };
		static private byte[] salt6 = { 0x26, 0x32, 0x16, 0xFC, 0xC2, 0xFA, 0xB3, 0x1C };

		[Test]
		public void KeyGeneration_Test1 () 
		{
			PKCS12.DeriveBytes db = new PKCS12.DeriveBytes ();
			db.HashName = "SHA1";
			db.IterationCount = 1;
			db.Password = password1;
			db.Salt = salt1;
			
			byte[] key = db.DeriveKey (24);
			AssertEquals ("Key(1)", "8A-AA-E6-29-7B-6C-B0-46-42-AB-5B-07-78-51-28-4E-B7-12-8F-1A-2A-7F-BC-A3", BitConverter.ToString (key));

			byte[] iv = db.DeriveIV (8);
			AssertEquals ("IV(2)", "79-99-3D-FE-04-8D-3B-76", BitConverter.ToString (iv));
		}

		[Test]
		public void KeyGeneration_Test2 () 
		{
			PKCS12.DeriveBytes db = new PKCS12.DeriveBytes ();
			db.HashName = "SHA1";
			db.IterationCount = 1;
			db.Password = password1;
			db.Salt = salt2;

			byte[] key = db.DeriveKey (24);
			AssertEquals ("Key(1)", "F3-A9-5F-EC-48-D7-71-1E-98-5C-FE-67-90-8C-5A-B7-9F-A3-D7-C5-CA-A5-D9-66", BitConverter.ToString (key));

			byte[] iv = db.DeriveIV (8);
			AssertEquals ("IV(2)", "C0-A3-8D-64-A7-9B-EA-1D", BitConverter.ToString (iv));

			db.Salt = salt3;
			byte[] mac = db.DeriveMAC (20);
			AssertEquals ("MAC(3)", "8D-96-7D-88-F6-CA-A9-D7-14-80-0A-B3-D4-80-51-D6-3F-73-A3-12", BitConverter.ToString (mac));
		}

		[Test]
		public void KeyGeneration_Test3 () 
		{
			PKCS12.DeriveBytes db = new PKCS12.DeriveBytes ();
			db.HashName = "SHA1";
			db.IterationCount = 1000;
			db.Password = password2;
			db.Salt = salt4;

			byte[] key = db.DeriveKey (24);
			AssertEquals ("Key(1)", "ED-20-34-E3-63-28-83-0F-F0-9D-F1-E1-A0-7D-D3-57-18-5D-AC-0D-4F-9E-B3-D4", BitConverter.ToString (key));

			byte[] iv = db.DeriveIV (8);
			AssertEquals ("IV(2)", "11-DE-DA-D7-75-8D-48-60", BitConverter.ToString (iv));
		}

		[Test]
		public void KeyGeneration_Test4 () 
		{
			PKCS12.DeriveBytes db = new PKCS12.DeriveBytes ();
			db.HashName = "SHA1";
			db.IterationCount = 1000;
			db.Password = password2;
			db.Salt = salt5;

			byte[] key = db.DeriveKey (24);
			AssertEquals ("Key(1)", "48-3D-D6-E9-19-D7-DE-2E-8E-64-8B-A8-F8-62-F3-FB-FB-DC-2B-CB-2C-02-95-7F", BitConverter.ToString (key));

			byte[] iv = db.DeriveIV (8);
			AssertEquals ("IV(2)", "9D-46-1D-1B-00-35-5C-50", BitConverter.ToString (iv));

			db.Salt = salt6;
			byte[] mac = db.DeriveMAC (20);
			AssertEquals ("MAC(3)", "5E-C4-C7-A8-0D-F6-52-29-4C-39-25-B6-48-9A-7A-B8-57-C8-34-76", BitConverter.ToString (mac));
		}

		[Test]
		public void Clone_Test ()
		{
			PKCS12 p12 = new PKCS12 (farscape_nopwd_pfx);
			PKCS12 cp12 = (PKCS12) p12.Clone ();

			Assert ("C1", ! (p12 == cp12));
		}

		[Test]
		public void AddCertificate_Test1 ()
		{
			PKCS12 p12 = new PKCS12 ();
			X509Certificate x509 = new X509Certificate (cert);
			p12.AddCertificate (x509);

			AssertEquals ("AC1.1", p12.Certificates.Count, 1);
			Assert ("AC1.2", Compare (p12.Certificates [0].RawData, cert));
		}

		[Test]
		public void AddCertificate_Test2 ()
		{
			PKCS12 p12 = new PKCS12 ();
			X509Certificate x509 = new X509Certificate (cert);

			IDictionary attrs = new Hashtable ();
			ArrayList attrValues = new ArrayList ();
			attrValues.Add (Encoding.BigEndianUnicode.GetBytes ("Friendly name"));
			attrs.Add (PKCS9.friendlyName, attrValues);

			p12.AddCertificate (x509, attrs);

			AssertEquals ("AC2.1", p12.Certificates.Count, 1);
			Assert ("AC2.2", Compare (p12.Certificates [0].RawData, cert));
		}

		[Test]
		public void RemoveCertificate_Test1 ()
		{
			PKCS12 p12 = new PKCS12 ();
			X509Certificate x509 = new X509Certificate (cert);
			p12.AddCertificate (x509);

			AssertEquals ("RC1.1", p12.Certificates.Count, 1);

			p12.RemoveCertificate (x509);

			AssertEquals ("RC1.2", p12.Certificates.Count, 0);
		}

		[Test]
		public void RemoveCertificate_Test2 ()
		{
			PKCS12 p12 = new PKCS12 ();
			X509Certificate x509 = new X509Certificate (cert);

			IDictionary attrs = new Hashtable ();
			ArrayList attrValues = new ArrayList ();
			attrValues.Add (Encoding.BigEndianUnicode.GetBytes ("Friendly name"));
			attrs.Add (PKCS9.friendlyName, attrValues);

			p12.AddCertificate (x509, attrs);

			AssertEquals ("RC2.1", p12.Certificates.Count, 1);

			p12.RemoveCertificate (x509, attrs);

			AssertEquals ("RC2.2", p12.Certificates.Count, 0);
		}

		[Test]
		public void AddPkcs8ShroudedKeyBag_Test1 ()
		{
			PKCS12 p12 = new PKCS12 ();

			p12.AddPkcs8ShroudedKeyBag (RSA.Create ());

			AssertEquals ("AP1", p12.Keys.Count, 1);
		}

		[Test]
		public void AddPkcs8ShroudedKeyBag_Test2 ()
		{
			PKCS12 p12 = new PKCS12 ();

			IDictionary attrs = new Hashtable ();
			ArrayList attrValues = new ArrayList ();
			attrValues.Add (Encoding.BigEndianUnicode.GetBytes ("Friendly name"));
			attrs.Add (PKCS9.friendlyName, attrValues);

			p12.AddPkcs8ShroudedKeyBag (RSA.Create (), attrs);

			AssertEquals ("AP2", p12.Keys.Count, 1);
		}

		[Test]
		public void RemovePkcs8ShroudedKeyBag_Test ()
		{
			PKCS12 p12 = new PKCS12 ();
			RSAManaged rsa = new RSAManaged (384);
			p12.AddPkcs8ShroudedKeyBag (rsa);
			AssertEquals ("RP.1", p12.Keys.Count, 1);

			RSAManaged rsa2 = new RSAManaged (384);
			p12.RemovePkcs8ShroudedKeyBag (rsa2);
			AssertEquals ("RP.2", p12.Keys.Count, 1);

			p12.RemovePkcs8ShroudedKeyBag (rsa);
			AssertEquals ("RP.3", p12.Keys.Count, 0);
		}

		[Test]
		public void AddKeyBag_Test1 ()
		{
			PKCS12 p12 = new PKCS12 ();

			p12.AddKeyBag (RSA.Create ());

			AssertEquals ("AK1", p12.Keys.Count, 1);
		}

		[Test]
		public void AddKeyBag_Test2 ()
		{
			PKCS12 p12 = new PKCS12 ();

			IDictionary attrs = new Hashtable ();
			ArrayList attrValues = new ArrayList ();
			attrValues.Add (Encoding.BigEndianUnicode.GetBytes ("Friendly name"));
			attrs.Add (PKCS9.friendlyName, attrValues);

			p12.AddKeyBag (RSA.Create (), attrs);

			AssertEquals ("AK2", p12.Keys.Count, 1);
		}

		[Test]
		public void RemoveKeyBag_Test ()
		{
			PKCS12 p12 = new PKCS12 ();
			RSAManaged rsa = new RSAManaged (384);
			p12.AddKeyBag (rsa);
			AssertEquals ("RK.1", p12.Keys.Count, 1);

			RSAManaged rsa2 = new RSAManaged (384);
			p12.RemoveKeyBag (rsa2);
			AssertEquals ("RK.2", p12.Keys.Count, 1);

			p12.RemoveKeyBag (rsa);
			AssertEquals ("RK.3", p12.Keys.Count, 0);
		}

		[Test]
		public void GetAsymmetricAlgorithm_Test ()
		{
			PKCS12 p12 = new PKCS12 ();

			IDictionary attrs = new Hashtable ();
			ArrayList attrValues = new ArrayList ();
			attrValues.Add (Encoding.BigEndianUnicode.GetBytes ("Friendly name"));
			attrs.Add (PKCS9.friendlyName, attrValues);

			p12.AddKeyBag (RSA.Create (), attrs);

			AssertEquals ("GA.1", p12.Keys.Count, 1);

			AsymmetricAlgorithm aa = p12.GetAsymmetricAlgorithm (attrs);

			AssertNotNull ("GA.2", aa);
		}

		[Test]
		public void GetCertificate_Test ()
		{
			PKCS12 p12 = new PKCS12 ();
			X509Certificate x509 = new X509Certificate (cert);

			IDictionary attrs = new Hashtable ();
			ArrayList attrValues = new ArrayList ();
			attrValues.Add (Encoding.BigEndianUnicode.GetBytes ("Friendly name"));
			attrs.Add (PKCS9.friendlyName, attrValues);

			p12.AddCertificate (x509, attrs);

			AssertEquals ("GC.1", p12.Certificates.Count, 1);

			X509Certificate xc = p12.GetCertificate (attrs);

			AssertNotNull ("GC.2", xc);
			Assert ("GC.3", Compare (x509.RawData, xc.RawData));
		}

		[Test]
		public void GetAttributes_Test1 ()
		{
			PKCS12 p12 = new PKCS12 ();

			IDictionary attrs = new Hashtable ();
			ArrayList attrValues = new ArrayList ();
			attrValues.Add (Encoding.BigEndianUnicode.GetBytes ("Friendly name"));
			attrs.Add (PKCS9.friendlyName, attrValues);

			AsymmetricAlgorithm aa = RSA.Create ();
			p12.AddKeyBag (aa, attrs);

			AssertEquals ("GA1.1", p12.Keys.Count, 1);

			IDictionary pattrs = p12.GetAttributes (aa);

			Assert ("GA1.2", pattrs.Contains (PKCS9.friendlyName));
		}

		[Test]
		public void GetAttributes_Test2 ()
		{
			PKCS12 p12 = new PKCS12 ();
			X509Certificate x509 = new X509Certificate (cert);

			IDictionary attrs = new Hashtable ();
			ArrayList attrValues = new ArrayList ();
			attrValues.Add (Encoding.BigEndianUnicode.GetBytes ("Friendly name"));
			attrs.Add (PKCS9.friendlyName, attrValues);

			p12.AddCertificate (x509, attrs);

			AssertEquals ("GA2.1", p12.Certificates.Count, 1);

			IDictionary pattrs = p12.GetAttributes (x509);

			Assert ("GA2.2", pattrs.Contains (PKCS9.friendlyName));
		}

		private bool Compare (byte[] array1, byte[] array2)
		{
			bool bResult = (array1.Length == array2.Length);
			if (bResult) {
				for (int i = 0; i < array1.Length; i++) {
					if (array1[i] != array2[i])
						return false;
				}
			}
			return bResult;
		}

		static public byte[] p12_truncated = { 0x30, 0x82, 0x06, 0x6E, 0x02, 0x01, 0x03, 0x30, 0x82, 0x06, 0x2A, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x06, 0x1B, 0x04, 0x82, 0x06, 0x17, 0x30, 0x82, 0x06, 0x13, 0x30, 0x82, 0x03, 0xB4, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x03, 0xA5, 0x04, 0x82, 0x03, 0xA1, 0x30, 0x82, 0x03, 0x9D, 0x30, 0x82, 0x03, 0x99, 0x06, 0x0B, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x0A, 0x01, 0x02, 0xA0, 0x82, 0x02, 0xAE, 0x30, 0x82, 0x02, 0xAA, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 
			0x0E, 0x04, 0x08, 0x5C, 0xCD, 0xF4, 0x4E, 0x3F, 0x08, 0xF9, 0x03, 0x02, 0x02, 0x07, 0xD0, 0x04, 0x82, 0x02, 0x88, 0xA7, 0xD8, 0xCE, 0x48, 0x16, 0xA6, 0x93, 0x74, 0x30, 0xFA, 0x81, 0xAC, 0x99, 0x8E, 0x47, 0xC9, 0x51, 0xBB, 0x7A, 0x10, 0x1A, 0x0E, 0x48, 0xC8, 0x47, 0xF1, 0x10, 0xE8, 0xFD, 0xB1, 0x99, 0xCF, 0x49, 0x87, 0x81, 0xA3, 0x50, 0xC8, 0x08, 0x3B, 0xB4, 0x48, 0xAB, 0x6B, 0x41, 0xA5, 0xC0, 0x03, 0x5E, 0x50, 0x4F, 0x58, 0x40, 0xF0, 0x91, 0xA9, 0x33, 0x99, 0xC4, 0x73, 0xA2, 0x84, 0x0B, 0x57, 0xCB, 0x33, 0x15, 0x5D, 0xAD, 0xAC, 0x8D, 0x62, 0x00, 0x8A, 0x0D, 0xB1, 0xA2, 0x61, 0xED, 0xEE, 0xF1, 
			0x63, 0xB5, 0xC2, 0x99, 0x47, 0x15, 0x7C, 0x5F, 0x83, 0x89, 0x58, 0x9F, 0xE5, 0xD5, 0x67, 0xDA, 0x90, 0x4A, 0xA3, 0xD1, 0xB8, 0xB0, 0x10, 0x12, 0x51, 0xD5, 0x13, 0xF2, 0xA8, 0xF4, 0xC9, 0x74, 0xE7, 0xD3, 0xEF, 0xA2, 0x33, 0xC2, 0xFD, 0x9B, 0xFA, 0x40, 0xE3, 0xB8, 0xF4, 0x62, 0xF9, 0x53, 0xDC, 0x56, 0x90, 0xE3, 0x15, 0x6E, 0x4E, 0xC1, 0xBC, 0xF0, 0x70, 0x7C, 0x10, 0x93, 0x4B, 0xF1, 0x5F, 0xC4, 0x45, 0xAA, 0x83, 0xF1, 0x84, 0x0E, 0x83, 0xE5, 0xA9, 0xE6, 0xE3, 0xA5, 0x10, 0xB3, 0x24, 0x4F, 0x57, 0x90, 0xC4, 0x16, 0x64, 0x64, 0x1F, 0x82, 0xCA, 0x9D, 0xB8, 0x7D, 0x87, 0xA2, 0x70, 0xE3, 0x58, 0xDF, 
			0x42, 0x92, 0xA0, 0xE6, 0xB2, 0xCA, 0x5F, 0x20, 0x0C, 0x63, 0x06, 0xBB, 0x17, 0xF0, 0x8F, 0xC7, 0x1C, 0x6D, 0x8C, 0x2D, 0xA1, 0xBE, 0x8C, 0xA3, 0x8D, 0x83, 0x42, 0xEF, 0xAD, 0x76, 0xD2, 0x9F, 0xC4, 0xFD, 0x59, 0x78, 0x4B, 0x4B, 0x33, 0x80, 0x11, 0xDE, 0x02, 0xED, 0x22, 0xA5, 0xBD, 0xBC, 0x23, 0x6A, 0x77, 0x4F, 0x32, 0xE9, 0x15, 0x3E, 0xC1, 0xDB, 0x6D, 0xC0, 0x4D, 0x4E, 0x6D, 0x92, 0x66, 0x45, 0x53, 0xD3, 0x34, 0x30, 0x43, 0x14, 0x82, 0x03, 0xB7, 0xB0, 0x51, 0x5F, 0xE9, 0x66, 0x67, 0xC6, 0x62, 0xB1, 0x72, 0xE1, 0x7F, 0xC7, 0x7F, 0x13, 0x6F, 0xCC, 0x18, 0xFF, 0xB5, 0x37, 0x64, 0xC9, 0x9E, 0xD2, 
			0xE7, 0x08, 0xFE, 0xA8, 0x26, 0xA7, 0xAC, 0x99, 0x47, 0xB0, 0x46, 0xDC, 0x7E, 0xA5, 0xED, 0xC5, 0x3B, 0x85, 0xD9, 0xBF, 0x8C, 0xD4, 0x97, 0xC2, 0x57, 0xDF, 0x6D, 0x88, 0x88, 0x2A, 0x90, 0xC3, 0x84, 0xEE, 0x86, 0x0E, 0x3F, 0x00, 0x5D, 0xC3, 0x5C, 0x4F, 0xCA, 0x80, 0xAA, 0x93, 0x15, 0xD9, 0x49, 0x85, 0x78, 0xDD, 0xC9, 0x95, 0xF6, 0x06, 0x2A, 0x43, 0xED, 0x60, 0x31, 0x13, 0xCF, 0xB1, 0xAA, 0x95, 0xB7, 0xBF, 0x64, 0x3C, 0xBB, 0x60, 0xA8, 0x6F, 0x30, 0x41, 0xEF, 0xBF, 0x80, 0xFA, 0xDF, 0xC4, 0x02, 0x16, 0x55, 0xDF, 0x47, 0x50, 0x4E, 0x20, 0xD2, 0x93, 0xFC, 0x96, 0x9E, 0xEC, 0x0D, 0x44, 0x45, 0x95, 
			0xCD, 0xD3, 0x28, 0x9F, 0xC3, 0xB7, 0x0B, 0xC2, 0x82, 0xF3, 0xDF, 0x62, 0x0F, 0x21, 0x36, 0xB0, 0x7C, 0xF7, 0xBF, 0xE9, 0x3F, 0xE4, 0xDA, 0x8A, 0x01, 0xA2, 0x23, 0x6D, 0x87, 0x42, 0x8C, 0x2D, 0x29, 0xE5, 0x2B, 0x7F, 0x7D, 0xB2, 0x6E, 0x19, 0x31, 0xBE, 0x4F, 0xF0, 0xB0, 0xC7, 0xF1, 0x16, 0xE8, 0xE3, 0x89, 0x01, 0x39, 0x75, 0xDF, 0xE3, 0x92, 0xDD, 0xC5, 0xFF, 0x0E, 0x35, 0xC7, 0x60, 0x0F, 0x65, 0x3C, 0xBF, 0xF1, 0x3B, 0x18, 0x8E, 0x44, 0x13, 0x00, 0xE3, 0xDA, 0xD8, 0xD0, 0xAC, 0x69, 0xCA, 0x5F, 0x2D, 0x78, 0x69, 0xA0, 0x5B, 0xFC, 0xA2, 0x8B, 0x5D, 0x78, 0x28, 0x25, 0x9A, 0xBC, 0x82, 0xDD, 0xFE, 
			0xF9, 0x19, 0xDD, 0xD3, 0xBB, 0x73, 0xA3, 0x4F, 0xB4, 0xF5, 0x1F, 0xF2, 0x28, 0x1C, 0xF2, 0x19, 0x1B, 0x1B, 0x55, 0x88, 0x86, 0xCE, 0xCC, 0xB2, 0x02, 0x82, 0x95, 0xD0, 0xED, 0x3D, 0xE3, 0xD2, 0xA8, 0x69, 0xA0, 0x2A, 0x57, 0x4E, 0x97, 0xDA, 0xAD, 0x88, 0xF2, 0x80, 0xDD, 0xED, 0x0F, 0x0E, 0xF7, 0x45, 0x56, 0x73, 0x74, 0x5B, 0xA6, 0x03, 0x83, 0xFE, 0x0D, 0x3F, 0x07, 0x9A, 0x89, 0x31, 0x31, 0x53, 0x19, 0x17, 0xEE, 0xDE, 0x2A, 0x5A, 0x38, 0x7E, 0x97, 0x8A, 0xEA, 0x25, 0xA8, 0x78, 0x24, 0x2D, 0xD9, 0xB6, 0x3F, 0xA8, 0x15, 0x56, 0xF3, 0x5B, 0x58, 0x80, 0xB9, 0x54, 0xEC, 0xBF, 0xB0, 0x12, 0xF7, 0x9A, 
			0xA5, 0x8E, 0x39, 0x3D, 0xBA, 0x6D, 0x41, 0xD9, 0x5D, 0x3E, 0x9E, 0x16, 0x0E, 0x2D, 0xB5, 0x22, 0x70, 0xB1, 0xBE, 0xAC, 0x5E, 0x80, 0xB6, 0xEE, 0x97, 0x5A, 0xD1, 0x30, 0x2D, 0x4C, 0xF4, 0x4B, 0x5E, 0xFB, 0xF3, 0x7B, 0xDB, 0xFC, 0x8F, 0xEC, 0x56, 0x73, 0xC1, 0xB3, 0x63, 0x30, 0x59, 0xE2, 0x04, 0x9E, 0x84, 0x6A, 0xF1, 0x5A, 0x31, 0x62, 0xB0, 0x59, 0xDA, 0x17, 0x2A, 0x34, 0x9A, 0x60, 0x4B, 0xB3, 0x8F, 0x31, 0x81, 0xD7, 0x30, 0x13, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x15, 0x31, 0x06, 0x04, 0x04, 0x01, 0x00, 0x00, 0x00, 0x30, 0x5B, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 
			0x0D, 0x01, 0x09, 0x14, 0x31, 0x4E, 0x1E, 0x4C, 0x00, 0x7B, 0x00, 0x33, 0x00, 0x39, 0x00, 0x31, 0x00, 0x42, 0x00, 0x39, 0x00, 0x42, 0x00, 0x36, 0x00, 0x32, 0x00, 0x2D, 0x00, 0x43, 0x00, 0x38, 0x00, 0x38, 0x00, 0x45, 0x00, 0x2D, 0x00, 0x34, 0x00, 0x37, 0x00, 0x45, 0x00, 0x38, 0x00, 0x2D, 0x00, 0x42, 0x00, 0x35, 0x00, 0x30, 0x00, 0x32, 0x00, 0x2D, 0x00, 0x39, 0x00, 0x36, 0x00, 0x32, 0x00, 0x39, 0x00, 0x33, 0x00, 0x45, 0x00, 0x34, 0x00, 0x43, 0x00, 0x30, 0x00, 0x46, 0x00, 0x32, 0x00, 0x35, 0x00, 0x7D, 0x30, 0x63, 0x06, 0x09, 0x2B, 0x06, 0x01, 0x04, 0x01, 0x82, 0x37, 0x11, 0x01, 0x31, 0x56, 0x1E, 
			0x54, 0x00, 0x4D, 0x00, 0x69, 0x00, 0x63, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x73, 0x00, 0x6F, 0x00, 0x66, 0x00, 0x74, 0x00, 0x20, 0x00, 0x42, 0x00, 0x61, 0x00, 0x73, 0x00, 0x65, 0x00, 0x20, 0x00, 0x43, 0x00, 0x72, 0x00, 0x79, 0x00, 0x70, 0x00, 0x74, 0x00, 0x6F, 0x00, 0x67, 0x00, 0x72, 0x00, 0x61, 0x00, 0x70, 0x00, 0x68, 0x00, 0x69, 0x00, 0x63, 0x00, 0x20, 0x00, 0x50, 0x00, 0x72, 0x00, 0x6F, 0x00, 0x76, 0x00, 0x69, 0x00, 0x64, 0x00, 0x65, 0x00, 0x72, 0x00, 0x20, 0x00, 0x76, 0x00, 0x31, 0x00, 0x2E, 0x00, 0x30, 0x30, 0x82, 0x02, 0x57, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x06, 
			0xA0, 0x82, 0x02, 0x48, 0x30, 0x82, 0x02, 0x44, 0x02, 0x01, 0x00, 0x30, 0x82, 0x02, 0x3D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x06, 0x30, 0x0E, 0x04, 0x08, 0x6B, 0x7A, 0x7C, 0x8D, 0xA7, 0xB3, 0x3B, 0x17, 0x02, 0x02, 0x07, 0xD0, 0x80, 0x82, 0x02, 0x10, 0xB5, 0x09, 0xC6, 0xFC, 0x02, 0x2C, 0x10, 0xA3, 0xAD, 0xB8, 0x84, 0x4E, 0x89, 0xA9, 0xBF, 0x07, 0x2C, 0xE0, 0xB8, 0x1C, 0x26, 0xA0, 0x7C, 0x07, 0x34, 0x3D, 0x49, 0xB5, 0x55, 0xF9, 0xAB, 0xE7, 0x8D, 0x8D, 0xF6, 0x25, 0x41, 0xBA, 0x55, 0x23, 
			0x18, 0xE2, 0x36, 0x8A, 0xC6, 0xA0, 0x22, 0x3A, 0x79, 0x24, 0xDC, 0xEC, 0x5B, 0xAA, 0xE6, 0xE9, 0x73, 0x89, 0x1E, 0x4B, 0xB4, 0xDC, 0x41, 0xCF, 0x0D, 0xA3, 0x51, 0x2E, 0x1D, 0xD2, 0xDE, 0x0A, 0x6D, 0xAF, 0x74, 0x6C, 0x04, 0x55, 0x29, 0x0B, 0xF9, 0xBA, 0x6F, 0x92, 0x79, 0x4A, 0xE2, 0xB6, 0x04, 0xA0, 0xBA, 0xB7, 0xF2, 0xD9, 0x0B, 0x1E, 0xB3, 0xF4, 0x21, 0x67, 0x0F, 0xA3, 0x91, 0xB5, 0x29, 0xB7, 0x4A, 0x44, 0xD5, 0xFA, 0xDE, 0x1B, 0x41, 0xA1, 0xD8, 0x0E, 0xFB, 0x61, 0xF1, 0x25, 0x5F, 0x4C, 0x69, 0x45, 0x35, 0x14, 0x12, 0xD8, 0x02, 0xD6, 0x9F, 0xD5, 0x62, 0xB9, 0x8F, 0x0F, 0xD8, 0x9A, 0x9D, 0xA6, 
			0x42, 0x2D, 0xDD, 0x84, 0x76, 0x46, 0xFE, 0x66, 0x97, 0x50, 0x19, 0xDD, 0xDA, 0xDB, 0x06, 0x75, 0xB9, 0xB2, 0x6E, 0x95, 0x0D, 0x6D, 0xC5, 0xD4, 0x4B, 0x35, 0x66, 0x3E, 0x3B, 0x95, 0x62, 0x0B, 0x9E, 0x00, 0x89, 0x85, 0xAA, 0x56, 0xD8, 0x4B, 0xEC, 0x01, 0x39, 0x2A, 0xBD, 0xEA, 0x13, 0x91, 0xC3, 0x76, 0x7D, 0x60, 0xCE, 0x5C, 0x22, 0x2A, 0xCE, 0x96, 0x8D, 0xB1, 0x9F, 0x76, 0xDF, 0xDF, 0xC2, 0x29, 0x25, 0xA4, 0xFB, 0x2B, 0x74, 0x09, 0xD2, 0xB5, 0x8D, 0x33, 0x65, 0x3C, 0xCB, 0x7C, 0xFF, 0x78, 0x22, 0x53, 0xAC, 0x64, 0x1C, 0x1C, 0x94, 0xAA, 0xF4, 0x29, 0x31, 0x60, 0x09, 0x51, 0xDB, 0x5F, 0x54, 0x26, 
			0x1E, 0xBD, 0x54, 0x4A, 0x2A, 0xD1, 0x0B, 0x39, 0x2C, 0xF1, 0xA2, 0x56, 0xC3, 0x0B, 0xBF, 0xB7, 0xCA, 0x38, 0xBF, 0x95, 0x80, 0x01, 0xCC, 0x90, 0x97, 0x3A, 0x04, 0xAB, 0xFC, 0xA6, 0xD0, 0x54, 0x7A, 0x11, 0x16, 0x51, 0x09, 0xC5, 0x7E, 0x77, 0x05, 0x49, 0x8C, 0xDF, 0x26, 0xD5, 0x86, 0xE8, 0xE3, 0x72, 0xD3, 0x9D, 0x47, 0x14, 0xB0, 0x3C, 0xF1, 0x7C, 0xCF, 0x37, 0x8E, 0x04, 0x8F, 0x3A, 0xD9, 0xCC, 0x95, 0x79, 0x27, 0x58, 0x1D, 0xCD, 0x06, 0x10, 0x4E, 0x08, 0x56, 0xF2, 0xF9, 0xCD, 0xB2, 0xA3, 0xB8, 0xBB, 0x5F, 0x5E, 0xF3, 0x51, 0x34, 0x61, 0xAA, 0x27, 0x5F, 0x54, 0x60, 0x6D, 0x6C, 0x96, 0x91, 0xAA, 
			0x6F, 0xC1, 0x04, 0xEF, 0xE0, 0x83, 0xA8, 0x24, 0x5F, 0xFE, 0x64, 0x59, 0x83, 0x24, 0x6A, 0x1D, 0x2A, 0x90, 0x38, 0x5D, 0x37, 0xD0, 0x25, 0xBC, 0x74, 0x39, 0xFE, 0x52, 0x2A, 0x36, 0xA4, 0x1A, 0x72, 0xB8, 0x5A, 0xB9, 0xAD, 0x89, 0x38, 0x02, 0x39, 0x95, 0x0A, 0xAF, 0x11, 0x5C, 0x7B, 0x86, 0x9A, 0x7C, 0x8D, 0x94, 0xDB, 0xC8, 0x3A, 0x72, 0x1D, 0x79, 0xC6, 0xE8, 0x0D, 0xF3, 0xBA, 0x01, 0x06, 0x1B, 0x11, 0x7E, 0x1F, 0xF4, 0xC0, 0x0E, 0x90, 0x3B, 0x6F, 0x03, 0xD9, 0xC8, 0x7F, 0x17, 0xE2, 0x97, 0xC9, 0x26, 0x54, 0x9D, 0x09, 0x9A, 0x45, 0xB9, 0xEE, 0x10, 0xED, 0x73, 0x39, 0xC9, 0x78, 0xCD, 0x99, 0x23, 
			0xCB, 0xA4, 0xD7, 0x83, 0x53, 0x3C, 0x9A, 0xFA, 0x04, 0x89, 0x91, 0xE3, 0xB1, 0xCC, 0xF2, 0xB5, 0xD1, 0xF6, 0x83, 0x19, 0xA4, 0xFC, 0x6B, 0x23, 0x60, 0x35, 0xD9, 0xC6, 0xF1, 0x08, 0xBB, 0xF2, 0x5A, 0x0B, 0x1C, 0x18, 0x20, 0xEF, 0x75, 0x8E, 0x4F, 0x96, 0x79, 0xE7, 0xA3, 0x4D, 0x9C, 0x38, 0x5A, 0xEE, 0x13, 0x3D, 0x73, 0x88, 0xD6, 0xAF, 0x89, 0x1F, 0x15, 0x1D, 0xFE, 0xB3, 0x55, 0x72, 0x71, 0x6C, 0xD6, 0xB3, 0x39, 0x5A, 0x59, 0xB3, 0x1C, 0x91, 0x9C, 0x55, 0x81, 0xF0, 0x87, 0x48, 0x96, 0x21, 0x00, 0xBB, 0x13, 0x23, 0x73, 0x5A, 0x30, 0x3B, 0x30, 0x1F, 0x30, 0x07, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 
			0x1A, 0x04, 0x14, 0x0B, 0xAE, 0xA0, 0x45, 0xEF, 0xA1, 0x40, 0x72, 0x6A, 0xBA, 0xF7, 0xDE, 0x7F, 0x59, 0x7A, 0x00, 0xFD, 0x25, 0xF8, 0x54, 0x04, 0x14, 0x11, 0x4F, 0xDA, 0xEC, 0x37, 0x6B, 0x44, 0x8E, 0xDD, 0x13, 0x24, 0x46, 0x8E, 0xA5, 0x6C, 0x36, 0x16, 0x7C, 0x40, 0x14, 0x02, 0x02, 0x07, 0xD0 };

		private static string xmlkey = "<RSAKeyValue><Modulus>pG9cK8b68qdr2z4TI4x+gcqiTBMhfRQhhOW8S13hnewv+RTJsqCFMDAPj22ch5likrE0CFFBpnEAYL2dgX+mrXUyqaaOf44UXwRgH72HKZuZSHgrTOOdwX4EqLNI4UkQRMwGLqum6z2jN26VRCk4y96cehepMI5347k3+sbl/XE=</Modulus><Exponent>EQ==</Exponent><P>yRv6YfcMYoINO4Fj1kpBaMWkzsJcl7zWe3PTyMKUAhbY+6VjXda5ztE8rPVvRL1BOKNI3zw0K0edbQ9mzC53tQ==</P><Q>0VDa9prBNFvTwMs5nEAeriFMIG07/KmkiaXv51EbDpc5vrgBlJsiWMpjl8SYL+baC0SISUagIDN6lFEPZMGMTQ==</Q><DP>I31oa6QRPo9rvzTzgCs4uCLg2TFbok6AM+dDfcf8AF5ihsLVTMuKM45WAGeMGzB06+CUY6E2Yf2UQGwhMxdCTQ==</DP><DQ>uLDBM/H1xMl+mw2rTaH88/9DK6utZneRLikA2ylyOgz2t1cQdBBpmaOFDXE6/Rb8oIfDi/MFwg9OCmWkK7nWJQ==</DQ><InverseQ>VdaO3DUp+QV5OGvEVWcWK26Ou0tdSevT6icqlj36iiwDA6Np3WoiBapzu1An/O1u7o/qo3igFsEjCvfz6urXFg==</InverseQ><D>OgkvlvrvKHdTPlIk317DPN4bKeiiaGF1Pfa67Wxtv0RNKrwK88AvAfLYUL1GTfnmjiBst6Q1Sc2HqaxVtTwctSAJljmkewv1HWxDHu1KKQGoWyFJxysEczm8UNP0qFAiqEJ3zrnWfcuKNVoRJHv4PlS4O5+zL58sC6pMEK1l35E=</D></RSAKeyValue>";

		[Test]
		public void CryptoApiLimitOnPasswordLength_Over ()
		{
			PKCS12.MaximumPasswordLength = PKCS12.CryptoApiPasswordLimit;
			// password too long - but the first 32 chars are ok
			PKCS12 p12 = new PKCS12 (p12_truncated, "1234567890123456789012345678901234567890");
			AssertEquals ("Key", xmlkey, (p12.Keys[0] as RSA).ToXmlString (true));
		}

		[Test]
		public void CryptoApiLimitOnPasswordLength_Identical ()
		{
			PKCS12.MaximumPasswordLength = PKCS12.CryptoApiPasswordLimit;
			// identical password
			PKCS12 p12 = new PKCS12 (p12_truncated, "12345678901234567890123456789012");
			AssertEquals ("Key", xmlkey, (p12.Keys[0] as RSA).ToXmlString (true));
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void CryptoApiLimitOnPasswordLength_Under ()
		{
			PKCS12.MaximumPasswordLength = PKCS12.CryptoApiPasswordLimit;
			// password too short
			new PKCS12 (p12_truncated, "1234567890123456789012345678901");
		}

		[Test] // see bug #76627
		public void MultipleKeys ()
		{
			RSAManaged r384a = new RSAManaged (384);
			RSAManaged r384b = new RSAManaged (384);
			RSAManaged r512 = new RSAManaged (512);
			PKCS12 p12 = new PKCS12 ();
			AssertEquals ("0", 0, p12.Keys.Count);
			p12.AddPkcs8ShroudedKeyBag (r512);
			AssertEquals ("1", 1, p12.Keys.Count);
			p12.AddPkcs8ShroudedKeyBag (r384a);
			AssertEquals ("2", 2, p12.Keys.Count);
			p12.AddPkcs8ShroudedKeyBag (r384b);
			AssertEquals ("3", 3, p12.Keys.Count);
		}
	}
}
